// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>

#include <maya/MAnimUtil.h>
#include <maya/MAnimControl.h>

static const char* DEFAULT_ATTR_GROUP_NAME = "DefaultGroup";

// Check if an attribute or one of its components is animated
static inline bool _isAttributeAnimated( const MPlug& attributePlug, bool checkIncoming = true )
{
    if( MAnimUtil::isAnimated( attributePlug ) )
    {
        return true;
    }

    // Check for compounds (vector) - only their children are animated
    else
    {
        MStatus status;
        MFnCompoundAttribute compoundAttr( attributePlug.attribute(), &status );
        if( status == MStatus::kSuccess )
        {
            for( unsigned int i=0; i<compoundAttr.numChildren(); ++i )
            {
                MObject childAttrObj = compoundAttr.child( i, &status );
                if( MStatus::kSuccess == status )
                {
                    MPlug childAttrPlug( attributePlug.node(), childAttrObj );
                    if( MAnimUtil::isAnimated( childAttrPlug ) )
                    {
                        return true;
                    }
                }
            }
        }
    }

    // Check for animated incoming connections
    MPlugArray sources;
    attributePlug.connectedTo( sources, true, false );
    if( sources.length() == 1 )
    {
        return _isAttributeAnimated( sources[0], false );
    }

    return false;
}


bool hctMayaSceneExporter::createAttribute( MObject nodeObject, MObject attrObject, hkxAttribute& hkx_attribute ) const
{
    const MTime::Unit timeUnit = MTime::uiUnit();
    const int numFrames = (int)( (m_options.m_endTime - m_options.m_startTime).as( timeUnit ) + 1 );

    const float timeInSeconds = (float) m_options.m_startTime.as(MTime::kSeconds);

    MPlug attributePlug( nodeObject, attrObject );
    const bool isAnimated = _isAttributeAnimated( attributePlug );
    const int arraySize = isAnimated ? numFrames : 1;

    hkx_attribute.m_name = HK_NULL;
    hkx_attribute.m_value = HK_NULL;

    MFnAttribute attribute(attrObject);
    MString attributeName = attribute.name();

    MStatus status;
    MFnUnitAttribute unitAttribute(attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        switch (unitAttribute.unitType())
        {
            case MFnUnitAttribute::kDistance:
            {
                hkxAnimatedFloat* animatedFloat = new hkxAnimatedFloat();

                animatedFloat->m_hint = hkxAttribute::HINT_SCALE;

                hkx_attribute.m_value = animatedFloat;

                animatedFloat->m_floats.setSize(1);

                attributePlug.getValue( animatedFloat->m_floats[0] );

                animatedFloat->removeReference();
                break;
            }
            case MFnUnitAttribute::kAngle:
            {
                hkxAnimatedFloat* animatedFloat = new hkxAnimatedFloat();

                animatedFloat->m_hint = hkxAttribute::HINT_FLIP;

                hkx_attribute.m_value = animatedFloat;

                animatedFloat->m_floats.setSize(1);

                attributePlug.getValue( animatedFloat->m_floats[0] );

                animatedFloat->removeReference();
                break;
            }
            default:
            {
                status.perror("Unsupported unit type");
            }
        }
    }

    MFnNumericAttribute numericAttribute(attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        MFnNumericData::Type numType = numericAttribute.unitType();

        switch (numType)
        {
            case MFnNumericData::kBoolean:
            {
                hkxSparselyAnimatedBool* animatedBool = new hkxSparselyAnimatedBool();

                hkx_attribute.m_value = animatedBool;

                animatedBool->m_bools.reserve(arraySize);
                animatedBool->m_times.reserve(arraySize);
                animatedBool->m_bools.setSize(1);
                animatedBool->m_times.setSize(1);

                bool theBool;
                attributePlug.getValue(theBool);
                animatedBool->m_bools[0] = theBool;
                animatedBool->m_times[0] = timeInSeconds;

                animatedBool->removeReference();
                break;
            }
            case MFnNumericData::kByte:
            case MFnNumericData::kShort:
            case MFnNumericData::kInt:
            {
                hkxSparselyAnimatedInt* animatedInt = new hkxSparselyAnimatedInt();

                hkx_attribute.m_value = animatedInt;

                animatedInt->m_ints.reserve(arraySize);
                animatedInt->m_times.reserve(arraySize);
                animatedInt->m_ints.setSize(1);
                animatedInt->m_times.setSize(1);

                attributePlug.getValue( animatedInt->m_ints[0] );
                animatedInt->m_times[0] = timeInSeconds;

                animatedInt->removeReference();
                break;
            }
            case MFnNumericData::kFloat:
            case MFnNumericData::kDouble:
            {
                hkxAnimatedFloat* animatedFloat = new hkxAnimatedFloat();

                animatedFloat->m_hint = hkxAttribute::HINT_NONE;

                hkx_attribute.m_value = animatedFloat;

                animatedFloat->m_floats.reserve(arraySize);
                animatedFloat->m_floats.setSize(1);

                attributePlug.getValue( animatedFloat->m_floats[0] );

                animatedFloat->removeReference();
                break;
            }
            case MFnNumericData::k3Double:
            case MFnNumericData::k3Float:
            {
                if (attributePlug.numChildren()!=3)
                {
                    status.perror("Unexpected number of children in k3Double attribute");
                    break;
                }

                hkxAnimatedVector* animatedVector = new hkxAnimatedVector();
                animatedVector->m_hint = hkxAttribute::HINT_NONE;

                hkx_attribute.m_value = animatedVector;

                animatedVector->m_vectors.reserve(4*arraySize);
                animatedVector->m_vectors.setSize(4);

                MPlug xPlug = attributePlug.child(0);
                MPlug yPlug = attributePlug.child(1);
                MPlug zPlug = attributePlug.child(2);

                float x,y,z;
                xPlug.getValue(x);
                yPlug.getValue(y);
                zPlug.getValue(z);
                animatedVector->m_vectors[0] = x;
                animatedVector->m_vectors[1] = y;
                animatedVector->m_vectors[2] = z;
                animatedVector->m_vectors[3] = 0;

                animatedVector->removeReference();
                break;
            }
            default:
            {
                status.perror("Unsupported numeric type");
            }
        }
    }

    MFnTypedAttribute typedAttribute(attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        if (MFnData::kString == typedAttribute.attrType())
        {
            hkxSparselyAnimatedString* animatedString = new hkxSparselyAnimatedString();
            hkx_attribute.m_value = animatedString;

            animatedString->m_strings.reserve(arraySize);
            animatedString->m_times.reserve(arraySize);

            animatedString->m_strings.setSize(1);
            animatedString->m_times.setSize(1);

            MString aString;
            attributePlug.getValue( aString);
            animatedString->m_strings[0] = aString.asChar();
            animatedString->m_times[0] = timeInSeconds;
            animatedString->removeReference();
        }
    }

    MFnEnumAttribute enumAttribute(attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        short minValue, maxValue;
        enumAttribute.getMin(minValue);
        enumAttribute.getMax(maxValue);
        MStatus statusFieldName;

        hkArray<hkxEnum::Item> itemsArray;
        for (short v=minValue; v<=maxValue; v++)
        {
            MString mayaName = enumAttribute.fieldName(v, &statusFieldName);
            if (MStatus::kSuccess== statusFieldName)
            {
                hkxEnum::Item& newItem = itemsArray.expandOne();
                newItem.m_name = mayaName.asChar();
                newItem.m_value = v;
            }
            else
            {
                status.perror("Invalid value in enum attribute");
            }
        }

        const int numItems = itemsArray.getSize();
        if (numItems>0)
        {
            MString enumMayaName = enumAttribute.name();
            hkxEnum* newEnum = new hkxEnum(/*enumMayaName.asChar()*/);
            newEnum->m_items.insertAt(0, itemsArray.begin(), itemsArray.getSize());

            hkxSparselyAnimatedEnum* animatedEnum = new hkxSparselyAnimatedEnum();

            hkx_attribute.m_value = animatedEnum;

            animatedEnum->m_enum = newEnum;

            animatedEnum->m_ints.reserve(arraySize);
            animatedEnum->m_times.reserve(arraySize);

            animatedEnum->m_ints.setSize(1);
            animatedEnum->m_times.setSize(1);

            attributePlug.getValue( animatedEnum->m_ints[0] );
            animatedEnum->m_times[0] = timeInSeconds;

            newEnum->removeReference();
            animatedEnum->removeReference();
        }
        else
        {
            status.perror("Enum attribute with no values");
        }
    }

    MFnMatrixAttribute matrixAttribute (attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        hkxAnimatedMatrix* animatedMatrix = new hkxAnimatedMatrix();

        // Assume matrices are always transformed
        animatedMatrix->m_hint = hkxAttribute::HINT_TRANSFORM_AND_SCALE;

        hkx_attribute.m_value = animatedMatrix;

        animatedMatrix->m_matrices.setSize(16);

        MObject anObject;
        status = attributePlug.getValue(anObject);
        MFnMatrixData matrixData (anObject);
        MMatrix aMatrix = matrixData.matrix(&status);

        hkMatrix4 hkm4;
        hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( aMatrix, hkm4 );
        hkm4.get4x4ColumnMajor(&animatedMatrix->m_matrices[0]);

        animatedMatrix->removeReference();
    }

    MFnMessageAttribute messageAttribute (attrObject, &status);
    if (MStatus::kSuccess == status)
    {
        MPlugArray plugArray;
        attributePlug.connectedTo( plugArray, true, false );
        if( plugArray.length() == 1 )
        {
            hkxSparselyAnimatedString* animatedString = new hkxSparselyAnimatedString();
            hkx_attribute.m_value = animatedString;

            animatedString->m_strings.reserve(arraySize);
            animatedString->m_times.reserve(arraySize);

            animatedString->m_strings.setSize(1);
            animatedString->m_times.setSize(1);

            MDagPath path;
            MDagPath::getAPathTo( plugArray[0].node(), path );
            MString name = path.partialPathName();  // the shortest unique name, which the HKX node also uses
            animatedString->m_strings[0] = name.asChar();

            animatedString->m_times[0] = timeInSeconds;

            animatedString->removeReference();
        }
    }

    const bool goodAttribute = hkx_attribute.m_value.getType() && hkx_attribute.m_value;

    if (goodAttribute)
    {
        hkx_attribute.m_name = attributeName.asChar();
    }

    return goodAttribute;

}

void hctMayaSceneExporter::addAnimatedAttributeToSample (const MPlug& mayaAttr, hkxAttribute* havokAttr)
{
    m_mayaAnimAttributes.append( mayaAttr );
    m_havokAnimAttributes.pushBack( havokAttr );
}

void hctMayaSceneExporter::sampleAttributes()
{
    // Get number of animated attributes
    const int numAnimAttribs = m_havokAnimAttributes.getSize();
    if( numAnimAttribs == 0 )
    {
        return;
    }

    // Get time increment
    const MTime::Unit timeUnit = MTime::uiUnit();
    const int numFrames = (int)( (m_options.m_endTime - m_options.m_startTime).as( timeUnit ) + 1 );
    const MTime tInc( 1, timeUnit ); // one frame at a time

    // Sample the attributes at each frame
    MTime curTime = m_options.m_startTime;
  bool wantProgress = MGlobal::mayaState() == MGlobal::kInteractive;
  if (wantProgress)
  {
    m_progressWindow.setRange( 0, numFrames );
  }
    for( int frame=0; frame<numFrames; ++frame )
    {
        MString str( "Sampling attributes (" );
        str += 100*frame/numFrames;
        str += "%)";
    if (wantProgress)
    {
      m_progressWindow.setStatus( str );
          m_progressWindow.setProgress( frame );
          if( m_progressWindow.isCancelled() )
          {
              return;
          }
    }
        // We know we have already sampled the first frame
        if( frame==0 )
        {
            curTime += tInc;
            continue;
        }

        MAnimControl::setCurrentTime( curTime );

        for( int i=0; i<numAnimAttribs; ++i )
        {
            const MPlug& maya_attribute = m_mayaAnimAttributes[i];
            hkxAttribute* hkx_attribute = m_havokAnimAttributes[i];

            if (hkxAnimatedFloat* animatedFloat = hkDynCast<hkxAnimatedFloat>(hkx_attribute->m_value))
            {
                maya_attribute.getValue( animatedFloat->m_floats.expandOne() );
                continue;
            }

            if (hkxAnimatedVector* animatedVector = hkDynCast<hkxAnimatedVector>(hkx_attribute->m_value))
            {
                MPlug xPlug = maya_attribute.child(0);
                MPlug yPlug = maya_attribute.child(1);
                MPlug zPlug = maya_attribute.child(2);

                float x,y,z;
                xPlug.getValue(x);
                yPlug.getValue(y);
                zPlug.getValue(z);

                hkFloat32* animVec = animatedVector->m_vectors.expandBy(4);
                animVec[0] = x;
                animVec[1] = y;
                animVec[2] = z;
                animVec[3] = 0;
                continue;
            }

            if (hkxSparselyAnimatedInt* animatedInt = hkDynCast<hkxSparselyAnimatedInt>(hkx_attribute->m_value))
            {
                int anInt;
                maya_attribute.getValue( anInt );
                if (anInt!= animatedInt->m_ints.back())
                {
                    animatedInt->m_ints.expandOne() = anInt;
                    animatedInt->m_times.expandOne() = (float) curTime.as(MTime::kSeconds);
                }
                continue;
            }

            if (hkxSparselyAnimatedBool* animatedBool = hkDynCast<hkxSparselyAnimatedBool>(hkx_attribute->m_value))
            {
                bool aBool;
                maya_attribute.getValue( aBool );
                if (aBool!=animatedBool->m_bools.back())
                {
                    animatedBool->m_bools.expandOne() = aBool;
                    animatedBool->m_times.expandOne() = (float) curTime.as(MTime::kSeconds);
                }
                continue;
            }

            if (hkxSparselyAnimatedString* animatedString = hkDynCast<hkxSparselyAnimatedString>(hkx_attribute->m_value))
            {
                // This shouldn't happen really, maya doesn't support animated strings
                const char* currentString = animatedString->m_strings.back();
                MString aString;
                maya_attribute.getValue( aString );
                if (hkString::strCmp(aString.asChar(), currentString)!=0) // different value
                {
                    animatedString->m_strings.expandOne() = aString.asChar();
                    animatedString->m_times.expandOne() = (float) curTime.as(MTime::kSeconds);
                }
                continue;
            }

            if (hkxAnimatedMatrix* animatedMatrix = hkDynCast<hkxAnimatedMatrix>(hkx_attribute->m_value))
            {
                // This shouldn't happen really, maya doesn't support animated matrices
                MObject anObject;
                maya_attribute.getValue(anObject);
                MFnMatrixData matrixData(anObject);
                MMatrix aMatrix = matrixData.matrix();
                hkMatrix4 hkm4;
                hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( aMatrix, hkm4 );
                hkm4.get4x4ColumnMajor(animatedMatrix->m_matrices.expandBy(16));
                continue;
            }

            HK_ASSERT_NO_MSG(0x0768cc12, "Unhandled animated hkx type");
        }

        curTime += tInc;
    }
}


MStatus hctMayaSceneExporter::recurseAttributeGroups( hkArray<AttributeGroup>& attributeGroups, const MObject& currentMayaNode, bool recurse )
{
    MStatus status = MStatus::kSuccess;
    MFnDependencyNode nodeFn( currentMayaNode );

    MString typeStr = nodeFn.typeName();
    const char* type = typeStr.asChar();

    //
    // Step through the current Maya node attributes
    //
    {
        int attributeCount = nodeFn.attributeCount();
        bool exportAllAttributes = false; // It becomes true as soon as we find an hkType attribute

        AttributeGroup* currentAttributeGroup = HK_NULL;

        if (m_attributeSelection.filterAttribute(type, HK_NULL, HK_NULL)!=hctAttributeSelectionUtil::USER_REMOVE)
        {
            // Create a new group
            currentAttributeGroup = attributeGroups.expandBy(1);

            // Store the group name
            currentAttributeGroup->m_name = type;
        }

        for ( int i=0; i<attributeCount; ++i )
        {
            MObject attrObject = nodeFn.attribute(i);
            MFnAttribute attribute( attrObject, &status );
            if ( !status )
            {
                status.perror( "Can't access attribute" );
                continue;
            }

            // Get attribute components
            MString attrNameMStr = attribute.name();
            const char* attrName = attrNameMStr.asChar();
            hkStringOld attributeName( attrName );

            MPlug attributePlug( nodeFn.object(), attribute.object() );

            // Ignore sub-attributes
            if( attributePlug.isChild() )
            {
                continue;
            }

            // Attributes named 'hkType___' create a new group
            if ( attributeName.asLowerCase().beginsWith("hktype") )
            {
                // Create a new group
                currentAttributeGroup = attributeGroups.expandBy(1);

                // Store the group name
                MString newTypeName;
                attributePlug.getValue( newTypeName );
                currentAttributeGroup->m_name = hkStringOld( newTypeName.asChar(), newTypeName.length() );

                exportAllAttributes = true;

                continue;
            }

            // Skip if attribute is hidden
            if( attribute.isHidden() )
            {
                continue;
            }

            // We are not exporting yet
            if( !currentAttributeGroup )
            {
                continue;
            }

            hctAttributeSelectionUtil::UserAction userAction = m_attributeSelection.filterAttribute(type, HK_NULL, attrName);

            if (userAction == hctAttributeSelectionUtil::USER_REMOVE)
            {
                continue;
            }

            if ( (userAction == hctAttributeSelectionUtil::USER_UNDEFINED) && !exportAllAttributes)
            {
                continue;
            }

            hkxAttribute hkxAttr;
            // Skip if creation of the HKX attribute fails
            if( !createAttribute( currentMayaNode, attrObject, hkxAttr ) )
            {
                continue;
            }

            // Store the hkx attribute and the temporary maya plug
            currentAttributeGroup->m_plugs.append( attributePlug );
            currentAttributeGroup->m_attributes.pushBack( hkxAttr );
        }
    }

    // Stop if no more recursion
    if( !recurse )
    {
        return status;
    }

    // Add attribute groups of direct non-transform children. no further recursion.
    {
        const MDagPath path = MDagPath::getAPathTo( currentMayaNode );
        for( unsigned int i=0; i<path.childCount(); ++i )
        {
            const MObject& child = path.child(i);
            if( !child.hasFn( MFn::kTransform ) )
            {
                recurseAttributeGroups( attributeGroups, child, false );
            }
        }
    }

    // Add attribute groups of children connected via outgoing 'message'->'hkMessage'
    {
        MPlugArray plugArray;
        MPlug messagePlug = nodeFn.findPlug( "message" );
        if ( messagePlug.connectedTo( plugArray, true, true, &status ) )
        {
            for ( unsigned int i=0; i<plugArray.length(); ++i )
            {
                if( plugArray[i].partialName(false, false, false, false, false, true, &status) == "hkMessage" )
                {
                    // Recurse this function with the connected node
                    recurseAttributeGroups( attributeGroups, plugArray[i].node(), false );
                }
            }
        }
    }

    return status;
}


MStatus hctMayaSceneExporter::addAttributeGroups( hkxAttributeHolder* hkx_attributeHolder, const MObject& maya_object )
{
    MStatus status;

    // Initialise maya groups
    hkArray<AttributeGroup> attributeGroups;

    // Collect any grouped/connected havok attributes
    status = recurseAttributeGroups( attributeGroups, maya_object );
    if (status != MStatus::kSuccess) return status;

    // Prune empty groups
    {
        for( int i = attributeGroups.getSize()-1; i>=0; --i )
        {
            if( attributeGroups[i].m_attributes.getSize() == 0 )
            {
                attributeGroups.removeAtAndCopy(i);
            }
        }
    }

    //
    // Construct the hkx attribute groups
    //

    if( attributeGroups.getSize() > 0 )
    {
        hkx_attributeHolder->m_attributeGroups.setSize(attributeGroups.getSize());

        for( int i=0; i<hkx_attributeHolder->m_attributeGroups.getSize(); ++i )
        {
            const AttributeGroup& attrGroup = attributeGroups[i];
            hkxAttributeGroup& hkxGroup = hkx_attributeHolder->m_attributeGroups[i];

            hkxGroup.m_name = attrGroup.m_name.cString();

            hkxGroup.m_attributes.setSize(attrGroup.m_attributes.getSize());

            for( int j=0; j<attrGroup.m_attributes.getSize(); ++j )
            {
                hkxGroup.m_attributes[j] = attrGroup.m_attributes[j];

                // Store animated attributes to be sampled later
                const MPlug& plug = attrGroup.m_plugs[j];
                if( _isAttributeAnimated( plug ) )
                {
                    addAnimatedAttributeToSample( plug, &hkxGroup.m_attributes[j] );
                }
            }
        }
    }

    return status;
}


void hctMayaSceneExporter::postProcessAttributes( hkxNode* hkx_node )
{
    if( !hkx_node )
    {
        return;
    }

    // Process this node
    m_attributeDatabase.processAttributes(hkx_node);

    // Process children
    {
        int numChildren = hkx_node->m_children.getSize();
        for( int i=0; i<numChildren; ++i )
        {
            postProcessAttributes( hkx_node->m_children[i] );
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
