// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <ContentTools/Maya/MayaSceneExport/Exporter/MeshWriter/hctMayaMeshWriter.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>

// The exporter - called by the export command but independent of it
class hctMayaSceneExporter
{
public:

    // Export options
    struct Options
    {
        bool        m_batchMode;
        bool        m_exportMeshes;
        bool        m_exportMaterials;
        bool        m_exportAttributes;
        bool        m_exportAnnotations;
        bool        m_exportLights;
        bool        m_exportCameras;
        bool        m_exportSplines;
        bool        m_exportVertexTangents;
        bool        m_exportVertexAnimations;
        hkArray<MTime> m_forceVertexAnimationSamplePoints;
        bool        m_visibleOnly;
        bool        m_selectedOnly;
        bool        m_expandSelectionToChildren;
        bool        m_expandSelectionToSkeleton;
        bool        m_autoSkinAttachments;
        MTime       m_startTime;
        MTime       m_endTime;
        MString     m_hkoFile;
        MString     m_environmentVariables;
        MString     m_configurationSet;
        bool        m_useRotatePivot;
        bool        m_doNotSplitVertices;
        bool        m_storeKeyframeSamplePoints;
        int         m_runConfig; // -1 for any / all

        Options();
    };

    hctMayaSceneExporter( const Options& options );
    ~hctMayaSceneExporter();

    // Do the export (full filter pipeline etc), will cleanup after itself
    MStatus doExport();

    // Do the export, but don't run the filter pipline, just return the raw base export data. Call cleanupScene to cleanup when done.
    MStatus createScene();
    hkRootLevelContainer* getCurrentRootContainer() { return m_currentRootContainer; }
    void cleanupScene();

    // Get the file dependencies
    MStatus getFileDependencies(hkArray<hkStringPtr>& fileDependencies);


    // The plugin object
    MString m_pluginPath;

    // Unique uvChooser Id base
    int m_uniqueUvChooserIdBase;

protected:

    // The mesh writer needs access to the resizable scene
    friend class hctMayaMeshWriter;

    // Structure which represents the root level container
    // class for scene data, but that has mem managed arrays
    // so that we can store the data easier.
    struct ResizeableScene
    {
        hkRefPtr<hkxNode> m_rootNode;
        hkArray< hkRefPtr<hkxCamera> > m_cameras;
        hkArray< hkRefPtr<hkxLight> > m_lights;
        hkArray< hkRefPtr<hkxMesh> > m_meshes;
        hkArray< hkRefPtr<hkxSpline> > m_splines;
        hkArray< hkRefPtr<hkxMaterial> > m_materials;
        hkArray< hkRefPtr<hkxTextureFile> > m_externalTextures;
        hkArray< hkRefPtr<hkxSkinBinding> > m_skinBindings;
        hkArray< hkRefPtr<hkxNodeSelectionSet> > m_selectionSets;

        inline void clear()
        {
            m_rootNode = HK_NULL;
            m_cameras.setSize(0);
            m_lights.setSize(0);
            m_meshes.setSize(0);
            m_splines.setSize(0);
            m_materials.setSize(0);
            m_externalTextures.setSize(0);
            m_skinBindings.setSize(0);
            m_selectionSets.setSize(0);
        }
    };

    // Structure to maintain exportable nodes
    struct ExportMapping
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, ExportMapping );
        ExportMapping() : m_parentNodeIndex(-1), m_hasBonesInParentTree(false), m_hkxNode(HK_NULL) {}
        ExportMapping( const ExportMapping& other ) { *this = other; }

        MDagPath            m_node;             // The Maya node
        int                   m_parentNodeIndex;    // Its parent ExportMapping index
        hkBool              m_animatable;       // Does it support MFnTransform?
    hkBool        m_hasBonesInParentTree;  // for forced rigid skinned attachments
        hkArray<hkMatrix4>  m_keyFrames;        // Its animation keys
        hkRefPtr<hkxNode>   m_hkxNode;          // The HKX node
    };

    // Structure to maintain exportable selection sets
    struct ExportedSet
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, ExportedSet );

        ExportedSet() {}
        ExportedSet( const ExportedSet& other ) { m_name = other.m_name; setSetMembers( other.m_setMembers); }

        const MDagPathArray& getSetMembers() const
        {
            return m_setMembers;
        }

        void setSetMembers( const MDagPathArray& source )
        {
            // EXP-1401 - In versions <= Maya 7.0 there is no assignment operator for MDagPathArray so we need to manually copy.
#if MAYA_API_VERSION >= 800
            m_setMembers = source;
#else
            m_setMembers.setLength( source.length() );
            for (unsigned int i = 0; i<source.length(); ++i)
            {
                m_setMembers[i] = source[i];
            }
#endif
        }

        MString               m_name;                // Name of the set
    protected:
        MDagPathArray         m_setMembers;          // Array of DAG paths of the set's members
    };

    // Structure to maintain attribute groups
    struct AttributeGroup
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, AttributeGroup );
        AttributeGroup() {}
        AttributeGroup( const AttributeGroup& other ) { *this = other; }

        hkStringOld             m_name;         // The name of the attribute group
        MPlugArray              m_plugs;        // Plugs to each of the Maya attributes
        hkArray<hkxAttribute>   m_attributes;   // The corresponding HKX attributes
    };


    // Filter utilities
    bool loadFilterManager();
    bool loadFilterSet();
    void saveFilterSet();

    // Export steps
    bool    collectExportNodesRecursive( const MDagPath& node, bool parentTreeHasBones, bool ancestorWasSelected, MSelectionList& selection );
    bool    createGenericObjectsFromMayaNodes(class hkResourceContainer* registeredClassNodes);

        /// convert a single MObject into a Havok class and add it to the resourceContainer
    void    convertMayaNode(MObject& mayaNode, hkResourceContainer* resourceContainer, hkArray<MString>& parentNames ) const;

    void    cleanupRegisteredClassNodes();


    const hkReflect::Type* getClassOfMayaNode( const MObject& mayaObject, int* attributeIndexOfFirstHavokAttributeOut ) const;
    static hkReflect::Var HK_CALL createEmptyHavokObject( const hkReflect::Type* klass );

    void    getDataFromMayaNode(const MObject& mayaObject, int firstAttributeIndex, hkResourceHandle* resourceHandle, const char* parentName ) const;
    void    collectNodesRecursively(MObject& node, hkArray<MObject>& nodesOut) const;
    void    precomputeAnimation();
    MStatus createHkxNodes();
    static void HK_CALL assignHkxSceneUUIDs(hkxScene* scene);
    static void HK_CALL assignHkxNodeUUID(hkxNode* node, hkArray<char>& memBuffer, hkMap<hkUuid, int>& knownUUIDs);
    void    verifyScene();

    // Node type processing
    MStatus     processMesh( const MDagPath& dagPath, hkRefVariant& variant, bool forceSkinned );
    MStatus     processNurbs ( const MDagPath& dagPath, hkRefVariant& variant, bool forceSkinned );
    hkxLight*   processLight( const MDagPath& dagPath, MStatus& status );
    hkxCamera*  processCamera( const MDagPath& dagPath, MStatus& status );
    hkxSpline*  processSpline( const MDagPath& dagPath, MStatus& status );

    // Attribute processing
    bool    createAttribute( MObject nodeObject, MObject attrObject, hkxAttribute& hkx_attribute ) const;
    void    sampleAttributes();
    void    addAnimatedAttributeToSample (const MPlug& mayaAttr, hkxAttribute* havokAttr);
    MStatus recurseAttributeGroups( hkArray<AttributeGroup>& attributeGroups, const MObject& currentMayaNode, bool recurse = true );
    MStatus addAttributeGroups( hkxAttributeHolder* hkx_attributeHolder, const MObject& maya_object );
    void    postProcessAttributes( hkxNode* hkx_node );

    // Node selection set processing
    void collectExportNodeSelectionSets();
    void createHkxNodeSelectionSet( const ExportedSet& exportedSet );

    // Annotation processing
    MStatus addAnnotations( hkxNode* hkx_node, const MDagPath& maya_node );

    // Utilities
    MStatus getLocalPivotTransform( const MDagPath& path, MMatrix& out );
    MStatus findBindPoses( MFnSkinCluster& skin, hkArray<MDagPath>& dagPaths, hkArray<MMatrix>& bindPoses );
  MStatus findBindPosesGeneric( MFnGeometryFilter& skin, hkArray<MDagPath>& dagPaths, hkArray<MMatrix>& bindPoses );
    MDagPath getMayaNode( const hkxNode* node );
    const hkxNode* getHkxNode( const MDagPath& path ) const;


    const Options&          m_options;              // The options passed on construction
    int                     m_configurationSetIndex;
    hkRootLevelContainer*   m_currentRootContainer;
    hctMayaMeshWriter       m_meshWriter;           // To handle mesh objects, materials, textures, etc.
    hctAttributeProcessingUtil m_attributeDatabase; // Lookup for attributes so that they can be given proper type info etc
    hctFilterProcessingUtil m_filters;              // The filter set we want to use
    hctMayaSceneExportUtilities::ProgressWindow m_progressWindow; // Progress bar

    // Utility to select attributes based on user-defined settings (EXP-868/EXP-914)
    hctAttributeSelectionUtil m_attributeSelection;

    hkArray< ExportMapping >    m_exportMappings;   // Exportable nodes with associated data
    hkArray< ExportedSet >  m_exportedSets;     // Exportable Maya selection sets
    ResizeableScene                 m_currentScene;     // Utility structure to allow handy resizing before final writing out

    // EXP-862
    // We keep a map from maya shapes exported to havok meshes exported.
    // This doesn't include skins (we don't support skin instancing).
    // We also use the pivot translation as meshes will be modified based on that
    struct ExportedMeshInstanceData
    {
        MObject m_mayaShape;
        MPoint m_pivotOffset;
        hkxMesh* m_havokMesh;
    };
    hkArray<ExportedMeshInstanceData> m_exportedMeshesInstanceData;
    hkxMesh* lookForAlreadyExportedMesh (const MDagPath& shapeDagPath);
    void registerExportedMesh (const MDagPath& shapeDagPath, hkxMesh* mesh);


    private:

        MPlugArray m_mayaAnimAttributes;
        hkArray<hkxAttribute*> m_havokAnimAttributes;
};

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
