// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

#include <ContentTools/Common/SdkUtils/ClassHierarchy/hctClassHierarchyUtil.h>

#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>
#include <ContentTools/Common/SceneExport/Error/hctSceneExportError.h>
#include <ContentTools/Common/SceneExport/Utils/hctSceneExportUtils.h>
#include <ContentTools/Common/Filters/Common/Utils/hctFilterUtils.h>

#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Types/Uuid/hkUuidMapOperations.cxx>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.maya"
#include <Common/Base/System/Log/hkLog.hxx>


#ifdef HAVOK_VISION_EXPORTER
namespace MayaVision {
  void GetCustomVisionData(const MObject& o, hkStringPtr& outStr);
}
#endif



// Default options
hctMayaSceneExporter::Options::Options()
:   m_batchMode( false ),
    m_exportMeshes( true ),
    m_exportMaterials( true ),
    m_exportAttributes( true ),
    m_exportAnnotations( true ),
    m_exportLights( true ),
    m_exportCameras( true ),
    m_exportSplines( true ),
    m_exportVertexTangents( true ), // default to true now as usually wanted
    m_exportVertexAnimations( false ),
    m_visibleOnly( false ),
    m_selectedOnly( false ),
    m_expandSelectionToChildren( false ), // Vision export treats all selected parents as implictly selecting children too (prefab support requires it for group nodes etc)
    m_expandSelectionToSkeleton( false ), // Backwards compatible with Havok
    m_autoSkinAttachments(false), // Vision requires this as true for attachments
    m_startTime( MAnimControl::minTime() ),
    m_endTime( MAnimControl::maxTime() ),
    m_hkoFile( "" ),
    m_environmentVariables(""),
    m_useRotatePivot( true ),
    m_doNotSplitVertices (false),
    m_storeKeyframeSamplePoints(true),
    m_runConfig(-1)
{
}



hctMayaSceneExporter::hctMayaSceneExporter( const Options& options )
:   m_options(options)
,   m_configurationSetIndex(-1)
,   m_currentRootContainer(HK_NULL)
,   m_uniqueUvChooserIdBase(0)
{
    m_meshWriter.setOwner( this );

    MStatus status;
    M3dView curView = M3dView::active3dView( &status );
    if( status == MStatus::kSuccess )
    {
        m_filters.setOwnerHandle( (HWND)( curView.applicationShell( &status ) ) );
    }
}

hctMayaSceneExporter::~hctMayaSceneExporter()
{
}


MStatus hctMayaSceneExporter::doExport()
{
    MStatus status = MStatus::kSuccess;

    HK_REPORT_SECTION_BEGIN(0xabba934, "Maya Scene Export");

    // EXP-969
    if ( m_options.m_selectedOnly )
    {
        Log_Info( "*Exporting Selected Only*" );
    }
    if ( m_options.m_visibleOnly )
    {
        Log_Info( "*Exporting Visible Only*" );
    }

    // Initialize, perform checks
    {
        // Ensure valid times
        if( m_options.m_endTime < m_options.m_startTime )
        {
            HK_ERROR(0x71073f74, "End time is earlier than start time. Can't continue." );
            return MStatus::kFailure;
        }

#ifndef HCT_NO_FILTERS
        // Must load the filter manager DLL
        if( !loadFilterManager() )
        {
            HK_ERROR(0x3ffcba7, "Couldn't load the Havok filter manager" );
            return MStatus::kFailure;
        }

        // Try to load any specified HKO file
        if( !loadFilterSet() )
        {
            m_filters.unload();
            // A warning will already be given if there was a problem
            return MStatus::kFailure;
        }
#endif
    }

    status = createScene();

    HK_ASSERT_NO_MSG(0x62e01de, m_currentRootContainer);

    HK_REPORT_SECTION_END();

#ifndef HCT_NO_FILTERS
    // Do the filtering
    if (status==MStatus::kSuccess) // EXP-1651
    {
        // Make sure that the Maya error is set to current before export, else this is not a safe cast
        m_filters.mergeErrors( (hctSceneExportError*) &hkError::getInstance() );

        // m_runConfig is either 0 (when quickConfigModeFlag is set) or -1 so in the latter case use the configuration set index
        const int configToRun = (m_options.m_runConfig == 0) ? 0 : m_configurationSetIndex;

        if ((m_options.m_batchMode) || (configToRun >= 0))
        {
            hkBool saveConfig;
            m_filters.processBatch(*m_currentRootContainer, saveConfig, configToRun, !m_options.m_batchMode);
            if( saveConfig )
            {
                saveFilterSet(); //EXP-1377 (we may have upgraded them)
            }
            Log_Info( "Havok filter set processed" );
        }
        else
        {
            hkBool saveConfig = true;
            m_filters.openFilterManager( *m_currentRootContainer, saveConfig );
            if( saveConfig )
            {
                saveFilterSet();
            }
        }
    }
#else

    // Asset filename
    MString assetFilePathM;
    MString assetFileNameM;
    status = MGlobal::executeCommand( "file -q -sceneName", assetFilePathM );
    status = MGlobal::executeCommand( "file -q -sceneName -shortName", assetFileNameM );

    hkStringBuf assetFilePath;
    hkStringBuf assetFileName;
    wchar_t filenameBuffer[1024];
    if( (assetFilePathM.length() == 0) || (assetFileNameM.length() == 0) )
    {
        assetFileName = "untitled.mb";
        _wgetcwd(filenameBuffer, 1024);
        hkUtf8::Utf8FromWide wf(filenameBuffer);
        assetFilePath.setJoin(wf.cString(), "\\", assetFileName.cString() );
    }
    else
    {
        assetFilePath = assetFilePathM.asUTF8();
        assetFileName = assetFileNameM.asUTF8();
    }

    bool saveFile = true;
    if (!m_options.m_batchMode)
    {
        assetFilePath.replace(".ma",".hkt");
        assetFileName.replace(".ma",".hkt");
        assetFilePath.replace(".mb",".hkt");
        assetFileName.replace(".mb",".hkt");
        if (!assetFilePath.endsWithCase(".hkt"))
        {
            assetFilePath.append(".hkt");
            assetFileName.append(".hkt");
        }

        OPENFILENAMEW op;
        hkString::memSet( &op, 0, sizeof(OPENFILENAMEW) );
        op.lStructSize = sizeof(OPENFILENAMEW);
        M3dView curView = M3dView::active3dView( &status );
        if( status == MStatus::kSuccess )
        {
            op.hwndOwner = (HWND)( curView.applicationShell( &status ) );
        }
        op.lpstrFilter = L"Havok Files (*.hkx;*.hkt;*.xml)\0*.hkx;*.hkt;*.xml;\0Binary Packfiles (*.hkx)\0*.hkx\0Tagfiles (*.hkt)\0*.hkt\0XML Packfiles (*.xml)\0*.xml\0All Files\0*.*\0\0";
        op.lpstrDefExt = L"hkt";
        op.nFilterIndex = 3;

        hkUtf8::WideFromUtf8 assetFileNameWide(assetFileName.cString());
        hkUtf8::WideFromUtf8 assetFilePathWide(assetFilePath.cString());

        op.lpstrInitialDir = assetFilePathWide.cString();
        wcscpy_s( filenameBuffer, assetFileNameWide.cString() );

        op.lpstrFile = filenameBuffer;

        op.nMaxFile = 1023;
        op.Flags = OFN_ENABLESIZING | OFN_EXPLORER | OFN_NOREADONLYRETURN | OFN_PATHMUSTEXIST;
        if ( GetSaveFileNameW(&op) )
        {
            hkUtf8::Utf8FromWide utf8File( op.lpstrFile );
            assetFilePath = utf8File.cString();
        }
        else
        {
            saveFile = false;
        }
    }
    else
    {
        //XX assume tools batch mode. ADD SCRIPT OPTION FOR FILE NAME
        assetFilePath.append(".hktAuto"); // so fullname.mb.hktAuto  so to not conflict with max ver etc of twp of same asset etc
    }

    if (saveFile)
    {
        assetFilePath.replace( '/', '\\' );
        hkSerializeUtil::save(m_currentRootContainer, assetFilePath.cString());
    }

#endif



    // Cleanup
    cleanupScene();

    // Release the filter dll
    m_filters.unload();

    return MStatus::kSuccess;
}

MStatus hctMayaSceneExporter::getFileDependencies(hkArray<hkStringPtr>& fileDependencies)
{
    MStatus status = MStatus::kSuccess;

    // Initialize, perform checks
    {
        // Ensure valid times
        if (m_options.m_endTime < m_options.m_startTime)
        {
            HK_ERROR(0x71073f74, "End time is earlier than start time. Can't continue.");
            return MStatus::kFailure;
        }

#ifndef HCT_NO_FILTERS
        // Must load the filter manager DLL
        if (!loadFilterManager())
        {
            HK_ERROR(0x3ffcba7, "Couldn't load the Havok filter manager");
            return MStatus::kFailure;
        }

        // Try to load any specified HKO file
        if (!loadFilterSet())
        {
            m_filters.unload();
            // A warning will already be given if there was a problem
            return MStatus::kFailure;
        }
#endif
    }

    // Create an empty scene with environment variables set up instead of createScene()
    {
        // Dirty all plugs to be sure that the scene is read correctly (EXP-381)
        MGlobal::executeCommand("dgdirty -a");

        // Clear lists
        m_currentScene.clear();
        m_exportMappings.clear();
        m_exportedSets.clear();
        m_progressWindow.setStatus("Initializing...");

        // Don't need the progress window any more
        m_progressWindow.endProgress();

        // Create the top level structures to process
        hkxEnvironment* envPtr = new hkxEnvironment();
        hkxEnvironment& environment = *envPtr;

        hkxScene* scenePtr = new hkxScene();
        hkxScene& scene = *scenePtr;

        // We are using this container to hold all registered classes. We use this because it
        // conveniently wraps an array of Named Variants, which is what we need. Rather than recreate
        // the functionality in a new class, we simply re-use hkRootLevelContainer.
        hkMemoryResourceContainer* resourceContainerPtr = new hkMemoryResourceContainer();
        hkMemoryResourceContainer& resourceContainer = *resourceContainerPtr;

        // Fill the root level container
        m_currentRootContainer = new hkRootLevelContainer();
        m_currentRootContainer->m_namedVariants.setSize(3);

        m_currentRootContainer->m_namedVariants[0].set("Environment Data", &environment);
        m_currentRootContainer->m_namedVariants[1].set("Scene Data", &scene);
        m_currentRootContainer->m_namedVariants[2].set("Resource Data", &resourceContainer);
        envPtr->removeReference();
        scenePtr->removeReference();
        resourceContainerPtr->removeReference();

        {
            // Modeller name and version
            hkStringBuf modellerString;
            modellerString.setJoin("Maya ", MGlobal::mayaVersion().asChar());
            scene.m_modeller = modellerString.cString();

            // Asset filename
            hkStringBuf asset("Untitled");
            {
                MString assetFilePath;
                status = MGlobal::executeCommand("file -q -sceneName", assetFilePath);
                if (status == MStatus::kSuccess && assetFilePath.length() > 0)
                {
                    asset = assetFilePath.asChar();
                    asset.replace('/', '\\');
                }
            }
            scene.m_asset = asset.cString();
        }

        // Fill the environment data
        {
            // Take data from the scene
            hkxSceneUtils::fillEnvironmentFromScene(scene, environment);
        }
    }

    HK_ASSERT_NO_MSG(0x62e01de, m_currentRootContainer);

#ifndef HCT_NO_FILTERS
    // Do the filtering
    if (status == MStatus::kSuccess) // EXP-1651
    {
        // Make sure that the Maya error is set to current before export, else this is not a safe cast
        m_filters.mergeErrors((hctSceneExportError*)&hkError::getInstance());

        // m_runConfig is either 0 (when quickConfigModeFlag is set) or -1 so in the latter case use the configuration set index
        const int configToRun = (m_options.m_runConfig == 0) ? 0 : m_configurationSetIndex;

        if ((m_options.m_batchMode) || (configToRun >= 0))
        {
            m_filters.getFileDependencies(*m_currentRootContainer, fileDependencies, configToRun);

            Log_Info("Havok filter set processed");
        }
    }
#endif

    // Cleanup
    cleanupScene();

    // Release the filter dll
    m_filters.unload();

    return MStatus::kSuccess;
}

MStatus hctMayaSceneExporter::createScene()
{
    MStatus status = MStatus::kSuccess;

    // Dirty all plugs to be sure that the scene is read correctly (EXP-381)
    MGlobal::executeCommand( "dgdirty -a" );

    // Clear lists
    m_currentScene.clear();
    m_exportMappings.clear();
    m_exportedSets.clear();
    m_progressWindow.setStatus( "Initializing..." );

    // Build the lists of export nodes
    MItDag rootIt( MItDag::kDepthFirst, MFn::kInvalid, &status );
    MDagPath rootPath;
    status = rootIt.getPath( rootPath );
    if( status == MStatus::kSuccess )
    {
        MSelectionList selection;
        MGlobal::getActiveSelectionList( selection );

        collectExportNodesRecursive( rootPath, false, false, selection );
        collectExportNodeSelectionSets();
    }

    // Make sure we have something to export
    if( m_exportMappings.getSize() == 0 )
    {
        HK_ERROR(0x25deada5, "Nothing to export" );
        m_progressWindow.endProgress();
        return MStatus::kFailure;
    }

    // Extract all the keyframes for the transform nodes
    precomputeAnimation();
    if( m_progressWindow.isCancelled() )
    {
        HK_WARN_ALWAYS(0xabba3a23, "Havok scene export cancelled" );
        m_progressWindow.endProgress();
        return MStatus::kFailure;
    }

    // Export the scene
    status = createHkxNodes();
    if( (status != MStatus::kSuccess) || m_progressWindow.isCancelled() )
    {
        if (status != MStatus::kSuccess )
        {
            HK_WARN_ALWAYS(0xabbab328, "Havok scene export failed in createHkxNodes" );
        }
        else
        {
            HK_WARN_ALWAYS(0xabbab328, "Havok scene export cancelled" );
        }
        m_progressWindow.endProgress();
        return MStatus::kFailure;
    }

    // Verify any skin mappings etc.
    verifyScene();

    // Don't need the progress window any more
    m_progressWindow.endProgress();

    // Create the top level structures to process
    hkxEnvironment* envPtr = new hkxEnvironment();
    hkxEnvironment& environment = *envPtr;

    hkxScene* scenePtr = new hkxScene();
    hkxScene& scene = *scenePtr;

    // We are using this container to hold all registered classes. We use this because it
    // conveniently wraps an array of Named Variants, which is what we need. Rather than recreate
    // the functionality in a new class, we simply re-use hkRootLevelContainer.
    hkMemoryResourceContainer* resourceContainerPtr = new hkMemoryResourceContainer();
    hkMemoryResourceContainer& resourceContainer = *resourceContainerPtr;

    // Fill the root level container
    m_currentRootContainer = new hkRootLevelContainer();
    m_currentRootContainer->m_namedVariants.setSize(3);

    m_currentRootContainer->m_namedVariants[0].set("Environment Data", &environment);
    m_currentRootContainer->m_namedVariants[1].set("Scene Data", &scene);
    m_currentRootContainer->m_namedVariants[2].set("Resource Data", &resourceContainer);
    envPtr->removeReference();
    scenePtr->removeReference();
    resourceContainerPtr->removeReference();

    {
        // Modeller name and version
        hkStringBuf modellerString;
        modellerString.setJoin( "Maya ", MGlobal::mayaVersion().asChar() );
        scene.m_modeller = modellerString.cString();

        // Asset filename
        hkStringBuf asset( "Untitled" );
        {
            MString assetFilePath;
            status = MGlobal::executeCommand( "file -q -sceneName", assetFilePath );
            if( status == MStatus::kSuccess && assetFilePath.length() > 0 )
            {
                asset = assetFilePath.asChar();
                asset.replace( '/', '\\' );
            }
        }
        scene.m_asset = asset.cString();

        // Scene length
        MTime animRange = m_options.m_endTime - m_options.m_startTime;
        scene.m_sceneLength = (hkReal)( animRange.as( MTime::kSeconds ) );
        scene.m_numFrames = (hkUint32)( animRange.as( MTime::uiUnit() ) );

        // Root node
        scene.m_rootNode = m_currentScene.m_rootNode;

        // Node selection sets
        if (m_currentScene.m_selectionSets.getSize() > 0)
        {
            scene.m_selectionSets.append( m_currentScene.m_selectionSets.begin(), m_currentScene.m_selectionSets.getSize() );
        }

        // Cameras
        if( m_options.m_exportCameras && m_currentScene.m_cameras.getSize() > 0 )
        {
            scene.m_cameras.append(m_currentScene.m_cameras.begin(), m_currentScene.m_cameras.getSize());
        }

        // Lights
        if( m_options.m_exportLights && m_currentScene.m_lights.getSize() > 0 )
        {
            scene.m_lights.append(m_currentScene.m_lights.begin(), m_currentScene.m_lights.getSize());
        }

        // Splines
        if (m_options.m_exportSplines && m_currentScene.m_splines.getSize() > 0)
        {
            scene.m_splines = m_currentScene.m_splines;
        }

        // If meshes aren't exported there'll be no skin bindings either
        if( m_options.m_exportMeshes )
        {
            // Meshes
            if( m_currentScene.m_meshes.getSize() > 0 )
            {
                scene.m_meshes.append(m_currentScene.m_meshes.begin(), m_currentScene.m_meshes.getSize());
            }


            // Skin bindings
            if( m_currentScene.m_skinBindings.getSize() > 0 )
            {
                scene.m_skinBindings.append(m_currentScene.m_skinBindings.begin(), m_currentScene.m_skinBindings.getSize());
            }

        }

        // If materials aren't exported textures won't be either.
        if( m_options.m_exportMaterials )
        {
            // Materials
            if( m_currentScene.m_materials.getSize() > 0 )
            {
                scene.m_materials.append(m_currentScene.m_materials.begin(), m_currentScene.m_materials.getSize());
            }

            // External textures
            if( m_currentScene.m_externalTextures.getSize() > 0 )
            {
                scene.m_externalTextures.append(m_currentScene.m_externalTextures.begin(), m_currentScene.m_externalTextures.getSize());
            }
        }
    }

    // Assign node UUIDs
    assignHkxSceneUUIDs(&scene);

    // Fill in the registered classes data.
    {
        createGenericObjectsFromMayaNodes(&resourceContainer);
    }

    // Fill the environment data
    {
        // Take data from the scene
        hkxSceneUtils::fillEnvironmentFromScene(scene, environment);

        // And then the user environment vars
        const char* envStr = m_options.m_environmentVariables.asChar();
        environment.interpretString(envStr); // problems will be reported
    }


    // EXP-867 : Reorder nodes by name
    {
        // If the user defined "do_not_reorder_nodes", we don't do so
        if (environment.getVariableValue("do_not_reorder_nodes")==HK_NULL

            // check the Windows environment variables also (EXP-1418)
            && ::getenv("do_not_reorder_nodes")==HK_NULL)
        {
            hkxSceneUtils::reorderNodesAlphabetically(scene);
        }
    }

    // EXP-969
    hkSceneExportUtils::reportSceneData(&scene);

    return status;
}

void hctMayaSceneExporter::assignHkxSceneUUIDs(hkxScene* scene)
{
    hkArray<char> memBuffer;
    hkMap<hkUuid, int> knownUUIDs;

    hkxNode* rootNode = scene->m_rootNode;
    assignHkxNodeUUID(rootNode, memBuffer, knownUUIDs);
}

void hctMayaSceneExporter::assignHkxNodeUUID(hkxNode* node, hkArray<char>& memBuffer, hkMap<hkUuid, int>& knownUUIDs)
{
    if ( !node )
    {
        return;
    }

    // Assign UUIDs to all its children
    {
        hkArray< hkRefPtr<hkxNode> >& childNodes = node->m_children;

        for (int k = childNodes.getSize() - 1; k >= 0; k--)
        {
            hkxNode* childNode = childNodes[k];
            assignHkxNodeUUID(childNode, memBuffer, knownUUIDs);
        }
    }

    // Assign a UUID to this node
    {
        memBuffer.clear();

        // Get the contents of the node
        node->m_uuid = hkUuid::getNil();
        hkArrayStreamWriter memWriter(&memBuffer, hkArrayStreamWriter::ARRAY_BORROW);
        const hkResult ret = hkSerializeUtil::save<hkxNode>(node, &memWriter);

        if ( ret.isSuccess() )
        {
            // Got the contents, digest it into an UUID
            hkUuid uuid;
            int hashSize = 16;
            if ( hctSdkUtils::computeMd5Hash(memBuffer.begin(), memBuffer.getSize(), reinterpret_cast<hkUint8*>(&uuid), hashSize).isSuccess() )
            {
                HK_ASSERT_NO_MSG(0x38461719, hashSize == 16);

                // Make sure the UUID is unique, i.e. does not collide with another in this file
                bool hadCollisions = false;
                for (hkMap<hkUuid, int>::Iterator it = knownUUIDs.findKey(uuid); knownUUIDs.isValid(it); it = knownUUIDs.findKey(uuid))
                {
                    uuid.setRandom();
                    hadCollisions = true;
                }

                // Found an unique UID!
                node->m_uuid = uuid;
                knownUUIDs.insert(uuid, 1);
                if ( hadCollisions )
                {
                    HK_WARN_ALWAYS(0xabbacddc, "The UUID of node " << node->m_name.cString() << " generated from its contents collides with another node. It will be randomly generated.");
                }
            }
        }
        else
        {
            HK_WARN_ALWAYS(0xabbacddc, "Failed to serialize the contents of node " << node->m_name.cString() << ". Its UUID will be randomly generated");
        }
    }
}

void hctMayaSceneExporter::cleanupScene()
{

    m_currentScene.clear();
    m_exportMappings.clear();
    m_exportedSets.clear();

    if (m_currentRootContainer)
    {
        delete m_currentRootContainer;
    }
    m_currentRootContainer = HK_NULL;
}

// Recursively finds if any children are selected
bool _childrenAreSelected(const MDagPath& node)
{
    MStatus status;
    bool result = false;
    int numChild = node.childCount();

    {
        MSelectionList selection;
        MGlobal::getActiveSelectionList(selection);

        // Check to see if the selection list contains the MDagPath in question as selection.hasItem(...) doesn't seem to work
        for (unsigned int i = 0; i < selection.length(); i ++)
        {
            MDagPath path;
            selection.getDagPath(i, path);
            if (path == node)
            {
                result = true;
                break;
            }
        }
    }

    if (!result)
    {
        for (int ni = 0; ni < numChild; ni ++)
        {
            MDagPath childPath = node;
            childPath.push(node.child(ni));

            if (_childrenAreSelected(childPath))
            {
                result = true;
                break;
            }

            childPath.pop();
        }
    }

    return result;
}

//
//  Main processing
//

bool hctMayaSceneExporter::collectExportNodesRecursive( const MDagPath& node, bool parentTreeHasBones, bool ancestorWasSelected, MSelectionList& selection )
{
    // We only want to export root/transform/joint nodes
    {
        const MFn::Type nodeApiType = node.apiType();
        const bool shouldExport = (node.length() == 0) ||
            (nodeApiType == MFn::kPluginTransformNode) ||
            (nodeApiType == MFn::kTransform) ||
            (nodeApiType == MFn::kJoint) ||
            (nodeApiType == MFn::kLodGroup) ||
            (nodeApiType == MFn::kLookAt) ||
#if MAYA_API_VERSION >= 201100
            (nodeApiType == MFn::kDagContainer) ||
#endif
            (nodeApiType == MFn::kUnknownTransform);
        if( !shouldExport )
        {
            return false;
        }
    }

    bool isSelected = selection.hasItem( node );

    // Recurse over the children of this node
    MDagPathArray children;
    bool hasBones = parentTreeHasBones || ( node.node().hasFn( MFn::kJoint ) );

    {
        MDagPath childPath = node;
        unsigned nc = node.childCount();

        for( unsigned ci=0; ci < nc; ++ci )
        {
            childPath.push( node.child(ci) );
            if( collectExportNodesRecursive( childPath, hasBones, isSelected || ancestorWasSelected, selection ) )
            {
                children.append( childPath );
            }
            childPath.pop();
        }
    }

    // Optionally exclude unselected/hidden LEAF nodes
    if( children.length() == 0 )
    {
        bool skipAsParentSelected = ancestorWasSelected && m_options.m_expandSelectionToChildren;
        if( m_options.m_selectedOnly && !skipAsParentSelected && !isSelected )
        {
            if (m_options.m_expandSelectionToSkeleton && hasBones)
            {
                // Find root node of skeleton
                MDagPath nodePath;
                MDagPath::getAPathTo(node.node(), nodePath);

                MStatus status;
                MObject nodeObject = nodePath.node(&status);

                // Get the root node of the skeleton first
                while( nodePath.length() > 1 && status == MStatus::kSuccess)
                {
                    nodePath.pop();
                    nodeObject = nodePath.node(&status);
                }

                MDagPath rootPath;
                MDagPath::getAPathTo(nodeObject, rootPath);

                if (!_childrenAreSelected(rootPath))
                {
                    return false;
                }
            }
            else
            {
                return false;
            }
        }

        if( m_options.m_visibleOnly )
        {
            // Check both the transform and the relevant child object
            MDagPath childPath = node;
            {
                unsigned int numChildren = node.childCount();
                for( unsigned int i=0; i<numChildren; ++i )
                {
                    MFn::Type type = node.child(i).apiType();
                    if( type == MFn::kMesh || node.hasFn( MFn::kLight ) || type == MFn::kCamera )
                    {
                        childPath.push( node.child(i) );
                        break;
                    }
                }
            }

            MFnDagNode dagFn( node );
            MFnDagNode childDagFn( childPath );
            if( dagFn.isIntermediateObject() || childDagFn.isIntermediateObject() )
            {
                return false;
            }

            bool visibility;
            dagFn.findPlug( "visibility" ).getValue( visibility );
            if( !visibility )
            {
                return false;
            }
            dagFn.findPlug( "overrideVisibility" ).getValue( visibility );
            if( !visibility )
            {
                return false;
            }
            childDagFn.findPlug( "visibility" ).getValue( visibility );
            if( !visibility )
            {
                return false;
            }
            childDagFn.findPlug( "overrideVisibility" ).getValue( visibility );
            if( !visibility )
            {
                return false;
            }
        }
    }

    // Set the parent index of the children
    const int numChildren = children.length();
    if( numChildren > 0 )
    {
        int parentIndex = m_exportMappings.getSize();
        int childIndex = numChildren - 1;
        for( int i=parentIndex-1; i>=0; --i )
        {
            if( m_exportMappings[i].m_node == children[childIndex] )
            {
                m_exportMappings[i].m_parentNodeIndex = parentIndex;
                if( --childIndex < 0 ) break;
            }
        }
    }

    // This node will be exported
    ExportMapping* ExportMapping = m_exportMappings.expandBy(1);
    ExportMapping->m_node = node;
    ExportMapping->m_animatable = ( node.hasFn( MFn::kTransform ) );
    ExportMapping->m_hasBonesInParentTree = hasBones;
    return true;
}


void hctMayaSceneExporter::collectExportNodeSelectionSets()
{
    MStatus status;

    // Create an iterator to discover all sets in the scene
    MItDependencyNodes iter(MFn::kSet);
    MObjectArray setsFound;

    // The iterator may run over the same set twice, so make an array with one instance of each
    for( ; !iter.isDone(); iter.next() )
    {
#if MAYA_API_VERSION >= 700
        MObject obj = iter.thisNode();
#else
        MObject obj = iter.item();
#endif

        bool alreadyAdded = false;
        for (unsigned int i=0; i<setsFound.length(); ++i)
        {
            if ( setsFound[i] == obj )
            {
                alreadyAdded = true;
                break;
            }
        }
        if (!alreadyAdded) setsFound.append(obj);
    }

    // Store paths of the DAG nodes referred to by each set
    for (unsigned int i=0; i<setsFound.length(); ++i)
    {
        MObject& obj = setsFound[i];
        if (obj.apiType() != MFn::kSet) continue;
        MFnSet set(obj, &status);
        if (status != MStatus::kSuccess) continue;

        MSelectionList members;
        bool flatten = false;
        set.getMembers(members, flatten);

        // Look for DAG nodes in the set
        MDagPathArray pathArray;
        MDagPath dagPath;
        MObject component;
        for (unsigned int j=0; j<members.length(); ++j)
        {
            status = members.getDagPath(j, dagPath, component);
            if (status == MStatus::kSuccess)
            {
                pathArray.append(dagPath);
            }
        }

        // If some DAG nodes were found in this set, prepare it for export
        if (pathArray.length()>0)
        {
            ExportedSet* exportedSet = m_exportedSets.expandBy(1);
            exportedSet->m_name = set.name();
            exportedSet->setSetMembers( pathArray );
        }
    }
}


void hctMayaSceneExporter::precomputeAnimation()
{
    MStatus status = MStatus::kSuccess;

    // Is there anything to do?
    int numNodes = m_exportMappings.getSize();
    if( numNodes == 0 )
    {
        return;
    }

    // Save original time
    const MTime originalTime = MAnimControl::currentTime();

    // Set to start of animation time
    MTime curTime = m_options.m_startTime;
    MAnimControl::setCurrentTime( curTime );

    // Get the required time increment
    MTime::Unit timeUnit = MTime::uiUnit();
    int frames = (int)( (m_options.m_endTime - m_options.m_startTime).as( timeUnit ) ) + 1;
    MTime tInc( 1, timeUnit ); // one frame at a time

    // some variables
    int i, f;

    // To properly check if a track is static we must compare future frames against the previous
    // frame beginning with the Bind Pose and NOT frame zero. To do this we need to find the
    // skin cluster node and retrieve the bind pose information for each influence object.
    hkArray<MDagPath> dagPaths;
    hkArray<MMatrix> bindPoses;

    // find all the skin clusters in the scene and extract the bind poses for each influence object
    MItDependencyNodes iter( MFn::kGeometryFilt );
    for( ; !iter.isDone(); iter.next() )
    {
        // create a skin cluster function set
        MObject object = iter.item();
        MFnSkinCluster skin( object, &status );
        if( status == MStatus::kSuccess )
        {
            // extract the bind pose for each influence object
            status = findBindPoses( skin, dagPaths, bindPoses );
            if( status != MStatus::kSuccess )
            {
                status.perror( "Failed to extract bind poses correctly from MFnSkinCluster!" );
            }
        }
        else
    {
      MFnGeometryFilter rigidSkin( object, &status );
      if( status == MStatus::kSuccess )
      {
        status = findBindPosesGeneric( rigidSkin, dagPaths, bindPoses );
        if( status != MStatus::kSuccess )
        {
          status.perror( "Failed to extract bind poses correctly from MFnGeometryFilter!" );
        }
      }
      else
          {
              status.perror( "Failed to create skin cluster function set!" );
          }
    }
    }

    // Prepare the animation keys
    for( i = 0; i < numNodes; ++i )
    {
        int numKeys = ( m_exportMappings[i].m_animatable ) ? frames: 1;
        m_exportMappings[i].m_keyFrames.setSize( numKeys );
    }

    // checks for static tracks ( MAnimUtils::isAnimated does not take dynamics etc. into account )
    hkArray<hkBool> staticNodes( numNodes, true );

    // stores previous frame's matrix for each node
    hkArray<MMatrix> nodeMatrices( numNodes );
    hkArray<MMatrix> initialMatrices( numNodes );

    // we must initialise these matrices to be the bind pose for each animated node
    hkBool found = false;
    for( i = 0; i < numNodes; i++ )
    {
        found = false;
        for( f = 0; f < bindPoses.getSize(); ++f )
        {
            if( dagPaths[f] == m_exportMappings[i].m_node )
            {
                nodeMatrices[i] = bindPoses[f];
                initialMatrices[i] = bindPoses[f];
                found = true;
                break;
            }
        }
        if( !found )
        {
            // the node was not found to be part of the skin cluster, so use frame 0 pose.
            // the root node is left as identity - it will give a warning otherwise
            if( m_exportMappings[i].m_node.length() > 0 )
            {
                getLocalPivotTransform( m_exportMappings[i].m_node, nodeMatrices[i] );
            }
            else
            {
                nodeMatrices[i] = MMatrix::identity;
            }
            initialMatrices[i] = nodeMatrices[i];
        }
    }

    // Precompute keys for all nodes at each frame
    m_progressWindow.setRange( 0, frames );

    MMatrix transform;
    for( f = 0; f < frames; ++f )
    {
        MAnimControl::setCurrentTime( curTime );

        // Update progress bar
        MString str( "Precomputing animation data (" );
        str += 100*f/frames;
        str += "%)";
        m_progressWindow.setStatus( str );
        m_progressWindow.setProgress( f );
        if( m_progressWindow.isCancelled() )
        {
            break;
        }

        // Evaluate all animated node keys
        for( i = 0; i < numNodes; ++i )
        {
            if( !m_exportMappings[i].m_animatable ) continue;

            // re-evaluate the transform for the current time
            getLocalPivotTransform( m_exportMappings[i].m_node, transform );

            // convert the transform to Havok format
            hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( transform, m_exportMappings[i].m_keyFrames[f] );

            // if the transform has changed then it can't be a static node
            if( ( nodeMatrices[i] != transform ) && ( staticNodes[i] ) )
            {
                staticNodes[i] = false;
            }

            // store this frame's transform
            nodeMatrices[i] = transform;
        }

        curTime += tInc; // inc by one field
    }

    // replace animation key data for static nodes with bind pose data
    if( !m_progressWindow.isCancelled() )
    {
        // Static nodes in animated scene data are exported with two keys
        const bool exportTwoFramesForStaticNodes = (frames > 1);

        for( int nodeIndex = 0;nodeIndex < numNodes; nodeIndex++ )
        {
            if( !m_exportMappings[nodeIndex].m_animatable || staticNodes[nodeIndex] )
            {
                // replace transform
                m_exportMappings[nodeIndex].m_keyFrames.setSize( exportTwoFramesForStaticNodes  ? 2: 1 );
                m_exportMappings[nodeIndex].m_keyFrames.optimizeCapacity( 0, true);

                // convert the bind pose transform to Havok format
                hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( initialMatrices[nodeIndex], m_exportMappings[nodeIndex].m_keyFrames[0] );

                if (exportTwoFramesForStaticNodes)
                {
                    hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( initialMatrices[nodeIndex], m_exportMappings[nodeIndex].m_keyFrames[1] );
                }
            }
        }
    }

    MAnimControl::setCurrentTime( originalTime ); // reset time back
}


MStatus hctMayaSceneExporter::createHkxNodes()
{
    // Make sure our scene helper struct is clean
    m_currentScene.clear();

    // Init the attribute database
    MString filterManagerPath;
    bool filtersOk = hctMayaUtilities::getFilterManagerPath( m_pluginPath.asChar(), filterManagerPath );
    hkStringOld attrDescsRoot = hkStringOld(filterManagerPath.asChar()) + "\\attributeProcessing";
    hkStringOld attrSelsRoot = hkStringOld(filterManagerPath.asChar()) + "\\attributeSelection";


    {
        m_attributeDatabase.init();

        // Load common attribute descriptions
        hkStringOld attrDescs = attrDescsRoot;
        m_attributeDatabase.loadAttributeDescriptions( attrDescs.cString() );

        // Load Maya-specific attribute descriptions
        attrDescs += "\\maya";
        m_attributeDatabase.loadAttributeDescriptions( attrDescs.cString() );

        if( m_attributeDatabase.isEmpty() && filtersOk)
        {
            // Changed from assert to warn so that it doesn't crash modellers in debug
            HK_WARN_ALWAYS(0xabbabf13, "Couldn't find any attribute description XML file(s) in either [" << attrDescsRoot.cString() << "] or [" << attrDescs.cString() << "], aborting." );
            return MStatus::kFailure;
        }
    }

    {
        m_attributeSelection.init();

        // Load maya-specific attribute selections
        hkStringOld mayaSelectionsPath = attrSelsRoot;
        mayaSelectionsPath += "\\maya";
        m_attributeSelection.loadAttributeSelections( mayaSelectionsPath.cString() );

    }

    // Need some nodes
    const int numExportMappings = m_exportMappings.getSize();
    if( numExportMappings == 0 )
    {
        return MStatus::kFailure;
    }

    // We set the time to the start frame to sample the first values
    const MTime originalTime = MAnimControl::currentTime();
    MAnimControl::setCurrentTime( m_options.m_startTime );

    // Get the active selection list
    MSelectionList selectionList;
    MGlobal::getActiveSelectionList( selectionList );


    // Allocate storage for the HKX nodes
    hkArray< hkRefPtr<hkxNode> > exportedNodes;
    exportedNodes.setSize(numExportMappings);

    // Finalize Maya->HKX node mapping
    for( int mappingIndex=0; mappingIndex<numExportMappings; ++mappingIndex )
    {
        exportedNodes[mappingIndex] = new hkxNode();
        m_exportMappings[mappingIndex].m_hkxNode = exportedNodes[mappingIndex];
        exportedNodes[mappingIndex]->removeReference(); // all ptrs above are refptrs
    }

    // The root node will be the last in the list
    m_currentScene.m_rootNode = exportedNodes[numExportMappings-1];

    // Make HKX node selection sets
    for( int s=0; s<m_exportedSets.getSize(); ++s )
    {
        ExportedSet& exportedSet = m_exportedSets[s];
        createHkxNodeSelectionSet( exportedSet );
    }

    // We keep references to the animated attributes - we'll sample all of them together
    m_mayaAnimAttributes.clear();
    m_havokAnimAttributes.clear();

    MStatus status;

    // Preprocess all nodes to first set their names
    // COM-883 Requires skin bindings to store node names.  These must be valid to be used.
    // This step is assumed to not take much time and is not included in the progress meter.
    for( int i=0; i<numExportMappings; ++i )
    {
        ExportMapping& ExportMapping = m_exportMappings[i];
        const MDagPath& node = ExportMapping.m_node;
        hkxNode* outNode = ExportMapping.m_hkxNode;

        // Set the (unique) name
        MString nodeName = ( node.length() == 0 )? "ROOT_NODE": node.partialPathName();

        // Replace FBXASC code on export EXP-2729
        hkStringBuf name( nodeName.asChar() );
        name.replace("FBXASC032", " ");
        outNode->m_name = name.cString();
    }

    // Convert the Maya nodes to HKX nodes
    m_progressWindow.setRange( 0, numExportMappings );

    for( int i=0; i<numExportMappings; ++i )
    {
        status = MStatus::kSuccess;

        ExportMapping& ExportMapping = m_exportMappings[i];
        const MDagPath& node = ExportMapping.m_node;
        hkxNode* outNode = ExportMapping.m_hkxNode;

        MString str( "Processing nodes (" );
        str += node.partialPathName();
        str += ")";

        m_progressWindow.setStatus( str );
        m_progressWindow.setProgress(i);
        if( m_progressWindow.isCancelled() )
        {
            // stop exporting
            status = MStatus::kFailure;
            break;
        }

        // Set selection status
        outNode->m_selected = selectionList.hasItem( node );
        bool currentNodeIsBone = node.node().hasFn( MFn::kJoint );
        outNode->m_bone = currentNodeIsBone;

        // Setup the children ptr array
        {
            outNode->m_children.setSize(0);
            for( int j=0; j<i; ++j )
            {
                if( m_exportMappings[j].m_parentNodeIndex == i )
                {
                    outNode->m_children.pushBack( exportedNodes[j] );
                }
            }
        }

        // Set keyframe data
        // Can use the memory directly from the keyframe array (as long as it is not deleted too soon)
        outNode->m_keyFrames.append( ExportMapping.m_keyFrames.begin(), ExportMapping.m_keyFrames.getSize() );

    if (m_options.m_storeKeyframeSamplePoints && outNode->m_keyFrames.getSize() > 2) // 2 frames get exported if totally redundant track
    {
            // fill keyframe times array (for Visions simple linear blending)
      hkArray<MTime> sampleTicks;
      hctUtilities::getKeyTimes(node, sampleTicks, m_options.m_startTime, m_options.m_endTime );
      if (sampleTicks.getSize() > 0)
      {
        const int numFrames = outNode->m_keyFrames.getSize();
        const int numKeys = sampleTicks.getSize();
        for (int ki =0; ki < numKeys; ++ki)
        {
          MTime keyTimeRebased = (sampleTicks[ki] - m_options.m_startTime);
          int keyFrameIndex = (int)keyTimeRebased.as( MTime::uiUnit() );
          float keyFrameTime = (float)keyTimeRebased.as( MTime::kSeconds );
          if (keyFrameIndex >= numFrames)
            break;
          if (outNode->m_linearKeyFrameHints.indexOf(keyFrameTime) < 0) // need to check if have it due to sub frame timer accuracy so can get more than one in one frame (just one per tick though as getKeyTimes() sorts etc)
                        outNode->m_linearKeyFrameHints.pushBack(keyFrameTime);
        }
      }
    }


        // Process the child mesh/light/camera node
        {
            MDagPath childPath = node;
            {
                unsigned int numChildren = node.childCount();
                for( unsigned int childIndex=0; childIndex<numChildren; ++childIndex )
                {
                    // Skip intermediate objects unless the user has flagged not to
                    {
                        hkxEnvironment environment;
                        environment.interpretString( m_options.m_environmentVariables.asChar() );
                        if( ( environment.getVariableValue("do_not_skip_intermediate_objects") == HK_NULL ) &&
                            ( ::getenv("do_not_skip_intermediate_objects") == HK_NULL ) &&
                            ( MFnDagNode( node.child(childIndex) ).isIntermediateObject() ) )
                        {
                            continue;
                        }
                    }

                    const MFn::Type apiType = node.child(childIndex).apiType();
                    if( apiType == MFn::kMesh || apiType==MFn::kNurbsSurface || node.hasFn( MFn::kLight ) || apiType == MFn::kCamera )
                    {
                        childPath.push( node.child(childIndex) );
                        break;
                    }
                }
            }
            if( childPath.isValid() )
            {
                const MFn::Type apiType = childPath.apiType();

                if( m_options.m_exportMeshes && ( apiType == MFn::kMesh) )
                {
                    status = processMesh( childPath, outNode->m_object, ExportMapping.m_hasBonesInParentTree && m_options.m_autoSkinAttachments  );
                    if( status != MStatus::kSuccess )
                    {
                        status.perror( "Error processing mesh" );
                    }
                }
                else if (m_options.m_exportMeshes && ( apiType == MFn::kNurbsSurface) )
                {
                    status = processNurbs( childPath, outNode->m_object, ExportMapping.m_hasBonesInParentTree && m_options.m_autoSkinAttachments  );
                    if( status != MStatus::kSuccess )
                    {
                        status.perror( "Error processing nurbs" );
                    }
                }
                else if( m_options.m_exportLights && (childPath.hasFn( MFn::kLight )) )
                {
                    hkxLight* newLight = processLight( childPath, status );
                    outNode->m_object = newLight;
                    if( newLight )
                    {
                        newLight->removeReference();
                    }
                    if( status != MStatus::kSuccess )
                    {
                        status.perror( "Error processing light" );
                    }
                }
                else if (m_options.m_exportSplines && (childPath.hasFn( MFn::kNurbsCurve)))
                {
                    hkxSpline* newSpline = processSpline (childPath, status);
                    outNode->m_object = newSpline;
                    if (newSpline)
                    {
                        newSpline->removeReference();
                    }
                    if ( status != MStatus::kSuccess )
                    {
                        status.perror( "Error processing nurbs curve" );
                    }
                }
                else if( m_options.m_exportCameras && (apiType== MFn::kCamera) )
                {
                    hkxCamera* newCamera = processCamera( childPath, status );
                    outNode->m_object = newCamera;
                    if( newCamera )
                    {
                        newCamera->removeReference();
                    }
                    if( status != MStatus::kSuccess )
                    {
                        status.perror( "Error processing camera" );
                    }
                }
            }

            if( status != MStatus::kSuccess )
            {
                char buf[1024];
                hkString::sprintf( buf, "Error while processing node [%s].", node.partialPathName().asChar() );
                status.perror( buf );
            }
        }

        // Add attribute groups
        if( m_options.m_exportAttributes )
        {
            addAttributeGroups( outNode, node.node() );
        }

        // Add annotations
        if( m_options.m_exportAnnotations )
        {
            addAnnotations( outNode, node );
        }

#ifdef HAVOK_VISION_EXPORTER
     // HCT Maya does not set the userprops string anyway, so we know it is empty, can just replace:
    MayaVision::GetCustomVisionData( node.node(), outNode->m_userProperties );
#endif

    }


    if( status == MStatus::kSuccess && m_currentScene.m_rootNode != HK_NULL && (!m_progressWindow.isCancelled()) )
    {
        // Sample all animated attributes together
        sampleAttributes();

        // Perform post processing on HKX attributes
        postProcessAttributes( m_currentScene.m_rootNode );
    }
    else
    {
        status = MStatus::kFailure;
    }


    // Restore original time
    MAnimControl::setCurrentTime( originalTime );

    return status;
}


void hctMayaSceneExporter::createHkxNodeSelectionSet( const ExportedSet& exportedSet )
{
    // Create a selection set
    hkxNodeSelectionSet* selectionSet = new hkxNodeSelectionSet();

    // Store the name of the selection set
    selectionSet->m_name = exportedSet.m_name.asChar(); // not unicode safe..

    // Find nodes in the selection set
    unsigned int numItems = exportedSet.getSetMembers().length();
    hkArray< hkRefPtr<hkxNode> > nodesInSet;

    for (unsigned int n=0; n<numItems; ++n)
    {
        const MDagPath& memberPath = exportedSet.getSetMembers()[n];

        // Find ExportMapping with this DAG node
        int mappingIndx = -1;
        for( int m=0; m<m_exportMappings.getSize(); ++m )
        {
            if ( m_exportMappings[m].m_node == memberPath )
            {
                mappingIndx = m;
                break;
            }
        }

        if (mappingIndx > -1)
        {
            ExportMapping& mapping = m_exportMappings[mappingIndx];
            if (mapping.m_hkxNode)
            {
                nodesInSet.pushBack(mapping.m_hkxNode);
            }
        }
    }

    // Allocate and assign nodes in the selection set if necessary
    // EXP-1477
    if( nodesInSet.getSize() > 0 )
    {
        // Allocate nodes
        selectionSet->m_selectedNodes.append(nodesInSet.begin(), nodesInSet.getSize() );
        m_currentScene.m_selectionSets.pushBack(selectionSet);
    }

    selectionSet->removeReference();
}


// Get any annotations attached to the node.
// In Maya, annotations are represented by enumerated attributes whose names start with 'hk'.
// The annotation's name is formed by concatenating the name of the attribute with its current value.
MStatus hctMayaSceneExporter::addAnnotations( hkxNode* hkx_node, const MDagPath& maya_node )
{
    hkArray<hkStringOld> annotationStrings;
    hkArray<hkReal> annotationTimes;

    MFnDagNode dagNode( maya_node.node() );
    int attributeCount = dagNode.attributeCount();
    MStatus status;

    for ( int index = 0; index < attributeCount; index++ )
    {
        MFnEnumAttribute enumAttribute( dagNode.attribute( index ), &status );
        if ( status == MStatus::kSuccess )
        {
            MString attributeName = enumAttribute.name();
            hkStringOld attName( attributeName.asChar(), attributeName.length() );
            // If we find an enumerated attribute beginning with 'hk', check its value at every frame.
            if ( attName.asUpperCase().beginsWith( "HK" ) )
            {
                MTime::Unit timeUnit = MTime::uiUnit();
                int frames = (int)( ( m_options.m_endTime - m_options.m_startTime ).as( timeUnit ) + 1 );

                MTime tInc( 1, timeUnit );
                MTime curTime = m_options.m_startTime;

                // Set the initial value to that of the last frame to catch annotations resulting from looping animations.
                short curValue;
                {
                    MAnimControl::setCurrentTime( m_options.m_endTime );
                    MPlug annotationData( dagNode.object(), enumAttribute.object() );
                    short finalValue;
                    annotationData.getValue( finalValue );
                    curValue = finalValue;
                }

                // Check each new value of the attribute with the current value.
                // If they're different, or there is a keyframe (EXP-2040), add an annotation.
                for( int f = 0; f < frames; ++f )
                {
                    MAnimControl::setCurrentTime( curTime );

                    MPlug annotationData( dagNode.object(), enumAttribute.object() );
                    short newValue;
                    annotationData.getValue( newValue );

                    MFnAnimCurve animCurve( annotationData );
                    unsigned int dummyInt;
                    if( newValue != curValue || animCurve.find( curTime, dummyInt ) )
                    {
                        curValue = newValue;

                        MString annotationName = enumAttribute.fieldName( curValue );

                        annotationStrings.pushBack( attName + hkStringOld( annotationName.asChar(), annotationName.length() ) );

                        MTime annotationTime = curTime - m_options.m_startTime;
                        annotationTimes.pushBack( (hkReal) annotationTime.as( MTime::kSeconds ) );
                    }

                    curTime += tInc;
                }
            }
        }
    }

    // Add all the saved annotations and their corresponding times to the hkxNode.
    int numAnnotations = annotationStrings.getSize();
    if( numAnnotations )
    {
        hkx_node->m_annotations.setSize(numAnnotations);

        for ( int i = 0; i < numAnnotations; i++ )
        {
            hkxNode::AnnotationData& desc = hkx_node->m_annotations[i];
            desc.m_time = annotationTimes[i];
            desc.m_description = annotationStrings[i].cString();
        }
    }

    return MStatus::kSuccess;
}

void hctMayaSceneExporter::convertMayaNode(MObject& mayaNode, hkResourceContainer* resourceContainer, hkArray<MString>& parentNames ) const
{
    MStatus status;

    int firstHavokAttributeIndex = -1;

    const hkReflect::Type* klass = getClassOfMayaNode( mayaNode, &firstHavokAttributeIndex );
    if ( !klass || ! hctSdkUtils::getTypeByName(klass->getName()) )
    {
        return;
    }

    MFnDependencyNode depNode(mayaNode);
    const char* nodeName = depNode.name(&status).asChar();

    hkReflect::Var object = createEmptyHavokObject( klass );    // todo pass in the type registry here

    const char* parentName = ( !parentNames.isEmpty() ) ? parentNames[0].asChar() : 0;

    // create container hierarchy
    hkResourceContainer* container = resourceContainer;
    {
        for ( int i = parentNames.getSize()-1; i>=0; i-- )
        {
            container = container->createContainer( parentNames[i].asChar() );
        }
    }


    hkResourceHandle* resourceHandle = container->createResource( nodeName, object.getAddress(), klass );

    getDataFromMayaNode(mayaNode, firstHavokAttributeIndex, resourceHandle, parentName);

    // try to set the member name to the name of the object
    {
        hkReflect::StringVar name = object["name"];
        if (name)
        {
            if (char** cstr = name.dynCast<char*>())
            {
                *cstr = hkString::strDup(nodeName);
            }
            else
            {
                name.setValue(nodeName);
            }
        }
    }
}

bool hctMayaSceneExporter::createGenericObjectsFromMayaNodes(hkResourceContainer* resourceContainer)
{
    MStatus status;

    for (int i = 0; i < m_exportMappings.getSize(); i++)
    {
        MDagPath dagPath;
        {
            const ExportMapping& em = m_exportMappings[i];
            if ( !em.m_node.hasFn( MFn::kDependencyNode ) )
            {
                continue;
            }
            dagPath = em.m_node;
        }
        //MString s = dagPath.partialPathName( &status );
        //MGlobal::displayWarning(s);

        // Get all parent node names if it exists including the current node
        hkArray<MString> parentNames;
        {
            for ( int pi = i; pi>=0; pi = m_exportMappings[pi].m_parentNodeIndex )
            {
                MString pn = m_exportMappings[pi].m_node.partialPathName( &status );
                parentNames.pushBack(pn);
            }
            if ( !parentNames.isEmpty() )
            {
                if ( parentNames.back().length() == 0 || hkString::strCmp( parentNames.back().asChar(), "world" )== 0)
                {
                    parentNames.popBack();
                }
            }
        }
        // iterate of all its children to get the attribute groups
        // We only want to export root/transform/joint nodes
        int numChildren = dagPath.childCount( &status );
        for ( int childIndex = 0; childIndex < numChildren; childIndex++ )
        {
            MObject mayaNode = dagPath.child( childIndex, &status );

            const MFn::Type nodeApiType = mayaNode.apiType();
            if (  (dagPath.length() == 0) ||
                (nodeApiType == MFn::kPluginTransformNode) ||
                (nodeApiType == MFn::kTransform) ||
                (nodeApiType == MFn::kJoint) ||
                (nodeApiType == MFn::kLodGroup))
            {
                continue; // node is already in our export mapping lists
            }
            if ( nodeApiType != MFn::kPluginLocatorNode && nodeApiType != MFn::kPluginDependNode )
            {
                continue;
            }
            convertMayaNode( mayaNode, resourceContainer, parentNames );
        }
    }
    // now iterate over all root nodes
    {
        hkArray<MString> parentNames;
        MItDependencyNodes nodeIterator( MFn::kDependencyNode, &status );
        for ( ; !nodeIterator.isDone();  nodeIterator.next() )
        {
            MObject mayaNode = nodeIterator.thisNode();
            MFnDagNode dagNodeFn(mayaNode, &status);
            if (dagNodeFn.parentCount() > 0 )
            {
                continue;
            }
            convertMayaNode( mayaNode, resourceContainer, parentNames );
        }
    }

    {
        hkError::getInstance().setEnabled(0xaf12e114, false);

        hkContainerResourceMap map( resourceContainer ) ;
        resourceContainer->tryToResolveLinks( map );

        hkError::getInstance().setEnabled(0xaf12e114, true);
    }

    // The main hctFilterProcessingUtil will register all (not just hkxScene) local classes types when it goes to export
    // so all the classes exported above will be merged in too.

    return true;
}

void hctMayaSceneExporter::cleanupRegisteredClassNodes()
{
    hctSdkUtils::unloadAllClasses();
}

const hkReflect::Type* hctMayaSceneExporter::getClassOfMayaNode( const MObject& mayaObject, int* attributeIndexOfFirstHavokAttribute ) const
{
    const hkArray<hctModelerNodeIdToClassMap>& classMap = hctSdkUtils::getNodeIdToClassMap();

    MFnDependencyNode nodeFn(mayaObject);

    MStatus status = MStatus::kSuccess;
    int attributeCount = nodeFn.attributeCount();
    const char* prefix = "hktypedestruction";
    int prefixLen = hkString::strLen(prefix);
    {
        for (int i = 0; i < attributeCount; ++i)
        {
            MObject attrObject = nodeFn.attribute(i);
            MFnAttribute attribute( attrObject, &status );
            const char* attributeName = attribute.name().asChar();

            if ( hkString::strNcasecmp( attributeName, prefix, prefixLen) == 0)
            {
                MPlug attributePlug( nodeFn.object(), attribute.object() );
                MString newTypeName;
                attributePlug.getValue( newTypeName );

                // Search for the type in our mapping table.
                const hkReflect::Type* klass = HK_NULL;
                {
                    const char* className = newTypeName.asChar();

                    int classCtr = 0;
                    while ( classCtr < classMap.getSize() )
                    {
                        if ( hkString::strCasecmp(className, classMap[classCtr].m_class->getName()) == 0 )
                        {
                            klass = classMap[classCtr].m_class;
                            *attributeIndexOfFirstHavokAttribute = i+1;
                            return klass;
                        }
                        classCtr++;
                    }
                }
            }
        }
    }
    attributeIndexOfFirstHavokAttribute = 0;
    return HK_NULL;
}


hkReflect::Var HK_CALL hctMayaSceneExporter::createEmptyHavokObject( const hkReflect::Type* klass )
{
    
    // Keep compatibility with the old behavior. Use a zeroed buffer so that reference count is disabled and objects
    // are never deleted (that might call destructors which were never meant to be called).
    void* object = hkAllocate<char>( klass->getSizeOf(), HK_MEMORY_CLASS_EXPORT);
    hkString::memSet(object, 0x0, klass->getSizeOf());

    hkReflect::Var res(object, klass);

    // Initialize vtables.
    hkReflect::Detail::initializeVtables(object, klass);

    // Set the defaults.
    res.writeDefault();
    return res;
}


void hctMayaSceneExporter::getDataFromMayaNode( const MObject& mayaObject, int firstAttributeIndex, hkResourceHandle* resourceHandle, const char* parentName ) const
{
    MFnDependencyNode nodeFn(mayaObject);

    MStatus status = MStatus::kSuccess;

    int attributeCount = nodeFn.attributeCount();
    hkReflect::Var object(resourceHandle->getObject(), resourceHandle->getObjectType());
    {
        for (int i = firstAttributeIndex; i < attributeCount; i++)
        {
            MObject attrObject = nodeFn.attribute(i);
            MFnAttribute attribute( attrObject, &status );
            const char* attributeName = attribute.name().asChar();

            MFn::Type objectType = attrObject.apiType();

            hkReflect::Var field = hctMayaUtilities::getField(attributeName, object);

            if ( !field.isValid() )
            {
                // If there's no class member that would match the Custom Attribute's name, we will try to add the Attribute and its value
                // as a property to the class (if the class supports this).
                hkReflect::Var attributesField = object["attributes"];
                if ( attributesField.isValid() )
                {
                    if ( hkArray<hkxAttribute>* attributesMember = attributesField.dynCast< hkArray<hkxAttribute> >())
                    {
                        hkxAttribute attributeNew;
                        bool attributeCreated = createAttribute(mayaObject, attrObject, attributeNew);
                        if ( attributeCreated )
                        {
                            attributesMember->pushBack(attributeNew);
                        }
                    }
                }
                continue;
            }

            MPlug attributePlug( nodeFn.object(), attribute.object() );

            hctMayaUtilities::attributeToClassMember(attributePlug, field);

            const hkReflect::Type* fieldType = field.getType();

            // Find attributes if they exist
            const hkLinkAttribute*      linkAttributes      = fieldType->findAttribute<hkLinkAttribute>();

            // If we have a link/pointer, we need to create extra data.
            bool pointerToRec = fieldType->asPointer() &&
                fieldType->asPointer()->getSubType() &&
                fieldType->asPointer()->getSubType()->asRecord();
            if ( objectType == MFn::kMessageAttribute && pointerToRec)
            {
                // If we link to another object(!) and this member has been classified as a link and the link is valid...
                if ( linkAttributes && (linkAttributes->m_type != hkLinkAttribute::NONE) )
                {
                    if ( attributePlug.isConnected() )
                    {
                        // Find the object we are pointing to, and copy the name over.

                        MStatus statusConnect;
                        MPlugArray outgoingConnections;
                        attributePlug.connectedTo(outgoingConnections, true, false, &statusConnect);
                        if (statusConnect != MStatus::kSuccess )
                        {
                            MGlobal::displayWarning(statusConnect.errorString());
                        }
                        if ( outgoingConnections.length() > 0 )
                        {
                            MObject destNode = outgoingConnections[0].node(&statusConnect);
                            if (statusConnect != MStatus::kSuccess )
                            {
                                MGlobal::displayWarning(statusConnect.errorString());
                            }

                            MString name;
                            MFnDependencyNode depNode(destNode, &statusConnect);
                            if (statusConnect != MStatus::kSuccess )
                            {
                                MGlobal::displayWarning(statusConnect.errorString());
                            }

                            name = depNode.name(&statusConnect);
                            if (statusConnect != MStatus::kSuccess )
                            {
                                MGlobal::displayWarning(statusConnect.errorString());
                            }

                            // Get link name
                            hkStringBuf linkName(name.asChar());
                            if ( linkAttributes->m_type == hkLinkAttribute::NODE_UUID )
                            {
                                // Get a DAG path
                                MDagPath dagPath;
                                statusConnect = MDagPath::getAPathTo(destNode, dagPath);
                                if ( statusConnect != MStatus::kSuccess )
                                {
                                    MGlobal::displayWarning(statusConnect.errorString());
                                }

                                // Locate its hkxNode
                                const hkxNode* linkedHkxNode = getHkxNode(dagPath);
                                if ( linkedHkxNode )
                                {
                                    const hkUuid& linkUuid = linkedHkxNode->m_uuid;
                                    linkUuid.toString(linkName);
                                }
                            }

                            // Register external link at resource handle.
                            {
                                hkStringOld memberName = attributeName;
                                memberName = memberName.replace(":", "."); // sync id: <0xaf1ee231>
                                resourceHandle->addExternalLink(memberName.cString(), linkName.cString());
                            }
                        }
                    }
                    else
                    {
                        hkStringOld memberName = attributeName;
                        memberName = memberName.replace(":", "."); // sync id: <0xaf1ee231>
                        hkStringOld replacementLinkName;
                        replacementLinkName.printf("UndefinedLink_%s", fieldType->isField() ?
                            fieldType->isField().getName() : fieldType->getName());
                        resourceHandle->addExternalLink(memberName.cString(), replacementLinkName.cString());
                    }
                }
            }
            else if( linkAttributes && linkAttributes->m_type == hkLinkAttribute::PARENT_NAME )
            {
                // We have a string that needs to store the name of the object's parent
                if (hkReflect::StringVar str = field)
                {
                    if (char** cstr = str.dynCast<char*>())
                    {
                        *cstr = hkString::strDup(parentName);
                    }
                    else
                    {
                        str.setValue(parentName);
                    }
                }
            }
        }
    }
    resourceHandle->setObject(object.getAddress(), object.getType());
}


void hctMayaSceneExporter::collectNodesRecursively(MObject& node, hkArray<MObject>& nodesOut) const
{
    MStatus status;
    MFnDagNode dagNode(node, &status);
    if ( status == MStatus::kSuccess )
    {
        // Ignore non-Mesh nodes.
        if ( node.apiType() == MFn::kMesh )
        {
            MObject& storedNode = nodesOut.expandOne();
            new (&storedNode) MObject(node);
        }

        // Process all child nodes.
        {
            for (int i = 0; i < int(dagNode.childCount()); i++)
            {
                MObject childNode = dagNode.child(i);
                MFnDagNode childDagNode(childNode, &status);
                if ( status == MStatus::kSuccess )
                {
                    collectNodesRecursively(childNode, nodesOut);
                }
            }
        }

    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
