// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Some extra Maya includes
#if MAYA_API_VERSION >= 20180000
#   include <maya/MDGContextGuard.h>
#endif
#include <maya/MFnSingleIndexedComponent.h>
#include <maya/MFnLambertShader.h>
#include <maya/MFnReflectShader.h>
#include <maya/MFnPhongShader.h>
#include <maya/MFnBlinnShader.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MItMeshPolygon.h>
#include <maya/MItMeshVertex.h>
#include <maya/MItMeshEdge.h>
#include <maya/MItGeometry.h>
#include <maya/MIkSystem.h>
#include <maya/MFnIkJoint.h>
#include <maya/MFnDoubleArrayData.h>

// Some extra Havok includes
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Math/hkMath.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/SceneData/Skin/hkxSkinUtils.h>

#define MAX_NUMBER_HAVOK_BONES 4
#define PEDANTIC_ASSERTS 0

hctMayaMeshWriter::hctMayaMeshWriter()
    :   m_currentUvChooserIdPtr(HK_NULL)
{

}

hctMayaMeshWriter::~hctMayaMeshWriter()
{
    cleanup();
}


void hctMayaMeshWriter::setOwner( hctMayaSceneExporter* owner )
{
    m_owner = owner;
    m_currentUvChooserIdPtr = &m_owner->m_uniqueUvChooserIdBase;
}

static void _countChildNodesRecursive(MObject& currNode, int &numChildren)
{
    MDagPath currNodePath;
    MDagPath::getAPathTo( currNode, currNodePath);

    int currNumChildren = currNodePath.childCount();

    for (int c=0; c< currNumChildren ; ++c)
    {
        MFn::Type nodeApiType = currNodePath.child(c).apiType();

        const bool shouldCount =
            (nodeApiType == MFn::kPluginTransformNode) ||
            (nodeApiType == MFn::kTransform) ||
            (nodeApiType == MFn::kJoint) ||
            (nodeApiType == MFn::kLodGroup);

        if (shouldCount)
        {
            MObject childObject = currNodePath.child(c);
            _countChildNodesRecursive(childObject, numChildren);
        }
    }
    numChildren++;
}
static int _getNumberOfBonesInSkeleton(MObject& bone)
{
    int numBonesInSkeleton = 0;

    if( bone.isNull() )
        return numBonesInSkeleton;

    MDagPath bonePath;
    MDagPath::getAPathTo( bone, bonePath);

    MStatus status;

    MObject parentObject;
    // Get the root node of the skeleton first
    while( bonePath.length() > 1 && status == MStatus::kSuccess)
    {
        // Get the parent path and object
        MDagPath parentPath = bonePath;
        parentPath.pop();
        parentObject = parentPath.node(&status);

        bone = parentObject;
        bonePath = parentPath;
    }

    //Count children
    _countChildNodesRecursive(bone, numBonesInSkeleton);

    return numBonesInSkeleton;
}

static MObject _getBoneRoot(MObject& bone)
{
    if( bone.isNull() )
        return bone;

    MDagPath bonePath;
    MDagPath::getAPathTo( bone, bonePath);

    MStatus status;

    MDagPath parentPath = bonePath;
    MObject parentObject = parentPath.node(&status);

    // Get the root node of the skeleton first
    while( bonePath.length() > 1 && status == MStatus::kSuccess)
    {
        // Get the parent path and object
        MDagPath parentBonePath = bonePath;
        parentBonePath.pop();
        parentObject = parentBonePath.node(&status);

        bone = parentObject;
        bonePath = parentBonePath;
    }

    return parentObject;
}

void hctMayaMeshWriter::findSkins()
{
    m_skinClusters.clear();
    m_allSkinInfluences.clear();
    MObject meshObject = m_mesh.object();

    for(MItDependencyGraph clusterIt( meshObject, MFn::kGeometryFilt, MItDependencyGraph::kUpstream); !clusterIt.isDone(); clusterIt.next())
    {
        // Get the current cluster.
        MObject thisNode = clusterIt.thisNode();

        // Early exit - ignore non-skinning stuff
        const bool skinCluster = thisNode.hasFn(MFn::kSkinClusterFilter) || thisNode.hasFn(MFn::kWeightGeometryFilt);
        if (!skinCluster) continue;

        MFnGeometryFilter fnGeomFilter(thisNode);

        // Collect bones that have an influence on our mesh, for both, ...
        hctMayaMeshWriter::Cluster& skinPlug = m_skinClusters.expandOne();
        skinPlug.cluster = clusterIt.thisNode();
        skinPlug.clusterPlugIndex = fnGeomFilter.indexForOutputShape(meshObject);

        MDagPathArray influences;
        // Smooth skin cluster
        if(thisNode.hasFn(MFn::kSkinClusterFilter))
        {
            MFnSkinCluster skinGeomFilter(clusterIt.thisNode());
            MStatus istat;
            skinGeomFilter.influenceObjects(influences,&istat);  // Get bones influencing our mesh.
            skinPlug.influenceLogicalIndices.setSize(influences.length(),-1);
            for (unsigned int inf=0; inf < influences.length(); ++inf)
            {
                MStatus status;
                unsigned int influenceIndex = skinGeomFilter.indexForInfluenceObject( influences[inf], &status );
                if( status != MStatus::kSuccess )
                {
                    status.perror( "Failed to retrieve joint influence index!" );
                    continue;
                }
                skinPlug.influenceLogicalIndices[inf] = influenceIndex;
            }
        }
        // Rigid binding
        else if(thisNode.hasFn(MFn::kWeightGeometryFilt))
        {
            MFnWeightGeometryFilter fnJointCluster(thisNode);
            for(MItDependencyGraph jointIt(thisNode, MFn::kJoint, MItDependencyGraph::kUpstream); !jointIt.isDone(); jointIt.next())
            {
                // Add first upstream joint (only one that is attached to this cluster).
                MDagPath dagPath;
                if(MFnDagNode(jointIt.thisNode()).getPath(dagPath) == MS::kSuccess)
                {
                    influences.append(dagPath);
                    break;
                }
            }
        }

        skinPlug.influenceIndexIntoMasterArray.setSize(influences.length(), -1);
        for (unsigned int ii=0; ii < influences.length(); ++ii)
        {
            unsigned int ai=0;
            for (; ai < m_allSkinInfluences.length(); ++ai)
            {
                if (m_allSkinInfluences[ai] == influences[ii])
                {
                    skinPlug.influenceIndexIntoMasterArray[ii] = ai;
                    break;
                }
            }

            if (ai == m_allSkinInfluences.length()) // not found
            {
                m_allSkinInfluences.append(influences[ii]);
                skinPlug.influenceIndexIntoMasterArray[ii] = m_allSkinInfluences.length() - 1;
            }
        }
    }
}

MStatus hctMayaMeshWriter::extract( const MDagPath& dagPath )
{
    MStatus status = MStatus::kSuccess;

    // Clear everything
    cleanup();

    // Set the dag path, extended to the shape if necessary
    m_dagPath = dagPath;
    m_dagPath.extendToShape();

    // Get the mesh data
    status = m_mesh.setObject( m_dagPath );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error creating MFnMesh for dagPath!" );
        return status;
    }

    // Check if it is a skinned mesh
    findSkins();

    // If so, we need to prepare a DG modifier
    MDGModifier dgMod;
    for (int si=0; si < m_skinClusters.getSize(); ++si )
    {
        // Get the plug for the skinned mesh coming into the shape.
        MPlug meshInputPlug = m_mesh.findPlug( "inMesh", &status );

        // Get the plug for the mesh coming into the skinCluster.
        // This is the original mesh before being deformed but after being edited/tweaked/etc.
        bool skinningAMesh = false;
        MPlug preSkinOutputPlug;
        MPlug skinInputPlug ;
        MFnGeometryFilter skinGeomFilter(m_skinClusters[si].cluster);
        {
            MPlug inputElemPlug = skinGeomFilter.findPlug( "input", &status );
            if( status == MStatus::kSuccess)
            {
                MPlug thisMeshPlug = inputElemPlug.elementByLogicalIndex(m_skinClusters[si].clusterPlugIndex);
                skinInputPlug = thisMeshPlug.child(0);
            }
            MObject inputToSkin;
            MStatus st = skinInputPlug.getValue(inputToSkin);

            skinningAMesh = inputToSkin.hasFn(MFn::kMesh);

            MPlugArray skinInputs;
            skinInputPlug.connectedTo(skinInputs, true, false, &status);

            if (skinInputs.length()==1)
            {
                preSkinOutputPlug = skinInputs[0];
            }

        }

        MPlug skinOutputPlug;
        MPlug afterSkinInputPlug;
        {
            // Get the out plug from the skin deformer
            MPlug outputPlug = skinGeomFilter.findPlug( "outputGeometry", &status );
            if (status == MStatus::kSuccess)
            {
                skinOutputPlug = outputPlug.elementByLogicalIndex(m_skinClusters[si].clusterPlugIndex);

                MPlugArray skinOutputs;
                skinOutputPlug.connectedTo(skinOutputs, false, true, &status);

                if (skinOutputs.length()==1)
                {
                    afterSkinInputPlug = skinOutputs[0];
                }
            }

        }

        // EXP-848
        if (!skinningAMesh)
        {
            HK_WARN_ALWAYS(0xabba1baa, "Skinning is only supported for polygonal meshes. Ignoring skin ("<<m_dagPath.partialPathName().asChar()<<").");
            m_skinClusters.setSize(0);
        }

        if (!preSkinOutputPlug.isNull() && !afterSkinInputPlug.isNull() && skinningAMesh)
        {
            dgMod.disconnect(skinOutputPlug, afterSkinInputPlug);
            dgMod.connect(preSkinOutputPlug, afterSkinInputPlug);
        }

    }

    // Do any DG modifications. Only need to do this while we extract
    // the vertex positions and normals
    dgMod.doIt();

    // get the object space vertex positions
    status = m_mesh.getPoints( m_vertexArray, MSpace::kObject );
    if( status != MStatus::kSuccess )
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    // may need to make them relative to the transform's rotate pivot
    if( m_owner->m_options.m_useRotatePivot )
    {
        MDagPath parentPath( m_dagPath );
        parentPath.pop();
        MFnTransform parentTransformFn( parentPath, &status );
        if( status == MStatus::kSuccess )
        {
            MPoint offset = parentTransformFn.rotatePivot( MSpace::kPreTransform );
            MFloatPoint floatOffset( (float)offset.x, (float)offset.y, (float)offset.z );
            int numV = m_vertexArray.length();
            for( int i=0; i<numV; ++i )
            {
                m_vertexArray[i] -= offset;
            }
        }
    }

    // get the object space normals
    status = m_mesh.getNormals( m_normalArray, MSpace::kObject );
    if( status != MStatus::kSuccess )
    {
        // [EXP-2866] Collision mesh can't be exported without a UV set.
        m_normalArray.setLength(0);
    }

    if (m_owner->m_options.m_exportVertexTangents)
    {
        status = m_mesh.getTangents( m_tangentsArray, MSpace::kObject );
        if( status != MStatus::kSuccess )
        {
            m_tangentsArray.setLength(0);
        }

        status = m_mesh.getBinormals( m_binormalsArray, MSpace::kObject );
        if( status != MStatus::kSuccess )
        {
            m_binormalsArray.setLength(0);
        }
    }

    // Restore the original mesh
    dgMod.undoIt();
    m_mesh.syncObject();


    // get the colours if there are any. Maya 7 has multiple sets per vertex, of which we only support the current set
#if (MAYA_API_VERSION >= 700)
    // If we have 1 or more color sets, of which we can get the current ones name, then we will try to export them
    int numVColors = m_mesh.numColorSets(&status);
    if( (numVColors > 0) && (status == MStatus::kSuccess))
    {
        MStringArray allColorSets;
        if (m_mesh.getColorSetNames(allColorSets) == MStatus::kSuccess )
        {
            int supportedNumColors = hkMath::min2<int>( numVColors, 2 ); // we only store 2 so far
            for (int ci=0; ci < supportedNumColors; ++ci)
            {
                if (ci == 0)
                {
                    m_colorSetNameA = allColorSets[0];
                    status = m_mesh.getFaceVertexColors( m_colorArrayA, &m_colorSetNameA);
                }
                else
                {
                    m_colorSetNameB = allColorSets[1];
                    status = m_mesh.getFaceVertexColors( m_colorArrayB, &m_colorSetNameB);
                }

                if( status != MStatus::kSuccess )
                {
                    status.perror( "MFnMesh::getFaceVertexColors" );
                    return status;
                }
            }
        }
    }
    else
    {
        m_colorArrayA.setLength(0);
        m_colorArrayB.setLength(0);
    }
#else
    status = m_mesh.getFaceVertexColors( m_colorArrayA );
    if( status != MStatus::kSuccess )
    {
        status.perror( "MFnMesh::getFaceVertexColors" );
        return status;
    }
#endif

    // get the UV Set names [EXP-2344]
    m_UVSetNames.clear();
    status = m_mesh.getUVSetNames(m_UVSetNames);
    if( status != MStatus::kSuccess )
    {
        status.perror( "MFnMesh::getUVSetNames" );
        return status;
    }
    m_numUVSetNames = hkMath::min2(m_UVSetNames.length(), MAX_UV_SETS);
    if (m_UVSetNames.length() > MAX_UV_SETS)
    {
        HK_WARN_ALWAYS(0xabba1bab, "Found " << m_UVSetNames.length() << " UV sets. Those beyond the maximum of " << MAX_UV_SETS << " will be ignored.");
    }

    // get UV's
    for (int i = 0, n = m_numUVSetNames; i < n; i ++)
    {
        status = m_mesh.getUVs(m_uArray[i], m_vArray[i], &m_UVSetNames[i]);
        if (status != MStatus::kSuccess)
        {
            hkStringBuf msg;
            msg.printf("MFnMesh::getUVs[%d]", i);
            status.perror(msg.cString());
            return status;
        }
    }

    // initialize the polygon export flags
    m_exportedPolygons.setSize( m_mesh.numPolygons(), false );

    return MStatus::kSuccess;
}

void hctMayaMeshWriter::cleanup()
{
    // Clear the current object's data only.
    // Keep the lists of exported materials and textures.

    m_dagPath.set( MDagPath() );
    m_mesh.setObject( MObject::kNullObj );
    m_skinClusters.clear();

    m_currentColorSetName = "";

    m_numUVSetNames = 0;
    m_UVSetNames.clear();

    m_vertexArray.setLength(0);
    m_colorArrayA.setLength(0);
    m_colorArrayB.setLength(0);
    m_normalArray.setLength(0);
    m_tangentsArray.setLength(0);
    m_binormalsArray.setLength(0);

    for (int i = 0; i < 8; i ++)
    {
        m_uArray[i].setLength(0);
        m_vArray[i].setLength(0);
    }

    m_exportedPolygons.clear();
}


// Structure to hold combined vertex information
struct FullVert
{
    int origVertIndex;
    int reducedVertIndex;
    int normalIndex;
    int vertColorAIndex;
    int vertColorBIndex;
    int texCoordIndex[hctMayaMeshWriter::MAX_UV_SETS]; // [EXP-2344]
    int vertTangentIndex;
};

inline hkUint8 _quantizeTo255( float v )
{
    if( v < 0 )
    {
        v = 0;
    }
    float v2 = ( v * 255 );
    if( v2 < 255.0f )
    {
        return (hkUint8)v2;
    }
    else
    {
        return 255;
    }
}

hkxVertexAnimation* hctMayaMeshWriter::getVertexAnimation( AnimatedVertData& meshData, const hkxMeshSection* section, hkxVertexAnimationStateCache& c, const hctUserChannelUtil::SectionToGlobalMap& dataMap )
{
    const int maxNumVerts = meshData.m_vertexArray.length();
    const int maxNumNormals = meshData.m_normalArray.length();

    hkxVertexAnimation* vanim = HK_NULL;

    const hkxVertexBuffer& curVertexState = c.getState();
    const hkxVertexDescription& curVertexDesc = curVertexState.getVertexDesc();

    // The hkxVertexAnimaion can store animate points for any data (uvs, colors, etc)
    // The orig vMax2 etc exporters only supported Pos and Normal, so we will just do the same here
    const hkxVertexDescription::ElementDecl* posDecl = curVertexDesc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0 );
    const hkxVertexDescription::ElementDecl* normalDecl = curVertexDesc.getElementDecl( hkxVertexDescription::HKX_DU_NORMAL, 0 );
    const hkVector4* curPosDataPtr = (hkVector4*)curVertexState.getVertexDataPtr( *posDecl );
    const hkVector4* curNormalDataPtr = (hkVector4*)curVertexState.getVertexDataPtr( *normalDecl );
    int posStride = posDecl->m_byteStride / sizeof(hkVector4);
    int normalStride = normalDecl->m_byteStride / sizeof(hkVector4);

    // Note we try to extract the information here from the same Mesh type (Mesh not MNMesh say) with the same mappings
    int numVerts = dataMap.m_sectionVertexIdToGlobalVertexId.getSize();

    hkArray<hkVector4> animatedPos;
    hkArray<hkVector4> animatedNormal;
    hkArray<int> animatedVid;

    // Work put minimal set of animated verts
    for(int vi=0; vi < numVerts; vi++)
    {
        int mainMeshVertId = dataMap.m_sectionVertexIdToGlobalVertexId[vi];
        int mainMeshNormalId = dataMap.m_sectionVertexIdToGlobalNormalId[vi];

        if ( (mainMeshVertId < 0) || (mainMeshVertId >= maxNumVerts)
            || (mainMeshNormalId < 0) || (mainMeshNormalId >= maxNumNormals) )
        {
            // now out of range.. topology changed?
            HK_WARN_ALWAYS(0xabba0000, "Inconsitant topology during animated vert export");
            if (vanim) delete vanim;
            return HK_NULL;
        }

        const MFloatPoint& aPoint = meshData.m_vertexArray[mainMeshVertId];
        const MFloatVector& aNormal = meshData.m_normalArray[mainMeshNormalId];

        // If normal or pos != last anim pos for that vert then is new vert
        const hkVector4* curPos = curPosDataPtr + posStride*vi;
        const hkVector4* curNormal = curNormalDataPtr + normalStride*vi;
        hkVector4 animPos; animPos.set(aPoint.x, aPoint.y, aPoint.z);
        hkVector4 animNormal; animNormal.set(aNormal.x, aNormal.y, aNormal.z);

        // assume if one different, both are, just to keep code down
        if (!animPos.equals3(*curPos) || !animNormal.equals3(*curNormal))
        {
            animatedPos.pushBack(animPos);
            animatedNormal.pushBack(animNormal);
            animatedVid.pushBack(vi);
        }
    }

    //do we have any, if so, create new hkxVertexAnimation
    if (animatedVid.getSize() > 0)
    {
        vanim = new hkxVertexAnimation();
        {
            hkxVertexDescription reqdesc;
            reqdesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_POSITION, hkxVertexDescription::HKX_DT_FLOAT, 3) );
            reqdesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_NORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3) );
            vanim->m_vertData.setNumVertices(animatedVid.getSize(), reqdesc );

            // Associate our P and N with the original ones:
            vanim->m_componentMap.setSize(2);
            vanim->m_componentMap[0].m_use = hkxVertexDescription::HKX_DU_POSITION;
            vanim->m_componentMap[0].m_useIndexLocal = 0;
            vanim->m_componentMap[0].m_useIndexOrig = 0;

            vanim->m_componentMap[1].m_use = hkxVertexDescription::HKX_DU_NORMAL;
            vanim->m_componentMap[1].m_useIndexLocal = 0;
            vanim->m_componentMap[1].m_useIndexOrig = 0;
        }

        const hkxVertexDescription& curAnimDesc = vanim->m_vertData.getVertexDesc();
        const hkxVertexDescription::ElementDecl* aposDecl = curAnimDesc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0 );
        const hkxVertexDescription::ElementDecl* anormalDecl = curAnimDesc.getElementDecl( hkxVertexDescription::HKX_DU_NORMAL, 0 );
        hkVector4* aPosDataPtr = (hkVector4*)vanim->m_vertData.getVertexDataPtr( *aposDecl );
        hkVector4* aNormalDataPtr = (hkVector4*)vanim->m_vertData.getVertexDataPtr( *anormalDecl );
        int aposStride = aposDecl->m_byteStride / sizeof(hkVector4);
        int anormalStride = anormalDecl->m_byteStride / sizeof(hkVector4);

        vanim->m_vertexIndexMap.setSize( animatedVid.getSize() );
        for (int ai=0; ai < animatedVid.getSize(); ++ai )
        {
            vanim->m_vertexIndexMap[ai] = animatedVid[ai];
            hkVector4* destP = aPosDataPtr + (ai * aposStride);
            hkVector4* destN = aNormalDataPtr + (ai * anormalStride);
            const hkVector4& srcP = animatedPos[ai];
            const hkVector4& srcN = animatedNormal[ai];
            *destP = srcP;
            *destN = srcN;

        }
    }

    // Cleanup



    return vanim;

}

// create vertex buffer
MStatus hctMayaMeshWriter::createVertexBuffer( MObject& component, hkArray<hkxVertexBuffer*> & newVBs, hkArray<MIntArray> & idxBuffers,
    const MFloatArray& weights, unsigned influenceCount, bool useInt8BoneIndices )
{
    MStatus status;
    newVBs.clear();

    MIntArray triIndices;
    triIndices.clear();

    // Maya will still have per vertex colour information and will return a full buffer of 'blank' colour
    // data, so we will check if all the colours are default values ( -1, -1, -1, -1 ) and if so use the
    // material colour.
    hkUint32 numVColors = 0;
    {
        MColor defaultColor( -1.0f, -1.0f, -1.0f, -1.0f );
        for( hkUint32 j = 0; j < m_colorArrayA.length(); j++ )
        {
            // if we have a non-default colour switch to use the per vertex colour
            if( m_colorArrayA[j] != defaultColor )
            {
                ++numVColors;
                break;
            }
        }

        if (numVColors < 1)
        {
            m_colorArrayA.setLength(0);
        }

        int numBColors = 0;
        for( hkUint32 j = 0; j < m_colorArrayB.length(); j++ )
        {
            // if we have a non-default colour switch to use the per vertex colour
            if( m_colorArrayB[j] != defaultColor )
            {
                ++numVColors;
                ++numBColors;
                break;
            }
        }

        if (numBColors < 1)
        {
            m_colorArrayB.setLength(0);
        }
    }

    int numTCoords = m_numUVSetNames;

    hctUserChannelUtil::SectionToGlobalMap sectionMap;
    // speed up the indexing process
    int reserveSize(100000);
    sectionMap.m_sectionTriangleIdToGlobalTriangleId.reserve(reserveSize);
    sectionMap.m_sectionTriangleIdToGlobalFaceId.reserve(reserveSize);

    // Create a list of unique vertices,
    // and fill the triangle index list
    hkArray<FullVert> fullVerts;
    hkArray<int> vertexIndexRemap(m_vertexArray.length(), -1);
    MFloatPointArray reducedVertexArray;
    hkArray<hkArray<int> > vertsToFullVerts;
    {
        FullVert thisFullVert;

        // Maya will store edges such that shared edges have the same direction.
        // This causes a lot more vertices to be created than we need when we don't have that
        // requirement, and also stops imported objects from exporting again without making holes (bad for Destruction).
        // To solve that we have the min set of vertices, so just based on position, by welding them all if possible.

        const int numBuckets = 50;
        hkArray<int> vertexBuckets[numBuckets];

        MItMeshPolygon polyIter( m_dagPath, component );
        // Iterate over the polygons, find the AABB of all the vertices used so we can bucket them

        hkAabb localAabb;
        localAabb.m_max.setAll(-HK_REAL_MAX);
        localAabb.m_min.setAll(HK_REAL_MAX);
        for( ; !polyIter.isDone(); polyIter.next() )
        {
            MIntArray mayaVertIndices;
            polyIter.getVertices( mayaVertIndices );
            for (hkUint32 idx = 0; idx < mayaVertIndices.length(); ++idx)
            {
                const MFloatPoint& p0 = m_vertexArray[ mayaVertIndices[idx] ];
                hkVector4 v0; v0.load3( (float*)&p0);
                localAabb.includePoint( v0 );
            }
        }

        hkVector4 ext; ext.setSub4(localAabb.m_max, localAabb.m_min);
        if (ext.lengthSquared3() < 0.0001f )
        {
            ext.setAll(1.0f);
        }
        const int bucketAxis = ext.getMajorAxis3(); // bucket along longest ext
        const float bucketScale = numBuckets / ext(bucketAxis);
        MFloatPoint bucketOffset;
        localAabb.m_min.store3( (float*)&bucketOffset );
        polyIter.reset();
        for( ; !polyIter.isDone(); polyIter.next() )
        {
            MIntArray mayaVertIndices;
            polyIter.getVertices( mayaVertIndices );
            // add verts to buckets so we can compare against them
            for (hkUint32 idx = 0; idx < mayaVertIndices.length(); ++idx)
            {
                const int curVertIndex = mayaVertIndices[idx];
                const MFloatPoint& pp = m_vertexArray[ curVertIndex ];
                MFloatVector p = pp - bucketOffset;
                float bucketFloat = p[bucketAxis] * bucketScale;

                int bucketIndex[3];
                bucketIndex[0] = hkMath::hkFloorToInt( bucketFloat );
                bucketIndex[0] = hkMath::clamp( bucketIndex[0], 0, numBuckets - 1);

                bucketIndex[1] = hkMath::hkFloorToInt( bucketFloat + 0.1f ); // +/- 10% (.1 of 1 range)
                bucketIndex[1] = hkMath::clamp( bucketIndex[1], 0, numBuckets - 1);
                bucketIndex[1] = (bucketIndex[1] == bucketIndex[0])? -1 : bucketIndex[1];

                bucketIndex[2] = hkMath::hkFloorToInt( bucketFloat - 0.1f );
                bucketIndex[2] = hkMath::clamp( bucketIndex[2], 0, numBuckets - 1);
                bucketIndex[2] = (bucketIndex[2] == bucketIndex[1]) || (bucketIndex[2] == bucketIndex[0])? -1 : bucketIndex[2];

                bool unique = true;
                for (int bi=0; bi < 3; ++bi)
                {
                    if (bucketIndex[bi] < 0)
                        continue;

                    int thisBucket = bucketIndex[bi];

                    // have we done this vert index yet?
                    if (vertexBuckets[thisBucket].indexOf( curVertIndex ) < 0)
                    {
                        // only add to vertexBucket if actual pos unique compared to rest in bucket
                        if (unique) // not found yet
                        {
                            for (int ii=0; ii < vertexBuckets[thisBucket].getSize(); ++ii)
                            {
                                const int prevUniqueVertIndex = vertexBuckets[thisBucket][ii];
                                const MFloatPoint& p0 = m_vertexArray[ prevUniqueVertIndex ];
                                if ( p0.isEquivalent( pp, 0.00001f ) )
                                {
                                    unique = false;
                                    vertexIndexRemap[ curVertIndex ] = vertexIndexRemap[ prevUniqueVertIndex ];
                                    break;
                                }

                            }
                        }

                        // mark index as checked in this bucket
                        vertexBuckets[thisBucket].pushBack( curVertIndex );
                    }
                    else
                    {
                        unique = false; // have already seen this index, let alone this pos.
                    }
                } // up to the three possible buckets

                if (unique)
                {
                    // Unique, so add to the reduced set
                    vertexIndexRemap[ curVertIndex ] = reducedVertexArray.length();
                    reducedVertexArray.append( pp );
                }
            }
        }

        // Want to reduce not expand the array! So any code that expands is broken.
        HK_ASSERT_NO_MSG(0x106d38fd, reducedVertexArray.length() <= (hkUint32)vertexIndexRemap.getSize() );

        // XXX Change me : takes up a lot of mem (always did)
        vertsToFullVerts.setSize( reducedVertexArray.length() );

        // presize the arrays and get the triangles once, rather than in the indexing loop
        MIntArray triangleCounts;
        MIntArray dummy;
        m_mesh.getTriangles(triangleCounts,dummy);
        unsigned int triangleCountSize(triangleCounts.length());
        hkArray<int> triangleSums;
        triangleSums.reserveExactly(triangleCountSize);
        
        if(triangleCountSize > 0)
        {
            triangleSums.pushBack(triangleCounts[0]);
        }

        for (unsigned int fi=1; fi <triangleCountSize; ++fi)
        {
            triangleSums.pushBack(triangleSums[fi - 1] + triangleCounts[fi]);
        }

        // Iterate over the polygons
        polyIter.reset();
        for( ; !polyIter.isDone(); polyIter.next() )
        {
            // Never export the same one twice (shouldn't really happen as render sel groups are exclusive)
            const int polyIndex = polyIter.index();
            if( m_exportedPolygons[polyIndex] ) continue;
            m_exportedPolygons[polyIndex] = true;

            // Get the triangulation data
            MIntArray faceVertIndices; // Indices of the vertices in this face
            MIntArray faceTriIndices; // Triangulation indices
            MPointArray dummyArray;
            polyIter.getVertices( faceVertIndices );
            polyIter.getTriangles(dummyArray, faceTriIndices );

            HK_ASSERT( 0xabba98d9, (faceTriIndices.length() % 3)==0, "Triangulation failed");

            for( unsigned int i=0; i<faceTriIndices.length(); ++i )
            {
                // Find for the face-relative (local) vertex index
                unsigned int j;
                int localVertIndex = 0;
                for( j=0; j<faceVertIndices.length(); ++j )
                {
                    if( faceVertIndices[j] == faceTriIndices[i] )
                    {
                        localVertIndex = j;
                        break;
                    }
                }
                HK_ASSERT( 0x6a45ea32, j<faceVertIndices.length(), "Triangle vertex not found in polygon vertex list." );

                // Gather the vertex details
                const int originalMayaVertIndex = faceTriIndices[i];
                thisFullVert.origVertIndex = originalMayaVertIndex;
                thisFullVert.reducedVertIndex = vertexIndexRemap[ originalMayaVertIndex ];
                thisFullVert.normalIndex = polyIter.normalIndex( localVertIndex );

#if MAYA_API_VERSION > 800
                if (m_owner->m_options.m_exportVertexTangents)
                {
                    thisFullVert.vertTangentIndex = polyIter.tangentIndex( localVertIndex, &status );
                #if PEDANTIC_ASSERTS
                    HK_ASSERT(0xabba719f, MS::kSuccess == status, "Tangent index retrieval failed.");
                #endif
                }
                else
#endif
                {
                    thisFullVert.vertTangentIndex = -1;
                }

                for (int n = 0; n < numTCoords; n ++)
                {
                    int uvIndex = -1;
                    status = polyIter.getUVIndex( localVertIndex, uvIndex, &m_UVSetNames[n] );
                #if PEDANTIC_ASSERTS
                    HK_ON_DEBUG(hkStringBuf msg; msg.printf("%s: Get UV Index Failed for vertex %d in UV channel %d (%s).", status.errorString().asChar(), localVertIndex, n, m_UVSetNames[n].asChar()););
                    HK_ASSERT(0xabba714f, MS::kSuccess == status, msg.cString());
                #endif
                    thisFullVert.texCoordIndex[n] = uvIndex;
                }


                int vertColorAIndex = -1;
                if (m_colorArrayA.length() > 0)
                {
                    status = m_mesh.getFaceVertexColorIndex( polyIter.index(), localVertIndex, vertColorAIndex, &m_colorSetNameA );
                #if PEDANTIC_ASSERTS
                    HK_ASSERT(0xabba726f, MS::kSuccess == status, "Get Face Vertex Color Index Failed for Color Array A.");
                #endif
                }
                thisFullVert.vertColorAIndex = vertColorAIndex;

                int vertColorBIndex = -1;
                if (m_colorArrayB.length() > 0)
                {
                    status = m_mesh.getFaceVertexColorIndex( polyIter.index(), localVertIndex, vertColorBIndex, &m_colorSetNameB );
                #if PEDANTIC_ASSERTS
                    HK_ASSERT(0xabba7165, MS::kSuccess == status, "Get Face Vertex Color Index Failed for Color Array B.");
                #endif
                }
                thisFullVert.vertColorBIndex = vertColorBIndex;


                // Check if we had this vertex already
                const int vertexIndex = thisFullVert.reducedVertIndex;

                hkArray<int>& fullVertsWithThatIndex = vertsToFullVerts[vertexIndex];

                bool reusedVertex = false;

                for (int fvi=0; fvi<fullVertsWithThatIndex.getSize(); fvi++)
                {
                    const int fullVertexIndex = fullVertsWithThatIndex[fvi];

                    // EXP-1073 : Do not compare normals or uvs if the user doesn't want to split
                    if (m_owner->m_options.m_doNotSplitVertices)
                    {
                        triIndices.append(fullVertexIndex);
                        reusedVertex = true;
                        break;
                    }

                    const FullVert& fv = fullVerts[fullVertexIndex];

                    bool found = false;

#if (MAYA_API_VERSION >= 700)
                    // can't memcmp as origvertex index stored in thwre now
                    // EXP-1581 - Do extra comparisons to account for normal/color/uv being the same, just not the same index (positions already taken care of)
                    if (!found)
                    {
                        const bool sameNormal = m_normalArray.length() == 0 || (thisFullVert.normalIndex == fv.normalIndex) || (m_normalArray[thisFullVert.normalIndex].isEquivalent(m_normalArray[fv.normalIndex]));
                        const bool sameTangents = m_tangentsArray.length() == 0 || (thisFullVert.vertTangentIndex == fv.vertTangentIndex) || (thisFullVert.vertTangentIndex < 0) || (m_tangentsArray[thisFullVert.vertTangentIndex].isEquivalent(m_tangentsArray[fv.vertTangentIndex]));
                        const bool sameVcolorA = (thisFullVert.vertColorAIndex == fv.vertColorAIndex) || (m_colorArrayA[thisFullVert.vertColorAIndex]==m_colorArrayA[fv.vertColorAIndex]);
                        const bool sameVcolorB = (thisFullVert.vertColorBIndex == fv.vertColorBIndex) || (m_colorArrayB[thisFullVert.vertColorBIndex]==m_colorArrayB[fv.vertColorBIndex]);

                        bool sameUV = false;

                        // Check each UV channel
                        for (int tcoordIndex = 0; tcoordIndex < numTCoords; tcoordIndex ++)
                        {
                            // Same uv based on index
                            sameUV = (thisFullVert.texCoordIndex[tcoordIndex] == fv.texCoordIndex[tcoordIndex] );

                            // Same uv based on array value, make sure indices are valid values first
                            if ((thisFullVert.texCoordIndex[tcoordIndex] >= 0) && (fv.texCoordIndex[tcoordIndex] >= 0) &&
                                (thisFullVert.texCoordIndex[tcoordIndex] < (int)m_uArray[tcoordIndex].length()) && (fv.texCoordIndex[tcoordIndex] < (int)m_uArray[tcoordIndex].length()) &&
                                (thisFullVert.texCoordIndex[tcoordIndex] < (int)m_vArray[tcoordIndex].length()) && (fv.texCoordIndex[tcoordIndex] < (int)m_vArray[tcoordIndex].length())
                                )
                            {
                                sameUV = sameUV ||
                                    ((m_uArray[tcoordIndex][thisFullVert.texCoordIndex[tcoordIndex]] == m_uArray[tcoordIndex][fv.texCoordIndex[tcoordIndex]]) &&
                                     (m_vArray[tcoordIndex][thisFullVert.texCoordIndex[tcoordIndex]] == m_vArray[tcoordIndex][fv.texCoordIndex[tcoordIndex]]));
                            }

                            if (!sameUV)
                            {
                                // Not the same, can stop checking subsequent channels
                                break;
                            }
                        }

                        found = sameNormal && sameTangents && sameVcolorA && sameVcolorB && sameUV;
                    }
#else
                    // check each UV channel
                    bool sameUV = true;
                    for (int i = 0; i < numTCoords; i ++)
                    {
                        sameUV = sameUV && (fv.texCoordIndex[i] == thisFullVert.texCoordIndex[i]);
                    }

                    // vertex colors aren't indexed well, so compare the actual colors
                    found = (fv.vertIndex == thisFullVert.vertIndex &&
                        fv.normalIndex == thisFullVert.normalIndex &&
                        sameUV &&
                        m_colorArrayA[fv.vertColorAIndex] == m_colorArrayA[thisFullVert.vertColorAIndex] );
#endif

                    if (found)
                    {
                        triIndices.append(fullVertexIndex);
                        reusedVertex = true;
                        break;
                    }
                }


                // Add the vertex to the buffer if necessary
                if( !reusedVertex )
                {
                    const int fullVertIndex = fullVerts.getSize();
                    fullVerts.pushBack( thisFullVert );

                    // keep track between our vertex id and maya's
                    sectionMap.m_sectionVertexIdToGlobalVertexId.pushBack(originalMayaVertIndex);
                    sectionMap.m_sectionVertexIdToGlobalSampleId.pushBack(fullVertIndex);
                    sectionMap.m_sectionVertexIdToGlobalNormalId.pushBack(hkUint16(thisFullVert.normalIndex));

                    // Add to the triangle index list
                    triIndices.append( fullVertIndex );
                    fullVertsWithThatIndex.pushBack(fullVertIndex);
                }

                // keep a map between each triangle id (ours) and the maya face index
                if (i%3 == 0)
                {
                    sectionMap.m_sectionTriangleIdToGlobalFaceId.pushBack(polyIndex);

                    // calculate the global index of the current triangle using the number of triangles of all the
                    // previous faces (this is necessary because the number of triangles of each *global* face is variable)
                    int curIndex(i/3);
                    if (polyIndex > 0)
                        curIndex += triangleSums[polyIndex-1];

                    sectionMap.m_sectionTriangleIdToGlobalTriangleId.pushBack(curIndex);
                }
            }

        }
    }

    // We create index buffers for each triangle, based on the availability of uv coordinates
    hkArray<hkArray<FullVert>>  vtxBuffers;
    hkArray<hctUserChannelUtil::SectionToGlobalMap> subMaps;
    {
        //int numTCoordsInFormat    = hkMath::min2<int>(numTCoords, 2); //only two queried from Maya. TODO: add all the map channels
        int numBuffers          = 1 << numTCoords;
        idxBuffers.setSize(numBuffers);
        vtxBuffers.setSize(numBuffers);
        subMaps.setSize(numBuffers);
        newVBs.setSize(numBuffers);
        hkString::memSet(newVBs.begin(), NULL, sizeof(hkxVertexBuffer*) * numBuffers);

        // Compute vertex flags
        hkArray<int> vtxFlags;
        int nVerts = fullVerts.getSize();
        vtxFlags.setSize(nVerts);
        for(int idx = 0 ; idx < nVerts ; idx++)
        {
            const FullVert & v = fullVerts[idx];

            vtxFlags[idx] = 0;

            for (int channel = 0; channel < numTCoords; channel ++)
            {
                if (v.texCoordIndex[channel] >= 0 &&
                    v.texCoordIndex[channel] < hkMath::min2((int)m_uArray[channel].length(), (int)m_uArray[channel].length())) // why use min2????
                {
                    vtxFlags[idx] |= (1 << channel);
                }
            }
        }

        // For each triangle in the big mesh, compute the uv flags and see in which index buffer to add
        int nTriIndices = triIndices.length();
        for(int idx = 0 ; idx < nTriIndices ; idx += 3)
        {
            // Get triangle vertices
            int ivA = triIndices[idx + 0];
            int ivB = triIndices[idx + 1];
            int ivC = triIndices[idx + 2];

            // Compute the minimum set of common uv coords
            int fABC = vtxFlags[ivA] & vtxFlags[ivB] & vtxFlags[ivC];

            // Add the triangle in the index buffer corresponding to fABC
            MIntArray & idxBuffer = idxBuffers[fABC];
            idxBuffer.append(ivA);
            idxBuffer.append(ivB);
            idxBuffer.append(ivC);

            // Maintain the mapping between the section and the original
            hctUserChannelUtil::SectionToGlobalMap & crtSectionMap = subMaps[fABC];
            const int triIndex = idx / 3;
            crtSectionMap.m_sectionTriangleIdToGlobalFaceId.pushBack(sectionMap.m_sectionTriangleIdToGlobalFaceId[triIndex]);
            crtSectionMap.m_sectionTriangleIdToGlobalTriangleId.pushBack(sectionMap.m_sectionTriangleIdToGlobalTriangleId[triIndex]);
        }
    }

    // We'll generate a section for each non-empty index buffer created above
    for(int cfg = idxBuffers.getSize() - 1 ; cfg >= 0 ; cfg--)
    {
        MIntArray & idxBuffer = idxBuffers[cfg];
        hctUserChannelUtil::SectionToGlobalMap & crtSectionMap = subMaps[cfg];

        // Create a vertex buffer only for the vertices in this index buffer. Re-index the buffer to point only to the new vertex buffer
        int nIndices = idxBuffer.length();
        if(nIndices)
        {
            hkArray<FullVert> & vtxBuffer = vtxBuffers[cfg];

            hkArray<int> reidx;

            // Create a re-index table
            int nAllVerts = fullVerts.getSize();
            reidx.setSize(fullVerts.getSize());
            hkString::memSet4(reidx.begin(), -1, nAllVerts);

            // Create the vertex buffer and re-index the index buffer
            int nVerts = 0;
            for(int idx = 0 ; idx < nIndices ; idx++)
            {
                // Get index in fullVerts
                int & oldIdx = idxBuffer[idx];

                // See if already re-indexed
                int & reIdx = reidx[oldIdx];
                if(reIdx < 0)
                {
                    // Must add as new vertex
                    vtxBuffer.expandOne() = fullVerts[oldIdx];

                    // Maintain section mappings
                    crtSectionMap.m_sectionVertexIdToGlobalVertexId.pushBack(sectionMap.m_sectionVertexIdToGlobalVertexId[oldIdx]);
                    crtSectionMap.m_sectionVertexIdToGlobalSampleId.pushBack(nVerts);
                    crtSectionMap.m_sectionVertexIdToGlobalNormalId.pushBack(sectionMap.m_sectionVertexIdToGlobalNormalId[oldIdx]);

                    // Update re-index table and current index buffer
                    oldIdx = nVerts;
                    reIdx = nVerts;

                    // Increase the number of vertices in this sub-section
                    nVerts++;
                }
                else
                {
                    oldIdx = reIdx;
                }
            }
        }
    }

    // Finally, generate the sections
    for(int cfg = idxBuffers.getSize() - 1 ; cfg >= 0 ; cfg--)
    {
        MIntArray & crtIdxBuffer = idxBuffers[cfg];
        if(!crtIdxBuffer.length())
        {
            continue;   // Nothing to do, configuration has no triangles!
        }

        // We have a non-empty index buffer and its associated vertex buffer. Create a section for it
        hkArray<FullVert> & crtVtxBuffer = vtxBuffers[cfg];

        // The number of generated uv coords is the position of the MSB bit in cfg
        int numTCoordsInFormat = 0;
        {
            int _cfg = cfg;
            while(_cfg)
            {
                if(cfg & (1 << numTCoordsInFormat))
                {
                    numTCoordsInFormat++;
                }
                _cfg >>= 1;
            }
        }

        // Warn caller if for instance uv coords 0 are missing but we have valid uv coords 1
        for(int tc = 0 ; tc < numTCoordsInFormat ; tc++)
        {
            if(!(cfg & (1 << tc)))
            {
                hkStringOld str;
                str.printf( "Mesh %s. Exporting uv set %d, but with no valid uvs!",
                    m_mesh.partialPathName().asChar(), tc);
                HK_WARN_ALWAYS(0x29e0971e, str.cString());
            }
        }


        // Set the correct format
        const bool isSkinned = (weights.length() > 0) && (influenceCount > 0);

        hkxVertexDescription desiredVertDesc;
        desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_POSITION, hkxVertexDescription::HKX_DT_FLOAT, 3) );
        bool haveNormals = m_normalArray.length() > 0;
        if (haveNormals)
        {
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_NORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3) );
        }

        bool haveTangents = m_owner->m_options.m_exportVertexTangents && (m_tangentsArray.length() > 0);
        if (haveTangents)
        {
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_TANGENT, hkxVertexDescription::HKX_DT_FLOAT, 3) );
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3) ); // could leave this out, but requires changes to some filters etc if encoded as tangent.w
        }

        int supportedNumColors = hkMath::min2<int>( numVColors, 2 );
        for (int vc=0; vc < supportedNumColors; ++vc)
        {
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_COLOR, hkxVertexDescription::HKX_DT_UINT32, 1) );
        }

        // Create uv buffer declarations
        for(int tcr=0 ; tcr < numTCoordsInFormat ; ++tcr)
        {
            // EXP-2344 Add name of the channel to the descriptor so it can be referenced later on.
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_TEXCOORD, hkxVertexDescription::HKX_DT_FLOAT, 2, m_UVSetNames[tcr].asUTF8()));
        }

        if (isSkinned)
        {
            desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, hkxVertexDescription::HKX_DT_UINT8, 4) );

            if(useInt8BoneIndices)
            {
                desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, hkxVertexDescription::HKX_DT_UINT8, 4) );
            }
            else
            {
                desiredVertDesc.m_decls.pushBack( hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, hkxVertexDescription::HKX_DT_INT16, 4) );
            }
        }

        // Allocate buffer
        hkxVertexBuffer *& newVB = newVBs[cfg];
        newVB = new hkxVertexBuffer();
        newVB->setNumVertices( crtVtxBuffer.getSize(), desiredVertDesc );

        // fill vertex data
        const hkxVertexDescription& vertDesc = newVB->getVertexDesc();
        const hkxVertexDescription::ElementDecl* posDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_POSITION, 0);
        const hkxVertexDescription::ElementDecl* normDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_NORMAL, 0 );
        const hkxVertexDescription::ElementDecl* tangDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_TANGENT, 0 );
        const hkxVertexDescription::ElementDecl* binormDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_BINORMAL, 0 );
        const hkxVertexDescription::ElementDecl* weightsDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
        const hkxVertexDescription::ElementDecl* indicesDecl = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

        hkArray<const hkxVertexDescription::ElementDecl*> vcolorDecls;
        vcolorDecls.setSize(supportedNumColors);
        int vcolorStride[2];
        char* vcolorBuf[2];
        for (int cc=0; cc < supportedNumColors; ++cc)
        {
            vcolorDecls[cc] = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_COLOR, cc );
            if ( vcolorDecls[cc] )
            {
                vcolorStride[cc] = vcolorDecls[cc]->m_byteStride;
                vcolorBuf[cc] = (char*)newVB->getVertexDataPtr( *vcolorDecls[cc] );
            }
            else
            {
                vcolorStride[cc] = 0;
                vcolorBuf[cc] = HK_NULL;
            }
        }

        hkArray<const hkxVertexDescription::ElementDecl*> texCoordDecls;
        texCoordDecls.setSize(numTCoordsInFormat);

        int texCoordStride[8];
        char* texCoordBuf[8];

        for (int i = 0; i < 8; i ++)
        {
            texCoordStride[i] = 0;
            texCoordBuf[i] = HK_NULL;
        }

        for (int tc=0; tc < numTCoordsInFormat; ++tc)
        {
            texCoordDecls[tc] = vertDesc.getElementDecl( hkxVertexDescription::HKX_DU_TEXCOORD, tc );
            if ( texCoordDecls[tc] )
            {
                texCoordStride[tc] = texCoordDecls[tc]->m_byteStride;
                texCoordBuf[tc] = (char*)newVB->getVertexDataPtr( *texCoordDecls[tc] );
            }
        }

        int posStride = posDecl? posDecl->m_byteStride : 0;
        int normStride = normDecl? normDecl->m_byteStride : 0;
        int tangentStride = tangDecl? tangDecl->m_byteStride : 0;
        int binormStride = binormDecl? binormDecl->m_byteStride : 0;
        int weightsStride = weightsDecl? weightsDecl->m_byteStride : 0;
        int indicesStride = indicesDecl? indicesDecl->m_byteStride : 0;

        char* posBuf = static_cast<char*>( posDecl? newVB->getVertexDataPtr( *posDecl) : HK_NULL );
        char* normBuf = static_cast<char*>( normDecl? newVB->getVertexDataPtr( *normDecl) : HK_NULL );
        char* tangBuf = static_cast<char*>( tangDecl? newVB->getVertexDataPtr( *tangDecl) : HK_NULL );
        char* binormBuf = static_cast<char*>( binormDecl? newVB->getVertexDataPtr( *binormDecl) : HK_NULL );
        char* weightsBuf = static_cast<char*>( weightsDecl? newVB->getVertexDataPtr( *weightsDecl) : HK_NULL );
        char* indicesBuf = static_cast<char*>( indicesDecl? newVB->getVertexDataPtr( *indicesDecl) : HK_NULL );

        hkArray<bool> doneBones( influenceCount );
        int numVerts = newVB->getNumVertices();
        for( int vi = 0; vi < numVerts; ++vi )
        {
            const FullVert& fv = crtVtxBuffer[vi];
            const int vertexIndex = fv.reducedVertIndex;
            const int origVertexIndex = fv.origVertIndex;

            // position
            float* pos = (float*)( posBuf );
            pos[0] = reducedVertexArray[vertexIndex].x;
            pos[1] = reducedVertexArray[vertexIndex].y;
            pos[2] = reducedVertexArray[vertexIndex].z;
            pos[3] = 1;

      if (normDecl)
      {
              // normal
              float* normal = (float*)( normBuf );
              normal[0] = m_normalArray[fv.normalIndex].x;
              normal[1] = m_normalArray[fv.normalIndex].y;
              normal[2] = m_normalArray[fv.normalIndex].z;
              normal[3] = 0;
      }

            if ( tangDecl && binormDecl )
            {
                float* tangent = (float*)( tangBuf );
                tangent[0] = m_tangentsArray[fv.vertTangentIndex].x;
                tangent[1] = m_tangentsArray[fv.vertTangentIndex].y;
                tangent[2] = m_tangentsArray[fv.vertTangentIndex].z;
                tangent[3] = 0;

                float* binormal = (float*)( binormBuf );
                binormal[0] = m_binormalsArray[fv.vertTangentIndex].x;
                binormal[1] = m_binormalsArray[fv.vertTangentIndex].y;
                binormal[2] = m_binormalsArray[fv.vertTangentIndex].z;
                binormal[3] = 0;
            }

            // colour
            if (( numVColors > 0) && vcolorDecls[0])
            {
                hkUint32* color = (hkUint32*)( vcolorBuf[0] );
                hkUint32 col = 0xffffffff;
                if( fv.vertColorAIndex >= 0 )
                {
                    MColor c = m_colorArrayA[fv.vertColorAIndex];
                    col = hkSceneExportUtils::floatsToARGB_saturate( c.r, c.g, c.b, c.a );
                }
                *color = col;
            }
            if (( numVColors > 1) && vcolorDecls[1])
            {
                hkUint32* color = (hkUint32*)(  vcolorBuf[1] );
                hkUint32 col = 0xffffffff;
                if( fv.vertColorAIndex >= 0 )
                {
                    MColor c = m_colorArrayB[fv.vertColorBIndex];
                    col = hkSceneExportUtils::floatsToARGB_saturate( c.r, c.g, c.b, c.a );
                }
                *color = col;
            }

            // Weights, indices
            if( isSkinned )
            {
                // The weights are given in component order
                HK_ON_DEBUG(unsigned maxW = weights.length());

                unsigned startW = influenceCount * origVertexIndex;
                float wf[4];
                unsigned ii[4];

                // there is a weight per influence, mostly all 0. Pick the highest 4 weights and make sure to normalize
                memset( &doneBones[0], 0, sizeof(bool)*influenceCount);
                {
                    for( int b = 0; b < 4; ++b )
                    {
                        wf[b] = 0;
                        ii[b] = 0;
                        for( unsigned y = 0; y < influenceCount; y++ )
                        {
                            if ( !doneBones[y] )
                            {
                                HK_ASSERT_NO_MSG(0x3bb14e54, ( startW + y ) < maxW); // weights array should be big enough
                                const float w = weights[ startW + y ];
                                if( w > wf[b] )
                                {
                                    HK_ASSERT(0x4532e345, y < 65535, " Too many influences on this bone" );
                                    ii[b] = y;
                                    wf[b] = w;
                                }
                            }
                        }
                        doneBones[ ii[b] ] = true;
                    }
                    // normalize so that they sum to 1
                    float tw = wf[0] + wf[1] + wf[2] + wf[3];
                    wf[0] /= tw;
                    wf[1] /= tw;
                    wf[2] /= tw;
                    wf[3] /= tw;
                }

                hkUint32* idx = (hkUint32*)( indicesBuf );
                if(useInt8BoneIndices)
                {
                    *idx =  unsigned int ( ii[0] ) << 24 |
                        unsigned int ( ii[1] ) << 16 |
                        unsigned int ( ii[2] ) << 8  |
                        unsigned int ( ii[3] );
                }
                else if (!useInt8BoneIndices)
                {
                    *idx =  unsigned int ( ii[2] ) << 16 | unsigned int ( ii[3] ) ;
                    idx++;
                    *idx =  unsigned int ( ii[0] ) << 16 | unsigned int ( ii[1] ) ;
                }

                hkUint8 tempQWeights[4];
                {
                    hkReal tempWeights[4];
                    for (int i=0; i<4; i++)
                    {
                        tempWeights[i] = wf[i];
                    }

                    hkxSkinUtils::quantizeWeights(tempWeights, tempQWeights);
                }

                unsigned int compressedW =  unsigned int ( tempQWeights[0] ) << 24 |
                    unsigned int ( tempQWeights[1] ) << 16 |
                    unsigned int ( tempQWeights[2] ) << 8  |
                    unsigned int ( tempQWeights[3] );

                hkUint32* weight = (hkUint32*)( weightsBuf );
                *weight = compressedW;
            }

            // ensure valid index
            for (int i = 0; i < numTCoordsInFormat; i ++)
            {
                if (texCoordBuf[i])
                {
                    float* uv = (float*)( texCoordBuf[i] );
                    if ( fv.texCoordIndex[i] >= 0 && fv.texCoordIndex[i] < (int)m_uArray[i].length() && fv.texCoordIndex[i] < (int)m_vArray[i].length() )
                    {
                        uv[0] = m_uArray[i][fv.texCoordIndex[i]];
                        uv[1] = m_vArray[i][fv.texCoordIndex[i]];
                    }
                    else
                    {
                        uv[0] = uv[1] = -1.0f; // EXP-1892 Prevent random values in streams
                    }
                }
            }

            // next
            posBuf      += posStride;
            normBuf     += normStride;
            tangBuf     += tangentStride;
            binormBuf   += binormStride;
            weightsBuf  += weightsStride;
            indicesBuf  += indicesStride;

            for (int tcn=0; tcn < numTCoordsInFormat; ++tcn)
                texCoordBuf[tcn] += texCoordStride[tcn];

            for (int ccn=0; ccn < supportedNumColors; ++ccn)
                vcolorBuf[ccn] += vcolorStride[ccn];
        }

        // Register current section mapping
        m_userChannelUtil.registerSection(subMaps[cfg]);
    }

    return MStatus::kSuccess;
}


MStatus hctMayaMeshWriter::createSection( MObject& component, const hkxMaterial* material, hkArray<hkxMeshSection*> & outSections, bool forceSkinned )
{
    MStatus status;
    outSections.clear();

    HK_ASSERT_NO_MSG(0x38a2de11, material);

    // Get skinning info
    bool useInt8BoneIndices = true;
    unsigned totalNumUniqueInfluences = m_allSkinInfluences.length() > 0 ? m_allSkinInfluences.length() : (forceSkinned? 1 : 0);
    // Initialize the weights array
    MFloatArray weights;
    {
        weights.setLength( m_mesh.numVertices() * totalNumUniqueInfluences );
        float resetValue =  m_allSkinInfluences.length() > 0 ? 0.f : (forceSkinned? 1.f : 0.f);
        for( unsigned i=0; i<weights.length(); ++i )
        {
            weights[i] = resetValue;
        }
    }

    if (m_skinClusters.getSize() > 0)
    {
        for( int si=0; si < m_skinClusters.getSize(); ++si)
        {
            unsigned int numInfObjects = m_skinClusters[si].influenceIndexIntoMasterArray.getSize();

            bool isSkin = m_skinClusters[si].cluster.hasFn(MFn::kSkinClusterFilter);
            MFnSkinCluster fnSkinCluster(m_skinClusters[si].cluster);
            MFnWeightGeometryFilter fnJointCluster(m_skinClusters[si].cluster);
            MSelectionList rigidSelectionSet;
            if(!isSkin)
            {
                MFnSet fnSet(fnJointCluster.deformerSet());
                fnSet.getMembers(rigidSelectionSet, true);
            }

            // Iterate through the vertices
            MItGeometry itMeshGeometry( m_dagPath, component, &status );
            if( status != MStatus::kSuccess )
            {
                return status;
            }
            for( ; !itMeshGeometry.isDone(); itMeshGeometry.next() )
            {
                // get bone weights for this vertex
                unsigned int infCount = 0;
                MFloatArray wts;
                {
                    MObject vertexObj = itMeshGeometry.component( &status );
                    if( status != MStatus::kSuccess )
                    {
                        continue;   // couldn't get the vertex
                    }

                    if ( isSkin)
                    {
                        status = fnSkinCluster.getWeights( m_dagPath, vertexObj, wts, infCount );
                        if( status != MStatus::kSuccess )
                        {
                            continue;   // no weights..
                        }
                    }
                    else if (rigidSelectionSet.hasItem( m_dagPath, vertexObj ) )
                    {
                        status = fnJointCluster.getWeights( m_dagPath, vertexObj, wts );
                        if( status != MStatus::kSuccess )
                        {
                            continue;   // should not happen as in sel set
                        }
                        infCount = wts.length();
                    }
                    else
                    {
                        continue;
                    }
                }

                HK_ASSERT_NO_MSG(0x22515ff0, numInfObjects == infCount );
                HK_ASSERT_NO_MSG(0x22515ff0, wts.length() == infCount );

                // map vertex section index to object index, ensuring positive weighting
                unsigned int vertStartIndex = itMeshGeometry.index( &status ) * totalNumUniqueInfluences;
                for( unsigned i=0; i < numInfObjects; ++i )
                {
                    int mainJointIndex = m_skinClusters[si].influenceIndexIntoMasterArray[i];
                    const unsigned relIndex = vertStartIndex + mainJointIndex;
                    hkReal weight = wts[i];
                    HK_ASSERT_NO_MSG(0x45aac3ee, relIndex < weights.length() );
                    weights[ relIndex ] = hkMath::max2( weight, 0.0f );
                }
            }
        }

        int totalNumBones = 0;
        if(m_allSkinInfluences.length() > 1)
        {
            MObjectArray uniqueRootNodes;
            for(unsigned b = 0; b < m_allSkinInfluences.length(); b++)
            {
                MObject boneObj = m_allSkinInfluences[b].node();
                const MObject boneRoot = _getBoneRoot(boneObj);

                if(!boneRoot.isNull())
                {
                    bool alreadyAdded = false;
                    for(unsigned urb = 0; urb < uniqueRootNodes.length(); urb++)
                    {
                        if(boneRoot == uniqueRootNodes[urb])
                        {
                            alreadyAdded = true;
                            break;
                        }
                    }

                    if(!alreadyAdded)
                    {
                        uniqueRootNodes.append(boneRoot);
                    }
                }
            }

            for(unsigned urb = 0; urb < uniqueRootNodes.length(); urb++)
            {
                totalNumBones += _getNumberOfBonesInSkeleton(uniqueRootNodes[urb]);
            }

            if(totalNumBones > 255)
                useInt8BoneIndices = false;
        }
    }
    else if (forceSkinned)  // force rigid skinned
    {
        // weights all 1.f already
    }


    // Create the vertex buffer and the triangle indices
    hkArray<hkxVertexBuffer*>   newVBs;
    hkArray<MIntArray>          triangleIndexBuffers;

    status = createVertexBuffer( component, newVBs, triangleIndexBuffers, weights, totalNumUniqueInfluences, useInt8BoneIndices );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Allocate and fill the index buffers
    for(int ib = triangleIndexBuffers.getSize() - 1 ; ib >= 0 ; ib--)
    {
        hkxVertexBuffer * newVB             = newVBs[ib];
        MIntArray       & triangleIndices   = triangleIndexBuffers[ib];
        hkxIndexBuffer  * newIB             = HK_NULL;

        if( triangleIndices.length() > 0 )
        {
            newIB = new hkxIndexBuffer();
            newIB->m_indexType = hkxIndexBuffer::INDEX_TYPE_TRI_LIST;
            newIB->m_length = triangleIndices.length();
            newIB->m_vertexBaseOffset = 0;

            if( newVB->getNumVertices() > (int)0x0ffff ) // then 16 bits can't index
            {
                newIB->m_indices32.setSize( newIB->m_length );
                hkUint32* curIndex = newIB->m_indices32.begin();
                for( unsigned int i=0; i<newIB->m_length; ++i )
                {
                    *curIndex = (hkUint32)triangleIndices[i];
                    curIndex++;
                }
            }
            else
            {
                newIB->m_indices16.setSize( newIB->m_length );

                hkUint16* curIndex = newIB->m_indices16.begin();
                for( unsigned int i=0; i<newIB->m_length; ++i )
                {
                    *curIndex = (hkUint16)triangleIndices[i];
                    curIndex++;
                }
            }
        }

        // Allocate and fill the mesh section
        if( newVB && newIB )
        {
            hkxMeshSection *& newSection = outSections.expandOne();

            newSection = new hkxMeshSection();

            // vertex buffer
            newSection->m_vertexBuffer = newVB;

            // index buffer
            newSection->m_indexBuffers.setSize(1);
            newSection->m_indexBuffers[0] = newIB;

            // material
            newSection->m_material = (hkxMaterial*)material;
        }

        if (newIB) newIB->removeReference();
        if (newVB) newVB->removeReference();
    }

    return MStatus::kSuccess;
}


MStatus hctMayaMeshWriter::extractDiffuseMaps(  MFnDependencyNode & surfaceShaderNode,
    hkArray<hkxTextureFile*> & outDiffuseMaps,
    hkArray<hkUint32> & outDiffuseMapUVs,
    hkArray<int>& uniqueUvChooserIds,
    hkxMaterial*& outMaterial)
{
    outMaterial->m_userData = 0;

    MStatus status = MStatus::kSuccess;
    MPlug colorPlug = surfaceShaderNode.findPlug( "color", &status );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Try to find a blend colors node
    MItDependencyGraph itBlendDG( colorPlug, MFn::kBlendColors,
        MItDependencyGraph::kUpstream,
        MItDependencyGraph::kBreadthFirst,
        MItDependencyGraph::kNodeLevel,
        &status );


    if( (status == MStatus::kSuccess) && !itBlendDG.isDone() )
    {
        // Retrieve blend material parameters
        MFnDependencyNode blendFn( itBlendDG.thisNode() );

        //[EXP-2316] UV map info for Vision
        {
            hctMayaUtilities::getPlugValue(itBlendDG.thisNode(), "repeatU", outMaterial->m_uvMapScale[0] );
            hctMayaUtilities::getPlugValue(itBlendDG.thisNode(), "repeatV", outMaterial->m_uvMapScale[1] );
            hctMayaUtilities::getPlugValue(itBlendDG.thisNode(), "offsetU", outMaterial->m_uvMapOffset[0] );
            hctMayaUtilities::getPlugValue(itBlendDG.thisNode(), "offsetV", outMaterial->m_uvMapOffset[1] );
            hctMayaUtilities::getPlugValue(itBlendDG.thisNode(), "rotateUV", outMaterial->m_uvMapRotation );
        }

        // Diffuse texture 0
        {
            MPlug color1Plug = blendFn.findPlug( "color1", &status );
            if ( status == MStatus::kSuccess)
            {
                MItDependencyGraph itDG( color1Plug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );

                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    hkxTextureFile* diffuseTexture = HK_NULL;
                    status = createTexture( itDG.thisNode(), diffuseTexture );

                    if ( diffuseTexture )
                    {
                        int uvChooserId = -1;
                        outDiffuseMaps.expandOne()      = diffuseTexture;
                        outDiffuseMapUVs.expandOne()    = getMapUVSet(itDG.thisNode(), uvChooserId);
                        uniqueUvChooserIds.expandOne()  = uvChooserId;
                    }
                }
            }
        }

        // Diffuse texture 1
        {
            MPlug color2Plug = blendFn.findPlug( "color2", &status );
            if ( status == MStatus::kSuccess)
            {
                MItDependencyGraph itDG( color2Plug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );

                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    hkxTextureFile* diffuseTexture = HK_NULL;
                    status = createTexture( itDG.thisNode(), diffuseTexture );

                    if ( diffuseTexture )
                    {
                        int uvChooserId = -1;
                        outDiffuseMaps.expandOne()      = diffuseTexture;
                        outDiffuseMapUVs.expandOne()    = getMapUVSet(itDG.thisNode(), uvChooserId);
                        uniqueUvChooserIds.expandOne()  = uvChooserId;
                    }
                }
            }
        }

        // Diffuse texture 2 (blend factor)
        {
            MPlug color3Plug = blendFn.findPlug( "blender", &status );
            if ( status == MStatus::kSuccess)
            {
                MItDependencyGraph itDG( color3Plug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );

                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    hkxTextureFile* diffuseTexture = HK_NULL;
                    status = createTexture( itDG.thisNode(), diffuseTexture );

                    if ( diffuseTexture )
                    {
                        int uvChooserId = -1;
                        outDiffuseMaps.expandOne()      = diffuseTexture;
                        outDiffuseMapUVs.expandOne()    = getMapUVSet(itDG.thisNode(), uvChooserId);
                        uniqueUvChooserIds.expandOne()  = uvChooserId;
                    }
                }
            }
        }

        // Mark the material as blend
        outMaterial->addProperty(hkxMaterial::PROPERTY_MTL_TYPE_BLEND, 2);
    }
    else    // Unknown / default color binding. Get texture
    {
        MItDependencyGraph itDG( colorPlug, MFn::kFileTexture,
            MItDependencyGraph::kUpstream,
            MItDependencyGraph::kBreadthFirst,
            MItDependencyGraph::kNodeLevel,
            &status );
        if( (status == MStatus::kSuccess) && !itDG.isDone() )
        {
            hkxTextureFile* diffuseTexture = HK_NULL;
            status = createTexture( itDG.thisNode(), diffuseTexture );

            if ( diffuseTexture )
            {
                int uvChooserId = -1;
                outDiffuseMaps.expandOne()      = diffuseTexture;
                outDiffuseMapUVs.expandOne()    = getMapUVSet(itDG.thisNode(), uvChooserId);
                uniqueUvChooserIds.expandOne()  = uvChooserId;
            }
        }
    }

    // No error!
    return status;
}

//
//  Adds a dynamic attribute on the uvChooser dependency node, to be able to connect to it later on if necessary

static int addUvSetUniqueID(MFnDependencyNode& uvChooserNode, int& uniqueId)
{
    // Check if we have the custom attribute
    MStatus status;
    const MString fullName("havokUniqueUvChooserId");
    const MString briefName("hkUniqueUvId");

    if ( !uvChooserNode.hasAttribute(fullName) )
    {
        // Create the custom attribute
        MFnNumericAttribute fnAttr;
        double attrDefault = -1;

        MObject newAttr = fnAttr.create(fullName, briefName, MFnNumericData::kInt, attrDefault, &status);
        if ( status != MStatus::kSuccess )
        {
            return -1;  // Failed to create dynamic attribute!
        }

        // Add it on the uvChooser node
        status = uvChooserNode.addAttribute(newAttr, MFnDependencyNode::kLocalDynamicAttr);
        if ( status != MStatus::kSuccess )
        {
            return -1;
        }
    }

    // Property already existing. Do nothing
    {
        MPlug plug = uvChooserNode.findPlug(fullName, &status);
        if ( status != MStatus::kSuccess )
        {
            return -1;
        }

        // Get value set on the attribute
        int prevUniqueId = -1;
        status = plug.getValue(prevUniqueId);
        if ( status != MStatus::kSuccess )
        {
            return -1;
        }

        if ( prevUniqueId < 0 )
        {
            // Uninitialized. Must set new value now
            plug.setValue(uniqueId);

            uniqueId += 1;
            return uniqueId - 1;
        }
        else
        {
            // Previously set. Do nothing
            return prevUniqueId;
        }
    }
}

int hctMayaMeshWriter::getMapUVSet(const MObject& textureNode, int& uniqueUvChooserId)
{
    MObject shape = m_dagPath.node();

    // Init uvChooser Id to none
    uniqueUvChooserId = -1;

    // Go back until we find an UV chooser
    MStatus status;
    MObject node = textureNode;
    while(node.apiType() != MFn::kUvChooser )
    {
        MFnDependencyNode dgFn(node);
        MPlug plug = dgFn.findPlug(MString("uvCoord"), &status);
        if(status != MStatus::kSuccess || !plug.isConnected())
        {
            break;  // Error, stop looking!
        }

        // Get the connection - there can be at most one input to a plug
        MPlugArray connections;
        plug.connectedTo(connections, true, false);
        if(connections.length() <= 0)
        {
            break;  // No more connections, stop looking!
        }

        // Go back one node
        node = connections[ 0].node();
    }

    // See if we got an UV chooser
    if (node.apiType() == MFn::kUvChooser)
    {
        // Find the uvSet connection to this shape
        MFnDependencyNode dgFn(node);
        MPlug plug = dgFn.findPlug(MString("uvSets"), &status);
        if (status == MStatus::kSuccess && plug.isArray())
        {
            // Iterate over all the elements in this array looking for one which connects to our shape
            unsigned int numUVSets = plug.evaluateNumElements();
            for(unsigned int i = 0; i < numUVSets; i++)
            {
                MPlug uvSetPlug = plug.elementByPhysicalIndex(i, &status);
                if(!status || !uvSetPlug.isConnected())
                {
                    continue;
                }

                // Get the connection - there can be at most one input to a plug
                MPlugArray connections;
                uvSetPlug.connectedTo(connections, true, false);
                if(connections.length() > 0 && connections[0].node() == shape)
                {
                    // Connected through name element
                    MPlug uvSetElement= connections[0].parent();
                    MPlug uvSetArray = uvSetElement.array();
                    if (uvSetArray.isArray())
                    {
                        unsigned int logicalIndex = uvSetElement.logicalIndex();
                        for (unsigned int child = 0; child < uvSetArray.numElements(); ++child)
                        {
                            MPlug childPlug = uvSetArray.elementByPhysicalIndex(child);
                            if (childPlug.logicalIndex() == logicalIndex)
                            {
                                // Assign a unique Id to the uvChooser, to be able to reconnect with it on import if case
                                int uniqueAvailableId = (*m_currentUvChooserIdPtr);
                                uniqueUvChooserId = addUvSetUniqueID(dgFn, uniqueAvailableId);
                                (*m_currentUvChooserIdPtr) = uniqueAvailableId;

                                return child;
                            }
                        }
                    }
                }
            }
        }
    }

    // No uvChooser found! Return the default uvSet
    MFnMesh mesh(shape);
    MStringArray setNames;
    mesh.getUVSetNames(setNames);

    MPlug uvSetPlug = mesh.findPlug("uvSet");
    for (unsigned int i = 0; i < uvSetPlug.numElements(); i++)
    {
        MPlug uvSetElememtPlug = uvSetPlug.elementByPhysicalIndex(i);
        MPlug uvSetNamePlug = uvSetElememtPlug.child(0);

        MString uvSetName;
        uvSetNamePlug.getValue(uvSetName);
        if ( uvSetName == setNames[0] )
        {
            return i;
        }
    }

    // Should not get here!
    return 0;
}

//
//  Sets the uvChooser id into the material properties

inline void addUvChooserIdToMaterial(hkxMaterial* mtl, int textureStage, int uvChooserId)
{
    int propKey = hkxMaterial::PROPERTY_MTL_UV_ID_STAGE0 + textureStage;
    if ( propKey >= hkxMaterial::PROPERTY_MTL_UV_ID_STAGE_MAX)
    {
        HK_WARN_ALWAYS(0x2248055e, "Attempting to set the uvChooser id property for a texture stage > 15. Not supported!");
        return;
    }

    mtl->addProperty(propKey, uvChooserId);
}

MStatus hctMayaMeshWriter::createMaterial( const MObject& shadingGroup, hkxMaterial*& outMaterial )
{
    MStatus status;

    // Check if we exported this material already
    HK_ASSERT( 0xabba9914, m_owner->m_currentScene.m_materials.getSize() == (int)m_processedShadingGroups.length(),
        "Processed shading groups do not map correctly to exported materials" );
    for( unsigned int i=0; i<m_processedShadingGroups.length(); ++i )
    {
        if( m_processedShadingGroups[i] == shadingGroup )
        {
            outMaterial = m_owner->m_currentScene.m_materials[i];
            outMaterial->addReference();
            return MStatus::kSuccess;
        }
    }

    MObject surfaceShaderNode;
    MObject displacementShaderNode;
    if (!shadingGroup.isNull())
    {
        MFnDependencyNode shadingGroupFn( shadingGroup );

        // Get the shader nodes
        {
            MPlug surfaceShaderPlug = shadingGroupFn.findPlug( "surfaceShader" );
            if( !surfaceShaderPlug.isNull() )
            {
                MPlugArray connectedPlugs;
                surfaceShaderPlug.connectedTo( connectedPlugs, true, false, &status );
                if( status == MStatus::kSuccess && connectedPlugs.length() == 1 )
                {
                    surfaceShaderNode = connectedPlugs[0].node();
                }
                else
                {
                    hkStringOld str; str.printf( "Couldn't get surface shader for %s", m_mesh.partialPathName().asChar());
                    HK_WARN_ALWAYS(0xabba1763, str.cString() );
                }
            }

            MPlug displacementShaderPlug = shadingGroupFn.findPlug( "displacementShader" );
            if( !displacementShaderPlug.isNull() )
            {
                MPlugArray connectedPlugs;
                displacementShaderPlug.connectedTo( connectedPlugs, true, false, &status );
                if( status == MStatus::kSuccess && connectedPlugs.length() == 1 )
                {
                    displacementShaderNode = connectedPlugs[0].node();
                }
            }
        }
    }

    // Create a new material
    outMaterial = new hkxMaterial();

    // Set default properties
    outMaterial->m_diffuseColor.set( 1.0, 1.0, 1.0 );
    outMaterial->m_ambientColor.set( 0.0, 0.0, 0.0 );
    outMaterial->m_emissiveColor.set( 0.0, 0.0, 0.0 );
    outMaterial->m_specularColor.set( 1.0, 1.0, 1.0 );
    outMaterial->m_uvMapAlgorithm = hkxMaterial::UVMA_MAYA_STYLE;
    outMaterial->m_uvMapOffset[0] = 0; outMaterial->m_uvMapOffset[1] = 0;
    outMaterial->m_uvMapScale[0] = 1.f; outMaterial->m_uvMapScale[1] = 1.f;
    outMaterial->m_uvMapRotation = 0;
    outMaterial->m_specularMultiplier = 0.f;
    outMaterial->m_specularExponent = 1.f;
    outMaterial->m_transparency = hkxMaterial::transp_none;
    outMaterial->m_userData = 0;

    // Need at least one of the shaders
    if( shadingGroup.isNull() || ( surfaceShaderNode.isNull() && displacementShaderNode.isNull() ))
    {
        outMaterial->m_name = "Default Null Material";
        m_owner->m_currentScene.m_materials.pushBack( outMaterial );
        m_processedShadingGroups.append(MObject::kNullObj);
        return MStatus::kFailure;
    }


    // Set the name [EXP-2316]
    {
        MFnDependencyNode namedObject(surfaceShaderNode.isNull()? displacementShaderNode : surfaceShaderNode);
        outMaterial->m_name = namedObject.name().asChar();
    }


    // Process the surface shader
    hkArray<hkxTextureFile*> diffuseMaps;
    hkArray<hkUint32> diffuseUVSets;
    hkArray<int> diffuseUvChooserIds;

    hkxTextureFile* bumpTexture         = HK_NULL;
    hkxTextureFile* specularTexture     = HK_NULL;
    hkxTextureFile* reflectionTexture   = HK_NULL;
    hkxTextureFile* opacityTexture      = HK_NULL;

    hkUint32 bumpMapUVSet       = 0;
    hkUint32 specularMapUVSet   = 0;
    hkUint32 reflectionMapUVSet = 0;
    hkUint32 opacityMapUVSet    = 0;

    int bumpUvChooserId         = -1;
    int specularUvChooserId     = -1;
    int reflectionUvChooserId   = -1;
    int opacityUvChooserId      = -1;

    // Other (unknown) textures
    hkArray<hkxTextureFile*> otherMaps;
    hkArray<hkUint32> otherMapsUVSets;
    hkArray<int> otherUvChooserIds;

    float specularRolloff = 1.0f;

    if( !surfaceShaderNode.isNull() )
    {
        // Get the colors
        {
            // Get the base props using a Lambert shader
            MFnLambertShader lambertShaderFn( surfaceShaderNode );
            MColor diffuseColor = lambertShaderFn.color();
            MColor emissiveColor = lambertShaderFn.incandescence();
            MColor ambientColor = lambertShaderFn.ambientColor();

            // Get the specular props using a suitable shader type
            MColor specularColor( 0.0, 0.0, 0.0 );
            float specularPower = 0;
            if( surfaceShaderNode.hasFn( MFn::kPhong ) )
            {
                MFnPhongShader phongShaderFn( surfaceShaderNode );
                emissiveColor = phongShaderFn.incandescence();
                ambientColor = phongShaderFn.ambientColor();
                specularPower = phongShaderFn.cosPower();
            }
            else if( surfaceShaderNode.hasFn( MFn::kBlinn ) )
            {
                MFnBlinnShader blinnShaderFn( surfaceShaderNode );
                emissiveColor = blinnShaderFn.incandescence();
                ambientColor = blinnShaderFn.ambientColor();
                specularColor = blinnShaderFn.specularColor();
                specularRolloff = blinnShaderFn.specularRollOff();
                specularColor *= specularRolloff; // Use the specular rolloff to scale specular intensity
                specularPower = blinnShaderFn.eccentricity(); // 0..1 :  low eccentricity == high spec power roughly
                outMaterial->m_specularExponent = blinnShaderFn.eccentricity();
                specularPower = 100 - specularPower*100;
                if( specularPower<1 ) specularPower = 1; // min 1 as the spec power.
            }
            else if( surfaceShaderNode.hasFn( MFn::kReflect ) )
            {
                MFnReflectShader reflectShaderFn( surfaceShaderNode );
                specularColor = reflectShaderFn.specularColor();
            }
            else if(surfaceShaderNode.hasFn(MFn::kPhongExplorer))
            {
                const char* s = surfaceShaderNode.hasFn(MFn::kPhong) ? "cosinePower" : "roughness";
                MPlug plug = MFnDependencyNode(surfaceShaderNode).findPlug(s, &status);
                if(status == MS::kSuccess)
                {
                    plug.getValue(outMaterial->m_specularExponent);
                }
            }


            // Set the properties
            outMaterial->m_diffuseColor.set( diffuseColor.r, diffuseColor.g, diffuseColor.b, diffuseColor.a );
            outMaterial->m_ambientColor.set( ambientColor.r, ambientColor.g, ambientColor.b, ambientColor.a );
            outMaterial->m_emissiveColor.set( emissiveColor.r, emissiveColor.g, emissiveColor.b, emissiveColor.a );
            outMaterial->m_specularColor.set( specularColor.r, specularColor.g, specularColor.b, specularPower );
        }

        // Check for texture maps
        {
            MFnDependencyNode shaderFn( surfaceShaderNode );

            // diffuse texture
            status = extractDiffuseMaps(shaderFn, diffuseMaps, diffuseUVSets, diffuseUvChooserIds, outMaterial);
            if (diffuseMaps.getSize()>0 )
            {
                // EXP-1306: Maya sets diffuseColor to (0,0,0,0) if a diffuse texture is applied
                outMaterial->m_diffuseColor = hkVector4(1,1,1,1);
            }

            // bump map
            MPlug normalCameraPlug = shaderFn.findPlug( "normalCamera", &status );
            if( status == MStatus::kSuccess )
            {
                MItDependencyGraph itDG( normalCameraPlug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );
                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    status = createTexture( itDG.thisNode(), bumpTexture );
                    bumpMapUVSet = getMapUVSet(itDG.thisNode(), bumpUvChooserId);
                }
            }

            // specular map
            MPlug specularColorPlug = shaderFn.findPlug( "specularColor", &status );
            if( status == MStatus::kSuccess )
            {
                MItDependencyGraph itDG( specularColorPlug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );
                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    status = createTexture( itDG.thisNode(), specularTexture );
                    specularMapUVSet = getMapUVSet(itDG.thisNode(), specularUvChooserId);
                    outMaterial->m_specularMultiplier = 1.f;
                    if (status == MStatus::kSuccess)
                    {
                        // EXP-1306: Maya sets specularColor to (0,0,0,0) if a specular texture is applied.
                        // Replace that with (c, c, c, sp), where sp is the user-specified spec. power, and c is the user-specified spec. rolloff
                        hkReal specularPower = outMaterial->m_specularColor(3);
                        outMaterial->m_specularColor = hkVector4(specularRolloff, specularRolloff, specularRolloff, specularPower);
                        outMaterial->m_specularExponent = specularPower;
                    }
                    else
                    {
                        // Calculate the specular multiplier as RGB average when we have no specular texture
                        MObject data;
                        if(specularColorPlug.getValue(data) == MS::kSuccess)
                        {
                            float c[3];
                            MFnNumericData val(data);
                            val.getData(c[0], c[1], c[2]);
                            outMaterial->m_specularMultiplier = (c[0]+c[1]+c[2])/3.f;  // RGB average.
                        }
                    }

                    if(surfaceShaderNode.hasFn(MFn::kBlinn))
                    {
                        MPlug plug = MFnDependencyNode(surfaceShaderNode).findPlug("eccentricity", &status);
                        if(status == MS::kSuccess)
                        {
                            float fExponent = 0.f;
                            plug.getValue(fExponent);
                            fExponent = (fExponent < .03125f) ? 128.f : (4.f/fExponent);
                            outMaterial->m_specularExponent = fExponent;
                        }
                    }
                    else if (surfaceShaderNode.hasFn(MFn::kPhong) || surfaceShaderNode.hasFn(MFn::kPhongExplorer))
                    {
                        const char* s = surfaceShaderNode.hasFn(MFn::kPhong) ? "cosinePower" : "roughness";
                        MPlug plug = MFnDependencyNode(surfaceShaderNode).findPlug(s, &status);
                        if(status == MS::kSuccess)
                        {
                            float fExponent = 0.f;
                            plug.getValue(fExponent);
                            fExponent *= 4.f;
                            outMaterial->m_specularExponent = fExponent;
                        }
                    }
                }
            }

            // reflection map
            MPlug reflectedColorPlug = shaderFn.findPlug( "reflectedColor", &status );
            if( status == MStatus::kSuccess )
            {
                MItDependencyGraph itDG( reflectedColorPlug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );
                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    createTexture( itDG.thisNode(), reflectionTexture );
                    reflectionMapUVSet = getMapUVSet(itDG.thisNode(), reflectionUvChooserId);
                }
            }

            // opacity map
            MPlug opacityPlug = shaderFn.findPlug( "transparency", &status );
            if( status == MStatus::kSuccess )
            {
                MItDependencyGraph itDG( opacityPlug, MFn::kFileTexture,
                    MItDependencyGraph::kUpstream,
                    MItDependencyGraph::kBreadthFirst,
                    MItDependencyGraph::kNodeLevel,
                    &status );
                if( (status == MStatus::kSuccess) && !itDG.isDone() )
                {
                    createTexture( itDG.thisNode(), opacityTexture );
                    outMaterial->m_transparency = hkxMaterial::transp_alpha;
                    opacityMapUVSet = getMapUVSet(itDG.thisNode(), opacityUvChooserId);
                }
            }

            // All other maps
            {
                MItDependencyGraph itDG(surfaceShaderNode, MFn::kFileTexture, MItDependencyGraph::kUpstream);
        for (; !itDG.isDone(); itDG.next())
        {
                    MObject shaderTextureNode = itDG.thisNode();
                    MFnDependencyNode mayaTexture(shaderTextureNode);

                    MString textureFile;
                    MPlug texFilePlug = mayaTexture.findPlug("fileTextureName", &status);
                    if( status == MStatus::kSuccess )
                    {
                        texFilePlug.getValue(textureFile);

                        // Add the texture, it will be removed afterwards if added multiple times
                        hkxTextureFile*& texFile = otherMaps.expandOne();
                        hkUint32& texUV = otherMapsUVSets.expandOne();
                        int& texUvChooserId = otherUvChooserIds.expandOne();

                        texFile         = HK_NULL;
                        texUV           = 0xFFFFFFFF;
                        texUvChooserId  = -1;

                        createTexture( itDG.thisNode(), texFile );
                        texUV = getMapUVSet(itDG.thisNode(), texUvChooserId);

                        if ( !texFile || (texUV == 0xFFFFFFFF) )
                        {
                            int lastIdx = otherMaps.getSize() - 1;
                            if ( texFile )
                            {
                                texFile->removeReference();
                            }

                            otherMaps.removeAt(lastIdx);
                            otherMapsUVSets.removeAt(lastIdx);
                            otherUvChooserIds.removeAt(lastIdx);
                        }
                    }
                }
            }
        }
    }

    // Process the displacement shader
    hkxTextureFile* displacementTexture = HK_NULL;
    hkUint32 displacementMapUVSet       = 0;
    int displacementUvChooserId         = -1;
    if( !displacementShaderNode.isNull() )
    {
        MFnDependencyNode shaderFn( displacementShaderNode );

        // displacement texture
        MPlug displacementPlug = shaderFn.findPlug( "displacement", &status );
        if( status == MStatus::kSuccess )
        {
            MItDependencyGraph itDG( displacementPlug, MFn::kFileTexture,
                MItDependencyGraph::kUpstream,
                MItDependencyGraph::kBreadthFirst,
                MItDependencyGraph::kNodeLevel,
                &status );
            if( (status == MStatus::kSuccess) && !itDG.isDone() )
            {
                createTexture( itDG.thisNode(), displacementTexture );
                displacementMapUVSet = getMapUVSet(itDG.thisNode(), displacementUvChooserId);
            }
        }
    }

    // Remove known textures from otherMaps
    int numOtherMaps = otherMaps.getSize();
    for (int mi = numOtherMaps - 1; mi >= 0; mi--)
    {
        hkxTextureFile* otherTexFile = otherMaps[mi];
        const char* otherTexFileName = otherTexFile->m_filename.cString();

        // Check if among diffuse maps
        bool found = false;
        for (int dmi = diffuseMaps.getSize() - 1; dmi >= 0; dmi--)
        {
            hkxTextureFile* tf = diffuseMaps[dmi];
            if ( tf->m_filename.compareTo(otherTexFileName) == 0 )
            {
                found = true;
                break;
            }
        }

        // Check if same as bump texture
        if (!found &&
            bumpTexture &&
            (bumpTexture->m_filename.compareTo(otherTexFileName) == 0) )
        {
            found = true;
        }

        // Check if same as specular texture
        if (!found &&
            specularTexture &&
            (specularTexture->m_filename.compareTo(otherTexFileName) == 0) )
        {
            found = true;
        }

        // Check if same as reflection texture
        if (!found &&
            reflectionTexture &&
            (reflectionTexture->m_filename.compareTo(otherTexFileName) == 0) )
        {
            found = true;
        }

        // Check if same as displacement texture
        if (!found &&
            displacementTexture &&
            (displacementTexture->m_filename.compareTo(otherTexFileName) == 0) )
        {
            found = true;
        }

        // Check if same as opacity texture
        if (!found &&
            opacityTexture &&
            (opacityTexture->m_filename.compareTo(otherTexFileName) == 0) )
        {
            found = true;
        }

        // If the texture was found, it can be removed
        if ( found )
        {
            otherMaps[mi]->removeReference();
            otherMaps.removeAt(mi);
            otherMapsUVSets.removeAt(mi);
            otherUvChooserIds.removeAt(mi);
        }
    }

    // Allocate and fill the texture stages
    {
        int numStages = diffuseMaps.getSize() +
            (bumpTexture ? 1 : 0) +
            (specularTexture ? 1 : 0) +
            (reflectionTexture ? 1 : 0) +
            (displacementTexture ? 1 : 0) +
            (opacityTexture ? 1 : 0) +
            otherMaps.getSize();
        outMaterial->m_stages.setSize(numStages);

        int i = 0;
        for(int k = 0; k < diffuseMaps.getSize(); k++)
        {
            outMaterial->m_stages[i].m_texture = diffuseMaps[k];
            diffuseMaps[k]->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_DIFFUSE;
            outMaterial->m_stages[i].m_tcoordChannel = diffuseUVSets[k];
            addUvChooserIdToMaterial(outMaterial, i, diffuseUvChooserIds[k]);
            i++;
        }
        if( bumpTexture )
        {
            outMaterial->m_stages[i].m_texture = bumpTexture;
            bumpTexture->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_NORMAL;
            outMaterial->m_stages[i].m_tcoordChannel = bumpMapUVSet;
            addUvChooserIdToMaterial(outMaterial, i, bumpUvChooserId);
            i++;
        }
        if ( specularTexture )
        {
            outMaterial->m_stages[i].m_texture = specularTexture;
            specularTexture->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_SPECULAR;
            outMaterial->m_stages[i].m_tcoordChannel = specularMapUVSet;
            addUvChooserIdToMaterial(outMaterial, i, specularUvChooserId);
            i++;
        }
        if( reflectionTexture )
        {
            outMaterial->m_stages[i].m_texture = reflectionTexture;
            reflectionTexture->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_REFLECTION;
            outMaterial->m_stages[i].m_tcoordChannel = reflectionMapUVSet;
            addUvChooserIdToMaterial(outMaterial, i, reflectionUvChooserId);
            i++;
        }
        if( displacementTexture )
        {
            outMaterial->m_stages[i].m_texture = displacementTexture;
            displacementTexture->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_DISPLACEMENT;
            outMaterial->m_stages[i].m_tcoordChannel = displacementMapUVSet;
            addUvChooserIdToMaterial(outMaterial, i, displacementUvChooserId);
            i++;
        }
        if( opacityTexture )
        {
            outMaterial->m_stages[i].m_texture = opacityTexture;
            opacityTexture->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_OPACITY;
            outMaterial->m_stages[i].m_tcoordChannel = opacityMapUVSet;
            outMaterial->m_transparency = hkxMaterial::transp_alpha;
            addUvChooserIdToMaterial(outMaterial, i, opacityUvChooserId);
            i++;
        }

        // Add all other maps
        for (int k = 0; k < otherMaps.getSize(); k++)
        {
            outMaterial->m_stages[i].m_texture = otherMaps[k];
            otherMaps[k]->removeReference();
            outMaterial->m_stages[i].m_usageHint = hkxMaterial::TEX_UNKNOWN;
            outMaterial->m_stages[i].m_tcoordChannel = otherMapsUVSets[k];
            addUvChooserIdToMaterial(outMaterial, i, otherUvChooserIds[k]);
            i++;
        }
    }

    // Run material through the attribute processing utility (EXP-611)
    MPlugArray maya_animAttributes;
    hkArray<hkxAttribute*> hkx_animAttributes;
    if( !surfaceShaderNode.isNull() )
    {
        status = m_owner->addAttributeGroups( outMaterial, surfaceShaderNode );
        if( status != MStatus::kSuccess)
        {
            hkStringOld str; str.printf( "Couldn't export materials for surface shader of %s", m_mesh.partialPathName().asChar() );
            HK_WARN_ALWAYS(0xabba5ed8, str.cString() );
        }
    }
    if( !displacementShaderNode.isNull() )
    {
        status = m_owner->addAttributeGroups( outMaterial, displacementShaderNode );
        if( status != MStatus::kSuccess)
        {
            hkStringOld str; str.printf( "Couldn't export materials for displacement shader of %s", m_mesh.partialPathName().asChar() );
            HK_WARN_ALWAYS(0xabba78df, str.cString() );
        }
    }

    // Store this material for future reference
    m_owner->m_currentScene.m_materials.pushBack( outMaterial );
    m_processedShadingGroups.append( shadingGroup );
    return MStatus::kSuccess;
}


MStatus hctMayaMeshWriter::createTexture( const MObject& textureNode, hkxTextureFile*& outTexture )
{
    MStatus status;

    // Check for valid shader node
    if( textureNode.isNull() )
    {
        outTexture = HK_NULL;
        return MStatus::kFailure;
    }

    // Function set for the node
    MFnDependencyNode textureFn( textureNode );

    // Get the texture filename
    MString textureName;
    MString textureFileName;
    status = textureFn.findPlug( "fileTextureName" ).getValue( textureFileName );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Couldn't retrieve 'fileTextureName' plug" );
        return status;
    }

    textureName = textureFn.name(&status);
    if( status != MStatus::kSuccess )
    {
        textureName = "";
    }

    // Check if we exported this texture file already
    const hkArray< hkRefPtr<hkxTextureFile> >& exportedTexFiles = m_owner->m_currentScene.m_externalTextures;
    for( int i=0; i<exportedTexFiles.getSize(); ++i )
    {
        if( hkString::strCasecmp( exportedTexFiles[i]->m_filename, textureFileName.asChar() ) == 0 )
        {
            outTexture = exportedTexFiles[i];
            outTexture->addReference();
            return MStatus::kSuccess;
        }
    }


    // Allocate a new hkxTexture
    outTexture = new hkxTextureFile();

    // Set the filename
    outTexture->m_filename = textureFileName.asChar();
    outTexture->m_originalFilename = outTexture->m_filename;

    int len = textureName.length();
    if (len > 0)
    {
        outTexture->m_name = textureName.asChar();
    }


    // Store it
    m_owner->m_currentScene.m_externalTextures.pushBack( outTexture );
    return MStatus::kSuccess;
}


static hkStringOld _strippedName(bool& hadSuffix, const hkStringOld& trueName, const char* suffix)
{
    hkStringOld stripped(trueName);

    const char* pSuffix = hkString::strStr(trueName.cString(), suffix);
    hadSuffix = (pSuffix != HK_NULL);
    if (hadSuffix)
    {
        stripped = hkString::strNdup( trueName.cString(), static_cast<int>(pSuffix - trueName.cString()) );
    }

    return stripped;
}


MStatus hctMayaMeshWriter::extractSelectionChannels( const MDagPath& dagPath )
{
    MStatus status;

    MObjectArray selectionGroups;
    MObjectArray selectionComponents;

    // Determine which instance this mesh's DAG path refers to, if any
    int instanceNum = 0;
    if( m_dagPath.isInstanced() )
    {
        instanceNum = m_dagPath.instanceNumber();
    }
    status = m_mesh.getConnectedSetsAndMembers( instanceNum, selectionGroups, selectionComponents, false);

    hkArray<hctUserChannelUtil::GlobalChannel> candidateVertexSelectionChannels;
    hkArray<bool> candidateVertexSelectionHadSuffix;
    hkArray<hctUserChannelUtil::GlobalChannel> candidateTriangleSelectionChannels;
    hkArray<bool> candidateTriangleSelectionHadSuffix;
    hkArray<hctUserChannelUtil::GlobalChannel> candidateEdgeSelectionChannels;
    hkArray<bool> candidateEdgeSelectionHadSuffix;

    const int numGroups = selectionGroups.length();
    for (int g=0; g<numGroups; g++)
    {
        MObject group = selectionGroups[g];
        MObject component = selectionComponents[g];

        // Only interested in object sets
        MFnDependencyNode groupFn (group);
        {
            MString typeName = groupFn.typeName();
            if (typeName!="objectSet")
            {
                continue;
            }
        }

        MString setName = groupFn.name();

        // Handle the type
        MFnComponent componentFn (component);
        MFn::Type type = componentFn.type(&status);
        switch (type)
        {
        case MFn::kMeshVertComponent:
            {
                hctUserChannelUtil::GlobalChannel vertexSelection;

                // Strip anything after and including the Havok vertex selection suffix [HCL-302]
                const char* vertexSelectionSuffix = "_hctVertexSelection";
                hkStringOld trueName = setName.asChar();
                bool hadSuffix;

                vertexSelection.m_channelName = _strippedName(hadSuffix, trueName, vertexSelectionSuffix);
                vertexSelection.m_channelType = hctUserChannelUtil::CT_VERTEX_SELECTION;

                MItMeshVertex vertIt (dagPath, component, &status);

                hctUserChannelUtil::ChannelDataItem item;
                for( ; !vertIt.isDone(); vertIt.next() )
                {
                    item.m_index = vertIt.index();
                    vertexSelection.m_channelData.pushBack(item);
                }

                candidateVertexSelectionChannels.pushBack(vertexSelection);
                candidateVertexSelectionHadSuffix.pushBack(hadSuffix);
                break;
            }

        case MFn::kMeshPolygonComponent:
            {
                hctUserChannelUtil::GlobalChannel triangleSelection;

                // Strip anything after and including the Havok face selection suffix [HCL-302]
                const char* faceSelectionSuffix = "_hctFaceSelection";
                hkStringOld trueName = setName.asChar();
                bool hadSuffix;

                triangleSelection.m_channelName = _strippedName(hadSuffix, trueName, faceSelectionSuffix);
                triangleSelection.m_channelType = hctUserChannelUtil::CT_FACE_SELECTION;

                MItMeshPolygon polyIt (dagPath, component, &status);

                hctUserChannelUtil::ChannelDataItem item;
                for( ; !polyIt.isDone(); polyIt.next() )
                {
                    item.m_index = polyIt.index();
                    triangleSelection.m_channelData.pushBack(item);
                }

                candidateTriangleSelectionChannels.pushBack(triangleSelection);
                candidateTriangleSelectionHadSuffix.pushBack(hadSuffix);
                break;
            }

        case MFn::kMeshEdgeComponent:
            {
                hctUserChannelUtil::GlobalChannel edgeSelection;

                // Strip anything after and including the Havok face selection suffix [HCL-302]
                const char* edgeSelectionSuffix = "_hctEdgeSelection";
                hkStringOld trueName = setName.asChar();
                bool hadSuffix;

                edgeSelection.m_channelName = _strippedName(hadSuffix, trueName, edgeSelectionSuffix);
                edgeSelection.m_channelType = hctUserChannelUtil::CT_EDGE_SELECTION;

                MItMeshEdge edgeIt (dagPath, component, &status);

                for( ; !edgeIt.isDone(); edgeIt.next() )
                {
                    // edge index refers to the mesh's global array of edges that are is for defining faces (and not
                    // triangles) therefore we need to look for the triangles which contain this edge manually...
                    MItMeshPolygon polyIter( dagPath );
                    int2 edgeVertexIndices;
                    m_mesh.getEdgeVertices(edgeIt.index(),edgeVertexIndices);

                    int globalTriIndex = -1;

                    for( ; !polyIter.isDone(); polyIter.next() )
                    {
                        // Get the triangulation data
                        MIntArray faceTriIndices; // Triangulation indices
                        MPointArray dummy;
                        polyIter.getTriangles( dummy, faceTriIndices );

                        HK_ASSERT( 0xabba98d9, (faceTriIndices.length() % 3)==0, "Triangulation failed");

                        const int faceTriCount = (int)faceTriIndices.length();
                        for (int i=0; i<faceTriCount; ++i)
                        {
                            if (i%3==0)
                            {
                                ++globalTriIndex;
                            }

                            int2 currentEdge;
                            currentEdge[0] = faceTriIndices[i];
                            currentEdge[1] = faceTriIndices[i + (i+1)%3 - i%3];

                            // look if the edge belongs in the face (else it has been created by the triangulation and is considered
                            // to be "hidden")

                            if ((currentEdge[0] == edgeVertexIndices[0] && currentEdge[1] == edgeVertexIndices[1]) ||
                                (currentEdge[0] == edgeVertexIndices[1] && currentEdge[1] == edgeVertexIndices[0]) )
                            {
                                hctUserChannelUtil::ChannelDataItem item;
                                item.m_index = 3*globalTriIndex + i%3;
                                edgeSelection.m_channelData.pushBack(item);
                            }
                        }
                    }
                }

                candidateEdgeSelectionChannels.pushBack(edgeSelection);
                candidateEdgeSelectionHadSuffix.pushBack(hadSuffix);
                break;
            }
        }
    }

    hkStringOld meshName = m_mesh.name().asChar();
    if (candidateVertexSelectionChannels.getSize()>0)
    {
        addExportableSelections(candidateVertexSelectionChannels, candidateVertexSelectionHadSuffix, meshName);
    }
    if (candidateTriangleSelectionChannels.getSize()>0)
    {
        addExportableSelections(candidateTriangleSelectionChannels, candidateTriangleSelectionHadSuffix, meshName);
    }
    if (candidateEdgeSelectionChannels.getSize()>0)
    {
        addExportableSelections(candidateEdgeSelectionChannels, candidateEdgeSelectionHadSuffix, meshName);
    }

    return MStatus::kSuccess;
}


struct UniqueName
{
    UniqueName(const hkStringOld& name, bool channelHadSuffix) : m_name(name), m_wasUnique(true), m_numDuplicates(0), m_channelHadSuffix(channelHadSuffix)  {}
    hkStringOld m_name;
    bool m_wasUnique;
    int m_numDuplicates;
    hctUserChannelUtil::ChannelType m_channelType;
    bool m_channelHadSuffix;
};

void hctMayaMeshWriter::addExportableSelections( const hkArray<hctUserChannelUtil::GlobalChannel>& candidateSelectionChannels,
    const hkArray<bool>& candidateSelectionHadSuffix, const hkStringOld& meshName )
{
    // There may be more than one selection with the same suffix-stripped name [HCL-302]. We deal with this as follows.
    // a) If at least one had a havok suffix  (i.e. was created in the channel tools), keep the first of those, OR
    // b) If none had a havok suffix, just keep the first one.

    // First find the set of unique stripped names
    hkArray<UniqueName> uniqueStrippedNames;
    for (int i=0; i<candidateSelectionChannels.getSize(); ++i)
    {
        const hctUserChannelUtil::GlobalChannel& channel = candidateSelectionChannels[i];
        const hkStringOld& name = channel.m_channelName;

        hkBool unique = true;
        for (int j=0; j<uniqueStrippedNames.getSize(); ++j)
        {
            if ( uniqueStrippedNames[j].m_name == name )
            {
                unique = false;
                uniqueStrippedNames[j].m_wasUnique = false;
                uniqueStrippedNames[j].m_numDuplicates++;
                uniqueStrippedNames[j].m_channelType = channel.m_channelType;
                break;
            }
        }
        if (unique)
        {
            UniqueName uniqueName(name, candidateSelectionHadSuffix[i]);
            uniqueStrippedNames.pushBack(uniqueName);
        }
    }

    // If some stripped names occurred more than once, we warn that all but one of the duplicates will be ignored
    for (int i=0; i<uniqueStrippedNames.getSize(); ++i)
    {
        const UniqueName& uniqueName = uniqueStrippedNames[i];
        if (!uniqueName.m_wasUnique)
        {
            hkStringOld typeName;
            switch(uniqueName.m_channelType)
            {
            case hctUserChannelUtil::CT_VERTEX_SELECTION:
                typeName = "vertex selections";
                break;
            case hctUserChannelUtil::CT_FACE_SELECTION:
                typeName = "face selections";
                break;
            case hctUserChannelUtil::CT_EDGE_SELECTION:
                typeName = "edge selections";
                break;
            default:
                typeName = "selections (of unknown type)"; break;
            }

            hkStringOld numCopies; numCopies.printf("%d", uniqueName.m_numDuplicates+1);
            HK_WARN_ALWAYS(0xabba2cdd, "Found "<< numCopies.cString() << " " << typeName.cString()<< " on mesh "<<meshName.cString() \
                << " with the same name: \"" << uniqueName.m_name.cString() << "\", " << (uniqueName.m_channelHadSuffix ? "ignoring all but one." : "merging them all into one selection."));
        }
    }

    for (int i=0; i<uniqueStrippedNames.getSize(); ++i)
    {
        const hkStringOld& name = uniqueStrippedNames[i].m_name;

        // a) Any tool created ones? Choose the first such found.
        bool foundToolCreatedSelection = false;
        for (int c=0; c<candidateSelectionChannels.getSize(); ++c)
        {
            if ( candidateSelectionChannels[c].m_channelName == name && candidateSelectionHadSuffix[c] )
            {
                foundToolCreatedSelection = true;
                m_userChannelUtil.addGlobalChannel(candidateSelectionChannels[c]);
                break;
            }
        }
        // b) Otherwise, merge them (EXP-2964)
        if (!foundToolCreatedSelection)
        {
            hctUserChannelUtil::GlobalChannel mergedChannel;
            bool mergedChannelCreated = false;

            for (int c=0; c<candidateSelectionChannels.getSize(); ++c)
            {
                if ( candidateSelectionChannels[c].m_channelName == name )
                {
                    if (uniqueStrippedNames[i].m_wasUnique)
                    {
                        m_userChannelUtil.addGlobalChannel(candidateSelectionChannels[c]);
                    }
                    else
                    {
                        if (!mergedChannelCreated)
                        {
                            // Initialise merged channel with the data of the first channel we find
                            mergedChannel = candidateSelectionChannels[c];
                            mergedChannelCreated = true;
                        }
                        else
                        {
                            // Append only channel data
                            mergedChannel.m_channelData.append(candidateSelectionChannels[c].m_channelData);
                        }
                    }
                }
            }

            if (mergedChannelCreated)
            {
                // Add merged channel
                m_userChannelUtil.addGlobalChannel(mergedChannel);
            }
        }
    }
}


MStatus hctMayaMeshWriter::extractFloatChannels( const MDagPath& dagPath )
{
    MStatus status;

    // Extract the painted attributes from the mesh (if any)
    hkArray<hctUserChannelUtil::GlobalChannel> sampleChannels;

    MObject meshObject = dagPath.node();
    MFnDagNode meshDagFn( meshObject, &status );

    MFnMesh meshFn(dagPath);
    unsigned int numVerts = meshFn.numVertices(&status);
    if( status != MStatus::kSuccess )
    {
        status.perror( "Can't count mesh vertices" );
        return status;
    }

    // Iterate over all dynamic multi/array attributes of the mesh (ignoring sub-attributes and hidden attributes)
    int attributeCount = meshDagFn.attributeCount();

    for ( int i=0; i<attributeCount; ++i )
    {
        MObject attrObject = meshDagFn.attribute(i);
        MFnAttribute attribute( attrObject, &status );
        if( status != MStatus::kSuccess )
        {
            status.perror( "Can't access attribute" );
            continue;
        }

        // Get attribute name and plug
        hkStringOld attributeName( ( attribute.name() ).asChar() );
        MPlug attributePlug( meshObject, attribute.object() );

        if( !attributePlug.isChild() && !attribute.isHidden() && attributePlug.isArray() && attributePlug.isDynamic() )
        {
            // Store the raw float array in a temporary user channel
            hctUserChannelUtil::GlobalChannel channel;
            channel.m_channelName = attributeName;
            channel.m_channelType = hctUserChannelUtil::CT_VERTEX_FLOAT;

            // There may not be a float attribute per-vertex, since some may not be stored if they have the default value.
            // Using elementByLogicalIndex ensures we get the default value for the float in that case.
            channel.m_channelData.setSize( numVerts );

            float value;
            for( unsigned int vi=0; vi<numVerts; ++vi )
            {
                MPlug plugElement = attributePlug.elementByLogicalIndex( vi );
                plugElement.getValue( value );
                channel.m_channelData[vi].m_float = value;
            }

            // Empty channels are not added
            if (numVerts != 0) sampleChannels.pushBack(channel);
        }
    }

    // If mesh has no raw float arrays, return.
    if ( !sampleChannels.getSize() ) return MStatus::kSuccess;

    // Look for Havok channel node siblings of the shape node
    MDagPath parentPath = dagPath; parentPath.pop();
    unsigned int numChildren = parentPath.childCount();

    for( unsigned int i=0; i<numChildren; ++i )
    {
        MFnDagNode node( parentPath.child(i) );
        if ( node.typeName() != "hkNodeChannel" ) continue;

        MDagPath channelNodePath = parentPath;
        channelNodePath.push( parentPath.child(i) );

        MObject channelNode = channelNodePath.node();
        MFnDagNode nodeFn( channelNode, &status );
        if( status != MStatus::kSuccess ) return status;

        // Extract name of raw channel referred to by this channel node
        MString rawChannelName;
        {
            nodeFn.findPlug( "channelName", &status ).getValue( rawChannelName );
            if( status != MStatus::kSuccess ) return status;
        }

        // Get the name of this channel node (which becomes the name of the exported channel)
        MString channelNodeName = node.name();

        // Extract channel scaling
        int rescale;
        {
            nodeFn.findPlug( "rescaleEnable", &status ).getValue( rescale );
            if( status != MStatus::kSuccess ) return status;
        }

        float chanMin = 0.0f;
        float chanMax = 1.0f;
        float chanDiff = 1.0f;
        if (rescale)
        {
            nodeFn.findPlug( "rescaleMin" ).getValue( chanMin );
            if( status != MStatus::kSuccess ) return status;
            nodeFn.findPlug( "rescaleMax" ).getValue( chanMax );
            if( status != MStatus::kSuccess ) return status;
            chanDiff = chanMax - chanMin;
        }

        // Extract channel dimension type
        enum
        {
            DIM_FLOAT=0,
            DIM_DISTANCE=1,
            DIM_ANGLE=2
        };
        int dimensionType;
        {
            nodeFn.findPlug( "dimensionType", &status ).getValue( dimensionType );
            if( status != MStatus::kSuccess ) return status;
        }

        // Find the extracted float array attribute in the mesh with the name stored in the channel node
        for (int c=0; c<sampleChannels.getSize(); ++c)
        {
            if ( !(rawChannelName == sampleChannels[c].m_channelName.cString()) ) continue;

            // Create a new channel (copy the raw channel initially)
            hctUserChannelUtil::GlobalChannel newChannel( sampleChannels[c] );
            newChannel.m_channelName = channelNodeName.asChar();

            switch (dimensionType)
            {
            case DIM_FLOAT:     newChannel.m_channelDimensions = hctUserChannelUtil::CD_FLOAT;    break;
            case DIM_DISTANCE:  newChannel.m_channelDimensions = hctUserChannelUtil::CD_DISTANCE; break;
            case DIM_ANGLE:     newChannel.m_channelDimensions = hctUserChannelUtil::CD_ANGLE;    break;
            default:            newChannel.m_channelDimensions = hctUserChannelUtil::CD_INVALID;  break;
            }

            if (rescale)
            {
                newChannel.m_scaleMin = chanMin;
                newChannel.m_scaleMax = chanMax;
            }

            m_userChannelUtil.addGlobalChannel( newChannel );
            break;
        }
    }

    return status;
}

static void addHiddenEdgesChannel(const MDagPath& dagPath, const MFnMesh& mesh, hctUserChannelUtil& userChannelUtil)
{
    hctUserChannelUtil::GlobalChannel invisibleEdgeChannel;
    invisibleEdgeChannel.m_channelName = "hidden_edges";
    invisibleEdgeChannel.m_channelType = hctUserChannelUtil::CT_EDGE_SELECTION;

    MItMeshPolygon polyIter( dagPath );

    int globalTriIndex = -1;

    for( ; !polyIter.isDone(); polyIter.next() )
    {
        //const int polyIndex = polyIter.index();

        // Get the triangulation data
        MIntArray faceEdgeIndices;
        MIntArray faceTriIndices; // Triangulation indices
        MPointArray dummy;
        polyIter.getTriangles( dummy, faceTriIndices );
        polyIter.getEdges(faceEdgeIndices);

        HK_ASSERT( 0xabba98d9, (faceTriIndices.length() % 3)==0, "Triangulation failed");

        const int faceEdgeCount = (int)faceEdgeIndices.length();
        const int faceTriCount = (int)faceTriIndices.length();
        for (int i=0; i<faceTriCount; ++i)
        {
            if (i%3==0)
            {
                ++globalTriIndex;
            }

            int2 currentEdge;
            currentEdge[0] = faceTriIndices[i];
            currentEdge[1] = faceTriIndices[i + (i+1)%3 - i%3];

            // look if the edge belongs in the face (else it has been created by the triangulation and is considered
            // to be "hidden")
            bool edgeFound = false;
            for (int j=0; j<faceEdgeCount; ++j)
            {
                int2 faceEdgeVertices;
                mesh.getEdgeVertices(faceEdgeIndices[j],faceEdgeVertices);

                if ((currentEdge[0] == faceEdgeVertices[0] && currentEdge[1] == faceEdgeVertices[1]) ||
                    (currentEdge[0] == faceEdgeVertices[1] && currentEdge[1] == faceEdgeVertices[0]) )
                {
                    edgeFound = true;
                    break;
                }
            }

            if (!edgeFound)
            {
                hctUserChannelUtil::ChannelDataItem item;
                item.m_index = 3*globalTriIndex + i%3;
                invisibleEdgeChannel.m_channelData.pushBack(item);
            }
        }
    }

    // add hidden edges if found any
    if (invisibleEdgeChannel.m_channelData.getSize()>0)
    {
        userChannelUtil.addGlobalChannel(invisibleEdgeChannel);
    }
}




// Create a mesh, and a skin binding if necessary
MStatus hctMayaMeshWriter::write( const MDagPath& dagPath, hkxMesh*& writtenMesh, hkxSkinBinding*& writtenSkin, bool forceSkinned )
{
    MStatus status;

    if( !m_owner )
    {
        status.perror( "Mesh writer has no owner" );
        return MStatus::kFailure;
    }

    // Extract the object's data
    status = extract( dagPath );
    if( status != MStatus::kSuccess )
    {
        return status;
    }

    // Get the shading groups and their component polygons
    MObjectArray shadingGroups;
    MObjectArray polygonComponents;
    {
        // Determine which instance this mesh's DAG path refers to, if any
        int instanceNum = 0;
        if( m_dagPath.isInstanced() )
        {
            instanceNum = m_dagPath.instanceNumber();
        }

        // Get the connected sets and members
        status = m_mesh.getConnectedSetsAndMembers( instanceNum, shadingGroups, polygonComponents, true );
        if( status != MStatus::kSuccess )
        {
            status.perror( "MFnMesh::getConnectedSetsAndMembers failed" );
            return status;
        }
    }

    // Extract float and selection user channels
    m_userChannelUtil.clear();

    status = extractFloatChannels( dagPath );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Float channel extraction failed" );
    }

    status = extractSelectionChannels( dagPath );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Selection channel extraction failed" );
    }

    addHiddenEdgesChannel(m_dagPath,m_mesh,m_userChannelUtil);

    // See if we have animation curves on the mesh
    hkArray<MTime> keyTimes;
    hkArray<MTime> linearKeyFrameHints;
    if ( m_owner->m_options.m_exportVertexAnimations )
    {
        if (m_owner->m_options.m_forceVertexAnimationSamplePoints.getSize() > 0)
        {
            keyTimes.insertAt(0, m_owner->m_options.m_forceVertexAnimationSamplePoints.begin(), m_owner->m_options.m_forceVertexAnimationSamplePoints.getSize());

            // EXP-2435 save key frames
            hctUtilities::getKeyTimes(m_dagPath, linearKeyFrameHints, m_owner->m_options.m_startTime, m_owner->m_options.m_endTime );

            // EXP-2838 Add key frames as well as m_forceVertexAnimationSamplePoints may not correlate with the key frames at all.
            keyTimes.insertAt(0, linearKeyFrameHints.begin(), linearKeyFrameHints.getSize());
            hkSceneExportUtils::sortAndRemoveDuplicates(keyTimes);
        }
        else // look for keys
        {
            hctUtilities::getKeyTimes(m_dagPath, keyTimes, m_owner->m_options.m_startTime, m_owner->m_options.m_endTime );
        }
    }


    // Create a user channel of per-vertex ints, to associate each exported vertex in each section with the original Maya normal indices (the normal 'ids').
    // Vertices which share a normal id we know should be rendered with the same normal, even though they may be exported as different vertices due
    // to differing UVs. This is used in Cloth to preserve hard/soft edges. A vertex int channel with the name "normalIds" is expected to be present by Cloth.
    // Note that each section will have in general completely different normal IDs. So this is not like a float channel, where vertices in different
    // sections corresponding to the same geometric vertex get the same float value as stored per-geometric-vertex in the channel data. Here, the
    // int is stored in a per-section map m_sectionVertexIdToGlobalNormalId, not in the channel data.
    // Therefore we use channel type CT_NORMAL_IDS for this specific case.
    hctUserChannelUtil::GlobalChannel normalIdChannel;
    normalIdChannel.m_channelName = hkStringOld("normalIds");
    normalIdChannel.m_channelType = hctUserChannelUtil::CT_NORMAL_IDS;


    // Create the mesh sections
    hkArray< hkRefPtr<hkxMeshSection> > meshSections;
    {
        // Each component forms a section, using the shader's material
        unsigned int compCount = polygonComponents.length();
        for( unsigned int i=0; i<compCount; i++ )
        {
            MObject component = polygonComponents[i];

            // create a material if necessary
            hkxMaterial* material = HK_NULL;
            if( m_owner->m_options.m_exportMaterials )
            {
                if ( createMaterial( shadingGroups[i], material ) != MStatus::kSuccess)
                {
                    hkStringOld str; str.printf( "Failed to find supported surface shader for material : %s", m_mesh.partialPathName().asChar());
                    HK_WARN_ALWAYS(0xabba1768, str.cString());
                }
            }

            // create and store the section
            hkArray<hkxMeshSection*> sections;
            status = createSection( component, material, sections, forceSkinned);

            if (material)
            {
                material->removeReference();
            }

            if( status != MStatus::kSuccess )
            {
                return status;
            }

            for(int sectionIdx = 0 ; sectionIdx < sections.getSize() ; ++sectionIdx )
            {
                hkxMeshSection* section = sections[sectionIdx];
                meshSections.pushBack( section );
                section->removeReference();
            }
        }

        // Any unexported polygons form another section, with default material
        MIntArray unexportedPolygonIndices;
        {
            long polyCount = m_exportedPolygons.getSize();
            for( long i=0; i<polyCount; ++i )
            {
                if( !m_exportedPolygons[i] )
                {
                    unexportedPolygonIndices.append(i);
                }
            }
        }
        if( unexportedPolygonIndices.length() > 0 )
        {
            MFnSingleIndexedComponent comp;
            comp.addElements( unexportedPolygonIndices );
            MObject compObj( comp.object() );

            hkArray<hkxMeshSection*> sections;
            hkxMaterial* defaultMaterial;
            createMaterial( MObject::kNullObj, defaultMaterial );
            createSection( compObj, defaultMaterial, sections, forceSkinned);

            for (int sectionIdx = 0; sectionIdx < sections.getSize(); ++sectionIdx)
            {
                hkxMeshSection * section = sections[sectionIdx];
                meshSections.pushBack( section );
                section->removeReference();
            }
        }
    }

    m_userChannelUtil.addGlobalChannel(normalIdChannel);

    // Write out the mesh sections
    if( meshSections.getSize() > 0 )
    {
        // allocate and fill the mesh object
        writtenMesh = new hkxMesh();
        writtenMesh->m_sections.setSize( meshSections.getSize() );
        for( int cs =0; cs < writtenMesh->m_sections.getSize(); ++cs )
        {
            writtenMesh->m_sections[cs] = meshSections[cs];
        }

        // also create a skin object if necessary
        if( (m_skinClusters.getSize() > 0) || forceSkinned)
        {
            MMatrix matrix;

            // allocate the skin storage
            writtenSkin = new hkxSkinBinding();

            // store the mesh pointer
            writtenSkin->m_mesh = writtenMesh;


            // Set the initial transform.
            // The pivot offset may have been removed from the mesh vertices,
            // so we should add it into the this transform.
            matrix = m_dagPath.inclusiveMatrix();
            if( m_owner->m_options.m_useRotatePivot )
            {
                MDagPath parentPath( m_dagPath );
                parentPath.pop();
                MFnTransform parentTransformFn( parentPath, &status );
                if( status == MStatus::kSuccess )
                {
                    MPoint pivot = parentTransformFn.rotatePivot( MSpace::kPreTransform );
                    MMatrix pivotMatrix = MMatrix::identity;
                    pivotMatrix(3,0) = pivot.x;
                    pivotMatrix(3,1) = pivot.y;
                    pivotMatrix(3,2) = pivot.z;
                    matrix = pivotMatrix * matrix;
                }
            }
            hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( matrix, writtenSkin->m_initSkinTransform );

            // allocate the bones storage
            unsigned int numBones = m_allSkinInfluences.length() > 0? m_allSkinInfluences.length() : (forceSkinned? 1 : 0);
            writtenSkin->m_nodeNames.setSize(numBones);
            writtenSkin->m_bindPose.setSize(numBones);

            if (m_skinClusters.getSize() > 0)
            {
                // find the 'bindPreMatrix' compound plug
                hkArray<bool> skinClusterBoundToPose(numBones,false);
                for (int si=0; si < m_skinClusters.getSize(); ++si)
                {
                    if ( m_skinClusters[si].influenceLogicalIndices.getSize() < 1)
                        continue; // not a skin cluster (deal with it later)

                    MFnSkinCluster skinGeomFilter(m_skinClusters[si].cluster);
                    MPlug bindPreMatrixPlug = skinGeomFilter.findPlug( "bindPreMatrix", &status );
                    if( status == MStatus::kFailure )
                    {
                        status.perror( "Failed to retrieve 'bindPreMatrix' plug!" );
                    }

                    // fill the mapping data
                    int numInfluencesOnThisCluster = m_skinClusters[si].influenceLogicalIndices.getSize();
                    for( int i=0; i<numInfluencesOnThisCluster; ++i )
                    {
                        unsigned int influenceIndex = m_skinClusters[si].influenceLogicalIndices[i];

                        // retrieve the bind pre-matrix for this joint
                        MObject matrixData;
                        bindPreMatrixPlug.elementByLogicalIndex( influenceIndex ).getValue( matrixData );
                        MFnMatrixData matrixDataFn( matrixData, &status );
                        if( status != MStatus::kSuccess )
                        {
                            status.perror( "Failed to create matrix function set!" );
                        }

                        // Store the inverse of the matrix
                        matrix = matrixDataFn.matrix().inverse();

                        int mainBoneIndex = m_skinClusters[si].influenceIndexIntoMasterArray[i];
                        skinClusterBoundToPose[mainBoneIndex] = true;
                        hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( matrix, writtenSkin->m_bindPose[mainBoneIndex]  );
                    }
                }

                // for all joints not a skin cluster, try a few things
                for (int bi=0; bi < skinClusterBoundToPose.getSize(); ++bi)
                {
                    // Store the allocated HKX node for the joint (may be null if the joint isn't being exported)
                    const hkxNode* tmpNode = m_owner->getHkxNode( m_allSkinInfluences[bi] );
                    if ( tmpNode != HK_NULL )
                    {
                        writtenSkin->m_nodeNames[bi] = tmpNode->m_name;
                    }

                    if (skinClusterBoundToPose[bi])
                        continue;

                    MFnIkJoint fnJoint(m_allSkinInfluences[bi]);
                    MPlug bindPoseMatrixPlug = fnJoint.findPlug("bindPose", &status);
                    MObject matrixObject;
                    if(status == MStatus::kSuccess)
                    {
                        status = bindPoseMatrixPlug.getValue(matrixObject);
                    }

                    // In the worst case, fall back to just grabbing the current world matrix.
                    if(status != MStatus::kSuccess)
                    {
                        MPlug jointWorldMatrixPlug = fnJoint.findPlug("worldMatrix").elementByLogicalIndex(0);
                        jointWorldMatrixPlug.getValue(matrixObject);
                    }

                    if(matrixObject != MObject::kNullObj)
                    {
                        matrix = MFnMatrixData(matrixObject).matrix();
                        hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( matrix, writtenSkin->m_bindPose[bi]  );
                    }
                }
            }
            else // force skinned to first parent bone it is attached to
            {
                writtenSkin->m_bindPose[0] = writtenSkin->m_initSkinTransform;
                MDagPath parentPath( m_dagPath );

                while(parentPath.isValid() && !parentPath.hasFn(MFn::kJoint))
                    parentPath.pop();

                if(parentPath.isValid())
                {
                    /*MPlug jointWorldMatrixPlug = MFnIkJoint(parentPath).findPlug("worldMatrix").elementByLogicalIndex(0);
                    MObject matrixObject;
                    jointWorldMatrixPlug.getValue(matrixObject);
                    if(matrixObject != MObject::kNullObj) {
                    MMatrix matrix = MFnMatrixData(matrixObject).matrix();
                    hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( matrix, writtenSkin->m_bindPose[0] );
                    }*/

                    MFnIkJoint fnJoint(parentPath);
                    MPlug bindPoseMatrixPlug = fnJoint.findPlug("bindPose", &status);
                    MObject matrixObject;
                    if(status == MStatus::kSuccess)
                    {
                        status = bindPoseMatrixPlug.getValue(matrixObject);
                    }

                    // In the worst case, fall back to just grabbing the current world matrix.
                    if(status != MStatus::kSuccess)
                    {
                        MPlug jointWorldMatrixPlug = fnJoint.findPlug("worldMatrix").elementByLogicalIndex(0);
                        jointWorldMatrixPlug.getValue(matrixObject);
                    }

                    if(matrixObject != MObject::kNullObj)
                    {
                        matrix = MFnMatrixData(matrixObject).matrix();
                        hctMayaSceneExportUtilities::convertMMatrixToHkMatrix4( matrix, writtenSkin->m_bindPose[0]  );
                    }

                    writtenSkin->m_nodeNames[0] = parentPath.partialPathName().asChar();
                }
            }
        }

        if (keyTimes.getSize() > 0)
        {
            // A bit nasty that we are hijacking the usersectionutil info here, should split it out
            const hkArray<hctUserChannelUtil::SectionToGlobalMap>& globalMaps = m_userChannelUtil.getSectionGlobalMaps();

            // Make a cache per section
            hkArray< hkRefPtr<hkxVertexAnimationStateCache> > caches;
            for (int si=0; si < meshSections.getSize(); ++si)
            {
                hkxVertexAnimationStateCache* c = new hkxVertexAnimationStateCache(meshSections[si]->m_vertexBuffer, false);
                caches.pushBack( c );
                c->removeReference();
            }

            AnimatedVertData vd;
            MStatus vastatus;
            MFnDependencyNode fnDependNode( m_dagPath.node(), &vastatus );

            MObject meshData;
            MPlug plugMesh = fnDependNode.findPlug( MString( "outMesh" ), &vastatus );

            for (int ki=0; ki < keyTimes.getSize(); ++ki)
            {
                // Change time and then get new vert and normal arrays
                vd.m_vertexArray.setLength(0);
                vd.m_normalArray.setLength(0);

                // Get the .outMesh plug for this mesh
                if( vastatus == MStatus::kSuccess )
                {
                    MDGContext ctx = MDGContext( keyTimes[ki] );
#if MAYA_API_VERSION >= 20180000
                    {
                        MDGContextGuard ctxGuard( ctx );
                        vastatus = plugMesh.getValue( meshData ); //no need to update whole DAG, just enough to get out new outMesh
                    }
#else
                    vastatus = plugMesh.getValue( meshData, ctx ); //no need to update whole DAG, just enough to get out new outMesh
#endif
                    MFnMesh fnMesh( meshData, &vastatus );
                    vastatus = fnMesh.getPoints(  vd.m_vertexArray );
                    if( vastatus == MStatus::kSuccess )
                    {
                        // may need to make them relative to the transform's rotate pivot
                        // XX is this right when using MDGContext approach above instead of setCurrentTime
                        if( m_owner->m_options.m_useRotatePivot )
                        {
                            MDagPath parentPath( m_dagPath );
                            parentPath.pop();
                            MFnTransform parentTransformFn( parentPath, &status );
                            if( vastatus == MStatus::kSuccess )
                            {
                                MPoint offset = parentTransformFn.rotatePivot( MSpace::kPreTransform );
                                MFloatPoint floatOffset( (float)offset.x, (float)offset.y, (float)offset.z );
                                int numV = vd.m_vertexArray.length();
                                for( int i=0; i<numV; ++i )
                                {
                                    vd.m_vertexArray[i] -= offset;
                                }
                            }
                        }

                        // get the object space normals
                        vastatus = fnMesh.getNormals(  vd.m_normalArray, MSpace::kObject );
                        if( vastatus == MStatus::kSuccess )
                        {
                            // Make new anims if any
                            for (int si=0; si < meshSections.getSize(); ++si)
                            {
                                hkxVertexAnimationStateCache& vcache = *caches[si];
                                hkxVertexAnimation* vanim = getVertexAnimation( vd, meshSections[si], vcache, globalMaps[si] );
                                if (vanim)
                                {
                                    MTime exportRelativeTicks = keyTimes[ki] - m_owner->m_options.m_startTime;
                                    vanim->m_time = (hkReal)exportRelativeTicks.as( MTime::kSeconds );
                                    meshSections[si]->m_vertexAnimations.pushBack(vanim);
                                    vanim->removeReference();

                                    // make it the current state
                                    vcache.apply( vanim );
                                }
                            }
                        }
                    }
                }
            }

            // EXP-2435
            for (int si=0; si < meshSections.getSize(); ++si)
            {
                for (int ki = 0; ki < linearKeyFrameHints.getSize(); ki ++)
                {
                    MTime exportRelativeTicks = linearKeyFrameHints[ki] - m_owner->m_options.m_startTime;
                    meshSections[si]->m_linearKeyFrameHints.pushBack( (float)exportRelativeTicks.as( MTime::kSeconds ));
                }
            }
        }


        m_userChannelUtil.storeChannelsInMesh(writtenMesh);

    }


    // Clear the extracted data
    cleanup();
    return MStatus::kSuccess;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
