// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <maya/MSimple.h>
#include <maya/MGlobal.h>
#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MSelectionList.h>
#include <maya/MIOStream.h>
#include <maya/MFloatPointArray.h>
#include <maya/MDGModifier.h>
#include <maya/MDagModifier.h>
#include <maya/MProgressWindow.h>

#include <Common/GeometryUtilities/Mesh/Converters/SceneDataToMesh/hkSceneDataToMeshConverter.h>
#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshBody.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshSystem.h>
#include <Common/GeometryUtilities/Mesh/Default/hkDefaultMeshMaterialRegistry.h>

#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Base/Serialize/Resource/hkResource.h>

// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>

#include <ContentTools/Maya/MayaSceneExport/Commands/ImportScene/hctCmdImportScene.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Importer/MeshImport/hctMayaMeshImport.h>

#include <ContentTools/Common/SdkUtils/Scene/hctSceneUtils.h>


// hctCmdImportScene command flags
#define filenameFlag            "-f"
#define filenameFlagLong        "-filename"

#define meshFlag                "-m"
#define meshFlagLong            "-meshes"

#define lightFlag               "-l"
#define lightFlagLong           "-lights"

#define cameraFlag              "-c"
#define cameraFlagLong          "-cameras"

// Initialize statics
MObject hctCmdImportScene::m_pluginObject;

/* Class to allow any scene import code to interact with Maya progress bar */
ProgressSceneUpdaterMaya* hctCmdImportScene::m_progressUpdater = NULL;

struct ProgressSceneUpdaterMaya
{
    ProgressSceneUpdaterMaya()
        : m_currentObjectName(NULL)
    {
        MProgressWindow::startProgress();
        MProgressWindow::setInterruptable( true );
        MProgressWindow::setProgressStatus( MString("Preparing to import scene") );
        MProgressWindow::setTitle( MString("Havok Scene Import Progress") );
    }

    const char* m_currentObjectName;

    virtual void progression (float percentage);
    virtual bool didUserCancel ();
};

void ProgressSceneUpdaterMaya::progression (float percentage)
{
    MProgressWindow::setProgress( static_cast<int>(percentage) );

    MString objectName( m_currentObjectName );

    if ( !m_currentObjectName && !objectName.length() )
    {
        MProgressWindow::setProgressStatus( MString("Importing Scene..") );
    }
    else
    {
        MProgressWindow::setProgressStatus( MString("Importing Scene, object [") + objectName +  MString("]") );
    }
}

bool ProgressSceneUpdaterMaya::didUserCancel()
{
    if (MProgressWindow::isCancelled())
    {
        MProgressWindow::endProgress();
        return true;
    }
    return false;
}

/* end of progress updater */

hctCmdImportScene::hctCmdImportScene()
{
    m_filename = MString("");
    m_importMeshes = true;
    m_importLights = false;
    m_importCameras = false;
}


hctCmdImportScene::~hctCmdImportScene()
{
    cleaupModifiers();
}


void hctCmdImportScene::cleaupModifiers()
{
    for (int mi=0; mi < (int)m_dagModifiers.size(); ++mi)
    {
        delete m_dagModifiers[mi];
    }
    m_dagModifiers.resize(0);
}

MStatus hctCmdImportScene::removeMeshGeneratedFromNurbs( const MDagPath& generatedMeshPath )
{
    MString convertedMeshName = generatedMeshPath.fullPathName();
    MString cmd = "delete "+convertedMeshName+";";
    MCommandResult res;
    MStatus status = MGlobal::executeCommand(cmd, res, false, false);

    return status;
}

MSyntax hctCmdImportScene::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag( filenameFlag, filenameFlagLong, MSyntax::kString );
    syntax.addFlag( meshFlag, meshFlagLong, MSyntax::kBoolean );
    syntax.addFlag( lightFlag, lightFlagLong, MSyntax::kBoolean );
    syntax.addFlag( cameraFlag, cameraFlagLong, MSyntax::kBoolean );
    syntax.enableQuery( true );

    return syntax;
}

MStatus hctCmdImportScene::redoIt()
{
    MStatus stat = MStatus::kSuccess;
    for (int mi=0; mi < (int)m_dagModifiers.size(); ++mi)
    {
        MStatus thisStat = m_dagModifiers[mi]->doIt();
        if ( thisStat != MStatus::kSuccess)
        {
            stat = thisStat;
        }
    }
    return stat;
}

MStatus hctCmdImportScene::undoIt()
{
    MStatus stat = MStatus::kSuccess;
    for (int mi= (int)m_dagModifiers.size() - 1; mi >= 0; --mi)
    {
        MStatus thisStat = m_dagModifiers[mi]->undoIt();
        if ( thisStat != MStatus::kSuccess)
        {
            stat = thisStat;
        }
    }
    return stat;
}


static void _createMeshNodesRecursive(hkxScene* scene, hkxNode* node, std::vector<MDagModifier*>& dagModifiers, hkMeshSystem* meshSys )
{
    hkxMesh* mesh = hkxSceneUtils::getMeshFromNode(node);
    if (mesh)
    {
        hkMeshShape* meshShape = hkSceneDataToMeshConverter::convert(meshSys, HK_NULL, scene, node, hkSceneDataToMeshConverter::SPACE_ONLY_USING_SCALE_SKEW);
        if (meshShape)
        {
            hkMeshBody* meshObject = new hkMemoryMeshBody(meshSys, meshShape, node->m_keyFrames[0], HK_NULL );

            MDagModifier* dagMod = new MDagModifier();
            MObject parentNode = MObject::kNullObj;
            MObject transformObject = hctMayaSceneConvertUtilities::createNodeFromMesh(scene, *meshObject, parentNode, *dagMod );

            if (transformObject == MObject::kNullObj)
            {
                delete dagMod;
                HK_WARN_ALWAYS(0xabba872e, "Could not create mesh node for " << node->m_name);
            }
            else
            {
                dagModifiers.push_back( dagMod );
            }
        }
        else
        {
            HK_WARN_ALWAYS(0xabbafcd4, "Could not create mesh interface for " << node->m_name);
        }
    }

    for (int c=0; c < node->m_children.getSize(); ++c)
    {
        _createMeshNodesRecursive( scene, node->m_children[c], dagModifiers, meshSys);
    }
}

static hctFilterProcessingUtil* loadAllAvailableClasses(const char* pluginPath)
{
    MString filterManagerPath;
    hctFilterProcessingUtil* filterUtil = new hctFilterProcessingUtil();
    if ( hctMayaUtilities::getFilterManagerPath( pluginPath , filterManagerPath ) )
    {
        filterUtil->load( filterManagerPath.asChar() );
    }
    else
    {
        filterUtil->load( HK_NULL );
    }

    return filterUtil;
}

static void unloadAllAvailableClasses(hctFilterProcessingUtil* filterUtil)
{
    if (filterUtil)
    {
        filterUtil->unload();
        delete filterUtil;
    }
}

MStatus hctCmdImportScene::doIt( const MArgList& args )
{
    // get command arguments
    MArgDatabase argData( syntax(), args );

    // defaults
    m_filename = MString("default.hkx");
    if (argData.isFlagSet( filenameFlag ))
    {
        argData.getFlagArgument( filenameFlag, 0, m_filename );
    }

    if (m_filename.length() < 1)
    {
        MStatus s;
        s.perror( "Error, selection is not a transform node" );
        return MStatus::kFailure;
    }

    bool haveSceneOptions = argData.isFlagSet( meshFlag ) || argData.isFlagSet( lightFlag ) || argData.isFlagSet( cameraFlag );

    m_importMeshes = !haveSceneOptions;
    if (argData.isFlagSet( meshFlag ))
    {
        argData.getFlagArgument( meshFlag, 0, m_importMeshes );
    }

    m_importLights = false;
    if (argData.isFlagSet( lightFlag ))
    {
        argData.getFlagArgument( lightFlag, 0, m_importLights );
    }

    m_importCameras = false;
    if (argData.isFlagSet( cameraFlag ))
    {
        argData.getFlagArgument( cameraFlag, 0, m_importCameras );
    }


    // Create a ProgressUpdaterMaya object to handle displaying information from the
    // hull utility in the progress window as it is running
    bool reserved = MProgressWindow::reserve();
    if ( !reserved )
    {
        // couldn't reserve the progress bar, fail
        return MStatus::kFailure;
    }


    MProgressWindow::setProgressMin(0);
    MProgressWindow::setProgressMax(100);

    if (m_progressUpdater) delete m_progressUpdater;
    m_progressUpdater = new ProgressSceneUpdaterMaya();

    hctFilterProcessingUtil* filterUtil = HK_NULL;
    {
        MFnPlugin pluginFn( m_pluginObject );
        const char* pluginPath = pluginFn.loadPath().asChar();
        filterUtil = loadAllAvailableClasses(pluginPath);
    }

    // Load HKX
    hkResource* scenePackfileData = HK_NULL;

    hkRootLevelContainer* root = hctSceneUtils::loadSceneFromFile( m_filename.asChar(), scenePackfileData );

    MStatus retStat = MStatus::kFailure;
    if (root && scenePackfileData)
    {
        // Iterate hkxNodes
        hkxScene* scene = root->findObject<hkxScene>();
        if (scene && scene->m_rootNode)
        {
            hkMemoryMeshSystem meshSystem;

            hkxNode* node = scene->m_rootNode;
            if (m_importMeshes && node)
            {
                _createMeshNodesRecursive( scene, node, m_dagModifiers, &meshSystem);
            }

            retStat = MStatus::kSuccess;
        }
        else
        {
            HK_WARN_ALWAYS(0xabbae4e0, "No scene graph data in HKX file. Currently unsupported import." );
        }
    }
    else
    {
        HK_WARN_ALWAYS(0xabbac77d, "File does not exist, or no root level container. Check path and file type." );
    }

    // cleanup
    if (scenePackfileData)
    {
        scenePackfileData->removeReference();
    }

    unloadAllAvailableClasses(filterUtil);


    if (m_progressUpdater)
    {
        MProgressWindow::setProgressStatus( MString("") );
        MProgressWindow::endProgress();
    }


    return retStat;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
