// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <maya/MSimple.h>
#include <maya/MGlobal.h>
#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MSelectionList.h>
#include <maya/MIOStream.h>
#include <maya/MFloatPointArray.h>
#include <maya/MDGModifier.h>
#include <maya/MDagModifier.h>
#include <maya/MProgressWindow.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <ContentTools/Maya/MayaSceneExport/Importer/MeshImport/hctMayaMeshImport.h>


// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>

#include <ContentTools/Maya/MayaSceneExport/Commands/ExecuteDestruction/hctCmdExecuteDestruction.h>
#include <ContentTools/Common/DestructionUtils/hctDestructionUtils.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>

#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.maya"
#include <Common/Base/System/Log/hkLog.hxx>


// Initialize statics
MObject hctCmdExecuteDestruction::m_pluginObject;

#define objectNameFlag "-n"
#define objectNameLongFlag "-nodename"

#define previewFlag "-p"
#define previewLongFlag "-preview"

#define quickFlag "-q"
#define quickLongFlag "-quick"

/* Class to allow any scene import code to interact with Maya progress bar */
ProgressDestructionUpdaterMaya* hctCmdExecuteDestruction::m_progressUpdater = NULL;

struct ProgressDestructionUpdaterMaya
{
    ProgressDestructionUpdaterMaya()
        : m_currentObjectName(NULL)
    {
        MProgressWindow::startProgress();
        MProgressWindow::setInterruptable( true );
        MProgressWindow::setProgressStatus( MString("Preparing to break up objects") );
        MProgressWindow::setTitle( MString("Havok Destruction Progress") );
    }

    const char* m_currentObjectName;

    virtual void progression (float percentage);
    virtual bool didUserCancel ();
};

void ProgressDestructionUpdaterMaya::progression (float percentage)
{
    MProgressWindow::setProgress( static_cast<int>(percentage) );

    MString objectName( m_currentObjectName );

    if ( !m_currentObjectName && !objectName.length() )
    {
        MProgressWindow::setProgressStatus( MString("Breaking Object..") );
    }
    else
    {
        MProgressWindow::setProgressStatus( MString("Breaking Object, name [") + objectName +  MString("]") );
    }
}

bool ProgressDestructionUpdaterMaya::didUserCancel()
{
    if (MProgressWindow::isCancelled())
    {
        MProgressWindow::endProgress();
        return true;
    }
    return false;
}

/* end of progress updater */


hctCmdExecuteDestruction::hctCmdExecuteDestruction()
{

}


hctCmdExecuteDestruction::~hctCmdExecuteDestruction()
{
    cleanupModifiers();
}


void hctCmdExecuteDestruction::cleanupModifiers()
{
    for (int mi=0; mi < (int)m_dagModifiers.size(); ++mi)
    {
        delete m_dagModifiers[mi];
    }
    m_dagModifiers.resize(0);
}

MSyntax hctCmdExecuteDestruction::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag( objectNameFlag, objectNameLongFlag, MSyntax::kString );
    syntax.addFlag( previewFlag, previewLongFlag);
    syntax.addFlag( quickFlag, quickLongFlag);
    syntax.enableQuery( true );

    return syntax;
}

MStatus hctCmdExecuteDestruction::redoIt()
{
    MStatus stat = MStatus::kSuccess;
    for (int mi=0; mi < (int)m_dagModifiers.size(); ++mi)
    {
        MStatus thisStat = m_dagModifiers[mi]->doIt();
        if ( thisStat != MStatus::kSuccess)
        {
            stat = thisStat;
        }
    }
    return stat;
}

MStatus hctCmdExecuteDestruction::undoIt()
{
    MStatus stat = MStatus::kSuccess;
    for (int mi= (int)m_dagModifiers.size() - 1; mi >= 0; --mi)
    {
        MStatus thisStat = m_dagModifiers[mi]->undoIt();
        if ( thisStat != MStatus::kSuccess)
        {
            stat = thisStat;
        }
    }
    return stat;
}

static hkxNode* _firstSelectedNode( hkxNode* parent )
{
    if (parent->m_selected)
    {
        return parent;
    }

    for (int c=0; c < parent->m_children.getSize(); ++c)
    {
        hkxNode* s = _firstSelectedNode( parent->m_children[c] );
        if (s) return s;
    }

    return HK_NULL;
}

static MObject _findTransformNode( const MDagPath& pathToNode )
{
    if (pathToNode.isValid())
    {
        if (pathToNode.hasFn( MFn::kTransform ))
        {
            return pathToNode.node();
        }
        else
        {
            MDagPath dagCopy = pathToNode;
            dagCopy.pop();
            return _findTransformNode( dagCopy );
        }
    }
    return MObject::kNullObj;
}

static MDagPath _findTransformDagPath( const MDagPath& pathToNode )
{
    if (pathToNode.isValid())
    {
        if (pathToNode.hasFn( MFn::kTransform ))
        {
            return pathToNode;
        }
        else
        {
            MDagPath dagCopy = pathToNode;
            dagCopy.pop();
            return _findTransformDagPath( dagCopy );
        }
    }
    return MDagPath();
}

MStatus hctCmdExecuteDestruction::doIt( const MArgList& args )
{
    // get command arguments
    MArgDatabase argData( syntax(), args );

    // defaults
    MSelectionList  origSelectionList;
    MSelectionList  currentSelectionList;
    MGlobal::getActiveSelectionList( origSelectionList);

    bool forceUseOfRotatePivot = true;

    bool reserved = MProgressWindow::reserve();
    if ( !reserved )
    {
        // couldn't reserve the progress bar, fail
        return MStatus::kFailure;
    }

    bool previewOnly = argData.isFlagSet( previewFlag );
    bool quickMode = argData.isFlagSet( quickFlag );
    bool showOptions = !(previewOnly || quickMode);

    if (argData.isFlagSet( objectNameFlag ))
    {
        MString objName;
        argData.getFlagArgument( objectNameFlag, 0, objName);
        MGlobal::selectByName(objName, MGlobal::kReplaceList );

        MGlobal::getActiveSelectionList( currentSelectionList );
    }
    else if (origSelectionList.isEmpty() && !showOptions)
    {
        MStatus status;
        status = MStatus::kInvalidParameter;
        status.perror( "Error, no current selection nor any name given, so nothing to destruct." );
        return status;
    }
    else
    {
        currentSelectionList = origSelectionList;
    }

    // Initialize memory manager
#ifdef HK_DEBUG
//  const bool userDebugMem = true; // See OutputDebug stream to see leaks (they will not output to Maya Console)
//  hkSceneExportMemory::threadSafeBaseSystemInit( userDebugMem);
#else
//  hkSceneExportMemory::threadSafeBaseSystemInit( false );
#endif

    hkError* origError = &hkError::getInstance();
    origError->addReference();
    hkMayaError* mayaErrorHandler = new hkMayaError();
    hkError::replaceInstance( mayaErrorHandler );


    MStatus retStat = MStatus::kFailure;

    MProgressWindow::setProgressMin(0);
    MProgressWindow::setProgressMax(100);

    if (m_progressUpdater) delete m_progressUpdater;
    m_progressUpdater = new ProgressDestructionUpdaterMaya();


    MProgressWindow::setProgressStatus( MString("Loading Destruction Util DLL..") );

    //->  find the destruction util dll
    //    fail if not found
    HMODULE destructionDll = HK_NULL;
    hctBaseDll* destUtilsDll = HK_NULL;
    hctDestructionUtils* destUtils = HK_NULL;
    {
        MString filterManagerPath;
        MString destructionUtilDll;
        MFnPlugin pluginFn( m_pluginObject );
        MString pluginPathString = pluginFn.loadPath();
        const char* pluginPath = pluginPathString.asChar();
        hctMayaUtilities::getFilterManagerPath( pluginPath, filterManagerPath );
        destructionUtilDll = filterManagerPath.asChar() + MString("\\utils\\hctDestructionUtils.dll");

        hkIfstream testStream( destructionUtilDll.asChar() );
        if (!testStream.isOk())
        {
            retStat = MStatus::kNotImplemented;
            retStat.perror( "Destruction Util DLL not found in filter manager's util dir" );
            goto QUIT_CLEANLY;
        }

        destructionDll = LoadLibrary( destructionUtilDll.asChar() );

        if (!destructionDll)
        {
            retStat = MStatus::kNotImplemented;
            retStat.perror( "Destruction Util DLL found but could not load. The DLL is corrupt." );
            goto QUIT_CLEANLY;
        }

        hctGetBaseDllInterfaceFunc destBaseDllFunc = ( hctGetBaseDllInterfaceFunc ) GetProcAddress(destructionDll, "getBaseDllInterface");
        if (!destBaseDllFunc)
        {
            retStat = MStatus::kInvalidParameter;
            retStat.perror( "Destruction DLL found but could not find baseDLL sync function. Install is invalid." );
            goto QUIT_CLEANLY;
        }

        MProgressWindow::setProgressStatus( MString("Waiting for init to finish caching filters..") );
        MGlobal::executeCommand("hkCmdWaitForFilterLoad", false, false);

        if (MProgressWindow::isCancelled())
        {
            retStat = MStatus::kFailure;
            retStat.perror( "Destruction did not complete, user canceled op." );
            goto QUIT_CLEANLY;
        }

        MProgressWindow::setProgressStatus( MString("Initializing Destruction DLL..") );

        destUtilsDll = destBaseDllFunc();
        if (destUtilsDll)
        {
            if ( destUtilsDll->getDllVersion() != HCT_CURRENT_VERSION )
            {
                retStat = MStatus::kInvalidParameter;
                retStat.perror( "Could not load Destruction Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall." );
                goto QUIT_CLEANLY;
            }

            hkMemoryInitUtil::SyncInfo baseInfo;
            hkSceneExportMemory::getBaseSystemSyncInfo( baseInfo );
            destUtilsDll->initDll( baseInfo, HK_NULL );

            hctGetDestructionUtilInterfaceFunc getUtilInterface = ( hctGetDestructionUtilInterfaceFunc ) GetProcAddress(destructionDll, "getDestructionUtilInterface");
            destUtils = getUtilInterface? getUtilInterface() : HK_NULL;
        }

        if (!destUtils || !destUtilsDll )
        {
            hctGetDestructionDllErrorFunc destDllErrorFunc = ( hctGetDestructionDllErrorFunc ) GetProcAddress(destructionDll, "getDestructionDllError");
            if (!destDllErrorFunc)
            {
                retStat = MStatus::kFailure;
                retStat.perror( "Could not query Destruction Util DLL. Maybe your install is corrupt." );
                goto QUIT_CLEANLY;
            }

            hctDestructionUtils::DllError destError = destDllErrorFunc();

            if (destError & hctDestructionUtils::DLL_KEYCODE_ILLEGAL)
            {
                retStat = MStatus::kFailure;
                retStat.perror( "The keycode for the Destruction Util DLL is invalid or is for a different product version. Are you using your runtime (instead of tools) keycode?" );
                goto QUIT_CLEANLY;
            }

            if (destError & hctDestructionUtils::DLL_KEYCODE_EXPIRED)
            {
                retStat = MStatus::kFailure;
                retStat.perror( "The keycode for the Destruction Util DLL is expired or is for a different product version. Please contact Havok support at http://support.havok.com" );
                goto QUIT_CLEANLY;
            }

            if (destError & hctDestructionUtils::DLL_INIT_ERROR)
            {
                retStat = MStatus::kFailure;
                retStat.perror( "The Destruction Util DLL failed to initialize properly. Please try restarting Maya. If this error repeats, please contact Havok support at http://support.havok.com" );
                goto QUIT_CLEANLY;
            }

            retStat = MStatus::kFailure;
            retStat.perror( "The Destruction Util DLL failed to load. Maybe your install is corrupt. If this error repeats, please contact Havok support at http://support.havok.com" );
            goto QUIT_CLEANLY;

        }

        // All good:

        hctGetDestructionDllErrorFunc destDllErrorFunc = ( hctGetDestructionDllErrorFunc ) GetProcAddress(destructionDll, "getDestructionDllError");
        if (!destDllErrorFunc)
        {
            retStat = MStatus::kFailure;
            retStat.perror( "Could not query Destruction Util DLL. Maybe your install is corrupt." );
            goto QUIT_CLEANLY;
        }

        hctDestructionUtils::DllError destError = destDllErrorFunc();

        if (destError & hctDestructionUtils::DLL_KEYCODE_WARNING)
        {
            MProgressWindow::setProgressStatus( MString("Destruction keycode expires soon") );
            MGlobal::displayWarning( MString("The Destruction keycode expires in 10 days or less.") );
            Sleep(3000);
        }

        destUtils->setFilterManagerPath(filterManagerPath.asChar());

        if (MProgressWindow::isCancelled())
        {
            retStat = MStatus::kFailure;
            retStat.perror( "Destruction did not complete, user canceled op." );
            goto QUIT_CLEANLY;
        }
    }

    //->  do a quick export, single frame, to get the current hkxScene to use
    {
        hctMayaSceneExporter::Options options;
        options.m_batchMode = true;
        options.m_doNotSplitVertices = false;
        options.m_visibleOnly = false;  
        if (forceUseOfRotatePivot)
        {
            options.m_useRotatePivot = true; // true by default anyway, but make sure :)
        }

        options.m_startTime = options.m_endTime = MAnimControl::currentTime();

        hctMayaSceneExporter exporter( options ); // sets base sys error handler to Mayas
        MFnPlugin pluginFn( m_pluginObject );
        MString pluginPathString = pluginFn.loadPath();
        exporter.m_pluginPath = pluginPathString.asChar();

        MProgressWindow::setProgressStatus( MString("Creating HKX scene..") );

        MStatus result = exporter.createScene();
        if (MProgressWindow::isCancelled())
        {
            result = MStatus::kFailure;
            result.perror( "Destruction did not complete, user canceled op." );
        }

        hkRootLevelContainer* sceneData = exporter.getCurrentRootContainer();
        if ( ( (result != MStatus::kSuccess) || (sceneData == HK_NULL) ) && !showOptions )
        {
            HK_WARN_ALWAYS(0xabba4534, "No scene data exported, so nothing to break up using Destruction");
            retStat = MStatus::kFailure;
            exporter.cleanupScene();
            goto QUIT_CLEANLY;
        }

        MProgressWindow::setProgressStatus( MString("Bringing up Destruction Util..") );

        destUtils->registerThreadCallback(hctFilterProcessingUtil::getDefaultThreadCallback());

        //->  give the scene and 'object to break' to the destruction utils
        MWindow mainMayaWindowHandle = M3dView::applicationShell();

        MItSelectionList selIter( currentSelectionList );
        hkArray<hkxNode*> selectedNodes;

        hkxScene* sceneGraph = sceneData->findObject<hkxScene>();
        bool haveNodes = false;
        while (!selIter.isDone())
        {
            MDagPath pathToNode;
            selIter.getDagPath(pathToNode);
            MString selNodeNameString = pathToNode.partialPathName();
            const char* selNodeName = selNodeNameString.asChar();
            hkxNode* selObject = sceneGraph->findNodeByName(selNodeName);
            if (selObject==HK_NULL)
            {
                // try to find the parent
                MDagPath parentTransformDag = _findTransformDagPath( pathToNode );
                selNodeNameString = parentTransformDag.partialPathName();
                selNodeName = selNodeNameString.asChar();
                selObject = sceneGraph->findNodeByName(selNodeName);
                if ( selObject == HK_NULL )
                {
                    HK_WARN_ALWAYS(0xabba5645, "Could not find selected node [" << selNodeName << "], was it exported ok?" );
                }
            }

            haveNodes = haveNodes || (selObject!=HK_NULL);
            selectedNodes.pushBack(selObject);
            selIter.next();
        }

        if ( !haveNodes && !showOptions ) // should process or preview but no nodes
        {
            exporter.cleanupScene();
            retStat = MStatus::kFailure;
            HK_WARN_ALWAYS(0xabba5646, "No selected objects where exported, so can't destruct them." );
            goto QUIT_CLEANLY;
        }


        hkArray< hctDestructionUtils::BodyInfo *> brokenBits;
        hctDestructionUtils::RunOptions runStyle = previewOnly? hctDestructionUtils::PREVIEW_RUN : ( quickMode? hctDestructionUtils::QUICK_RETURN_RUN : hctDestructionUtils::DEFAULT_RUN );

        //
        // Extract the conversion factor for converting [m] to the current physics distance unit (which can differ from [m]).
        // Necessary for some hardcoded distance checks/asserts/warnings (e.g. default convex radius).
        //
        hkReal distanceConversionFactor = 1.0f;
        {
            MDistance internalMeterToUnitConverter;
            internalMeterToUnitConverter.setUnit(MDistance::kMeters);
            internalMeterToUnitConverter.setValue(1.0f);
            distanceConversionFactor = hkReal(internalMeterToUnitConverter.as(MDistance::internalUnit()));
        }
        destUtils->setDistanceConversionFactor(distanceConversionFactor);

        hkBool wasEnabled0xabba71e0 = hkError::getInstance().isEnabled(0xabba71e0);
        hkError::getInstance().setEnabled(0xabba71e0, false);

        destUtils->breakObjects(sceneData, selectedNodes.begin(), selectedNodes.getSize(), brokenBits, (void*)mainMayaWindowHandle, runStyle );

        // Manually append filter manager's error output to Maya's error output.
        {
            hctSceneExportError* filterManagerErrorHandler = destUtils->getFilterUtil()->getFilterManagerInterface()->getErrorHandler();
            for (int i = filterManagerErrorHandler->numLogEntries()-1; i >= 0; i--)
            {
                const hctSceneExportError::LogEntry& logEntry = filterManagerErrorHandler->getLogEntry(i);
                if (logEntry.m_type != hkError::MESSAGE_REPORT &&
                    logEntry.m_id != 0xabba10b0 &&
                    logEntry.m_id != 0xabbabfd2 &&
                    logEntry.m_id != 0xabbabfd4 &&
                    logEntry.m_id != 0xabbabfd6 &&
                    logEntry.m_id != 0xabbabad4 &&
                    logEntry.m_id != 0xabbabfd1)
                {
                    mayaErrorHandler->message(logEntry.m_type, 0, logEntry.m_text.cString(), __FILE__, __LINE__);
                }
            }
        }

        hkError::getInstance().setEnabled(0xabba71e0, wasEnabled0xabba71e0);

        if (MProgressWindow::isCancelled())
        {
            retStat = MStatus::kFailure;
            retStat.perror( "Destruction did not complete, user canceled op." );
        }
        else if (brokenBits.getSize() > 0)
        {
            //->  take the results of the destruction and add them back to the scene
            //    may be no results, as util may iterate through previews etc, and/or cancel
            hkArray<MObject> createdNodes;
            createdNodes.reserve(brokenBits.getSize());

            int numSuccess = 0;
            int numFail = 0;
            int numPartialFail = 0;
            int physicsShapeCtr = 0;

            for (int bbi = 0; bbi < brokenBits.getSize(); ++bbi)
            {
                hctDestructionUtils::BodyInfo& newPiece = *(brokenBits[bbi]);

                hkxNode* origNode = newPiece.getOriginalNode();
                hkMeshBody* graphicsBody = newPiece.getGraphicsBody();
                hkArray<hkMeshBody*> childPhysicsShapes;
                newPiece.getPhysicsShapes( childPhysicsShapes );

                // We will abort if either there is no original node to be found or if we have neither
                // a graphics shape nor at least one physics shape for this node. Having no graphics
                // shape but 1+ physics shapes is a valid situation produced by e.g. the ConvexDecompose
                // fracture algorithm.
                if ( !origNode || (!graphicsBody && childPhysicsShapes.getSize() == 0) )
                {
                    createdNodes.pushBack(MObject::kNullObj);
                    ++numFail;
                    continue;
                }

                MDagPath pathToNode;
                {
                    int selIndex = selectedNodes.indexOf(origNode);
                    HK_ASSERT_NO_MSG(0x62b38707, selIndex >= 0 );
                    currentSelectionList.getDagPath( selIndex, pathToNode );
                }

                // Find parent transform node
                MObject parentTransformNode = _findTransformNode( pathToNode );
                int pieceParentIndex = newPiece.getParentIndex();
                if ( pieceParentIndex >=0 && createdNodes[pieceParentIndex] != MObject::kNullObj)
                {
                    parentTransformNode = createdNodes[pieceParentIndex];
                }

                MFnTransform fnParent(parentTransformNode);
                MString parentNameString = fnParent.name();
                const char* parentName = parentNameString.asChar();

                if ( graphicsBody )
                {
                    MDagModifier* dagMod = new MDagModifier();

                    const char* childMeshName = childPhysicsShapes.getSize() ? "graphicsMesh" : "mesh";

                    MObject meshTransformNode = hctMayaSceneConvertUtilities::createNodeFromMesh(sceneGraph, *graphicsBody, parentTransformNode, *dagMod, graphicsBody->getName(), childMeshName, forceUseOfRotatePivot );

                    createdNodes.pushBack(meshTransformNode);

                    if (meshTransformNode == MObject::kNullObj)
                    {
                        delete dagMod;
                        HK_WARN_ALWAYS(0xabba6712, "Could not create mesh node for " << graphicsBody->getName() );

                        ++numFail;
                    }
                    else
                    {

                        m_dagModifiers.push_back( dagMod ); // so we can undo :)

                        // create RB and BB data for new bits
                        // can just call the scripts
                        MFnTransform fn(meshTransformNode);
                        MString cmd;
                        MCommandResult cmdResult;

                        // Physics Shape Node(s)
                        int numChildren = childPhysicsShapes.getSize();
                        if (numChildren == 0)
                        {
                            cmd = MString("hkProcPhysics_addShapeNode2 ") + fn.fullPathName();
                            MStatus sStat = MGlobal::executeCommand(cmd, cmdResult, false, false);
                            if (sStat!= MStatus::kSuccess)
                            {
                                MString nameString = fn.name();
                                HK_WARN_ALWAYS(0xabba5423, "Could not set new shape node for " << nameString.asChar()  );
                                ++numPartialFail;
                            }
                            else
                            {
                                ++numSuccess;
                            }
                        }
                        else // the physics != graphics for the broken pieces
                        {
                            for (int csi=0; csi < numChildren; ++csi)
                            {
                                hkMeshBody* childMesh = childPhysicsShapes[csi];
                                if (childMesh)
                                {
                                    // May or may not need a transform node. We need one though as
                                    MDagModifier* childDagMod = new MDagModifier();
                                    MObject childTransformNode = hctMayaSceneConvertUtilities::createNodeFromMesh(sceneGraph, *childMesh, meshTransformNode, *childDagMod, "physics", "physicsMesh" );

                                    if (childTransformNode != MObject::kNullObj)
                                    {
                                        m_dagModifiers.push_back( childDagMod ); // so we can undo :)

                                        MFnTransform fnChild(childTransformNode);
                                        cmd = MString("hkProcPhysics_addShapeNode2 ") + fnChild.fullPathName();
                                        MStatus sStat = MGlobal::executeCommand(cmd, cmdResult, false, false);
                                        if (sStat!= MStatus::kSuccess)
                                        {
                                            ++numPartialFail;
                                            MString nameString = fnChild.name();
                                            HK_WARN_ALWAYS(0xabba82e2, "Could not set new shape node for " << nameString.asChar()  );
                                        }
                                        else
                                        {
                                            ++numSuccess;
                                        }
                                    }
                                    else
                                    {
                                        delete childDagMod;
                                        ++numPartialFail;
                                        MString nameString = fn.name();
                                        HK_WARN_ALWAYS(0xabbaa091, "Could not create child mesh [" << (childMesh->getName()? childMesh->getName() : "noname") << "] for node " << nameString.asChar() );
                                    }
                                }
                            }
                        }
                        // Destruction info
                        // The parent tree has the breakable body in it, we just need to add shapes for each new bit
                        {
                            cmd = MString("hkProcDestruction_addBreakableShapeNode ") + fn.fullPathName();
                            MStatus dStat = MGlobal::executeCommand(cmd, cmdResult, false, false);
                            if (dStat != MStatus::kSuccess)
                            {
                                ++numPartialFail;
                                MString nameString = fn.name();
                                HK_WARN_ALWAYS(0xabba7a53, "Could not set new Destruction Shape for " << nameString.asChar() );
                            }
                        }

                        // Set the piece fixed if needed
                        if ( newPiece.getQualityType() == hctDestructionUtils::QUALITY_FIXED )
                        {
                            MString qValue; qValue.set(double(hctDestructionUtils::QUALITY_FIXED), 0);
                            cmd             = MString("hkProcDestruction_setNodeAttValue (\"") + fn.name() + MString("\", \"hkdShape\", \"bodyQualityType\", ") + qValue + MString(");");
                            MStatus dStat   = MGlobal::executeCommand(cmd, cmdResult, false, false);

                            if ( dStat != MStatus::kSuccess )
                            {
                                numPartialFail++;
                                MString nameString = fn.name();
                                HK_WARN_ALWAYS(0xabba7a53, "Could not set new bodyQualityType for " << nameString.asChar());
                            }
                        }

                    }
                }
                else // !graphicsBody
                {
                    //
                    // Special case for "Convex Decompose Physics" utility:
                    // No graphics body but at least one physics shape.
                    //

                    char nodeNameBuffer[1024];

                    {
                        for (int physicsShapeIndex = 0; physicsShapeIndex < childPhysicsShapes.getSize(); physicsShapeIndex++)
                        {
                            hkMeshBody* physicsShape = childPhysicsShapes[physicsShapeIndex];

                            MDagModifier* dagMod = new MDagModifier();

                            hkString::snprintf(nodeNameBuffer, 1000, "%s_Physics%i", parentName, physicsShapeCtr++);

                            MObject meshTransformNode = hctMayaSceneConvertUtilities::createNodeFromMesh(sceneGraph, *physicsShape, parentTransformNode, *dagMod, nodeNameBuffer, "physicsMesh", forceUseOfRotatePivot);

                            createdNodes.pushBack(meshTransformNode);

                            if ( meshTransformNode == MObject::kNullObj )
                            {
                                HK_WARN_ALWAYS(0xabba6715, "Could not create physics shape node for " << physicsShape->getName());
                                delete dagMod;
                                ++numFail;
                            }
                            else // meshTransformNode is valid
                            {
                                m_dagModifiers.push_back(dagMod); // so we can undo :)

                                MString cmd;
                                MCommandResult cmdResult;

                                //
                                // Add the "Havok Shape" modifier to the new imported physics node.
                                //
                                {
                                    MFnTransform fn(meshTransformNode);
                                    cmd = MString("hkProcPhysics_addShapeNode ") + fn.fullPathName();
                                    MStatus dStat = MGlobal::executeCommand(cmd, cmdResult, false, false);
                                    if ( dStat != MStatus::kSuccess )
                                    {
                                        MString nameString = fn.name();
                                        HK_WARN_ALWAYS(0xabba5426, "Could not set new Shape for " << nameString.asChar() );
                                        ++numPartialFail;
                                    }
                                    else
                                    {
                                        ++numSuccess;
                                    }
                                }

                                //
                                // Remove (if available) the old "Havok Shape" node from the original object.
                                //
                                {
                                    MFnTransform fn(parentTransformNode);
                                    cmd = MString("hkProcDestruction_removePhysicsShapeNode ") + fn.fullPathName();
                                    MGlobal::executeCommand(cmd, cmdResult, false, true); // need to enable Undo here!
                                }
                            }
                        }
                    }
                }

                // cleanup
                delete brokenBits[bbi];
            }// all new pieces

            if (numSuccess > 0)
            {
                retStat = MStatus::kSuccess;
                if ( numFail > 0 )
                {
                    Log_Warning( "Destruction created {} new pieces, but had some failures. Check the 'Output Window' for specific errors.", numSuccess );
                }
                else if ( numPartialFail )
                {
                    Log_Warning( "Destruction created {} new pieces, but had some partial failures. Check the 'Output Window' for specific errors.", numSuccess );
                }
                else
                {
                    Log_Info( "Destruction created {} new pieces, no failures", numSuccess );
                }
            }
            else
            {
                retStat = MStatus::kFailure;
                retStat.perror("Destruction did not complete properly, no pieces created. ");
            }
        }
        else if (showOptions)
        {
            retStat = MStatus::kSuccess;
            Log_Info( "Destruction Util options success." );
        }
        else
        {
            retStat = MStatus::kFailure;
            retStat.perror("Nothing was returned from the Destruction Util. No new pieces created.");
        }

        //-> Do some cleanup
        exporter.cleanupScene();
    }

QUIT_CLEANLY:

    if (retStat != MStatus::kSuccess)
    {
        Log_Error( "Failed to destruct object(s). Check the 'Output Window' for specific errors." );
    }

    if (destUtils)
    {
        delete destUtils;
    }

    // cleanup
    if (destUtilsDll)
    {
        destUtilsDll->quitDll();
    }

    hkError::replaceInstance( origError ); // takes ref

    // reset selection
    MGlobal::setActiveSelectionList( origSelectionList );

    if (m_progressUpdater)
    {
        MProgressWindow::setProgressStatus( MString("") );
        MProgressWindow::endProgress();
    }

    // Exit the memory manager
//  hkSceneExportMemory::threadSafeBaseSystemQuit();

    if ( destructionDll )
    {
        FreeLibrary( destructionDll );
    }

    return retStat;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
