// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <maya/MSimple.h>
#include <maya/MGlobal.h>
#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MSelectionList.h>
#include <maya/MIOStream.h>
#include <maya/MFloatPointArray.h>
#include <maya/MDGModifier.h>
#include <maya/MDagModifier.h>
#include <maya/MProgressWindow.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SdkUtils/Geometry/hctCreateConvexHullUtil.h>
#include <ContentTools/Maya/MayaSceneExport/Commands/CreateConvexHull/hctCmdCreateConvexHull.h>


// hctCmdCreateConvexHull command flags
#define maxVertsFlag                "-mv"
#define maxVertsFlagLong            "-maxverts"

#define encloseInputsFlag           "-en"
#define encloseInputsFlagLong       "-enclose"


/* Class to allow convex hull utility code to interact with Maya progress bar */
ProgressUpdaterMaya* hctCmdCreateConvexHull::m_progressUpdater = NULL;

struct ProgressUpdaterMaya : public hctSdkUtils::ProgressUpdater
{
    ProgressUpdaterMaya()
    {
        m_currentObjectName = NULL;

        MProgressWindow::startProgress();
        MProgressWindow::setInterruptable( true );
        MProgressWindow::setProgressStatus( MString("Preparing to run convex hull utility") );
        MProgressWindow::setTitle( MString("Convex hull utility progress") );
    }

    virtual void progression (float percentage);
    virtual bool didUserCancel ();
};

void ProgressUpdaterMaya::progression (float percentage)
{
    MProgressWindow::setProgress( static_cast<int>(percentage) );

    MString objectName( m_currentObjectName );

    if ( !objectName.length() )
    {
        MProgressWindow::setProgressStatus( MString("Creating convex hull of object") );
    }
    else
    {
        MProgressWindow::setProgressStatus( MString("Creating convex hull of ") + objectName );
    }
}

bool ProgressUpdaterMaya::didUserCancel()
{
    if (MProgressWindow::isCancelled())
    {
        MProgressWindow::endProgress();
        return true;
    }
    return false;
}

/* end of progress updater */


MStatus hctCmdCreateConvexHull::removeMeshGeneratedFromNurbs( const MDagPath& generatedMeshPath )
{
    MString convertedMeshName = generatedMeshPath.fullPathName();
    MString cmd = "delete "+convertedMeshName+";";
    MCommandResult res;
    MStatus status = MGlobal::executeCommand(cmd, res, false, false);

    return status;
}

MStatus hctCmdCreateConvexHull::createHullMesh( hkGeometry& geom, MString hullName )
{
    // create a mesh from the input geometry
    int numPoints = geom.m_vertices.getSize();
    MPointArray points (numPoints);

    for (int vi=0; vi<numPoints; vi++)
    {
        MPoint p;
        p.x = geom.m_vertices[vi](0);
        p.y = geom.m_vertices[vi](1);
        p.z = geom.m_vertices[vi](2);
        points[vi] = p;
    }

    int numTriangles = geom.m_triangles.getSize();
    MIntArray polygonCounts( numTriangles, 3 );
    MIntArray polygonConnects( numTriangles*3 );

    for (int ti=0; ti<numTriangles; ti++)
    {
        polygonConnects[3*ti  ] = geom.m_triangles[ti].m_a;
        polygonConnects[3*ti+1] = geom.m_triangles[ti].m_b;
        polygonConnects[3*ti+2] = geom.m_triangles[ti].m_c;
    }

    /* create a new identity transform under the scene root, as follows:

        root
          |
          +---- new identity transform node ( foo_hull )
                           |
                           +---- generated hull ( foo_hullShape )

       where foo is the name of the first object in the original selection.
    */

    //create parent transform for mesh, parented to scene root
    MStatus status;
    MTypeId typeId = MFn::kTransform;
    MObject meshTransform = dagMod.createNode( MString("transform"), MObject::kNullObj, &status );
    if ( status != MStatus::kSuccess)
    {
        return status;
    }

    //// rename the transform
    dagMod.renameNode(meshTransform, hullName + "_hull");
    dagMod.doIt();

    // NB, the following mesh creation/connection is not done using MDagModifier, but that's OK since on undoing,
    // the transform containing the mesh is deleted, effectively undoing the creation and connection.

    // create a mesh parented to this transform
    MFnMesh meshFn;
    meshFn.create( numPoints, numTriangles, points, polygonCounts, polygonConnects, meshTransform, &status );
    if (status != MStatus::kSuccess)
    {
        return status;
    }

    MDagPath generatedMeshPath;
    meshFn.getPath( generatedMeshPath );

    MDagPath generatedMeshTransformPath( generatedMeshPath );
    generatedMeshTransformPath.pop();

    // connect mesh to shading group
    // ( MEL commmand: connectAttr -na ($mesh + ".instObjGroups") "initialShadingGroup.dagSetMembers" )
    MString command = MString( "connectAttr -na ( \"" ) + generatedMeshPath.fullPathName() \
            + MString( "\" + \".instObjGroups\") \"initialShadingGroup.dagSetMembers\" " );

    MGlobal::executeCommand( command );

    return status;
}


MSyntax hctCmdCreateConvexHull::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag( maxVertsFlag, maxVertsFlagLong, MSyntax::kUnsigned );
    syntax.addFlag( encloseInputsFlag, encloseInputsFlagLong, MSyntax::kBoolean );
    syntax.setObjectType( MSyntax::kSelectionList );
    syntax.enableQuery( true );

    return syntax;
}

MStatus hctCmdCreateConvexHull::redoIt()
{
    return dagMod.doIt();
}

MStatus hctCmdCreateConvexHull::undoIt()
{
    return dagMod.undoIt();
}

MStatus hctCmdCreateConvexHull::doIt( const MArgList& args )
{
    // get command arguments
    MArgDatabase argData( syntax(), args );

    // defaults
    m_maxVerts = 64;
    m_encloseInputs = true;

    if (argData.isFlagSet( maxVertsFlag ))
    {
        unsigned maxVerts;
        argData.getFlagArgument( maxVertsFlag, 0, maxVerts );
        m_maxVerts = maxVerts;
    }
    if (argData.isFlagSet( encloseInputsFlag ))
    {
        bool encloseInputs;
        argData.getFlagArgument( encloseInputsFlag, 0, encloseInputs );
        m_encloseInputs = encloseInputs;
    }

    // get the objects in the command selection list
    MSelectionList      commandList;
    argData.getObjects( commandList );
    if (commandList.length()==0)
    {
        // no object was specified, use the first object in the current selection list (transform is expected)
        MGlobal::getActiveSelectionList( commandList );

        // if no object is selected, return failure
        if (commandList.length()==0) return MStatus::kFailure;
    }

    unsigned int numSelected = commandList.length();
    if (!numSelected) return MStatus::kFailure;;

    // Extract the combined set of world space vertices from the selected meshes/nurbs
    MPointArray vertexArray;
    MDagPath selectionPath;

    for (unsigned int n=0; n<numSelected; ++n)
    {
        commandList.getDagPath( n, selectionPath );

        // check whether the selection is a transform node, if not, return failure
        {
            MFnTransform tNodeFn;
            MStatus status = tNodeFn.setObject( selectionPath );
            if (status != MStatus::kSuccess)
            {
                status.perror( "Error, selection is not a transform node" );
                return MStatus::kFailure;
            }
        }

        MStatus result = getWorldVerticesFromObject( selectionPath, vertexArray );
        if (result != MStatus::kSuccess)
        {
            // fail (do not generate any hull) if we could not extract the vertices of one of the selected objects
            return MStatus::kFailure;
        }
    }

    commandList.getDagPath( 0, selectionPath );
    MString hullName = selectionPath.partialPathName();

    return createMultiHull( vertexArray, hullName );
}


MStatus hctCmdCreateConvexHull::getWorldVerticesFromObject( const MDagPath& dagPath, MPointArray& vertexArray )
{
    MStatus status;

    // find the first mesh/nurbs under the transform
    const MDagPath& node = dagPath;
    MDagPath meshPath = node;
    unsigned int numChildren = node.childCount();

    bool isNurbs = false;

    for( unsigned int i=0; i<numChildren; ++i )
    {
        MFn::Type type = node.child(i).apiType();

        if( type == MFn::kMesh )
        {
            meshPath.push( node.child(i) );
            isNurbs = false;
            break;
        }

        // generate a temporary mesh from the nurbs surface
        if (type == MFn::kNurbsSurface )
        {
            MDagPath nurbsPath = node;
            nurbsPath.push( node.child(i) );

            MDagPath generatedMeshPath;
            status = hctUtilities::getMeshFromNurbs( nurbsPath, generatedMeshPath );
            if( status != MStatus::kSuccess )
            {
                status.perror( "Error creating mesh from nurbs " + nurbsPath.fullPathName() );
                return status;
            }

            meshPath = generatedMeshPath;
            isNurbs = true;
            break;
        }
    }

    // if there was no mesh, return failure
    if( !meshPath.isValid() ) return MStatus::kFailure;
    if( meshPath.apiType() != MFn::kMesh && meshPath.apiType() != MFn::kNurbsSurface )
    {
        status.perror( "Error, no mesh or nurbs found" );
        return MStatus::kFailure;
    }

    // Get the mesh data
    MFnMesh mesh;
    status = mesh.setObject( meshPath );
    if( status != MStatus::kSuccess )
    {
        status.perror( "Error creating MFnMesh" );
        return status;
    }

    //get the world space transform of the mesh
    MDagPath meshTransformPath = meshPath; meshTransformPath.pop();
    MMatrix nodeToWorld = meshTransformPath.inclusiveMatrix();


    // get the object space vertex positions
    MPointArray newVertexArray;
    status = mesh.getPoints( newVertexArray, MSpace::kObject );
    if( status != MStatus::kSuccess )
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    // transform into world space
    for (unsigned int vi=0; vi<newVertexArray.length(); ++vi)
    {
        MPoint& p = newVertexArray[vi];
        p *= nodeToWorld;
    }

    // append the current object's vertices to the multi-hull vertex array
    unsigned int oldLength   = vertexArray.length();
    unsigned int extraLength = newVertexArray.length();
    unsigned int newLength   = oldLength + extraLength;
    vertexArray.setLength( newLength );

    for (unsigned int vi=0; vi<extraLength; ++vi)
    {
        vertexArray.set( newVertexArray[vi], oldLength + vi );
    }

    // remove temporary mesh created from nurbs
    if (isNurbs)
    {
        removeMeshGeneratedFromNurbs( meshPath );
    }

    return MStatus::kSuccess;
}


MStatus hctCmdCreateConvexHull::createMultiHull( MPointArray& vertexArray, MString hullName )
{
    // Create a ProgressUpdaterMaya object to handle displaying information from the
    // hull utility in the progress window as it is running
    bool reserved = MProgressWindow::reserve();
    if ( !reserved )
    {
        // couldn't reserve the progress bar, fail
        return  MStatus::kFailure;
    }

    MProgressWindow::setProgressMin(0);
    MProgressWindow::setProgressMax(100);

    delete m_progressUpdater;
    m_progressUpdater = new ProgressUpdaterMaya();

    MStatus status;

    {
        // use the hctCreateConvexHullUtil to generate a hull from the input mesh
        hctCreateConvexHullUtil::Input utilityInput;

        utilityInput.m_progressUpdater = m_progressUpdater;

        int nPoints = vertexArray.length();
        for (int i=0; i<nPoints; i++)
        {
            MPoint p = vertexArray[i];
            hkVector4& vertex = *utilityInput.m_vertices.expandBy(1);
            vertex.set( hkReal(p.x), hkReal(p.y), hkReal(p.z), hkReal(0.0) );
        }

        // take input params from command arguments
        utilityInput.m_maxNumVerts   = m_maxVerts;
        utilityInput.m_encloseInputs = m_encloseInputs;

        // call the utility
        hctCreateConvexHullUtil::Output utilityOutput;
        hkResult hullUtilResult = hctCreateConvexHullUtil::createCoarseHull(utilityInput, utilityOutput);

        if (hullUtilResult.isFailure())
        {
            status.perror( "Failed to generate hull" );
            setResult( FALSE );
            status = MStatus::kFailure;
            goto cleanup;
        }
        else
        {
            hkStringBuf report_str;
            report_str.printf("Convex hull generated with fractional volume reduction from original hull of %4.2f percent.", \
                                    (float)utilityOutput.m_volumeReductionPercentage);

            MGlobal::displayInfo( report_str.cString() );
        }

        // create a new mesh from the output geometry, and put it in the node
        MDagPath generatedMeshPath;
        status = createHullMesh( utilityOutput.m_convexGeom, hullName );

        if( status != MStatus::kSuccess )
        {
            status.perror( "Failed to create hull mesh" );
            goto cleanup;
        }

        setResult( TRUE );
        status = MStatus::kSuccess;
    }

cleanup:

    if (m_progressUpdater)
    {
        MProgressWindow::setProgressStatus( MString("") );
        MProgressWindow::endProgress();
    }


    return status;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
