// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>
#include <ContentTools/Maya/MayaSceneExport/Commands/CreateConvexDecomposition/hctCmdCreateConvexDecomposition.h>

#include <maya/MSimple.h>
#include <maya/MGlobal.h>
#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MSelectionList.h>
#include <maya/MIOStream.h>
#include <maya/MFloatPointArray.h>
#include <maya/MDGModifier.h>
#include <maya/MDagModifier.h>
#include <maya/MProgressWindow.h>
#include <maya/MPlug.h>
#include <maya/MFnTransform.h>
#include <maya/MFnDependencyNode.h>
#include <maya/MFnMessageAttribute.h>
#include <maya/MFnNumericAttribute.h>

#include <ContentTools/Maya/MayaSceneExport/Exporter/hctMayaSceneExporter.h>
#include <ContentTools/Maya/MayaSceneExport/Utilities/hctMayaUtilities.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>

MObject hctCmdCreateConvexDecomposition::m_pluginObject;

// hctCmdCreateConvexDecomposition command flags
#define checkDllValidityFlag            "-cv"
#define checkDllValidityFlagLong        "-checkValidity"

#define generateGuardsFlag              "-gg"
#define generateGuardsFlagLong          "-generateGuards"

#define detachPartsFlag                 "-dp"
#define detachPartsFlagLong             "-detachParts"

#define accuracyFlag                    "-ac"
#define accuracyFlagLong                "-accuracy"

#define singleOutputFlag                "-so"
#define singleOutputFlagLong            "-singleOutput"

#define singleOutputCombineFlag         "-sc"
#define singleOutputCombineFlagLong     "-singleOutputCombine"

#define useExistingFlag                 "-ue"
#define useExistingFlagLong             "-useExisting"

#define reduceOverlapFlag               "-ro"
#define reduceOverlapFlagLong           "-reduceOverlap"

#define octreeDepthFlag                 "-od"
#define octreeDepthFlagLong             "-octreeDepth"

#define decomposeMethodFlag             "-dm"
#define decomposeMethodFlagLong         "-decomposeMethod"

#define reduceMethodFlag                "-rm"
#define reduceMethodFlagLong            "-reduceMethod"

#define wrapIterationsFlag              "-wi"
#define wrapIterationsFlagLong          "-wrapIterations"

#define wrapConcavityFlag               "-wc"
#define wrapConcavityFlagLong           "-wrapConcavity"

#define MAYA_CD_SINGLE_NODE_OUTPUT 1

//
// Progress bar job-handler
//

void hctMayaConvexDecompProgressHandler::processJobs()
{
    m_isCancelled_mainThread = m_isCancelled = false;
    char message[160];
    int percentage;

    while ( m_newJobEvent->acquire(), true )
    {
        m_jobQueueLock->enter();
        hkString::strCpy(message, m_message );
        percentage = m_percentage;
        m_jobQueueLock->leave();

        if ( 0 == strcmp( "quit", message ) )
        {
            break;
        }

        MProgressWindow::setProgress( percentage );
        MProgressWindow::setProgressStatus( MString( message ) );

        m_interfaceLock->enter();
        if (MProgressWindow::isCancelled())
        {
            m_isCancelled_mainThread = m_isCancelled = true;
        }
        m_interfaceLock->leave();
    }
}

void hctMayaConvexDecompProgressHandler::setupBar()
{
    bool reserved = MProgressWindow::reserve();

    HK_ASSERT( 0x6dbb5221, reserved, "MProgressWindow could not be reserved" );

    if ( !reserved )
    {
        return;
    }

    hkStringBuf report_str;
    if (m_isCombine)
    {
        report_str.printf( "Combine" );
    }
    else
    {
        report_str.printf( "Decomposition" );
    }

    MProgressWindow::startProgress();
    MProgressWindow::setInterruptable( true );
    MProgressWindow::setProgressStatus( MString( report_str.cString() ) );
    MProgressWindow::setProgressMin(0);
    MProgressWindow::setProgressMax(100);

    if (m_isCombine)
    {
        MProgressWindow::setTitle( MString( "Combine progress" ) );
    }
    else
    {
        MProgressWindow::setTitle( MString( "Decompose progress" ) );
    }

    MGlobal::displayInfo( report_str.cString() );

    m_nFinished = 0;
    m_nJobs = 0;
    m_isCancelled = false;
    m_wasCancelPressed = false;
}

void hctMayaConvexDecompProgressHandler::cleanupBar( const bool result )
{
    MProgressWindow::setProgressStatus( MString( "" ) );
    MProgressWindow::endProgress();

    hkStringBuf report_str;
    if (m_isCombine)
    {
        report_str.printf( "Combine complete" );
    }
    else
    {
        report_str.printf( "Decomposition complete" );
    }

    MGlobal::displayInfo( report_str.cString() );
}

//
// Utility functions
//

// Find and load the convex decomposition DLL
bool _loadLibs( hctConvexDecompUtils** convexDecompUtils, hctBaseDll** convexDecompBaseDll, HMODULE& convexDecompHmodule, MString pluginPath, MString& errorMessage, hkBool& displayErrorMessage )
{
    MString convexDecompUtilsDllName;
    {
        MString filterManagerPath;
        const char* pluginPathString = pluginPath.asChar();
        hctMayaUtilities::getFilterManagerPath( pluginPathString, filterManagerPath );
        convexDecompUtilsDllName = filterManagerPath.asChar() + MString( "\\utils\\hctConvexDecompUtils.dll" );

        displayErrorMessage = FALSE;

        // Find library file
        hkIfstream testStream( convexDecompUtilsDllName.asChar() );
        if ( !testStream.isOk() )
        {
            errorMessage = "Convex Decomposition Util DLL not found in filter manager's util dir";
            HK_WARN_ALWAYS( 0x7b5ac0db, errorMessage.asChar() );
            return false;
        }

        displayErrorMessage = TRUE;

        // Load library
        convexDecompHmodule = LoadLibrary( convexDecompUtilsDllName.asChar() );
        if ( !convexDecompHmodule )
        {
            errorMessage = "Convex Decomposition Util DLL found but could not load. The DLL is corrupt.";
            HK_WARN_ALWAYS( 0x299d37e4, errorMessage.asChar() );
            return false;
        }

        // Get base DLL interface
        hctGetBaseDllInterfaceFunc convexDecompBaseDllFunc = ( hctGetBaseDllInterfaceFunc ) GetProcAddress( convexDecompHmodule, "getBaseDllInterface" );
        if ( !convexDecompBaseDllFunc )
        {
            errorMessage = "Convex Decomposition DLL found but could not find baseDLL sync function. Install is invalid.";
            HK_WARN_ALWAYS( 0x6fdba42b, errorMessage.asChar() );
            return false;
        }

        // Get convexDecompUtil DLL interface
        *convexDecompBaseDll = convexDecompBaseDllFunc();
        if ( *convexDecompBaseDll )
        {
            if ( (*convexDecompBaseDll)->getDllVersion() != HCT_CURRENT_VERSION )
            {
                errorMessage = "Could not load Convex Decomposition Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall.";
                HK_WARN_ALWAYS( 0x7f425fa0, errorMessage.asChar() );
                return false;
            }

            hkMemoryInitUtil::SyncInfo baseSystemInfo;
            hkSceneExportMemory::getBaseSystemSyncInfo( baseSystemInfo );
            (*convexDecompBaseDll)->initDll( baseSystemInfo, HK_NULL  );

            hctGetConvexDecompUtilInterfaceFunc convexDecompInterfaceFunc = ( hctGetConvexDecompUtilInterfaceFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompUtilInterface" );
            *convexDecompUtils = convexDecompInterfaceFunc ? convexDecompInterfaceFunc() : HK_NULL;

            if ( *convexDecompUtils )
            {
                // Set callback in dll such that a memory router created on a thread can be set back here in the modeler dll
                (*convexDecompUtils)->setThreadCallback( &hkSceneExportMemory::g_defaultThreadCallback );
            }
        }

        // Handle errors
        if ( !(*convexDecompUtils) || !(*convexDecompBaseDll) )
        {
            hctGetConvexDecompDllErrorFunc convexDecompErrorFunc = ( hctGetConvexDecompDllErrorFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompDllError" );
            if ( !convexDecompErrorFunc )
            {
                errorMessage = "Could not query Convex Decomposition Util DLL. Maybe your install is corrupt.";
                HK_WARN_ALWAYS( 0x34e5aa70, errorMessage.asChar() );
                return false;
            }

            hctConvexDecompUtils::DllError convexDecompError = convexDecompErrorFunc();

            if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_ILLEGAL )
            {
                errorMessage = "The keycode for the Convex Decomposition Util DLL is invalid or is for a different product version. Please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x5a9a1283, errorMessage.asChar() );
                return false;
            }

            if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_EXPIRED )
            {
                errorMessage = "The keycode for the Convex Decomposition Util DLL is expired or is for a different product version. Please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x7441b3ff, errorMessage.asChar() );
                return false;
            }

            if ( convexDecompError & hctConvexDecompUtils::DLL_INIT_ERROR )
            {
                errorMessage = "The Convex Decomposition Util DLL failed to initialize properly. Please try restarting Maya. If this error repeats, please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x146fc145, errorMessage.asChar() );
                return false;
            }

            errorMessage = "The Convex Decomposition Util DLL failed to load. Maybe your install is corrupt. If this error repeats, please contact Havok support at http://support.havok.com";
            HK_WARN_ALWAYS(0x21270eed, errorMessage.asChar() );
            return false;
        }

        // Warn user if keycode is about to expire
        hctGetConvexDecompDllErrorFunc convexDecompErrorFunc = ( hctGetConvexDecompDllErrorFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompDllError" );
        if ( !convexDecompErrorFunc )
        {
            errorMessage = "Could not query Convex Decomposition Util DLL. Maybe your install is corrupt.";
            HK_WARN_ALWAYS(0x7b16680f, errorMessage.asChar() );
            return false;
        }

        hctConvexDecompUtils::DllError convexDecompError = convexDecompErrorFunc();

        if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_WARNING )
        {
            errorMessage = "The Convex Decomposition keycode expires in 10 days or less.";
            displayErrorMessage = TRUE;
            HK_WARN_ALWAYS(0x4dc2efa9, errorMessage.asChar() );
        }
    }
    return true;
}

void _cleanupLibs( hctConvexDecompUtils* convexDecompUtils, hctBaseDll* convexDecompBaseDll, HMODULE& convexDecompHmodule )
{
    if ( convexDecompUtils )
    {
        delete convexDecompUtils;
    }

    // Cleanup
    if ( convexDecompBaseDll )
    {
        convexDecompBaseDll->quitDll();
    }

    if ( convexDecompHmodule )
    {
        FreeLibrary( convexDecompHmodule );
    }
}

//
// Maya interface
//

MSyntax hctCmdCreateConvexDecomposition::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag(checkDllValidityFlag, checkDllValidityFlagLong, MSyntax::kBoolean);
    syntax.addFlag(detachPartsFlag, detachPartsFlagLong, MSyntax::kBoolean);
    syntax.addFlag(singleOutputFlag, singleOutputFlagLong, MSyntax::kBoolean);
    syntax.addFlag(accuracyFlag, accuracyFlagLong, MSyntax::kDouble);
    syntax.addFlag(singleOutputCombineFlag, singleOutputCombineFlagLong, MSyntax::kBoolean);
    syntax.addFlag(useExistingFlag, useExistingFlagLong, MSyntax::kBoolean);
    syntax.addFlag(generateGuardsFlag, generateGuardsFlagLong, MSyntax::kBoolean);
    syntax.addFlag(reduceOverlapFlag, reduceOverlapFlagLong, MSyntax::kLong);
    syntax.addFlag(octreeDepthFlag, octreeDepthFlagLong, MSyntax::kLong);
    syntax.addFlag(decomposeMethodFlag, decomposeMethodFlagLong, MSyntax::kLong);
    syntax.addFlag(reduceMethodFlag, reduceMethodFlagLong, MSyntax::kLong);
    syntax.addFlag(wrapIterationsFlag, wrapIterationsFlagLong, MSyntax::kLong);
    syntax.addFlag(wrapConcavityFlag, wrapConcavityFlagLong, MSyntax::kDouble);
    syntax.setObjectType(MSyntax::kSelectionList);
    syntax.enableQuery(true);

    return syntax;
}

MStatus hctCmdCreateConvexDecomposition::redoIt()
{
    return dagMod.doIt();
}

MStatus hctCmdCreateConvexDecomposition::undoIt()
{
    return dagMod.undoIt();
}

bool g_displayDllAlert = true;

MStatus hctCmdCreateConvexDecomposition::doIt(const MArgList& args)
{
    HMODULE convexDecompHmodule = HK_NULL;
    hctBaseDll* convexDecompBaseDll = HK_NULL;
    MFnPlugin pluginFn( m_pluginObject );
    MString pluginPath = pluginFn.loadPath();
    MString errorMessage;
    hkBool displayErrorMessage;
    m_convexDecompUtils = HK_NULL;
    bool isLibLoaded = _loadLibs( &m_convexDecompUtils, &convexDecompBaseDll, convexDecompHmodule, pluginPath, errorMessage, displayErrorMessage );

    // Get command arguments
    MArgDatabase argData(syntax(), args);

    // Set defaults
    m_checkValidity = false;
    m_detachParts = true;
    m_singleOut = false;
    m_accuracy = 0.0f;
    m_singleOutCombine = false;
    m_useExisting = false;
    m_generateGuards = false;
    m_reduceOverlapSteps = 0;
    m_octreeDepth = 0;
    m_decomposeMethod = 0;
    m_reduceMethod = 1;
    m_wrapIterations = 0;
    m_wrapConcavity = 0.0f;

    if (argData.isFlagSet( checkDllValidityFlag ))
    {
        bool checkValidity;
        argData.getFlagArgument( checkDllValidityFlag, 0, checkValidity );
        m_checkValidity = checkValidity;
    }
    if (argData.isFlagSet( detachPartsFlag ))
    {
        bool detachParts;
        argData.getFlagArgument( detachPartsFlag, 0, detachParts );
        m_detachParts = detachParts;
    }
    if (argData.isFlagSet( singleOutputFlag ))
    {
        bool singleOut;
        argData.getFlagArgument( singleOutputFlag, 0, singleOut );
        m_singleOut = singleOut;
    }
    if (argData.isFlagSet( accuracyFlag ))
    {
        double accuracy;
        argData.getFlagArgument( accuracyFlag, 0, accuracy );
        m_accuracy = (float)accuracy;
    }
    if (argData.isFlagSet( singleOutputCombineFlag ))
    {
        bool singleOutCombine;
        argData.getFlagArgument( singleOutputCombineFlag, 0, singleOutCombine );
        m_singleOutCombine = singleOutCombine;
    }
    if (argData.isFlagSet( useExistingFlag ))
    {
        bool useExisting;
        argData.getFlagArgument( useExistingFlag, 0, useExisting );
        m_useExisting = useExisting;
    }
    if (argData.isFlagSet( generateGuardsFlag ))
    {
        bool generateGuards;
        argData.getFlagArgument( generateGuardsFlag, 0, generateGuards );
        m_generateGuards = generateGuards;
    }
    if (argData.isFlagSet( reduceOverlapFlag ))
    {
        int reduceOverlapSteps;
        argData.getFlagArgument( reduceOverlapFlag, 0, reduceOverlapSteps );
        m_reduceOverlapSteps = reduceOverlapSteps;
    }
    if (argData.isFlagSet( octreeDepthFlag ))
    {
        int octreeDepth;
        argData.getFlagArgument( octreeDepthFlag, 0, octreeDepth );
        m_octreeDepth = octreeDepth;
    }
    if (argData.isFlagSet( decomposeMethodFlag ))
    {
        int decomposeMethod;
        argData.getFlagArgument( decomposeMethodFlag, 0, decomposeMethod );
        m_decomposeMethod = decomposeMethod;
    }
    if (argData.isFlagSet( reduceMethodFlag ))
    {
        int reduceMethod;
        argData.getFlagArgument( reduceMethodFlag, 0, reduceMethod );
        m_reduceMethod = reduceMethod;
    }
    if (argData.isFlagSet( wrapIterationsFlag ))
    {
        int wrapIterations;
        argData.getFlagArgument( wrapIterationsFlag, 0, wrapIterations );
        m_wrapIterations = wrapIterations;
    }
    if (argData.isFlagSet( wrapConcavityFlag ))
    {
        double wrapConcavity;
        argData.getFlagArgument( wrapConcavityFlag, 0, wrapConcavity );
        m_wrapConcavity = (float) wrapConcavity;
    }

    // Get the objects in the command selection list
    MSelectionList commandList;
    argData.getObjects(commandList);

    if ( m_checkValidity )
    {
        clearResult();

        if ( isLibLoaded )
        {
            setResult( 1 );
        }
        else
        {
            setResult( 0 );
        }

        if ( ( 0 != errorMessage.length() ) && g_displayDllAlert && displayErrorMessage )
        {
            MessageBox( NULL, errorMessage.asChar(), "Warning", MB_OK );
            g_displayDllAlert = false;
        }

        goto CLEANUP_LIBS;
    }

    clearResult();
    setResult( 0 );

    if ( isLibLoaded )
    {
        if (0 == commandList.length())
        {
            goto CLEANUP_LIBS;
        }

        MStatus status;
        if (0 == m_useExisting)
        {
            status = convexDecomposition( commandList );
        }
        else
        {
            status = combineConvexShapes( commandList );
        }

        int resultValue = ( status == MStatus::kSuccess ) ? 1 : 0;
        clearResult();
        setResult( resultValue );
    }

CLEANUP_LIBS:
    _cleanupLibs( m_convexDecompUtils, convexDecompBaseDll, convexDecompHmodule );
    return MStatus::kSuccess;
}

//
// Convex Decomposition implementation
//

MStatus hctCmdCreateConvexDecomposition::convexDecomposition( const MSelectionList& commandList )
{
    if (commandList.length() < 2)
    {
        return MStatus::kFailure;
    }

    MDagPath baseDagPath;
    commandList.getDagPath( 0, baseDagPath );
    MString baseDagPathName = baseDagPath.partialPathName();

    // Extract set of world space vertices from selected meshes
    MStatus status;
    hctConvexDecompResults results;
    hkGeometry geometryFromScene;
    status = createGeometryFromNodes( baseDagPath, commandList, geometryFromScene, results );
    if (status != MStatus::kSuccess)
    {
        status.perror( "Error creating geometry, " );
        MProgressWindow::endProgress();
        return status;
    }

    // There are quite a lot of absolute epsilons in CD, therefore better to work with a normalized mesh
    hkTransform normalizedToWorld;
    {
        hkVector4 normalizeCenter;
        m_convexDecompUtils->normalizeGeometry( geometryFromScene, 10.0f, &normalizedToWorld, &normalizeCenter );
    }

    results.nInShapes = commandList.length() - 1;

    status = displayWarning( results.nInTriangles, 1 );
    if (MStatus::kSuccess != status)
    {
        MProgressWindow::endProgress();
        return status;
    }

    bool decomposeResult;
    hkArray<hkArray<hkVector4>> shapes;
    {
        hctConvexDecompUtils::Input input;
        hctMayaConvexDecompProgressHandler progressHandler;
        input.m_progressJobHandler = (hctConvexDecompProgressHandler*)&progressHandler;
        input.setupDecomposeInput( m_detachParts, m_reduceOverlapSteps, m_accuracy, m_octreeDepth, m_decomposeMethod, m_reduceMethod, m_wrapIterations, m_wrapConcavity );

        // call utility
        decomposeResult = m_convexDecompUtils->decompose( input, geometryFromScene, shapes );
    }

    // check result and create geometry from vertices
    if ( !decomposeResult || 0 == shapes.getSize() )
    {
        status.perror( "Failed to generate decomposition" );
        MProgressWindow::endProgress();
        return MStatus::kFailure;
    }
    else
    {
        // create group node to which to 'parent' shapes
        MObject groupHead;
        groupHead = dagMod.createNode( MString( "transform" ), MObject::kNullObj, &status );
        if (MStatus::kSuccess != status)
        {
            MProgressWindow::endProgress();
            return status;
        }
        dagMod.renameNode( groupHead, baseDagPathName + "__hkConvDecGroup" );

        MSelectionList newSelList;
        MSelectionList connectList;

        status = createOutputNodesFromPoints( baseDagPathName, baseDagPath, groupHead, shapes, normalizedToWorld, newSelList, connectList, results );
        if (MStatus::kSuccess != status)
        {
            MProgressWindow::endProgress();
            return status;
        }

        MGlobal::setActiveSelectionList( newSelList );

        connectNodesToBaseMesh( connectList, baseDagPath );
        connectNodesToShadingGroup( newSelList );

        // execute all the operations given to this modifier
        dagMod.doIt();

        displayResults( results, true );
    }

    MProgressWindow::endProgress();

    return status;
}

MStatus hctCmdCreateConvexDecomposition::combineConvexShapes( const MSelectionList& commandList )
{
    // Original mesh is passed as a parameter, such that new shapes can be parented under it
    MDagPath baseDagPath;
    commandList.getDagPath( 0, baseDagPath );
    MString baseDagPathName = baseDagPath.partialPathName();

    // Command list contains nodes within the original mesh
    MStatus status;
    hctConvexDecompResults results;
    hkGeometry geometryFromScene;
    status = createGeometryFromNodes( baseDagPath, commandList, geometryFromScene, results );
    if (status != MStatus::kSuccess)
    {
        status.perror("Error creating geometry");
        MProgressWindow::endProgress();
        return status;
    }

    hkTransform normalizedToWorld;
    hkVector4 normalizeCenter;
    hkReal normalizeScale;
    normalizeScale = m_convexDecompUtils->normalizeGeometry( geometryFromScene, 10.0f, &normalizedToWorld, &normalizeCenter );

    // retrieve nodes that are to be combined (those selected in scene)
    MSelectionList combineList;
    status = MGlobal::getActiveSelectionList( combineList );
    if (status != MStatus::kSuccess)
    {
        MProgressWindow::endProgress();
        return status;
    }

    results.reset();

    // extract set of world space vertices from selected meshes
    hkArray<hkArray<hkVector4>> shapesToCombine;
    status = getVertexArrayFromSelectionList( combineList, shapesToCombine, results );

    status = displayWarning( results.nInPlanes, false );
    if (MStatus::kSuccess != status)
    {
        MProgressWindow::endProgress();
        return status;
    }

    m_convexDecompUtils->normalizeShapes( shapesToCombine, normalizeScale, normalizeCenter );

    bool combineResult;
    hkArray<hkArray<hkVector4>> shapes;
    {
        hctConvexDecompUtils::Input input;
        hctMayaConvexDecompProgressHandler progressHandler;
        input.m_progressJobHandler = (hctConvexDecompProgressHandler*)&progressHandler;
        input.setupCombineInput( m_generateGuards, m_accuracy, m_octreeDepth, m_reduceMethod );

        // Call utility
        combineResult = m_convexDecompUtils->combine( input, geometryFromScene, shapes, &shapesToCombine );
    }

    // Check result and create geometry from vertices
    if ( !combineResult || 0 == shapes.getSize() )
    {
        status.perror( "Failed to generate decomposition" );
        MProgressWindow::endProgress();
        return MStatus::kFailure;
    }
    else
    {
        MObject groupHead = retrieveGroupHead( combineList, &status );
        if (MStatus::kSuccess != status)
        {
            MProgressWindow::endProgress();
            return MStatus::kFailure;
        }

        MSelectionList newSelList;
        MSelectionList connectList;

        status = createOutputNodesFromPoints( baseDagPathName, baseDagPath, groupHead, shapes, normalizedToWorld, newSelList, connectList, results );
        if( status != MStatus::kSuccess )
        {
            status.perror("Error creating nodes");
            MProgressWindow::endProgress();
            return MStatus::kFailure;
        }

        connectNodesToBaseMesh( connectList, baseDagPath );

        connectNodesToShadingGroup( newSelList );

        deleteCombinedNodes( combineList );

        MGlobal::selectCommand( newSelList );

        // execute all the operations given to this modifier
        status = dagMod.doIt();
        if ( status != MStatus::kSuccess )
        {
            status.perror( "Changes to DAG failed" );
            MProgressWindow::endProgress();
            return MStatus::kFailure;
        }

        displayResults( results, false );
    }

    MProgressWindow::endProgress();

    return status;
}

MStatus hctCmdCreateConvexDecomposition::deleteCombinedNodes( MSelectionList& combineList )
{
    MDagPath dagPath;
    MStatus status;
    MObject node;
    MFn::Type type;

    for (unsigned int i = 0; i < combineList.length(); ++i)
    {
        combineList.getDagPath( i, dagPath );

        node = dagPath.node( &status );
        if (status != MStatus::kSuccess)
        {
            return MStatus::kFailure;
        }

        type = node.apiType();
        if (type == MFn::kMesh)
        {
            dagPath.pop();
            node = dagPath.node( &status );
            if (status != MStatus::kSuccess)
            {
                return MStatus::kFailure;
            }
        }

        status = dagMod.deleteNode( node );
        if (status != MStatus::kSuccess)
        {
            return MStatus::kFailure;
        }
    }

    return MStatus::kSuccess;
}

MObject hctCmdCreateConvexDecomposition::retrieveGroupHead( MSelectionList& combineList, MStatus* status )
{
    MDagPath dagPath;
    MObject object;

    *status = combineList.getDagPath( 0, dagPath );
    if (MStatus::kSuccess != *status)
    {
        *status = MStatus::kFailure;
        return object;
    }

    object = dagPath.node( status );
    if (MStatus::kSuccess != *status)
    {
        *status = MStatus::kFailure;
        return object;
    }

    MFn::Type type = object.apiType();
    if (MFn::kMesh == type)
    {
        dagPath.pop();
        object = dagPath.node( status );
        if (MStatus::kSuccess != *status)
        {
            *status = MStatus::kFailure;
            return object;
        }
    }

    dagPath.pop();

    object = dagPath.node( status );
    if (MStatus::kSuccess != *status)
    {
        *status = MStatus::kFailure;
        return object;
    }

    return object;
}

MStatus hctCmdCreateConvexDecomposition::connectNodesToBaseMesh( MSelectionList& nodeList, MDagPath& baseDagPath )
{
    MStatus status;

    MFnDependencyNode baseDepNode (baseDagPath.node(), &status);
    if (MStatus::kSuccess != status)
    {
        MGlobal::displayError( "Error creating base dagpath dependency node" );
        return MStatus::kFailure;
    }

    MPlug basePlug = baseDepNode.findPlug( "message", &status );
    if (MStatus::kSuccess != status)
    {
        MGlobal::displayError( "Error creating base message plug" );
        return MStatus::kFailure;
    }

    MFnMessageAttribute fnMessageAttrib;
    MFnNumericAttribute fnNumericAttrib;
    MFnTransform fnTransorm;
    MDagPath newHullDagPath;
    MFnDependencyNode newHullDepNode;
    for (unsigned int i = 0; i < nodeList.length(); ++i)
    {
        nodeList.getDagPath( i, newHullDagPath );

        MObject newHullObj = newHullDagPath.node( &status );
        if (MStatus::kSuccess != status)
        {
            MGlobal::displayError( "Error creating new hull mobject" );
            return MStatus::kFailure;
        }

        newHullDepNode.setObject( newHullObj );

        // attribute to connect to original mesh
        {
            MObject connectionAttribute = fnMessageAttrib.create( "origMesh", "om", &status );
            if (MStatus::kSuccess != status)
            {
                MGlobal::displayError( "Error creating base attribute" );
                return MStatus::kFailure;
            }

            newHullDepNode.addAttribute( connectionAttribute );
            MPlug newHullPlug = newHullDepNode.findPlug( connectionAttribute );

            status = dagMod.connect( basePlug, newHullPlug );
            if (MStatus::kSuccess != status)
            {
                MGlobal::displayError( "Error connecting nodes" );
                return MStatus::kFailure;
            }
        }
    }

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::connectNodesToShadingGroup( const MSelectionList& dagList )
{
    MSelectionList sgSelList;
    MObject sgObj;
    {
        MGlobal::getSelectionListByName( MString( "initialShadingGroup" ), sgSelList );
        sgSelList.getDependNode( 0, sgObj );
    }

    MStatus status;
    MDagPath nodeDag;
    MObject nodeObj;
    MItSelectionList iter( dagList );

    for ( ; !iter.isDone(); iter.next() )
    {
        iter.getDagPath( nodeDag, nodeObj );
        nodeDag.extendToShape();

        MFnSet fnSG( sgObj, &status );
        if ( fnSG.restriction() != MFnSet::kRenderableOnly )
        {
            return MStatus::kFailure;
        }

        status = fnSG.addMember( nodeDag, nodeObj );
        if ( status != MStatus::kSuccess )
        {
            return MStatus::kFailure;
        }
    }

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::getVertexArrayFromSelectionList( const MSelectionList& dagList, hkArray<hkArray<hkVector4> >& shapesToCombine, hctConvexDecompResults& results )
{
    MStatus status;
    MDagPath dagPath;
    MFnTransform fnTrans;
    MDagPath meshDagPath;
    MObject object;
    MFn::Type type;

    for (unsigned int i = 0; i < dagList.length(); ++i)
    {
        dagList.getDagPath( i, dagPath );

        object = dagPath.node();
        type = object.apiType();

        if (type == MFn::kMesh)
        {
            meshDagPath = dagPath;
        }
        else
        {
            status = fnTrans.setObject( dagPath );
            if (status != MStatus::kSuccess)
            {
                MGlobal::displayError("Error, selection is not a transform node");
                return MStatus::kFailure;
            }

            for (unsigned int dagIndex = 0; dagIndex < dagPath.childCount(); ++dagIndex)
            {
                type = dagPath.child(dagIndex).apiType();
                if (type == MFn::kMesh)
                {
                    const MDagPath& temp = dagPath;
                    meshDagPath = temp;
                    meshDagPath.push( dagPath.child(dagIndex) );
                    break;
                }
            }
        }

        if (!meshDagPath.isValid() ||
            meshDagPath.apiType() != MFn::kMesh)
        {
            return MStatus::kFailure;
        }

        hkArray<hkVector4>& vertices = shapesToCombine.expandOne();

        status = getVerticesFromDagPath( meshDagPath, vertices, results );
        if (status != MStatus::kSuccess)
        {
            return MStatus::kFailure;
        }
    }

    results.nInShapes = dagList.length();

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::getVerticesFromDagPath( const MDagPath& baseDagPath, hkArray<hkVector4>& vertices, hctConvexDecompResults& results )
{
    MStatus status;

    MFnMesh mesh;
    status = mesh.setObject( baseDagPath );

    if (status != MStatus::kSuccess)
    {
        status.perror( "Error creating MFnMesh" );
        return status;
    }

    // get the object space vertex positions
    MPointArray newVertexArray;
    status = mesh.getPoints( newVertexArray, MSpace::kWorld );
    if (status != MStatus::kSuccess)
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    MIntArray triangleCounts;
    MIntArray triangleVertices;
    mesh.getTriangles( triangleCounts, triangleVertices );

    // transform into world space
    vertices.setSize( newVertexArray.length() );

    for (unsigned int vi = 0; vi < newVertexArray.length(); ++vi)
    {
        MPoint& p = newVertexArray[vi];
        hkVector4 v( static_cast<hkReal>(p.x), static_cast<hkReal>(p.y), static_cast<hkReal>(p.z) );
        vertices[vi] = v;
    }

    // calculate n planes for results
    {
        hkGeometry geometry;
        geometry.m_vertices = vertices;

        hkArray<hkGeometry::Triangle>& nodeTriangles = geometry.m_triangles;
        nodeTriangles.setSize( triangleVertices.length() / 3 );

        int numTriangleVertices = triangleVertices.length();
        for (int ti = 0, tvi = 0; tvi < numTriangleVertices; ++ti)
        {
            nodeTriangles[ti].m_a = triangleVertices[tvi++];
            nodeTriangles[ti].m_b = triangleVertices[tvi++];
            nodeTriangles[ti].m_c = triangleVertices[tvi++];
        }

        int nPlanes = m_convexDecompUtils->getConvexHullPlanes( geometry );

        results.nInPlanes += nPlanes;
    }

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::getGeometryFromDagPath( const MDagPath& baseDagPath, const MDagPath& dagPath, hkGeometry& combinedGeometry, hctConvexDecompResults& results )
{
    MStatus status;
    const MDagPath& node = dagPath;
    MDagPath meshPath = node;

    for (unsigned int i = 0; i < dagPath.childCount(); ++i)
    {
        MFn::Type type = dagPath.child(i).apiType();

        if (type == MFn::kMesh)
        {
            meshPath.push(dagPath.child(i));
            break;
        }
    }

    if (!meshPath.isValid() ||
        meshPath.apiType() != MFn::kMesh)
    {
        return MStatus::kFailure;
    }

    MFnMesh mesh;
    status = mesh.setObject( meshPath );
    if (status != MStatus::kSuccess)
    {
        status.perror( "Error creating MFnMesh" );
        return status;
    }

    // get the object space vertex positions
    MPointArray newVertexArray;
    status = mesh.getPoints( newVertexArray, MSpace::kWorld );
    if (status != MStatus::kSuccess)
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    // transform into world space
    hkGeometry objectGeometry;
    hkArray<hkVector4>& geometryVertices = objectGeometry.m_vertices;

    geometryVertices.setSize ( newVertexArray.length() );
    for (unsigned int vi = 0; vi < newVertexArray.length(); ++vi)
    {
        MPoint& p = newVertexArray[vi];
        hkVector4 v( static_cast<hkReal>(p.x), static_cast<hkReal>(p.y), static_cast<hkReal>(p.z) );
        geometryVertices[vi] = v;
    }

    // get vertex indices for each triangle
    MIntArray triangleCounts;
    MIntArray triangleVertices;

    status = mesh.getTriangles( triangleCounts, triangleVertices );
    if (status != MStatus::kSuccess)
    {
        status.perror( "MFnMesh::getTriangles" );
        return status;
    }

    hkArray<hkGeometry::Triangle>& geometryTriangles = objectGeometry.m_triangles;
    geometryTriangles.setSize(triangleVertices.length() / 3);

    results.nInTriangles += triangleVertices.length() / 3;

    int numTriangleVertices = triangleVertices.length();
    for (int ti = 0, tvi = 0; tvi < numTriangleVertices; ++ti)
    {
        geometryTriangles[ti].m_a = triangleVertices[tvi++];
        geometryTriangles[ti].m_b = triangleVertices[tvi++];
        geometryTriangles[ti].m_c = triangleVertices[tvi++];
    }

    m_convexDecompUtils->mergeGeometries( objectGeometry, combinedGeometry );

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::createDagPathFromGeometry( const hkGeometry& geometry, const MString& shapeName, MObject& baseShape, MObject& groupHead, MSelectionList& newSelList, MSelectionList& connectList, hctConvexDecompResults& results, const hkTransform* normalizedToWorld )
{
    // create a mesh from the input geometry
    int numPoints = geometry.m_vertices.getSize();
    MPointArray points(numPoints);

    bool doTransform = false;
    hkTransform ntw;
    if (NULL != normalizedToWorld)
    {
        ntw = *normalizedToWorld;
        doTransform = true;
    }

    hkVector4 v = geometry.m_vertices[0];
    if (doTransform)
    {
        v.setTransformedPos( ntw, v );
    }
    hkVector4 lowerBounds = v;
    hkVector4 upperBounds = v;

    for (int vi = 0; vi < numPoints; ++vi)
    {
        hkVector4 vertex = geometry.m_vertices[vi];
        if (doTransform)
        {
            vertex.setTransformedPos( ntw, vertex);
        }

        MPoint p;
        p.x = vertex(0);
        p.y = vertex(1);
        p.z = vertex(2);
        points[vi] = p;
        lowerBounds.setMax4( lowerBounds, vertex);
        upperBounds.setMin4( upperBounds, vertex);
    }

    int numTriangles = geometry.m_triangles.getSize();
    MIntArray polygonCounts(numTriangles, 3);
    MIntArray polygonConnects(numTriangles * 3);

    for (int ti = 0; ti < numTriangles; ++ti)
    {
        polygonConnects[3 *ti     ] = geometry.m_triangles[ti].m_a;
        polygonConnects[3 * ti + 1] = geometry.m_triangles[ti].m_b;
        polygonConnects[3 * ti + 2] = geometry.m_triangles[ti].m_c;
    }

    // create parent transform for mesh, parented to group
    MStatus status;
    MObject meshTransform = dagMod.createNode(MString("transform"), groupHead, &status);
    if (MStatus::kSuccess != status)
    {
        return status;
    }

    // doIt is called after all hulls are created
    MString name = shapeName + "__hkConvDec";
    dagMod.renameNode( meshTransform, name );

    // get cog of shape to set pivot point
    hkVector4 ext;
    ext.setSub4( upperBounds, lowerBounds );
    ext.mul4( 0.5f );
    ext.add4( lowerBounds );

    MPoint pivotPoint;
    pivotPoint.x = ext(0);
    pivotPoint.y = ext(1);
    pivotPoint.z = ext(2);

    // get matrix from base object
    MFnTransform fnTransorm;
    fnTransorm.setObject( baseShape );

    MVector trans = fnTransorm.getTranslation( MSpace::kWorld );

    fnTransorm.setObject( meshTransform );
    fnTransorm.translateBy( trans, MSpace::kWorld );
    fnTransorm.setRotatePivot( pivotPoint, MSpace::kObject, true );
    fnTransorm.setScalePivot( pivotPoint, MSpace::kObject, true );

    // NB, the following mesh creation/connection is not done using MDagModifier, but that's OK since on undoing,
    // the transform containing the mesh is deleted, effectively undoing the creation and connection.
    

    // create a mesh parented to this transform
    MFnMesh meshFn;
    meshFn.create( numPoints, numTriangles, points, polygonCounts, polygonConnects, meshTransform, &status );
    if (MStatus::kSuccess != status)
    {
        return MStatus::kFailure;
    }

    MDagPath generatedMeshPath;
    meshFn.getPath( generatedMeshPath );
    newSelList.add( generatedMeshPath );

    MDagPath meshTransformPath;
    MDagPath::getAPathTo( meshTransform, meshTransformPath );
    connectList.add( meshTransformPath );

    int nPlanes = m_convexDecompUtils->getConvexHullPlanes( geometry );

    results.nOutPlanes += nPlanes;
    ++results.nOutShapes;

    return status;
}

MStatus hctCmdCreateConvexDecomposition::displayResults( hctConvexDecompResults& results, const bool isDecompose )
{
    MString cmd = MString("hkProcPhysics_displayResults( ") + results.nInShapes + MString(",") + results.nOutShapes + MString(",") + results.nInTriangles + MString(",") + results.nInPlanes + MString(",") + results.nOutPlanes + MString(",") + isDecompose + MString(" )");

    MGlobal::executeCommand( cmd );

    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::displayWarning( const int nTriangles, const bool isDecompose )
{
    if (isDecompose && nTriangles > m_triangleWarningThreshold)
    {
        MString result;
        MString cmd = MString("hkProcPhysics_displayWarning( ") + nTriangles + MString(",") + isDecompose + MString(" )");
        MGlobal::executeCommand(cmd, result, false, false);

        if (result == "Continue")
        {
            return MStatus::kSuccess;
        }
        return MStatus::kFailure;
    }
    // n planes are used for combine
    else if (!isDecompose && nTriangles > m_triangleWarningThreshold / 2)
    {
        MString result;
        MString cmd = MString("hkProcPhysics_displayWarning( ") + nTriangles + MString(",") + isDecompose + MString(" )");
        MGlobal::executeCommand(cmd, result, false, false);

        if (result == "Continue")
        {
            return MStatus::kSuccess;
        }
        return MStatus::kFailure;
    }
    return MStatus::kSuccess;
}

MStatus hctCmdCreateConvexDecomposition::createGeometryFromNodes( MDagPath& baseDagPath, const MSelectionList& commandList, hkGeometry& geometryFromScene, hctConvexDecompResults& results )
{
    MStatus status = MStatus::kFailure;
    MDagPath dagPath;
    MFnTransform tNodeFn;

    for (unsigned int i = 1; i < commandList.length(); ++i) // N.B. starts at 1
    {
        commandList.getDagPath( i, dagPath );

        status = tNodeFn.setObject( dagPath );
        if (status != MStatus::kSuccess)
        {
            status.perror( "Error, selection is not a transform node" );
            MProgressWindow::endProgress();
            return status;
        }

        status = getGeometryFromDagPath( baseDagPath, dagPath, geometryFromScene, results );
        if (status != MStatus::kSuccess)
        {
            MProgressWindow::endProgress();
            return status;
        }
    }
    return status;
}

MStatus hctCmdCreateConvexDecomposition::createOutputNodesFromPoints( MString& baseDagPathName, MDagPath& baseDagPath, MObject& groupHead, hkArray< hkArray<hkVector4> >& shapes, hkTransform& normalizedToWorld, MSelectionList& newSelList, MSelectionList& connectList, hctConvexDecompResults& results )
{
    MStatus status;

    MObject baseObject = baseDagPath.node();
    hkGeometry combinedGeometry;
    for (int si = 0; si < shapes.getSize(); ++si)
    {
        hkGeometry shapeGeometry;
        m_convexDecompUtils->createConvexGeometry( shapes[si], shapeGeometry );

        if (m_singleOut)
        {
            m_convexDecompUtils->mergeGeometries( shapeGeometry, combinedGeometry );
        }
        else
        {
            // create a new mesh from the output geometry
            status = createDagPathFromGeometry( shapeGeometry, baseDagPathName, baseObject, groupHead, newSelList, connectList, results, &normalizedToWorld );

            if( status != MStatus::kSuccess )
            {
                status.perror("Failed to create decomposition mesh");
                return status;
            }

            status = MStatus::kSuccess;
        }
    }

    if (m_singleOut)
    {
        // create a new mesh from the output geometry
        status = createDagPathFromGeometry( combinedGeometry, baseDagPathName, baseObject, groupHead, newSelList, connectList, results, &normalizedToWorld );
        if (status != MStatus::kSuccess)
        {
            status.perror( "Failed to create decomposition mesh" );
            return status;
        }

        status = MStatus::kSuccess;
    }

    return status;
}

MStatus hctCmdCreateConvexDecomposition::addNodesToShadingGroup( const MSelectionList& nodeList )
{
    MStatus status;
    MDagPath meshPath;
    for ( unsigned int i = 0; i < nodeList.length(); ++i )
    {
        status = nodeList.getDagPath( i, meshPath );

        if ( MStatus::kSuccess != status )
        {
            MProgressWindow::endProgress();
            return status;
        }

        MString command = MString( "connectAttr -na ( \"" ) + meshPath.fullPathName() + MString( "\" + \".instObjGroups\") \"initialShadingGroup.dagSetMembers\" " );

        MGlobal::executeCommand( command );
    }

    return MStatus::kSuccess;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
