// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <maya/MSimple.h>
#include <maya/MGlobal.h>
#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MSelectionList.h>
#include <maya/MFloatPointArray.h>
#include <maya/MFnIkJoint.h>

#include <ContentTools/Maya/MayaSceneExport/Utilities/hctUtilities.h>

// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SdkUtils/Cloth/BonesFromMesh/hctBonesFromMeshUtil.h>
#include <ContentTools/Maya/MayaSceneExport/Commands/CreateBonesFromMesh/hctCmdCreateBonesFromMesh.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>

// hctCmdCreateBonesFromMesh command flags
#define boneNormalAxis "-bna"
#define boneNormalAxisLong  "-boneNormalAxis"
#define boneTangentAxis "-bta"
#define boneTangentAxisLong "-boneTangentAxis"
#define globalTangentAxis "-gta"
#define globalTangentAxisLong   "-globalTangentAxis"
#define boneCreationMethod "-bcm"
#define boneCreationMethodLong  "-boneCreationMethod"

MStatus hctCmdCreateBonesFromMesh::createBonesFromMesh(const MSelectionList& commandList)
{
    MStatus status = MStatus::kFailure;

    if (commandList.length() == 0)
    {
        hkStringBuf warningMsg;
        warningMsg.printf("Failed to create bones: no mesh was selected.");
        MessageBox (0, warningMsg.cString(), "Bones From Mesh Utility Warning", MB_ICONEXCLAMATION | MB_OK);

        status.perror("Failed to create bones: no mesh was selected.");
        return status;
    }

    if(m_boneNormalAxis == m_boneTangentAxis)
    {
        hkStringBuf warningMsg;
        warningMsg.printf("Bone normal axis and bone tangent axis cannot be the same.");
        MessageBox (0, warningMsg.cString(), "Bones From Mesh Utility Warning", MB_ICONEXCLAMATION | MB_OK);

        status.perror( "Bone normal axis and bone tangent axis cannot be the same." );
        return status;
    }

    for(int objIdx = 0; objIdx < (int) commandList.length(); objIdx++)
    {
        // Extract set of world space vertices from selected mesh
        MDagPath dagPath;
        hkGeometry simulationMeshGeometry;
        {
            commandList.getDagPath( objIdx, dagPath );

            MFnTransform tNodeFn;
            status = tNodeFn.setObject( dagPath );
            if (status != MStatus::kSuccess)
            {
                MString errorMsg("Failed to create bones for " + dagPath.partialPathName() + " :object is not a transform node");
                status.perror( errorMsg );
                continue;
            }

            status = getGeometryFromDagPath( dagPath, simulationMeshGeometry);

            if (status != MStatus::kSuccess)
            {
                MString errorMsg("Failed to create bones for " + dagPath.partialPathName() + " :hctCmdCreateBonesFromMesh::getGeometryFromDagPath() returned false");
                status.perror( errorMsg );
                continue;
            }
        }

        int numVerts = simulationMeshGeometry.m_vertices.getSize();
        int numTriangles = simulationMeshGeometry.m_triangles.getSize();

        // Set the size of the array depending on the bone creation method
        hkArray<hkTransform> transformsOut;
        hkArray<hkBool> boneAxisChanged;
        {
            int numBones = 0;
            //For the moment, create per vertex
            if(m_boneCreationMethod == hctBonesFromMeshUtil::CREATE_BONE_PER_VERTEX)
            {
                numBones = numVerts;
            }
            else if(m_boneCreationMethod == hctBonesFromMeshUtil::CREATE_BONE_PER_TRIANGLE)
            {
                numBones = numTriangles;
            }
            transformsOut.setSize(numBones);
            boneAxisChanged.setSize(numBones, false);
        }

        // Create the bones
        if( hctBonesFromMeshUtil::createBones( simulationMeshGeometry, transformsOut, (hctBonesFromMeshUtil::BoneAxis)m_boneNormalAxis,
                                                    (hctBonesFromMeshUtil::BoneAxis)m_boneTangentAxis, (hctBonesFromMeshUtil::GlobalTangentAxis)m_globalTangentAxis,
                                                        boneAxisChanged, (hctBonesFromMeshUtil::BoneCreationMethod)m_boneCreationMethod ) )
        {
            int numBoneAxesChanged = 0;

            for(int jointIdx = 0; jointIdx < transformsOut.getSize() ; jointIdx++)
            {
                MStatus statusJointCreate;
                MFnIkJoint ikJointFn;

                MObject jointObj = ikJointFn.create(MObject::kNullObj, &statusJointCreate);

                if(statusJointCreate != MStatus::kSuccess)
                {
                    status.perror( "Error creating joint, " );
                    continue;
                }

                //Set the transform
                {
                    MFnDagNode dagNodeJoint(jointObj);
                    MDagPath jointDagPath;
                    dagNodeJoint.getPath(jointDagPath);

                    MFnTransform parentTransformFn(jointDagPath, &status);

                    HK_ALIGN16(float f[4][4]);
                    transformsOut[jointIdx].get4x4ColumnMajor((float*)f);
                    MMatrix mat( f );
                    parentTransformFn.set( mat );
                    const double scale[3] = {1.0,1.0,1.0};
                    parentTransformFn.setScale(scale);
                }

                // Rename the joint
                {
                    MDagModifier dagMod;
                    MString newName = dagPath.partialPathName() + "_BoneSim" ;
                    dagMod.renameNode(jointObj, newName);
                    dagMod.doIt();
                }

                if( boneAxisChanged[jointIdx] )
                {
                    numBoneAxesChanged++;
                }
            }

            if(numBoneAxesChanged > 0)
            {
                hkStringBuf warningMsg;
                warningMsg.printf("Global tangent axis was the same as the vertex/triangle normal for %d joints, new global tangent axis picked in these cases", numBoneAxesChanged);
                MessageBox (0, warningMsg.cString(), "Bones From Mesh Utility Warning", MB_ICONEXCLAMATION | MB_OK);
            }
        }
    }

    return MStatus::kSuccess;
}


MSyntax hctCmdCreateBonesFromMesh::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag( boneNormalAxis, boneNormalAxisLong, MSyntax::kUnsigned );
    syntax.addFlag( boneTangentAxis, boneTangentAxisLong, MSyntax::kUnsigned );
    syntax.addFlag( globalTangentAxis, globalTangentAxisLong, MSyntax::kUnsigned );
    syntax.addFlag( boneCreationMethod, boneCreationMethodLong, MSyntax::kUnsigned );
    syntax.setObjectType( MSyntax::kSelectionList );
    syntax.enableQuery( false );
    syntax.enableEdit( false );

    return syntax;
}

MStatus hctCmdCreateBonesFromMesh::redoIt()
{
    return dagMod.doIt();
}

MStatus hctCmdCreateBonesFromMesh::undoIt()
{
    return dagMod.undoIt();
}

MStatus hctCmdCreateBonesFromMesh::doIt( const MArgList& args )
{
    MStatus status;

    m_boneNormalAxis = hctBonesFromMeshUtil::B_Z_AXIS;
    m_boneTangentAxis = hctBonesFromMeshUtil::B_X_AXIS;
    m_globalTangentAxis = hctBonesFromMeshUtil::GT_Y_POSITIVE_AXIS;
    m_boneCreationMethod = hctBonesFromMeshUtil::CREATE_BONE_PER_VERTEX;

    // get command arguments
    MArgDatabase argData( syntax(), args );

    if (argData.isFlagSet( boneNormalAxis ))
    {
        unsigned bna;
        argData.getFlagArgument( boneNormalAxis, 0, bna);
        m_boneNormalAxis = bna;
    }

    if (argData.isFlagSet( boneTangentAxis ))
    {
        unsigned bta;
        argData.getFlagArgument( boneTangentAxis, 0, bta);
        m_boneTangentAxis = bta;
    }

    if (argData.isFlagSet( globalTangentAxis ))
    {
        unsigned gta;
        argData.getFlagArgument( globalTangentAxis, 0, gta);
        m_globalTangentAxis = gta;
    }

    if (argData.isFlagSet( boneCreationMethod ))
    {
        unsigned bcm;
        argData.getFlagArgument( boneCreationMethod, 0, bcm);
        m_boneCreationMethod = bcm;
    }

    // get the objects in the command selection list
    MSelectionList commandList;
    argData.getObjects( commandList );

    status = createBonesFromMesh( commandList );

    return status;
}



MStatus hctCmdCreateBonesFromMesh::getGeometryFromDagPath( /*const MDagPath& baseDagPath, */const MDagPath& dagPath, hkGeometry& geometry )
{
    MStatus status;
    const MDagPath& node = dagPath;
    MDagPath meshPath = node;

    for (unsigned int i = 0; i < dagPath.childCount(); ++i)
    {
        MFn::Type type = dagPath.child(i).apiType();

        if (type == MFn::kMesh)
        {
            meshPath.push(dagPath.child(i));
            break;
        }
    }

    if (!meshPath.isValid() ||
        meshPath.apiType() != MFn::kMesh)
    {
        return MStatus::kFailure;
    }

    MFnMesh mesh;
    status = mesh.setObject( meshPath );
    if (status != MStatus::kSuccess)
    {
        status.perror( "Error creating MFnMesh" );
        return status;
    }

    // get the object space vertex positions
    MPointArray newVertexArray;
    status = mesh.getPoints( newVertexArray, MSpace::kWorld );
    if (status != MStatus::kSuccess)
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    // transform into world space
    hkArray<hkVector4>& geometryVertices = geometry.m_vertices;

    geometryVertices.setSize ( newVertexArray.length() );
    for (unsigned int vi = 0; vi < newVertexArray.length(); ++vi)
    {
        MPoint& p = newVertexArray[vi];
        hkVector4 v( static_cast<hkReal>(p.x), static_cast<hkReal>(p.y), static_cast<hkReal>(p.z) );
        geometryVertices[vi] = v;
    }

    // get vertex indices for each triangle
    MIntArray triangleCounts;
    MIntArray triangleVertices;

    status = mesh.getTriangles( triangleCounts, triangleVertices );
    if (status != MStatus::kSuccess)
    {
        status.perror( "MFnMesh::getTriangles" );
        return status;
    }

    hkArray<hkGeometry::Triangle>& geometryTriangles = geometry.m_triangles;
    geometryTriangles.setSize(triangleVertices.length() / 3);

    int numTriangleVertices = triangleVertices.length();
    for (int ti = 0, tvi = 0; tvi < numTriangleVertices; ++ti)
    {
        geometryTriangles[ti].m_a = triangleVertices[tvi++];
        geometryTriangles[ti].m_b = triangleVertices[tvi++];
        geometryTriangles[ti].m_c = triangleVertices[tvi++];
    }

    return MStatus::kSuccess;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
