// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT    : COMMON
// VISIBILITY : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Maya/MayaSceneExport/hctMayaSceneExport.h>

#include <maya/MProgressWindow.h>

#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>
#include <Common/Internal/GeometryProcessing/CollisionGeometryOptimizer/hkgpCgo.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>

#include <ContentTools/Maya/MayaSceneExport/Commands/CollideGeometryOptimizer/hctCmdCgo.h>


// 'hkCmdCgo' command flags:

#define maxDistance_flag            "-e"
#define maxDistance_flagL           "-maxError"

#define targetNumVertices_flag      "-v"
#define targetNumVertices_flagL     "-targetNumVertices"

#define targetRatio_flag            "-r"
#define targetRatio_flagL           "-targetRatio"

#define shrinking_flag              "-s"
#define shrinking_flagL             "-shrinking"

#define multipass_flag              "-m"
#define multipass_flagL             "-multipass"

#define protMat_flag                "-pm"
#define protMat_flagL               "-protectMaterial"

#define protGeo_flag                "-pg"
#define protGeo_flagL               "-protectGeometry"

#define useVertexWeight_flag        "-vw"
#define useVertexWeight_flagL       "-useVertexWeight"


MSyntax hctCmdCgo::getCommandSyntax()
{
    MSyntax syntax;

    syntax.addFlag( maxDistance_flag, maxDistance_flagL, MSyntax::kDistance );
    syntax.addFlag( shrinking_flag, shrinking_flagL, MSyntax::kDouble );
    syntax.addFlag( multipass_flag, multipass_flagL, MSyntax::kBoolean );
    syntax.addFlag( targetNumVertices_flag, targetNumVertices_flagL, MSyntax::kLong );
    syntax.addFlag( targetRatio_flag, targetRatio_flagL, MSyntax::kDouble );
    syntax.addFlag( protMat_flag, protMat_flagL, MSyntax::kBoolean );
    syntax.addFlag( protGeo_flag, protGeo_flagL, MSyntax::kBoolean );
    syntax.addFlag( useVertexWeight_flag, useVertexWeight_flagL, MSyntax::kBoolean );
    syntax.setObjectType( MSyntax::kSelectionList );
    syntax.enableQuery( true );

    return syntax;
}

struct hctCmdCgoProgressUpdate : public hkgpCgo::IProgress
{
    hctCmdCgoProgressUpdate(int baseVertices) : m_valid(false), m_baseVertices(baseVertices)
    {
        m_currentProgress = -1;
        m_valid = MProgressWindow::reserve();
        m_canceled = false;
        if (m_valid)
        {
            MProgressWindow::startProgress();
            MProgressWindow::setProgressRange(0, 100);
            MProgressWindow::setInterruptable(true);
        }
    }

    ~hctCmdCgoProgressUpdate()
    {
        MProgressWindow::setProgressStatus(MString(""));
        MProgressWindow::endProgress();
    }

    bool step(const hkgpCgo::Config& config, hkReal error, int numVertices, int numTriangles)
    {
        if (m_valid)
        {
            const int   vOffset = (m_baseVertices - config.m_maxVertices);
            hkReal      eProgress = hkMath::clamp(config.m_maxDistance > 0.0f ? error / config.m_maxDistance : 0.0f, 0.0f, 1.0f);
            hkReal      vProgress = hkMath::clamp(vOffset > 0 ? 1.0f - (numVertices - config.m_maxVertices) / (hkReal)vOffset : 0.0f, 0.0f, 1.0f);
            int         progress = (int)(hkMath::max2(eProgress, vProgress) * 100.0f + 0.5f);
            if (m_currentProgress != progress)
            {
                if (m_currentProgress == -1)
                {
                    MProgressWindow::setTitle(MString("Collision Geometry Optimizer Utility progress"));
                    MProgressWindow::setProgressStatus(MString("Analyze"));
                }
                m_currentProgress = progress;
                MProgressWindow::setProgress(progress);
            }
            m_canceled |= MProgressWindow::isCancelled();
        }
        return !m_canceled;
    }

    int         m_currentProgress;
    int         m_baseVertices;
    bool        m_canceled;
    bool        m_valid;
};

MStatus hctCmdCgo::redoIt()
{
    return m_dagMod.doIt();
}

MStatus hctCmdCgo::undoIt()
{
    return m_dagMod.undoIt();
}

MStatus hctCmdCgo::doIt( const MArgList& args )
{
    // get command arguments
    MArgDatabase argData( syntax(), args );

    // defaults
    m_maxDistance       = MDistance( 1.0f, MDistance::uiUnit() );
    m_shrinking         = 0.0;
    m_multipass         = false;
    m_protMat           = false;
    m_protGeo           = false;
    m_useVertexWeight   = false;
    m_targetNumVertices = 0;
    m_targetRatio       = 0.0;

    if (argData.isFlagSet(maxDistance_flag))        argData.getFlagArgument( maxDistance_flag, 0, m_maxDistance );
    if (argData.isFlagSet(shrinking_flag))          argData.getFlagArgument( shrinking_flag, 0, m_shrinking );
    if (argData.isFlagSet(targetNumVertices_flag))  argData.getFlagArgument( targetNumVertices_flag, 0, m_targetNumVertices );
    if (argData.isFlagSet(targetRatio_flag))        argData.getFlagArgument( targetRatio_flag, 0, m_targetRatio );
    if (argData.isFlagSet(protGeo_flag))            argData.getFlagArgument( protGeo_flag, 0, m_protGeo );
//  if (argData.isFlagSet(protMat_flag))            argData.getFlagArgument( protMat_flag, 0, m_protMat );
//  if (argData.isFlagSet(useVertexWeight_flag))    argData.getFlagArgument( useVertexWeight_flag, 0, m_useVertexWeight );
    if (argData.isFlagSet(multipass_flag))          argData.getFlagArgument( multipass_flag, 0, m_multipass );

    if ((m_targetNumVertices == 0) && (m_targetRatio == 0.0))
    {
        MString errorText("CGO Input Error: Either -targetNumVertices or -targetRatio need to be set to a value > 0.");
        MGlobal::displayInfo(errorText);
        return MStatus::kFailure;
    }

    if ((m_targetNumVertices > 0) && (m_targetRatio > 0.0))
    {
        MString errorText("CGO Input Error: -targetNumVertices and -targetRatio cannot both be set to a value > 0.");
        MGlobal::displayInfo(errorText);
        return MStatus::kFailure;
    }

    if (m_targetRatio > 1.0)
    {
        MString errorText("CGO Input Error: -targetRatio needs to be in the range [0, 1].");
        MGlobal::displayInfo(errorText);
        return MStatus::kFailure;
    }

    // Start worker threads.
    hkDefaultTaskQueue taskQueue;
    hkCpuThreadPoolCinfo threadPoolCinfo;
    threadPoolCinfo.m_numThreads = hkHardwareInfo::getNumHardwareThreads() - 1;
    hkCpuThreadPool threadPool( threadPoolCinfo );
    threadPool.processTaskQueue( &taskQueue );

    // By design, a value of 0 will be interpreted as "any distance is allowed".
    // Any other value will need to be converted to [m] from Maya's UI unit system.
    hkReal maxDistance = m_maxDistance.value() <= 0.0f ? HK_REAL_MAX : (hkReal)m_maxDistance.asMeters();

    hkgpCgo::Config config;
    config.m_taskQueue                  = &taskQueue;
    config.m_maxDistance                = maxDistance;
    config.m_maxShrink                  = (hkReal)m_shrinking;
    config.m_maxVertices                = hkMath::max2(0, m_targetNumVertices);
    config.m_protectMaterialBoundaries  = m_protMat;
    config.m_protectNakedBoundaries     = m_protGeo;
    config.m_multiPass                  = m_multipass;
    config.m_semantic                   = m_useVertexWeight ? hkgpCgo::Config::VS_WEIGHT : hkgpCgo::Config::VS_NONE;

    // (Try to) Use the nodes as passed in through the command's argument list.
    MSelectionList commandList;
    argData.getObjects( commandList );

    // If no nodes were specified use the first object in the current active selection instead.
    if ( commandList.length() == 0 )
    {
        MGlobal::getActiveSelectionList( commandList );
    }

    unsigned int numSelected = commandList.length();
    for ( unsigned int n = 0; n < numSelected; n++ )
    {
        MDagPath selectionPath;
        commandList.getDagPath( n, selectionPath );

        hkGeometry geometry;
        MStatus sourceGeometryCreated = getGeometryFromDagPath( selectionPath, geometry );

        if (sourceGeometryCreated == MStatus::kSuccess)
        {
            const int numOriginalVertices  = geometry.m_vertices.getSize();
            const int numOriginalTriangles = geometry.m_triangles.getSize();

            if (m_targetRatio > 0.0)
            {
                config.m_maxVertices = int(numOriginalVertices * m_targetRatio);
            }

            hkStringBuf infos;

            infos.printf( "Optimize '%s' until the error is greater than %f or the number of vertices is less than %d", selectionPath.partialPathName().asChar(), m_maxDistance.value(), config.m_maxVertices );
            MString minfos(infos.cString());
            MGlobal::displayInfo(minfos);

            hctCmdCgoProgressUpdate progressUpdate( numOriginalVertices );

            hkgpCgo::optimize( config, geometry, &progressUpdate );

            createMesh( geometry, selectionPath );

            infos.printf( "Triangles %d => %d    Vertices %d => %d", numOriginalTriangles, geometry.m_triangles.getSize(), numOriginalVertices, geometry.m_vertices.getSize() );
            minfos = infos.cString();
            MGlobal::displayInfo(minfos);
        }
    }

    // Stop worker threads.
    taskQueue.close();
    threadPool.waitForCompletion();

    return MStatus::kSuccess;
}

MStatus hctCmdCgo::getGeometryFromDagPath( const MDagPath& dagPath, hkGeometry& geometry )
{
    MStatus status;
    MDagPath meshPath = dagPath;

    for (unsigned int i = 0; i < dagPath.childCount(); ++i)
    {
        MFn::Type type = dagPath.child(i).apiType();

        if (type == MFn::kMesh)
        {
            meshPath.push(dagPath.child(i));
            break;
        }
    }

    if (!meshPath.isValid() ||
        meshPath.apiType() != MFn::kMesh)
    {
        return MStatus::kFailure;
    }

    MFnMesh mesh;
    status = mesh.setObject( meshPath );
    if (status != MS::kSuccess)
    {
        status.perror( "Error creating MFnMesh" );
        return status;
    }

    // get the object space vertex positions
    MPointArray newVertexArray;
    status = mesh.getPoints(newVertexArray, MSpace::kObject);
    if (status != MS::kSuccess)
    {
        status.perror( "MFnMesh::getPoints" );
        return status;
    }

    // transform into world space
    hkArray<hkVector4>& geometryVertices = geometry.m_vertices;

    MDistance unitConverterX;
    MDistance unitConverterY;
    MDistance unitConverterZ;
    unitConverterX.setUnit(MDistance::internalUnit());
    unitConverterY.setUnit(MDistance::internalUnit());
    unitConverterZ.setUnit(MDistance::internalUnit());

    geometryVertices.setSize ( newVertexArray.length() );
    for (unsigned int vi = 0; vi < newVertexArray.length(); ++vi)
    {
        MPoint& p = newVertexArray[vi];
        unitConverterX.setValue(p.x);
        unitConverterY.setValue(p.y);
        unitConverterZ.setValue(p.z);
        hkReal x = (hkReal)unitConverterX.asMeters();
        hkReal y = (hkReal)unitConverterY.asMeters();
        hkReal z = (hkReal)unitConverterZ.asMeters();
        geometryVertices[vi] = hkVector4( x, y, z );
    }

    // get vertex indices for each triangle
    MIntArray triangleCounts;
    MIntArray triangleVertices;

    status = mesh.getTriangles( triangleCounts, triangleVertices );
    if (status != MS::kSuccess)
    {
        status.perror( "MFnMesh::getTriangles" );
        return status;
    }

    hkArray<hkGeometry::Triangle>& geometryTriangles = geometry.m_triangles;
    geometryTriangles.setSize(triangleVertices.length() / 3);

    int numTriangleVertices = triangleVertices.length();
    for (int ti = 0, tvi = 0; tvi < numTriangleVertices; ++ti)
    {
        geometryTriangles[ti].m_a = triangleVertices[tvi++];
        geometryTriangles[ti].m_b = triangleVertices[tvi++];
        geometryTriangles[ti].m_c = triangleVertices[tvi++];
        geometryTriangles[ti].m_material = 0;
    }

    return MStatus::kSuccess;
}

MStatus hctCmdCgo::createMesh( const hkGeometry& geom, MDagPath& sourceMeshDagPath )
{
    // Create Maya compatible structures from the input geometry.
    int numPoints = geom.m_vertices.getSize();
    MFloatPointArray points (numPoints);

    // We had CGO work in [m], so we need to convert the vertices back to Maya's internal unit.
    MDistance unitConverterX;
    MDistance unitConverterY;
    MDistance unitConverterZ;
    unitConverterX.setUnit(MDistance::kMeters);
    unitConverterY.setUnit(MDistance::kMeters);
    unitConverterZ.setUnit(MDistance::kMeters);

    for (int vi = 0; vi < numPoints; vi++)
    {
        unitConverterX.setValue(geom.m_vertices[vi](0));
        unitConverterY.setValue(geom.m_vertices[vi](1));
        unitConverterZ.setValue(geom.m_vertices[vi](2));

        MFloatPoint p;
        p.x = (float)unitConverterX.asUnits(MDistance::internalUnit());
        p.y = (float)unitConverterY.asUnits(MDistance::internalUnit());
        p.z = (float)unitConverterZ.asUnits(MDistance::internalUnit());

        points[vi] = p;
    }

    int numTriangles = geom.m_triangles.getSize();
    MIntArray polygonCounts( numTriangles, 3 );
    MIntArray polygonConnects( numTriangles*3 );

    for (int ti=0; ti<numTriangles; ti++)
    {
        polygonConnects[3*ti  ] = geom.m_triangles[ti].m_a;
        polygonConnects[3*ti+1] = geom.m_triangles[ti].m_b;
        polygonConnects[3*ti+2] = geom.m_triangles[ti].m_c;
    }

    MStatus status;

    // Create a temporary transform node.
    MObject meshTransform = m_dagMod.createNode( MString("transform"), MObject::kNullObj, &status );
    if ( status != MStatus::kSuccess)
    {
        return status;
    }

    // Create a new shape/mesh node from the input geometry.
    MFnMesh meshFn;
    MObject shapeNode = meshFn.create( numPoints, numTriangles, points, polygonCounts, polygonConnects, meshTransform, &status );
    if ( status != MStatus::kSuccess )
    {
        return status;
    }

    // Rename the new shape node and parent it to the source mesh's transform node.
    m_dagMod.renameNode( shapeNode, sourceMeshDagPath.partialPathName() + "_cgo" );
    m_dagMod.reparentNode( shapeNode, sourceMeshDagPath.transform() );
    m_dagMod.doIt();

    // The deletion of the (temporary) transform node needs to go into its own separate 'undo' step or otherwise
    // Maya will optimize the creation of the new mesh shape away again. :)
    m_dagMod.deleteNode( meshTransform );
    m_dagMod.doIt();

    // Assign the same shader groups to the newly created mesh that the original mesh has.
    // We go through MEL for this as this seems to be the easiest way.
    {
        MDagPath generatedMeshPath;
        meshFn.getPath(generatedMeshPath);

        MString command = \
            "string $shader[] = `listConnections -source true -type shadingEngine " + sourceMeshDagPath.fullPathName() + "`; " \
            "for ($i=0; $i<size($shader); $i++) " \
            "{ " \
            "\tsets -forceElement $shader[$i] " + generatedMeshPath.fullPathName() + "; " \
            "}";

        MGlobal::executeCommand(command);
    }

    return status;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
