// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <ContentTools/Common/SceneExport/Error/hctSceneExportError.h>
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
// Error handler
class hkMaxSceneError : public hctSceneExportError
{
public:
    hkMaxSceneError () {}

    /*virtual*/ int message(Message type, int id, const char* desc, const char* file, int line)
    {

        // Max doesn't have a reporting area - instead we use a message box, for major errors only
        if( type == MESSAGE_ERROR )
        {
            HCT_SCOPED_CONVERSIONS;

            MessageBox( GetCOREInterface()->GetMAXHWnd(), TO_WINDOWS(desc), TEXT("Havok Export Error"), MB_OK | MB_ICONWARNING );
        }

        /*int brk =*/ hctSceneExportError::message(type, id, desc, file, line);

        // never trigger breakpoint so the modeler doesn't crash
        return 0;
    }
};

class MaxProgressUpdater: public hctSdkUtils::ProgressUpdater
{
public:

    static DWORD WINAPI __dummyUpdateFn(LPVOID arg) { return 0; }

    MaxProgressUpdater(MCHAR* text)
    {
        m_currentObjectName = NULL;

        LPVOID arg=0;
        Interface* ip = GetCOREInterface();
        ip->ProgressStart(text, TRUE, __dummyUpdateFn, arg);
    }

    ~MaxProgressUpdater()
    {
        Interface* ip = GetCOREInterface();
        ip->ProgressEnd();
    }

    virtual void progression(float percentage);
    virtual void progression(float percentage, MCHAR* text);

    virtual bool didUserCancel ();

};

class hctMaxUtils
{
public:

    //
    // MESHES
    //

    // GEOSPHERE
    static bool createGeoSphereMesh (float radius, int segments, bool hideEdges, Mesh& meshOut);
    // DISC
    // If the disc is not of full (two pi range) and "closeOpenDisc" is true, extra faces will be added for the "slice"
    // SegmentsPerRadian specifies the number of vertices used for each circular base.
    static bool createDiscMesh (const Point3& verticalAxis, const Point3& zeroAxis, float radius, float height, float limitMin, float limitMax, float segPerRadian, bool closeOpenDisc, Mesh& meshOut);
    // CONE
    static bool createConeMesh (const Point3& axis, float angle, float height,  bool closeCone, bool hideEdges, Mesh& meshOut);

    //
    // SHAPES
    //

    // CIRCLE (Bezier) :
    static bool createCircleBezier (float radius, int planeAxis, BezierShape& bezierOut);

    // CIRCLE (PolyShape) - Calls the above
    static bool createCircleShape (float radius, int planeAxis, int steps, PolyShape& polyShapeOut);

    // SPHERE (Bezier)
    static bool createSphereBezier (float radius, BezierShape& bezierOut);

    // CIRCLE (PolyShape) - Calls the above
    static bool createSphereShape (float radius, int steps, PolyShape& polyShapeOut);


    //
    // DISPLAY STUFF
    //

    // Using the render() method from the maxsdk causes problems when using the software renderer.
    // This implementation uses individual GraphicsWindow::Segment() calls instead.
    static void renderPolyShape (const PolyShape& polyShape, GraphicsWindow* gw);

    //
    // MAX STUFF
    //

    // Finds the first node referencing this object (modifier)
    static INode* findNodeRef(const ReferenceTarget *rt);
    static INodeTab findNodeRefs(const ReferenceTarget *rt);

    // Displays a dashed line
    static void drawDashedLine (GraphicsWindow* gw, const Point3& start, const Point3& end, float sizeOn=0.05f, float sizeOff=0.05f);

    // As above, but gradually changing color
    static void drawColoredDashedLine (GraphicsWindow* gw,
        const Point3& start, const Point3& end,
        const Point3& startColor, const Point3& endColor,
        float sizeOn=0.05f, float sizeOff=0.05f);

    // Presents a "select by name" dialog and allows a single selection of a node.
    // Returns NULL if no selection was done. "Title" specifies the title in the dialog.
    // (It uses the Interace::DoHitByNameDialog() method with a custom callback)
    static INode* selectOneNodeUsingDialog (class Interface* ip, const char* title);

    static hkxMaterial::TextureType maxSlotToTextureType(int slot);
    static void getSubTexMaps( Texmap* texmap, hkArray< Texmap* >& subMaps );

    // For a given material this function fills the supplied array with the indices of all used map channels.
    static void findUsedChannels(Mtl* mat, hkArray<int>& usedChannels);

    // If the material is a Viewport/Render Shell material, get the viewport material. Otherwise return the same material
    static Mtl* hctMaxUtils::getViewportMaterial (Mtl* material);

    // Returns the number of sub-materials of the given multiMaterial
    static int getNumSubMaterials(Mtl* multiMaterial);

    // Returns the sub-materials of the given multiMaterial
    static void findSubMaterials(Mtl* multiMaterial, int lookingFor, hkArray<Mtl*>& mats, hkArray<int>& blends);

    // Retrieves the full name of the given sub-material
    static Mtl* findSubMaterialName(Mtl* multiMaterial, int lookingFor, hkStringPtr& nameOut);

    //
    // MATH STUFF
    //

    // Returns a rotation that would convert the axis "from" into "to"
    static Matrix3 shortestRotationBetweenAxis (const Point3& from, const Point3& to);

    // Returns a unit vector perpendicular to the given vector
    static Point3 calculatePerpendicularUnitVector (const Point3& vectorIn);

    //
    // PHYSICS STUFF
    //
    // Returns the twist between the two spaces in a ragdoll constraint. Replicates the behaviour at run-time.
    static float getRagdollTwist (Matrix3& parentSpaceWorld, Matrix3& childSpaceWorld);

    // Garbage collection safe way to call a maxscript command (and get the result back etc).
    static BOOL evaluateMAXScript(Interface *ip, const char *command);



    // Converts a 3ds max Matrix3 into an hkMatrix4
    inline static void convertToMatrix4( Matrix3& in, hkMatrix4& out );

    inline static void convertToMatrix3( hkMatrix4& in, Matrix3& out );

    // Returns the given point3 as an hkVector4
    inline static hkVector4 asHkVector4 (const Point3& point3);

    // Converts the given horizontal FOV to a vertical FOV using 3ds max's current rendering aspect ratio
    inline static float convertHorizontalToVerticalFOV (const float hFov);

    static bool HK_CALL getFilterManagerPath( hkStringOld& filterManagerPath );

  static bool HK_CALL isBone(INode* n);

  static void HK_CALL getControllerKeyTimes(INode* pNode, hkArray<int>& keyList, int startTime, int endTime, bool sort = true);
  static void HK_CALL getControllerKeyTimes(Control* pC,  hkArray<int>& keyList, int startTime, int endTime, bool sort = true);
  static void HK_CALL getAnimatableKeyTimes(INode* pNode, hkArray<int>& keyList, int startFrame, int endFrame, bool sort = true);

  static inline int HK_CALL getFramesFromTicks(int ticks)
  {
    int tpF = GetTicksPerFrame();
    int frames = ticks / tpF;
    return frames;
  }

  static inline int HK_CALL getTicksFromFrame(int frame)
  {
    int tpF = GetTicksPerFrame();
    int ticks = frame * tpF;
    return ticks;
  }


private:

    static void setupCircleSpline (float radius, int planeAxis, Spline3D& splineOut);
};





//
// UNDO UTILITY DEFINES (EXP-37)
//

#define BEGIN_UNDOABLE_ACTION   \
    bool remember_to_accept = false; \
    const bool isHoldingAlready = theHold.Holding() != FALSE; \
    const bool isUndoingOrRedoing = theHold.RestoreOrRedoing() != FALSE; \
    if (!isHoldingAlready && (!isUndoingOrRedoing) ) \
{ \
    theHold.Begin(); \
    remember_to_accept = true; \
}

#define END_UNDOABLE_ACTION(undo_string) \
    if (remember_to_accept) \
{\
    theHold.Accept(undo_string);\
}

// EXP-37 : Max suspends undo when setting an Inode*, we need to temporarily
// resume it using this macros
#define BEGIN_RESUME_UNDO   \
    const bool wasUndoSuspended = (theHold.IsSuspended() != FALSE); \
    const bool isUndoingOrRedoing = theHold.RestoreOrRedoing() != FALSE; \
    bool rememberToSuspend = false; \
    if (wasUndoSuspended && (!isUndoingOrRedoing))   \
{ \
    theHold.Resume(); \
    rememberToSuspend = true;\
}

#define END_RESUME_UNDO \
    if (rememberToSuspend) \
{ \
    theHold.Suspend(); \
}

#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.inl>

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
