// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

void MaxProgressUpdater::progression (float percentage)
{
    Interface* ip = GetCOREInterface();
    ip->ProgressUpdate( static_cast<int>(percentage) );
}

void MaxProgressUpdater::progression (float percentage, MCHAR* text)
{
    Interface* ip = GetCOREInterface();
    ip->ProgressUpdate( static_cast<int>(percentage), FALSE, text );
}

bool MaxProgressUpdater::didUserCancel()
{
    Interface* ip = GetCOREInterface();
    if ( ip->GetCancel() )
    {
        return true;
    }

    return false;
}

// This registry key contains the path to the filter manager.
// The most recently used filter set is also stored from here.
#ifdef HK_PLATFORM_WIN64
    static const char* _HAVOK_FILTERS_REG_KEY = "Software\\Havok\\hkFilters_x64";
#else
    static const char* _HAVOK_FILTERS_REG_KEY = "Software\\Havok\\hkFilters";
#endif

bool HK_CALL hctMaxUtils::getFilterManagerPath( hkStringOld& filterManagerPath )
{
    HCT_SCOPED_CONVERSIONS;

    Interface* ip = GetCOREInterface();

#if defined (APP_MAX_SYS_ROOT_DIR) // max9
    hkStringOld maxRootDir = FROM_MAX( ip->GetDir(APP_MAX_SYS_ROOT_DIR) );
#else
    hkStringOld maxRootDir = FROM_MAX( ip->GetDir(APP_MAXROOT_DIR) );
#endif

    hkBool filterManagerFound = false;

    // Always check env var before reg keys so that scripts can change the path easily for a local Process
    {
        filterManagerPath = ::getenv( ENVVAR_FILTER_ROOT );
        const hkStringOld filterManagerDll = filterManagerPath + "\\hctFilterManager.dll";
        hkIfstream dllFile( filterManagerDll.cString() );
        filterManagerFound = dllFile.isOk();
    }

    if (!filterManagerFound)
    {
        HKEY animReg;
        DWORD dispos;

        RegCreateKeyExA( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

        BYTE filtersPath[1024];
        DWORD filtersPathSize = 1024;

        // Check the MaxPath value.
        if( RegQueryValueExA( animReg, "MaxPath", NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
        {
            filterManagerPath = (const char *)(filtersPath);

            const hkStringOld filterManagerDll = filterManagerPath + "\\hctFilterManager.dll";
            hkIfstream dllFile( filterManagerDll.cString() );
            filterManagerFound = dllFile.isOk();
        }

        if( !filterManagerFound )
        {
            // filtersPathSize will be reduced to 0 if the MaxPath registry entry is empty. Reset filtersPathSize before reading the FilterPath value.
            filtersPathSize = 1024;

            // If MaxPath isn't set, check the default FilterPath value.
            if( RegQueryValueExA( animReg, "FilterPath", NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
            {
                filterManagerPath = (const char*)(filtersPath);

                const hkStringOld filterManagerDll = filterManagerPath + "\\hctFilterManager.dll";
                hkIfstream dllFile( filterManagerDll.cString() );
                filterManagerFound = dllFile.isOk();
            }
            RegCloseKey(animReg);
        }
    }

    return ( filterManagerPath.getLength() > 0 ) && filterManagerFound;
}

// Garbage collection safe way to call a maxscript command (and get the result back etc).
BOOL hctMaxUtils::evaluateMAXScript(Interface *ip, const char *command)
{
    HCT_SCOPED_CONVERSIONS;

    BOOL result=TRUE;
    init_thread_locals();
    four_typed_value_locals(Parser* parser,Value* code,Value* result,StringStream* source);

    MSTR commandStr = TO_MAX(command);

    vl.parser = new Parser (thread_local(current_stdout));
    vl.source = new StringStream(commandStr);
    vl.source->log_to(thread_local(current_stdout));

    {
#if MAX_VERSION_MAJOR < 19
        save_current_frames();
#else
        ScopedSaveCurrentFrames frameSave;
#endif
        try
        {
            ip->RedrawViews(ip->GetTime(), REDRAW_BEGIN);

            vl.source->flush_whitespace();

            vl.code = vl.parser->compile_all(vl.source);
            vl.result = vl.code->eval();
            vl.result->print();

            vl.source->flush_whitespace();
            vl.source->close();

            ip->RedrawViews(ip->GetTime(), REDRAW_END);
        }
        catch (...)
        {
#if MAX_VERSION_MAJOR < 19
            restore_current_frames();
#endif
            result = FALSE;
            vl.source->close();
        }
#if MAX_VERSION_MAJOR < 19
        pop_value_locals();
#endif
    }

    return result;
}


/*static*/ bool hctMaxUtils::createGeoSphereMesh (float radius , int segments, bool hideEdges, Mesh& meshOut)
{
#define GEOSPHERE_CLASS_ID Class_ID(0, 32670)

    SimpleObject2* geoSphereObj = (SimpleObject2*) GetCOREInterface()->CreateInstance(GEOMOBJECT_CLASS_ID, GEOSPHERE_CLASS_ID);

    if (!geoSphereObj) return false;

    // block IDs
    enum { geo_creation_type, geo_type_in, geo_params, };
    // geo_creation_type param IDs
    enum { geo_create_meth, };
    // geo_type_in param IDs
    enum { geo_ti_pos, geo_ti_radius, };
    // geo_param param IDs
    enum { geo_radius, geo_segs, geo_type, geo_hemi, geo_smooth, geo_basepivot, geo_mapping, };

    IParamBlock2* pblock2 = geoSphereObj->GetParamBlockByID(geo_params);

    pblock2->SetValue(geo_radius, 0, radius);
    pblock2->SetValue(geo_segs, 0, segments);
    pblock2->SetValue(geo_type, 0, 2); // icosahedric
    pblock2->SetValue(geo_hemi, 0, FALSE);
    pblock2->SetValue(geo_smooth, 0, FALSE); // WARNING! HCL-220 : do not change this to true or display will not work with Software or OpenGL as display drivers
    pblock2->SetValue(geo_basepivot, 0, FALSE);
    pblock2->SetValue(geo_mapping, 0, TRUE);

    geoSphereObj->BuildMesh(0);

    meshOut = geoSphereObj->mesh;

    geoSphereObj->DeleteThis();

    if (hideEdges)
    {
        for (int i=0; i<meshOut.numFaces; i++)
        {
            meshOut.faces[i].setEdgeVisFlags(0,0,0);
            meshOut.faces[i].setMatID(0);
        }
    }

    return true;

}


/*bool hctMaxUtils::createSphereMesh (float radius, int segments, bool hideEdges, Mesh& meshOut)
{
// Use one of the manipulator helper methods
{
Mesh* sphereMesh = MakeSphere(Point3(0,0,0), radius, segments);
meshOut = *sphereMesh;
sphereMesh->DeleteThis();
}

if (hideEdges)
{
for (int i=0; i<meshOut.numFaces; i++)
{
meshOut.faces[i].setEdgeVisFlags(0,0,0);
meshOut.faces[i].setMatID(0);
}
}

return true;
}
*/

/*static*/ bool hctMaxUtils::createDiscMesh (const Point3& verticalAxis, const Point3& zeroAxis, float radius, float height, float limitMin, float limitMax, float segmentsPerRadian, bool closeOpenDisc, Mesh& meshOut)
{
    // NOTE : We could use max cylinder's object as we do for geosphere

    if (limitMin>limitMax)
    {
        limitMin = limitMax;
    }

    float totalAngle = limitMax - limitMin;
    const bool fullCircle = (totalAngle >= TWOPI);
    const bool closedDisc = fullCircle || (!closeOpenDisc);
    if (fullCircle)
    {
        totalAngle = TWOPI;
    }

    const int numSegments = max((int) ( (totalAngle) * segmentsPerRadian ), 1); // (one each 0.1 rad)

    const float segmentSize = totalAngle / (float) numSegments;

    const float halfHeight = height * 0.5f;

    // Vertices
    {
        // 2 for origin, two for each segment limit (segments+1)
        // For closed discs we could get rid of two of them but this way it's easier
        const int numVerts = 2+(2*(numSegments+1));
        meshOut.setNumVerts(numVerts);

        // First two vertices : center, top and down;
        meshOut.setVert(0, 0.0f, 0.0f, halfHeight);
        meshOut.setVert(1, 0.0f, 0.0f, -halfHeight);

        for (int seg=0; seg<=numSegments; seg++)
        {
            const float angle = limitMin + seg * segmentSize;
            const float x = radius * cosf(angle);
            const float y = radius * sinf(angle);
            meshOut.setVert(2+ seg*2,   x,y, halfHeight);
            meshOut.setVert(2+ seg*2+1, x,y, -halfHeight);
        }
    }

    // Faces
    {

        // For each segment, one triangle at top, one at bottom and two at the side
        // Unless it's full circle, we also need four more triangles to close the sides of the pie
        const int numFaces = closedDisc ? (numSegments*4) : (numSegments*4 + 4);
        meshOut.setNumFaces(numFaces);

        for (int seg=0; seg<numSegments; seg++)
        {
            // Indices of vertices
            // top0,bot0 are in the center (top and bottom)
            // top1,2 and bot1,2 are this segments start and p_end, counterclockwise looking from the top
            int top0 = 0;
            int top1 = 2 + seg*2;
            int top2 = 2 + (seg+1)*2;
            int bot0 = 1;
            int bot1 = 2 + seg*2 + 1;
            int bot2 = 2 + (seg+1)*2 +1;

            // top face
            meshOut.faces[seg*4].setVerts(top0,top1,top2);
            meshOut.faces[seg*4].setEdgeVisFlags(0,1,0);
            meshOut.faces[seg*4].setSmGroup(1);
            // bottom face
            meshOut.faces[seg*4+1].setVerts(bot0,bot2,bot1);
            meshOut.faces[seg*4+1].setEdgeVisFlags(0,1,0);
            meshOut.faces[seg*4].setSmGroup(2);
            // side
            meshOut.faces[seg*4+2].setVerts(bot1,top2,top1);
            meshOut.faces[seg*4+2].setEdgeVisFlags(0,0,0);
            meshOut.faces[seg*4+2].setSmGroup(4);
            meshOut.faces[seg*4+3].setVerts(top2,bot1,bot2);
            meshOut.faces[seg*4+3].setEdgeVisFlags(0,0,0);
            meshOut.faces[seg*4+3].setSmGroup(4);
        }

        // finally, if the disc is not closed, two extra squares (four extra triangles)
        if (!closedDisc)
        {
            int a,b,c,d;

            const int beginFace = numSegments*4;
            a = 0; // center top
            b = 1; // center bottom
            c = 3; // begin bottom
            d = 2; // begin top
            meshOut.faces[beginFace].setVerts(a,b,c);
            meshOut.faces[beginFace].setEdgeVisFlags(1,1,0);
            meshOut.faces[beginFace].setSmGroup(8);
            meshOut.faces[beginFace+1].setVerts(c,d,a);
            meshOut.faces[beginFace+1].setEdgeVisFlags(1,1,0);
            meshOut.faces[beginFace+1].setSmGroup(8);

            const int endFace = beginFace+2;
            a = 1; // center bottom
            b = 0; // center top
            c = 2 + numSegments*2; // p_end top
            d = 2 + numSegments*2 +1; //p_end bottom
            meshOut.faces[endFace].setVerts(a,b,c);
            meshOut.faces[endFace].setEdgeVisFlags(1,1,0);
            meshOut.faces[endFace].setSmGroup(16);
            meshOut.faces[endFace+1].setVerts(c,d,a);
            meshOut.faces[endFace+1].setEdgeVisFlags(1,1,0);
            meshOut.faces[endFace+1].setSmGroup(16);
        }
    }

    /*
    ** Transform the vertices
    */
    {
        // First step : construct a matrix that will convert the points to the desired configuration
        const Point3 newXAxis = zeroAxis;
        const Point3 newZAxis = verticalAxis;
        const Point3 newYAxis = newZAxis ^ newXAxis;
        const Matrix3 oldToNew(newXAxis, newYAxis, newZAxis,Point3(0,0,0));

        // Transform all the vertices
        for (int v=0; v<meshOut.numVerts; v++)
        {
            meshOut.verts[v] = meshOut.verts[v] * oldToNew;
        }
    }

    meshOut.buildNormals();
    meshOut.BuildStripsAndEdges();

    return true;
}

// Builds a cone mesh around the given axis
/*static*/ bool hctMaxUtils::createConeMesh (const Point3& axis, float angle, float height, bool closeCone, bool hideEdges, Mesh& meshOut)
{
    const int numSegments = 16;

    if (angle+0.01 >=HALFPI) angle = HALFPI-0.01f;

    Point3 new_point(height,0.0f,0.0f);
    float radius = height * tanf(angle);
    float currentAngle = 0.0f;

    const int extraVerts = closeCone ? 2 : 1;

    meshOut.setNumVerts(extraVerts+numSegments);
    meshOut.setNumFaces(closeCone ? numSegments*2 : numSegments);

    meshOut.setVert(0,Point3(height*0.01f,0.0f,0.0f)); // cheat, we avoid singularity a the origin for booleans

    if (closeCone)
    {
        meshOut.setVert(1,Point3(height,0.0f,0.0f));
    }

    // Vertices
    {
        for (int i= 0; i<numSegments; i++)
        {
            currentAngle = 2*PI*i/numSegments;
            new_point.y = cosf(currentAngle) * radius;
            new_point.z = sinf(currentAngle) * radius;
            meshOut.setVert(i+extraVerts, new_point);
        }
    }

    // Faces
    {
        for (int i = 0; i < numSegments; i++)
        {
            const int face_num = closeCone ? i*2 : i;

            const int vert2 = ((i-1+numSegments)%numSegments) + extraVerts;
            const int vert1 = (i%numSegments) + extraVerts;

            meshOut.faces[face_num].setVerts(0, vert1, vert2);
            meshOut.faces[face_num].setSmGroup(1);
            meshOut.faces[face_num].setMatID(0);
            // Cone Side
            if (hideEdges)
            {
                meshOut.faces[face_num].setEdgeVisFlags(0,1,0);
            }
            else
            {
                meshOut.faces[face_num].setEdgeVisFlags(1,1,1);
            }

            // Cone base
            if (closeCone)
            {
                meshOut.faces[face_num+1].setVerts(1, vert2, vert1);
                meshOut.faces[face_num+1].setSmGroup(2);
                meshOut.faces[face_num+1].setMatID(0);

                if (hideEdges)
                {
                    meshOut.faces[face_num+1].setEdgeVisFlags(0,1,0);
                }
                else
                {
                    meshOut.faces[face_num+1].setEdgeVisFlags(1,1,1);
                }
            }
        }
    }

    // Now, transform the points to reach desired orientation of axis (originally X)
    {
        Matrix3 transform = hctMaxUtils::shortestRotationBetweenAxis(Point3(1,0,0), axis);
        for (int v=0; v<meshOut.numVerts; v++)
        {
            meshOut.verts[v] = meshOut.verts[v] * transform;
        }
    }

    meshOut.buildNormals();
    meshOut.BuildStripsAndEdges();

    return true;
}


//////////////////////////////////////////////////////////////////////////
// Circles and wire spheres
//////////////////////////////////////////////////////////////////////////

/*static*/ void hctMaxUtils::setupCircleSpline (float radius, int planeAxis, Spline3D& splineOut)
{
    const float CIRCLE_VECTOR_LENGTH = 0.5517861843f;
    float vector = CIRCLE_VECTOR_LENGTH * radius;

    int x = (planeAxis+1) % 3;
    int y = (planeAxis+2) % 3;
    int z = planeAxis;

    for(int ix=0; ix<4; ++ix)
    {
        float angle = 6.2831853f * (float)ix / 4.0f;
        float sinfac = (float)sin(angle), cosfac = (float)cos(angle);
        Point3 p;
        p[x] = cosfac * radius;
        p[y] = sinfac * radius;
        p[z] = 0.0f;

        Point3 rotvec;
        rotvec[x] = sinfac * vector;
        rotvec[y] = -cosfac * vector;
        rotvec[z] = 0.0f;

        splineOut.AddKnot( SplineKnot( KTYPE_BEZIER, LTYPE_CURVE, p, p + rotvec, p - rotvec ) );
    }

    splineOut.SetClosed();
    splineOut.ComputeBezPoints();

}

// CIRCLE (Bezier)
/*static*/ bool hctMaxUtils::createCircleBezier (float radius, int planeAxis, BezierShape& bezierOut)
{
    bezierOut.steps = -1;
    bezierOut.optimize = true;

    Spline3D *spline = bezierOut.NewSpline();

    setupCircleSpline(radius, planeAxis, *spline);

    bezierOut.UpdateSels();
    bezierOut.InvalidateGeomCache();

    return true;
}

// CIRCLE (PolyShape) - Calls the above
/*static*/  bool hctMaxUtils::createCircleShape (float radius,int planeAxis, int steps, PolyShape& polyShapeOut)
{
    BezierShape bezierCircle;
    bool res = createCircleBezier(radius, planeAxis, bezierCircle);

    if (!res) return false;

    bezierCircle.MakePolyShape(polyShapeOut, steps);

    return true;
}

// SPHERE (Bezier)
/*static*/ bool hctMaxUtils::createSphereBezier (float radius, BezierShape& bezierOut)
{
    bezierOut.steps = -1;
    bezierOut.optimize = true;

    for (int axis=0; axis<3; axis++)
    {
        Spline3D *spline = bezierOut.NewSpline();

        setupCircleSpline(radius, axis, *spline);
    }

    bezierOut.UpdateSels();
    bezierOut.InvalidateGeomCache();

    return true;

}

// CIRCLE (PolyShape) - Calls the above
/*static*/   bool hctMaxUtils::createSphereShape (float radius, int steps, PolyShape& polyShapeOut)
{
    BezierShape bezierSphere;
    bool res = createSphereBezier(radius, bezierSphere);

    if (!res) return false;

    bezierSphere.MakePolyShape(polyShapeOut, steps);

    return true;
}

/*static*/ void hctMaxUtils::renderPolyShape (const PolyShape& polyShape, GraphicsWindow* gw)
{
    for (int l=0;l<polyShape.numLines; l++)
    {
        PolyLine& pl = polyShape.lines[l];

        gw->startSegments();

        const int numSegments = pl.IsClosed() ? pl.numPts : pl.numPts-1;
        for (int p=0; p<numSegments; p++)
        {
            Point3 points[2] = {pl.pts[p].p, pl.pts[(p+1) % pl.numPts].p};
            gw->segment(points,1);
        }
        gw->endSegments();
    }
}


static INode* GetNodeRef(ReferenceMaker *rm)
{

    if (rm->SuperClassID()==BASENODE_CLASS_ID)
    {
        return (INode *)rm;
    }
    else
    {
        return rm->IsRefTarget()? hctMaxUtils::findNodeRef((ReferenceTarget *)rm):NULL ;
    }

}

// Finds the first node referencing this object (modifier)
INode* hctMaxUtils::findNodeRef(const ReferenceTarget *rt)
{
    DependentIterator di(const_cast<ReferenceTarget*> (rt));

    ReferenceMaker *rm;

    INode *nd = NULL;

    while ((rm=di.Next())!=NULL)
    {

        nd = GetNodeRef(rm);

        if (nd) return nd;

    }

    return NULL;

}

INodeTab hctMaxUtils::findNodeRefs(const ReferenceTarget *rt)
{
    INodeTab result;

    DependentIterator di(const_cast<ReferenceTarget*> (rt));

    ReferenceMaker *rm;

    while ((rm=di.Next())!=NULL)
    {

        INode* nd = GetNodeRef(rm);

        if (nd)
        {
            result.Append(1,&nd);
        }

    }

    return result;
}

// Displays a dashed line
/*static*/ void hctMaxUtils::drawDashedLine (GraphicsWindow* gw, const Point3& start, const Point3& p_end, float sizeOn, float sizeOff)
{
    const Point3 distance = p_end - start;
    float progress = 0.0f;
    gw->startSegments();
    while (progress < 1.0f)
    {
        Point3 points[2];
        points[0] = start + distance * progress;
        points[1] = start + distance * min(1.0f, progress+sizeOn);

        gw->segment(points,TRUE);

        progress += (sizeOn+sizeOff);
    }
    gw->endSegments();

}

/*static*/ void hctMaxUtils::drawColoredDashedLine (GraphicsWindow* gw,
                                                    const Point3& start, const Point3& p_end,
                                                    const Point3& startColor, const Point3& endColor,
                                                    float sizeOn, float sizeOff)
{
    const Point3 distance = p_end - start;
    const Point3 colorDistance = endColor - startColor;
    float progress = 0.0f;
    gw->startSegments();
    while (progress < 1.0f)
    {
        Point3 points[2];
        points[0] = start + distance * progress;
        points[1] = start + distance * min(1.0f, progress+sizeOn);

        Point3 color = startColor + colorDistance * (progress + sizeOn*0.5f);
        gw->setColor(LINE_COLOR, color);
        gw->segment(points,TRUE);

        progress += (sizeOn+sizeOff);
    }
    gw->endSegments();

}

// Returns the twist between the two spaces in a ragdoll constraint. Replicates the behaviour at run-time.
float hctMaxUtils::getRagdollTwist (Matrix3& parentSpaceWorld, Matrix3& childSpaceWorld)
{
    Point3 twistParent = parentSpaceWorld.VectorTransform(Point3(1,0,0));
    Point3 twistChild =  childSpaceWorld.VectorTransform(Point3(1,0,0));

    Point3 planeParent = parentSpaceWorld.VectorTransform(Point3(0,1,0));
    Point3 planeChild = childSpaceWorld.VectorTransform(Point3(0,1,0));

    Point3 twistAxisWS;
    {
        twistAxisWS = twistParent + twistChild;
        const float twistLength = twistAxisWS.Length();
        if (twistLength>1e-16f)
        {
            twistAxisWS*=(1.0f/twistLength);
        }
        else
        {
            twistAxisWS = twistParent;
        }
    }

    const Point3 perp1 = twistAxisWS ^ planeParent; // crossprod
    const Point3 perp2 = perp1 ^ twistAxisWS; // crossprod

    const float d1 = perp1 % planeChild; // dotprod
    const float d2 = perp2 % planeChild; // dotprod
    const float currentTwistAngle  = atan2f(d1, d2);

    return currentTwistAngle;
}

//
// MATH STUFF
//

Matrix3 hctMaxUtils::shortestRotationBetweenAxis (const Point3& from, const Point3& to)
{
    const float dotProd = from %  to;   // dot product = cos(theta)

    // If the axis are pointing in opposite directions, there are infinite solutions
    if (dotProd==-1.0f)
    {
        // Choose an arbitrary perpendicular axis
        int minCompoment = from.MinComponent();
        Point3 nonParalelAxis(0,0,0);
        nonParalelAxis[minCompoment] = 1.0f;

        const Point3 perpAxis = from ^ nonParalelAxis;
        Matrix3 m;
        m.SetRotate(AngAxis(perpAxis, PI));

        return m;
    }

    // Using cos(theta) = 2*cos^2(theta/2) - 1
    const float c = (dotProd + 1.0f) * 0.5f;
    const float cosT2 = sqrtf( c );

    Point3 vec = from ^ to; // cross product

    // Using sin(theta) = 2*cos(theta/2)*sin(theta/2)
    const float rescaleSin  = 0.5f / cosT2;
    vec *= rescaleSin;

    Quat rotQ (vec.x, vec.y, vec.z, cosT2);

    Matrix3 m;
    rotQ.MakeMatrix(m, true);

    return m;
}


/*static*/ Point3 hctMaxUtils::calculatePerpendicularUnitVector (const Point3& vectorIn)
{
    const int minComponent = MinComponent(vectorIn);
    const int other1 = (minComponent+1) % 3;
    const int other2 = (minComponent+2) % 3;

    Point3 result;
    result[minComponent] = 0.0f;
    result[other1] = vectorIn[other2];
    result[other2] = -vectorIn[other1];

    result.Normalize();

    return result;
}

class _OurHitByNameCallback : public HitByNameDlgCallback
{
    public:

        _OurHitByNameCallback (const char* title) : m_result (NULL)
        {
            HCT_SCOPED_CONVERSIONS;

            m_title = TO_MAX(title);
        }

        INode* getResult () const { return m_result; }

        /*virtual*/ CONST15 MCHAR* dialogTitle () { return m_title; }
        /*virtual*/ CONST15 MCHAR* buttonText () {return TEXT("Pick");}
        /*virtual*/ BOOL useFilter() {return FALSE;}
        /*virtual*/ BOOL singleSelect() {return TRUE;}
        /*virtual*/ BOOL useProc() {return TRUE;}
        /*virtual*/ void proc (INodeTab& nodeTab) { if (nodeTab.Count()==1) m_result = nodeTab[0]; }

    private:

        MSTR m_title;
        INode* m_result;

};

/*static*/ INode* hctMaxUtils::selectOneNodeUsingDialog (Interface* ip, const char* title)
{
    _OurHitByNameCallback callback(title);

    BOOL okVal = ip->DoHitByNameDialog(&callback);

    if (okVal)
    {
        return callback.getResult();
    }
    else
    {
        return NULL;
    }
}

/*static*/ hkxMaterial::TextureType hctMaxUtils::maxSlotToTextureType(int slot)
{
    switch (slot)
    {
    case ID_AM: /*Ambient (value 0)*/
    case ID_DI: /*Diffuse (value 1)*/
        return hkxMaterial::TEX_DIFFUSE;

    case ID_SP: /*Specular Color (value 2)*/
        return hkxMaterial::TEX_SPECULAR;

    case ID_SH: /*Shininess (specular level)(value 3). */
        return hkxMaterial::TEX_SPECULAR;

    case ID_SS: /*Shininess strength /Specular Power (gloss) (value 4). */
        return hkxMaterial::TEX_GLOSS;

    case ID_SI: /*Self-illumination (value 5)*/
        return hkxMaterial::TEX_EMISSIVE;

    case ID_OP: /*Opacity (value 6)*/
        return hkxMaterial::TEX_OPACITY;

    case ID_FI: /*Filter color (value 7)*/
        return hkxMaterial::TEX_UNKNOWN;; // not handled really.. might as well export it anyway and just modulate it in

    case ID_RR: /*Refraction (value 10)*/
        return hkxMaterial::TEX_REFRACTION;

    case ID_RL: /*Reflection (value 9)*/
        return hkxMaterial::TEX_REFLECTION;

    case ID_BU: /*Bump (value 8)*/
        return hkxMaterial::TEX_NORMAL; // assume a normal map, don't really use just bump anymore

    case ID_DP: /*Displacement (value 11)*/
        return hkxMaterial::TEX_DISPLACEMENT;
    }

    return hkxMaterial::TEX_UNKNOWN; // user type.. might as well export it anyway
    //return hkxMaterial::TEX_NOTEXPORTED;
}


/*static*/ void hctMaxUtils::getSubTexMaps( Texmap* texmap, hkArray< Texmap* >& subMaps )
{
    int numSubTexs = texmap->NumSubTexmaps();
    bool hadChildren = false;
    if (numSubTexs > 0)
    {
        for (int it=0; it < numSubTexs; ++it)
        {
            Texmap* subTex = texmap->GetSubTexmap(it);
            if (subTex)
            {
                hadChildren = true;
                getSubTexMaps( subTex, subMaps); // recurse..
            }
        }
    }

    // if no sub tex maps, or sub tex map slots do not have textures
    // then we just use the parent tex map (eg Checker can have empty child slots)
    if (!hadChildren)
    {
        subMaps.pushBack( texmap );
    }
}

/*static*/ void hctMaxUtils::findUsedChannels(Mtl* mat, hkArray<int>& usedChannels )
{
    //Sub materials handled further up


    // don't clear the used channel array as we may be recursively called
    int numSlots = mat->NumSubTexmaps();
    for (int slot=0; slot<numSlots; slot++)
    {
        const int hkslot = hctMaxUtils::maxSlotToTextureType(slot);
        if (hkslot < 0) // not exported by HCT
            continue;

        Texmap* tmap = mat->GetSubTexmap(slot);
        const int state = mat->SubTexmapOn(slot); // 0 - off, > 0 - on

        if (state && tmap) // on and not null
        {
            hkArray< Texmap* > sourceTexMaps;
            hctMaxUtils::getSubTexMaps( tmap, sourceTexMaps);

            for (int sti=0; sti < sourceTexMaps.getSize(); ++sti)
            {
                int channel = sourceTexMaps[sti]->GetMapChannel();
                if ((channel>=0) && usedChannels.indexOf(channel) < 0)
                {
                    usedChannels.pushBack(channel);
                }
            }
        }
    }

#ifdef MAX6_OR_HIGHER

    IDxMaterial2* dx2Mat = (IDxMaterial2*) mat->GetInterface( IDXMATERIAL2_INTERFACE );
    if (dx2Mat)
    {
        int numBitmaps = dx2Mat->GetNumberOfEffectBitmaps();
        for (int bi=0; bi < numBitmaps; ++bi)
        {
            PBBitmap* pbm = dx2Mat->GetEffectBitmap(bi);
            if (pbm)
            {
                const MCHAR* bmFile = pbm->bi.Filename();
                if (bmFile)
                {
                    int texChannel = dx2Mat->GetBitmapMappingChannel( bi );
                    if ((texChannel>=0) && usedChannels.indexOf(texChannel) < 0)
                    {
                        usedChannels.pushBack(texChannel);
                    }
                }
            }
        }
    }

#endif
}

/*static*/ bool HK_CALL hctMaxUtils::isBone(INode* n)
{
  Object* pObj = n->GetObjectRef();
  if (!pObj) return false; // root node (or node with no shape)

  if (pObj->ClassID() == BONE_OBJ_CLASSID  )//|| pObj->ClassID().PartA() == BONE_CLASS_ID)
  {
    // BONE
    return true;
  }
  else if ((pObj->ClassID() == SKELOBJ_CLASS_ID) || (pObj->ClassID() == BIPED_CLASS_ID))
  {
    // BIPED
    return true;
  }
  else
  {
    // Geom as Bone?
    return n->GetBoneNodeOnOff();
  }
}

// Vertex anims etc
void HK_CALL hctMaxUtils::getAnimatableKeyTimes(INode* pNode, hkArray<int>& keyList, int startFrame, int endFrame, bool sort)
{
  Animatable *anim = (Animatable*)pNode;

  if(!anim->IsAnimated())
    return;

  const int iNumSubAnims = anim->NumSubs();

  int startTicks = getTicksFromFrame(startFrame);
  int endTicks = getTicksFromFrame(endFrame);

  for(int s=0; s < iNumSubAnims;  s++)
  {
    //Get the pointer to the sub animation
    //and check whether it is valid
    Animatable *subAnim = anim->SubAnim(s);

    if(!subAnim)
    {
      continue;
    }

    //Get the number of the keys of the current sub animation
    const int iNumKeys = subAnim->NumKeys();

    //Iterate over all the keyframes and store the keyframe
    //times. Check if they match the given animation time range
    int keyTime =0;
    for(int k=0; k<iNumKeys;k++)
    {
      keyTime = (int)subAnim->GetKeyTime(k);
      bool validKey = ( (endFrame != -1) && ((keyTime >= startTicks) && (keyTime <= endTicks)) ) ||
         ( (startFrame == 0) && (endFrame == -1) );

      if (validKey && (keyList.indexOf(keyTime) < 0) ) // valid and not a duplicate
      {
         keyList.pushBack(keyTime);
      }
      // handle case of [EXP-2436], where no keys in the range but range is affected by keys outside, so have to mark at start and p_end
      else if ( (keyTime < startTicks) && (keyList.indexOf(startTicks) < 0) )
      {
         keyList.pushBack(startTicks);
      }
      else if ( (keyTime > endTicks) && (keyList.indexOf(endTicks) < 0) )
      {
         keyList.pushBack(endTicks);
      }
    }

    //If this sub animation has further sub animation go through
    //them to check for further keyframes to be considered
    //Depth-First
    if(subAnim->NumSubs() > 0)
    {
      getAnimatableKeyTimes((INode*)subAnim, keyList, startFrame, endFrame, false);
    }
  }

  // there will be no duplicates, but will not be sorted. Best to sort now before use
  if( sort && (keyList.getSize() > 1))
  {
    hkSort( keyList.begin(), keyList.getSize() );
  }
}

void HK_CALL hctMaxUtils::getControllerKeyTimes(INode* pNode, hkArray<int>& keyList, int startTime, int endTime, bool sort)
{
  getControllerKeyTimes(pNode->GetTMController(), keyList, startTime, endTime, sort);
}

// Bone controllers like CAT
void HK_CALL hctMaxUtils::getControllerKeyTimes(Control* pC, hkArray<int>& keyList, int startTime, int endTime, bool sort)
{
  if(!pC)
    return;

  //if(!pC->IsAnimated())
  //  return;

  int startTicks = getTicksFromFrame(startTime);
  int endTicks   = getTicksFromFrame(endTime);

  //Get the number of the keys of the current sub animation
  const int iNumKeys = pC->NumKeys();

  //Iterate over all the keyframes and store the keyframe
  //times. Check if they match the given animation time range
  int keyTime =0;
  for(int k=0; k<iNumKeys;k++)
  {
    keyTime = (int)pC->GetKeyTime(k);
    if(endTime != -1  && (keyTime >= startTicks && keyTime <= endTicks))
    {
      keyList.pushBack(keyTime);
    }
    else if(startTime == 0 && endTime == -1)
    {
      keyList.pushBack(keyTime);
    }
  }

  // Loop over references.
  for(int n=pC->NumRefs(),r=0; r<n; ++r)
  {
    RefTargetHandle ref = pC->GetReference(r);
    if(ref)
      getControllerKeyTimes((Control*)ref->GetInterface(I_CONTROL), keyList, startTime, endTime);
  }

  // Recurse.
  for(int n=pC->NumSubs(),s=0; s<n; ++s)
  {
    Animatable* a = pC->SubAnim(s);
    if(a)
      getControllerKeyTimes((Control*)a->GetInterface(I_CONTROL), keyList, startTime, endTime);
  }


  // there will be no duplicates, but will not be sorted. Best to sort now before use
  if( sort && (keyList.getSize() > 1))
  {
    hkSort( keyList.begin(), keyList.getSize() );
  }
}

//
//  If the material is a Viewport/Render Shell material, get the viewport material. Otherwise return the same material

Mtl* hctMaxUtils::getViewportMaterial (Mtl* material)
{
    if (!material) return HK_NULL;

    int vpSubMat = -1;
    if ( material->ClassID() != Class_ID(MIXMAT_CLASS_ID, 0) )
    {
        vpSubMat = material->VPDisplaySubMtl();
    }

    // -1 == not implemented (not a "Shell" material)
    if (vpSubMat >= 0)
    {
        Mtl* vpmat = material->GetSubMtl(vpSubMat);
        return getViewportMaterial(vpmat); // For the weird case of chained Shell materials
    }
    else
    {
        return material;
    }
}

//
//  Returns the number of sub-materials of the given multiMaterial

int hctMaxUtils::getNumSubMaterials(Mtl* multiMaterial)
{
    // Check if the material is a shell etc we should just bypass into a child for
    Mtl* viewMat =  getViewportMaterial(multiMaterial);
    if ( !viewMat )
    {
        return 0;
    }

    Class_ID theId = viewMat->ClassID();
    ULONG idA = theId.PartA();
    if ( (idA != MULTI_CLASS_ID) && (idA != MIXMAT_CLASS_ID) && (idA != BAKE_SHELL_CLASS_ID) ) // then assume straightup, flat mat
    {
        return 0;
    }

    if (idA == BAKE_SHELL_CLASS_ID) // take the viewport mat, and go again
    {
        return getNumSubMaterials(viewMat);
    }

    // multi mat
    return viewMat->NumSubMtls();
}

void hctMaxUtils::findSubMaterials(Mtl* multiMaterial, int lookingFor, hkArray<Mtl*>& mats, hkArray<int>& blends)
{
    // Check if the material is a shell etc we should just bypass into a child for
    Mtl* viewMat =  getViewportMaterial(multiMaterial);
    if ( !viewMat )
    {
        return;
    }

    Class_ID theId = viewMat->ClassID();
    ULONG idA = theId.PartA();
    if ( (idA != MULTI_CLASS_ID) && (idA != MIXMAT_CLASS_ID) && (idA != BAKE_SHELL_CLASS_ID) ) // then assume straightup, flat mat
    {
        mats.pushBack( viewMat );
    }
    else // multi mat
    {
        if ( idA == MIXMAT_CLASS_ID ) // Blend, so add all sub mats
        {
            for (int mi = 0; mi < viewMat->NumSubMtls(); ++mi)
            {
                findSubMaterials( viewMat->GetSubMtl(mi), lookingFor, mats, blends);
            }

            // Add the blend map too, after the others (as destruction assumes in the propery set in this file that it is the last)
            if (viewMat->NumSubTexmaps() > 0)
            {
                blends.pushBack(mats.getSize()); // Destruction likes to know which are blends etc
                mats.pushBack( viewMat );
            }
        }
        else if (idA == BAKE_SHELL_CLASS_ID) // take the viewport mat, and go again
        {
            findSubMaterials( viewMat, lookingFor, mats, blends );
        }
        else
        {
            // mutli/mat. Recurse into wanted mat only
            const int numSlots  = viewMat->NumSubMtls();
            const int modSlot   = lookingFor % numSlots;
            Mtl* subMat         = viewMat->GetSubMtl(modSlot);
            if (subMat)
            {
                findSubMaterials(subMat,lookingFor,mats, blends);
            }
        }
    }
}

//
//  Retrieves the full name of the given sub-material

Mtl* hctMaxUtils::findSubMaterialName(Mtl* keyMtl, int lookingFor, hkStringPtr& nameOut)
{
    nameOut = HK_NULL;
    if ( !keyMtl )
    {
        return HK_NULL;
    }

    HCT_SCOPED_CONVERSIONS;

    const MSTR keyMtlName = keyMtl->GetName();
    hkStringBuf strb(FROM_MAX(keyMtlName));

    hkArray<Mtl*> mtls;
    hkArray<int> blends;
    hctMaxUtils::findSubMaterials(keyMtl, lookingFor, mtls, blends);

    for (int mi = 0; mi < mtls.getSize(); mi++)
    {
        if ( mtls[mi] != keyMtl )
        {
            const MSTR subMatName = mtls[mi]->GetName();
            strb.appendJoin("/", FROM_MAX(subMatName));
        }
    }

    nameOut = strb;
    return mtls.getSize() ? mtls[mtls.getSize() - 1] : keyMtl;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
