// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SdkUtils/ClassHierarchy/hctClassHierarchyUtil.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxDestructionUtilities.h>

void hctMaxDestructionUtilities::initDestructionPlugin()
{
    hctSdkUtils::loadAllUtils();

    // Sync statics
    const HMODULE dllHandle = hctSdkUtils::m_destructionUtilsDll;
    if ( dllHandle )
    {
        // Get base Dll interface
        typedef class hctBaseDll* (HK_CALL *hctGetBaseDllInterfaceFunc)();
        hctGetBaseDllInterfaceFunc destBaseDllFunc = (hctGetBaseDllInterfaceFunc) GetProcAddress(dllHandle, "getBaseDllInterface");
        if ( !destBaseDllFunc )
        {
            HK_WARN_ALWAYS(0x6fdba42b,"Destruction Util DLL found but could not find baseDLL sync function. Install is invalid.");
            hctSdkUtils::unloadDestructionUtils();
            return;
        }

        // Sync statics
        hctBaseDll* dll = destBaseDllFunc();
        hctSdkUtils::m_baseDestructionUtilsDll = dll;
        if ( dll )
        {
            if ( dll->getDllVersion() != HCT_CURRENT_VERSION )
            {
                HK_WARN_ALWAYS(0x7f425fa0, "Could not load Destruction Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall." );
                hctSdkUtils::m_baseDestructionUtilsDll = HK_NULL;
                hctSdkUtils::unloadDestructionUtils();
                return;
            }

            // Init the shared mem system.
            hkMemoryInitUtil::SyncInfo baseSystemInfo;
            hkSceneExportMemory::getBaseSystemSyncInfo(baseSystemInfo);
            dll->initDll(baseSystemInfo, HK_NULL);
        }
        else
        {
            // We failed to sync statics, most likely the key-code is not good / expired. Unload the dll!
            hctSdkUtils::unloadDestructionUtils();
        }
    }
    hctSdkUtils::loadAllClasses();
}

void hctMaxDestructionUtilities::quitDestructionPlugin()
{
    // Unsync statics
    hctBaseDll* dll = hctSdkUtils::m_baseDestructionUtilsDll;
    if ( dll )
    {
        hctSdkUtils::m_baseDestructionUtilsDll->quitDll();
        hctSdkUtils::m_baseDestructionUtilsDll = HK_NULL;
    }

    hctSdkUtils::unloadAllUtils();
    hctSdkUtils::unloadAllClasses();
}


def_visible_primitive( hctGetDerivedClassesInfo, "hctGetDerivedClassesInfo" );
Value* hctGetDerivedClassesInfo_cf( Value** arg_list, int count )
{
    HCT_SCOPED_CONVERSIONS;

    // We expect exactly 2 parameters: baseClassName (string), outputArray (array)
    check_arg_count(hctGetDerivedClassesInfo, 2, count);

    // Convert generic input values into proper types.
    const char* baseClassName = FROM_MAX(arg_list[0]->to_string());
    Array* outputArray = reinterpret_cast<Array*>(arg_list[1]);

    // Get all classes that are derived from the supplied base class.
    hkArray<hctModelerNodeIdToClassMap> classes;
    hctSdkUtils::getClassesByBaseName(baseClassName, classes);

    // Get the UI style and hide all classes that aren't supposed to be visible
    const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAX);

    // Fill output array with both the class name and the display name.
    {
        for (int i = 0; i < classes.getSize(); i++)
        {
            HCT_SCOPED_CONVERSIONS;

            hctModelerNodeIdToClassMap& classInfo   = classes[i];
            const hkReflect::Type* klass                    = classInfo.m_class;
            const hkUiAttribute* uiAttrib           = klass ? klass->findAttribute<hkUiAttribute>() : HK_NULL;
            const bool bHidden                      = uiAttrib && (uiAttrib->m_hideCriteria & hideCriteria);

            if ( !bHidden )
            {
                // Let MAXScript know that these values must not be freed upon p_end of function.
                three_typed_value_locals(String* className, String* displayName, String* classification);

                // Append class name to array.
                vl.className =  new String(TO_MAX(classInfo.m_class->getName()));
                outputArray->append(vl.className);

                // Append display name to array.
                vl.displayName =  new String(TO_MAX(classInfo.m_displayName));
                outputArray->append(vl.displayName);

                // Append internal classification to array.
                hkStringBuf classification;
                classification.printf("%d", classInfo.m_internalClassification);
                vl.classification = new String(TO_MAX(classification));
                outputArray->append(vl.classification);
            }
        }
    }

    return &true_value;
}


def_visible_primitive( hctVersionCheck, "hctVersionCheck" );
Value* hctVersionCheck_cf( Value** arg_list, int count )
{
    // We expect no paraeters.
    check_arg_count(hctGetDerivedClassesInfo, 0, count);

    // Some version checking.
    {
        HCT_SCOPED_CONVERSIONS;

        const char* versionString = FROM_MAX(UtilityInterface::GetCurrentVersion());
        int majorVersion = hkString::atoi(versionString);
        int minorVersion = 0;
        const char* minorVersionString = hkString::strChr(versionString, '.');
        if ( minorVersionString )
        {
            minorVersion = hkString::atoi(minorVersionString+1);
        }

        switch ( majorVersion )
        {
            case 11:
            {
                if ( minorVersion < 1 )
                {
                    ExecuteMAXScriptScript(TEXT("messagebox \"Havok Destruction requires 3ds Max 2009 Service Pack 1 to be installed!\" title:\"Havok Destruction - IMPORTANT\""));
                }
                break;
            }
            default:
            {
                break;
            }
        }
    }

    return &true_value;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
