// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Max/MaxSceneExport/Utilities/hctSceneExportUtility.h>
#include <ContentTools/Max/MaxFpInterfaces/Export/hctSceneExportUtilityInterface.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>

// EXP-1070 : The Maps (rollouts) in the utility
enum
{
    MAP_EXPORT,
    MAP_SCENE_DATA,
    MAP_FRAME_RANGE,
    MAP_FILTER_SETUP,
    MAP_ADVANCED,
};


///////////////////////////////////////////////////////////////////////////////////////////
//
// The class descriptor for the hctSceneExportUtility utility plugin.
//
///////////////////////////////////////////////////////////////////////////////////////////

class hkSceneExportUtilityClassDesc : public ClassDesc2
{
public:
    int             IsPublic() { return TRUE; }
    void*           Create( BOOL loading = FALSE );
    const MCHAR *   ClassName() { return GetString(IDS_EXPORT_UTILITY_CLASS_NAME); }
    SClass_ID       SuperClassID() { return UTILITY_CLASS_ID; }
    Class_ID        ClassID() { return HK_SCENE_EXPORT_UTILITY_CLASS_ID; }
    const MCHAR*    Category() { return GetString(IDS_EXPORT_UTILITY_CATEGORY); }
    const MCHAR*    InternalName() { return _T("hkSceneExportUtility"); }       // Returns fixed parsable name (scripter-visible name).
    HINSTANCE       HInstance() { return hInstance; }                   // Returns owning module handle.

    // Persistant settings in the max file.
    virtual BOOL     NeedsToSave() { return TRUE; }
    virtual IOResult Save( ISave *s );
    virtual IOResult Load( ILoad *l );
};

ClassDesc2* getHkSceneExportUtilityDesc()
{
    static hkSceneExportUtilityClassDesc maxSceneExportDesc;

    return &maxSceneExportDesc;
}

void* hkSceneExportUtilityClassDesc::Create( BOOL loading )
{
    return getSceneExportUtilityInstance();
}

IOResult hkSceneExportUtilityClassDesc::Save( ISave *s )
{
    return getSceneExportUtilityInstance()->Save(s);
}

IOResult hkSceneExportUtilityClassDesc::Load( ILoad *l )
{
    return getSceneExportUtilityInstance()->Load(l);
}

void _preMergeCallback(void*, NotifyInfo*)
{
    getSceneExportUtilityInstance()->backupUtilityData();
}

void _postMergeCallback(void*, NotifyInfo*)
{
    getSceneExportUtilityInstance()->restoreUtilityData();
}

hctSceneExportUtility::hctSceneExportUtility()
:   m_iutil(NULL),
m_interface(NULL)
{
    // Create the plugin's parameter blocks.
    m_pbSceneExport = NULL;
    getHkSceneExportUtilityDesc()->MakeAutoParamBlocks(this);

    RegisterNotification(_preMergeCallback, NULL, NOTIFY_FILE_PRE_MERGE);
    RegisterNotification(_postMergeCallback, NULL, NOTIFY_FILE_POST_MERGE);
}

hctSceneExportUtility::~hctSceneExportUtility()
{
    UnRegisterNotification (_preMergeCallback, NULL);
    UnRegisterNotification (_postMergeCallback, NULL);
}


bool hctSceneExportUtility::exportScene( BOOL batchMode )
{
    // Set the export options from the paramBlock values.
    hctMaxSceneExportOptions exportOptions;
    getExportOptions(exportOptions);

    exportOptions.m_batchMode = (batchMode != FALSE);

    hctMaxSceneExporter* sceneExporter = new hctMaxSceneExporter(exportOptions, m_filterSetOptions);

    // Use our 'message box' error handler only if we are not in batch mode
    hctSceneExportError* errorHandler = exportOptions.m_batchMode? new hctSceneExportError(): new hkMaxSceneError();
    hkError* normError = &hkError::getInstance();
    normError->addReference();
    hkError::replaceInstance( errorHandler );

    bool okVal = sceneExporter->exportAndProcess();

    delete sceneExporter;

    if( errorHandler )
    {
        hkError::replaceInstance( normError );
    }

    return okVal;
}

bool hctSceneExportUtility::openVideoBrowser()
{
    return hctSdkUtils::openVideoBrowser("-modeller COMMON+MAX");
}

bool hctSceneExportUtility::openContentToolsChm()
{
    return hctSdkUtils::openDocumentationChmFile();
}

void hctSceneExportUtility::getOptionsFileDependencies(Tab<MSTR*>& dependencies)
{
    // Set the export options from the paramBlock values.
    hctMaxSceneExportOptions exportOptions;
    getExportOptions(exportOptions);

    exportOptions.m_batchMode = true;// (batchMode != FALSE);

    hctMaxSceneExporter* sceneExporter = new hctMaxSceneExporter(exportOptions, m_filterSetOptions);

    // Use our 'message box' error handler only if we are not in batch mode
    hctSceneExportError* errorHandler = exportOptions.m_batchMode ? new hctSceneExportError() : new hkMaxSceneError();
    hkError* normError = &hkError::getInstance();
    normError->addReference();
    hkError::replaceInstance(errorHandler);

    hkArray<hkStringPtr> fileDependencies;
    sceneExporter->getFileDependencies(fileDependencies);

    delete sceneExporter;

    if (errorHandler)
    {
        hkError::replaceInstance(normError);
    }

    dependencies.SetCount(fileDependencies.getSize());

    for (int i = 0; i < fileDependencies.getSize(); i++)
    {
        HCT_SCOPED_CONVERSIONS;
        MSTR* maxStr = new MSTR( TO_MAX(fileDependencies[i].cString()) );
        dependencies[i] = maxStr;

    }
}


// Import is in its own cpp


//
// Argument parsing.
//
void  hctSceneExportUtility::getExportOptions (hctMaxSceneExportOptions& exportOptionsOut)
{
    HCT_SCOPED_CONVERSIONS;

    if (m_pbSceneExport->GetInt(PA_UseOptionsFile)!=FALSE)
    {
        exportOptionsOut.m_optionsFile = FROM_MAX(m_pbSceneExport->GetStr( PA_OptionsFile ));
    }

    if (m_pbSceneExport->GetInt(PA_UseConfigurationSet) != FALSE)
    {
        exportOptionsOut.m_configurationSet = FROM_MAX(m_pbSceneExport->GetStr(PA_ConfigurationSet));
    }

    exportOptionsOut.m_environmentVariables = FROM_MAX( m_pbSceneExport->GetStr (PA_EnvironmentVariables) );

    exportOptionsOut.m_visibleOnly = (m_pbSceneExport->GetInt( PA_VisibleOnly ) != FALSE);

    exportOptionsOut.m_selectedOnly = (m_pbSceneExport->GetInt( PA_SelectedOnly ) != FALSE);

    exportOptionsOut.m_exportMeshes = (m_pbSceneExport->GetInt( PA_ExportMeshes ) != FALSE);

    exportOptionsOut.m_exportMaterials = (m_pbSceneExport->GetInt( PA_ExportMaterials ) != FALSE);

    exportOptionsOut.m_exportAttributes = (m_pbSceneExport->GetInt( PA_ExportAttributes ) != FALSE);

    exportOptionsOut.m_exportAnnotations = (m_pbSceneExport->GetInt( PA_ExportAnnotations ) != FALSE);

    exportOptionsOut.m_exportLights = (m_pbSceneExport->GetInt( PA_ExportLights ) != FALSE);

    exportOptionsOut.m_exportCameras = (m_pbSceneExport->GetInt( PA_ExportCameras ) != FALSE);

    exportOptionsOut.m_addDefaultCamera = exportOptionsOut.m_exportCameras && (m_pbSceneExport->GetInt( PA_AddDefaultCamera ) != FALSE);

    exportOptionsOut.m_doNotSplitVertices = (m_pbSceneExport->GetInt( PA_DoNotSplitVertices) != FALSE);

    // EXP-2481 More options

    exportOptionsOut.m_respectModifierSkinPose = (m_pbSceneExport->GetInt( PA_RespectModifierSkinPose) != FALSE);

    exportOptionsOut.m_autoSkinAttachments = (m_pbSceneExport->GetInt( PA_AutoSkinAttachments) != FALSE);

    exportOptionsOut.m_exportWireColorsAsVertexColors = (m_pbSceneExport->GetInt( PA_ExportWireColorsAsVertexColors) != FALSE);

    exportOptionsOut.m_exportMergeVertsOnlyIfSameSrcId = (m_pbSceneExport->GetInt( PA_ExportMergeVertsOnlyIfSameSrcId) != FALSE);

    exportOptionsOut.m_storeKeyframeSamplePoints = (m_pbSceneExport->GetInt( PA_StoreKeyframeSamplePoints) != FALSE);

    exportOptionsOut.m_expandSelectionToSkeleton = (m_pbSceneExport->GetInt( PA_ExpandSelectionToSkeleton ) != FALSE);
    exportOptionsOut.m_expandSelectionToParents  = (m_pbSceneExport->GetInt( PA_ExpandSelectionToParents ) != FALSE);
    exportOptionsOut.m_expandSelectionToChildren = (m_pbSceneExport->GetInt( PA_ExpandSelectionToChildren ) != FALSE);

    exportOptionsOut.m_exportVertexTangents = (m_pbSceneExport->GetInt(PA_ExportTangents) != FALSE);

    // Animation Range
    Interface* coreInterface = GetCOREInterface();
    switch(m_pbSceneExport->GetInt(PA_AnimatedDataExport))
    {
    case ADE_CURRENT_FRAME:
        {
            const TimeValue now = coreInterface->GetTime();
            exportOptionsOut.m_animationStart = now;
            exportOptionsOut.m_animationEnd = now;
            break;
        }
    case ADE_SPECIFIC_RANGE:
        {
            exportOptionsOut.m_animationStart = m_pbSceneExport->GetInt(PA_AnimationStart);
            exportOptionsOut.m_animationEnd = m_pbSceneExport->GetInt(PA_AnimationEnd);
            break;
        }
    case ADE_CURRENT_RANGE:
    default:
        {
            const Interval currentRange = coreInterface->GetAnimRange();
            exportOptionsOut.m_animationStart = currentRange.Start();
            exportOptionsOut.m_animationEnd = currentRange.End();
            break;
        }
    }
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// Stripped-down version of the old hctSceneExportUtility scene export plugin.
// Allows filter sets saved with the old plugin to be loaded and used with the new version.
//
///////////////////////////////////////////////////////////////////////////////////////////

class hkOldSceneExporter: public SceneExport
{
    // SceneExport methods.
    int ExtCount() { return 0; }
    const MCHAR* Ext(int n) { return NULL; }
    const MCHAR* LongDesc() { return _T(""); }
    const MCHAR* ShortDesc() { return _T(""); }
    const MCHAR* AuthorName() { return _T(""); }
    const MCHAR* CopyrightMessage() { return _T("(c) Microsoft Corporation"); }
    const MCHAR* OtherMessage1() { return _T(""); }
    const MCHAR* OtherMessage2() { return _T(""); }
    unsigned int Version() { return 120; }
    void ShowAbout(HWND hWnd) {}
    int DoExport( const MCHAR *name, ExpInterface *ei, Interface *i, BOOL suppressPrompts=FALSE, DWORD options = 0 ) { return TRUE; }
};

#define hkSceneExportUtility_CLASS_ID_old   Class_ID(0x16052618, 0x2e0770f6)

class hkOldSceneExporterClassDesc : public ClassDesc2
{
public:

    int             IsPublic() { return FALSE; }
    void*           Create(BOOL loading = FALSE) { return new hkOldSceneExporter(); }
    const MCHAR*    ClassName() { return GetString(IDS_EXPORT_UTILITY_CLASS_NAME); }
    SClass_ID       SuperClassID() { return SCENE_EXPORT_CLASS_ID; }
    Class_ID        ClassID() { return hkSceneExportUtility_CLASS_ID_old; }
    const MCHAR*    Category() { return GetString(IDS_EXPORT_UTILITY_CATEGORY); }

    virtual BOOL     NeedsToSave() { return TRUE; }
    virtual IOResult Load( ILoad *iload );

    const MCHAR*    InternalName() { return _T("hkSceneExporter_old"); }
    HINSTANCE       HInstance() { return hInstance; }
};

ClassDesc2* getHkOldSceneExporterDesc()
{
    static hkOldSceneExporterClassDesc oldExporterDesc;
    return &oldExporterDesc;
}

IOResult hkOldSceneExporterClassDesc::Load( ILoad* iload )
{
    return getSceneExportUtilityInstance()->Load( iload );
}

/*
** Check wheter the old physics exporter is present
*/

/*static*/ bool hctSceneExportUtility::isOldPhysicsExporterPresent ()
{
    static const Class_ID oldPhysicsExporterClassID(0x6f477f46, 0x2959095f);

    Interface* ip = GetCOREInterface();

    void* oldExporter = ip->CreateInstance(UTILITY_CLASS_ID, oldPhysicsExporterClassID);

    return oldExporter != NULL;
}



///////////////////////////////////////////////////////////////////////////////////////////
//
// Save and load functions for the filter options.
//
///////////////////////////////////////////////////////////////////////////////////////////

// Filter sets are stored in a global array of chars, used by the plugin as a buffer for loading/saving.
#define FILTER_SET_OPTIONS_CHUNK_ID         1
#define OPTIONS_FILE_CHUNK_ID               2
#define SCENE_EXPORT_OPTIONS_CHUNK_ID       3
#define ENVIRONMENT_VARIABLES_CHUNK_ID      4
#define DONOTSPLITVERTICES_CHUNK_ID         5
#define EXTRA_OPTIONS_CHUNK_ID              6
#define OPTIONS_FILE_WIDE_CHUNK_ID          7
#define CONFIGURATION_SET_CHUNK_ID          8

IOResult hctSceneExportUtility::Save(ISave *isave)
{
    HCT_SCOPED_CONVERSIONS;

    // The number of bytes written.
    ULONG nb;

    // Storage for parameters before writing them out.
    int intParam;

    // Save the filter options with version info so that we can ignore if too old / new.
    if (m_filterSetOptions.m_data.Count() > 0)
    {
        isave->BeginChunk( FILTER_SET_OPTIONS_CHUNK_ID );
        {
            isave->Write( &m_filterSetOptions.m_version, sizeof(DWORD), &nb );
            if (m_filterSetOptions.m_data.Count() > 0)
            {
                // In v2013 max checks the string length but we are not storing a string, we are storing an array of chars,
                // i.e., there is no '\0' at the end
                isave->WRITEVOID( (void*) &(m_filterSetOptions.m_data[0]), m_filterSetOptions.m_data.Count(), &nb );
            }
        }
        isave->EndChunk();
    }

    // Save the options file.
    isave->BeginChunk( OPTIONS_FILE_WIDE_CHUNK_ID );
    {
        MSTR strParam = m_pbSceneExport->GetStr( PA_OptionsFile );
        isave->WriteWString( strParam );
    }
    isave->EndChunk();

    // Save the remaining export options.
    isave->BeginChunk( SCENE_EXPORT_OPTIONS_CHUNK_ID );
    {
        intParam = m_pbSceneExport->GetInt( PA_UseOptionsFile );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_VisibleOnly );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_SelectedOnly );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportMeshes );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportMaterials );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportAttributes );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportAnnotations );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportLights );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_ExportCameras );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = FALSE; // Deprecated PA_UseOldPhysics
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_AddDefaultCamera );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_AnimatedDataExport );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_AnimationStart );
        isave->Write( &intParam, sizeof(intParam), &nb );

        intParam = m_pbSceneExport->GetInt( PA_AnimationEnd );
        isave->Write( &intParam, sizeof(intParam), &nb );
    }
    isave->EndChunk();

    // Save the environment variables
    isave->BeginChunk( ENVIRONMENT_VARIABLES_CHUNK_ID );
    {
        MSTR strParam = m_pbSceneExport->GetStr( PA_EnvironmentVariables );
        isave->WriteCString(strParam);
    }
    isave->EndChunk();

    // Save the configuration set
    isave->BeginChunk( CONFIGURATION_SET_CHUNK_ID );
    {
        intParam = m_pbSceneExport->GetInt(PA_UseConfigurationSet);
        isave->Write(&intParam, sizeof(intParam), &nb);

        MSTR strParam = m_pbSceneExport->GetStr(PA_ConfigurationSet);
        isave->WriteCString(strParam);
    }
    isave->EndChunk();

    // Save the "Do Not Split Vertices" flag
    isave->BeginChunk( DONOTSPLITVERTICES_CHUNK_ID);
    {
        intParam = m_pbSceneExport->GetInt(PA_DoNotSplitVertices);
        isave->Write(&intParam, sizeof(intParam), &nb);
    }
    isave->EndChunk();

    isave->BeginChunk( EXTRA_OPTIONS_CHUNK_ID);
    {
        intParam = m_pbSceneExport->GetInt(PA_RespectModifierSkinPose);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_AutoSkinAttachments);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExportWireColorsAsVertexColors);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExportMergeVertsOnlyIfSameSrcId);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_StoreKeyframeSamplePoints);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExpandSelectionToSkeleton);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExpandSelectionToParents);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExpandSelectionToChildren);
        isave->Write(&intParam, sizeof(intParam), &nb);

        intParam = m_pbSceneExport->GetInt(PA_ExportTangents);
        isave->Write(&intParam, sizeof(intParam), &nb);
    }
    isave->EndChunk();


    return IO_OK;
}

IOResult hctSceneExportUtility::Load( ILoad *iload )
{
    ULONG nb;
    IOResult res;
    int intParam;
    MCHAR* strParam;

    // Reset all defaults
    m_pbSceneExport->ResetAll();

    // Checks if ok to understand the rest / try to later on. Allows old files to be loaded and saved, aslong as the exporter does not have to read them.
    while ( IO_OK == ( res=iload->OpenChunk() ) )
    {
        switch( iload->CurChunkID() )
        {
        case FILTER_SET_OPTIONS_CHUNK_ID:
            {
                DWORD fmv;
                res = iload->Read(&fmv, sizeof(DWORD), &nb);

                if ( res==IO_OK )
                {
                    m_filterSetOptions.m_version = fmv;
                    const int len = (int) (iload->CurChunkLength() - sizeof(DWORD)); // size of the data, less header and the first version dword
                    m_filterSetOptions.m_data.SetCount(len);
                    iload->READVOID( (void*) &(m_filterSetOptions.m_data[0]), len, &nb );
                }
            }
            break;

        case ENVIRONMENT_VARIABLES_CHUNK_ID:
            {
                res = iload->ReadCStringChunk( &strParam );
                m_pbSceneExport->SetValue( PA_EnvironmentVariables, 0, strParam );
            }
            break;


        case CONFIGURATION_SET_CHUNK_ID:
            {
                res = iload->Read(&intParam, sizeof(intParam), &nb);
                m_pbSceneExport->SetValue(PA_UseConfigurationSet, 0, intParam);

                res = iload->ReadCStringChunk(&strParam);
                m_pbSceneExport->SetValue(PA_ConfigurationSet, 0, strParam);
            }
            break;

        case OPTIONS_FILE_CHUNK_ID:
            {
                res = iload->ReadCStringChunk( &strParam );
                m_pbSceneExport->SetValue( PA_OptionsFile, 0, strParam );
            }
            break;
        case OPTIONS_FILE_WIDE_CHUNK_ID:
            {
                res = iload->ReadWStringChunk( &strParam );
                m_pbSceneExport->SetValue( PA_OptionsFile, 0, strParam );
            }
            break;

        case SCENE_EXPORT_OPTIONS_CHUNK_ID:
            {
                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_UseOptionsFile, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_VisibleOnly, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_SelectedOnly, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportMeshes, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportMaterials, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportAttributes, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportAnnotations, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportLights, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_ExportCameras, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                // Ignore this value, is the deprecated PA_UseOldPhysics;

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue( PA_AddDefaultCamera, 0, intParam );

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue(PA_AnimatedDataExport, 0, intParam);

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue(PA_AnimationStart, 0, intParam);

                res = iload->Read( &intParam, sizeof(intParam), &nb );
                m_pbSceneExport->SetValue(PA_AnimationEnd, 0, intParam);
            }
            break;

            case DONOTSPLITVERTICES_CHUNK_ID:
                {
                    res = iload->Read( &intParam, sizeof(intParam), &nb);
                    m_pbSceneExport->SetValue(PA_DoNotSplitVertices, 0, intParam);
                }
            break;

            case EXTRA_OPTIONS_CHUNK_ID:
                {
                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    m_pbSceneExport->SetValue(PA_RespectModifierSkinPose, 0, intParam);

                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    m_pbSceneExport->SetValue(PA_AutoSkinAttachments, 0, intParam);

                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    m_pbSceneExport->SetValue(PA_ExportWireColorsAsVertexColors, 0, intParam);

                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    m_pbSceneExport->SetValue(PA_ExportMergeVertsOnlyIfSameSrcId, 0, intParam);

                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    m_pbSceneExport->SetValue(PA_StoreKeyframeSamplePoints, 0, intParam);

                    res = iload->Read( &intParam, sizeof(intParam), &nb );
                    if ( res == IO_OK) // may be older file, so leave these at defaults if not set
                    {
                        m_pbSceneExport->SetValue(PA_ExpandSelectionToSkeleton, 0, intParam);

                        res = iload->Read( &intParam, sizeof(intParam), &nb );
                        m_pbSceneExport->SetValue(PA_ExpandSelectionToParents, 0, intParam);

                        res = iload->Read( &intParam, sizeof(intParam), &nb );
                        m_pbSceneExport->SetValue(PA_ExpandSelectionToChildren, 0, intParam);

                        res = iload->Read(&intParam, sizeof(intParam), &nb);
                        if (res == IO_OK) // may be older file, so leave these at defaults if not set
                        {
                            m_pbSceneExport->SetValue(PA_ExportTangents, 0, intParam);
                        }
                    }
                }
        }

        iload->CloseChunk();
        if (res!=IO_OK)
        {
            return res;
        }
    }

    updateUI();

    return IO_OK;
}

void hctSceneExportUtility::updateUI()
{
    if ( m_interface == NULL )
    {
        return;
    }

    // Filter Setup
    {
        IParamMap2* pmap = m_pbSceneExport->GetMap( MAP_FILTER_SETUP );
        HWND hWnd = pmap->GetHWnd();

        if (hWnd)
        {
            ICustEdit* optionsFileCtrl = GetICustEdit( GetDlgItem( hWnd, IDC_CUSTOM_OPTIONS_FILE ) );

            // Activate/deactivate the edit box and file chooser.
            if ( m_pbSceneExport->GetInt( PA_UseOptionsFile ) != FALSE )
            {
                optionsFileCtrl->Enable();
                EnableWindow( GetDlgItem( hWnd, IDC_SELECT_OPTIONS_FILE ), TRUE );
            }
            else
            {
                optionsFileCtrl->Disable();
                EnableWindow( GetDlgItem( hWnd, IDC_SELECT_OPTIONS_FILE ), FALSE );
            }

            ICustEdit* configurationSetCtrl = GetICustEdit(GetDlgItem(hWnd, IDC_ED_CONFIGURATION_SET));

            // Activate/deactivate the edit box and file chooser.
            if (m_pbSceneExport->GetInt(PA_UseConfigurationSet) != FALSE)
            {
                configurationSetCtrl->Enable();
            }
            else
            {
                configurationSetCtrl->Disable();
            }
        }
    }

    // Frame Range
    // Enable/disable start&p_end frame controls (label, edit, spinner)
    {
        IParamMap2* pmap = m_pbSceneExport->GetMap( MAP_FRAME_RANGE );
        HWND hWnd = pmap->GetHWnd();

        if (hWnd)
        {
            const int enableRangeCtrls = (m_pbSceneExport->GetInt(PA_AnimatedDataExport) == ADE_SPECIFIC_RANGE) ? TRUE:FALSE;
            ICustEdit *iEdit;
            ISpinnerControl* iSpinner;

            EnableWindow(GetDlgItem(hWnd, IDC_LAB_ANIMATION_START), enableRangeCtrls);

            iEdit = GetICustEdit(GetDlgItem(hWnd, IDC_ED_ANIMATION_START));
            iEdit->Enable(enableRangeCtrls);
            ReleaseICustEdit(iEdit);

            iSpinner = GetISpinner(GetDlgItem(hWnd, IDC_SP_ANIMATION_START));
            iSpinner->Enable(enableRangeCtrls);
            ReleaseISpinner(iSpinner);

            EnableWindow(GetDlgItem(hWnd, IDC_LAB_ANIMATION_END), enableRangeCtrls);

            iEdit = GetICustEdit(GetDlgItem(hWnd, IDC_ED_ANIMATION_END));
            iEdit->Enable(enableRangeCtrls);
            ReleaseICustEdit(iEdit);

            iSpinner = GetISpinner(GetDlgItem(hWnd, IDC_SP_ANIMATION_END));
            iSpinner->Enable(enableRangeCtrls);
            ReleaseISpinner(iSpinner);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////////////
//
// Dialog Procs
//
///////////////////////////////////////////////////////////////////////////////////////////


// EXPORT Rollout
class ExportRolloutDlgProc : public ParamMap2UserDlgProc
{
public:

    INT_PTR DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
    void DeleteThis() {}
};

INT_PTR ExportRolloutDlgProc::DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    switch (msg)
    {
    case WM_INITDIALOG:
        {
            HWND exportButtonWnd = GetDlgItem(hWnd, IDC_B_EXPORT_SCENE);
            ICustButton* ibutton = GetICustButton(exportButtonWnd);

            ibutton->SetText(TEXT("Export Scene"));

            ReleaseICustButton(ibutton);

            HWND labVersion = GetDlgItem (hWnd, IDC_LAB_VERSION);
            SetWindowText (labVersion, TEXT(HCT_CURRENT_VERSION_STRING));

        }
        break;

    case WM_DESTROY:
        break;

    case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
            case IDC_B_EXPORT_SCENE:
                {
                    // AUTOMATICALLY HANDLED AS A PUBLISHED ACTION
                    break;
                }
                    case IDC_B_PRESET_DEFAULT:
                        {
                            // Default settings
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_DoNotSplitVertices, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_AutoSkinAttachments, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_RespectModifierSkinPose, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExportWireColorsAsVertexColors, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExportMergeVertsOnlyIfSameSrcId, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_StoreKeyframeSamplePoints, 0, FALSE);

                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExpandSelectionToSkeleton, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExpandSelectionToParents, 0, TRUE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExpandSelectionToChildren, 0, FALSE);
                            getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_ExportTangents, 0, TRUE); // With new tools etc we would assume tangents in assets
                            break;
                        }
            }
        }
        break;

    case WM_LBUTTONDOWN:
    case WM_LBUTTONUP:
    case WM_MOUSEMOVE:
        getSceneExportUtilityInstance()->m_interface->RollupMouseMessage( hWnd, msg, wParam, lParam );
        break;

    default:
        return FALSE;
    }
    return FALSE;//TRUE;
}


static ExportRolloutDlgProc g_utilityExportDlgProc;


// FILTER SETUP Rollout

class FilterSetupRolloutDlgProc : public ParamMap2UserDlgProc
{
public:

    INT_PTR DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
    void DeleteThis() {}
};

INT_PTR FilterSetupRolloutDlgProc::DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    static ICustEdit* optionsFileCtrl = NULL;

    switch (msg)
    {
    case WM_INITDIALOG:
        {
            // Get a handle to the custom control used to display the selected options file.
            optionsFileCtrl = GetICustEdit( GetDlgItem( hWnd, IDC_CUSTOM_OPTIONS_FILE ) );

            EnableWindow( GetDlgItem( hWnd, IDC_SELECT_OPTIONS_FILE ), FALSE);
            optionsFileCtrl->Disable();
        }
        break;

    case WM_DESTROY:
        ReleaseICustEdit( optionsFileCtrl );
        break;

    case WM_COMMAND:
        {
            switch (LOWORD(wParam))
            {
            case IDC_CB_USE_OPTIONS_FILE:
                {
                    // Activate/deactivate the edit box and file chooser.
                    if ( IsDlgButtonChecked( hWnd, IDC_CB_USE_OPTIONS_FILE ) == TRUE )
                    {
                        optionsFileCtrl->Enable();
                        EnableWindow( GetDlgItem( hWnd, IDC_SELECT_OPTIONS_FILE ), TRUE );
                    }
                    else
                    {
                        optionsFileCtrl->Disable();
                        EnableWindow( GetDlgItem( hWnd, IDC_SELECT_OPTIONS_FILE ), FALSE );
                    }
                }
                break;

            case IDC_SELECT_OPTIONS_FILE:
                {
                    // Open up a file chooser.
                    MCHAR filename[2048];
                    filename[0] = '\0';

                    OPENFILENAME op;
                    hkString::memSet( &op, 0, sizeof(OPENFILENAME) );
                    op.lStructSize = sizeof(OPENFILENAME);
                    op.hwndOwner = hWnd;
                    op.lpstrFilter = TEXT("Havok Option Files (*.hko)\0*.hko\0All Files (*.*)\0*.*\0\0");
                    op.lpstrInitialDir = '\0';
                    op.lpstrFile = filename;
                    op.lpstrDefExt = TEXT("hko");
                    op.nMaxFile = 2048;
                    op.Flags = OFN_ENABLESIZING | OFN_EXPLORER | OFN_NOREADONLYRETURN | OFN_PATHMUSTEXIST;

                    if ( GetOpenFileName(&op) )
                    {
                        optionsFileCtrl->SetText( op.lpstrFile );

                        getSceneExportUtilityInstance()->m_pbSceneExport->SetValue(PA_OptionsFile, 0, op.lpstrFile);
                    }

                }
                break;
            }
        }
        break;

    case WM_LBUTTONDOWN:
    case WM_LBUTTONUP:
    case WM_MOUSEMOVE:
        getSceneExportUtilityInstance()->m_interface->RollupMouseMessage( hWnd, msg, wParam, lParam );
        break;

    default:
        return FALSE;
    }
    return FALSE;//TRUE;
}


static FilterSetupRolloutDlgProc g_utilityFilterSetupDlgProc;


///////////////////////////////////////////////////////////////////////////////////////////
//
// The descriptor for the scene export param block.
//
///////////////////////////////////////////////////////////////////////////////////////////

static ParamBlockDesc2 pbdExportScene
(
 PB_SceneExport, _T("exportScene"), IDS_EXPORT_UTILITY_ACTION_EXPORT_SCENE, getHkSceneExportUtilityDesc(),
 P_AUTO_UI + P_AUTO_CONSTRUCT + P_MULTIMAP, PBLOCK_SceneExport,

 5,
 MAP_EXPORT,
 IDD_EXPORT_UTILITY_ROLLOUT_EXPORT,
 IDS_EXPORT_UTILITY_ROLLOUT_EXPORT,
 0,0,
 &g_utilityExportDlgProc,
 MAP_SCENE_DATA,
 IDD_EXPORT_UTILITY_ROLLOUT_SCENE_DATA,
 IDS_EXPORT_UTILITY_ROLLOUT_SCENE_DATA,
 0,0,
 NULL,
 MAP_FRAME_RANGE,
 IDD_EXPORT_UTILITY_ROLLOUT_FRAME_RANGE,
 IDS_EXPORT_UTILITY_ROLLOUT_FRAME_RANGE,
 0, APPENDROLL_CLOSED,
 NULL,
 MAP_FILTER_SETUP,
 IDD_EXPORT_UTILITY_ROLLOUT_FILTER_SETUP,
 IDS_EXPORT_UTILITY_ROLLOUT_FILTER_SETUP,
 0, APPENDROLL_CLOSED,
 &g_utilityFilterSetupDlgProc,
 MAP_ADVANCED,
 IDD_EXPORT_UTILITY_ROLLOUT_ADVANCED,
 IDS_EXPORT_UTILITY_ROLLOUT_ADVANCED,
 0, APPENDROLL_CLOSED,
 NULL,

 // SCENE DATA ROLLOUT

 PA_VisibleOnly, _T("visibleOnly"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_VISIBLE_ONLY,
 p_default, FALSE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_VISIBLE_ONLY,
 p_end,
 PA_SelectedOnly, _T("selectedOnly"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_SELECTED_ONLY,
 p_default, FALSE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_SELECTED_ONLY,
 p_end,
 PA_ExportMeshes, _T("exportMeshes"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_MESHES,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_MESHES,
 p_end,
 PA_ExportMaterials, _T("exportMaterials"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_MATERIALS,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_MATERIALS,
 p_enable_ctrls, 1, PA_DoNotSplitVertices,
 p_end,
 PA_ExportAttributes, _T("exportAttributes"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_ATTRIBUTES,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_ATTRIBUTES,
 p_end,
 PA_ExportAnnotations, _T("exportAnnotations"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_ANNOTATIONS,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_ANNOTATIONS,
 p_end,
 PA_ExportLights, _T("exportLights"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_LIGHTS,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_LIGHTS,
 p_end,
 PA_ExportCameras, _T("exportCameras"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_CAMERAS,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_CAMERAS,
 p_enable_ctrls, 1, PA_AddDefaultCamera,
 p_end,
 PA_AddDefaultCamera, _T("addDefaultCamera"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_ADD_DEFAULT_CAMERA,
 p_default, TRUE,
 p_ui,  MAP_SCENE_DATA,
 TYPE_SINGLECHEKBOX, IDC_CB_ADD_DEFAULT_CAMERA,
 p_end,

 // FRAME RANGE ROLLOUT

 PA_AnimatedDataExport, _T("animatedDataExport"), TYPE_INT, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_ANIMATED_DATA_EXPORT,
 p_default, ADE_CURRENT_RANGE,
 p_ui,  MAP_FRAME_RANGE,
 TYPE_RADIO, 3, IDC_RB_EXPORT_CURRENT_FRAME, IDC_RB_EXPORT_CURRENT_RANGE, IDC_RB_EXPORT_SPECIFIC_RANGE,
 p_vals, ADE_CURRENT_FRAME, ADE_CURRENT_RANGE, ADE_SPECIFIC_RANGE,
 p_end,
 PA_AnimationStart, _T("animationStart"), TYPE_INT, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_ANIMATION_START,
 p_default, 0,
 p_range, -100000000, 1000000000,
 p_ui,  MAP_FRAME_RANGE,
 TYPE_SPINNER, EDITTYPE_TIME, IDC_ED_ANIMATION_START, IDC_SP_ANIMATION_START, SPIN_AUTOSCALE,
 p_end,
 PA_AnimationEnd, _T("animationEnd"), TYPE_INT, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_ANIMATION_END,
 p_default, 16000,
 p_range, -100000000, 1000000000,
 p_ui,  MAP_FRAME_RANGE,
 TYPE_SPINNER, EDITTYPE_TIME, IDC_ED_ANIMATION_END, IDC_SP_ANIMATION_END, SPIN_AUTOSCALE,
 p_end,

 // FILTER SETUP ROLLOUT

 PA_UseOptionsFile, _T("useOptionsFile"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_USE_OPTIONS_FILE,
 p_default, FALSE,
 p_ui,  MAP_FILTER_SETUP,
 TYPE_SINGLECHEKBOX, IDC_CB_USE_OPTIONS_FILE,
 p_end,
 PA_OptionsFile, _T("optionsFile"), TYPE_STRING, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_OPTIONS_FILE,
 p_default, _T(""),
 p_ui,  MAP_FILTER_SETUP,
 TYPE_EDITBOX, IDC_CUSTOM_OPTIONS_FILE,
 p_end,
 PA_UseConfigurationSet, _T("useConfigurationSet"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_USE_CONFIGURATION_SET,
 p_default, FALSE,
 p_ui, MAP_FILTER_SETUP,
 TYPE_SINGLECHEKBOX, IDC_CB_USE_CONFIGURATION_SET,
 p_end,
 PA_ConfigurationSet, _T("configurationSet"), TYPE_STRING, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_CONFIGURATION_SET,
 p_default, _T(""),
 p_ui, MAP_FILTER_SETUP,
 TYPE_EDITBOX, IDC_ED_CONFIGURATION_SET,
 p_end,
 PA_EnvironmentVariables, _T("environmentVariables"), TYPE_STRING, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_ENVIRONMENT_VARIABLES,
 p_default, _T(""),
 p_ui,  MAP_FILTER_SETUP,
 TYPE_EDITBOX, IDC_ED_ENVIRONMENT_VARIABLES,
 p_end,

    // DO NOT SPLIT VERTICES ROLLOUT

 PA_DoNotSplitVertices, _T("doNotSplitVertices"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_DO_NOT_SPLIT_VERTICES,
 p_default, FALSE,
 p_ui,  MAP_ADVANCED,
 TYPE_SINGLECHEKBOX, IDC_CB_DO_NOT_SPLIT_VERTICES,
 p_end,

    // EXTRA OPTIONS ROLLOUT

    PA_RespectModifierSkinPose, _T("respectModifierSkinPose"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_RESPECT_MODIFIER_SKIN_POSE,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_RESPECT_MODIFIER_SKIN_POSE,
        p_end,

    PA_AutoSkinAttachments, _T("autoSkinAttachments"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_AUTO_SKIN_ATTACHMENTS,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_AUTO_SKIN_ATTACHMENTS,
        p_end,

    PA_ExportWireColorsAsVertexColors, _T("exportWireColorsAsVertexColors"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_WIRE_COLORS_AS_VERTEX_COLORS,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_WIRE_COLORS_AS_VERTEX_COLORS,
        p_end,

    PA_ExportMergeVertsOnlyIfSameSrcId, _T("exportMergeVertsOnlyIfSameSrcId"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_MERGE_VERTS_ONLY_IF_SAME_SRC_ID,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_MERGE_VERTS_ONLY_IF_SAME_SRC_ID,
        p_end,

    PA_StoreKeyframeSamplePoints, _T("storeKeyframeSamplePoints"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_KEYFRAME_SAMPLE_POINTS,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_KEYFRAME_SAMPLE_POINTS,
        p_end,

    PA_ExpandSelectionToSkeleton, _T("expandSelectionToSkeleton"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPAND_SELECTION_TO_SKELETON,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPAND_SELECTION_TO_SKELETON,
        p_end,

    PA_ExpandSelectionToParents, _T("expandSelectionToParents"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPAND_SELECTION_TO_PARENTS,
        p_default, TRUE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPAND_SELECTION_TO_PARENTS,
        p_end,

    PA_ExpandSelectionToChildren, _T("expandSelectionToChildren"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPAND_SELECTION_TO_CHILDREN,
        p_default, FALSE,
        p_ui,   MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPAND_SELECTION_TO_CHILDREN,
        p_end,

    PA_ExportTangents, _T("exportTangents"), TYPE_BOOL, P_RESET_DEFAULT, IDS_EXPORT_UTILITY_PA_EXPORT_TANGENTS,
        p_default, TRUE,
        p_ui, MAP_ADVANCED,
        TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_TANGENTS,
        p_end,

 p_end
 );


// Access the only instance of the utility. We create a global rather than local static
// variable here to cause registration of the pre- and post-merge callbacks on startup of Max, because
// registration immediately before a merge will not necessarily produce a pre-merge callback (EXP-1451).
static hctSceneExportUtility theInstance;

hctSceneExportUtility* getSceneExportUtilityInstance ()
{
    return &theInstance;
}

///////////////////////////////////////////////////////////////////////////////////////////
//
// Required hctSceneExportUtility functions, inherited from ReferenceMaker and UtilityObj.
//
///////////////////////////////////////////////////////////////////////////////////////////

IParamBlock2* hctSceneExportUtility::m_pbSceneExport = NULL;

int hctSceneExportUtility::NumRefs()
{
    return 1;
}

int hctSceneExportUtility::NumParamBlocks()
{
    return 1;
}

IParamBlock2* hctSceneExportUtility::GetParamBlock( int id )
{
    switch (id)
    {
    case PBLOCK_SceneExport:
        return m_pbSceneExport;

    default:
        HK_ASSERT( 0x25549adf, 0, "Unexpected ID in hctSceneExportUtility::GetParamBlock()" );
        return NULL;
    }
}

IParamBlock2* hctSceneExportUtility::GetParamBlockByID( BlockID id )
{
    switch (id)
    {
    case PB_SceneExport:
        return ( GetParamBlock( PBLOCK_SceneExport ) );

    default:
        HK_ASSERT( 0x592786a2, 0, "Unexpected ID in hctSceneExportUtility::GetParamBlockByID()" );
        return NULL;
    }
}

RefTargetHandle hctSceneExportUtility::GetReference( int i )
{
    switch (i)
    {
    case PBLOCK_SceneExport:
        return GetParamBlock( i );

    default:
        HK_ASSERT( 0x79243184, 0, "Unexpected ID in hctSceneExportUtility::GetReference()" );
        return NULL;
    }
}

void hctSceneExportUtility::SetReference( int i, RefTargetHandle rtarg )
{
    switch (i)
    {
    case PBLOCK_SceneExport:
        m_pbSceneExport = (IParamBlock2*) rtarg;
        break;
    default:
        HK_ASSERT( 0xedd3d352, 0, "Unexpected ID in hctSceneExportUtility::SetReference()" );
    }
}

#if MAX_VERSION_MAJOR>=17
    RefResult hctSceneExportUtility::NotifyRefChanged( const Interval& changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message, BOOL propagate )
#else
    RefResult hctSceneExportUtility::NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message )
#endif
{
    switch (message)
    {
    case REFMSG_CHANGE:
        pbdExportScene.InvalidateUI();
        updateUI();
        break;
    }
    return REF_SUCCEED;
}

void hctSceneExportUtility::BeginEditParams( Interface *ip, IUtil *iu )
{
    this->m_iutil = iu;
    this->m_interface = ip;

    // Add the rollups owned by param blocks.
    getHkSceneExportUtilityDesc()->BeginEditParams( (IObjParam *)ip, this, 0, this );

    updateUI();
}

void hctSceneExportUtility::EndEditParams( Interface *ip, IUtil *iu )
{
    // Clean up the rollups owned by param blocks.
    getHkSceneExportUtilityDesc()->EndEditParams( (IObjParam *)ip, this, END_EDIT_REMOVEUI, this );

    this->m_interface = NULL;
    this->m_iutil = NULL;
}

// EXP-1047 : Avoid overriding the utility data during scene merges
void hctSceneExportUtility::backupUtilityData ()
{
    m_utilityDataBackup.m_paramValues.SetCount(0);

    IParamBlock2* pblock2 = GetParamBlock(PBLOCK_SceneExport);

    for (int paramIndex=0; paramIndex<pblock2->NumParams(); paramIndex++)
    {
        PB2Value value;
        ParamID paramID = pblock2->IndextoID(paramIndex);
        switch ((int) pblock2->GetParameterType(paramID))
        {
        case TYPE_BOOL:
        case TYPE_INT:
            value.i = pblock2->GetInt(paramID);
            break;
        case TYPE_STRING:
            value.s = MAX_STRDUP(pblock2->GetStr(paramID));
            break;
        case TYPE_FLOAT:
            value.f = pblock2->GetFloat(paramID);
            break;
        default:
            continue;
        }

        m_utilityDataBackup.m_paramValues.Append(1, &value);
    }
    m_utilityDataBackup.m_filterSetOptions = m_filterSetOptions;
}

void hctSceneExportUtility::restoreUtilityData ()
{
    IParamBlock2* pblock2 = GetParamBlock(PBLOCK_SceneExport);

    if ( m_utilityDataBackup.m_paramValues.Count() != pblock2->NumParams() ) return;

    int curIndex=0;
    for (int paramIndex=0; paramIndex<pblock2->NumParams(); paramIndex++)
    {
        ParamID paramID = pblock2->IndextoID(paramIndex);
        switch ((int) pblock2->GetParameterType(paramID))
        {
        case TYPE_BOOL:
        case TYPE_INT:
            pblock2->SetValue(paramID, 0, m_utilityDataBackup.m_paramValues[curIndex++].i);
            break;
        case TYPE_STRING:
            pblock2->SetValue(paramID, 0, m_utilityDataBackup.m_paramValues[curIndex++].s);
            break;
        case TYPE_FLOAT:
            pblock2->SetValue(paramID, 0, m_utilityDataBackup.m_paramValues[curIndex++].f);
            break;
        }
    }

    assert(curIndex == m_utilityDataBackup.m_paramValues.Count());

    m_filterSetOptions = m_utilityDataBackup.m_filterSetOptions;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
