// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>

#include <ContentTools/Max/MaxSceneExport/Utilities/Destruction/hctDestructionUtility.h>
#include <ContentTools/Max/MaxFpInterfaces/Physics/DestructionUtility/hctDestructionUtilityInterface.h>

#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxSceneExport/Importer/MeshImport/hctMaxMeshImport.h>

#include <ContentTools/Max/MaxSceneExport/Utilities/Destruction/hctDestructionUtilityMessageLogDlg.h>
#include <ContentTools/Max/MaxSceneExport/resource.h>

// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>

#include <ContentTools/Common/DestructionUtils/hctDestructionUtils.h>

#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.3dsmax"
#include <Common/Base/System/Log/hkLog.hxx>


extern HINSTANCE hInstance;

hctMaxDestructionUtility* getDestructionUtilityInstance()
{
    static hctMaxDestructionUtility* theInstance = HK_NULL;
    if ( !theInstance )
    {
        theInstance = new hctMaxDestructionUtility();
    }
    return theInstance;
}

///////////////////////////////////////////////////////////////////////////////////////////
//
// The class descriptor
//
///////////////////////////////////////////////////////////////////////////////////////////

class hkDestructionUtilityClassDesc : public ClassDesc2
{
public:
    int             IsPublic() { return TRUE; }
    void*           Create( BOOL loading = FALSE ) { return getDestructionUtilityInstance(); }
    const MCHAR *   ClassName() { return GetString( IDS_DESTRUCTION_UTILITY_CLASS_NAME ); }
    SClass_ID       SuperClassID() { return UTILITY_CLASS_ID; }
    Class_ID        ClassID() { return HK_DESTRUCTION_UTILITY_CLASS_ID; }
    const MCHAR*    Category() { return GetString( IDS_DESTRUCTION_UTILITY_CATEGORY ); }
    const MCHAR*    InternalName() { return _T("hkDestructionUtility"); } // Returns fixed parsable name (scripter-visible name)
    HINSTANCE       HInstance() { return hInstance; }                    // Returns owning module handle
};


ClassDesc2* getHkDestructionUtilityDesc()
{
    static hkDestructionUtilityClassDesc maxDestructionUtilDesc;
    return &maxDestructionUtilDesc;
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// The dialog
//
///////////////////////////////////////////////////////////////////////////////////////////

class DestructionUtilityDlgProc : public ParamMap2UserDlgProc
{
    public:
        INT_PTR DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
        void DeleteThis() {}
};

INT_PTR DestructionUtilityDlgProc::DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    // NOTE: Buttons are handled by the action publishing interface, so no need to handle them here
    switch (msg)
    {
        case WM_LBUTTONDOWN:
        case WM_LBUTTONUP:
        case WM_MOUSEMOVE:
            getDestructionUtilityInstance()->m_interface->RollupMouseMessage( hWnd, msg, wParam, lParam );
            break;
        case WM_COMMAND:
            {
                //switch (LOWORD(wParam))
                //{
                ////case IDC_B_TEST:
                ////    {
                ////        getConvexHullUtilityInstance()->doTheTest();
                ////        break;
                ////    }

                //}
            }
            break;

        default:
            return FALSE;
    }
    return TRUE;
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// The param block descriptor
//
///////////////////////////////////////////////////////////////////////////////////////////

static DestructionUtilityDlgProc g_theDestructionUtilityDlgProc;

enum
{
    MAP_GENERAL_PROPERTIES_ROLLOUT
};

static ParamBlockDesc2 pbdDestructionUtil
(
    PB_DESTRUCTION_UTILITY_PBLOCK, _T("destruct"), 0, getHkDestructionUtilityDesc(),
    P_AUTO_CONSTRUCT + P_AUTO_UI + P_MULTIMAP, PB_DESTRUCTION_UTILITY_PBLOCK,

    // One rollout
    1,
    MAP_GENERAL_PROPERTIES_ROLLOUT,
        IDD_DESTRUCTION_UTILITY_ROLLOUT_PARAMS,
        IDS_DESTRUCTION_UTILITY_ROLLOUT_PARAMS,
        0, 0,
        &g_theDestructionUtilityDlgProc,

    // no parameters, just some buttons

        p_end,

    p_end
);


///////////////////////////////////////////////////////////////////////////////////////////
//
// The utility implementation
//
///////////////////////////////////////////////////////////////////////////////////////////

hctMaxDestructionUtility::hctMaxDestructionUtility()
{
    m_pblock = NULL;
    getHkDestructionUtilityDesc()->MakeAutoParamBlocks(this);
    m_logDialogWindow = 0;
}

hctMaxDestructionUtility::~hctMaxDestructionUtility()
{
    if (m_logDialogWindow)
    {
        DestroyWindow(m_logDialogWindow);
    }
}

#if MAX_VERSION_MAJOR>=17
    RefResult hctMaxDestructionUtility::NotifyRefChanged( const Interval& changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message, BOOL propagate )
#else
    RefResult hctMaxDestructionUtility::NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message )
#endif
{
    switch (message)
    {
    case REFMSG_CHANGE:
        pbdDestructionUtil.InvalidateUI();
        updateUI();
        break;
    }
    return REF_SUCCEED;
}

// Declare the callback function
static void _selectionChanged (void *param, NotifyInfo *info)
{
    getDestructionUtilityInstance()->updateUI();
}

void hctMaxDestructionUtility::BeginEditParams( Interface *ip, IUtil *iu )
{
    m_interface = ip;

    // Add the rollups owned by param blocks
    getHkDestructionUtilityDesc()->BeginEditParams( (IObjParam *)ip, this, 0, this );
    updateUI();

    RegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);
}

void hctMaxDestructionUtility::EndEditParams( Interface *ip, IUtil *iu )
{
    // Clean up the rollups owned by param blocks
    getHkDestructionUtilityDesc()->EndEditParams( (IObjParam *)ip, this, END_EDIT_REMOVEUI, this );

    m_interface = NULL;

    UnRegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);
}

bool hctMaxDestructionUtility::canDoDestruction()
{
    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();

    if( ip->GetSelNodeCount() == 0 )
    {
        return false;
    }

    // Need all selected nodes to have mesh to activate utility
    for( int i=0; i<ip->GetSelNodeCount(); ++i )
    {
        INode* node = ip->GetSelNode(i);

        Object *obj = node->EvalWorldState( now ).obj;
        if( !obj->CanConvertToType( Class_ID(TRIOBJ_CLASS_ID, 0) ) ) return false;
    }

    // XXX CK:TODO check keycode / destruction params etc

    return true;
}

bool hctMaxDestructionUtility::destructionMessageLog()
{
    if (!m_logDialogWindow)
    {
        m_logDialogWindow = CreateDialogParam(hInstance, MAKEINTRESOURCE(IDD_DESTRUCTION_MESSAGE_LOG),
            0, hctDestructionUtilityMessageLogProc, (LPARAM)this );
    }

    return (m_logDialogWindow ? true : false);
}

void hctMaxDestructionUtility::getMessageLogString(hkStringOld& logString) const
{
    for (int i=0; i<m_errorHandler.numLogEntries(); ++i)
    {
        const hkMaxSceneError::LogEntry& entry = m_errorHandler.getLogEntry(i);
        logString += entry.m_text + "\r\n";
    }
}

void hctMaxDestructionUtility::printMessages(HWND richEditControl) const
{
    for (int i = 0; i < m_errorHandler.numLogEntries(); i++)
    {
        const hkMaxSceneError::LogEntry& entry = m_errorHandler.getLogEntry(i);
        hctPrintLogEntryToRichEditControl(entry, richEditControl);
    }
}

bool hctMaxDestructionUtility::isWarningOrErrorInLog() const
{
    for (int i = 0; i < m_errorHandler.numLogEntries(); i++)
    {
        const hkMaxSceneError::LogEntry& entry = m_errorHandler.getLogEntry(i);
        if ( entry.m_type == hkMaxSceneError::MESSAGE_WARNING || entry.m_type == hkMaxSceneError::MESSAGE_ERROR )
        {
            return true;
        }
    }

    return false;
}

void hctMaxDestructionUtility::destructionMessageLogUpdated() const
{
    if (m_logDialogWindow)
    {
        SendNotifyMessage( m_logDialogWindow, WM_COMMAND, IDC_UPDATE, 0 );
    }
}

bool hctMaxDestructionUtility::destruct( const Tab<INode*>& selectedNodes, Tab<INode*>& allNodes, Tab<INode*>& newNodes, hctMaxDestructionUtility::RunOptions runOptions)
{
    HCT_SCOPED_CONVERSIONS;

    Interface* ip = GetCOREInterface();
    //TimeValue now = ip->GetTime();

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    ip->PushPrompt( _T("Fracturing nodes..") );

    bool retStat = false;

    {
        // call the utility
        MaxProgressUpdater progressUpdater(_T("Loading Destruction Util DLL.."));

        m_errorHandler.clearLog();

        hkError* origError = &hkError::getInstance();
        origError->addReference();
        hkError::replaceInstanceAndAddReference( &m_errorHandler );

        //->  find the destruction util dll
        //    fail if not found
        HMODULE destructionDll = HK_NULL;
        hctBaseDll* destUtilsDll = HK_NULL;
        hctDestructionUtils* destUtils = HK_NULL;
        hkStringOld destructionUtilDll;
        {
            hkStringOld filterManagerPath;
            hctMaxUtils::getFilterManagerPath( filterManagerPath );
            destructionUtilDll = filterManagerPath + "\\utils\\hctDestructionUtils.dll";

            hkIfstream testStream( destructionUtilDll.cString() );
            if (!testStream.isOk())
            {
                HK_ERROR(0x7b5ac0db,"Destruction Util DLL not found in filter manager's util dir");
                goto QUIT_CLEANLY;
            }

            destructionDll = LoadLibraryA( destructionUtilDll.cString() );
            if (!destructionDll)
            {
                HK_ERROR(0x299d37e4,"Destruction Util DLL found but could not load. The DLL is corrupt.");
                goto QUIT_CLEANLY;
            }

            hctGetBaseDllInterfaceFunc destBaseDllFunc = ( hctGetBaseDllInterfaceFunc ) GetProcAddress(destructionDll, "getBaseDllInterface");
            if (!destBaseDllFunc)
            {
                HK_ERROR(0x6fdba42b,"Destruction DLL found but could not find baseDLL sync function. Install is invalid.");
                goto QUIT_CLEANLY;
            }

            progressUpdater.progression(0, _T("Waiting for init to finish caching filters..") );

            MSTR waitCmd = _T("hkSceneExportUtility.waitForFilterLoad()");
            if ( !ExecuteMAXScriptScript( waitCmd ) )
            {
                HK_ERROR(0x597e229f,"Could not wait on background filter load, Destruction Util DLL statics may get corrupted.");
            }

            if (progressUpdater.didUserCancel())
            {
                HK_ERROR(0x1a5c44d7,"Destruction did not complete, user canceled op.");
                goto QUIT_CLEANLY;
            }

            progressUpdater.progression(1, _T("Initializing Destruction DLL..") );

            destUtilsDll = destBaseDllFunc();
            if (destUtilsDll)
            {
                if ( destUtilsDll->getDllVersion() != HCT_CURRENT_VERSION )
                {
                    HK_ERROR(0x7f425fa0, "Could not load Destruction Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall." );
                    goto QUIT_CLEANLY;
                }

                hkMemoryInitUtil::SyncInfo baseSystemInfo;
                hkSceneExportMemory::getBaseSystemSyncInfo( baseSystemInfo );
                destUtilsDll->initDll( baseSystemInfo, HK_NULL  );

                hctGetDestructionUtilInterfaceFunc getUtilInterface = ( hctGetDestructionUtilInterfaceFunc ) GetProcAddress(destructionDll, "getDestructionUtilInterface");
                destUtils = getUtilInterface? getUtilInterface() : HK_NULL;
            }

            if (!destUtils || !destUtilsDll )
            {
                hctGetDestructionDllErrorFunc destDllErrorFunc = ( hctGetDestructionDllErrorFunc ) GetProcAddress(destructionDll, "getDestructionDllError");
                if (!destDllErrorFunc)
                {
                    HK_ERROR(0x34e5aa70, "Could not query Destruction Util DLL. Maybe your install is corrupt." );
                    goto QUIT_CLEANLY;
                }

                hctDestructionUtils::DllError destError = destDllErrorFunc();

                if (destError & hctDestructionUtils::DLL_KEYCODE_ILLEGAL)
                {
                    HK_ERROR(0x5a9a1283, "The keycode for the Destruction Util DLL is invalid or is for a different product version. Please contact Havok support at http://support.havok.com" );
                    goto QUIT_CLEANLY;
                }

                if (destError & hctDestructionUtils::DLL_KEYCODE_EXPIRED)
                {
                    HK_ERROR(0x7441b3ff, "The keycode for the Destruction Util DLL is expired or is for a different product version. Please contact Havok support at http://support.havok.com" );
                    goto QUIT_CLEANLY;
                }

                if (destError & hctDestructionUtils::DLL_INIT_ERROR)
                {
                    HK_ERROR(0x146fc145, "The Destruction Util DLL failed to initialize properly. Please try restarting Maya. If this error repeats, please contact Havok support at http://support.havok.com" );
                    goto QUIT_CLEANLY;
                }

                HK_ERROR(0x21270eed, "The Destruction Util DLL failed to load. Maybe your install is corrupt. If this error repeats, please contact Havok support at http://support.havok.com" );
                goto QUIT_CLEANLY;
            }

            // All good:

            hctGetDestructionDllErrorFunc destDllErrorFunc = ( hctGetDestructionDllErrorFunc ) GetProcAddress(destructionDll, "getDestructionDllError");
            if (!destDllErrorFunc)
            {
                HK_ERROR(0x7b16680f, "Could not query Destruction Util DLL. Maybe your install is corrupt." );
                goto QUIT_CLEANLY;
            }

            hctDestructionUtils::DllError destError = destDllErrorFunc();

            if (destError & hctDestructionUtils::DLL_KEYCODE_WARNING)
            {
                progressUpdater.progression(1, _T("Destruction keycode expires soon") );
                HK_WARN_ALWAYS(0xabba6c1b, "The Destruction keycode expires in 10 days or less." );
                Sleep(3000);
            }

            destUtils->setFilterManagerPath(filterManagerPath.cString());

            if (progressUpdater.didUserCancel())
            {
                HK_ERROR(0x59a2cea, "Destruction did not complete, user canceled op." );
                goto QUIT_CLEANLY;
            }
        }

        //->  do a quick export, single frame, to get the current hkxScene to use
        {
            hkBool wasEnabled0xabba0d39 = hkError::getInstance().isEnabled(0xabba0d39);
            hkBool wasEnabled0xabba1441 = hkError::getInstance().isEnabled(0xabba1441);
            hkError::getInstance().setEnabled(0xabba0d39, false);
            hkError::getInstance().setEnabled(0xabba1441, false);

            hctMaxSceneExportOptions options;
            options.m_batchMode = true;
            options.m_doNotSplitVertices = false;
            options.m_visibleOnly = true;

            hctFilterSetOptions filterSetOptions;
            filterSetOptions.m_data.SetCount(0);
            filterSetOptions.m_version = 0;
            hctMaxSceneExporter exporter( options, filterSetOptions ); // sets base sys error handler to Max's

            bool showConfigDialog = (runOptions == hctMaxDestructionUtility::DEFAULT_RUN);

            progressUpdater.progression(2, _T("Creating HKX scene..") );

            bool haveScene = exporter.createScene(false) || showConfigDialog;
            if (progressUpdater.didUserCancel())
            {
                HK_ERROR(0x5a68d518, "Destruction did not complete, user canceled op." );
            }

            hkRootLevelContainer* sceneData = exporter.getCurrentRootContainer();
            if (!haveScene || (sceneData == HK_NULL))
            {
                HK_WARN_ALWAYS(0xabba4534, "No scene data exported... so nothing to fracture using Destruction.");
                exporter.cleanupScene();
                goto QUIT_CLEANLY;
            }

            progressUpdater.progression(3, _T("Bringing up Destruction Util..") );

            destUtils->registerThreadCallback(hctFilterProcessingUtil::getDefaultThreadCallback());

            //->  give the scene and 'object to break' to the destruction utils
            hkArray<hkxNode*> selectedHavokNodes;

            if ( !showConfigDialog )
            {
                hkxScene* sceneGraph = sceneData->findObject<hkxScene>();
                for (int sni=0; sni < selectedNodes.Count(); ++sni)
                {
                    HCT_SCOPED_CONVERSIONS;

                    MSTR nodeName = selectedNodes[sni]->NodeName();
                    const char* selNodeName = FROM_MAX(nodeName);

                    hkxNode* selObject = sceneGraph->findNodeByName(selNodeName);
                    if (selObject==HK_NULL)
                    {
                        //XX todo: try to find the parent
                        //if ( selObject == HK_NULL )
                        {
                            HK_WARN_ALWAYS(0xabba5645, "Could not find selected node [" << selNodeName << "], was it exported ok?" );
                        }
                    }

                    if (selObject)
                    {
                        selectedHavokNodes.pushBack(selObject);
                    }
                }
            }

            if ( (selectedHavokNodes.getSize() < 1) && !showConfigDialog )
            {
                exporter.cleanupScene();
                HK_WARN_ALWAYS(0xabba5646, "No object selected. Nothing to fracture." );
                goto QUIT_CLEANLY;
            }

            //
            // Extract the conversion factor for converting [m] to the current physics distance unit (which can differ from [m]).
            // Necessary for some hardcoded distance checks/asserts/warnings (e.g. default convex radius).
            //
            float distanceConversionFactor = 1.0f;
            {
                int systemUnitType;
                float systemConversionScale;
                GetMasterUnitInfo(&systemUnitType, &systemConversionScale);
                float meterConversionScale = GetMasterScale(UNITS_METERS);
                distanceConversionFactor = systemConversionScale / meterConversionScale;
            }
            destUtils->setDistanceConversionFactor(distanceConversionFactor);

            hkArray< hctDestructionUtils::BodyInfo *> brokenBits;
            hctDestructionUtils::RunOptions runStyle = runOptions == hctMaxDestructionUtility::PREVIEW_RUN ?
                                                hctDestructionUtils::PREVIEW_RUN :
                                                ( runOptions == hctMaxDestructionUtility::QUICK_RETURN_RUN ? hctDestructionUtils::QUICK_RETURN_RUN :
                                                    hctDestructionUtils::DEFAULT_RUN );

            hkBool wasEnabled0xabba6578 = hkError::getInstance().isEnabled(0xabba6578);
            hkBool wasEnabled0xabba6576 = hkError::getInstance().isEnabled(0xabba6576);
            hkBool wasEnabled0x34df5494 = hkError::getInstance().isEnabled(0x34df5494);
            hkBool wasEnabled0xf3768206 = hkError::getInstance().isEnabled(0xf3768206);
            hkBool wasEnabled0xabba4566 = hkError::getInstance().isEnabled(0xabba4566);
            hkBool wasEnabled0xabba1ee2 = hkError::getInstance().isEnabled(0xabba1ee2);
            hkBool wasEnabled0xabba1ee3 = hkError::getInstance().isEnabled(0xabba1ee3);
            hkBool wasEnabled0xf0325456 = hkError::getInstance().isEnabled(0xf0325456);
            hkBool wasEnabled0xabba1095 = hkError::getInstance().isEnabled(0xabba1095);
            hkBool wasEnabled0xabba1096 = hkError::getInstance().isEnabled(0xabba1096);
            hkBool wasEnabled0xabba1097 = hkError::getInstance().isEnabled(0xabba1097);
            hkBool wasEnabled0xf022dedf = hkError::getInstance().isEnabled(0xf022dedf);
            hkBool wasEnabled0xabba71e0 = hkError::getInstance().isEnabled(0xabba71e0);

            hkError::getInstance().setEnabled(0xabba6576, false);
            hkError::getInstance().setEnabled(0xabba6578, false);
            hkError::getInstance().setEnabled(0x34df5494, false);
            hkError::getInstance().setEnabled(0xf3768206, false);
            hkError::getInstance().setEnabled(0xabba4566, false);
            hkError::getInstance().setEnabled(0xabba1ee2, false);
            hkError::getInstance().setEnabled(0xabba1ee3, false);
            hkError::getInstance().setEnabled(0xf0325456, false);
            hkError::getInstance().setEnabled(0xabba1095, false);
            hkError::getInstance().setEnabled(0xabba1096, false);
            hkError::getInstance().setEnabled(0xabba1097, false);
            hkError::getInstance().setEnabled(0xf022dedf, false);
            hkError::getInstance().setEnabled(0xabba71e0, false);

            destUtils->breakObjects(sceneData, selectedHavokNodes.begin(), selectedHavokNodes.getSize(), brokenBits, (void*)ip->GetMAXHWnd(), runStyle);

            // Merge the two separate error handlers into one.
            {
                hctSceneExportError* filterManagerErrorHandler = destUtils->getFilterUtil()->getFilterManagerInterface()->getErrorHandler();
                for (int i = filterManagerErrorHandler->numLogEntries()-1; i >= 0; i--)
                {
                    const hctSceneExportError::LogEntry& logEntry = filterManagerErrorHandler->getLogEntry(i);
                    if (logEntry.m_type != hkError::MESSAGE_REPORT &&
                        logEntry.m_id != 0xabba10b0 &&
                        logEntry.m_id != 0xabbabfd2 &&
                        logEntry.m_id != 0xabbabfd4 &&
                        logEntry.m_id != 0xabbabfd6 &&
                        logEntry.m_id != 0xabbabad4 &&
                        logEntry.m_id != 0xabbabfd1)
                    {
                        m_errorHandler.mergeLogEntry(logEntry);
                    }
                }
            }

            hkError::getInstance().setEnabled(0xabba71e0, wasEnabled0xabba71e0);
            hkError::getInstance().setEnabled(0xf022dedf, wasEnabled0xf022dedf);
            hkError::getInstance().setEnabled(0xabba1097, wasEnabled0xabba1097);
            hkError::getInstance().setEnabled(0xabba1096, wasEnabled0xabba1096);
            hkError::getInstance().setEnabled(0xabba1095, wasEnabled0xabba1095);
            hkError::getInstance().setEnabled(0xabba6576, wasEnabled0xabba6576);
            hkError::getInstance().setEnabled(0xabba6578, wasEnabled0xabba6578);
            hkError::getInstance().setEnabled(0x34df5494, wasEnabled0x34df5494);
            hkError::getInstance().setEnabled(0xf3768206, wasEnabled0xf3768206);
            hkError::getInstance().setEnabled(0xabba4566, wasEnabled0xabba4566);
            hkError::getInstance().setEnabled(0xabba0d39, wasEnabled0xabba0d39);
            hkError::getInstance().setEnabled(0xabba1441, wasEnabled0xabba1441);
            hkError::getInstance().setEnabled(0xabba1ee2, wasEnabled0xabba1ee2);
            hkError::getInstance().setEnabled(0xabba1ee3, wasEnabled0xabba1ee3);
            hkError::getInstance().setEnabled(0xf0325456, wasEnabled0xf0325456);

            if (progressUpdater.didUserCancel())
            {
                HK_ERROR(0x38cf10b0, "Destruction did not complete, user canceled op." );
            }
            else if (brokenBits.getSize() > 0)
            {
                //->  take the results of the destruction and add them back to the scene
                //    may be no results, as util may iterate through previews etc, and/or cancel
                int numSuccess = 0;
                int numFail = 0;
                int numPartialFail = 0;
                int physicsShapeCtr = 0;

                hkArray<INode*> createdNodes;
                createdNodes.reserve(brokenBits.getSize());

                for (int bbi = 0; bbi < brokenBits.getSize(); ++bbi)
                {
                    hctDestructionUtils::BodyInfo& newPiece = *(brokenBits[bbi]);

                    hkxNode* origNode = newPiece.getOriginalNode();
                    hkMeshBody* graphicsBody = newPiece.getGraphicsBody();
                    hkArray<hkMeshBody*> childPhysicsShapes;
                    newPiece.getPhysicsShapes( childPhysicsShapes );

                    // We will abort if either there is no original node to be found or if we have neither
                    // a graphics shape nor at least one physics shape for this node. Having no graphics
                    // shape but 1+ physics shapes is a valid situation produced by e.g. the ConvexDecompose
                    // fracture algorithm.
                    if ( !origNode || (!graphicsBody && childPhysicsShapes.getSize() == 0))
                    {
                        if ( origNode )
                        {
                            Log_Warning( "Newly created child #{} of node {} has neither graphics nor physics and will be ignored.", bbi, origNode->m_name );
                        }
                        ++numFail;
                        createdNodes.pushBack(HK_NULL);
                        continue;
                    }

                    // find Max parent node
                    INode* origParentMaxNode = HK_NULL;
                    int pieceParentIndex = newPiece.getParentIndex();
                    if ((pieceParentIndex >= 0) && (createdNodes[pieceParentIndex] != HK_NULL))
                    {
                        origParentMaxNode = createdNodes[pieceParentIndex];
                    }
                    else
                    {
                        for (int nni=0; nni < selectedNodes.Count(); ++nni)
                        {
                            HCT_SCOPED_CONVERSIONS;

                            MSTR nodeNameStr = selectedNodes[nni]->NodeName();
                            const char* nodeName = FROM_MAX(nodeNameStr);

                            if (hkString::strCmp( nodeName, origNode->m_name ) == 0 )
                            {
                                origParentMaxNode = selectedNodes[nni];
                                break;
                            }
                        }
                    }

                    if ( graphicsBody )
                    {
                        //const char* childMeshName = childPhysicsShapes.getSize() ? "graphicsMesh" : "mesh";

                        INode* newNode = hctMaxSceneConvertUtilities::createNodeFromMesh( *graphicsBody, origParentMaxNode, allNodes, graphicsBody->getName() );

                        createdNodes.pushBack(newNode);

                        if (newNode == HK_NULL)
                        {
                            HK_WARN_ALWAYS(0xabba6712, "Could not create mesh node for " << graphicsBody->getName() );
                            ++numFail;
                        }
                        else
                        {
                            newNodes.Append(1, &newNode);

                            // create RB and BB data for new bits
                            // can just call the scripts

                            // Physics Shape Node(s)
                            int numChildren = childPhysicsShapes.getSize();
                            if (numChildren == 0)
                            {
                                MSTR createShapeCmd = _T("hvkPhysics_createShape $'");
                                createShapeCmd += newNode->NodeName();
                                createShapeCmd += _T("'");
                                if ( !ExecuteMAXScriptScript( createShapeCmd ) )
                                {
                                    HK_WARN_ALWAYS(0xabba5423, "Could not set new shape node for " << FROM_MAX(newNode->NodeName())  );
                                    ++numPartialFail;
                                }
                                else
                                {
                                    ++numSuccess;
                                }
                            }
                            else // the physics != graphics for the broken pieces
                            {
                                for (int csi=0; csi < numChildren; ++csi)
                                {
                                    hkMeshBody* childMesh = childPhysicsShapes[csi];
                                    if (childMesh)
                                    {
                                        INode* newChildNode = hctMaxSceneConvertUtilities::createNodeFromMesh( *childMesh, newNode, allNodes, "physics");
                                        if (newChildNode != HK_NULL)
                                        {
                                            MSTR createShapeChildCmd = _T("hvkPhysics_createShape $'");
                                            createShapeChildCmd += newChildNode->NodeName();
                                            createShapeChildCmd += _T("'");
                                            if ( !ExecuteMAXScriptScript( createShapeChildCmd ) )
                                            {
                                                ++numPartialFail;
                                                HK_WARN_ALWAYS(0xabbaf861, "Could not set new shape node for " << FROM_MAX(newChildNode->NodeName()) << " under node " << FROM_MAX(newNode->NodeName())  );
                                            }
                                            else
                                            {
                                                ++numSuccess;
                                            }
                                        }
                                        else
                                        {
                                            ++numPartialFail;
                                            HK_WARN_ALWAYS(0xabba2e9a, "Could not create child mesh [" << (childMesh->getName()? childMesh->getName() : "noname") << "] for node " <<  FROM_MAX(newNode->NodeName()) );
                                        }

                                        ++numSuccess;
                                    }
                                }
                            }

                            // Destruction info
                            // The parent tree has the breakable body in it, we just need to add shapes for each new bit
                            {

                                MSTR createhkdShapeCmd = TEXT("hvkDestruction_createGenericModifier $'");
                                createhkdShapeCmd += newNode->NodeName();
                                createhkdShapeCmd += TEXT("'\thkdShape");

                                if ( !ExecuteMAXScriptScript( createhkdShapeCmd ) )
                                {
                                    ++numPartialFail;
                                    HK_WARN_ALWAYS(0xabba55f4, "Could not set new Destruction Shape for " << FROM_MAX(newNode->NodeName()) );
                                }
                            }

                            // Set the piece fixed if needed
                            if ( newPiece.getQualityType() == hctDestructionUtils::QUALITY_FIXED )
                            {
                                MSTR createShapeCmd = _T("hvkDestruction_setModifierAttValue $'");
                                createShapeCmd += newNode->NodeName();
                                createShapeCmd += _T("' \"hkdShape\" \"bodyQualityType\" ");
                                createShapeCmd.printf(_T("%s%d"), createShapeCmd.data(), (hctDestructionUtils::QUALITY_FIXED + 1));

                                if ( !ExecuteMAXScriptScript( createShapeCmd ) )
                                {
                                    HK_WARN_ALWAYS(0xabba5423, "Could set fixed property on " << FROM_MAX(newNode->NodeName())  );
                                    ++numPartialFail;
                                }
                                else
                                {
                                    ++numSuccess;
                                }
                            }
                        }
                    }
                    else // !graphicsBody
                    {
                        //
                        // Special case for "Convex Decompose Physics" utility:
                        // No graphics body but at least one physics shape.
                        //

                        char newChildNameBuffer[1024];

                        {
                            for (int physicsShapeIndex = 0; physicsShapeIndex < childPhysicsShapes.getSize(); physicsShapeIndex++)
                            {
                                hkMeshBody* physicsShape = childPhysicsShapes[physicsShapeIndex];

                                hkString::snprintf( newChildNameBuffer, 1000, "%s_Physics%i", FROM_MAX(origParentMaxNode->NodeName()), physicsShapeCtr++ );
                                INode* newNode = hctMaxSceneConvertUtilities::createNodeFromMesh( *physicsShape, origParentMaxNode, allNodes, newChildNameBuffer );

                                createdNodes.pushBack(newNode);

                                if ( newNode == HK_NULL )
                                {
                                    HK_WARN_ALWAYS(0xabba6713, "Could not create physics shape node for " << physicsShape->getName() );
                                    ++numFail;
                                }
                                else
                                {
                                    newNodes.Append(1, &newNode);

                                    //
                                    // Add the "Havok Shape" modifier to the new imported physics node.
                                    //
                                    {
                                        MSTR createShapeCmd = _T("hvkPhysics_createShape $'");
                                        createShapeCmd += newNode->NodeName();
                                        createShapeCmd += _T("'");
                                        if ( !ExecuteMAXScriptScript( createShapeCmd ) )
                                        {
                                            HK_WARN_ALWAYS(0xabba5423, "Could not set new shape node for " << FROM_MAX(newNode->NodeName()) );
                                            ++numPartialFail;
                                        }
                                        else
                                        {
                                            ++numSuccess;
                                        }
                                    }

                                    //
                                    // Remove (if available) the old "Havok Shape" modifier from the original object.
                                    //
                                    {
                                        MSTR createShapeCmd = _T("hvkDestruction_removeShapeModifier $'");
                                        createShapeCmd += origParentMaxNode->NodeName();
                                        createShapeCmd += _T("'");
                                        ExecuteMAXScriptScript( createShapeCmd );
                                    }
                                }
                            }
                        }
                    }

                    // cleanup
                    delete brokenBits[bbi];

                }// all new pieces

                if (numSuccess > 0)
                {
                    retStat = true; // Success, or at least partially so.

                    if ( numFail > 0 )
                    {
                        Log_Warning( "In-Modeler Fracture created a total of {} new Fracture Pieces, but had some failures.", numSuccess );
                    }
                    else if ( numPartialFail )
                    {
                        Log_Warning( "In-Modeler Fracture created a total of {} new Fracture Pieces, but had some partial failures.", numSuccess );
                    }
                    else
                    {
                        Log_Info( "In-Modeler Fracture created a total of {} new Fracture Pieces.", numSuccess );
                    }
                }
                else
                {
                    Log_Error( "In-Modeler Fracture did not complete properly. No Fracture Pieces created." );
                }
            }
            else if (showConfigDialog)
            {
                retStat = true;
                Log_Info( "Destruction Util options success." );
            }
            else
            {
                Log_Warning( "No Fracture Pieces created." );
            }

            //-> Do some cleanup
            exporter.cleanupScene();
        }

QUIT_CLEANLY:

        if ( !retStat || isWarningOrErrorInLog() )
        {
            //Log_Error( "In-Modeler Fracture failed." );

            // force showing the message log window
            destructionMessageLog();
        }

        if (destUtils)
        {
            delete destUtils;
        }

        // cleanup
        if (destUtilsDll)
        {
            destUtilsDll->quitDll();
        }

        hkError::replaceInstance( origError ); // takes ref

        if ( destructionDll )
        {
            FreeLibrary( destructionDll );
        }
    }

    destructionMessageLogUpdated();

    ip->PopPrompt();
    SetCursor(LoadCursor(NULL, IDC_ARROW));
    ip->RedrawViews( ip->GetTime() );

    return retStat;
}

static void _deleteAllNodes( Interface* ip, INode* root )
{
    if ( !root ) return;

    for (int c = root->NumberOfChildren()-1; c >= 0; --c )
    {
        _deleteAllNodes( ip, root->GetChildNode(c) );
    }

    ip->DeleteNode(root, FALSE, TRUE );

}
bool hctMaxDestructionUtility::deleteCreatedChildren( Tab<INode*>& selectedNodes )
{
    Interface* ip = GetCOREInterface();
    MSTR destrcutionChildPatern(TEXT("*__C*"));
    bool deletedSomeNodes = false;

    theHold.Begin();
    {
        for (int si=0; si < selectedNodes.Count(); ++si)
        {
            INode* parent = selectedNodes[si];
            if (parent)
            {
                for (int j = parent->NumberOfChildren() - 1; j >=0 ; --j )
                {
                    INode* child = parent->GetChildNode(j);
                    if (child)
                    {
                        MSTR name = child->GetName();
                        if (MatchPattern(name, destrcutionChildPatern, FALSE))
                        {
                            _deleteAllNodes( ip, child );
                            deletedSomeNodes = true;
                        }
                    }
                }
            }
        }
    }
    theHold.Accept( _T("Delete Destruction Child Nodes") );

    if (deletedSomeNodes)
    {
        ip->RedrawViews( ip->GetTime() );
    }

    return deletedSomeNodes;
}


void hctMaxDestructionUtility::updateUI()
{
    if( !m_interface )
    {
        return;
    }

    getHkDestructionUtilityDesc()->InvalidateUI();
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
