// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Max/MaxSceneExport/Utilities/ConvexHull/hctConvexHullUtility.h>

#include <ContentTools/Max/MaxFpInterfaces/Physics/ConvexHullUtility/hctConvexHullUtilityInterface.h>


// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SdkUtils/Geometry/hctCreateConvexHullUtil.h>


namespace hctConvexHullUtilityUiInternals
{
    enum
    {
        MAP_GENERAL_PROPERTIES_ROLLOUT
    };

    ///////////////////////////////////////////////////////////////////////////////////////////
    //
    // The class descriptor
    //
    ///////////////////////////////////////////////////////////////////////////////////////////
    class _ClassDesc2 : public ClassDesc2
    {
    public:
        int             IsPublic() { return TRUE; }
        void*           Create( BOOL loading = FALSE ) { return getConvexHullUtilityInstance(); }
        const MCHAR *   ClassName() { return GetString( IDS_CONVEX_HULL_UTILITY_CLASS_NAME ); }
        SClass_ID       SuperClassID() { return UTILITY_CLASS_ID; }
        Class_ID        ClassID() { return HK_CONVEX_HULL_UTILITY_CLASS_ID; }
        const MCHAR*    Category() { return GetString( IDS_CONVEX_HULL_UTILITY_CATEGORY ); }
        const MCHAR*    InternalName() { return _T("hkConvexHullUtility"); } // Returns fixed parsable name (scripter-visible name)
        HINSTANCE       HInstance() { return hInstance; }                    // Returns owning module handle
};

    ///////////////////////////////////////////////////////////////////////////////////////////
    //
    // The dialog
    //
    ///////////////////////////////////////////////////////////////////////////////////////////
    class _DlgProc : public ParamMap2UserDlgProc
    {
        public:

            INT_PTR DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
            {
                // NOTE: Buttons are handled by the action publishing interface, so no need to handle them here
    switch (msg)
                {
                    case WM_LBUTTONDOWN:
                    case WM_LBUTTONUP:
                    case WM_MOUSEMOVE:
            getConvexHullUtilityInstance()->m_interface->RollupMouseMessage( hWnd, msg, wParam, lParam );
                        break;
                    case WM_COMMAND:
                        {
                            //switch (LOWORD(wParam))
                            //{
                            ////case IDC_B_TEST:
                            ////    {
                            ////        getConvexHullUtilityInstance()->doTheTest();
                            ////        break;
                            ////    }

                            //}
                        }
                        break;

                    default:
                        return FALSE;
                }
                return TRUE;
            }

            void DeleteThis() {}
    };
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// The utility implementation
//
///////////////////////////////////////////////////////////////////////////////////////////

static hctConvexHullUtility theInstance;

hctConvexHullUtility* getConvexHullUtilityInstance()
{
    return &theInstance;
}

ClassDesc2* getHkConvexHullUtilityDesc()
{
    return getConvexHullUtilityInstance()->m_classDesc;
}


hctConvexHullUtility::hctConvexHullUtility()
{
    m_classDesc = new hctConvexHullUtilityUiInternals::_ClassDesc2();

    m_dlgProc = new hctConvexHullUtilityUiInternals::_DlgProc();

    // Create the UI.
    m_pBlock2 = new ParamBlockDesc2
    (
        PB_CONVEX_HULL_UTILITY_PBLOCK, _T("createConvexHull"), 0, m_classDesc,
        P_AUTO_CONSTRUCT + P_AUTO_UI + P_MULTIMAP, PB_CONVEX_HULL_UTILITY_PBLOCK,

        // One rollout
        1,
        hctConvexHullUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
        IDD_CONVEX_HULL_UTILITY_ROLLOUT_PARAMS,
        IDS_CONVEX_HULL_UTILITY_ROLLOUT_PARAMS,
        0, 0,
            m_dlgProc,

        // parameters
        PA_CONVEX_HULL_UTILITY_MAXVERTS,
            _T("maxVerts"), TYPE_INT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CONVEX_HULL_UTILITY_PA_MAXVERTS,
            p_default,      64,
            p_range,        4, 10000,
            p_ui,           hctConvexHullUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_SPINNER, EDITTYPE_INT, IDC_ED_HULL_MAXVERTS, IDC_SP_HULL_MAXVERTS,
                            SPIN_AUTOSCALE,
            p_end,

        PA_CONVEX_HULL_UTILITY_ENCLOSE_INPUTS,
            _T("encloseInputs"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CONVEX_HULL_UTILITY_PA_ENCLOSE_INPUTS,
            p_default,      TRUE,
            p_ui,           hctConvexHullUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_HULL_CB_ENCLOSE_INPUTS,
            p_end,

        PA_CONVEX_HULL_UTILITY_MIN_WEIGHT,
            _T("minWeight"), TYPE_FLOAT, P_ANIMATABLE |P_RESET_DEFAULT, IDS_CONVEX_HULL_UTILITY_PA_MIN_WEIGHT,
            p_default,      0.1f,
            p_range,            0.0f,1.0f,
//          p_ui,           hctConvexHullUtilityInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
//                          TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_MIN_WEIGHT, IDC_SP_MIN_WEIGHT, SPIN_AUTOSCALE,
            p_end,

        p_end
    );

    m_interface = NULL;
    m_pblock = NULL;
    m_classDesc->MakeAutoParamBlocks(this);
}

hctConvexHullUtility::~hctConvexHullUtility()
{
    delete m_pBlock2;
    delete m_dlgProc;
    delete m_classDesc;
}

#if MAX_VERSION_MAJOR>=17
RefResult hctConvexHullUtility::NotifyRefChanged( const Interval& changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message, BOOL propagate )
#else
RefResult hctConvexHullUtility::NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message )
#endif
{
    switch (message)
    {
    case REFMSG_CHANGE:
        m_pBlock2->InvalidateUI();
        updateUI();
        break;
    }
    return REF_SUCCEED;
}

// Declare the callback function
static void _selectionChanged (void *param, NotifyInfo *info)
{
    getConvexHullUtilityInstance()->updateUI();
}

void hctConvexHullUtility::BeginEditParams( Interface *ip, IUtil *iu )
{
    m_interface = ip;

    // Add the rollups owned by param blocks
    m_classDesc->BeginEditParams( (IObjParam *)ip, this, 0, this );
    updateUI();

    RegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);
}

void hctConvexHullUtility::EndEditParams( Interface *ip, IUtil *iu )
{
    // Clean up the rollups owned by param blocks
    m_classDesc->EndEditParams( (IObjParam *)ip, this, END_EDIT_REMOVEUI, this );

    m_interface = NULL;

    UnRegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);
}

bool hctConvexHullUtility::canCreateHulls()
{
    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();

    if( ip->GetSelNodeCount() == 0 )
    {
        return false;
    }

    // Need all selected nodes to have mesh to activate utility
    for( int i=0; i<ip->GetSelNodeCount(); ++i )
    {
        INode* node = ip->GetSelNode(i);

        Object *obj = node->EvalWorldState( now ).obj;
        if( !obj->CanConvertToType( Class_ID(TRIOBJ_CLASS_ID, 0) ) ) return false;
    }

    return true;
}

bool hctConvexHullUtility::canCreateSkinHulls()
{
    Interface* ip = GetCOREInterface();

    // Need all selected nodes to have mesh to activate utility
    for( int i=0; i<ip->GetSelNodeCount(); ++i )
    {
        INode* boneNode = ip->GetSelNode(i);

        DependentIterator di(boneNode);

        ReferenceMaker *rm;

        while ((rm=di.Next())!=NULL)
        {
            Class_ID klass = rm->ClassID();
            if (klass == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
            {
                Modifier* modifierItem = (Modifier*) rm;
                if (modifierItem->IsEnabled()!=FALSE)
                {
                    return true;
                }
            }

            ISkin* skinMod = (ISkin*) rm->GetInterface(I_SKIN);
            if (skinMod)
            {
                Modifier* modifierItem = (Modifier*) rm;
                if (modifierItem->IsEnabled()!=FALSE)
                {
                    return true;
                }
            }
        }
    }

    return false;
}


bool hctConvexHullUtility::getWorldVerticesFromNode (INode* node, Tab<Point3>& worldVerticesOut)
{
    if (!node) return false;

    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();

    Object *obj = node->EvalWorldState( now ).obj;
    if( ! obj->CanConvertToType( Class_ID(TRIOBJ_CLASS_ID, 0) ) ) return false;

    TriObject *triObject = (TriObject*) obj->ConvertToType( now, Class_ID(TRIOBJ_CLASS_ID, 0) );

    const Mesh& mesh = triObject->mesh;

    for (int v=0; v<mesh.numVerts; v++)
    {
        Point3 vertexWorld = mesh.verts[v] * node->GetObjectTM(now);
        worldVerticesOut.Append(1, &vertexWorld);
    }

    if (obj!=triObject)
    {
        triObject->DeleteThis();
    }

    return true;
}


INode* hctConvexHullUtility::createMultiHull (const Tab<INode*>& nodes)
{
    if (nodes.Count()==0) return NULL;

    Tab<Point3> verticesWorld;

    for (int ni=0; ni<nodes.Count(); ni++)
    {
        INode* node = nodes[ni];
        getWorldVerticesFromNode(node, verticesWorld);
    }

    if (verticesWorld.Count()<3) return NULL;

    // Transform vertices into local space of the main node
    INode* mainNode = nodes[0];

    INode* result = createHullNodeFromWorldVertices(verticesWorld, mainNode);

    return result;
}

INode* hctConvexHullUtility::createHull (INode* node)
{
    if (!node) return NULL;

    Tab<INode*> nodes;
    nodes.Append(1, &node);

    INode* result = createMultiHull(nodes);

    return result;
}

/*static*/ INode* hctConvexHullUtility::createHullNodeFromWorldVertices (const Tab<Point3>& worldVertices, INode* mainNode)
{
    Interface* ip = GetCOREInterface();
    TimeValue now = ip->GetTime();

    INode* result = NULL;

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    {
        MSTR buffer;
        buffer.printf(TEXT("Generating hull for node : %s "), mainNode->GetName());
        ip->PushPrompt(buffer);
    }

    {
        // call the utility
        MaxProgressUpdater progressUpdater( _T( "Running Convex Hull Utility..") );
        hctCreateConvexHullUtil::Input input;
        {
            const Matrix3 worldToNode = Inverse (mainNode->GetNodeTM(now));
            for (int vi=0; vi<worldVertices.Count(); vi++)
            {
                Point3 localVert = worldVertices[vi] * worldToNode;
                input.m_vertices.pushBack(hkVector4(localVert.x, localVert.y, localVert.z));
            }

            input.m_maxNumVerts = m_pblock->GetInt(PA_CONVEX_HULL_UTILITY_MAXVERTS);
            input.m_encloseInputs = (m_pblock->GetInt( PA_CONVEX_HULL_UTILITY_ENCLOSE_INPUTS ) != FALSE);
            input.m_progressUpdater = &progressUpdater;

            // run the convex hull utility
            hctCreateConvexHullUtil::Output output;
            hkResult hullUtilResult = hctCreateConvexHullUtil::createCoarseHull(input, output);

            if (hullUtilResult.isSuccess())
            {
                theHold.Begin();
                {
                    // create a triangle mesh from the generated hull vertices/faces
                    Mesh hullMesh;
                    hullMesh.setNumVerts( output.m_convexGeom.m_vertices.getSize() );
                    hullMesh.setNumFaces( output.m_convexGeom.m_triangles.getSize() );

                    for (int vi=0; vi<hullMesh.getNumVerts(); ++vi)
                    {
                        hkVector4& p = output.m_convexGeom.m_vertices[vi];
                        hullMesh.setVert( vi, p(0), p(1), p(2) );
                    }
                    for (int ti=0; ti<hullMesh.getNumFaces(); ++ti)
                    {
                        hkGeometry::Triangle& triangle = output.m_convexGeom.m_triangles[ti];
                        hullMesh.faces[ti].setVerts( triangle.m_a, triangle.m_b, triangle.m_c );
                        hullMesh.faces[ti].setEdgeVisFlags(EDGE_VIS, EDGE_VIS, EDGE_VIS);
                        hullMesh.faces[ti].setSmGroup(1);
                    }

                    //create an object to hold the mesh
                    TriObject* triObj = CreateNewTriObject();
                    triObj->mesh = hullMesh;

                    //create a node to hold the object, and give it a name
                    result = ip->CreateObjectNode( triObj );
                    MSTR newNodeName = MSTR(mainNode->GetName()) + TEXT("_Hull");
                    ip->MakeNameUnique(newNodeName);
                    result->SetName(newNodeName);

                    //set the node transform to that of the selected object
                    const TimeValue nodeNowTime = ip->GetTime();
                    result->SetNodeTM(nodeNowTime, mainNode->GetNodeTM(nodeNowTime));
                }
                theHold.Accept( _T("Create Convex Hull") );
            }
        }
    }

    ip->PopPrompt();
    SetCursor(LoadCursor(NULL, IDC_ARROW));

    ip->ForceCompleteRedraw();

    return result;
}

void hctConvexHullUtility::updateUI()
{
    if( !m_interface )
    {
        return;
    }

    m_classDesc->InvalidateUI();

    /*
    IParamMap2* pmap = m_pblock->GetMap( PB_FX_SPHERES_PBLOCK );
    HWND hWnd = pmap->GetHWnd();

    // Enable/disable controls here...
    // We don't have any parameters (which need to be updated) so this method is very simple
    */
}

INode* hctConvexHullUtility::createSkinHull (INode* node)
{
    if (!node) return NULL;

    Tab<INode*> nodes;
    nodes.Append(1, &node);

    INode* result = createSkinMultiHull(nodes);

    return result;
}

INode* hctConvexHullUtility::createSkinMultiHull (const Tab<INode*>& nodes)
{

    // Look for any skin associated with the node
    Tab<Point3> verticesWorld;

    for (int ni=0; ni<nodes.Count(); ni++)
    {
        INode* boneNode = nodes[ni];

        DependentIterator di(boneNode);

        ReferenceMaker *rm;

        while ((rm=di.Next())!=NULL)
        {
            Class_ID klass = rm->ClassID();
            if (klass == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
            {
                Modifier* modifierItem = (Modifier*) rm;
                IPhysiqueExport* physique = (IPhysiqueExport *) rm->GetInterface(I_PHYINTERFACE);
                if (physique && modifierItem->IsEnabled())
                {
                    INodeTab skinNodes = hctMaxUtils::findNodeRefs((Modifier*) rm);

                    for (int si=0; si<skinNodes.Count(); si++)
                    {
                        INode* skinNode = skinNodes[si];
                        addPhysiqueVertices(physique, skinNode, boneNode, verticesWorld);
                    }
                }

            }
            ISkin* skinMod = (ISkin*) rm->GetInterface(I_SKIN);
            if (skinMod)
            {
                Modifier* modifierItem = (Modifier*) rm;
                if (modifierItem->IsEnabled())
                {
                    INodeTab skinNodes = hctMaxUtils::findNodeRefs((Modifier*) rm);

                    for (int si=0; si<skinNodes.Count(); si++)
                    {
                        INode* skinNode = skinNodes[si];
                        addSkinVertices(skinMod, skinNode, boneNode, verticesWorld);
                    }
                }
            }
        }
    }

    if (verticesWorld.Count()<3) return NULL;

    // Transform vertices into local space of the main node
    INode* mainNode = nodes[0];

    INode* result = createHullNodeFromWorldVertices(verticesWorld, mainNode);

    return result;

}


void hctConvexHullUtility::addPhysiqueVertices (IPhysiqueExport* physiqueMod, INode* skinNode, INode* boneNode,  Tab<Point3>& vertsWorldInOut)
{
    IPhyContextExport* contextData = physiqueMod->GetContextInterface(skinNode);

    if (!contextData) return;

    const float minWeight = m_pblock->GetFloat(PA_CONVEX_HULL_UTILITY_MIN_WEIGHT);

    contextData->ConvertToRigid(TRUE);

    const int numverts = contextData->GetNumberVertices();
    BitArray selVerts(numverts);

    selVerts.ClearAll();

    for(int vi=0;vi<numverts;vi++)
    {
        IPhyVertexExport* vertexExp = contextData->GetVertexInterface(vi);

        if (!vertexExp)
        {
            continue;
        }

        const int vType = vertexExp->GetVertexType();
        if (vType & BLENDED_TYPE )
        {
            IPhyBlendedRigidVertex *blendedVtx = (IPhyBlendedRigidVertex*) vertexExp;
            const int numAssociatedBones =  blendedVtx->GetNumberNodes();
            for (int bi=0; bi<numAssociatedBones; bi++)
            {
                if (blendedVtx->GetNode(bi)==boneNode)
                {
                    if (blendedVtx->GetWeight(bi)>=minWeight)
                    {
                        selVerts.Set(vi);
                    }
                }
            }
        }
        else
        {
            IPhyRigidVertex* rigidVtx = (IPhyRigidVertex*) vertexExp;
            if (rigidVtx->GetNode() == boneNode)
            {
                selVerts.Set(vi);
            }
        }

        contextData->ReleaseVertexInterface(vertexExp);
    }

    physiqueMod->ReleaseContextInterface(contextData);

    const TimeValue now = GetCOREInterface()->GetTime();

    Object *obj = skinNode->EvalWorldState( now ).obj;
    if( ! obj->CanConvertToType( Class_ID(TRIOBJ_CLASS_ID, 0) ) ) return;

    TriObject *triObject = (TriObject*) obj->ConvertToType( now, Class_ID(TRIOBJ_CLASS_ID, 0) );

    const Mesh& mesh = triObject->mesh;

    for (int vi=0; vi<mesh.numVerts; vi++)
    {
        if (selVerts[vi])
        {
            const Point3 newVert = mesh.verts[vi];
            Point3 vertInWorld = newVert * skinNode->GetObjectTM(now);
            vertsWorldInOut.Append(1, &vertInWorld);
        }
    }

    if (obj!=triObject)
    {
        triObject->DeleteThis();
    }
}


void hctConvexHullUtility::addSkinVertices (ISkin* skin, INode* skinNode, INode* boneNode, Tab<Point3>& vertsWorldInOut)
{
    ISkinContextData* contextData = skin->GetContextInterface(skinNode);

    const float minWeight = m_pblock->GetFloat(PA_CONVEX_HULL_UTILITY_MIN_WEIGHT);

    // look for the index of the bone in the skin
    int ourBoneIndex = -1;
    {
        for (int bi=0; bi<skin->GetNumBones(); bi++)
        {
            if (skin->GetBone(bi)==boneNode)
            {
                ourBoneIndex = bi;
                break;
            }
        }
    }

    if (ourBoneIndex<0) return;

    const int numVertices = contextData->GetNumPoints();

    BitArray selVerts(numVertices);

    selVerts.ClearAll();

    for (int vi=0; vi<numVertices; vi++)
    {
        for (int bi=0; bi<contextData->GetNumAssignedBones(vi); bi++)
        {
            if (contextData->GetAssignedBone(vi, bi) == ourBoneIndex)
            {
                if (contextData->GetBoneWeight(vi, bi) >= minWeight)
                {
                    selVerts.Set(vi);
                }
            }
        }
    }

    const TimeValue now = GetCOREInterface()->GetTime();

    Object *obj = skinNode->EvalWorldState( now ).obj;
    if( ! obj->CanConvertToType( Class_ID(TRIOBJ_CLASS_ID, 0) ) ) return;

    TriObject *triObject = (TriObject*) obj->ConvertToType( now, Class_ID(TRIOBJ_CLASS_ID, 0) );

    const Mesh& mesh = triObject->mesh;

    for (int vi=0; vi<mesh.numVerts; vi++)
    {
        if (selVerts[vi])
        {
            const Point3 newVert = mesh.verts[vi];
            Point3 vertInWorld = newVert * skinNode->GetObjectTM(now);
            vertsWorldInOut.Append(1, &vertInWorld);
        }
    }

    if (obj!=triObject)
    {
        triObject->DeleteThis();
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
