// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Max/MaxFpInterfaces/Physics/CgoUtility/hctCgoUtilityInterface.h>
#include <ContentTools/Max/MaxSceneExport/Utilities/Cgo/hctCgoUtility.h>

#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>
#include <Common/Internal/GeometryProcessing/CollisionGeometryOptimizer/hkgpCgo.h>

// Connection to Havok SDK through hksdkutils
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>


#define HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT 99999999

namespace hctCgoLocals
{
    static void getSelectionInfo(int& numSelectedPolygonalNodes, int& numVerticesInSelection)
    {
        numSelectedPolygonalNodes = 0;
        numVerticesInSelection = 0;

        Interface* ip = GetCOREInterface();
        int numSelectedNodes = ip->GetSelNodeCount();
        if (numSelectedNodes == 0)
        {
            return;
        }

        const TimeValue now = ip->GetTime();

        for (int i = 0; i < numSelectedNodes; i++)
        {
            TriObject* triObject = (TriObject*)ip->GetSelNode(i)->EvalWorldState(now).obj->ConvertToType(now, Class_ID(TRIOBJ_CLASS_ID, 0));
            if (triObject)
            {
                numSelectedPolygonalNodes++;
                numVerticesInSelection += triObject->mesh.getNumVerts();
            }
        }
    }

    static void updateTargetNumVerticesRatioTooltip(hctCgoUtility* cgoUtil, int currentNumVertices)
    {
        MSTR tooltipText;
        if (cgoUtil->m_targetNumVerticesUpperLimit != HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT)
        {
            tooltipText.printf(TEXT("%d out of %d vertices"), currentNumVertices, cgoUtil->m_targetNumVerticesUpperLimit);
        }

        cgoUtil->m_targetNumVerticesEditCtrl->SetTooltip(true, tooltipText);
        cgoUtil->m_targetRatioEditCtrl->SetTooltip(true, tooltipText);
    }

    static void targetNumVerticesValueChanged(hctCgoUtility* cgoUtil)
    {
        int targetNumVertices = cgoUtil->m_targetNumVerticesSpinnerCtrl->GetIVal();
        cgoUtil->m_manualTargetNumVertices = targetNumVertices;

        cgoUtil->targetNumVerticesModified(targetNumVertices);
    }

    static void targetVertexRatioValueChanged(hctCgoUtility* cgoUtil)
    {
        float currentRatio = cgoUtil->m_targetRatioSpinnerCtrl->GetFVal();
        cgoUtil->m_targetVertexRatio = currentRatio / 100.0f;

        cgoUtil->targetVertexRatioModified();
    }

    // User has switched to RETAIN NUMBER OF VERTICES mode.
    static void retainNumVertices(hctCgoUtility* cgoUtil)
    {
        cgoUtil->m_ignoreSetValueCallback = true;

        cgoUtil->m_targetRetainmentType = 0;

        // Disable the 'vertex ratio' edit control.
        cgoUtil->m_targetRatioEditCtrl->Disable();
        cgoUtil->m_targetRatioSpinnerCtrl->Disable();

        // The 'vertex ratio' has no meaning when switching back to 'number of vertices' mode if there currently is "no"
        // upper limit to the target number of vertices. Therefore we simply force the target number to the minimum of
        // 4 vertices.
        if (cgoUtil->m_targetNumVerticesUpperLimit == HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT)
        {
            cgoUtil->m_manualTargetNumVertices = 4;
            cgoUtil->m_targetNumVerticesEditCtrl->SetText(cgoUtil->m_manualTargetNumVertices);
            cgoUtil->m_targetNumVerticesSpinnerCtrl->SetValue(cgoUtil->m_manualTargetNumVertices, TRUE);
            cgoUtil->targetNumVerticesModified(cgoUtil->m_manualTargetNumVertices);
        }

        // Enable (and unhide) the 'target number of vertices' edit control and (re)set its value.
        cgoUtil->m_targetNumVerticesEditCtrl->Enable();
        cgoUtil->m_targetNumVerticesSpinnerCtrl->Enable();

        cgoUtil->m_ignoreSetValueCallback = false;
    }

    // User has switched to RETAIN VERTEX RATIO mode.
    static void retainVertexRatio(hctCgoUtility* cgoUtil)
    {
        cgoUtil->m_ignoreSetValueCallback = true;

        cgoUtil->m_targetRetainmentType = 1;

        // Disable the 'number of target vertices' edit control.
        cgoUtil->m_targetNumVerticesEditCtrl->Disable();
        cgoUtil->m_targetNumVerticesSpinnerCtrl->Disable();

        // Enable (and unhide) the 'vertex ratio' edit control and (re)set its value.
        cgoUtil->m_targetRatioEditCtrl->Enable();
        cgoUtil->m_targetRatioSpinnerCtrl->Enable();

        // If there is currently "no" upper limit to the target number of vertices then the ratio has no meaning either
        // and we force it to (an arbitrary) 50%.
        if (cgoUtil->m_targetNumVerticesUpperLimit == HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT)
        {
            cgoUtil->m_targetVertexRatio = 0.5f;
            cgoUtil->m_targetRatioEditCtrl->SetText(cgoUtil->m_targetVertexRatio*100.0f);
            cgoUtil->m_targetRatioSpinnerCtrl->SetValue(cgoUtil->m_targetVertexRatio*100.0f, TRUE);
            cgoUtil->targetVertexRatioModified();
        }

        cgoUtil->m_ignoreSetValueCallback = false;
    }
}
using namespace hctCgoLocals;



namespace hctCgoUtilityUiInternals
{
    enum
    {
        MAP_GENERAL_PROPERTIES_ROLLOUT
    };

    ///////////////////////////////////////////////////////////////////////////////////////////
    //
    // The class descriptor
    //
    ///////////////////////////////////////////////////////////////////////////////////////////
    class _ClassDesc2 : public ClassDesc2
    {
    public:
        int             IsPublic() { return TRUE; }
        void*           Create(BOOL loading = FALSE) { return getCgoUtilityInstance(); }
        const MCHAR *   ClassName() { return GetString(IDS_CGO_UTILITY_CLASS_NAME); }
        SClass_ID       SuperClassID() { return UTILITY_CLASS_ID; }
        Class_ID        ClassID() { return HK_CGO_UTILITY_CLASS_ID; }
        const MCHAR*    Category() { return GetString(IDS_CGO_UTILITY_CATEGORY); }
        const MCHAR*    InternalName() { return _T("hkCgoUtility"); }   // Returns fixed parsable name (scripter-visible name)
        HINSTANCE       HInstance() { return hInstance; }                // Returns owning module handle
    };

    ///////////////////////////////////////////////////////////////////////////////////////////
    //
    // The dialog
    //
    ///////////////////////////////////////////////////////////////////////////////////////////
    class _DlgProc : public ParamMap2UserDlgProc
    {
        public:
            INT_PTR DlgProc( TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
            {
                // NOTE: Buttons are handled by the action publishing interface, so no need to handle them here
                switch (msg)
                {
                    case WM_LBUTTONDOWN:
                    case WM_LBUTTONUP:
                    case WM_MOUSEMOVE:
                    {
                        getCgoUtilityInstance()->m_interface->RollupMouseMessage(hWnd, msg, wParam, lParam);
                        break;
                    }

                    case WM_COMMAND:
                    {
                        break;
                    }

                    default:
                    {
                        return FALSE;
                    }
                }
                return TRUE;
            }

            void DeleteThis() {}
    };

    class _PBAccessor : public PBAccessor
    {
        public:

            _PBAccessor() {}

            void Set(PB2Value& v, ReferenceMaker* owner, ParamID id, int tabIndex, TimeValue t)
            {
                hctCgoUtility* cgoUtil = static_cast<hctCgoUtility*>(owner);

                if (id == PA_CGO_UTILITY_TARGET_NUM_VERTICES)
                {
                    if (!cgoUtil->m_ignoreSetValueCallback)
                    {
                        hctCgoLocals::targetNumVerticesValueChanged(cgoUtil);
                    }
                }
                else if (id == PA_CGO_UTILITY_TARGET_RATIO)
                {
                    if (!cgoUtil->m_ignoreSetValueCallback)
                    {
                        hctCgoLocals::targetVertexRatioValueChanged(cgoUtil);
                    }
                }
                else if (id == PA_CGO_UTILITY_RETAINMENT_TYPE)
                {
                    if (v.i == 0)
                    {
                        hctCgoLocals::retainNumVertices(cgoUtil);
                    }
                    else
                    {
                        hctCgoLocals::retainVertexRatio(cgoUtil);
                    }
                }
            }
    };
}


///////////////////////////////////////////////////////////////////////////////////////////
//
// The utility implementation
//
///////////////////////////////////////////////////////////////////////////////////////////

static hctCgoUtility theInstance;

hctCgoUtility* getCgoUtilityInstance()
{
    return &theInstance;
}

ClassDesc2* getHkCgoUtilityDesc()
{
    return getCgoUtilityInstance()->m_classDesc;
}


hctCgoUtility::hctCgoUtility()
{
    m_classDesc = new hctCgoUtilityUiInternals::_ClassDesc2();

    m_dlgProc = new hctCgoUtilityUiInternals::_DlgProc();

    m_pBAccessor = new hctCgoUtilityUiInternals::_PBAccessor();

    // Create the UI.
    m_pBlock2 = new ParamBlockDesc2
    (
        PB_CGO_UTILITY_PBLOCK, _T("CGO"), 0, getHkCgoUtilityDesc(),
        P_AUTO_CONSTRUCT + P_AUTO_UI + P_MULTIMAP, PB_CGO_UTILITY_PBLOCK,

        // One rollout
        1,
        hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
            IDD_CGO_UTILITY_ROLLOUT_PARAMS,
            IDS_CGO_UTILITY_ROLLOUT_PARAMS,
            0, 0,
            m_dlgProc,

        // parameters
        PA_CGO_UTILITY_MAXERROR,
            _T("maxError"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_MAXERROR,
            p_default,      100.0f,
            p_range,        0.0f, 100000.0f,
            p_tooltip,      IDS_CGO_UTILITY_TT_MAX_ERROR,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_SPINNER, EDITTYPE_UNIVERSE, IDC_ED_CGO_MAXERROR, IDC_SP_CGO_MAXERROR,
                            SPIN_AUTOSCALE,
            p_end,

        PA_CGO_UTILITY_SHRINKING,
            _T("shrinking"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_SHRINKING,
            p_default,      0.0f,
            p_range,        0.0f, 100.0f,
            p_tooltip,      IDS_CGO_UTILITY_TT_SHRINKING,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_CGO_SHRINKING, IDC_SP_CGO_SHRINKING,
                            SPIN_AUTOSCALE,
            p_end,

        PA_CGO_UTILITY_TARGET_NUM_VERTICES,
            _T("targetNumVerts"), TYPE_INT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_TARGET_NUM_VERTICES,
            p_default,      4,
            p_range,        4, 99999999,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_SPINNER, EDITTYPE_INT, IDC_ED_CGO_TARGET_NUM_VERTICES, IDC_SP_CGO_TARGET_NUM_VERTICES,
                            SPIN_AUTOSCALE,
            p_accessor,     m_pBAccessor,
            p_end,

        PA_CGO_UTILITY_TARGET_RATIO,
            _T("targetRatio"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_TARGET_RATIO,
            p_default,      100.0f,
            p_range,        0.0f, 100.0f,
            p_enabled,      FALSE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_CGO_TARGET_RATIO, IDC_SP_CGO_TARGET_RATIO,
                            SPIN_AUTOSCALE,
            p_accessor,     m_pBAccessor,
            p_end,

        PA_CGO_UTILITY_RETAINMENT_TYPE,
            _T("retaintmentType"), TYPE_INT, P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_RETAINMENT_TYPE,
            p_default,      0,
            p_tooltip,      IDS_CGO_UTILITY_TT_RETAINMENT_TYPE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT,
                            TYPE_RADIO, 2, IDC_RB_CGO_RETAINMENT_TYPE_VERTICES, IDC_RB_CGO_RETAINMENT_TYPE_RATIO,
            p_vals,         0, 1,
            p_accessor,     m_pBAccessor,
            p_end,

        PA_CGO_UTILITY_PROTECT_MATERIALS,
            _T("protectMaterials"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_PROTECT_MATERIALS,
            p_default,      FALSE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_CGO_MAT_BOUNDS,
            p_end,

        PA_CGO_UTILITY_PROTECT_EDGES,
            _T("protectEdges"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_PROTECT_EDGES,
            p_default,      FALSE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_CGO_GEO_BOUNDS,
            p_end,

        PA_CGO_UTILITY_USE_WEIGHTS,
            _T("useWeights"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_USE_WEIGHTS,
            p_default,      FALSE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_CGO_VTX_WEIGHTS,
            p_end,

        PA_CGO_UTILITY_DELETE_INPUT,
            _T("deleteInput"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_DELETE_INPUT,
            p_default,      TRUE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_CGO_DELETE_INPUT,
            p_end,

        PA_CGO_UTILITY_MULTIPASS,
            _T("multiPass"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_CGO_UTILITY_PA_MULTIPASS,
            p_default,      TRUE,
            p_ui,           hctCgoUtilityUiInternals::MAP_GENERAL_PROPERTIES_ROLLOUT, TYPE_SINGLECHEKBOX, IDC_CGO_MULTIPASS,
            p_end,

        p_end
    );

    m_pblock = NULL;

    m_targetVertexRatio = 1.0f;
    m_manualTargetNumVertices = 4;
    m_targetNumVerticesUpperLimit = 1;
    m_targetRetainmentType = 0;
    m_ignoreSetValueCallback = true;

    getHkCgoUtilityDesc()->MakeAutoParamBlocks(this);
}

hctCgoUtility::~hctCgoUtility()
{
    delete m_pBlock2;
    delete m_pBAccessor;
    delete m_dlgProc;
    delete m_classDesc;
}

#if MAX_VERSION_MAJOR>=17
RefResult hctCgoUtility::NotifyRefChanged( const Interval& changeInt, RefTargetHandle hTarget, PartID& partID,  RefMessage message, BOOL propagate )
#else
RefResult hctCgoUtility::NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message )
#endif
{
    switch (message)
    {
        case REFMSG_CHANGE:
        {
            m_pBlock2->InvalidateUI();
            updateUI();
            break;
        }
    }
    return REF_SUCCEED;
}

// Simply forward to our own implementation.
static void _selectionChanged (void *param, NotifyInfo *info)
{
    getCgoUtilityInstance()->selectionChanged();
}

void hctCgoUtility::TopologyChanged(NodeKeyTab& nodes)
{
    Interface* ip = GetCOREInterface();

    // Only act if exactly one node is selected.
    // See hctCgoUtility::selectionChanged():
    // "If there are several polygonal nodes in the selection then we will not limit the upper bound for the target
    // number of vertices."
    if (ip->GetSelNodeCount() == 1)
    {
        for (int cni = 0; cni < nodes.Count(); cni++)
        {
            INode* changedNode = NodeEventNamespace::GetNodeByKey(nodes[cni]);
            if (GetCOREInterface()->GetSelNode(0) == changedNode)
            {
                // For sake of simplicity we will just fake a selection change.
                selectionChanged();
            }
        }
    }
}

void hctCgoUtility::selectionChanged()
{
    int numSelectedPolygonalNodes;
    int numVerticesInSelection;
    hctCgoLocals::getSelectionInfo(numSelectedPolygonalNodes, numVerticesInSelection);

    // If no node has been selected (or none of the selected nodes are polygonal) then there's no upper limit to the
    // target number of vertices, yet the 'Optimize' button has to be disabled.
    if (numSelectedPolygonalNodes == 0)
    {
        m_optimizeButtonCtrl->Disable();

        m_targetNumVerticesUpperLimit = HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT;
    }
    else
    {
        m_optimizeButtonCtrl->Enable();

        if (numSelectedPolygonalNodes == 1)
        {
            m_targetNumVerticesUpperLimit = numVerticesInSelection;
        }
        else
        {
            // If there are several polygonal nodes in the selection then we will not limit the upper bound for the target
            // number of vertices.
            m_targetNumVerticesUpperLimit = HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT;
        }
    }

    // We need to set the (new) upper limit.
    m_targetNumVerticesSpinnerCtrl->SetLimits(4, m_targetNumVerticesUpperLimit, FALSE);

    if (m_targetRetainmentType == 0)
    {
        // try (it might still get clipped!) to keep the old number of target vertices and adjust the resulting percentage.
        int targetNumVertices = m_targetNumVerticesSpinnerCtrl->GetIVal();

        // First try to restore the last manually set value (in case it had to be clipped previously).
        if (targetNumVertices < m_manualTargetNumVertices)
        {
            targetNumVertices = m_manualTargetNumVertices;
        }

        targetNumVertices = targetNumVerticesModified(targetNumVertices);
    }
    else
    {
        targetVertexRatioModified();
    }
}

int hctCgoUtility::targetNumVerticesModified(int targetNumVertices)
{
    m_ignoreSetValueCallback = true;

    // Manually cap the value at the maximum number of vertices in the currently selected mesh.
    targetNumVertices = hkMath::min2(targetNumVertices, m_targetNumVerticesUpperLimit);

    // Visually 'flag' the control if we had to cap the value.
    if (targetNumVertices < m_manualTargetNumVertices)
    {
        m_targetNumVerticesEditCtrl->SetBold(TRUE);
    }
    else
    {
        m_targetNumVerticesEditCtrl->SetBold(FALSE);
    }

    // Update the 'target number of vertices' display.
    m_targetNumVerticesEditCtrl->SetText(targetNumVertices);
    m_targetNumVerticesSpinnerCtrl->SetValue(targetNumVertices, TRUE);

    hctCgoLocals::updateTargetNumVerticesRatioTooltip(this, targetNumVertices);

    // Adjust the 'vertex ratio' according to the currently selected mesh and the number of its vertices.
    if (m_targetNumVerticesUpperLimit != HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT)
    {
        m_targetVertexRatio = float(targetNumVertices) / float(m_targetNumVerticesUpperLimit);
        m_targetRatioEditCtrl->SetText(m_targetVertexRatio * 100.0f);
        m_targetRatioSpinnerCtrl->SetValue(m_targetVertexRatio * 100.0f, TRUE);
    }
    else
    {
        MSTR label; label.printf(TEXT("n/a"));
        m_targetRatioEditCtrl->SetText(label);
    }

    m_ignoreSetValueCallback = false;

    return targetNumVertices;
}

void hctCgoUtility::targetVertexRatioModified()
{
    m_ignoreSetValueCallback = true;

    // Keep the set percentage (of the maximum number of vertices) and adjust the de facto number of target vertices.
    // Note that any manual change to the ratio also counts as a manual change to the 'target number of vertices'!
    // Only do this if we have some mesh selected, though!
    if (m_targetNumVerticesUpperLimit != HK_CGO_UTILITY_TARGET_NUM_VERTICES_HARD_UPPER_LIMIT)
    {
        m_manualTargetNumVertices = m_targetVertexRatio * m_targetNumVerticesUpperLimit;

        // Also reset the visual marker that might indicate that we up until now had to cap the displayed number of target vertices.
        m_targetNumVerticesEditCtrl->SetBold(FALSE);

        // Clip the 'target number of vertices' to a minimum of 4.
        if (m_manualTargetNumVertices < 4)
        {
            m_manualTargetNumVertices = 4;

            // Re-adjust the ratio after clipping the target number of vertices and update the display.
            m_targetVertexRatio = float(m_manualTargetNumVertices) / float(m_targetNumVerticesUpperLimit);
            m_targetRatioEditCtrl->SetText(m_targetVertexRatio * 100.0f);
            m_targetRatioSpinnerCtrl->SetValue(m_targetVertexRatio * 100.0f, TRUE);
        }

        // Update the 'target number of vertices' display.
        m_targetNumVerticesEditCtrl->SetText(m_manualTargetNumVertices);
        m_targetNumVerticesSpinnerCtrl->SetValue(m_manualTargetNumVertices, TRUE);
    }
    else
    {
        MSTR label;
        label.printf(TEXT("n/a"));
        m_targetNumVerticesEditCtrl->SetText(label);
    }

    m_ignoreSetValueCallback = false;
}

void hctCgoUtility::BeginEditParams( Interface *ip, IUtil *iu )
    {
    m_interface = ip;

    // Add the rollups owned by param blocks
    getHkCgoUtilityDesc()->BeginEditParams( (IObjParam *)ip, this, 0, this );

    IParamMap2* pMap = m_pblock->GetMap();
    if (pMap)
    {
        m_targetNumVerticesEditCtrl     = GetICustEdit(GetDlgItem(pMap->GetHWnd(), IDC_ED_CGO_TARGET_NUM_VERTICES));
        m_targetNumVerticesSpinnerCtrl  = GetISpinner(GetDlgItem(pMap->GetHWnd(), IDC_SP_CGO_TARGET_NUM_VERTICES));
        m_targetRatioEditCtrl           = GetICustEdit(GetDlgItem(pMap->GetHWnd(), IDC_ED_CGO_TARGET_RATIO));
        m_targetRatioSpinnerCtrl        = GetISpinner(GetDlgItem(pMap->GetHWnd(), IDC_SP_CGO_TARGET_RATIO));
        m_optimizeButtonCtrl            = GetICustButton(GetDlgItem(pMap->GetHWnd(), IDC_B_OPTIMIZE));
    }
    else
    {
        m_targetNumVerticesEditCtrl     = NULL;
        m_targetNumVerticesSpinnerCtrl  = NULL;
        m_targetRatioEditCtrl           = NULL;
        m_targetRatioSpinnerCtrl        = NULL;
        m_optimizeButtonCtrl            = NULL;
    }

    selectionChanged();

    updateUI();

    RegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);

    // Register ourselves for node callbacks. We are only interested in the 'topologyChanged' callback, though.
    m_callbackKey = GetISceneEventManager()->RegisterCallback(this);
}

void hctCgoUtility::EndEditParams( Interface *ip, IUtil *iu )
{
    // Clean up the rollups owned by param blocks
    getHkCgoUtilityDesc()->EndEditParams( (IObjParam *)ip, this, END_EDIT_REMOVEUI, this );

    m_interface = NULL;

    GetISceneEventManager()->UnRegisterCallback(m_callbackKey);

    UnRegisterNotification(_selectionChanged, this, NOTIFY_SELECTIONSET_CHANGED);
}

bool hctCgoUtility::canOptimize()
{
    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();

    if (ip->GetSelNodeCount() == 0)
    {
        return false;
    }

    // All selected nodes need to be meshes, otherwise we won't allow for optimization.
    for (int i = 0; i < ip->GetSelNodeCount(); ++i)
    {
        INode* node = ip->GetSelNode(i);
        Object *obj = node->EvalWorldState(now).obj;
        if (!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))
        {
            return false;
        }
    }

    return true;
}

INode* hctCgoUtility::optimize (INode* node)
{
    INode*      ret = NULL;
    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();
    Object*     curObject = node->EvalWorldState(now).obj;
    TriObject*  triObject = (TriObject*) curObject->ConvertToType( now, Class_ID( TRIOBJ_CLASS_ID, 0 ) );
    if(triObject)
    {
        Mesh& nodeMesh = triObject->mesh;

        const int numVerts = nodeMesh.getNumVerts();
        const int numFaces = nodeMesh.getNumFaces();

        //Create the geometry
        hkGeometry nodeGeometry;

        Matrix3 objectToWorld = node->GetObjTMAfterWSM(now);

        {
            // copy vertices to geometry
            hkArray<hkVector4>& nodeVertices = nodeGeometry.m_vertices;
            nodeVertices.setSize( numVerts );

            for ( int vi = 0; vi < numVerts; ++vi )
            {
                Point3 pWorld = nodeMesh.verts[vi] * objectToWorld;

                hkVector4 v( pWorld.x, pWorld.y, pWorld.z , 1.0f);
                nodeVertices[vi] = v;
            }

            if (nodeMesh.vertCol)
            {
                for (int fi = 0; fi < numFaces; ++fi)
                {
                    for (int i = 0; i < 3; ++i)
                    {
                        int vi = (int) nodeMesh.faces[fi].v[i];
                        int ti = (int) nodeMesh.vcFace[fi].t[i];
                        nodeVertices[vi](3) = hkMath::min2(nodeVertices[vi](3), nodeMesh.vertCol[ti].Length() / hkMath::sqrt(3.0f));
                    }
                }
            }

            // copy triangles to geometry
            hkArray<hkGeometry::Triangle>& nodeTriangles = nodeGeometry.m_triangles;
            nodeTriangles.setSize( numFaces );

            for (int ti = 0; ti < numFaces; ++ti)
            {
                Face& f = nodeMesh.faces[ti];
                nodeTriangles[ti].m_a = f.getVert(0);
                nodeTriangles[ti].m_b = f.getVert(1);
                nodeTriangles[ti].m_c = f.getVert(2);
                nodeTriangles[ti].m_material = (int) f.getMatID();
            }
        }

        const hkReal    maxError = m_pblock->GetFloat(PA_CGO_UTILITY_MAXERROR);
        const int       targetNumVertices = m_pblock->GetInt(PA_CGO_UTILITY_TARGET_NUM_VERTICES);
        const hkReal    shrinking = m_pblock->GetFloat(PA_CGO_UTILITY_SHRINKING) / 100.0f;
        const bool      protectMaterials = m_pblock->GetInt(PA_CGO_UTILITY_PROTECT_MATERIALS) ? true : false;
        const bool      protectEdges = m_pblock->GetInt(PA_CGO_UTILITY_PROTECT_EDGES) ? true : false;
        const bool      useWeights = m_pblock->GetInt(PA_CGO_UTILITY_USE_WEIGHTS) ? true : false;
        const bool      deleteInput = m_pblock->GetInt(PA_CGO_UTILITY_DELETE_INPUT) ? true : false;
        const bool      multiPass = m_pblock->GetInt(PA_CGO_UTILITY_MULTIPASS) ? true : false;

        struct ProgressUpdate : hkgpCgo::IProgress
        {
            ProgressUpdate(Interface* ip, int bv) : m_ip(ip), m_baseVertices(bv)
            {
                m_currentProgress = -1;
                m_ip->ProgressStart(TEXT("Havok CGO"), TRUE, &placeHolder, NULL);
            }

            ~ProgressUpdate()
            {
                m_ip->ProgressEnd();
            }

            bool step(const hkgpCgo::Config& config, hkReal error, int numVertices, int numTriangles)
            {
                const int   vOffset = (m_baseVertices - config.m_maxVertices);
                hkReal      eProgress = hkMath::clamp(config.m_maxDistance > 0.0f ? error / config.m_maxDistance : 0.0f, 0.0f, 1.0f);
                hkReal      vProgress = hkMath::clamp(vOffset > 0 ? 1.0f - (numVertices - config.m_maxVertices) / (hkReal)vOffset : 0.0f, 0.0f, 1.0f);
                int         progress = (int) (hkMath::max2(eProgress,vProgress) * 100.0f + 0.5f);
                if (m_currentProgress != progress)
                {
                    m_ip->ProgressUpdate(progress, true);
                    m_currentProgress = progress;
                }
                m_canceled = m_ip->GetCancel() ? true : false;
                return !m_canceled;
            }

            static DWORD WINAPI placeHolder(LPVOID arg) { return(0); }

            int         m_currentProgress;
            int         m_baseVertices;
            bool        m_canceled;
            Interface*  m_ip;
        } progress(ip, nodeGeometry. m_vertices.getSize());

        hkgpCgo::Config                 config;
        hkArray<hkgpCgo::ClusterData>   clusters; clusters.setSize(16);
        int                             pass = 1;

        // start worker threads
        hkDefaultTaskQueue taskQueue;
        hkCpuThreadPoolCinfo threadPoolCinfo;
        threadPoolCinfo.m_numThreads = hkHardwareInfo::getNumHardwareThreads() - 1;
        hkCpuThreadPool threadPool( threadPoolCinfo );
        threadPool.processTaskQueue( &taskQueue );

        config.m_taskQueue                  = &taskQueue;
        config.m_maxDistance                = maxError <= 0.0f ? HK_REAL_MAX : maxError;
        config.m_maxShrink                  = shrinking;
        config.m_maxVertices                = targetNumVertices;
        config.m_protectMaterialBoundaries  = protectMaterials;
        config.m_protectNakedBoundaries     = protectEdges;
        config.m_semantic                   = useWeights ? hkgpCgo::Config::VS_WEIGHT : hkgpCgo::Config::VS_NONE;

        for (bool continueOptimizing = true; continueOptimizing; ++pass)
        {
            const int numTrianglesBefore = nodeGeometry.m_triangles.getSize();
            const int numVerticesBefore = nodeGeometry.m_vertices.getSize();

            if (pass == 1)
            {
                MSTR s; s.printf(TEXT("print \"CGO : Starting with %d triangles and %d vertices.\""), numTrianglesBefore, numVerticesBefore);
                ExecuteMAXScriptScript(s);
            }

            continueOptimizing = false;
            hkgpCgo::optimize(config, nodeGeometry, &progress, pass == 1 ? &clusters : HK_NULL);

            const int numTrianglesAfter = nodeGeometry.m_triangles.getSize();
            const int numVerticesAfter = nodeGeometry.m_vertices.getSize();

            continueOptimizing = (numTrianglesAfter < numTrianglesBefore) && multiPass;

            if (numTrianglesAfter < numTrianglesBefore)
            {
                const int percent = (int) (0.5f + (numVerticesAfter * 100.0f) / (float)progress.m_baseVertices);
                MSTR s; s.printf(TEXT("print \"CGO : Pass #%d, T:%d V:%d (%d%%)\""), pass, numTrianglesAfter, numVerticesAfter, percent);
                ExecuteMAXScriptScript(s);
            }
            else if (pass == 1)
            {
                MSTR s = TEXT("print \"CGO : Could not optimize the geometry. Try increasing the maximum allowed error.\"");
                ExecuteMAXScriptScript(s);
            }
        }

        // stop worker threads
        taskQueue.close();
        threadPool.waitForCompletion();

        if (!progress.m_canceled)
        {
            theHold.Begin();
            {
                // create a triangle mesh.
                TriObject*  triObj = CreateNewTriObject();
                Mesh&       mesh = triObj->mesh;
                mesh.setNumVerts( nodeGeometry.m_vertices.getSize() );
                mesh.setNumFaces( nodeGeometry.m_triangles.getSize() );

                Matrix3 WorldToObject(objectToWorld);
                WorldToObject.Invert();

                for (int vi=0; vi<mesh.getNumVerts(); ++vi)
                {
                    hkVector4& p = nodeGeometry.m_vertices[vi];
                    Point3 pWorld( p(0), p(1), p(2) );
                    Point3 pObject = pWorld * WorldToObject;
                    mesh.setVert(vi, pObject);
                }
                for (int ti=0; ti<mesh.getNumFaces(); ++ti)
                {
                    hkGeometry::Triangle& triangle = nodeGeometry.m_triangles[ti];
                    mesh.faces[ti].setVerts( triangle.m_a, triangle.m_b, triangle.m_c );
                    mesh.faces[ti].setEdgeVisFlags(EDGE_VIS, EDGE_VIS, EDGE_VIS);
                    mesh.faces[ti].setMatID((MtlID)triangle.m_material);
                    mesh.faces[ti].setSmGroup(0);
                }

                // Create a node to hold the object, and give it a unique name
                ret = ip->CreateObjectNode( triObj );
                ret->SetMtl(node->GetMtl());
                MSTR newNodeName = MSTR(node->GetName()) + TEXT("_Cgo");
                ip->MakeNameUnique(newNodeName);
                ret->SetName(newNodeName);

                // Inherit the source node's transform
                ret->SetNodeTM(now, node->GetNodeTM(now));

                // Inherit the source node's pivot point
                ret->SetObjOffsetPos(node->GetObjOffsetPos());
                ret->SetObjOffsetRot(node->GetObjOffsetRot());
                ret->SetObjOffsetScale(node->GetObjOffsetScale());

                if ( deleteInput )
                {
                    ip->SelectNode(ret);
                    ip->DeleteNode(node);
                }
            }
            theHold.Accept( _T("Optimized") );
        }
    }

    return ret;
}

void hctCgoUtility::updateUI()
{
    if( !m_interface )
    {
        return;
    }

    getHkCgoUtilityDesc()->InvalidateUI();
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
