
------------------------------------------------------------------------------
-- EXTERNAL METHODS
------------------------------------------------------------------------------

global hvkDestruction_getGenericModifierOfType


------------------------------------------------------------------------------
-- LOCAL METHODS
------------------------------------------------------------------------------

function hvkDestructionDialog_createBreakableBody =
(
	global returnValue = -1
	
	rollout requester "Create Breakable Body"
	(
		label _label1 "You can create individual Destruction Shape && Body modifiers for all"   align:#left
		label _label2 "objects in the selection (UNIQUE) or you can create one single modifier" align:#left
		label _label3 "each that is being shared between all selected objects (SHARED)."        align:#left
		label _label4 ""
		
		button reqButton1 "Unique" width:80 across:3
		button reqButton2 "Shared" width:80
		button reqButton3 "Cancel" width:80
		
		on reqButton1 pressed do
		(
			-- Unique
			DestroyDialog requester
			returnValue = 0
		)

		on reqButton2 pressed do
		(
			-- Shared
			DestroyDialog requester
			returnValue = 1
		)

		on reqButton3 pressed do
		(
			-- Cancel
			DestroyDialog requester
			returnValue = -1
		)
	)

	createDialog requester 370 110 modal:true
	
	return returnValue
)


------------------------------------------------------------------------------
-- GLOBAL METHODS
------------------------------------------------------------------------------


--
--  This function will perform a pre-check on the current selection.
--  o If no Mesh node is selected, we will abort with a user warning.
--  o If exactly one node is selected, we will check if this one already has an hkdShape/hkdBody modifier combo.
--    If so, we will switch to the hkdShape modifier and abort.
--  o If 2+ nodes are selected, we will prompt the user whether he wishes to create unique hkdBody & hkdShape
--    modifiers for each node or if he wants to share them between all selected nodes.
--
--  Return values:
--   -1 : abort (reason can be: user-abort, error)
--    0 : each node has its own unique modifiers
--    1 : all nodes share the same hkdShape and hkdBody modifiers
--
function hvkDestruction_createBreakableBody_preCheck =
(
	if ( selection.count == 0 ) then
	(
		messageBox("Please select a Mesh node first.")
		return -1
	)
	else if ( selection.count == 1 ) then
	(
		return 0
	)

	result = hvkDestructionDialog_createBreakableBody()
	return result
)


--
-- This function will go through all nodes in the current selection and, depending on the user's choice,
-- will create unique/individual hkdSHape & hkdBody modifiers for all nodes or create one shared
-- hkdShape/hkdBody combo. It will also create physics shapes and rigid bodies if they are not yet present
-- on the node's modifier stack.
--
-- Note: a physics shape will only be created (and added) if there's no rigid body modifier on the node's
--       stack. See code below for details.
--
-- Input values:
--   userInput = 0 : create unique modifiers
--   uderInput = 1 : create shared modifiers
--
function hvkDestruction_createBreakableBody userInput =
(
	maxVersionNumber = maxVersion()

	sharedhkdShape = undefined
	sharedhkdBody  = undefined
	
	if ( userInput == 1 ) then
	(
		-- Create SHARED modifiers.
		sharedhkdShape = hkdShape()
		sharedhkdBody  = hkdBody()

		if ( maxVersionNumber[1] >= 11000 ) then -- [not really affected by HKD-159 but let's keep it consistent for 3ds Max 9 and 2008]
		(
			sharedhkdShape.strength = 1.0
		)
	)

	hkdShapeNumAdded = 0
	hkdBodyNumAdded  = 0
	lastNodeAddedTo  = undefined

	physicsShapeNotCreated = 0	

	for selectedNode in $selection do
	(
		--
		-- First search the current node's modifier stack for any modifiers that we might otherwise have to create.
		--
		hkpShapeModifier     = undefined
		hkpRigidBodyModifier = undefined
		hkdShapeModifier     = undefined
		hkdBodyModifier      = undefined
		(
			for modifier in selectedNode.modifiers do
			(
				if ( classOf modifier == hkShapeModifier ) then
				(
					hkpShapeModifier = modifier
					continue
				)
				if ( classOf modifier == hkRigidBodyModifier ) then
				(
					hkpRigidBodyModifier = modifier
					continue
				)
				if ( hkIsOfGenericModifierType modifier "hkdShape" ) then
				(
					hkdShapeModifier = modifier
					continue
				)
				if ( hkIsOfGenericModifierType modifier "hkdBody" ) then
				(
					hkdBodyModifier = modifier
					continue
				)
			)
		)

		-- DevNote: we don't scan the subtree here like we do in Maya. This is ok with Oli.
		hkpRigidBodyOnNode = hkpRigidBodyModifier

		-- If there's no rigid body modifier on the current node but node is part of a hierarchy, we need
		-- to scan the hierarchy as well.
		if ( hkpRigidBodyModifier == undefined ) then
		(
			ancestorNode = selectedNode.parent
			while ( ancestorNode != undefined AND hkpRigidBodyModifier == undefined ) do
			(
				hkpRigidBodyModifier = hvkPhysics_getRigidBodyModifier ancestorNode
				hkdBodyModifier      = hvkDestruction_getGenericModifierOfType ancestorNode "hkdBody"
				ancestorNode         = ancestorNode.parent
			) 
		)

		--
		-- No physics shape modifier on stack yet
		--
		if ( hkpShapeModifier == undefined ) then
		(
			-- Only create a physics shape if there wasn't a rigid body on the stack yet.
			-- Reason: if we call this function on the root node of a compound rigid body with proxies we don't
			-- want a shape to be inserted on top-level
			if ( hkpRigidBodyOnNode == undefined ) then
			(
				hvkPhysics_createShape selectedNode
			)
			else
			(
				physicsShapeNotCreated = 1
			)
		)

		--
		-- No physics rigid body modifier on stack yet
		--
		if ( hkpRigidBodyModifier == undefined ) then
		(
			hkpRigidBodyModifier = hvkPhysics_createRigidBody selectedNode
			hkpRigidBodyModifier.mass = 1.0
		)

		--
		-- No Destruction shape modifier on stack yet
		--
		if ( hkdShapeModifier == undefined ) then
		(
			if ( userInput == 0 ) then
			(
				-- Unique: create individual modifiers for each object in selection
				uniqueShapeMod = hvkDestruction_createGenericModifier selectedNode hkdShape
				if ( maxVersionNumber[1] >= 11000 ) then -- [HKD-159 has been fixed in 3ds Max 2009 and later only]
				(
					uniqueShapeMod.strength = 1.0
				)
			)	
			else
			(
				-- Shared: use the same modifier for all objects in selection
				addModifier selectedNode sharedhkdShape
			)
			
			lastNodeAddedTo = selectedNode

			hkdShapeNumAdded = hkdShapeNumAdded + 1
		)

		--
		-- No Destruction body modifier on stack yet
		--
		if ( hkdBodyModifier == undefined ) then
		(
			if ( userInput == 0 ) then
			(
				-- Unique: create individual modifiers for each object in selection
				hvkDestruction_createGenericModifier selectedNode hkdBody
			)
			else
			(
				-- Shared: use the same modifier for all objects in selection
				addModifier selectedNode sharedhkdBody
			)

			hkdBodyNumAdded = hkdBodyNumAdded + 1
		)
	)

	selectionBackup = selection as array
	
	--
	-- If only one node was selected or we created shared modifiers we will bring them up in the Modify Panel.
	--
	if ( selection.count == 1 OR userInput == 1 ) then
	(
		hkdShapeModifier = hvkDestruction_getGenericModifierOfType selection[1] "hkdShape"

		-- Another workaround: modPanel.setCurrentObject is supposed to switch to the Modify panel (which it doesn't)
		-- so we have to do this manually. But this might reset the current selection to one node only when opening a
		-- dialog window later on. So we need force a refresh on the original selection. This will potentially switch
		-- modifiers again, though. Argh.
		-- At least for one selected node we can achieve that. :P
		setCommandPanelTaskMode mode:#modify

		modPanel.setCurrentObject hkdShapeModifier ui:true
	)

	if ( selectionBackup.count > 1 ) then
	(
		select selectionBackup
	)

	--
	-- Error Messaging
	--
	(
		errorMsg = ""
		
		if ( userInput == 1 ) then
		(
			if ( selection.count != hkdShapeNumAdded ) then
			(
				hvkDestruction_createErrorMessage &errorMsg "Some objects in selection already had a Destruction Shape modifier attached. These will still use the old one and not use the shared new modifier."
			)
			if ( selection.count != hkdBodyNumAdded ) then
			(
				hvkDestruction_createErrorMessage &errorMsg "Some objects in selection already had a Destruction Body modifier attached. These will still use the old one and not use the shared new modifier."
			)
		)

		if ( physicsShapeNotCreated == 1 ) then
		(
			hvkDestruction_createErrorMessage &errorMsg "One or more nodes in the selection were missing a physics shape modifier but had a Rigid Body modifier. No physics shape modifier was created for these."
		)
		
		if ( errorMsg != "" ) then
		(
			messagebox(errorMsg)
		)
	)
)


--
--  This function will perform a pre-check on the current selection.
--  o If less than 2 Mesh nodes are selected, we will abort with a user warning.
--  o If 2+ nodes are selected and if we are not automatically creating a new
--    Dummy parent node, we will let the user verify the master node
--
--  Return values:
--   -1 : abort (reason can be: user-abort, error)
--    0 : Proceed
--
function hvkDestruction_createBreakableBodyFromParts_preCheck createParentDummy &masterNodeOut =
(
	if ( createParentDummy == false ) then
	(
		if ( selection.count < 2 ) then
		(
			messageBox("Please select at least two Mesh nodes first.")
			return -1
		)

		-- Working ourselves up from the intended master node, check if any
		-- ancestor is also part of the selection. Select topmost node to
		-- avoid cyclic errors.

		-- collect all nodes in hierarchy first
		nodesInHierarchy = #()
		masterNode = selection[selection.count]
		while ( masterNode != undefined ) do
		(
			append nodesInHierarchy masterNode
			masterNode = masterNode.parent
		)
		-- traverse hierarchy from top to bottom, abort once we find the first (== topmost) node
		-- that is part of the selection
		for i = nodesInHierarchy.count to 1 by -1 do
		(
			masterNode = nodesInHierarchy[i]
			if ( (findItem (selection as array) masterNode) > 0 ) then
			(
				exit
			)
		)

		result = queryBox ("This will parent all shapes to "+masterNode.name+". Continue ?") title:"Attach Nodes as Fracture Pieces"				
		if ( result ) then
		(
			masterNodeOut = masterNode
			return 0
		)
		return -1
	)

	return 0
)


--
-- This function will:
-- + create a new Dummy parent node if requested
-- + on the ROOT node:
--   - add a physics rigid body modifier to the ROOT (if not present yet)
--   - add hkdBody and hkdShape modifiers to the ROOT (if not present yet)
-- + on all SLAVE nodes
--   - remove any physics rigid body and hkdBody modifier
--   - add a physics shape if the node didn't have a rigid body modifier before
--   - add an hkdShape modifier
--   - link the node to the MASTER node
--
function hvkDestruction_createBreakableBodyFromParts createParentDummy masterNode =
(
	maxVersionNumber = maxVersion()

	errorMsg                         = ""
	errorSlaveRigidBodyRemoved       = 0
	errorSlaveDestructionBodyRemoved = 0

	-- Create a Dummy node and size it so that it will enclose all child nodes.
	if ( createParentDummy ) then
	(
		-- Collect ALL nodes in selection (including nodes in subtrees)
		allNodesInSelectionRecursive = #()
		for node in selection do
		(
			hvkDestruction_collectAllNodesInScene node &allNodesInSelectionRecursive
		)

		totalBoundingBoxMin = selection[1].min
		totalBoundingBoxMax = selection[1].max
		for node in allNodesInSelectionRecursive do
		(
			boundingBoxMin = node.min
			boundingBoxMax = node.max
			if ( boundingBoxMin[1] < totalBoundingBoxMin[1] ) then ( totalBoundingBoxMin[1] = boundingBoxMin[1] )
			if ( boundingBoxMin[2] < totalBoundingBoxMin[2] ) then ( totalBoundingBoxMin[2] = boundingBoxMin[2] )
			if ( boundingBoxMin[3] < totalBoundingBoxMin[3] ) then ( totalBoundingBoxMin[3] = boundingBoxMin[3] )
			if ( boundingBoxMax[1] > totalBoundingBoxMax[1] ) then ( totalBoundingBoxMax[1] = boundingBoxMax[1] )
			if ( boundingBoxMax[2] > totalBoundingBoxMax[2] ) then ( totalBoundingBoxMax[2] = boundingBoxMax[2] )
			if ( boundingBoxMax[3] > totalBoundingBoxMax[3] ) then ( totalBoundingBoxMax[3] = boundingBoxMax[3] )
		)
		totalBoundingBoxExtents = totalBoundingBoxMax - totalBoundingBoxMin
		rootPosition = totalBoundingBoxMin + (totalBoundingBoxExtents/2)
		rootName     = selection[1].name + "_CompoundRoot"

		dummyRootNode         = Dummy()
		dummyRootNode.name    = rootName
		dummyRootNode.pos     = rootPosition
		dummyRootNode.boxsize = totalBoundingBoxExtents

		masterNode = dummyRootNode
	)

	ancestorNode                   = undefined
	hkpRigidBodyModifierInAncestry = undefined
	hkdBodyModifierInAncestry      = undefined
	(
		ancestorNode = masterNode
		hkpRigidBodyModifierInAncestry = hvkPhysics_getRigidBodyModifier ancestorNode
		if ( hkpRigidBodyModifierInAncestry == undefined ) then
		(
			-- walk up the hierarchy unless we hit the root node or a node that holds an hkpRigidBody modifier
			while ( ancestorNode.parent != undefined AND hkpRigidBodyModifierInAncestry == undefined ) do
			(
				ancestorNode = ancestorNode.parent
				hkpRigidBodyModifierInAncestry = hvkPhysics_getRigidBodyModifier ancestorNode
			)
		)
		hkdBodyModifierInAncestry = hvkDestruction_getGenericModifierOfType ancestorNode "hkdBody"
	)

	if ( hkpRigidBodyModifierInAncestry == undefined ) then
	(
		hkpRigidBodyModifierInAncestry      = hvkPhysics_createRigidBody ancestorNode
		hkpRigidBodyModifierInAncestry.mass = 1.0
	)

	if ( hkdBodyModifierInAncestry == undefined ) then
	(
		hkdShapeModifierInAncestry = hvkDestruction_getGenericModifierOfType ancestorNode "hkdShape"
		if ( hkdShapeModifierInAncestry == undefined ) then
		(
			hkdShapeModifierInAncestry = hvkDestruction_createGenericModifier ancestorNode hkdShape
			if ( maxVersionNumber[1] >= 11000 ) then -- [HKD-159 has been fixed in 3ds Max 2009 and later only]
			(
				hkdShapeModifierInAncestry.strength = 1.0
			)
		)

		hvkDestruction_createGenericModifier ancestorNode hkdBody
	)

	constraintModifierTypes = hvkPhysics_getHavokConstraintModifierTypes()

	(
		for slaveNode in selection do
		(
			if ( slaveNode == masterNode ) then
			(
				continue
			)

			hkpRigidBodyModifier = hvkPhysics_getRigidBodyModifier slaveNode
			if ( hkpRigidBodyModifier == undefined ) then
			(
				hvkPhysics_createShape slaveNode
			)
			else
			(
				hvkDestruction_moveHavokConstraintModifiers constraintModifierTypes slaveNode ancestorNode
				deleteModifier slaveNode hkpRigidBodyModifier
				errorSlaveRigidBodyRemoved = 1
			)

			hkdBodyModifier = hvkDestruction_getGenericModifierOfType slaveNode "hkdBody"
			if ( hkdBodyModifier != undefined ) then
			(
				deleteModifier slaveNode hkdBodyModifier
				errorSlaveDestructionBodyRemoved = 1
			)

			hkdShapeModifier = hvkDestruction_getGenericModifierOfType slaveNode "hkdShape"
			if ( hkdShapeModifier == undefined ) then
			(
				hkdShapeModifier = hvkDestruction_createGenericModifier slaveNode hkdShape
				if ( maxVersionNumber[1] >= 11000 ) then -- [HKD-159 has been fixed in 3ds Max 2009 and later only]
				(
					hkdShapeModifier.strength = 1.0
				)
			)

			slaveNode.parent = masterNode
		)
	)

	if ( errorSlaveRigidBodyRemoved == 1 ) then
	(
		hvkDestruction_createErrorMessage &errorMsg "Removed hkpRigidBody modifier(s) from some of the slave nodes."
	)

	if ( errorSlaveDestructionBodyRemoved == 1 ) then
	(
		hvkDestruction_createErrorMessage &errorMsg "Removed hkdBody modifier(s) from some of the slave nodes."
	)

	if ( errorMsg != "" ) then
	(
		messagebox(errorMsg)
	)
)

