// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Objects/TaperedCapsule/hctTaperedCapsuleObject.h>
#include <ContentTools/Max/MaxSceneExport/Modifiers/ClothCollidable/hctClothCollidableModifier.h>

#define DEFAULT_TAPEREDCAPSULE_OBJECT_VERSION   0
#define CURRENT_TAPEREDCAPSULE_OBJECT_VERSION   7000

// The class descriptor for tapered capsule
class hkTaperedCapsuleClassDesc: public ClassDesc2
{
    public:
    int             IsPublic() { return TRUE; }
    void *          Create (BOOL loading = FALSE);
    const MCHAR *   ClassName() { return GetString(IDS_TAPEREDCAPSULE_CLASS_NAME); }
    SClass_ID       SuperClassID() { return GEOMOBJECT_CLASS_ID; }
    Class_ID        ClassID() { return HK_TAPEREDCAPSULE_CLASS_ID; }
    const MCHAR*    Category() { return GetString(IDS_HAVOK_HELPERS_CATEGORY); }
    const MCHAR*    InternalName() { return TEXT("hkTaperedCapsule"); }
    HINSTANCE       HInstance() { return hInstance; }
};

ClassDesc2* getHkTaperedCapsuleClassDesc()
{
    static hkTaperedCapsuleClassDesc taperedCapsuleDesc;
    return &taperedCapsuleDesc;
}


void* hkTaperedCapsuleClassDesc::Create (BOOL loading)
{
    hctTaperedCapsuleObject* newTaperedCapsule =  new hctTaperedCapsuleObject(loading);

    return newTaperedCapsule;
}

// per instance tapered capsule block
static ParamBlockDesc2 hkTaperedCapsule_ParamBlockDesc ( PB_TAPEREDCAPSULE_OBJ_PBLOCK, _T("hkTaperedCapsule"),  0, getHkTaperedCapsuleClassDesc(),
    P_AUTO_CONSTRUCT + P_AUTO_UI, PB_TAPEREDCAPSULE_OBJ_PBLOCK,
    //
    // Rollout
    IDD_TAPEREDCAPSULE_OBJECT_ROLLOUT_PROPERTIES, IDS_TAPEREDCAPSULE_OBJECT_ROLLOUT_PROPERTIES, 0, 0, NULL,

    // GENERAL PROPERTIES PARAMETERS
    // Start Radius
    PA_TAPEREDCAPSULE_OBJ_RADIUS,
        _T("radius"), TYPE_FLOAT, P_ANIMATABLE , IDS_TAPEREDCAPSULE_OBJECT_PA_RADIUS,
        p_default,      1.0f, // meters - we'll scale later in the constructor
        p_range,        0.0f, 99999999.0,
        p_ui,           TYPE_SPINNER, EDITTYPE_UNIVERSE,
                        IDC_ED_CAPSULE_RADIUS, IDC_SP_CAPSULE_RADIUS,
                        SPIN_AUTOSCALE,
        p_end,

    // End Radius
    PA_TAPEREDCAPSULE_OBJ_TAPER,
        _T("taper"), TYPE_FLOAT, P_ANIMATABLE , IDS_TAPEREDCAPSULE_OBJECT_PA_TAPER,
        p_default,      1.0f, // meters - we'll scale later in the constructor
        p_range,        0.0f, 99999999.0,
        p_ui,           TYPE_SPINNER, EDITTYPE_FLOAT,
                        IDC_ED_CAPSULE_TAPER, IDC_SP_CAPSULE_TAPER,
                        0.01f,
        p_end,

    // Height
    PA_TAPEREDCAPSULE_OBJ_HEIGHT,
        _T("height"), TYPE_FLOAT, P_ANIMATABLE , IDS_TAPEREDCAPSULE_OBJECT_PA_HEIGHT,
        p_default,      3.0f, // meters - we'll scale later in the constructor
        p_range,        0.0f, 99999999.0,
        p_ui,           TYPE_SPINNER, EDITTYPE_UNIVERSE,
                        IDC_ED_CAPSULE_HEIGHT, IDC_SP_CAPSULE_HEIGHT,
                        SPIN_AUTOSCALE,
        p_end,

    // Version: INTERNAL: Used to upgrade assets
    PA_TAPEREDCAPSULE_OBJ_VERSION_INTERNAL,
        _T(""), TYPE_INT,  P_RESET_DEFAULT | P_READ_ONLY, 0 ,
        p_default,  DEFAULT_TAPEREDCAPSULE_OBJECT_VERSION,
        p_end,

    p_end
    );



IObjParam* hctTaperedCapsuleObject::m_interface = NULL;

hctTaperedCapsuleObject::hctTaperedCapsuleObject(BOOL loading)
{
    getHkTaperedCapsuleClassDesc()->MakeAutoParamBlocks(this);
}

void hctTaperedCapsuleObject::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev)
{
    SimpleObject2::BeginEditParams(ip, flags, prev);

    this->m_interface = ip;

    // throw up all the appropriate auto-rollouts
    getHkTaperedCapsuleClassDesc()->BeginEditParams(ip, this, flags, prev);

}

void hctTaperedCapsuleObject::EndEditParams(IObjParam *ip, ULONG flags, Animatable *next)
{
    SimpleObject2::EndEditParams(ip, flags, next);
    this->m_interface = NULL;
    // tear down the appropriate auto-rollouts
    getHkTaperedCapsuleClassDesc()->EndEditParams(ip, this, flags, next);
}

// Return an inclusive range of linear values [phi0, phi1] given indices in inclusive range [i0, i1]
static inline double _interpPhi(double phi0, double phi1, int i, int i0, int i1)
{
    return phi0 + double(i - i0) * (phi1 - phi0) / double(i1 - i0);
}

void hctTaperedCapsuleObject::BuildMesh(TimeValue t)
{
    float radius;
    float taper;
    float height;
    // Start the validity interval at forever and whittle it down.
    ivalid = FOREVER;

    if(!OKtoDisplay(t))
    {
        return;
    }

    pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_RADIUS, t, radius, ivalid);
    pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_TAPER, t, taper, ivalid);
    pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_HEIGHT, t, height, ivalid);

    Point3 axis = Point3::ZAxis;
    short stacks = 2;
    const short slices = 16;

    double bigRadius;
    double smallRadius;

    double startRadius = radius;
    double endRadius = taper * startRadius;

    if (startRadius > endRadius)
    {
        bigRadius = startRadius;
        smallRadius = endRadius;
        axis *= -1.0;
    }
    else
    {
        bigRadius = endRadius;
        smallRadius = startRadius;
    }

    if (height <= bigRadius - smallRadius)
    {
        height = bigRadius - smallRadius + HK_REAL_EPSILON;
    }

    double TOL = 10.0 * double(HK_REAL_EPSILON);

    double radiusDiff = bigRadius - smallRadius + TOL;
    if (height < radiusDiff) height = radiusDiff;
    const double Delta = radiusDiff * bigRadius / height;
    //const double Delta = (bigRadius - smallRadius) * bigRadius / height;


    // Create the mesh data
    {
        Tab<Point3> slice;
        for( int i = 0; i < slices; ++i )
        {
            Point3 s( (cos(2*PI*i / slices)), (sin(2*PI*i / slices)), 0.0f);
            slice.Append(1, &s);
        }

        int completeHemisphereStacks = max(4, int(double(slices)/2 * bigRadius / (bigRadius + Delta)));

        int partialHemisphereStacks = max(3, int(double(slices)/2 * Delta / (bigRadius + Delta)));

        if (fabs(taper-1.f) < HK_REAL_EPSILON)
        {
            partialHemisphereStacks = 0;
        }

        const int capStacks = completeHemisphereStacks + partialHemisphereStacks;
        const int totalStacks = capStacks + stacks + capStacks;

        const int numPoints = (totalStacks + 1) * slices;

        Tab<Point3> points;
        for( int i = 0; i < numPoints; ++i )
        {
            Point3 p;
            points.Append(1, &p);
        }

        double sinTheta = (bigRadius - smallRadius) / height;
        double cosTheta = sqrt(1.0f - sinTheta*sinTheta);

        double theta = asin(sinTheta);
        double halfPi = 0.5f*PI;
        double halfPiPlusTheta = halfPi + theta;
        double halfPiMinusTheta = halfPi - theta;
        double dphi_big = theta / (double(partialHemisphereStacks) + TOL);
        double dphi_small = halfPiMinusTheta / capStacks;

        double h2 = 0.5f * height;
        double hmin = -h2 - smallRadius * sinTheta;
        double hmax = h2 - bigRadius * sinTheta;
        double rmax = bigRadius * cosTheta;
        double rmin = smallRadius * cosTheta;

        //Create the points
        {
            for( int i = 0; i < totalStacks+1; ++i )
            {
                double r;
                double h;

                // big cap hemisphere
                if( i < completeHemisphereStacks )
                {
                    double phi = _interpPhi(0.0, halfPi, i, 0, completeHemisphereStacks-1);
                    r = bigRadius * sin(phi);
                    h = h2 + bigRadius * cos(phi);
                }

                // big cap partial hemisphere
                else if ( i < capStacks )
                {
                    double phi = _interpPhi(halfPi+dphi_big, halfPiPlusTheta-dphi_big, i, completeHemisphereStacks, capStacks-1);
                    r = bigRadius * sin(phi);
                    h = h2 + bigRadius * cos(phi);
                }

                // cone
                else if( i < capStacks + stacks + 1)
                {
                    double alpha = double(i-capStacks) / stacks;
                    const double epsilon = 0.02f; // Needed to avoid degenerate triangles when radii are equal
                    double beta = epsilon + (1.0f - 2.0f*epsilon)* alpha;
                    r = beta * rmin + (1.0f - beta) * rmax;
                    h = beta * hmin + (1.0f - beta) * hmax;
                }

                // small cap hemisphere
                else
                {
                    int i0 = capStacks + stacks + 1;
                    int i1 = totalStacks;

                    double phi = _interpPhi(halfPiMinusTheta-dphi_small, 0.0f, i, i0, i1);
                    r = smallRadius * sin(phi);
                    h = -(h2 + smallRadius * cos(phi));
                }

                int cur = i*slices;
                for( int j = 0; j < slices; ++j )
                {
                    points[cur+j] = slice[j] * float(r);
                    points[cur+j].z = h*axis.z;

                }
            }
        }

        int numTris = totalStacks * slices * 2;

        mesh.setNumVerts(numPoints);
        mesh.setNumFaces(numTris);
        mesh.setSmoothFlags(false);
        mesh.setNumTVerts (0);
        mesh.setNumTVFaces (0);
        mesh.setSmoothFlags(1);

        //
        // Set the vertices
        //
        {
            for(int i = 0; i < numPoints; i++)
            {
                mesh.setVert (i, points[i]);
            }
        }

        //
        // Set the faces
        //
        {
            //
            // Set all edge flags
            //
            {
                for (int i=0; i< numTris; i++)
                {
                    mesh.faces[i].setEdgeVisFlags(1, 1, 1);
                }
            }

            for( int i = 0; i < totalStacks; ++i )
            {
                int st = i*slices;

                for( int j = 0; j < slices; ++j )
                {
                    int curTriIdx = (st + j)*2;

                    mesh.faces[curTriIdx].setSmGroup(1);
                    mesh.faces[curTriIdx].setMatID(0);
                    mesh.faces[curTriIdx].setVerts (st + j, (st + slices + (j+1)%slices), (st + (j+1)%slices) );

                    mesh.faces[curTriIdx + 1].setSmGroup(1);
                    mesh.faces[curTriIdx + 1].setMatID(0);
                    mesh.faces[curTriIdx + 1].setVerts (st + j, (st + slices + j) , (st + slices + (j+1)%slices));

                    if(axis.z < 0)
                    {
                        mesh.faces[curTriIdx].setVerts (st + j, (st + (j+1)%slices), (st + slices + (j+1)%slices));
                        mesh.faces[curTriIdx + 1].setVerts (st + j, (st + slices + (j+1)%slices), (st + slices + j));
                    }
                }
            }
        }
    }

    mesh.InvalidateTopologyCache();

}

BOOL hctTaperedCapsuleObject::OKtoDisplay(TimeValue t)
{
    if(pblock2)
    {
        float radius;
        float taper;

        pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_RADIUS, t, radius, FOREVER);
        pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_TAPER, t, taper, FOREVER);

        if((radius == 0.0f) || (taper == 0.0f))
        {
            return FALSE;
        }

        return TRUE;
    }

    return FALSE;
}

void hctTaperedCapsuleObject::UpdateUI()
{

}

void hctTaperedCapsuleObject::InvalidateUI()
{
    // if this was caused by a NotifyDependents from pblock2, LastNotifyParamID()
    // will contain ID to update, else it will be -1 => inval whole rollout
    hkTaperedCapsule_ParamBlockDesc.InvalidateUI(pblock2->LastNotifyParamID());
}

enum HIT_TYPE
{
    CONE_HIT,
    SMALLSPHERE_HIT,
    BIGSPHERE_HIT
};
struct TaperedCapsuleHit
{
    HIT_TYPE m_type;
    double m_hitFraction;
};

void sphereRayCast(const Point3& pprime, const Point3& p, const Point3& C, float R, bool& gotHit, double& at, Point3& norm)
{
    Point3 tmp = p - pprime;
    Point3 tmp2 = pprime - C;

    double a = LengthSquared(tmp);
    double b = double(2.0f) * DotProd(tmp, tmp2);
    double c = LengthSquared(tmp2) - double(R*R);
    double det = b * b - 4.0f * a * c;

    if (det < 0.0f)
    {
        gotHit = false;
        return;
    }

    gotHit = true;
    at = - ( b + Sqrt(det) ) / ( 2.0f * a + FLT_EPSILON );

    Point3 hitPoint = ((1 - float(at)) * pprime + (float(at) * p));
    norm = hitPoint - C;
    norm = norm.Normalize();
}

int hctTaperedCapsuleObject::IntersectRay(TimeValue t, Ray& ray, float& at, Point3& norm)
{
    Point3 smallSphereCentre, bigSphereCentre;

    double bigRadius;
    double smallRadius;
    float radius, taper, height;
    {
        pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_RADIUS, t, radius, ivalid);
            pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_TAPER, t, taper, ivalid);
        pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_HEIGHT, t, height, ivalid);

        double startRadius = radius;
        double endRadius = radius*taper;

        bigRadius = startRadius;
        smallRadius = endRadius;

        if(startRadius <= endRadius)
        {
            smallRadius = startRadius;
            bigRadius = endRadius;

            smallSphereCentre = Point3(0.0f, 0.0f, -height/2.0f);
            bigSphereCentre = Point3(0.0f, 0.0f, height/2.0f);
        }
        else if(startRadius > endRadius)
        {
            smallRadius = endRadius;
            bigRadius = startRadius;

            smallSphereCentre = Point3(0.0f, 0.0f, height/2.0f);
            bigSphereCentre = Point3(0.0f, 0.0f, -height/2.0f);
        }

    }

    const double realMax = 3.40282e+38f;
    const double r = double(smallRadius);
    const double R = double(bigRadius);
    const double l = double(height);
    const double H = R * ( l * l - (R - r) * (R - r) ) / ((R - r) * l);
    const double zmin = r * H / R;

    const double sinTheta = (R - r) / l;
    const double W = Sqrt(H * R * sinTheta);
    const double cosTheta = H / Sqrt(W*W + H*H);
    const double tanTheta = W / H;
    const double tanThetaSqr = tanTheta * tanTheta;

    Point3 coneAxis = bigSphereCentre - smallSphereCentre;
    coneAxis = coneAxis.Normalize();
    double d = r / sinTheta;
    Point3 coneApex = smallSphereCentre - coneAxis*float(d);

    const Point3& lhat = coneAxis;
    const Point3& V = coneApex;

    const Point3 p = ray.p + ((ray.dir)*float(realMax));
    const Point3 pprime = ray.p;

    Point3 n = p - pprime;

    double ndotlhat = DotProd(n, lhat);

    Point3 alpha;
    {
        Point3 tmp = pprime - V;
        tmp = CrossProd(tmp, lhat);
        alpha = CrossProd(lhat, tmp);
    }

    Point3 beta;
    {
        Point3 tmp = pprime - V;
        beta = lhat*DotProd(tmp, lhat);
    }


    // Solve quadratic to find intersection with cone portion
    double a = LengthSquared(n) - double(1.0f + tanTheta) * ndotlhat * ndotlhat;
    double b;
    {
        Point3 tmp = (alpha - beta)*float(tanThetaSqr);
        b = double(2.0f) * DotProd(n, tmp);
    }

    double c = LengthSquared(alpha) - double(tanThetaSqr) * LengthSquared(beta);
    double det = b * b - double(4.0f) * a * c;

    bool gotHit = false;

    TaperedCapsuleHit hits[3];
    hits[0].m_hitFraction = realMax;
    hits[1].m_hitFraction = realMax;
    hits[2].m_hitFraction = realMax;

    if ( det >= 0.0f )
    {
        // Cone portion was hit (should actually always be true, geometrically).
        const double half = double(0.5f);
        double hitFractionLow  = -half * ( b + Sqrt(det) ) / a ; // TO DO: consider preventing a == 0.0.
        double hitFractionHigh = -half * ( b - Sqrt(det) ) / a ;

        //Check if this is ok, setAddMul
        Point3 tmp = beta + n*float(hitFractionLow);
        double zlow = DotProd(tmp, lhat);

        //Check if this is ok, setAddMul
        tmp = beta + n*float(hitFractionHigh);
        double zhigh = DotProd(tmp, lhat);

        // We have to distinguish between hits with the positive-z and negative-z cone (only the former are wanted).
        double hitFractionCone = 0.0f;

        bool loHit = ( zlow  >= 0.0f );//&& hitFractionLow  >=0.0f );
        bool hiHit = ( zhigh >= 0.0f );//&& hitFractionHigh >=0.0f  );
        double zhit = -realMax;

        // If the first hit lies on upper cone, use that, otherwise if the second hit lies on the upper cone, use that.
        // If neither lies on the upper cone, there is no hit.
        if (loHit)
        {
            zhit = zlow;
            hitFractionCone = hitFractionLow;
        }
        else if (hiHit)
        {
            zhit = zhigh;
            hitFractionCone = hitFractionHigh;
        }

        // Check whether ray hits cone portion in valid range
        if ( zhit >= zmin && zhit <= H )
        {
            gotHit = true;
            hits[0].m_hitFraction = hitFractionCone;
            hits[0].m_type = CONE_HIT;
        }
    }

    // Check spheres
    Point3 smallSphere_normal, bigSphere_normal;
    {
        bool gotHitSmall, gotHitBig;
        sphereRayCast(pprime, p, smallSphereCentre, r, gotHitSmall, hits[1].m_hitFraction, smallSphere_normal);
        sphereRayCast(pprime, p, bigSphereCentre,   R, gotHitBig,   hits[2].m_hitFraction, bigSphere_normal);

        hits[1].m_type = SMALLSPHERE_HIT;
        hits[2].m_type = BIGSPHERE_HIT;
        gotHit = gotHit || gotHitSmall || gotHitBig;
    }

    int hitResult = 0;

    if (hitResult)
    {
        // Return the first intersection
        {
            TaperedCapsuleHit temp;
            if (hits[1].m_hitFraction < hits[0].m_hitFraction)
            {
                temp.m_hitFraction = hits[1].m_hitFraction;
                temp.m_type = hits[1].m_type;

                hits[1].m_hitFraction = hits[0].m_hitFraction;
                hits[1].m_type = hits[0].m_type;

                hits[0].m_hitFraction = temp.m_hitFraction;
                hits[0].m_type = temp.m_type;

            }
            if (hits[2].m_hitFraction < hits[1].m_hitFraction)
            {
                temp.m_hitFraction = hits[2].m_hitFraction;
                temp.m_type = hits[2].m_type;

                hits[2].m_hitFraction = hits[1].m_hitFraction;
                hits[2].m_type = hits[1].m_type;

                hits[1].m_hitFraction = temp.m_hitFraction;
                hits[1].m_type = temp.m_type;
            }
            if (hits[1].m_hitFraction < hits[0].m_hitFraction)
            {
                temp.m_hitFraction = hits[1].m_hitFraction;
                temp.m_type = hits[1].m_type;

                hits[1].m_hitFraction = hits[0].m_hitFraction;
                hits[1].m_type = hits[0].m_type;

                hits[0].m_hitFraction = temp.m_hitFraction;
                hits[0].m_type = temp.m_type;
            }
        }

        at = float(hits[0].m_hitFraction);
        hitResult = gotHit;


        // Compute hit normal
        switch (hits[0].m_type)
        {
            case SMALLSPHERE_HIT:
            {
                norm = smallSphere_normal;
            }
            break;

            case BIGSPHERE_HIT:
            {
                norm = bigSphere_normal;
            }
            break;

            case CONE_HIT:
            {
                Point3 relHitPoint = ((1 - at) * pprime + (at * p)) - V;

                Point3 xperphat = CrossProd(relHitPoint, lhat);
                xperphat = CrossProd(lhat, xperphat);
                xperphat = xperphat.Normalize();

                Point3 hitNormal = ((xperphat)*float(cosTheta)) - (lhat*float(sinTheta));
                hitNormal = hitNormal.Normalize();
                norm = hitNormal;
            }
            break;
        }
    }

    return hitResult;
}

RefTargetHandle hctTaperedCapsuleObject::Clone(RemapDir& remap)
{
    hctTaperedCapsuleObject* newob = new hctTaperedCapsuleObject(FALSE);
    newob->ReplaceReference(0, remap.CloneRef(pblock2));
    newob->ivalid.SetEmpty();
    BaseClone(this, newob, remap);

    return(newob);
}
Object* hctTaperedCapsuleObject::ConvertToType(TimeValue t, Class_ID obtype)
{
    return SimpleObject2::ConvertToType(t, obtype);
}

int hctTaperedCapsuleObject::CanConvertToType(Class_ID obtype)
{
    if (obtype==defObjectClassID || obtype==triObjectClassID)
    {
        return 1;
    }
    else
    {
        return SimpleObject2::CanConvertToType(obtype);
    }
}


void hctTaperedCapsuleObject::GetCollapseTypes(Tab<Class_ID> &clist, Tab<MSTR*> &nlist)
{
    Object::GetCollapseTypes(clist, nlist);
}





class TaperedCapsuleObjCreateCallBack : public CreateMouseCallBack
{

    private:
        IPoint2 m_sp0;
        IPoint2 m_sp1;
        IPoint2 m_sp2;

        hctTaperedCapsuleObject *m_taperedCapsuleObject;
        Point3 m_p0;
        Point3 m_p1;

    public:
        int proc( ViewExp *vpt, int msg, int point, int flags, IPoint2 m, Matrix3& mat);
        void setObj(hctTaperedCapsuleObject *obj) { m_taperedCapsuleObject = obj; }
};


int TaperedCapsuleObjCreateCallBack::proc(ViewExp *vpt, int msg, int point, int flags, IPoint2 m, Matrix3& mat )
{
    float radius;
    float taper;
    float height;

    Point3 d;
    //Point3 radiusPoint;

    if (msg == MOUSE_FREEMOVE)
    {
        vpt->SnapPreview(m, m, NULL, SNAP_IN_3D);
    }


    if (msg == MOUSE_POINT || msg == MOUSE_MOVE )
    {
        switch(point)
        {
            case 0:
                // only happens with MOUSE_POINT msg
                m_sp0 = m;
                m_taperedCapsuleObject->pblock2->SetValue(PA_TAPEREDCAPSULE_OBJ_HEIGHT, 0, 0.0f);

                m_p0 = vpt->SnapPoint(m, m, NULL, SNAP_IN_3D);
                m_p0.x = 0.0f;
                m_p0.y = 0.0f;
                m_p0.z = 0.0f;

                mat.SetTrans(m_p0);

                break;
            case 1:
                m_sp1 = m;

                m_p1 = vpt->SnapPoint(m, m, NULL, SNAP_IN_3D);
                m_p1.z = m_p0.z + (float)0.1;

                mat.SetTrans(m_p0);

                radius = Length( (m_p0 - m_p1) );
                m_taperedCapsuleObject->pblock2->SetValue(PA_TAPEREDCAPSULE_OBJ_RADIUS, 0, radius);

                hkTaperedCapsule_ParamBlockDesc.InvalidateUI();

                break;

            case 2:
                m_sp2 = m;
                m_p1.z = m_p0.z + vpt->SnapLength(vpt->GetCPDisp(m_p1,Point3(0,0,1), m_sp1, m));

                height = m_p1.z - m_p0.z;

                m_taperedCapsuleObject->pblock2->SetValue(PA_TAPEREDCAPSULE_OBJ_HEIGHT, 0, height);
                hkTaperedCapsule_ParamBlockDesc.InvalidateUI();

                break;

            case 3:
                d.x = vpt->SnapLength(vpt->GetCPDisp(m_p1, Point3(0,1,0), m_sp2, m));

                if (d.x < 0.0f)
                {
                    d.x = -d.x;
                }

                m_taperedCapsuleObject->pblock2->GetValue(PA_TAPEREDCAPSULE_OBJ_RADIUS, 0, radius, FOREVER);

                taper = d.x/radius;
                m_taperedCapsuleObject->pblock2->SetValue(PA_TAPEREDCAPSULE_OBJ_TAPER,0,taper);

                hkTaperedCapsule_ParamBlockDesc.InvalidateUI();

                if ( msg == MOUSE_POINT )
                {

                        return CREATE_STOP;
                }

                break;
        }
    }
    else if (msg == MOUSE_ABORT)
    {
        return CREATE_ABORT;
    }

    return TRUE;
}

static TaperedCapsuleObjCreateCallBack taperedCapsuleCreateCB;

CreateMouseCallBack* hctTaperedCapsuleObject::GetCreateMouseCallBack()
{
    taperedCapsuleCreateCB.setObj(this);
    return(&taperedCapsuleCreateCB);
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
