// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64
// PRODUCT      : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <sstream>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <ContentTools/Max/MaxSceneExport/resource.h>

#include <ContentTools/Max/MaxSceneExport/Modifiers/ScriptedFracture/hctScriptedFractureModifier.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.3dsmax"
#include <Common/Base/System/Log/hkLog.hxx>


//
//  Max description class

class hkScriptedFractureModifierDesc : public ClassDesc2
{
    public:

        int             IsPublic()                          {   return TRUE;                                    }
        const MCHAR *   ClassName()                         {   return _T("hkdScriptedFracture");               }
        SClass_ID       SuperClassID()                      {   return OSM_CLASS_ID;                            }
        Class_ID        ClassID()                           {   return HK_SCRIPTED_FRACTURE_MODIFIER_CLASS_ID;  }
        const MCHAR*    Category()                          {   return GetString(IDS_HAVOK_MODIFIERS_CATEGORY); }
        const MCHAR*    InternalName()                      {   return _T("hkdScriptedFracture");               }
        HINSTANCE       HInstance()                         {   return hInstance;                               }


        void* Create(BOOL loading = FALSE)
        {
            hctScriptedFractureModifier* newModifier =  new hctScriptedFractureModifier(this);

            if ( !loading )
            {
                // We need to make this a separate call (instead of putting it into the constructor),
                // as MAXScript crashes when we add a new parameter to the constructor. Reason unknown.
                newModifier->initUI();
            }

            return newModifier;
        }
};

ClassDesc2* gethkScriptedFractureModifierDesc()
{
    static hkScriptedFractureModifierDesc scriptedFractureModifierDesc;
    return &scriptedFractureModifierDesc;
}

//
//  This pointer is for the script helpers

ReferenceTarget* g_scriptSelf = HK_NULL;

//
//  Max script getter

Value* hctScriptedFractureModifier::hkScript_get()
{
    return MAXClass::make_wrapper_for( g_scriptSelf );
}

//
//  Max script setter

Value* hctScriptedFractureModifier::hkScript_set( Value* val )
{
    // We do nothing here. As this global variable is not exposed to any external MAXScript there shouldn't
    // be a need to set it from MAXScript.
    return val;
}

//
//  Constructor
//  Creates basic rollout for selecting a script file

hctScriptedFractureModifier::hctScriptedFractureModifier(ClassDesc2* theClassDesc)
:   hctBasicModifier( theClassDesc )
{
    // Create accessor for self
    g_scriptSelf = this;
    define_system_global(TEXT("hkScriptedFractureTransfer"), hkScript_get, hkScript_set);

    // Generate uid
    uid = rand();
}

//
//  Initializes the UI

void hctScriptedFractureModifier::initUI()
{
    hkStringBuf uidString;
    uidString.appendPrintf( "0x%08x", uid );

    // Add rollout
    M_STD_OSTRINGSTREAM execString;
    execString << TEXT("modifierParams = attributes \"hkdScriptedFracture_" << uidString.cString() << "\" attribID:#(175204460," << uidString.cString() << ")\n");
    execString << TEXT("(\n");
    execString << TEXT("\tparameters classInternalData\n");
    execString << TEXT("\t(\n");
    execString << TEXT("\t\tinternalClassVersion type:#integer default:1\n");
    execString << TEXT("\t)\n");
    execString << TEXT("\tparameters script rollout:script\n");
    execString << TEXT("\t(\n");
    execString << TEXT("\t\tscriptPath type:#string default:\"Select script\" ui:scriptPath\n");
    execString << TEXT("\t)\n");
    execString << TEXT("\trollout script \"Script\"\n");
    execString << TEXT("\t(\n");
    execString << TEXT("\t\t\tedittext scriptPath align:#left width:104 across:2\n");
    execString << TEXT("\t\t\tbutton browseScript \"...\" align:#right\n");
    execString << TEXT("\t\t\ton browseScript pressed do\n");
    execString << TEXT("\t\t\t(\n");
    execString << TEXT("\t\t\t\tfile = getOpenFileName caption:\"Open script\" types:\"Script(*.lua;*.hks)|*.lua;*.hks|All|*.*|\"\n");
    execString << TEXT("\t\t\t\tif ( file != undefined ) then\n");
    execString << TEXT("\t\t\t\t(\n");
    execString << TEXT("\t\t\t\t\tscriptPath.text = file\n");
    execString << TEXT("\t\t\t\t\tcurrentModifier = modpanel.getCurrentObject()\n");
    execString << TEXT("\t\t\t\t\thkScriptedFractureSelectScript file currentModifier\n");
    execString << TEXT("\t\t\t\t)\n");
    execString << TEXT("\t\t\t)\n");
    execString << TEXT("\t)\n");
    execString << TEXT(")\n");
    execString << TEXT("CustAttributes.add hkScriptedFractureTransfer modifierParams\n");
    M_STD_STRING execStringStr = execString.str();

    ExecuteMAXScriptScript(const_cast<MCHAR*>(execStringStr.c_str()));
}

//
//  Attempts to reset the UI

void hctScriptedFractureModifier::resetUI()
{
    ExecuteMAXScriptScript(TEXT("try( deleteItem hkScriptedFractureTransfer.custAttributes 2 ) catch ()\n"));
}

//
//  Destructor

hctScriptedFractureModifier::~hctScriptedFractureModifier()
{}

//
//  Runs the script, then creates the params rollout with the extracted dynamic parameters.

hkResult hctScriptedFractureModifier::parse( const char* scriptFileName )
{
    // Create accessor for self
    g_scriptSelf = this;
    define_system_global(TEXT("hkScriptedFractureTransfer"), hctScriptedFractureModifier::hkScript_get, hctScriptedFractureModifier::hkScript_set);

    // Attempt to resolve the path
    hkStringBuf scriptFilePath;
    {
        HCT_SCOPED_CONVERSIONS;

        MSTR pathStr = GetCOREInterface()->GetCurFilePath();
        if ( hctFilterUtils::resolvePath(scriptFileName, FROM_MAX(pathStr), scriptFilePath).isFailure() )
        {
            resetUI();
            HK_WARN_ALWAYS(0xabba5649, "Failed to open script file " << scriptFilePath << "!");
            return HK_FAILURE;
        }
    }

    // Attempt to load the script and resolve the params
    hkArray< hkStringMap<const char*>* > params;
    typedef hkResult (HK_CALL *FunPtr)(const char*, hkArray< hkStringMap<const char*>* >&);
    FunPtr getLuaScriptParameters = (FunPtr)hctSdkUtils::getUtilityFunction("getLuaScriptParameters");
    if ( !getLuaScriptParameters ||
         (getLuaScriptParameters(scriptFilePath.cString(), params).isFailure()) )
    {
        resetUI();
        HK_WARN_ALWAYS(0xabba5649, "Failed to parse global parameters from the script file " << scriptFilePath << "!");
        return HK_FAILURE;
    }

    try
    {
        std::stringstream execString;
        std::stringstream paramString;
        std::stringstream openString;

        hkStringBuf uidString;
        uidString.appendPrintf( "0x%08x", uid );

        // Exec header
        execString << "modifierParams = attributes \"hkdScriptedFracture_" << uidString.cString() << "\" attribID:#(195204460," << uidString.cString() << ")\n";
        execString << "(\n";
        execString << "\tparameters params rollout:params\n";
        execString << "\t(\n";

        // Param header
        paramString << "\trollout params \"Params\"\n";
        paramString << "\t(\n";

        // Create ui for all the dynamic parameters
        const int numParams = params.getSize();
        for ( int i=0; i<numParams; i++ )
        {
            const hkStringMap<const char*>* attributes = params[i];

            // Get id
            const char* id; attributes->get( "id", &id );

            // Get label
            const char *label = attributes->getWithDefault( "label", id );

            paramString << "\t\t\tlabel " << id << "Label \"" << label << "\" align:#left\n";

            // Get value
            const char* value; attributes->get( "value", &value );

            // Get type
            const char *type = attributes->getWithDefault( "type", "spinner" );

            // Check type
            if ( hkString::strCmp( type, "text" ) == 0 )
            {
                // Simple string
                execString << "\t\t" << id << " type:#string ui:" << id << " default:" << value << "\n";
                paramString << "\t\t\tedittext " << id << "\n";
            }
            else if ( hkString::strCmp( type, "spinner" ) == 0 )
            {
                // Float spinner
                execString << "\t\t" << id << " type:#float ui:" << id << " default:" << value << "\n";
                paramString << "\t\t\tspinner " << id << " type:#float align:#left fieldwidth:43";
                if ( attributes->hasKey( "min" ) && attributes->hasKey( "max" ) )
                {
                    const char *min; attributes->get( "min", &min );
                    const char *max; attributes->get( "max", &max );
                    paramString << " range:[" << min << "," << max << "," << value << "]";
                }
                paramString << "\n";
            }
            else if ( hkString::strCmp( type, "intspinner" ) == 0 )
            {
                // Integer spinner
                execString << "\t\t" << id << " type:#integer ui:" << id << " default:" << value << "\n";
                paramString << "\t\t\tspinner " << id << " type:#integer align:#left fieldwidth:43";
                if ( attributes->hasKey( "min" ) && attributes->hasKey( "max" ) )
                {
                    const char *min; attributes->get( "min", &min );
                    const char *max; attributes->get( "max", &max );
                    paramString << " range:[" << min << "," << max << "," << value << "]";
                }
                paramString << "\n";
            }
            else if ( hkString::strCmp( type, "checkbox" ) == 0 )
            {
                // Integer spinner
                execString << "\t\t" << id << " type:#boolean ui:" << id << " default:" << value << "\n";
                paramString << "\t\t\tcheckbox " << id << " align:#left checked:" << value << " fieldwidth:43\n";
            }
            else if ( hkString::strCmp( type, "enum" ) == 0 )
            {
                // Get items
                const char* items; attributes->get( "items", &items );

                // Combobox
                execString << "\t\t" << id << " type:#string default:\"" << value << "\"\n";
                paramString << "\t\t\tdropdownlist " << id << "_list align:#left items:" << items << " fieldwidth:43\n";

                // Handle selection
                paramString << "on " << id << "_list selected idx do\n";
                paramString << "(\n";
                paramString << "  " << id << " = " << id << "_list.items[idx]\n";
                paramString << ")\n";

                // Init
                openString << "  (\n";
                openString << "    local idx = findItem " << id << "_list.items " << id << "\n";
                openString << "    if ( idx != 0 ) do\n";
                openString << "    (\n";
                openString << "      " << id << "_list.selection = idx\n";
                openString << "    )\n";
                openString << "  )\n";
            }
            else if ( hkString::strCmp( type, "vector" ) == 0 )
            {
                // Vector spinner
                execString << "\t\t" << id << " type:#point3 default:" << value << "\n";
                paramString << "\t\t\tspinner " << id << "_x type:#float align:#left fieldwidth:42 across:3\n";
                paramString << "\t\t\tspinner " << id << "_y type:#float align:#left fieldwidth:42\n";
                paramString << "\t\t\tspinner " << id << "_z type:#float align:#left fieldwidth:42\n";
                // Param -> spinners
                paramString << "on " << id << " changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << "_x.value = val.x\n";
                paramString << "  " << id << "_y.value = val.y\n";
                paramString << "  " << id << "_z.value = val.z\n";
                paramString << ")\n";
                // Spinner X
                paramString << "on " << id << "_x changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".x = val\n";
                paramString << ")\n";
                // Spinner Y
                paramString << "on " << id << "_y changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".y = val\n";
                paramString << ")\n";
                // Spinner Z
                paramString << "on " << id << "_z changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".z = val\n";
                paramString << ")\n";
                // init
                openString << "  " << id << "_x.value = " << id << ".x\n";
                openString << "  " << id << "_y.value = " << id << ".y\n";
                openString << "  " << id << "_z.value = " << id << ".z\n";
            }
            else if ( hkString::strCmp( type, "quaternion" ) == 0 )
            {
                // Vector spinner
                execString << "\t\t" << id << " type:#quat default:" << value << "\n";
                paramString << "\t\t\tspinner " << id << "_x type:#float align:#left fieldwidth:42 across:3\n";
                paramString << "\t\t\tspinner " << id << "_y type:#float align:#left fieldwidth:42\n";
                paramString << "\t\t\tspinner " << id << "_z type:#float align:#left fieldwidth:42\n";
                paramString << "\t\t\tspinner " << id << "_w type:#float align:#left fieldwidth:42\n";
                // Param -> spinners
                paramString << "on " << id << " changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << "_x.value = val.x\n";
                paramString << "  " << id << "_y.value = val.y\n";
                paramString << "  " << id << "_z.value = val.z\n";
                paramString << "  " << id << "_w.value = val.w\n";
                paramString << ")\n";
                // Spinner X
                paramString << "on " << id << "_x changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".x = val\n";
                paramString << ")\n";
                // Spinner Y
                paramString << "on " << id << "_y changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".y = val\n";
                paramString << ")\n";
                // Spinner Z
                paramString << "on " << id << "_z changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".z = val\n";
                paramString << ")\n";
                // Spinner W
                paramString << "on " << id << "_w changed val do\n";
                paramString << "(\n";
                paramString << "  " << id << ".w = val\n";
                paramString << ")\n";
                // init
                openString << "  " << id << "_x.value = " << id << ".x\n";
                openString << "  " << id << "_y.value = " << id << ".y\n";
                openString << "  " << id << "_z.value = " << id << ".z\n";
                openString << "  " << id << "_w.value = " << id << ".w\n";
            }
            else if ( hkString::strCmp( type, "node" ) == 0 )
            {
                // Node
                execString << "\t\t" << id << " type:#string ui:" << id << " default:\"Undefined\"\n";
                paramString << "\t\t\tedittext " << id << " align:#left readonly:true height:21 width:98 across:4\n";
                paramString << "\t\t\tbutton " << id << "FollowButton \">\" align:#right width:13 offset:[43,0]\n";
                paramString << "\t\t\tpickbutton " << id << "PickButton \"...\" align:#right width:13 offset:[23,0]\n";
                paramString << "\t\t\tbutton " << id << "UnlinkButton \"X\" align:#right width:13 offset:[3,0]\n";
                // on follow
                paramString << "on " << id << "FollowButton pressed do\n";
                paramString << "(\n";
                paramString << "  if ( " << id << "PickButton.object != undefined ) then\n";
                paramString << "  (\n";
                paramString << "    select " << id << "PickButton.object\n";
                paramString << "  )\n";
                paramString << ")\n";
                // on pick
                paramString << "on " << id << "PickButton picked obj do\n";
                paramString << "(\n";
                paramString << "  " << id << "PickButton.object = hvkDestruction_verifyMeshLink " << id << "PickButton.object\n";
                paramString << "  " << id << ".text = " << id << "PickButton.object.name\n";
                paramString << ")\n";
                // on unlink
                paramString << "on " << id << "UnlinkButton pressed do\n";
                paramString << "(\n";
                paramString << "  " << id << "PickButton.object = undefined\n";
                paramString << "  " << id << ".text = \"Undefined\"\n";
                paramString << ")\n";
            }
        }

        // Exec footer
        execString << "\t)\n";

        // Add open string to params
        if ( openString.str().length() > 0 )
        {
            paramString << "on params open do\n";
            paramString << "(\n";
            paramString << openString.str();
            paramString << ")\n";
        }

        // Param footer
        paramString << "\t)\n";

        // Add params to exec
        execString << paramString.str().c_str();

        // Finish exec
        execString << ")\n";
        execString << "try( deleteItem hkScriptedFractureTransfer.custAttributes 2 ) catch ()\n";
        execString << "CustAttributes.add hkScriptedFractureTransfer modifierParams\n";
        std::string execStringStr = execString.str();

        HCT_SCOPED_CONVERSIONS;

        const MCHAR* maxStr = TO_MAX(execStringStr.c_str());

        ExecuteMAXScriptScript(CONST15_CAST(maxStr));
    }
    catch(...)
    {
        Log_Error( "ScriptedFracture Params exception!" );
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

//  Max delegate for parsing a script

def_visible_primitive( hkScriptedFractureSelectScript, "hkScriptedFractureSelectScript" );
Value* hkScriptedFractureSelectScript_cf( Value** arg_list, int count )
{
    check_arg_count(hkScriptedFractureSelectScript, 2, count);
    type_check(arg_list[0], String, TEXT("hkScriptedFractureSelectScript scriptPath"));

    HCT_SCOPED_CONVERSIONS;

    MSTR maxScriptPath = arg_list[0]->to_string();
    const hkStringBuf scriptPath(FROM_MAX(maxScriptPath));

    hctScriptedFractureModifier* self = static_cast<hctScriptedFractureModifier*>(arg_list[1]->to_reftarg());
    const hkResult ret = self->parse( scriptPath.cString() );
    Value* v = (ret.isSuccess()) ? &true_value : &false_value;
    return v;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
