// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <ContentTools/Max/MaxSceneExport/Modifiers/Generic/hctGenericModifierUtil.h>
#include <ContentTools/Common/Filters/Common/Utils/hctLocaleScope.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>


void hctGenericModifierUtil::addCustomAttributesToModifier(int modifierId)
{
    hctLocaleScope scope;

    HCT_SCOPED_CONVERSIONS;

    //
    // Build a custom MAXScript from the collected data and execute it. This will add all the class's members as
    // custom attributes to the modifier.
    //
    {
        std::stringstream execString;
        hctGenericModifierUtil::createCustomAttributesDefinition(modifierId, execString);
        execString << "CustAttributes.add hkCurrentGenericModifierTempTransfer modifierParams\n";
        std::string execStringStr = execString.str();
        const MCHAR* maxStr = TO_MAX(execStringStr.c_str());
        ExecuteMAXScriptScript(CONST15_CAST(maxStr));

#if 0
        // Debug: dump the generated MAXScript to a file.
        // Warning: enabling this block can cause a 'Privileged Instruction' error when creating the
        //          modifier from the Modifier dropdown list.
        {
            FileStream filestreamTest;
            filestreamTest.open("c:\\hkModifierMAXScriptDebugDump.ms", "w");
            filestreamTest.printf(execString.str().c_str());
            filestreamTest.close();
        }
#endif

#if 0
        // DEBUG: Execute the specified MAXScript file.
        {
            FileStream filestream;
            filestream.open("c:\\hkModifierMAXScriptDebugDump.ms", "r");
            bool result;
            ExecuteScript(&filestream, &result);
            filestream.close();
        }
#endif
    }
}


void hctGenericModifierUtil::createCustomAttributesDefinition(int modifierId, std::stringstream& execString)
{
    const hkReflect::Type* klass = hctSdkUtils::getUiClass(modifierId).m_class;

    //
    // Find out if this node supports documentation.
    //
    bool docsAvailable = false;
    {
        if( const hkDocumentationAttribute* documentationAttribute = klass->findAttribute<hkDocumentationAttribute>() )
        {
            const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAX);
            const char* docsSectionTag = documentationAttribute->m_getDocsSectionTag(hideCriteria);
            if ( docsSectionTag && docsSectionTag[0] != 0 )
            {
                docsAvailable = true;
            }
        }
    }

    //
    // Create a hierarchy from all the class's members (taking GROUP tagging and structs into account).
    // This will for now include INVISIBLE members as well. Those will get removed later on below right
    // before we build the actual GUI creation code.
    //
    std::string groupPathDelimiter("_"); // sync id: <0xaf1ee232>
    hkReal unitConversionModifier = hkReal(GetMasterScale(UNITS_METERS)); // i.e. metersPerUnit
    std::vector<hctClassHierarchyUtil::ParamGroup*> groupsArray;
    hctClassHierarchyUtil::createDisplayHierarchyFromClass(klass, groupPathDelimiter, unitConversionModifier, groupsArray);

    //
    // Output MAXScript header.
    //
    {
        hkStringOld attributeName("");
        //attributeName.printf("ModifierID%.4d", modifierId);
        attributeName.printf("%s", klass->getName());

        hkStringOld attribIdString("");
        attribIdString.printf("attribID:#(%d,%d)", 0xaf14+modifierId, 0xaf70+modifierId);

        execString << "modifierParams = attributes \"" << attributeName.cString() << "\" " << attribIdString.cString() << "\n";
        execString << "(\n";
    }

    //
    // Output class internal data block as Custom Attributes.
    //
    {
        execString << "\tparameters classInternalData\n";
        execString << "\t(\n";
        execString << "\t\tinternalClassVersion type:#integer default:" << klass->getVersion() << "\n";
        execString << "\t)\n";
    }

    // Query UI scheme and set additional hide criteria
    const hkAttributeHideCriteria::Types hideCriteria = hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::MODELER_IS_MAX);

    //
    // Output the definition for Custom Attributes.
    //
    {
        for (unsigned int i = 0; i < groupsArray.size(); i++)
        {
            hctClassHierarchyUtil::ParamGroup* group = groupsArray[i];
            buildParametersBlockForGroup(hideCriteria, group, execString);
        }
    }

    //
    // Remove all VISIBLE members from hierarchies.
    //
    hctClassHierarchyUtil::pruneInvisibleMembers(hideCriteria, groupsArray);

    //
    // Remove all empty subtrees from hierarchies.
    //
    hctClassHierarchyUtil::pruneEmptyGroups(groupsArray);

    //
    // Create a dedicated HELP rollout.
    //
    execString << "\n";
    if ( docsAvailable )
    {
        buildHelpRollout(klass, execString);
    }

    //
    // Add a disclaimer if a certain feature is currently under review.
    //
    if (hctSdkUtils::getUiClass(modifierId).m_internalClassification == hctModelerNodeIdToClassMap::UNDERGOING_REVIEW)
    {
        buildUnderReviewNotificationRollout(klass, execString);
    }

    //
    // Output the MAXScript GUI definition for Custom Attributes.
    //
    {
        for (unsigned int i = 0; i < groupsArray.size(); i++)
        {
            hctClassHierarchyUtil::ParamGroup* group = groupsArray[i];
            buildRolloutBlockForGroup(group, execString);
        }
    }

    //
    // Output MAXScript footer.
    //
    {
        execString << ")\n";
    }
}


// ================================================================================================
// PARAMETERS BLOCK
// ================================================================================================

void hctGenericModifierUtil::buildParametersBlockForGroup(hkAttributeHideCriteria::Types hideCriteria, const hctClassHierarchyUtil::ParamGroup* group, std::stringstream& execString)
{
    //
    // Recursively build the parameters list from the group (and all of its subgroups).
    //
    std::stringstream indent; indent << "\t\t";
    std::stringstream paramBlockParametersString;
    std::stringstream gizmoCollectorString;
    int numVisibleParameters = 0;
    buildParametersBlockForGroupRecursively(hideCriteria, group, indent, paramBlockParametersString, gizmoCollectorString, numVisibleParameters);

    //
    // Build the complete block incl. header and parentheses.
    //
    std::stringstream paramBlockString;
    {
        paramBlockString << "\tparameters " << group->m_groupInternalName;

        if ( numVisibleParameters > 0 )
        {
            paramBlockString << " rollout:" << group->m_groupInternalName;
        }

        paramBlockString << "\n";
        paramBlockString << "\t(\n";
        paramBlockString << paramBlockParametersString.str();

        if ( gizmoCollectorString.str().length() && (gizmoCollectorString.str()[0] != 0) )
        {
            paramBlockString << "\t\t" << group->m_groupInternalName << "HkGizmos type:#string default:\"" << gizmoCollectorString.str() << "\"\n";
        }

        paramBlockString << "\t)\n";
    }

    //
    // Append the complete parameters block to the output string.
    //
    execString << paramBlockString.str();
}


void hctGenericModifierUtil::buildParametersBlockForGroupRecursively(hkAttributeHideCriteria::Types hideCriteria, const hctClassHierarchyUtil::ParamGroup* group, const std::stringstream& indent, std::stringstream& groupParamBlockString, std::stringstream& gizmoCollectorString, int& numVisibleParameters)
{
    {
        for (int i = 0; i < int(group->m_members.size()); i++)
        {
            const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData = group->m_members[i];

            hkReflect::FieldDecl        member              = memberData.m_member;
            const hkUiAttribute*        uiAttribute         = member.getType()->findAttribute<hkUiAttribute>();

            hkStringOld    validNameBuffer;
            const bool isHidden         = uiAttribute && (uiAttribute->m_hideCriteria & hideCriteria);
            hkBool      isVisible       = memberData.m_visible && !isHidden;
            const char* internalName    = hctSdkUtils::getValidName(memberData.m_memberInternalName.c_str(), validNameBuffer);

            if ( isVisible )
            {
                numVisibleParameters++;
            }

            struct BuildParamVisitor : public hkReflect::TypeVisitor<BuildParamVisitor, void, std::stringstream&>
            {
                BuildParamVisitor(bool isVisible, const char* internalName,
                    const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData,
                    const std::stringstream& indent, std::stringstream& gizmoCollectorString)
                    : isVisible(isVisible)
                    , internalName(internalName)
                    , memberData(memberData)
                    , indent(indent)
                    , gizmoCollectorString(gizmoCollectorString)
                {
                }

                bool isVisible;
                const char* internalName;
                const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData;
                const std::stringstream& indent;
                std::stringstream& gizmoCollectorString;

                void visit(const hkReflect::VoidType*, std::stringstream&) { HK_UNREACHABLE(0x1b282f91, "Void field found" ); }
                void visit(const hkReflect::ArrayType*, std::stringstream&) { HK_WARN(0x78872830, "Field should not be an array"); }
                void visit(const hkReflect::RecordType*, std::stringstream&) { HK_WARN(0x77ffea2b, "Field should not be a record"); }

                void visit(const hkReflect::IntType* type, std::stringstream& groupParamBlockString)
                {
                    groupParamBlockString << indent.str() << internalName << " type:#integer default:";
                    if (type->findAttribute<hk::Presets>())
                    {
                        int enumDefault = atoi(memberData.m_defaultValue.c_str()) + 1;
                        groupParamBlockString << enumDefault;
                    }
                    else
                    {
                        groupParamBlockString << memberData.m_defaultValue;
                    }
                    if ( isVisible )
                    {
                        groupParamBlockString << " ui:" << internalName;
                    }
                    groupParamBlockString << "\n";
                }

                void visit(const hkReflect::FloatType* type, std::stringstream& groupParamBlockString)
                {
                    groupParamBlockString << indent.str() << internalName << " type:#float default:" << memberData.m_defaultValue;
                    if ( isVisible )
                    {
                        groupParamBlockString << " ui:" << internalName;

                        const hkGizmoAttribute*     gizmoAttribute      = type->findAttribute<hkGizmoAttribute>();
                        if ( gizmoAttribute )
                        {
                            // add on/off functionality
                            HK_ASSERT(0x736400ba, gizmoAttribute->m_type == hkGizmoAttribute::SPHERE, "gizmo does not work with this variable type\n");
                            gizmoCollectorString << internalName << ":SPHERE ";
                        }
                    }
                    groupParamBlockString << "\n";
                }

                void visit(const hkReflect::BoolType* type, std::stringstream& groupParamBlockString)
                {
                    groupParamBlockString << indent.str() << internalName << " type:#integer default:" << memberData.m_defaultValue;
                    if ( isVisible )
                    {
                        groupParamBlockString << " ui:" << internalName;
                    }
                    groupParamBlockString << "\n";
                }

                void visitUuid(std::stringstream& groupParamBlockString)
                {
                    groupParamBlockString << indent.str() << internalName << " type:#string default:\"" << memberData.m_defaultValue << "\"";
                    if ( isVisible )
                    {
                        groupParamBlockString << " ui:" << internalName;
                    }
                    groupParamBlockString << "\n";
                }

                void visit(const hkReflect::StringType* type, std::stringstream& groupParamBlockString)
                {
                    groupParamBlockString << indent.str() << internalName << " type:#string default:\"" << memberData.m_defaultValue << "\"";
                    if ( isVisible )
                    {
                        groupParamBlockString << " ui:" << internalName;
                    }
                    groupParamBlockString << "\n";
                }

                void visitVector4(const hkReflect::Type* member, std::stringstream& groupParamBlockString)
                {
                    const hkGizmoAttribute*     gizmoAttribute = member->findAttribute<hkGizmoAttribute>();
                    groupParamBlockString << indent.str() << internalName << " type:#point3 default:[" << memberData.m_defaultValue << "]\n";
                    if ( isVisible )
                    {
                        if ( gizmoAttribute )
                        {
                            // add on/off functionality
                            gizmoCollectorString << internalName;
                            switch (gizmoAttribute->m_type)
                            {
                                case hkGizmoAttribute::PLANE:
                                {
                                    gizmoCollectorString << ":PLANE ";
                                    break;
                                }

                                case hkGizmoAttribute::ARROW:
                                {
                                    gizmoCollectorString << ":ARROW ";
                                    break;
                                }

                                default:
                                {
                                    HK_ASSERT(0x736400ba, 0, "gizmo does not work with this variable type\n");
                                    break;
                                }
                            }
                        }
                    }
                }

                void visit(const hkReflect::PointerType* type, std::stringstream& groupParamBlockString)
                {
                    const hkLinkAttribute*      linkAttribute = type->findAttribute<hkLinkAttribute>();
                    if ( !linkAttribute || linkAttribute->m_type == hkLinkAttribute::NONE )
                    {
                        return;
                    }

                    if ( (linkAttribute->m_type == hkLinkAttribute::MESH) || (linkAttribute->m_type == hkLinkAttribute::NODE_UUID) )
                    {
                        groupParamBlockString << indent.str() << internalName << " type:#node";
                        if ( isVisible )
                        {
                            groupParamBlockString << " ui:" << internalName;
                        }
                        groupParamBlockString << "\n";
                    }
                    else
                    {
                        groupParamBlockString << indent.str() << internalName << " type:#maxObject\n";
                    }
                }
            } visitor(isVisible, internalName, memberData, indent, gizmoCollectorString);

            if (member.getType()->equals<hkUuid>())
            {
                visitor.visitUuid(groupParamBlockString);
            }
            else if (member.getType()->equals<hkVector4>())
            {
                visitor.visitVector4(member.getType(), groupParamBlockString);
            }
            else
            {
                visitor.dispatch(member.getType(), groupParamBlockString);
            }
        }
    }

    //
    // Recursively add members from whole subtree. Here in the Params block all members of the subtree will be flattened.
    //
    {
        for (unsigned int i = 0; i < group->m_subGroups.size(); i++)
        {
            hctClassHierarchyUtil::ParamGroup* subGroup = group->m_subGroups[i];
            buildParametersBlockForGroupRecursively(hideCriteria, subGroup, indent, groupParamBlockString, gizmoCollectorString, numVisibleParameters);
        }
    }
}


// ================================================================================================
// ROLLOUT BLOCK
// ================================================================================================

void hctGenericModifierUtil::buildRolloutBlockForGroup(const hctClassHierarchyUtil::ParamGroup* group, std::stringstream& execString)
{
    //
    // Recursively build the parameters list from the group (and all of its subgroups).
    //
    std::stringstream indent; indent << "\t";
    std::stringstream rolloutBlockParametersString;
    std::stringstream onRolloutOpenString;
    std::stringstream onDataChangedString;
    int labelCtr = 0;
    buildRolloutBlockForGroupRecursively(group, false, indent, rolloutBlockParametersString, onRolloutOpenString, onDataChangedString, 2, labelCtr);

    if ( onRolloutOpenString.str().length() && (onRolloutOpenString.str()[0] != 0) )
    {
        rolloutBlockParametersString << "\t\t" << "on " << group->m_groupInternalName << " open do\n";
        rolloutBlockParametersString << "\t\t" << "(\n";
        rolloutBlockParametersString << onRolloutOpenString.str();
        rolloutBlockParametersString << "\t\t" << ")\n";
    }

    if ( onDataChangedString.str().length() && (onDataChangedString.str()[0] != 0) )
    {
        rolloutBlockParametersString << onDataChangedString.str();
    }


    //
    // Build the complete block incl. header and parentheses.
    //
    std::stringstream rolloutBlockString;
    {
        rolloutBlockString << "\trollout " << group->m_groupInternalName << " \"" << group->m_groupLabel << "\"\n";
        rolloutBlockString << "\t(\n";
        rolloutBlockString << rolloutBlockParametersString.str();
        rolloutBlockString << "\t)\n";
    }

    //
    // Append the complete parameters block to the output string.
    //
    execString << rolloutBlockString.str();
}


void hctGenericModifierUtil::buildRolloutBlockForGroupRecursively(const hctClassHierarchyUtil::ParamGroup* group, bool isSubGroup, const std::stringstream& indentIn, std::stringstream& rolloutParamBlockString, std::stringstream& onRolloutOpenString, std::stringstream& onDataChangedString, int indentLevel, int& labelCtr)
{
    // Advance indent.
    std::stringstream indent;
    indent << indentIn.str() << "\t";

    {
        for (int i = 0; i < int(group->m_members.size()); i++)
        {
            const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData = group->m_members[i];

            hkReflect::FieldDecl member = memberData.m_member;

            hkStringOld    validNameBuffer;
            const char* internalName = hctSdkUtils::getValidName(memberData.m_memberInternalName.c_str(), validNameBuffer);
            const char* label        = memberData.m_memberLabel.c_str();

            struct BuildRolloutVisitor : public hkReflect::TypeVisitor<BuildRolloutVisitor, void, std::stringstream&>
            {
                BuildRolloutVisitor(bool isSubGroup, const std::stringstream& indent, const char* internalName,
                    const char* label, const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData,
                    std::stringstream& onRolloutOpenString, std::stringstream& onDataChangedString, int& labelCtr)
                    : isSubGroup(isSubGroup)
                    , indent(indent)
                    , internalName(internalName)
                    , label(label)
                    , memberData(memberData)
                    , onRolloutOpenString(onRolloutOpenString)
                    , onDataChangedString(onDataChangedString)
                    , labelCtr(labelCtr)
                {
                }

                bool isSubGroup;
                const std::stringstream& indent;
                const char* internalName;
                const char* label;
                const hctClassHierarchyUtil::ParamGroup::ClassAndMember& memberData;
                std::stringstream& onRolloutOpenString;
                std::stringstream& onDataChangedString;
                int& labelCtr;

                void visit(const hkReflect::VoidType*, std::stringstream&) { HK_UNREACHABLE(0x2fd722af, "Void field found" ); }
                void visit(const hkReflect::ArrayType*, std::stringstream&) { HK_WARN(0x38d38283, "Field should not be an array"); }
                void visit(const hkReflect::RecordType*, std::stringstream&) { HK_WARN(0x7b8941db, "Field should not be a record"); }

                void visitUuid(std::stringstream& rolloutParamBlockString)
                {
                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << internalName << "Label \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-5,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[-10,0]\n"; }
                        }

                        // 'Edittext' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "edittext " << internalName << " align:#left fieldwidth:156 ";
                            if ( !memberData.m_editable )   {   rolloutParamBlockString << " readOnly:true ";   }
                            if ( isSubGroup )               {   rolloutParamBlockString << "offset:[-19,0]\n";  }
                            else                            {   rolloutParamBlockString << "offset:[-14,0]\n";  }
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visit(const hkReflect::IntType* type, std::stringstream& rolloutParamBlockString)
                {
                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << internalName << "Label \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-3,1] "; }
                            else              { rolloutParamBlockString << "offset:[-8,1] "; }
                            rolloutParamBlockString << "across:2\n";
                        }

                        // 'Spinner' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "spinner " << internalName << " type:#integer align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "fieldwidth:39 "; }
                            else              { rolloutParamBlockString << "fieldwidth:43 "; }

                            if (const hkRangeInt32Attribute* ranges = type->findAttribute<hkRangeInt32Attribute>())
                            {
                                hkStringOld RangesString;
                                RangesString.printf("range:[%d,%d,%s] ", ranges->m_absmin, ranges->m_absmax, memberData.m_defaultValue.c_str());
                                rolloutParamBlockString << RangesString.cString();
                            }

                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[21,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[22,0]\n"; }
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visit(const hkReflect::FloatType* type, std::stringstream& rolloutParamBlockString)
                {
                    const hkSemanticsAttribute* semanticsAttributes = type->findAttribute<hkSemanticsAttribute>();
                    bool useWorldUnits = semanticsAttributes && (semanticsAttributes->m_type == hkSemanticsAttribute::DISTANCE);

                    hkReal unitConversionModifier = 1.0f;
                    if ( useWorldUnits )
                    {
                        unitConversionModifier = hkReal(GetMasterScale(UNITS_METERS)); // i.e. metersPerUnit
                    }

                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << internalName << "Label \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-3,1] "; }
                            else              { rolloutParamBlockString << "offset:[-8,1] "; }
                            rolloutParamBlockString << "across:2\n";
                        }

                        // 'Spinner' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "spinner " << internalName;
                            if ( useWorldUnits ) { rolloutParamBlockString << " type:#worldunits "; }
                            else                 { rolloutParamBlockString << " type:#float ";      }
                            rolloutParamBlockString << "align:#left ";
                            if ( isSubGroup )    { rolloutParamBlockString << "fieldwidth:39 "; }
                            else                 { rolloutParamBlockString << "fieldwidth:43 "; }

                            if (const hkRangeRealAttribute* ranges = type->findAttribute<hkRangeRealAttribute>())
                            {
                                hkStringOld RangesString;
                                RangesString.printf("range:[%f,%f,%s] ", ranges->m_absmin/unitConversionModifier, ranges->m_absmax/unitConversionModifier, memberData.m_defaultValue.c_str());
                                rolloutParamBlockString << RangesString.cString();
                            }

                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[21,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[22,0]\n"; }
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visit(const hkReflect::BoolType* type, std::stringstream& rolloutParamBlockString)
                {
                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Checkbox' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "checkbox " << internalName << " \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-5,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[-8,0]\n"; }
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visit(const hkReflect::StringType* type, std::stringstream& rolloutParamBlockString)
                {
                    const hkLinkAttribute*      linkAttribute = type->findAttribute<hkLinkAttribute>();
                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << internalName << "Label \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-5,0]\n";  }
                            else              { rolloutParamBlockString << "offset:[-10,0]\n"; }
                        }

                        // 'Edittext' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "edittext " << internalName << " align:#left ";
                            if ( linkAttribute && linkAttribute->m_type == hkLinkAttribute::IMGSELECT )
                            {
                                //rolloutParamBlockString << "fieldwidth:131 readonly:true across:2 ";
                                rolloutParamBlockString << "fieldwidth:131 across:2 ";
                            }
                            else
                            {
                                rolloutParamBlockString << "fieldwidth:156 ";
                            }
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-19,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[-14,0]\n"; }
                        }

                        // Optionnally pick button
                        if ( linkAttribute && linkAttribute->m_type == hkLinkAttribute::IMGSELECT )
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "mapbutton " << internalName << "Pick \"...\" tooltip:\"Select crack lines image\" align:#right width:13 height:19 offset:[5,0]\n";
                            {
                                onDataChangedString << indent.str() << "on " << internalName << "Pick picked obj do\n";
                                onDataChangedString << indent.str() << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t" << "if ( classof obj as string == \"Bitmaptexture\" ) then\n";
                                    onDataChangedString << indent.str() << "\t(\n";
                                    onDataChangedString << indent.str() << "\t\t" << internalName << ".text = mapPaths.getFullFilePath obj.filename\n";
                                    onDataChangedString << indent.str() << "\t)\n";
                                    onDataChangedString << indent.str() << "\telse\n";
                                    onDataChangedString << indent.str() << "\t(\n";
                                    onDataChangedString << indent.str() << "\t\tmessagebox \"Please select a bitmap object\"\n";
                                    onDataChangedString << indent.str() << "\t)\n";
                                }
                                onDataChangedString << indent.str() << ")\n";
                            }
                        }

                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visitVector4(const hkReflect::Type* member, std::stringstream& rolloutParamBlockString)
                {
                    const hkSemanticsAttribute* semanticsAttributes = member->findAttribute<hkSemanticsAttribute>();
                    bool useWorldUnits = semanticsAttributes && (semanticsAttributes->m_type == hkSemanticsAttribute::POSITION);

                    hkReal unitConversionModifier = 1.0f;
                    if ( useWorldUnits )
                    {
                        unitConversionModifier = hkReal(GetMasterScale(UNITS_METERS)); // i.e. metersPerUnit
                    }

                    hkVector4 defaultValues;
                    hkReflect::Var typeDefaultValue = member->getDefault();
                    if (!typeDefaultValue)
                    {
                        defaultValues.setZero4();
                    }
                    else
                    {
                        defaultValues.set(*typeDefaultValue.dynCast<hkVector4>());
                        defaultValues.mul4( 1.0f / unitConversionModifier );
                    }

                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << internalName << "Label \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-4,0]\n"; }
                            else              { rolloutParamBlockString << "offset:[-8,0]\n"; }
                        }

                        hkReal absMinValue = -1000.0f;
                        hkReal absMaxValue =  1000.0f;
                        if (const hkRangeRealAttribute* ranges = member->findAttribute<hkRangeRealAttribute>())
                        {
                            absMinValue = ranges->m_absmin / unitConversionModifier;
                            absMaxValue = ranges->m_absmax / unitConversionModifier;
                        }

                        // X 'Spinner' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "spinner " << internalName << "X align:#left across:3 ";

                            if ( isSubGroup )    { rolloutParamBlockString << "fieldwidth:39 offset:[-5,0] ";  }
                            else                 { rolloutParamBlockString << "fieldwidth:42 offset:[-10,0] "; }

                            if ( useWorldUnits ) { rolloutParamBlockString << "type:#worldunits "; }
                            else                 { rolloutParamBlockString << "type:#float ";      }

                            rolloutParamBlockString << "range:[" << absMinValue << "," << absMaxValue << "," << defaultValues(0) << "]\n";
                        }

                        // Y 'Spinner' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "spinner " << internalName << "Y align:#left ";

                            if ( isSubGroup )    { rolloutParamBlockString << "fieldwidth:39 offset:[-3,0] "; }
                            else                 { rolloutParamBlockString << "fieldwidth:42 offset:[-5,0] "; }

                            if ( useWorldUnits ) { rolloutParamBlockString << "type:#worldunits "; }
                            else                 { rolloutParamBlockString << "type:#float ";      }

                            rolloutParamBlockString << "range:[" << absMinValue << "," << absMaxValue << "," << defaultValues(1) << "]\n";
                        }

                        // Z 'Spinner' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "spinner " << internalName << "Z align:#right ";

                            if ( isSubGroup )    { rolloutParamBlockString << "fieldwidth:39 offset:[4,0] "; }
                            else                 { rolloutParamBlockString << "fieldwidth:43 offset:[9,0] "; }

                            if ( useWorldUnits ) { rolloutParamBlockString << "type:#worldunits "; }
                            else                 { rolloutParamBlockString << "type:#float ";      }

                            rolloutParamBlockString << "range:[" << absMinValue << "," << absMaxValue << "," << defaultValues(2) << "]\n";
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";

                    {
                        // Update actions if spinners or 3d variable has changed
                        {
                            { // varX = var.x, varY = var.y, varZ = var.z
                                onDataChangedString << indent.str() << "on " << internalName << " changed val do\n";
                                onDataChangedString << indent.str() << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t" << internalName << "X = val.x\n";
                                    onDataChangedString << indent.str() << "\t" << internalName << "Y = val.y\n";
                                    onDataChangedString << indent.str() << "\t" << internalName << "Z = val.z\n";
                                }
                                onDataChangedString << indent.str() << ")\n";
                            }
                            { // var.x = varX
                                onDataChangedString << indent.str() << "on " << internalName << "X changed val do\n";
                                onDataChangedString << indent.str() << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t" << internalName << ".x = val\n";
                                }
                                onDataChangedString << indent.str() << ")\n";
                            }
                            { // var.y = varY
                                onDataChangedString << indent.str() << "on " << internalName << "Y changed val do\n";
                                onDataChangedString << indent.str() << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t" << internalName << ".y = val\n";
                                }
                                onDataChangedString << indent.str() << ")\n";
                            }
                            { // var.z = varZ
                                onDataChangedString << indent.str() << "on " << internalName << "Z changed val do\n";
                                onDataChangedString << indent.str() << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t" << internalName << ".z = val\n";
                                }
                                onDataChangedString << indent.str() << ")\n";
                            }
                        }

                        {
                            {
                                onRolloutOpenString << indent.str() << "\t" << internalName << "X.value = " << internalName << ".x\n";
                            }
                            {
                                onRolloutOpenString << indent.str() << "\t" << internalName << "Y.value = " << internalName << ".y\n";
                            }
                            {
                                onRolloutOpenString << indent.str() << "\t" << internalName << "Z.value = " << internalName << ".z\n";
                            }
                        }
                    }
                }

                void visit(const hkReflect::PointerType* type, std::stringstream& rolloutParamBlockString)
                {
                    const hkLinkAttribute*      linkAttribute = type->findAttribute<hkLinkAttribute>();
                    if (!linkAttribute || linkAttribute->m_type == hkLinkAttribute::NONE)
                    {
                        return;
                    }

                    hkStringOld labelCtrString;
                    labelCtrString.printf("Label_%.4d", labelCtr++);

                    rolloutParamBlockString << indent.str() << "--(\n";
                    if ( (linkAttribute->m_type == hkLinkAttribute::MESH) || (linkAttribute->m_type == hkLinkAttribute::NODE_UUID) )
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << labelCtrString.cString() << " \"" << label << "\" align:#left ";

                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-3,2]\n"; }
                            else              { rolloutParamBlockString << "offset:[-8,2]\n"; }
                        }

                        // 'Edittext' & 'Pickbutton' GUI elements
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "edittext " << internalName << "TextBox align:#left readonly:true height:21 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "width:104 offset:[-7,-3] ";  }
                            else              { rolloutParamBlockString << "width:114 offset:[-12,-3] "; }
                            rolloutParamBlockString << "across:4\n";

                            rolloutParamBlockString << indent.str() << "\t" << "button " << internalName << "FollowLinkButton \">\" align:#right width:13 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[43,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[48,-3]\n"; }

                            rolloutParamBlockString << indent.str() << "\t" << "pickbutton " << internalName << " \"...\" align:#right width:13 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[23,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[28,-3]\n"; }

                            rolloutParamBlockString << indent.str() << "\t" << "button " << internalName << "UnlinkButton \"X\" align:#right width:13 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[3,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[8,-3]\n"; }
                        }

                        {
                            onDataChangedString << indent.str() << "on " << internalName << "FollowLinkButton pressed do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << "if ( " << internalName << ".object != undefined ) then\n";
                                onDataChangedString << indent.str() << "\t" << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t\t" << "select " << internalName << ".object\n";
                                }
                                onDataChangedString << indent.str() << "\t" << ")\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }
                        {
                            onDataChangedString << indent.str() << "on " << internalName << " picked obj do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << internalName << ".object = hvkDestruction_verifyMeshLink " << internalName << ".object\n";
                                onDataChangedString << indent.str() << "\t" << internalName << "TextBox.text = " << internalName << ".object.name\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }
                        {
                            onDataChangedString << indent.str() << "on " << internalName << "UnlinkButton pressed do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << internalName << ".object = undefined\n";
                                onDataChangedString << indent.str() << "\t" << internalName << "TextBox.text = \"Undefined\"\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }
                        {
                            onRolloutOpenString << indent.str() << "\t" << "if ( " << internalName << ".object != undefined ) then\n";
                            onRolloutOpenString << indent.str() << "\t" << "(\n";
                            {
                                onRolloutOpenString << indent.str() << "\t\t" << internalName << "TextBox.text = " << internalName << ".object.name\n";
                            }
                            onRolloutOpenString << indent.str() << "\t" << ")\n";
                            onRolloutOpenString << indent.str() << "\t" << "else\n";
                            onRolloutOpenString << indent.str() << "\t" << "(\n";
                            {
                                onRolloutOpenString << indent.str() << "\t\t" << internalName << "TextBox.text = \"Undefined\"\n";
                            }
                            onRolloutOpenString << indent.str() << "\t" << ")\n";
                        }
                    }
                    else
                    {
                        // 'Label' GUI element
                        {
                            rolloutParamBlockString << indent.str() << "\t" << "label " << labelCtrString.cString() << " \"" << label << "\" align:#left ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[-3,2]\n"; }
                            else              { rolloutParamBlockString << "offset:[-8,2]\n"; }
                        }

                        // 'Edittext' & 'Buttons' GUI elements
                        {
                            //rolloutParamBlockString << indent.str() << "edittext " << internalName << "TextBox align:#left readonly:true width:120 height:21 across:2\n";
                            rolloutParamBlockString << indent.str() << "\t" << "edittext " << internalName << "TextBox align:#left readonly:true height:19 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "width:86 offset:[-7,-2] ";  }
                            else              { rolloutParamBlockString << "width:99 offset:[-12,-2] "; }
                            rolloutParamBlockString << "across:4\n";

                            rolloutParamBlockString << indent.str() << "\t" << "button " << internalName << "FollowLinkButton \">\" align:#right width:11 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[23,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[30,-3]\n"; }

                            rolloutParamBlockString << indent.str() << "\t" << "button " << internalName << "Button \"Select\" align:#right width:35 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[25,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[31,-3]\n"; }

                            rolloutParamBlockString << indent.str() << "\t" << "button " << internalName << "UnlinkButton \"X\" align:#right width:11 ";
                            if ( isSubGroup ) { rolloutParamBlockString << "offset:[3,-3]\n"; }
                            else              { rolloutParamBlockString << "offset:[8,-3]\n"; }
                        }

                        {
                            onDataChangedString << indent.str() << "on " << internalName << "FollowLinkButton pressed do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << "hvkDestruction_followLink " << internalName << " selection[1]\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }
                        {
                            onDataChangedString << indent.str() << "on " << internalName << "UnlinkButton pressed do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << internalName << " = undefined\n";
                                onDataChangedString << indent.str() << "\t" << internalName << "TextBox.text = \"Undefined\"\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }
                        {
                            const hkReflect::Type* linkedClass = type->getSubType();

                            onDataChangedString << indent.str() << "on " << internalName << "Button pressed do\n";
                            onDataChangedString << indent.str() << "(\n";
                            {
                                onDataChangedString << indent.str() << "\t" << "pickedModifier = hvkDestruction_openGenericSelectionWindow \"" << linkedClass->getName() << "\" \"Select a node to link to...\" " << internalName << "\n";
                                onDataChangedString << indent.str() << "\t" << "if ( pickedModifier != -1 ) then\n";
                                onDataChangedString << indent.str() << "\t" << "(\n";
                                {
                                    onDataChangedString << indent.str() << "\t\t" << internalName << " = pickedModifier\n";
                                    onDataChangedString << indent.str() << "\t\t" << "if ( " << internalName << " != undefined ) then\n";
                                    onDataChangedString << indent.str() << "\t\t" << "(\n";
                                    {
                                        onDataChangedString << indent.str() << "\t\t\t" << internalName << "TextBox.text = pickedModifier.name as string\n";
                                    }
                                    onDataChangedString << indent.str() << "\t\t" << ")\n";
                                    onDataChangedString << indent.str() << "\t\t" << "else\n";
                                    onDataChangedString << indent.str() << "\t\t" << "(\n";
                                    {
                                        onDataChangedString << indent.str() << "\t\t\t" << internalName << "TextBox.text = \"Undefined\"\n";
                                    }
                                    onDataChangedString << indent.str() << "\t\t" << ")\n";
                                }
                                onDataChangedString << indent.str() << "\t" << ")\n";
                            }
                            onDataChangedString << indent.str() << ")\n";
                        }

                        //
                        // ON ROLLOUT OPEN DO...
                        //
                        {
                            onRolloutOpenString << indent.str() << "\t" << "if ( " << internalName << " != undefined ) then\n";
                            onRolloutOpenString << indent.str() << "\t" << "(\n";
                            {
                                onRolloutOpenString << indent.str() << "\t\t" << "hvkDestruction_assertLinkedModifierIsOnStack selection[1] " << internalName << "\n";
                                onRolloutOpenString << indent.str() << "\t\t" << internalName << "TextBox.text = " << internalName << ".name\n";
                            }
                            onRolloutOpenString << indent.str() << "\t" << ")\n";
                            onRolloutOpenString << indent.str() << "\t" << "else\n";
                            onRolloutOpenString << indent.str() << "\t" << "(\n";
                            {
                                onRolloutOpenString << indent.str() << "\t\t" << internalName << "TextBox.text = \"Undefined\"\n";
                            }
                            onRolloutOpenString << indent.str() << "\t" << ")\n";
                        }
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }

                void visitEnum(const hk::Presets* presets, std::stringstream& rolloutParamBlockString)
                {
                    rolloutParamBlockString << indent.str() << "--(\n";
                    {
                        // 'Dropdownlist' GUI element
                        rolloutParamBlockString << indent.str() << "\t" << "dropdownlist " << internalName << " \"" << label << "\" ";
                        if ( isSubGroup ) { rolloutParamBlockString << "offset:[-3,0] width:143 "; }
                        else              { rolloutParamBlockString << "offset:[-8,0] width:153 "; }
                        rolloutParamBlockString << "items:#(";

                        for (int i = 0; i < presets->getNumPresets(); i++)
                        {
                            rolloutParamBlockString << "\"" << presets->getPresetName(i) << "\"";
                            if (i < presets->getNumPresets() - 1)
                            {
                                rolloutParamBlockString << ", ";
                            }
                        }
                        int enumDefault = atoi(memberData.m_defaultValue.c_str()) + 1;
                        rolloutParamBlockString << ") selection:" << enumDefault << "\n";
                    }
                    rolloutParamBlockString << indent.str() << "--)\n";
                }
            } visitor(isSubGroup, indent, internalName, label, memberData,
                    onRolloutOpenString, onDataChangedString, labelCtr);

            if (member.getType()->extendsOrEquals<hkUuid>())
            {
                visitor.visitUuid(rolloutParamBlockString);
            }
            else if (member.getType()->extendsOrEquals<hkVector4>())
            {
                visitor.visitVector4(member.getType(), rolloutParamBlockString);
            }
            else if (const hk::Presets* presets = member.getType()->findAttribute<hk::Presets>())
            {
                HK_ASSERT_NO_MSG(0x7399994b, member.getType()->asInteger());
                visitor.visitEnum(presets, rolloutParamBlockString);
            }
            else
            {
                visitor.dispatch(member.getType(), rolloutParamBlockString);
            }
        }
    }

    //
    // Recursively add members from whole subtree.
    //
    {
        for (unsigned int i = 0; i < group->m_subGroups.size(); i++)
        {
            hctClassHierarchyUtil::ParamGroup* subGroup = group->m_subGroups[i];

            rolloutParamBlockString << indent.str() << "group \"" << subGroup->m_groupLabel << "\"\n";
            rolloutParamBlockString << indent.str() << "(\n";

            hctGenericModifierUtil::buildRolloutBlockForGroupRecursively(subGroup, true, indent, rolloutParamBlockString, onRolloutOpenString, onDataChangedString, indentLevel, labelCtr);

            rolloutParamBlockString << indent.str() << ")\n";
        }
    }
}


void hctGenericModifierUtil::buildHelpRollout(const hkReflect::Type* klass, std::stringstream& execString)
{
    if( klass->getName() )
    {
        execString << "\trollout helpGroup \"Help\"\n";
        execString << "\t(\n";
        execString << "\t\tbutton helpButton \"Open Documentation\"\n";
        execString << "\t\ton helpButton pressed do\n";
        execString << "\t\t(\n";
        execString << "\t\t\thkOpenChmDocs \"" << klass->getName() << "\"\n";
        execString << "\t\t)\n";
        execString << "\t)\n";
    }
}

void hctGenericModifierUtil::buildUnderReviewNotificationRollout(const hkReflect::Type* klass, std::stringstream& execString)
{
    if (klass->getName())
    {
        execString << "\trollout urnGroup \"Attention\"\n";
        execString << "\t(\n";
        execString << "\t\tlabel line1 \"\"\n";
        execString << "\t\tlabel line2 \"UNDER REVIEW\"\n";
        execString << "\t\tlabel line3 \"\"\n";
        execString << "\t\tlabel line4 \"Please contact\"\n";
        execString << "\t\thyperLink supportLink \"Havok Support\" color:blue align:#center address:\"http://support.havok.com\"\n";
        execString << "\t\tlabel line5 \"if you are actively\"\n";
        execString << "\t\tlabel line6 \"using this feature.\"\n";
        execString << "\t\tlabel line7 \"\"\n";
        execString << "\t)\n";
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
