// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Modifiers/ExportChannel/hctExportChannelModifier.h>


class hkExportChannelModifierDesc : public ClassDesc2
{
public:
    int             IsPublic() { return TRUE; }
    void *          Create (BOOL loading = FALSE) { return new hctExportChannelModifier(this); };
    const MCHAR *   ClassName() { return GetString( IDS_EXPORT_CHANNEL_MODIFIER_CLASS_NAME ); }
    SClass_ID       SuperClassID() { return OSM_CLASS_ID; }
    Class_ID        ClassID() { return HK_EXPORT_CHANNEL_MODIFIER_CLASS_ID; }
    const MCHAR*    Category() { return GetString( IDS_HAVOK_MODIFIERS_CATEGORY ); }
    const MCHAR*    InternalName() { return TEXT("hkExportChannelModifier"); }
    HINSTANCE       HInstance() { return hInstance; }
};

ClassDesc2* getHkExportChannelModifierDesc()
{
    static hkExportChannelModifierDesc exportChannelModifierDesc;
    return &exportChannelModifierDesc;
}

// We use multi map parameter blocks so we can move parameters to different rollouts in the future
// without extra hassle
enum
{
    MAP_GENERAL_PROPERTIES_ROLLOUT
};

/*
** Dialog Proc
*/

class hkExportChannelSpacesDlgProc : public ParamMap2UserDlgProc
{
public:

    hkExportChannelSpacesDlgProc ( hctExportChannelModifier* theModifier ) { m_theModifier = theModifier; }

    INT_PTR DlgProc(TimeValue t, IParamMap2 *pmap, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

    void DeleteThis() { } // Nothing, we use a static instance

    void SetThing ( ReferenceTarget* m )
    {
        m_theModifier = static_cast<hctExportChannelModifier*>( m );
    }

    void updateVertexPaintingTool() const;

protected:

    hctExportChannelModifier *m_theModifier;

};

INT_PTR hkExportChannelSpacesDlgProc::DlgProc(TimeValue t, IParamMap2 *pmap, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch ( LOWORD(wParam) )
    {
    case IDC_ED_EXPORT_CHANNEL_MOD_EXPORT_NAME :
    case IDC_SP_EXPORT_CHANNEL_MOD_CHANGE_CHANNEL :
    case IDC_CB_EXPORT_CHANNEL_MOD_DISPLAY :
        updateVertexPaintingTool();  // update the toolbox UI
        return TRUE;

    default:
        return m_theModifier->handleComboBoxMessages(hWnd, msg, wParam, lParam); // handle combo boxes
    }
}

void hkExportChannelSpacesDlgProc::updateVertexPaintingTool() const
{
    Interface* ip = GetCOREInterface();
    hctMaxUtils::evaluateMAXScript( ip, "hvkVertexPaintingTools_Refresh()" );
}

static hkExportChannelSpacesDlgProc g_pbExportChannelPaintDlgProc (NULL);


static ParamBlockDesc2 hkExportChannel_ParamBlockDesc (
    PB_EXPORT_CHANNEL_MOD_PBLOCK, _T("hkExportChannelModifier"),  0, getHkExportChannelModifierDesc(),
    P_AUTO_CONSTRUCT + P_AUTO_UI + P_MULTIMAP, PB_EXPORT_CHANNEL_MOD_PBLOCK,

    // One Rollout
    1,
    // General properties rollout
    MAP_GENERAL_PROPERTIES_ROLLOUT,
    IDD_EXPORT_CHANNEL_MODIFIER_ROLLOUT,
    IDS_EXPORT_CHANNEL_MODIFIER_ROLLOUT_GENERAL_PROPERTIES,
    0, 0,
    &g_pbExportChannelPaintDlgProc,


    PA_EXPORT_CHANNEL_MOD_EXPORT_NAME,
    _T("channelExportName"), TYPE_STRING, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_EXPORT_NAME,
    p_default,      0,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_EDITBOX,   IDC_ED_EXPORT_CHANNEL_MOD_EXPORT_NAME,
    p_end,

    PA_EXPORT_CHANNEL_MOD_CHANNEL_ID,
    _T("channelID"), TYPE_INT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_CHANNEL_ID_SPINNER,
    p_default,      0,
    p_range,        0, 99,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_INT, IDC_ED_EXPORT_CHANNEL_MOD_CHANGE_CHANNEL, IDC_SP_EXPORT_CHANNEL_MOD_CHANGE_CHANNEL,
    SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_CHANNEL_NAME,
    _T("channelName"), TYPE_STRING, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_CHANNEL_NAME,
    p_default,      0,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_EDITBOX,   IDC_ED_EXPORT_CHANNEL_MOD_CHANNEL_NAME,
    p_end,

    PA_EXPORT_CHANNEL_MOD_TYPE,
    _T("channelType"), TYPE_INT, P_RESET_DEFAULT | P_ANIMATABLE, IDS_EXPORT_CHANNEL_MODIFIER_PA_CHANNEL_TYPE,
    p_default,      HAVOK_CHANNEL_FLOAT,
    p_end,

    PA_EXPORT_CHANNEL_MOD_RESCALE,
    _T("rescale"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_RESCALE,
    p_default,      FALSE,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_CHANNEL_MOD_RESCALE,
    p_enable_ctrls, 4, PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT,
    p_end,

    PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST,
    _T("rescaleMinD"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_RESCALE_MIN,
    p_default,      0.0f,
    p_range,        -10000000.0f, 10000000.0f,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_UNIVERSE, IDC_ED_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, IDC_SP_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST,
    _T("rescaleMaxD"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_RESCALE_MAX,
    p_default,      0.01f,
    p_range,        -10000000.0f, 10000000.0f,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_UNIVERSE, IDC_ED_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, IDC_SP_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT,
    _T("rescaleMinF"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_RESCALE_MIN,
    p_default,      0.0f,
    p_range,        -10000000.0f, 10000000.0f,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, IDC_SP_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT,
    _T("rescaleMaxF"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_RESCALE_MAX,
    p_default,      1.0f,
    p_range,        -10000000.0f, 10000000.0f,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT, IDC_SP_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT, SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_DISPLAY_ACTIVE,
    _T("activeDisplay"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_ACTIVE,
    p_default,      FALSE,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_CHANNEL_MOD_DISPLAY,
    p_enable_ctrls, 3, PA_EXPORT_CHANNEL_MOD_DISPLAY_SELECTED, PA_EXPORT_CHANNEL_MOD_USE_TRANSPARENCY, PA_EXPORT_CHANNEL_MOD_OPACITY,
    p_end,

    // should the radius be displayed only when the object is selected
    PA_EXPORT_CHANNEL_MOD_DISPLAY_SELECTED,
    _T("displaySelectedOnly"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_DISPLAY_SELECTED_ONLY,
    p_default,      TRUE,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_CHANNEL_MOD_DISPLAY_SELECTED_ONLY,
    p_end,

    // use transparency for radius
    PA_EXPORT_CHANNEL_MOD_USE_TRANSPARENCY,
    _T("displayRadiusAsTransparent"), TYPE_BOOL, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_USE_TRANSPARENCY,
    p_default,      TRUE,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SINGLECHEKBOX, IDC_CB_EXPORT_CHANNEL_MOD_USE_TRANSPARENCY,
    p_enable_ctrls, 1, PA_EXPORT_CHANNEL_MOD_OPACITY,
    p_end,

    // set transparency
    PA_EXPORT_CHANNEL_MOD_OPACITY,
    _T("displayRadiusOpacity"), TYPE_FLOAT, P_ANIMATABLE | P_RESET_DEFAULT, IDS_EXPORT_CHANNEL_MODIFIER_PA_OPACITY,
    p_default,      0.5f,
    p_range,        0.0f, 1.0f,
    p_ui,           MAP_GENERAL_PROPERTIES_ROLLOUT,
    TYPE_SPINNER, EDITTYPE_FLOAT, IDC_ED_EXPORT_CHANNEL_MOD_OPACITY, IDC_SP_EXPORT_CHANNEL_MOD_OPACITY, SPIN_AUTOSCALE,
    p_end,

    PA_EXPORT_CHANNEL_MOD_DISPLAY_TYPE,
    _T("channelDisplayType"), TYPE_INT, P_RESET_DEFAULT | P_ANIMATABLE, IDS_EXPORT_CHANNEL_MODIFIER_PA_CHANNEL_DISPLAY_TYPE,
    p_default,      HAVOK_CHANNEL_FLOAT,
    p_end,

    p_end
    );

class ChannelTypeComboBoxDescriptor : public hctBasicModifier::ComboBoxDescriptor
{
public:

    /*virtual*/ BlockID getParamBlockID() {return PB_EXPORT_CHANNEL_MOD_PBLOCK;}
    /*virtual*/ ParamID getParamID() {return PA_EXPORT_CHANNEL_MOD_TYPE;}
    /*virtual*/ MapID getParamMapID() {return MAP_GENERAL_PROPERTIES_ROLLOUT;}
    /*virtual*/ int getControlID() {return IDC_EXPORT_CHANNEL_MOD_CHANNEL_TYPE;}
    /*virtual*/ int getNumElements() {return 3;}
    /*virtual*/ const char* getElementString (int i)
    {
        static const char* m_elementStrings [] = { "Float", "Distance", "Angle" };
        return m_elementStrings[i];
    }

    /*virtual*/ const int getElementValue (int i) {return i;}
};

class ChannelDisplayTypeComboBoxDescriptor : public hctBasicModifier::ComboBoxDescriptor
{
public:

    /*virtual*/ BlockID getParamBlockID() {return PB_EXPORT_CHANNEL_MOD_PBLOCK;}
    /*virtual*/ ParamID getParamID() {return PA_EXPORT_CHANNEL_MOD_DISPLAY_TYPE;}
    /*virtual*/ MapID getParamMapID() {return MAP_GENERAL_PROPERTIES_ROLLOUT;}
    /*virtual*/ int getControlID() {return IDC_EXPORT_CHANNEL_MOD_CHANNEL_DISPLAY_TYPE;}
    /*virtual*/ int getNumElements() {return 2;}
    /*virtual*/ const char* getElementString (int i)
    {
        static const char* m_elementStrings [] = { "Radius", "Normal Distance" };
        return m_elementStrings[i];
    }

    /*virtual*/ const int getElementValue (int i) {return i;}
};

hctBasicModifier::ComboBoxDescriptor* hctExportChannelModifier::getComboBoxDescriptor ( int comboBoxID )
{
    static ChannelTypeComboBoxDescriptor channelTypeComboBox;
    static ChannelDisplayTypeComboBoxDescriptor channelDisplayTypeComboBox;

    switch ( comboBoxID )
    {
    case 0 :
        return &channelTypeComboBox;

    case 1 :
        return &channelDisplayTypeComboBox;

    default:
        break;
    }

    assert(0); // should never get here
    return NULL;
}

// STATIC DATA FOR DISPLAY
bool hctExportChannelModifier::m_isSphereMeshCreated = false;
Mesh hctExportChannelModifier::m_sphereDisplayMeshLowRes;
Mesh hctExportChannelModifier::m_sphereDisplayMeshHighRes;
bool hctExportChannelModifier::m_isDistanceMeshCreated = false;
Mesh hctExportChannelModifier::m_dummyDisplayMesh = hctExportChannelModifier::m_sphereDisplayMeshLowRes;

hctExportChannelModifier::hctExportChannelModifier(ClassDesc2* theClassDesc)
: hctBasicModifier(theClassDesc),
m_maxRadius( .0f ),
m_bBox( Point3(.0f, .0f , .0f), Point3(.0f, .0f , .0f) ),
m_areMaterialsSet( false ),
m_updateParticleData( true ),
m_updateNormals( true ),
m_particleValObjRatioX( .0f ),
m_particleValObjRatioY( .0f ),
m_channelType( -1 )
{
    theClassDesc->MakeAutoParamBlocks(this);

    // set the mesh used to display per particle radius
    createSphereDisplayMeshes();
    createDistanceDisplayMeshes();

    // setup display materials
    setupMaterials();

    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );
    const float metersPerUnit = float(GetMasterScale(UNITS_METERS));
    // rescale min
    {
        const float meters = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST );
        const float units = meters / metersPerUnit;
        pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, 0, units );
    }
    // rescale max
    {
        const float meters = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST );
        const float units = meters / metersPerUnit;
        pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, 0, units );
    }
}


void hctExportChannelModifier::createSphereDisplayMeshes()
{
    if( !m_isSphereMeshCreated )
    {
        hctMaxUtils::createGeoSphereMesh( 1.0f, 1, true, m_sphereDisplayMeshLowRes );
        hctMaxUtils::createGeoSphereMesh( 1.0f, 2, true, m_sphereDisplayMeshHighRes );
        m_isSphereMeshCreated = true;
    }
}

void hctExportChannelModifier::createDistanceDisplayMeshes()
{
    if( !m_isDistanceMeshCreated )
    {
        if( !m_isSphereMeshCreated ) createSphereDisplayMeshes();

        m_isDistanceMeshCreated = true;
    }
}

void hctExportChannelModifier::setupMaterials()
{
    if( !m_areMaterialsSet )
    {
        Point3 materialColor = GetColorManager()->GetColorAsPoint3( HK_COLOR_CLOTH );

        m_sphereDisplayMaterial.shinStrength = 0.1f;
        m_sphereDisplayMaterial.Ka = m_sphereDisplayMaterial.Ks = m_sphereDisplayMaterial.Kd = materialColor;
        m_sphereDisplayMaterial.selfIllum = 0.8f;
        m_sphereDisplayMaterial.opacity = (GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK ) )->GetFloat( PA_EXPORT_CHANNEL_MOD_OPACITY, 0 );


        Point3 negCol( 255/255.0f, 47/255.0f, 52/255.0f );
        m_sphereDisplayMaterialNegRadius = m_sphereDisplayMaterial;
        m_sphereDisplayMaterialNegRadius.Ka = m_sphereDisplayMaterialNegRadius.Ks = m_sphereDisplayMaterialNegRadius.Kd = negCol;

        m_areMaterialsSet = true;
    }
}

hctExportChannelModifier::~hctExportChannelModifier()
{
}

void hctExportChannelModifier::BeginEditParams( IObjParam *ip, ULONG flags,Animatable *prev )
{
    g_pbExportChannelPaintDlgProc.SetThing(this);
    hctBasicModifier::BeginEditParams(ip, flags, prev);
    updateUI();
}

void hctExportChannelModifier::EndEditParams( IObjParam *ip, ULONG flags,Animatable *next)
{
    hctBasicModifier::EndEditParams( ip, flags, next );
    g_pbExportChannelPaintDlgProc.SetThing( NULL );
}


void hctExportChannelModifier::updateUI()
{
    if ( !m_interface ) return;
    const TimeValue now = m_interface->GetTime();

    // the controls have a meaning if there is one and only one node selected
    BOOL oneNodeSelected = GetCOREInterface()->GetSelNodeCount() == 1;
    INode* selNode = ( oneNodeSelected ) ? GetCOREInterface()->GetSelNode( 0 ) : NULL;
    bool deleteTriObject = false;
    TriObject* triObject = NULL;

    if( selNode ) // convert to triObject to get the mesh
    {
        Object *obj = selNode->EvalWorldState( now ).obj;
        // will be true if the triObject will be created for us
        triObject = extractTriObjectFromObject( now, obj, deleteTriObject );
    }

    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );
    IParamMap2* map = pblock->GetMap();
    if (!map) return;

    // enable or disable controls
    IParamMap2* pmap = pblock->GetMap( MAP_GENERAL_PROPERTIES_ROLLOUT );
    HWND dlgWnd = pmap->GetHWnd();
    if( dlgWnd )
    {
        BOOL isChannelvalid = false;

        // Channel ID controls
        if( oneNodeSelected && triObject )
        {
            int channelId = pblock->GetInt( PA_EXPORT_CHANNEL_MOD_CHANNEL_ID );

            // get the mesh
            Mesh& theMesh = triObject->GetMesh();

            isChannelvalid = ( channelId != -1 ) && theMesh.mapSupport( channelId );
            {
                // Set Channel Name
                ICustStatus* iCustStatusCtrl = GetICustStatus( GetDlgItem( dlgWnd, IDC_ED_EXPORT_CHANNEL_MOD_CHANNEL_NAME ) );
                // if the channel is supported and the name is found
                if( isChannelvalid )
                {
                    // set the name
                    MSTR channelName; getMeshChannelName ( selNode, channelId, channelName );
                    iCustStatusCtrl->SetText( channelName ) ;
                }
                else
                {
                    iCustStatusCtrl->SetText( TEXT("unsupported") ) ;
                }
                ReleaseICustStatus( iCustStatusCtrl );
            }
        }
        {
            // if the type has changed we have to update the rescale controls
            const int type = pblock->GetInt( PA_EXPORT_CHANNEL_MOD_TYPE );

            const BOOL isDistance = ( type == HAVOK_CHANNEL_DISTANCE );
            const BOOL wasDistance = ( m_channelType == HAVOK_CHANNEL_DISTANCE );

            // update rescale controls visibility
            pmap->Show( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, !isDistance );
            pmap->Show( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT, !isDistance );
            pmap->Show( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST,   isDistance );
            pmap->Show( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST,   isDistance );

            const int chType = m_channelType;
            m_channelType = type; // avoid recursion

            if( chType != type && chType != -1 ) // convert values of we changed type and it's not startup
            {
                // Convert...
                if( isDistance ) // ...from float to universe
                {
                    float val = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT, now );
                    pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, now, val );
                    val = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, now );
                    pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, now, val );
                }
                else if( wasDistance )// ...or from universe to float
                {
                    float val = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST, now );
                    pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT, now, val );
                    val = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST, now );
                    pblock->SetValue( PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT, now, val );
                }
            }

            const BOOL displayOn = pblock->GetInt(PA_EXPORT_CHANNEL_MOD_DISPLAY_ACTIVE);

            // Channel display makes sense just if the type is distance
            pmap->Enable( PA_EXPORT_CHANNEL_MOD_DISPLAY_ACTIVE, isDistance );
            pmap->Enable( PA_EXPORT_CHANNEL_MOD_DISPLAY_SELECTED, isDistance && displayOn);
            pmap->Enable( PA_EXPORT_CHANNEL_MOD_DISPLAY_TYPE, isDistance );
            pmap->Enable( PA_EXPORT_CHANNEL_MOD_OPACITY, isDistance && displayOn);
            pmap->Enable( PA_EXPORT_CHANNEL_MOD_USE_TRANSPARENCY, isDistance && displayOn);
        }
    }

    // if the obj!=triobject, it means that the triobj has been created for us, and we will have to destroy it
    if( deleteTriObject )
    {
        delete triObject;
    }

    g_pbExportChannelPaintDlgProc.updateVertexPaintingTool();  // update the toolbox UI

}

void hctExportChannelModifier::NotifyInputChanged(Interval changeInt, PartID partID, RefMessage message, ModContext *mc)
{
    if ( partID != 0xffffffff )
    {
        if ( ( partID & PART_VERTCOLOR ) ) // painting in progress, update vertex color data
        {
            m_updateParticleData = true;
            NotifyDependents(changeInt, partID, message);
        }
        if ( ( partID & PART_GEOM ) || ( ( partID & PART_TOPO ) ) ) // geometry is changing, rebuild the normals
        {
            m_updateNormals = true;
        }
    }
}

void hctExportChannelModifier::ModifyObject(TimeValue t, ModContext &mc, ObjectState * os, INode *node)
{
    m_updateParticleData = true;
    hctBasicModifier::ModifyObject(t,mc,os,node);
}



hctBasicXTCObject* hctExportChannelModifier::getXTCObject (int number, TimeValue t, ModContext &mc, ObjectState * os, INode *node)
{
    return new hctExportChannelXTCObject( this );
}

bool hctExportChannelModifier::getMeshChannelName (INode* inode, int channelId, MSTR& nameOut)
{
    MSTR buf;
    buf.printf(_T("MapChannel:%i"), channelId );
    return ( inode->GetUserPropString(buf, nameOut) != 0 );
}


int hctExportChannelModifier::getMapChannelID( IParamBlock2* pblock )
{
    return pblock->GetInt( PA_EXPORT_CHANNEL_MOD_CHANNEL_ID );
}

void hctExportChannelModifier::updateNormalsForParticle( Point3& normalOut, int vIdx, Mesh& theMesh )
{
    RVertex* rv = theMesh.getRVertPtr( vIdx );  // get the rendering vertex
    int numNormals = rv->rFlags & NORCT_MASK;

    if (numNormals == 1)
    {
        normalOut = rv->rn.getNormal();
    }
    else if( numNormals != 0 )
    {
        normalOut = rv->ern[0].getNormal();
    }
}


void hctExportChannelModifier::gatherParticleNormalsData( TimeValue t, INode* inode, Object* object, Tab<hctExportChannelModifier::ParticleData>& data )
{
    if (inode == NULL)
    {
        inode = hctMaxUtils::findNodeRef( this );
    }
    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );

    // will be true if the triObject will be created for us
    bool deleteTriObject = false;
    TriObject* triObject = extractTriObjectFromObject( t, object, deleteTriObject );

    // get the mesh
    Mesh& theMesh = triObject->GetMesh();
    // clear data
    data.ZeroCount();
    {
        int channelID = getMapChannelID( pblock );
        if( theMesh.mapSupport( channelID ) )
        {
            // build the normals if they are not already built
            theMesh.checkNormals( true );

            // prepare data array
            const int numVerts = theMesh.getNumVerts();
            data.SetCount( numVerts );
            m_particleValueMap.SetCount( data.Count() );

            // iterate over vertices and get positions
            for( int idx=0; idx < numVerts; ++idx)
            {
                data[idx].position = theMesh.getVert( idx );
                // prepare normals arrays
                data[idx].normal.Set(0,0,0);

                m_particleValueMap[idx] = -1;
            }

            // number of faces
            int numFaces = theMesh.getNumFaces();
            // faces
            Face* faces = theMesh.faces;
            TVFace* tFaces = theMesh.mapFaces( channelID );

            for(int faceIdx = 0; faceIdx<numFaces; ++faceIdx)
            {
                const int idx0 = faces[faceIdx].getVert( 0 );
                updateNormalsForParticle( (data[ idx0 ].normal), idx0, theMesh );

                const int idx1 = faces[faceIdx].getVert( 1 );
                updateNormalsForParticle( (data[ idx1 ].normal), idx1, theMesh );

                const int idx2 = faces[faceIdx].getVert( 2 );
                updateNormalsForParticle( (data[ idx2 ].normal), idx2, theMesh );

                const DWORD vertex0 = tFaces[faceIdx].getTVert( 0 );
                const DWORD vertex1 = tFaces[faceIdx].getTVert( 1 );
                const DWORD vertex2 = tFaces[faceIdx].getTVert( 2 );
                m_particleValueMap[ idx0 ] = vertex0;
                m_particleValueMap[ idx1 ] = vertex1;
                m_particleValueMap[ idx2 ] = vertex2;
            }
        }
    }

    // if the obj!=triobject, it means that the triobj has been created for us, and we will have to destroy it
    if( deleteTriObject )
    {
        delete triObject;
    }

    // flag as updated
    m_updateNormals = false;
}

void hctExportChannelModifier::gatherParticleVertexColorData( TimeValue t, INode* inode, Object* object, Tab<hctExportChannelModifier::ParticleData>& data )
{
    if (inode == NULL)
    {
        inode = hctMaxUtils::findNodeRef( this );
    }
    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );

    // will be true if the triObject will be created for us
    bool deleteTriObject = false;
    TriObject* triObject = extractTriObjectFromObject( t, object, deleteTriObject );

    // get the mesh
    Mesh& theMesh = triObject->GetMesh();

    m_maxRadius = .0f;
    {
        // get max and min scale values
        const BOOL rescale = pblock->GetInt( PA_EXPORT_CHANNEL_MOD_RESCALE, t ) != FALSE;

        // the control used to scale depends on the type
        const BOOL isDIstance = ( pblock->GetInt( PA_EXPORT_CHANNEL_MOD_TYPE ) == HAVOK_CHANNEL_DISTANCE );
        const ParamID resMinPar = ParamID(( isDIstance ) ? PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_DIST : PA_EXPORT_CHANNEL_MOD_RESCALE_MIN_FLOAT);
        const ParamID resMaxPar = ParamID(( isDIstance ) ? PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_DIST : PA_EXPORT_CHANNEL_MOD_RESCALE_MAX_FLOAT);
        // get scaling values
        const float scaleMin = ( rescale ) ? pblock->GetFloat( resMinPar, t ) :  .0f;
        const float scaleMax = ( rescale ) ? pblock->GetFloat( resMaxPar, t ) : 1.0f;
        const float scale = scaleMax - scaleMin;

        const int channelID = getMapChannelID( pblock );

        if( theMesh.mapSupport( channelID ) )
        {
            // painted values
            UVVert* values = theMesh.mapVerts( channelID );
            const float valdiv = ( 1 / 3.0f ) * scale;

            // use this just for isolated vertices
            Point3 dummyBlack( 0, 0, 0 );

            for ( int vidx = 0; vidx < data.Count(); ++vidx )
            {
                // get the color
                const Point3& val = ( m_particleValueMap[ vidx ] > 0 ) ? values[ m_particleValueMap[ vidx ] ] : dummyBlack;
                data[ vidx ].value = ( val.x + val.y + val.z ) * valdiv + scaleMin; // scale and store it
                if( data[ vidx ].value > m_maxRadius ) m_maxRadius = data[ vidx ].value; // keep max value
            }
        }
    }

    // flag as updated
    m_updateParticleData = false;

    // keep the ratio used to enlarge the viewport
    m_bbox = theMesh.getBoundingBox();
    const Point3 width = m_bbox.Width();
    m_particleValObjRatioX =  ( m_maxRadius / width.x );
    m_particleValObjRatioY =  ( m_maxRadius / width.y );

    // if the obj!=triobject, it means that the triobj has been created for us, and we will have to destroy it
    if( deleteTriObject )
    {
        delete triObject;
    }
}

void hctExportChannelModifier::checkAndUpdateData( TimeValue t, INode* inode, Object* object, Tab<hctExportChannelModifier::ParticleData>& data )
{
    if ( m_updateNormals )
    {
        gatherParticleNormalsData( t, inode, object, data );
        m_updateNormals = false;
        m_updateParticleData = true;
    }
    if ( m_updateParticleData )
    {
        gatherParticleVertexColorData( t, inode, object, data );
        m_updateParticleData = false;
    }
}

TriObject* hctExportChannelModifier::extractTriObjectFromObject( TimeValue t, Object* obj, bool& deleteTriObject )
{
    // We ask the object to convert itself to a TriObject
    if ( (obj==NULL) || !obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID,0)) )
    {
        return HK_NULL;
    }
    TriObject* triObject = reinterpret_cast<TriObject*>( obj->ConvertToType( t, Class_ID(TRIOBJ_CLASS_ID,0) ) );
    // if the two pointers are different the new one was created for us
    deleteTriObject = ( obj != triObject );

    return triObject;
}

// returns true if the inode is currently selected
static bool _isNodeSelected (const INode* inode)
{
    Interface* ip = GetCOREInterface();
    const int numSel = ip->GetSelNodeCount();
    for (int i=0; i<numSel; ++i)
    {
        if (ip->GetSelNode(i)==inode) return true;
    }

    return false;
}

BOOL hctExportChannelModifier::shouldDisplay(TimeValue t, INode* inode)
{
    IParamBlock2* pBlock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );

    const int type = pBlock->GetInt( PA_EXPORT_CHANNEL_MOD_TYPE );
    const BOOL isDistance = ( type == HAVOK_CHANNEL_DISTANCE );

    // is Display checked in the UI?
    BOOL shouldDisplay = isDistance && ( pBlock->GetInt( PA_EXPORT_CHANNEL_MOD_DISPLAY_ACTIVE, t ) == TRUE );
    // is the node selected?
    shouldDisplay = shouldDisplay && ( (pBlock->GetInt( PA_EXPORT_CHANNEL_MOD_DISPLAY_SELECTED )==TRUE)?_isNodeSelected( inode ):TRUE );

    return shouldDisplay;
}

Material* hctExportChannelModifier::getSphereDisplayMaterial(TimeValue t)
{
    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );
    if( pblock->GetInt( PA_EXPORT_CHANNEL_MOD_USE_TRANSPARENCY, t ) == TRUE )
    {
        m_sphereDisplayMaterial.opacity = pblock->GetFloat( PA_EXPORT_CHANNEL_MOD_OPACITY, t );
    }
    else
    {
        m_sphereDisplayMaterial.opacity = 1.0;
    }

    return &m_sphereDisplayMaterial;
}


Material* hctExportChannelModifier::getSphereDisplayMaterialNegRadius()
{
    m_sphereDisplayMaterialNegRadius.opacity = m_sphereDisplayMaterial.opacity;
    return &m_sphereDisplayMaterialNegRadius;
}

int hctExportChannelModifier::getDisplayType()
{
    IParamBlock2* pblock = GetParamBlock( PB_EXPORT_CHANNEL_MOD_PBLOCK );
    assert(pblock);
    return pblock->GetInt( PA_EXPORT_CHANNEL_MOD_DISPLAY_TYPE );
}


void hctExportChannelModifier::GetWorldBoundBox(TimeValue t,INode* inode, ViewExp *vpt, Box3& box3, ModContext *mc)
{
    box3 = *(mc->box);

    const float maxR = ( m_particleValObjRatioX > m_particleValObjRatioY ) ? m_particleValObjRatioX : m_particleValObjRatioY;
    Point3 bw = box3.Width();
    float l = (bw.x > bw.y) ? bw.x : bw.y;
    if( maxR > .0f )
    {
        box3.EnlargeBy( maxR*l );

        // save it
        m_bBox = box3;
    }
}

void hctExportChannelModifier::GetLocalBoundBox(TimeValue t, INode* inode, ViewExp* vpt, Box3& box3 )
{
    Box3 worldBox;
    GetWorldBoundBox(t,inode,vpt,worldBox,NULL);

    Matrix3 worldToLocal = Inverse(inode->GetNodeTM(t));

    box3 = worldBox * worldToLocal;
}


void hctExportChannelXTCObject::MaybeEnlargeViewportRect(GraphicsWindow *gw, Rect &rect)
{
    const float particleValObjRatioX = m_modifier->getParticleValObjRatioX();
    const float particleValObjRatioY = m_modifier->getParticleValObjRatioY();
    int h = int(rect.h() * particleValObjRatioY);
    int w = int(rect.w() * particleValObjRatioX);

    rect.bottom += h;
    rect.top    -= w;
    rect.left   -= w;
    rect.right  += h;
}


// XT Object
unsigned long hctExportChannelXTCObject::m_channelDependencies = (TOPO_CHANNEL|GEOM_CHANNEL|TEXMAP_CHANNEL/*|SELECT_CHANNEL|MTL_CHANNEL|SUBSEL_TYPE_CHANNEL|DISP_ATTRIB_CHANNEL|VERTCOLOR_CHANNEL|GFX_DATA_CHANNEL|DISP_APPROX_CHANNEL|EXTENSION_CHANNEL|TM_CHANNEL|GLOBMTL_CHANNEL*/);

hctExportChannelXTCObject::hctExportChannelXTCObject(hctExportChannelModifier* theModifier)
:m_modifier( theModifier ),
m_translation( .0f, .0f, .0f ),
m_updateData(false)
{
    HK_ASSERT(0x42839548, m_modifier != NULL, "NULL modifier provided");
}

Mesh& hctExportChannelXTCObject::getDisplayMesh( int elementCount, int type )
{
    switch ( type )
    {
    case HAVOK_CHANNEL_DISPLAY_RADIUS:
        return  hctExportChannelModifier::getSphereDisplayMesh( elementCount > 500 );
    case HAVOK_CHANNEL_DISPLAY_NORMAL_DISTANCE :
        return hctExportChannelModifier::getDistanceDisplayMesh( elementCount > 500 );
    default :
        break;
    }
    assert(0);
    return hctExportChannelModifier::getDummyDisplayMesh();
}

int hctExportChannelXTCObject::Display( TimeValue t, INode* inode, ViewExp *vpt, int flags, Object *pObj )
{
    // check if the radius must be displayed only with the node as selected
    if ( !m_modifier->shouldDisplay(t,inode) ) return 1;

    // we use the modifier's data array
    Tab<hctExportChannelModifier::ParticleData>* data = m_modifier->getParticleData();

    // Check if data is up to date
    m_modifier->checkAndUpdateData( t, inode, pObj, *data );

    GraphicsWindow *gw = vpt->getGW();

    // draw particle values, if any
    if( data->Count() != 0 )
    {

        Material* material = m_modifier->getSphereDisplayMaterial(t);

        const DWORD savedLimits=gw->getRndLimits();

        if( material->opacity < 1.0f )
        {
            gw->setRndLimits( GW_BACKCULL | GW_LIGHTING | GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
        }
        else
        {
            gw->setRndLimits( GW_BACKCULL | GW_LIGHTING | GW_Z_BUFFER );
        }

        // save the translation
        m_translation = inode->GetObjectTM(t).GetTrans();

        Point3 blackColor( .0f, .0f, .0f);
        gw->setColor(LINE_COLOR, blackColor);

        const int displayType = m_modifier->getDisplayType();
        Mesh& displayMesh = getDisplayMesh( data->Count(), displayType ) ;

        // Display depending on type
        switch ( displayType )
        {
        case HAVOK_CHANNEL_DISPLAY_RADIUS :
            displaySpheres( t, gw, data, displayMesh, inode, material );
            break;
        case HAVOK_CHANNEL_DISPLAY_NORMAL_DISTANCE :
            displayNormalDistances( t, gw, data, displayMesh, inode, material );
            break;
        }

        gw->setRndLimits( savedLimits );
    }
    return 1;
}

void hctExportChannelXTCObject::displaySpheres( TimeValue t, GraphicsWindow* gw, Tab<hctExportChannelModifier::ParticleData>* data, Mesh& displayMesh, INode* inode, Material* material )
{
    Matrix3 objectToWorld = inode->GetObjTMAfterWSM( t );

    Matrix3 meshToWorld;

    Material* negRadMat = m_modifier->getSphereDisplayMaterialNegRadius();

    // Display for Sphere- make function
    for ( int i=0; i<data->Count(); ++i )
    {
        const hctExportChannelModifier::ParticleData& particleData = (*data)[i];
        const float& radius = particleData.value;

        // we do not want object scaling to affect spheres scale, so we transform the position first and then apply our own scaling
        Point3 pos; objectToWorld.TransformPoints( &(particleData.position), &pos, 1 );
        // translate into particle position
        meshToWorld.IdentityMatrix();
        meshToWorld.PreTranslate( pos );

        // scale the mesh with radius
        const float scale = ( radius > .0f ) ? radius : -radius;
        meshToWorld.PreScale( Point3( scale, scale, scale ) );

        gw->setTransform( meshToWorld );
        Material* renderMat = ( radius > 0 ) ? material : negRadMat;
        gw->setMaterial( *renderMat );

        if ( material->opacity < 1.0f )
        {
            // transparency
            gw->setTransparency( GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
        }

        // render the mesh
        displayMesh.render( gw, renderMat, NULL, COMP_ALL );
    }
    gw->setTransparency(0);
}



void hctExportChannelXTCObject::displayNormalDistances( TimeValue t, GraphicsWindow* gw, const Tab<hctExportChannelModifier::ParticleData>* data, Mesh& displayMesh, INode* inode, Material* material )
{
    // get node transform
    Matrix3 objectToWorld = inode->GetObjTMAfterWSM( t );

    Matrix3 normalTransform;
    {
        // Normals are transformed by the inverse transposed
        Matrix3 inverse = Inverse( objectToWorld );
        Matrix3& inverseTransposed = normalTransform;
        inverseTransposed.SetRow(0, inverse.GetColumn3(0));
        inverseTransposed.SetRow(1, inverse.GetColumn3(1));
        inverseTransposed.SetRow(2, inverse.GetColumn3(2));
        inverseTransposed.SetRow(3, Point3(0,0,0));
    }
    normalTransform.NoScale(); // remove scaling

    // identity display transform, we transform the normals ourselves
    {
        Matrix3 trans; trans.IdentityMatrix();
        gw->setTransform( trans );
    }

    Tab<Point3> transformedPositionsAndNormals; //  ( position, normal ) pairs for all the particles
    Tab<float> scalesValues;

    transformedPositionsAndNormals.SetCount(data->Count()*2);
    for ( int partIdx = 0; partIdx < data->Count(); ++partIdx )
    {
        hctExportChannelModifier::ParticleData& particleData = (*data)[partIdx];
        const Point3& normal = (particleData.normal);

        // transform position
        Point3 pos ; objectToWorld.TransformPoints( &(particleData.position), &(pos), 1 );

        transformedPositionsAndNormals[ partIdx*2 ] = pos; // store position

        // transform normal
        Point3 normalWorld = normal * normalTransform;//.TransformVectors( &(normal), &normalWorld, 1 );
        // store p_end point
        transformedPositionsAndNormals[ partIdx*2 + 1 ] = pos + normalWorld * particleData.value;

        // save scaling
        scalesValues.Append( 1, &(particleData.value) );
    }

    // Display the normals
    gw->startSegments();
    assert( (transformedPositionsAndNormals.Count() % 2) == 0 ); // must be 2*numNormals
    const int numNormals = transformedPositionsAndNormals.Count() / 2;
    for ( int i=0; i<numNormals; ++i )
    {
        gw->segment( transformedPositionsAndNormals.Addr( i*2 ), TRUE );
    }
    gw->endSegments();

    if ( material->opacity < 1.0f )
    {
        // transparency
        gw->setTransparency( GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
    }


    Material* negRadMat = m_modifier->getSphereDisplayMaterialNegRadius();

    Matrix3 meshToWorld;
    // display little spheres on top of the normals
    for ( int i=0; i<numNormals; ++i )
    {
        const float& radius = scalesValues[i] * 0.1f;

        meshToWorld.IdentityMatrix();
        meshToWorld.PreTranslate( transformedPositionsAndNormals[ 2*i + 1 ] );
        // scale the mesh with radius
        meshToWorld.PreScale( Point3( radius, radius, radius ) );
        gw->setTransform( meshToWorld );

        Material* renderMat = ( radius > 0 ) ? material : negRadMat;
        // set material and transparency
        gw->setMaterial( *renderMat );

        // render the mesh
        displayMesh.render( gw, renderMat, NULL, COMP_ALL );
    }
    gw->setTransparency(0);
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
