// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Modifiers/Constraints/Wheel/hctWheelConstraintModifier.h>

/*static*/ Mesh hctWheelConstraintModifier::m_parentSpaceMesh;
/*static*/ Mesh hctWheelConstraintModifier::m_parentSuspensionMesh;
/*static*/ Mesh hctWheelConstraintModifier::m_childSpaceMesh;
/*static*/ Mesh hctWheelConstraintModifier::m_halfCylinder;
/*static*/ Mesh hctWheelConstraintModifier::m_middleVolume;
/*static*/ bool hctWheelConstraintModifier::m_staticsMeshesSetup = false;

class hkWheelConstraintModifierDesc : public ClassDesc2
{
    public:
        int             IsPublic() { return TRUE; }
        void *          Create (BOOL loading = FALSE) { return new hctWheelConstraintModifier(this); }
        const MCHAR *   ClassName() { return GetString(IDS_WHEEL_MODIFIER_CLASS_NAME); }
        SClass_ID       SuperClassID() { return OSM_CLASS_ID; }
        Class_ID        ClassID() { return HK_CONSTRAINT_WHEEL_CLASS_ID; }
        const MCHAR*    Category() { return GetString(IDS_HAVOK_MODIFIERS_CATEGORY); }
        const MCHAR*    InternalName() { return _T("hkWheelConstraintModifier");; }
        HINSTANCE       HInstance() { return hInstance; }
};

ClassDesc2* getHkWheelConstraintModifierDesc()
{
    static hkWheelConstraintModifierDesc modifierDesc;
    return &modifierDesc;
}

class hkWheelConstraintDescriptor : public hctConstraintDescriptor
{
    public:

        // Return the Class Descriptor of the modifier implementing the constraint
        /*virtual*/ ClassDesc2* getClassDesc() const {return getHkWheelConstraintModifierDesc();}

        // Return the constraint name (name for the pblock)
        /*virtual*/ MCHAR* getConstraintName() const {return TEXT("hkWheelConstraint");}

        // Return a string ID for the Class Name
        /*virtual*/ int getClassNameStringID () const {return IDS_WHEEL_MODIFIER_CLASS_NAME;}

        // Return a string ID for the Object NAme
        /*virtual*/ int getObjectNameStringID () const {return IDS_WHEEL_MODIFIER_OBJECT_NAME;}

    public:

        // Constructor, sets up the paramblockdesc2
        hkWheelConstraintDescriptor(): hctConstraintDescriptor()
        {
            m_commonSpacesPBD = hctConstraintModifier::setupCommonSpacesPBD(this);

            hctConstraintModifier::addCommonFPInterface(this);
        }
};

static hkWheelConstraintDescriptor g_wheelConstraintDescriptor;

hctConstraintDescriptor* hctWheelConstraintModifier::getConstraintDescriptor()
{
    return &g_wheelConstraintDescriptor;
}

// Null class so order is as we want it
static ParamBlockDesc2 hkWheel_ParamBlockDesc
    ( PB_WHEEL_MOD_PBLOCK, TEXT("hkWheelConstraintMerge"),  0, getHkWheelConstraintModifierDesc(), P_AUTO_CONSTRUCT + P_AUTO_UI, PB_WHEEL_MOD_PBLOCK,

    // Rollout; we use the default combobox handler dlg proc to handle combobox changes
    IDD_WHEEL_MODIFIER_ROLLOUT_PROPERTIES,
    IDS_WHEEL_MODIFIER_ROLLOUT_PROPERTIES,
    0, 0, hctBasicModifier::getComboBoxHandlerDlgProc(),

    PA_WHEEL_MOD_SUSPENSION_OFFSET,
        _T("suspensionOffset"), TYPE_ANGLE, P_RESET_DEFAULT | P_ANIMATABLE, IDS_WHEEL_MODIFIER_PA_SUSPENSION_OFFSET,
        p_default, 0.0f,
        p_range,    -180.0f, 180.0f,
        p_ui,       TYPE_SPINNER, EDITTYPE_FLOAT,
                    IDC_ED_OFFSET, IDC_SP_OFFSET, SPIN_AUTOSCALE,
        p_end,

    PA_WHEEL_MOD_SUSPENSION_LIMIT_MIN,
        _T("suspensionLimitMin"), TYPE_WORLD, P_RESET_DEFAULT | P_ANIMATABLE, IDS_WHEEL_MODIFIER_PA_SUSPENSION_LIMIT_MIN,
        p_default,      -1.0f, // Meters - will be scaled in the constructor (EXP-1006)
        p_range,        -10000000.0f, 0.0f,
        p_ui,           TYPE_SPINNER, EDITTYPE_UNIVERSE,
            IDC_ED_LIMIT_MIN, IDC_SP_LIMIT_MIN,
            SPIN_AUTOSCALE,
        p_end,

    PA_WHEEL_MOD_SUSPENSION_LIMIT_MAX,
        _T("suspensionLimitMax"), TYPE_WORLD, P_RESET_DEFAULT | P_ANIMATABLE, IDS_WHEEL_MODIFIER_PA_SUSPENSION_LIMIT_MAX,
        p_default,      1.0f, // Meters - will be scaled in the constructor (EXP-1006)
        p_range,        0.0f, 10000000.0f,
        p_ui,           TYPE_SPINNER, EDITTYPE_UNIVERSE,
            IDC_ED_LIMIT_MAX, IDC_SP_LIMIT_MAX,
            SPIN_AUTOSCALE,
        p_end,

    PA_WHEEL_MOD_SUSPENSION_STRENGTH,
        _T("suspensionStrength"), TYPE_FLOAT, P_RESET_DEFAULT | P_ANIMATABLE, IDS_WHEEL_MODIFIER_PA_SUSPENSION_STRENGTH,
        p_default, 0.0f,
        p_range, 0.0f, 10000000.0f,
        p_ui,           TYPE_SPINNER, EDITTYPE_FLOAT,
            IDC_ED_SUSPENSION_STRENGTH, IDC_SP_SUSPENSION_STRENGTH,
            SPIN_AUTOSCALE,
        p_end,

    PA_WHEEL_MOD_SUSPENSION_DAMPING,
        _T("suspensionDamping"), TYPE_FLOAT, P_RESET_DEFAULT | P_ANIMATABLE, IDS_WHEEL_MODIFIER_PA_SUSPENSION_DAMPING,
        p_default, 0.0f,
        p_range, 0.0f, 10000000.0f,
        p_ui,           TYPE_SPINNER, EDITTYPE_FLOAT,
            IDC_ED_SUSPENSION_DAMPING, IDC_SP_SUSPENSION_DAMPING,
            SPIN_AUTOSCALE,
        p_end,

    p_end

    );


//Constructor/Destructor
hctWheelConstraintModifier::hctWheelConstraintModifier(ClassDesc2* theClassDesc)
: hctConstraintModifier(theClassDesc)
{
    setupStaticMeshes();

    // EXP-1006 : Set the max/max suspension limits defaults based on system units
    const float metersPerUnit = float(GetMasterScale(UNITS_METERS));
    IParamBlock2* pblock2 = GetParamBlock(PB_WHEEL_MOD_PBLOCK);
    if (pblock2)
    {
        // Limit min
        {
            const float meters = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MIN);
            const float units = meters / metersPerUnit;
            pblock2->SetValue(PA_WHEEL_MOD_SUSPENSION_LIMIT_MIN, 0, units);
        }

        // Limit max
        {
            const float meters = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MAX);
            const float units = meters / metersPerUnit;
            pblock2->SetValue(PA_WHEEL_MOD_SUSPENSION_LIMIT_MAX, 0, units);
        }

    }


}

hctWheelConstraintModifier::~hctWheelConstraintModifier()
{

}

void hctWheelConstraintModifier::getSuspensionTransform (TimeValue t, INode* inode, Matrix3& transformOut)
{
    getSubobjectTransform(SOBJ_CONSTRAINT_MOD_PARENT_SPACE, t, inode, transformOut);

    IParamBlock2* pblock2 = GetParamBlock(PB_WHEEL_MOD_PBLOCK);
    const float angle = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_OFFSET, t);

    transformOut.PreRotateX(angle);
}

void hctWheelConstraintModifier::getTopVolumeDisplayTransform (TimeValue t, INode* inode, ViewExp* vpt, Matrix3& transformOut)
{
    IParamBlock2* pblock2 = GetParamBlock(PB_WHEEL_MOD_PBLOCK);
    const float limit = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MAX,t );

    getSuspensionTransform(t, inode, transformOut);

    // Add the distance upwards
    transformOut.PreTranslate(Point3(0.0f,0.0f,limit));

    // Scale to display
    const float scale = getScaleDisplay(t, inode, vpt);
    const Point3 scaleV (scale,scale,scale);
    transformOut.PreScale(scaleV, FALSE);
}

void hctWheelConstraintModifier::getBottomVolumeDisplayTransform (TimeValue t, INode* inode, ViewExp* vpt, Matrix3& transformOut)
{
    IParamBlock2* pblock2 = GetParamBlock(PB_WHEEL_MOD_PBLOCK);
    const float limit = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MIN,t );

    getSuspensionTransform (t, inode, transformOut);

    // Add the distance downwards
    transformOut.PreTranslate(Point3(0.0f,0.0f,limit));

    // Rotate to flip
    transformOut.PreRotateY(PI);

    // Scale to display
    const float scale = getScaleDisplay(t, inode, vpt);
    const Point3 scaleV (scale,scale,scale);
    transformOut.PreScale(scaleV, FALSE);
}

void hctWheelConstraintModifier::getMiddleVolumeDisplayTransform (TimeValue t, INode* inode, ViewExp* vpt, Matrix3& transformOut)
{
    IParamBlock2* pblock2 = GetParamBlock(PB_WHEEL_MOD_PBLOCK);
    const float limitMin = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MIN,t );
    const float limitMax = pblock2->GetFloat(PA_WHEEL_MOD_SUSPENSION_LIMIT_MAX,t );

    getSuspensionTransform(t, inode, transformOut);

    const float middle = 0.5f * (limitMin + limitMax);

    transformOut.PreTranslate(Point3(0.0f,0.0f,middle));

    // Z scale depends on the range
    const float zScale = limitMax - limitMin;

    // X,Y Scale to display
    const float scale = getScaleDisplay(t, inode, vpt);
    const Point3 scaleV (scale,scale,zScale);
    transformOut.PreScale(scaleV, FALSE);

}

int hctWheelConstraintModifier::Display(TimeValue t, INode* inode, ViewExp *vpt, int flagst, ModContext *mc)
{

    // Default display : two axis
    const float scale = getScaleDisplay(t, inode, vpt);
    const Point3 scaleV (scale,scale,scale);

    GraphicsWindow *gw=vpt->getGW();

    // Backup original limits
    const DWORD savedLimits=gw->getRndLimits();


    // Draw child volume
    {
        Matrix3 childSpaceTM; getSubobjectTransform(SOBJ_CONSTRAINT_MOD_CHILD_SPACE, t, inode, childSpaceTM);
        Matrix3 scaledTM = childSpaceTM;
        scaledTM.PreScale(scaleV, FALSE);
        gw->setTransform(scaledTM);

        Material childMaterialsArray[] = {m_childSpaceMaterial};

        gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_WIREFRAME);
        gw->setMaterial(m_childSpaceMaterial);
        m_childSpaceMesh.render(gw, childMaterialsArray, NULL, COMP_ALL);

        gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_Z_BUFFER);
        gw->setMaterial(m_childSpaceMaterial);
        m_childSpaceMesh.render(gw, childMaterialsArray, NULL, COMP_ALL);

    }

    // Draw parent steering+rotation
    {
        Matrix3 parentSpaceTM; getSubobjectTransform(SOBJ_CONSTRAINT_MOD_PARENT_SPACE, t, inode, parentSpaceTM);

        Matrix3 scaledTM = parentSpaceTM;
        scaledTM.PreScale(scaleV, FALSE);
        gw->setTransform(scaledTM);

        // Draw axis
        {

            Material parentMaterialsArray[] = {m_parentSpaceMaterial};

            gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_WIREFRAME);
            gw->setMaterial(m_parentSpaceMaterial);
            m_parentSpaceMesh.render(gw, parentMaterialsArray, NULL, COMP_ALL);

            gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_FLAT | GW_Z_BUFFER);
            gw->setMaterial(m_parentSpaceMaterial);
            m_parentSpaceMesh.render(gw, parentMaterialsArray, NULL, COMP_ALL);
        }
    }

    // Draw parent suspension axis
    {
        Matrix3 parentSuspensionTM; getSuspensionTransform(t, inode, parentSuspensionTM);

        Matrix3 scaledTM = parentSuspensionTM;
        scaledTM.PreScale(scaleV, FALSE);
        gw->setTransform(scaledTM);

        // Draw axis
        {

            Material parentMaterialsArray[] = {m_parentSpaceMaterial};

            gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_WIREFRAME);
            gw->setMaterial(m_parentSpaceMaterial);
            m_parentSuspensionMesh.render(gw, parentMaterialsArray, NULL, COMP_ALL);

            gw->setRndLimits(GW_BACKCULL | (GW_LIGHTING) | GW_FLAT | GW_Z_BUFFER);
            gw->setMaterial(m_parentSpaceMaterial);
            m_parentSuspensionMesh.render(gw, parentMaterialsArray, NULL, COMP_ALL);
        }

    }

    // Draw suspension limits
    {
        Material limitsMaterialsArray[] = {m_limitsMaterial};

        // TOP disc
        {
            Matrix3 topSpaceTM; getTopVolumeDisplayTransform(t, inode, vpt, topSpaceTM);
            gw->setTransform(topSpaceTM);

            // note : do not use GW_LIGHTING with wireframe and alpha
            gw->setRndLimits(GW_BACKCULL | GW_WIREFRAME);
            gw->setMaterial(m_limitsMaterial);
            m_halfCylinder.render(gw, limitsMaterialsArray, NULL, COMP_ALL);

            gw->setRndLimits(GW_BACKCULL | GW_LIGHTING | /*GW_Z_BUFFER |*/ GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
            gw->setTransparency(GW_TRANSPARENCY | GW_TRANSPARENT_PASS);
            gw->setMaterial(m_limitsMaterial);
            m_halfCylinder.render(gw, limitsMaterialsArray, NULL, COMP_ALL);
            gw->setTransparency(0);
        }

        // BOTTOM disc
        {
            Matrix3 bottomSpaceTM; getBottomVolumeDisplayTransform(t, inode, vpt, bottomSpaceTM);
            gw->setTransform(bottomSpaceTM);

            // note : do not use GW_LIGHTING with wireframe and alpha
            gw->setRndLimits(GW_BACKCULL | GW_WIREFRAME);
            gw->setMaterial(m_limitsMaterial);
            m_halfCylinder.render(gw, limitsMaterialsArray, NULL, COMP_ALL);

            gw->setRndLimits(GW_BACKCULL | GW_LIGHTING | /*GW_Z_BUFFER |*/ GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
            gw->setTransparency(GW_TRANSPARENCY | GW_TRANSPARENT_PASS);
            gw->setMaterial(m_limitsMaterial);
            m_halfCylinder.render(gw, limitsMaterialsArray, NULL, COMP_ALL);
            gw->setTransparency(0);
        }

        // Middle volume
        {
            Matrix3 middleVolumeTM; getMiddleVolumeDisplayTransform(t, inode, vpt, middleVolumeTM);
            gw->setTransform(middleVolumeTM);

            // note : do not use GW_LIGHTING with wireframe and alpha
            gw->setRndLimits(GW_BACKCULL | GW_WIREFRAME);
            gw->setMaterial(m_limitsMaterial);
            m_middleVolume.render(gw, limitsMaterialsArray, NULL, COMP_ALL);

            gw->setRndLimits(GW_BACKCULL | GW_LIGHTING | /*GW_Z_BUFFER |*/ GW_TRANSPARENCY | GW_TRANSPARENT_PASS );
            gw->setTransparency(GW_TRANSPARENCY | GW_TRANSPARENT_PASS);
            gw->setMaterial(m_limitsMaterial);
            m_middleVolume.render(gw, limitsMaterialsArray, NULL, COMP_ALL);
            gw->setTransparency(0);

        }
    }

    // Restore original limits
    gw->setRndLimits(savedLimits);

    return 0;
}




void hctWheelConstraintModifier::GetWorldBoundBox(TimeValue t,INode* inode, ViewExp *vpt, Box3& box3, ModContext *mc)
{
    const Box3 parentSpaceBoxLocal = m_parentSpaceMesh.getBoundingBox();
    const Box3 childSpaceBoxLocal = m_childSpaceMesh.getBoundingBox();
    const Box3 halfCylinderBoxLocal = m_halfCylinder.getBoundingBox();
    const Box3 parentSuspensionBoxLocal = m_parentSuspensionMesh.getBoundingBox();

    const float scale = getScaleDisplay(t, inode, vpt);
    const Point3 scaleV (scale,scale,scale);

    Matrix3 parentSpaceTM;
    getSubobjectTransform(SOBJ_CONSTRAINT_MOD_PARENT_SPACE, t, inode, parentSpaceTM);
    parentSpaceTM.PreScale(scaleV, FALSE);

    Matrix3 suspensionSpaceTM;
    getSuspensionTransform(t, inode, suspensionSpaceTM);
    suspensionSpaceTM.PreScale(scaleV, FALSE);

    Matrix3 childSpaceTM;
    getSubobjectTransform(SOBJ_CONSTRAINT_MOD_CHILD_SPACE, t, inode, childSpaceTM);
    childSpaceTM.PreScale(scaleV, FALSE);

    Matrix3 topHalfTM;
    getTopVolumeDisplayTransform(t, inode, vpt, topHalfTM);
    Matrix3 bottomHalfTM;
    getBottomVolumeDisplayTransform(t, inode, vpt, bottomHalfTM);

    const Box3 parentSpaceBoxWorld = parentSpaceBoxLocal * parentSpaceTM;
    const Box3 parentSuspensionBoxWorld = parentSuspensionBoxLocal * suspensionSpaceTM;
    const Box3 childSpaceBoxWorld = childSpaceBoxLocal * childSpaceTM;
    const Box3 topHalfBoxWorld = halfCylinderBoxLocal * topHalfTM;
    const Box3 bottomHalfBoxWorld = halfCylinderBoxLocal * bottomHalfTM;

    box3 = parentSpaceBoxWorld;
    box3 += parentSuspensionBoxWorld;
    box3 += childSpaceBoxWorld;
    box3 += topHalfBoxWorld;
    box3 += bottomHalfBoxWorld;

    // note: we don't add the bbox of the middle volume because by adding the top and the bottom bits we already enclose it
}


void hctWheelConstraintModifier::setupStaticMeshes()
{
    if (m_staticsMeshesSetup) return;

    // The axis have a length of 100.0 - Make it unit size
    const float vertexScale = 1.0f/100.0f;

    // PARENT SPACE (ROTATION + STEERING)
    {
        static float theVertices[] = {
            0.0000f,  0.0000f,  0.0000f ,    -0.0000f,  3.6820f,  0.0000f ,  -2.6035f,  2.6035f,  0.0000f ,
            -3.6820f, -0.0000f,  0.0000f ,   -2.6035f, -2.6035f,  0.0000f ,   0.0000f, -3.6820f,  0.0000f ,
            2.6035f, -2.6035f,  0.0000f ,     3.6820f,  0.0000f,  0.0000f ,   2.6035f,  2.6035f,  0.0000f ,
            -0.0000f,  3.6820f, 22.0917f ,   -2.6035f,  2.6035f, 22.0917f ,  -3.6820f,  0.0000f, 22.0917f ,
            -2.6035f, -2.6035f, 22.0917f ,    0.0000f, -3.6820f, 22.0917f ,   2.6035f, -2.6035f, 22.0917f ,
            3.6820f,  0.0000f, 22.0917f ,     2.6035f,  2.6035f, 22.0917f ,  -0.0000f,  0.0000f, 22.0917f ,
            0.0000f, -0.0000f, 22.0917f ,     7.3639f,  0.0000f, 22.0917f ,   5.2071f,  5.2071f, 22.0917f ,
            0.0000f,  7.3639f, 22.0917f ,    -5.2071f,  5.2071f, 22.0917f ,  -7.3639f, -0.0000f, 22.0917f ,
            -5.2071f, -5.2071f, 22.0917f ,    0.0000f, -7.3639f, 22.0917f ,   5.2071f, -5.2071f, 22.0917f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f ,
            -0.0000f, 10.0000f, 11.0349f ,   -0.0000f, 10.0000f, 12.0349f ,  -0.0000f,  8.0000f, 12.0349f ,
            -0.0000f,  8.0000f, 11.0349f ,    3.8268f,  9.2388f, 11.0349f ,   3.8268f,  9.2388f, 12.0349f ,
            3.0615f,  7.3910f, 12.0349f ,     3.0615f,  7.3910f, 11.0349f ,   7.0711f,  7.0711f, 11.0349f ,
            7.0711f,  7.0711f, 12.0349f ,     5.6569f,  5.6569f, 12.0349f ,   5.6569f,  5.6569f, 11.0349f ,
            9.2388f,  3.8268f, 11.0349f ,     9.2388f,  3.8268f, 12.0349f ,   7.3910f,  3.0615f, 12.0349f ,
            7.3910f,  3.0615f, 11.0349f ,    10.0000f, -0.0000f, 11.0349f ,  10.0000f, -0.0000f, 12.0349f ,
            8.0000f, -0.0000f, 12.0349f ,     8.0000f, -0.0000f, 11.0349f ,   9.2388f, -3.8268f, 11.0349f ,
            9.2388f, -3.8268f, 12.0349f ,     7.3910f, -3.0615f, 12.0349f ,   7.3910f, -3.0615f, 11.0349f ,
            7.0711f, -7.0711f, 11.0349f ,     7.0711f, -7.0711f, 12.0349f ,   5.6569f, -5.6569f, 12.0349f ,
            5.6569f, -5.6569f, 11.0349f ,     3.8268f, -9.2388f, 11.0349f ,   3.8268f, -9.2388f, 12.0349f ,
            3.0615f, -7.3910f, 12.0349f ,     3.0615f, -7.3910f, 11.0349f ,  -0.0000f,-10.0000f, 11.0349f ,
            -0.0000f,-10.0000f, 12.0349f ,   -0.0000f, -8.0000f, 12.0349f ,  -0.0000f, -8.0000f, 11.0349f ,
            -3.8268f, -9.2388f, 11.0349f ,   -3.8268f, -9.2388f, 12.0349f ,  -3.0615f, -7.3910f, 12.0349f ,
            -3.0615f, -7.3910f, 11.0349f ,   -7.0711f, -7.0711f, 11.0349f ,  -7.0711f, -7.0711f, 12.0349f ,
            -5.6569f, -5.6569f, 12.0349f ,   -5.6569f, -5.6569f, 11.0349f ,  -9.2388f, -3.8268f, 11.0349f ,
            -9.2388f, -3.8268f, 12.0349f ,   -7.3910f, -3.0615f, 12.0349f ,  -7.3910f, -3.0615f, 11.0349f ,
            -10.0000f,  0.0000f, 11.0349f , -10.0000f,  0.0000f, 12.0349f ,  -8.0000f,  0.0000f, 12.0349f ,
            -8.0000f,  0.0000f, 11.0349f ,   -9.2388f,  3.8268f, 11.0349f ,  -9.2388f,  3.8268f, 12.0349f ,
            -7.3910f,  3.0615f, 12.0349f ,   -7.3910f,  3.0615f, 11.0349f ,  -7.0711f,  7.0711f, 11.0349f ,
            -7.0711f,  7.0711f, 12.0349f ,   -5.6569f,  5.6569f, 12.0349f ,  -5.6569f,  5.6569f, 11.0349f ,
            -3.8268f,  9.2388f, 11.0349f ,   -3.8268f,  9.2388f, 12.0349f ,  -3.0615f,  7.3910f, 12.0349f ,
            -3.0615f,  7.3910f, 11.0349f ,    0.0000f,  0.0000f,  0.0000f ,  -0.0000f,  5.0000f, -0.0000f ,
            -0.0000f,  3.5355f,  3.5355f ,   -0.0000f, -0.0000f,  5.0000f ,  -0.0000f, -3.5355f,  3.5355f ,
            0.0000f, -5.0000f, -0.0000f ,     0.0000f, -3.5355f, -3.5355f ,   0.0000f,  0.0000f, -5.0000f ,
            0.0000f,  3.5355f, -3.5355f ,    30.0000f,  5.0000f,  0.0000f ,  30.0000f,  3.5355f,  3.5355f ,
            30.0000f,  0.0000f,  5.0000f ,   30.0000f, -3.5355f,  3.5355f ,  30.0000f, -5.0000f,  0.0000f ,
            30.0000f, -3.5355f, -3.5355f ,   30.0000f,  0.0000f, -5.0000f ,  30.0000f,  3.5355f, -3.5355f ,
            30.0000f,  0.0000f,  0.0000f ,   30.0000f,  0.0000f,-10.0000f ,  30.0000f,  7.0711f, -7.0711f ,
            30.0000f, 10.0000f,  0.0000f ,   30.0000f,  7.0711f,  7.0711f ,  30.0000f, -0.0000f, 10.0000f ,
            30.0000f, -7.0711f,  7.0711f ,   30.0000f,-10.0000f, -0.0000f ,  30.0000f, -7.0711f, -7.0711f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f };

        m_parentSpaceMesh.setNumVerts(135);

        for (int v=0; v<135; v++)
        {
            m_parentSpaceMesh.setVert(v, vertexScale * Point3(theVertices[v*3], theVertices[v*3+1], theVertices[v*3+2]));
        }

        static int theFaces[] = {
            0,  2,  1, 0,1,0, 1,1 ,       0,  3,  2, 0,1,0, 1,1 ,         0,  4,  3, 0,1,0, 1,1 ,
            0,  5,  4, 0,1,0, 1,1 ,       0,  6,  5, 0,1,0, 1,1 ,         0,  7,  6, 0,1,0, 1,1 ,
            0,  8,  7, 0,1,0, 1,1 ,       0,  1,  8, 0,1,0, 1,1 ,         1, 10,  9, 0,1,1, 4,2 ,
            1,  2, 10, 1,1,0, 4,2 ,       2, 11, 10, 0,1,1, 4,2 ,         2,  3, 11, 1,1,0, 4,2 ,
            3, 12, 11, 0,1,1, 4,2 ,       3,  4, 12, 1,1,0, 4,2 ,         4, 13, 12, 0,1,1, 4,2 ,
            4,  5, 13, 1,1,0, 4,2 ,       5, 14, 13, 0,1,1, 4,2 ,         5,  6, 14, 1,1,0, 4,2 ,
            6, 15, 14, 0,1,1, 4,2 ,       6,  7, 15, 1,1,0, 4,2 ,         7, 16, 15, 0,1,1, 4,2 ,
            7,  8, 16, 1,1,0, 4,2 ,       8,  9, 16, 0,1,1, 4,2 ,         8,  1,  9, 1,1,0, 4,2 ,
            17,  9, 10, 0,1,0, 1,0 ,         17, 10, 11, 0,1,0, 1,0 ,        17, 11, 12, 0,1,0, 1,0 ,
            17, 12, 13, 0,1,0, 1,0 ,         17, 13, 14, 0,1,0, 1,0 ,        17, 14, 15, 0,1,0, 1,0 ,
            17, 15, 16, 0,1,0, 1,0 ,         17, 16,  9, 0,1,0, 1,0 ,        18, 20, 19, 0,1,0, 1,1 ,
            18, 21, 20, 0,1,0, 1,1 ,         18, 22, 21, 0,1,0, 1,1 ,        18, 23, 22, 0,1,0, 1,1 ,
            18, 24, 23, 0,1,0, 1,1 ,         18, 25, 24, 0,1,0, 1,1 ,        18, 26, 25, 0,1,0, 1,1 ,
            18, 19, 26, 0,1,0, 1,1 ,         19, 28, 27, 0,1,1, 4,2 ,        19, 20, 28, 1,1,0, 4,2 ,
            20, 29, 28, 0,1,1, 4,2 ,         20, 21, 29, 1,1,0, 4,2 ,        21, 30, 29, 0,1,1, 4,2 ,
            21, 22, 30, 1,1,0, 4,2 ,         22, 31, 30, 0,1,1, 4,2 ,        22, 23, 31, 1,1,0, 4,2 ,
            23, 32, 31, 0,1,1, 4,2 ,         23, 24, 32, 1,1,0, 4,2 ,        24, 33, 32, 0,1,1, 4,2 ,
            24, 25, 33, 1,1,0, 4,2 ,         25, 34, 33, 0,1,1, 4,2 ,        25, 26, 34, 1,1,0, 4,2 ,
            26, 27, 34, 0,1,1, 4,2 ,         26, 19, 27, 1,1,0, 4,2 ,        35, 27, 28, 0,1,0, 1,0 ,
            35, 28, 29, 0,1,0, 1,0 ,         35, 29, 30, 0,1,0, 1,0 ,        35, 30, 31, 0,1,0, 1,0 ,
            35, 31, 32, 0,1,0, 1,0 ,         35, 32, 33, 0,1,0, 1,0 ,        35, 33, 34, 0,1,0, 1,0 ,
            35, 34, 27, 0,1,0, 1,0 ,         36, 41, 40, 0,1,1, 2,2 ,        36, 37, 41, 1,1,0, 2,2 ,
            37, 42, 41, 0,1,1, 3,0 ,         37, 38, 42, 1,1,0, 3,0 ,        38, 43, 42, 0,1,1, 2,3 ,
            38, 39, 43, 1,1,0, 2,3 ,         39, 40, 43, 0,1,1, 3,1 ,        39, 36, 40, 1,1,0, 3,1 ,
            40, 45, 44, 0,1,1, 2,2 ,         40, 41, 45, 1,1,0, 2,2 ,        41, 46, 45, 0,1,1, 3,0 ,
            41, 42, 46, 1,1,0, 3,0 ,         42, 47, 46, 0,1,1, 2,3 ,        42, 43, 47, 1,1,0, 2,3 ,
            43, 44, 47, 0,1,1, 3,1 ,         43, 40, 44, 1,1,0, 3,1 ,        44, 49, 48, 0,1,1, 2,2 ,
            44, 45, 49, 1,1,0, 2,2 ,         45, 50, 49, 0,1,1, 3,0 ,        45, 46, 50, 1,1,0, 3,0 ,
            46, 51, 50, 0,1,1, 2,3 ,         46, 47, 51, 1,1,0, 2,3 ,        47, 48, 51, 0,1,1, 3,1 ,
            47, 44, 48, 1,1,0, 3,1 ,         48, 53, 52, 0,1,1, 2,2 ,        48, 49, 53, 1,1,0, 2,2 ,
            49, 54, 53, 0,1,1, 3,0 ,         49, 50, 54, 1,1,0, 3,0 ,        50, 55, 54, 0,1,1, 2,3 ,
            50, 51, 55, 1,1,0, 2,3 ,         51, 52, 55, 0,1,1, 3,1 ,        51, 48, 52, 1,1,0, 3,1 ,
            52, 57, 56, 0,1,1, 2,2 ,         52, 53, 57, 1,1,0, 2,2 ,        53, 58, 57, 0,1,1, 3,0 ,
            53, 54, 58, 1,1,0, 3,0 ,         54, 59, 58, 0,1,1, 2,3 ,        54, 55, 59, 1,1,0, 2,3 ,
            55, 56, 59, 0,1,1, 3,1 ,         55, 52, 56, 1,1,0, 3,1 ,        56, 61, 60, 0,1,1, 2,2 ,
            56, 57, 61, 1,1,0, 2,2 ,         57, 62, 61, 0,1,1, 3,0 ,        57, 58, 62, 1,1,0, 3,0 ,
            58, 63, 62, 0,1,1, 2,3 ,         58, 59, 63, 1,1,0, 2,3 ,        59, 60, 63, 0,1,1, 3,1 ,
            59, 56, 60, 1,1,0, 3,1 ,         60, 65, 64, 0,1,1, 2,2 ,        60, 61, 65, 1,1,0, 2,2 ,
            61, 66, 65, 0,1,1, 3,0 ,         61, 62, 66, 1,1,0, 3,0 ,        62, 67, 66, 0,1,1, 2,3 ,
            62, 63, 67, 1,1,0, 2,3 ,         63, 64, 67, 0,1,1, 3,1 ,        63, 60, 64, 1,1,0, 3,1 ,
            64, 69, 68, 0,1,1, 2,2 ,         64, 65, 69, 1,1,0, 2,2 ,        65, 70, 69, 0,1,1, 3,0 ,
            65, 66, 70, 1,1,0, 3,0 ,         66, 71, 70, 0,1,1, 2,3 ,        66, 67, 71, 1,1,0, 2,3 ,
            67, 68, 71, 0,1,1, 3,1 ,         67, 64, 68, 1,1,0, 3,1 ,        68, 73, 72, 0,1,1, 2,2 ,
            68, 69, 73, 1,1,0, 2,2 ,         69, 74, 73, 0,1,1, 3,0 ,        69, 70, 74, 1,1,0, 3,0 ,
            70, 75, 74, 0,1,1, 2,3 ,         70, 71, 75, 1,1,0, 2,3 ,        71, 72, 75, 0,1,1, 3,1 ,
            71, 68, 72, 1,1,0, 3,1 ,         72, 77, 76, 0,1,1, 2,2 ,        72, 73, 77, 1,1,0, 2,2 ,
            73, 78, 77, 0,1,1, 3,0 ,         73, 74, 78, 1,1,0, 3,0 ,        74, 79, 78, 0,1,1, 2,3 ,
            74, 75, 79, 1,1,0, 2,3 ,         75, 76, 79, 0,1,1, 3,1 ,        75, 72, 76, 1,1,0, 3,1 ,
            76, 81, 80, 0,1,1, 2,2 ,         76, 77, 81, 1,1,0, 2,2 ,        77, 82, 81, 0,1,1, 3,0 ,
            77, 78, 82, 1,1,0, 3,0 ,         78, 83, 82, 0,1,1, 2,3 ,        78, 79, 83, 1,1,0, 2,3 ,
            79, 80, 83, 0,1,1, 3,1 ,         79, 76, 80, 1,1,0, 3,1 ,        80, 85, 84, 0,1,1, 2,2 ,
            80, 81, 85, 1,1,0, 2,2 ,         81, 86, 85, 0,1,1, 3,0 ,        81, 82, 86, 1,1,0, 3,0 ,
            82, 87, 86, 0,1,1, 2,3 ,         82, 83, 87, 1,1,0, 2,3 ,        83, 84, 87, 0,1,1, 3,1 ,
            83, 80, 84, 1,1,0, 3,1 ,         84, 89, 88, 0,1,1, 2,2 ,        84, 85, 89, 1,1,0, 2,2 ,
            85, 90, 89, 0,1,1, 3,0 ,         85, 86, 90, 1,1,0, 3,0 ,        86, 91, 90, 0,1,1, 2,3 ,
            86, 87, 91, 1,1,0, 2,3 ,         87, 88, 91, 0,1,1, 3,1 ,        87, 84, 88, 1,1,0, 3,1 ,
            88, 93, 92, 0,1,1, 2,2 ,         88, 89, 93, 1,1,0, 2,2 ,        89, 94, 93, 0,1,1, 3,0 ,
            89, 90, 94, 1,1,0, 3,0 ,         90, 95, 94, 0,1,1, 2,3 ,        90, 91, 95, 1,1,0, 2,3 ,
            91, 92, 95, 0,1,1, 3,1 ,         91, 88, 92, 1,1,0, 3,1 ,        92, 97, 96, 0,1,1, 2,2 ,
            92, 93, 97, 1,1,0, 2,2 ,         93, 98, 97, 0,1,1, 3,0 ,        93, 94, 98, 1,1,0, 3,0 ,
            94, 99, 98, 0,1,1, 2,3 ,         94, 95, 99, 1,1,0, 2,3 ,        95, 96, 99, 0,1,1, 3,1 ,
            95, 92, 96, 1,1,0, 3,1 ,         96, 37, 36, 0,1,1, 2,2 ,        96, 97, 37, 1,1,0, 2,2 ,
            97, 38, 37, 0,1,1, 3,0 ,         97, 98, 38, 1,1,0, 3,0 ,        98, 39, 38, 0,1,1, 2,3 ,
            98, 99, 39, 1,1,0, 2,3 ,         99, 36, 39, 0,1,1, 3,1 ,        99, 96, 36, 1,1,0, 3,1 ,
            100,102,101, 0,1,0, 1,1 ,       100,103,102, 0,1,0, 1,1 ,       100,104,103, 0,1,0, 1,1 ,
            100,105,104, 0,1,0, 1,1 ,       100,106,105, 0,1,0, 1,1 ,       100,107,106, 0,1,0, 1,1 ,
            100,108,107, 0,1,0, 1,1 ,       100,101,108, 0,1,0, 1,1 ,       101,110,109, 0,1,1, 4,2 ,
            101,102,110, 1,1,0, 4,2 ,       102,111,110, 0,1,1, 4,2 ,       102,103,111, 1,1,0, 4,2 ,
            103,112,111, 0,1,1, 4,2 ,       103,104,112, 1,1,0, 4,2 ,       104,113,112, 0,1,1, 4,2 ,
            104,105,113, 1,1,0, 4,2 ,       105,114,113, 0,1,1, 4,2 ,       105,106,114, 1,1,0, 4,2 ,
            106,115,114, 0,1,1, 4,2 ,       106,107,115, 1,1,0, 4,2 ,       107,116,115, 0,1,1, 4,2 ,
            107,108,116, 1,1,0, 4,2 ,       108,109,116, 0,1,1, 4,2 ,       108,101,109, 1,1,0, 4,2 ,
            117,119,118, 0,1,0, 1,1 ,       117,120,119, 0,1,0, 1,1 ,       117,121,120, 0,1,0, 1,1 ,
            117,122,121, 0,1,0, 1,1 ,       117,123,122, 0,1,0, 1,1 ,       117,124,123, 0,1,0, 1,1 ,
            117,125,124, 0,1,0, 1,1 ,       117,118,125, 0,1,0, 1,1 ,       118,127,126, 0,1,1, 4,2 ,
            118,119,127, 1,1,0, 4,2 ,       119,128,127, 0,1,1, 4,2 ,       119,120,128, 1,1,0, 4,2 ,
            120,129,128, 0,1,1, 4,2 ,       120,121,129, 1,1,0, 4,2 ,       121,130,129, 0,1,1, 4,2 ,
            121,122,130, 1,1,0, 4,2 ,       122,131,130, 0,1,1, 4,2 ,       122,123,131, 1,1,0, 4,2 ,
            123,132,131, 0,1,1, 4,2 ,       123,124,132, 1,1,0, 4,2 ,       124,133,132, 0,1,1, 4,2 ,
            124,125,133, 1,1,0, 4,2 ,       125,126,133, 0,1,1, 4,2 ,       125,118,126, 1,1,0, 4,2 ,
            134,126,127, 0,1,0, 1,0 ,       134,127,128, 0,1,0, 1,0 ,       134,128,129, 0,1,0, 1,0 ,
            134,129,130, 0,1,0, 1,0 ,       134,130,131, 0,1,0, 1,0 ,       134,131,132, 0,1,0, 1,0 ,
            134,132,133, 0,1,0, 1,0 ,       134,133,126, 0,1,0, 1,0 };

        m_parentSpaceMesh.setNumFaces(248);

        for (int f=0; f<248; f++)
        {
            m_parentSpaceMesh.faces[f].setVerts(theFaces[f*8], theFaces[f*8+1], theFaces[f*8+2]);
            m_parentSpaceMesh.faces[f].setEdgeVisFlags(theFaces[f*8+3], theFaces[f*8+4], theFaces[f*8+5]);
            m_parentSpaceMesh.faces[f].setSmGroup(theFaces[f*8+6]);
            m_parentSpaceMesh.faces[f].setMatID( (MtlID)theFaces[f*8+7]);
        }

        m_parentSpaceMesh.buildNormals();
        m_parentSpaceMesh.BuildStripsAndEdges();


    }

    // SUSPENSION
    {
        static float theVertices[] = {
            -10.0000f,-50.0000f, -0.7205f ,  10.0000f,-50.0000f, -0.7205f , -10.0000f, 50.0000f, -0.7205f ,
            10.0000f, 50.0000f, -0.7205f ,  -10.0000f,-50.0000f,  0.8359f ,  10.0000f,-50.0000f,  0.8359f ,
            -10.0000f, 50.0000f,  0.8359f ,  10.0000f, 50.0000f,  0.8359f ,  -0.0000f,  3.6820f,  0.0000f ,
            -2.6035f,  2.6035f,  0.0000f ,   -3.6820f, -0.0000f,  0.0000f ,  -2.6035f, -2.6035f,  0.0000f ,
            0.0000f, -3.6820f,  0.0000f ,     2.6035f, -2.6035f,  0.0000f ,   3.6820f,  0.0000f,  0.0000f ,
            2.6035f,  2.6035f,  0.0000f ,    -0.0000f,  3.6820f, 22.0917f ,  -2.6035f,  2.6035f, 22.0917f ,
            -3.6820f,  0.0000f, 22.0917f ,   -2.6035f, -2.6035f, 22.0917f ,   0.0000f, -3.6820f, 22.0917f ,
            2.6035f, -2.6035f, 22.0917f ,     3.6820f,  0.0000f, 22.0917f ,   2.6035f,  2.6035f, 22.0917f ,
            0.0000f, -0.0000f, 22.0917f ,     7.3639f,  0.0000f, 22.0917f ,   5.2071f,  5.2071f, 22.0917f ,
            0.0000f,  7.3639f, 22.0917f ,    -5.2071f,  5.2071f, 22.0917f ,  -7.3639f, -0.0000f, 22.0917f ,
            -5.2071f, -5.2071f, 22.0917f ,    0.0000f, -7.3639f, 22.0917f ,   5.2071f, -5.2071f, 22.0917f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f ,
            0.0000f, -0.0000f, 36.8195f ,     0.0000f, -0.0000f, 36.8195f ,   0.0000f, -0.0000f, 36.8195f };

        m_parentSuspensionMesh.setNumVerts(42);

        for (int v=0; v<42; v++)
        {
            m_parentSuspensionMesh.setVert(v, vertexScale * Point3(theVertices[v*3], theVertices[v*3+1], theVertices[v*3+2]));
        }

        static int theFaces[] = {
            0,  2,  3, 1,1,0, 2,1 ,       3,  1,  0, 1,1,0, 2,1 ,         4,  5,  7, 1,1,0, 3,0 ,
            7,  6,  4, 1,1,0, 3,0 ,       0,  1,  5, 1,1,0, 4,4 ,         5,  4,  0, 1,1,0, 4,4 ,
            1,  3,  7, 1,1,0, 5,3 ,       7,  5,  1, 1,1,0, 5,3 ,         3,  2,  6, 1,1,0, 6,5 ,
            6,  7,  3, 1,1,0, 6,5 ,       2,  0,  4, 1,1,0, 7,2 ,         4,  6,  2, 1,1,0, 7,2 ,
            8, 17, 16, 0,1,1, 4,2 ,       8,  9, 17, 1,1,0, 4,2 ,         9, 18, 17, 0,1,1, 4,2 ,
            9, 10, 18, 1,1,0, 4,2 ,      10, 19, 18, 0,1,1, 4,2 ,        10, 11, 19, 1,1,0, 4,2 ,
            11, 20, 19, 0,1,1, 4,2 ,         11, 12, 20, 1,1,0, 4,2 ,        12, 21, 20, 0,1,1, 4,2 ,
            12, 13, 21, 1,1,0, 4,2 ,         13, 22, 21, 0,1,1, 4,2 ,        13, 14, 22, 1,1,0, 4,2 ,
            14, 23, 22, 0,1,1, 4,2 ,         14, 15, 23, 1,1,0, 4,2 ,        15, 16, 23, 0,1,1, 4,2 ,
            15,  8, 16, 1,1,0, 4,2 ,         24, 26, 25, 0,1,0, 1,1 ,        24, 27, 26, 0,1,0, 1,1 ,
            24, 28, 27, 0,1,0, 1,1 ,         24, 29, 28, 0,1,0, 1,1 ,        24, 30, 29, 0,1,0, 1,1 ,
            24, 31, 30, 0,1,0, 1,1 ,         24, 32, 31, 0,1,0, 1,1 ,        24, 25, 32, 0,1,0, 1,1 ,
            25, 34, 33, 0,1,1, 4,2 ,         25, 26, 34, 1,1,0, 4,2 ,        26, 35, 34, 0,1,1, 4,2 ,
            26, 27, 35, 1,1,0, 4,2 ,         27, 36, 35, 0,1,1, 4,2 ,        27, 28, 36, 1,1,0, 4,2 ,
            28, 37, 36, 0,1,1, 4,2 ,         28, 29, 37, 1,1,0, 4,2 ,        29, 38, 37, 0,1,1, 4,2 ,
            29, 30, 38, 1,1,0, 4,2 ,         30, 39, 38, 0,1,1, 4,2 ,        30, 31, 39, 1,1,0, 4,2 ,
            31, 40, 39, 0,1,1, 4,2 ,         31, 32, 40, 1,1,0, 4,2 ,        32, 33, 40, 0,1,1, 4,2 ,
            32, 25, 33, 1,1,0, 4,2 ,         41, 33, 34, 0,1,0, 1,0 ,        41, 34, 35, 0,1,0, 1,0 ,
            41, 35, 36, 0,1,0, 1,0 ,         41, 36, 37, 0,1,0, 1,0 ,        41, 37, 38, 0,1,0, 1,0 ,
            41, 38, 39, 0,1,0, 1,0 ,         41, 39, 40, 0,1,0, 1,0 ,        41, 40, 33, 0,1,0, 1,0 };

        m_parentSuspensionMesh.setNumFaces(60);

        for (int f=0; f<60; f++)
        {
            m_parentSuspensionMesh.faces[f].setVerts(theFaces[f*8], theFaces[f*8+1], theFaces[f*8+2]);
            m_parentSuspensionMesh.faces[f].setEdgeVisFlags(theFaces[f*8+3], theFaces[f*8+4], theFaces[f*8+5]);
            m_parentSuspensionMesh.faces[f].setSmGroup(theFaces[f*8+6]);
            m_parentSuspensionMesh.faces[f].setMatID((MtlID)theFaces[f*8+7]);
        }

        m_parentSuspensionMesh.buildNormals();
        m_parentSuspensionMesh.BuildStripsAndEdges();
    }


    // CHILD
    {
        static float theVertices[] = {
            -10.0000f,  0.0000f,  0.0000f , -10.0000f,  0.0000f,-50.0000f , -10.0000f, 15.4509f,-47.5528f ,
            -10.0000f, 29.3893f,-40.4508f , -10.0000f, 40.4509f,-29.3893f , -10.0000f, 47.5528f,-15.4508f ,
            -10.0000f, 50.0000f,  0.0000f , -10.0000f, 47.5528f, 15.4509f , -10.0000f, 40.4508f, 29.3893f ,
            -10.0000f, 29.3893f, 40.4509f , -10.0000f, 15.4508f, 47.5528f , -10.0000f, -0.0000f, 50.0000f ,
            -10.0000f,-15.4509f, 47.5528f , -10.0000f,-29.3893f, 40.4508f , -10.0000f,-40.4509f, 29.3893f ,
            -10.0000f,-47.5528f, 15.4508f , -10.0000f,-50.0000f, -0.0000f , -10.0000f,-47.5528f,-15.4509f ,
            -10.0000f,-40.4508f,-29.3893f , -10.0000f,-29.3893f,-40.4509f , -10.0000f,-15.4508f,-47.5528f ,
            10.0000f,  0.0000f,-50.0000f ,   10.0000f, 15.4509f,-47.5528f ,  10.0000f, 29.3893f,-40.4508f ,
            10.0000f, 40.4509f,-29.3893f ,   10.0000f, 47.5528f,-15.4508f ,  10.0000f, 50.0000f,  0.0000f ,
            10.0000f, 47.5528f, 15.4509f ,   10.0000f, 40.4508f, 29.3893f ,  10.0000f, 29.3893f, 40.4509f ,
            10.0000f, 15.4508f, 47.5528f ,   10.0000f, -0.0000f, 50.0000f ,  10.0000f,-15.4509f, 47.5528f ,
            10.0000f,-29.3893f, 40.4508f ,   10.0000f,-40.4509f, 29.3893f ,  10.0000f,-47.5528f, 15.4508f ,
            10.0000f,-50.0000f, -0.0000f ,   10.0000f,-47.5528f,-15.4509f ,  10.0000f,-40.4508f,-29.3893f ,
            10.0000f,-29.3893f,-40.4509f ,   10.0000f,-15.4508f,-47.5528f ,  10.0000f,  0.0000f, -0.0000f ,
            30.0000f,  0.0000f,  0.0000f ,   30.0000f,  0.0000f,-10.0000f ,  30.0000f,  7.0711f, -7.0711f ,
            30.0000f, 10.0000f,  0.0000f ,   30.0000f,  7.0711f,  7.0711f ,  30.0000f, -0.0000f, 10.0000f ,
            30.0000f, -7.0711f,  7.0711f ,   30.0000f,-10.0000f, -0.0000f ,  30.0000f, -7.0711f, -7.0711f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f ,
            50.0000f,  0.0000f,  0.0000f ,   50.0000f,  0.0000f,  0.0000f ,  50.0000f,  0.0000f,  0.0000f ,
            10.0000f,  5.0000f, -0.0000f ,   10.0000f,  3.5355f,  3.5355f ,  10.0000f, -0.0000f,  5.0000f ,
            10.0000f, -3.5355f,  3.5355f ,   10.0000f, -5.0000f, -0.0000f ,  10.0000f, -3.5355f, -3.5355f ,
            10.0000f,  0.0000f, -5.0000f ,   10.0000f,  3.5355f, -3.5355f ,  30.0000f,  5.0000f,  0.0000f ,
            30.0000f,  3.5355f,  3.5355f ,   30.0000f,  0.0000f,  5.0000f ,  30.0000f, -3.5355f,  3.5355f ,
            30.0000f, -5.0000f,  0.0000f ,   30.0000f, -3.5355f, -3.5355f ,  30.0000f,  0.0000f, -5.0000f ,
            30.0000f,  3.5355f, -3.5355f };

        m_childSpaceMesh.setNumVerts(76);

        for (int v=0; v<76; v++)
        {
            m_childSpaceMesh.setVert(v, vertexScale * Point3(theVertices[v*3], theVertices[v*3+1], theVertices[v*3+2]));
        }

        static int theFaces[] = {
            0,  2,  1, 0,1,0, 1,1 ,       0,  3,  2, 0,1,0, 1,1 ,         0,  4,  3, 0,1,0, 1,1 ,
            0,  5,  4, 0,1,0, 1,1 ,       0,  6,  5, 0,1,0, 1,1 ,         0,  7,  6, 0,1,0, 1,1 ,
            0,  8,  7, 0,1,0, 1,1 ,       0,  9,  8, 0,1,0, 1,1 ,         0, 10,  9, 0,1,0, 1,1 ,
            0, 11, 10, 0,1,0, 1,1 ,       0, 12, 11, 0,1,0, 1,1 ,         0, 13, 12, 0,1,0, 1,1 ,
            0, 14, 13, 0,1,0, 1,1 ,       0, 15, 14, 0,1,0, 1,1 ,         0, 16, 15, 0,1,0, 1,1 ,
            0, 17, 16, 0,1,0, 1,1 ,       0, 18, 17, 0,1,0, 1,1 ,         0, 19, 18, 0,1,0, 1,1 ,
            0, 20, 19, 0,1,0, 1,1 ,       0,  1, 20, 0,1,0, 1,1 ,         1, 22, 21, 0,1,1, 4,2 ,
            1,  2, 22, 1,1,0, 4,2 ,       2, 23, 22, 0,1,1, 4,2 ,         2,  3, 23, 1,1,0, 4,2 ,
            3, 24, 23, 0,1,1, 4,2 ,       3,  4, 24, 1,1,0, 4,2 ,         4, 25, 24, 0,1,1, 4,2 ,
            4,  5, 25, 1,1,0, 4,2 ,       5, 26, 25, 0,1,1, 4,2 ,         5,  6, 26, 1,1,0, 4,2 ,
            6, 27, 26, 0,1,1, 4,2 ,       6,  7, 27, 1,1,0, 4,2 ,         7, 28, 27, 0,1,1, 4,2 ,
            7,  8, 28, 1,1,0, 4,2 ,       8, 29, 28, 0,1,1, 4,2 ,         8,  9, 29, 1,1,0, 4,2 ,
            9, 30, 29, 0,1,1, 4,2 ,       9, 10, 30, 1,1,0, 4,2 ,        10, 31, 30, 0,1,1, 4,2 ,
            10, 11, 31, 1,1,0, 4,2 ,         11, 32, 31, 0,1,1, 4,2 ,        11, 12, 32, 1,1,0, 4,2 ,
            12, 33, 32, 0,1,1, 4,2 ,         12, 13, 33, 1,1,0, 4,2 ,        13, 34, 33, 0,1,1, 4,2 ,
            13, 14, 34, 1,1,0, 4,2 ,         14, 35, 34, 0,1,1, 4,2 ,        14, 15, 35, 1,1,0, 4,2 ,
            15, 36, 35, 0,1,1, 4,2 ,         15, 16, 36, 1,1,0, 4,2 ,        16, 37, 36, 0,1,1, 4,2 ,
            16, 17, 37, 1,1,0, 4,2 ,         17, 38, 37, 0,1,1, 4,2 ,        17, 18, 38, 1,1,0, 4,2 ,
            18, 39, 38, 0,1,1, 4,2 ,         18, 19, 39, 1,1,0, 4,2 ,        19, 40, 39, 0,1,1, 4,2 ,
            19, 20, 40, 1,1,0, 4,2 ,         20, 21, 40, 0,1,1, 4,2 ,        20,  1, 21, 1,1,0, 4,2 ,
            41, 21, 22, 0,1,0, 1,0 ,         41, 22, 23, 0,1,0, 1,0 ,        41, 23, 24, 0,1,0, 1,0 ,
            41, 24, 25, 0,1,0, 1,0 ,         41, 25, 26, 0,1,0, 1,0 ,        41, 26, 27, 0,1,0, 1,0 ,
            41, 27, 28, 0,1,0, 1,0 ,         41, 28, 29, 0,1,0, 1,0 ,        41, 29, 30, 0,1,0, 1,0 ,
            41, 30, 31, 0,1,0, 1,0 ,         41, 31, 32, 0,1,0, 1,0 ,        41, 32, 33, 0,1,0, 1,0 ,
            41, 33, 34, 0,1,0, 1,0 ,         41, 34, 35, 0,1,0, 1,0 ,        41, 35, 36, 0,1,0, 1,0 ,
            41, 36, 37, 0,1,0, 1,0 ,         41, 37, 38, 0,1,0, 1,0 ,        41, 38, 39, 0,1,0, 1,0 ,
            41, 39, 40, 0,1,0, 1,0 ,         41, 40, 21, 0,1,0, 1,0 ,        42, 44, 43, 0,1,0, 1,1 ,
            42, 45, 44, 0,1,0, 1,1 ,         42, 46, 45, 0,1,0, 1,1 ,        42, 47, 46, 0,1,0, 1,1 ,
            42, 48, 47, 0,1,0, 1,1 ,         42, 49, 48, 0,1,0, 1,1 ,        42, 50, 49, 0,1,0, 1,1 ,
            42, 43, 50, 0,1,0, 1,1 ,         43, 52, 51, 0,1,1, 4,2 ,        43, 44, 52, 1,1,0, 4,2 ,
            44, 53, 52, 0,1,1, 4,2 ,         44, 45, 53, 1,1,0, 4,2 ,        45, 54, 53, 0,1,1, 4,2 ,
            45, 46, 54, 1,1,0, 4,2 ,         46, 55, 54, 0,1,1, 4,2 ,        46, 47, 55, 1,1,0, 4,2 ,
            47, 56, 55, 0,1,1, 4,2 ,         47, 48, 56, 1,1,0, 4,2 ,        48, 57, 56, 0,1,1, 4,2 ,
            48, 49, 57, 1,1,0, 4,2 ,         49, 58, 57, 0,1,1, 4,2 ,        49, 50, 58, 1,1,0, 4,2 ,
            50, 51, 58, 0,1,1, 4,2 ,         50, 43, 51, 1,1,0, 4,2 ,        59, 51, 52, 0,1,0, 1,0 ,
            59, 52, 53, 0,1,0, 1,0 ,         59, 53, 54, 0,1,0, 1,0 ,        59, 54, 55, 0,1,0, 1,0 ,
            59, 55, 56, 0,1,0, 1,0 ,         59, 56, 57, 0,1,0, 1,0 ,        59, 57, 58, 0,1,0, 1,0 ,
            59, 58, 51, 0,1,0, 1,0 ,         60, 69, 68, 0,1,1, 4,2 ,        60, 61, 69, 1,1,0, 4,2 ,
            61, 70, 69, 0,1,1, 4,2 ,         61, 62, 70, 1,1,0, 4,2 ,        62, 71, 70, 0,1,1, 4,2 ,
            62, 63, 71, 1,1,0, 4,2 ,         63, 72, 71, 0,1,1, 4,2 ,        63, 64, 72, 1,1,0, 4,2 ,
            64, 73, 72, 0,1,1, 4,2 ,         64, 65, 73, 1,1,0, 4,2 ,        65, 74, 73, 0,1,1, 4,2 ,
            65, 66, 74, 1,1,0, 4,2 ,         66, 75, 74, 0,1,1, 4,2 ,        66, 67, 75, 1,1,0, 4,2 ,
            67, 68, 75, 0,1,1, 4,2 ,         67, 60, 68, 1,1,0, 4,2 };

        m_childSpaceMesh.setNumFaces(128);

        for (int f=0; f<128; f++)
        {
            m_childSpaceMesh.faces[f].setVerts(theFaces[f*8], theFaces[f*8+1], theFaces[f*8+2]);
            m_childSpaceMesh.faces[f].setEdgeVisFlags(theFaces[f*8+3], theFaces[f*8+4], theFaces[f*8+5]);
            m_childSpaceMesh.faces[f].setSmGroup(theFaces[f*8+6]);
            m_childSpaceMesh.faces[f].setMatID((MtlID)theFaces[f*8+7]);
        }

        m_childSpaceMesh.buildNormals();
        m_childSpaceMesh.BuildStripsAndEdges();
    }

    // HALF CYLINDER (volume top and bottom)
    {
        hctMaxUtils::createDiscMesh(Point3(1,0,0), Point3(0,1,0), vertexScale*50, vertexScale*20, 0, PI, 10, false, m_halfCylinder);
    }

    // MIDDLE VOLUME (box joining top and bottom halfs)
    {
        static float theVertices[] = {
            -10.0000f,-50.0000f,-50.0000f ,  10.0000f,-50.0000f,-50.0000f , -10.0000f, 50.0000f,-50.0000f ,
            10.0000f, 50.0000f,-50.0000f ,  -10.0000f,-50.0000f, 50.0000f ,  10.0000f,-50.0000f, 50.0000f ,
            -10.0000f, 50.0000f, 50.0000f ,  10.0000f, 50.0000f, 50.0000f };

        m_middleVolume.setNumVerts(8);

        for (int v=0; v<8; v++)
        {
            m_middleVolume.setVert(v, vertexScale * Point3(theVertices[v*3], theVertices[v*3+1], theVertices[v*3+2]));
        }

        static int theFaces[] = {
            0,  1,  5, 0,1,0, 4,4 ,       5,  4,  0, 0,1,0, 4,4 ,         1,  3,  7, 0,1,0, 5,3 ,
            7,  5,  1, 0,1,0, 5,3 ,       3,  2,  6, 0,1,0, 6,5 ,         6,  7,  3, 0,1,0, 6,5 ,
            2,  0,  4, 0,1,0, 7,2 ,       4,  6,  2, 0,1,0, 7,2 };

        m_middleVolume.setNumFaces(8);

        for (int f=0; f<8; f++)
        {
            m_middleVolume.faces[f].setVerts(theFaces[f*8], theFaces[f*8+1], theFaces[f*8+2]);
            m_middleVolume.faces[f].setEdgeVisFlags(theFaces[f*8+3], theFaces[f*8+4], theFaces[f*8+5]);
            m_middleVolume.faces[f].setSmGroup(0);
        }

        m_middleVolume.buildNormals();
        m_middleVolume.BuildStripsAndEdges();

    }

    m_staticsMeshesSetup = true;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
