// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>
#include <Common/GeometryUtilities/Mesh/Memory/hkMemoryMeshMaterial.h>
#include <Common/GeometryUtilities/Mesh/Utils/MeshSectionLockSet/hkMeshSectionLockSet.h>

#include <ContentTools/Common/SceneExport/Importer/MeshImport/hctModelerMeshBuilder.h>
#include <ContentTools/Common/SceneExport/Importer/MeshImport/hctMeshImportUtilities.h>
#include <ContentTools/Max/MaxSceneExport/Importer/MeshImport/hctMaxMeshBuilder.h>

#include <ContentTools/Max/MaxSceneExport/Importer/MeshImport/hctMaxMeshImport.h>


struct HavokMaxMaterial
{
    Mtl* m_maxMaterial;
    MSTR m_multiMaterialSlotName;
    hkStringPtr m_mtlName;  ///< Full material name
};


static Mtl* _findMaterialOnNodeByName(INode* node, const char* materialName, MSTR& slotName, hkStringPtr& fullMaterialNameOut)
{
    fullMaterialNameOut = HK_NULL;

    if ( !node )
    {
        return HK_NULL;
    }

    Mtl* keyMtl = node->GetMtl();
    if ( !keyMtl )
    {
        return HK_NULL;
    }

    // Compare against the root material
    {
        HCT_SCOPED_CONVERSIONS;
        const MSTR keyMtlName = keyMtl->GetName();
        hkStringBuf sbKeyName (FROM_MAX(keyMtlName));

        if ( materialName && !hkString::strCmp(sbKeyName.cString(), materialName) )
        {
            fullMaterialNameOut = materialName;
            return keyMtl;
        }
    }

    // Get the sub-materials
    const int numSubMtls = hctMaxUtils::getNumSubMaterials(keyMtl);
    for (int subMtlIdx = 0; subMtlIdx < numSubMtls; subMtlIdx++)
    {
        hkStringPtr strb;
        Mtl* subMtl = hctMaxUtils::findSubMaterialName(keyMtl, subMtlIdx, strb);

        if ( materialName && !hkString::strCmp(strb.cString(), materialName) )
        {
            fullMaterialNameOut = strb;
            slotName = subMtl->GetName();
            return subMtl;
        }
    }

    return HK_NULL;
}


static Mtl* _findMaterialInSceneByName(INode* parentNode, const Tab<INode*>& sceneNodes, const char* materialName, MSTR& slotName, hkStringPtr& fullMaterialNameOut)
{
    // Check parent node first (if available) as the material is most likely to be found there.
    if ( parentNode )
    {
        Mtl* material = _findMaterialOnNodeByName(parentNode, materialName, slotName, fullMaterialNameOut);
        if ( material ) { return material; }
    }

    // Now check all nodes in scene (except the already checked parent node).
    for (int nodeIndex = 0; nodeIndex < sceneNodes.Count(); nodeIndex++)
    {
        INode* node = sceneNodes[nodeIndex];
        if ( node != parentNode )
        {
            Mtl* material = _findMaterialOnNodeByName(node, materialName, slotName, fullMaterialNameOut);
            if ( material ) { return material; }
        }
    }

    return HK_NULL;
}


static MultiMtl* _findMatchingMultiMaterialOnNode(INode* node, const hkArray<HavokMaxMaterial>& sourceMaterials)
{
    if ( node == HK_NULL )
    {
        return HK_NULL;
    }

    Mtl* material = node->GetMtl();
    if ( (material == HK_NULL) || !material->IsMultiMtl() )
    {
        return HK_NULL;
    }

    MultiMtl* multiMaterial = reinterpret_cast<MultiMtl*>(material);
    int highestMaterialId = multiMaterial->NumSubMtls();

    int sourceMaterialIndex = 0;
    for (int subMaterialIndex = 0; subMaterialIndex < highestMaterialId; subMaterialIndex++)
    {
        Mtl* subMaterial = multiMaterial->GetSubMtl(subMaterialIndex);
        if ( subMaterial )
        {
            // If we have more sub-materials in the multi-material than in the source material then this is no match!
            if ( sourceMaterialIndex >= sourceMaterials.getSize() )
            {
                return HK_NULL;
            }

            // If the sub-material is not matching the current source material then we have no match!
            if ( subMaterial != sourceMaterials[sourceMaterialIndex].m_maxMaterial )
            {
                return HK_NULL;
            }

            // Advance to next material in source.
            sourceMaterialIndex++;
        }
    }

    // If there are still some materials left in the source, but none in the actual material, then this is no match!
    if ( sourceMaterialIndex < sourceMaterials.getSize() )
    {
        return HK_NULL;
    }

    return multiMaterial;
}


static Mtl* _findMatchingMultiMaterialInScene(INode* parentNode, const Tab<INode*>& sceneNodes, const hkArray<HavokMaxMaterial>& sourceMaterials)
{
    if ( parentNode )
    {
        MultiMtl* multiMaterial = _findMatchingMultiMaterialOnNode(parentNode, sourceMaterials);
        if ( multiMaterial ) { return multiMaterial; }
    }

    for (int nodeIndex = 0; nodeIndex < sceneNodes.Count(); nodeIndex++)
    {
        INode* node = sceneNodes[nodeIndex];
        if ( node != parentNode )
        {
            MultiMtl* multiMaterial = _findMatchingMultiMaterialOnNode(node, sourceMaterials);
            if ( multiMaterial ) { return multiMaterial; }
        }
    }

    return HK_NULL;
}


INode* hctMaxSceneConvertUtilities::createNodeFromMesh(const hkMeshBody&  meshBody,
                                                       INode*             parentNode,
                                                       Tab<INode*>&       allSceneNodes,
                                                       const char*        meshName)
{
    hkMeshSectionLockSet lockSet;
    int numSections = hctModelerMeshBuilderUtilities::prepareMeshSectionLockSet(meshBody, &lockSet);

    hkMatrix4 transform;
    meshBody.getTransform(transform);

    if ( (numSections == 0) || (!transform.isOk()) )
    {
        return HK_NULL;
    }

    INode* maxNode = HK_NULL;

    //
    // Create the 3DS MAX object.
    //
    theHold.Begin();
    {
        Interface* ip = GetCOREInterface();

        //create an object to hold the mesh
        PolyObject* polyObj = CreateEditablePolyObject();

        //create a node to hold the object, and give it a name
        maxNode = ip->CreateObjectNode( polyObj );

        MNMesh& maxMesh = polyObj->GetMesh();

        maxMesh.Init();

        //
        // Get (and, if necessary, create) the material for the new mesh.
        //
        Mtl* nodeMaterial = HK_NULL;
        bool mtlSet = false;
        {
            // Collect (and if necessary create) all materials from mesh.
            bool newMaterialCreated = false;
            hkArray<HavokMaxMaterial> materials;
            {
                for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
                {
                    HCT_SCOPED_CONVERSIONS;

                    const hkMeshSection& meshSection = lockSet.getSection(sectionIndex);
                    if ( meshSection.m_material )
                    {
                        HavokMaxMaterial& materialInfo = materials.expandOne();
                        materialInfo.m_maxMaterial   = _findMaterialInSceneByName(parentNode, allSceneNodes, meshSection.m_material->getName(), materialInfo.m_multiMaterialSlotName, materialInfo.m_mtlName);
                        if ( materialInfo.m_maxMaterial == HK_NULL )
                        {
                            materialInfo.m_maxMaterial = NewDefaultStdMat();

                            MSTR newMaterialName = parentNode ? MSTR(parentNode->GetName()) : TO_MAX(meshBody.getName());
                            newMaterialName.printf(TEXT("%s_%.2d"), newMaterialName.data(), sectionIndex);
                            materialInfo.m_maxMaterial->SetName(newMaterialName);
                            materialInfo.m_mtlName = FROM_MAX(newMaterialName);
                            meshSection.m_material->setName(materialInfo.m_mtlName.cString());
                            newMaterialCreated = true;
                        }
                    }
                }
            }

            int numMaterials = materials.getSize();
            if ( numMaterials == 1 )
            {
                // Only one single material: use it for the mesh.
                nodeMaterial = materials[0].m_maxMaterial;
            }
            else
            {
                // Try to find the exact matching (and already existing) multi-material in the scene to re-use it.
                if ( !newMaterialCreated )
                {
                    nodeMaterial = _findMatchingMultiMaterialInScene(parentNode, allSceneNodes, materials);
                }

                // If we couldn't re-use an existing multi-material we need to create a new one.
                if ( nodeMaterial == HK_NULL )
                {
                    MultiMtl* multiMaterial = NewDefaultMultiMtl();
                    multiMaterial->SetNumSubMtls(numMaterials);
                    for (int materialIndex = 0; materialIndex < numMaterials; materialIndex++)
                    {
                        Mtl* maxMaterial = materials[materialIndex].m_maxMaterial;
                        multiMaterial->SetSubMtlAndName(materialIndex, maxMaterial, materials[materialIndex].m_multiMaterialSlotName);
                    }

                    nodeMaterial = multiMaterial;
                }

                // This will assign a default name to the material
                maxNode->SetMtl(nodeMaterial);
                mtlSet = true;

                // We need to update the names of our mesh materials to match with the new multi-material
                HK_ASSERT_NO_MSG(0xabba1001, numSections >= numMaterials);
                for (int si = 0, mi = 0; si < numSections; si++)
                {
                    hkMeshSection& meshSection  = const_cast<hkMeshSection&>(lockSet.getSection(si));
                    hkMeshMaterial* origMtl     = meshSection.m_material;

                    if ( origMtl )
                    {
                        HavokMaxMaterial& mtlInfo   = materials[mi];
                        const char* mtlName         = origMtl->getName();

                        if ( !mtlName || hkString::strCmp(mtlName, mtlInfo.m_mtlName.cString()) )
                        {
                            HK_ASSERT_NO_MSG(0xabba1002, false);
                        }

                        // Get new name
                        hkStringPtr strb;
                        hctMaxUtils::findSubMaterialName(nodeMaterial, mi, strb);

                        // Create new material with the new name
                        if ( strb.cString() && hkString::strCmp(mtlName, strb.cString()) )
                        {
                            hkMemoryMeshMaterial* newMtl = new hkMemoryMeshMaterial();
                            *newMtl = *static_cast<hkMemoryMeshMaterial*>(origMtl);
                            newMtl->setName(strb.cString());
                            meshSection.m_material = newMtl;
                        }
                        mi++;
                    }
                }
            }
        }

        if ( !mtlSet )
        {
            mtlSet= true;
            maxNode->SetMtl(nodeMaterial);
        }

        // Commit mesh changes
        lockSet.clear();
        hctModelerMeshBuilderUtilities::prepareMeshSectionLockSet(meshBody, &lockSet);

        //
        // Extract the indices for the 3ds Max UV channels that are actually in use as well as the highest index.
        //
        hkArray<int> uvChannelsInUse;
        int highestUvChannelIndex = 0;
        {
            hctMaxUtils::findUsedChannels(nodeMaterial, uvChannelsInUse);
            for (int i = 0; i < uvChannelsInUse.getSize(); i++)
            {
                if ( uvChannelsInUse[i] > highestUvChannelIndex )
                {
                    highestUvChannelIndex = uvChannelsInUse[i];
                }
            }
        }

        // If original object has no material assigned or material uses no texture map channel then we need to make
        // sure that at least the default texture map channel (map channel 1) is initialized as the original
        // hkMeshShape always seems to have at least one USAGE_TEX_COORD set in the vertex buffer.
        if ( uvChannelsInUse.getSize() == 0 )
        {
            uvChannelsInUse.pushBack(1);
            highestUvChannelIndex = 1;
        }

        //
        // Initialize Channels:
        // 0  : Vertex Color channel.
        // 1  : Default UV channel.
        // 2+ : additional channels
        //
        // This has to be done before the actual mesh is created. Otherwise 3ds Max will run into problems.
        //
        {
            maxMesh.SetMapNum(1+highestUvChannelIndex); // 1+ = Always initialize the Vertex Color channel.

            // Init the Vertex Color channel.
            maxMesh.InitMap(0);

            // Init the UV map channels.
            {
                for (int uvChannelIndex = 0; uvChannelIndex < uvChannelsInUse.getSize(); uvChannelIndex++)
                {
                    maxMesh.InitMap(uvChannelsInUse[uvChannelIndex]);
                }
            }
        }

        // Setup the virtual modeler interface.
        hctMaxMeshBuilder maxMeshBuilder(maxNode, maxMesh, meshName, transform, uvChannelsInUse, parentNode);

        // Register the material that has been retrieved or created above.
        maxMeshBuilder.registerMaterial(nodeMaterial);

        // Do the actual mesh conversion.
        hkResult res = hctModelerMeshBuilderUtilities::convertHavokMeshToModelerNode(meshBody, lockSet, maxMeshBuilder);
        if ( res.isFailure() )
        {
            theHold.Cancel();
            return HK_NULL;
        }

        // Enable the next line to assist in debugging any problems that cause 3dsMax to crash right after creating a
        // new node with this importer.
        //maxMesh.CheckAllData();

        // Add the new node to the list of scene nodes so that subsequent nodes might find the material on the newly created node.
        allSceneNodes.Append(1, &maxNode);
    }
    theHold.Accept( _T("Create Mesh") );

    return maxNode;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
