// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#include <ContentTools/Max/MaxSceneExport/Importer/MeshImport/hctMaxMeshBuilder.h>

// Additional includes
#include <Common/Base/Math/Matrix/hkMatrixDecomposition.h>


hctMaxMeshBuilder::hctMaxMeshBuilder(INode* maxNode, MNMesh& maxMesh, const char* meshName, hkMatrix4& transform, hkArray<int>& mapChannelsInUse, INode* parentMaxNode)
:   m_maxMesh(maxMesh)
,   m_mapChannelsInUse(mapChannelsInUse)
,   m_colorChannel(0)
,   m_uvChannel(1)
{
    // Attach new node to parent and update node's WORLD SPACE matrix.
    if ( parentMaxNode )
    {
        parentMaxNode->AttachChild(maxNode, 0);
    }

    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();

    Matrix3 nodeToWorldTransform3dsMax;
    hctMaxUtils::convertToMatrix3(transform, nodeToWorldTransform3dsMax);

    if ( parentMaxNode )
    {
        Matrix3 parentToWorldTransform3dsMax = parentMaxNode->GetNodeTM(now);
        // nodeToWorldTransform3dsMax is actually nodeToParentTransform3dsMax here!
        nodeToWorldTransform3dsMax = nodeToWorldTransform3dsMax * parentToWorldTransform3dsMax;

        // To compensate for scaling on the original node (or its hierarchy) we will "undo" the scaling on the newly
        // created children. By scaling with the inverse scaling factor of the parent node this essentially brings the
        // new node's scale back to 1/1/1.
        // Note that this only works as long as the grandparent nodes (or above) are uniformly scaled! The original node
        // can still be non-uniformly scaled, though.
        {
            hkMatrix4 worldFromParentMatrix;
            hctMaxUtils::convertToMatrix4(parentToWorldTransform3dsMax, worldFromParentMatrix);

            // Separate scale/skew from rotation/translation.
            hkMatrixDecomposition::Decomposition decomposition;
            hkMatrixDecomposition::decomposeMatrix(worldFromParentMatrix, decomposition);

            hkVector4 scaleFactors = decomposition.m_scale;
            hkReal scaleEps = 0.0001f; // HK_REAL_EPSILON won't work here
            if ((hkMath::fabs(scaleFactors(0)-scaleFactors(1)) > scaleEps) ||
                (hkMath::fabs(scaleFactors(0)-scaleFactors(2)) > scaleEps) ||
                (hkMath::fabs(scaleFactors(1)-scaleFactors(2)) > scaleEps)   )
            {
                HK_WARN_ALWAYS(0xaf41e153, "'" << parentMaxNode->GetName() << "' : The non-uniform scale (" << scaleFactors(0) << ", " << scaleFactors(1) << ", " << scaleFactors(2) << ") on this node can cause graphical distortions of the Fracture Pieces. If that's the case, you might consider baking the scaling into the mesh itself (by using 'Reset Scale' on the node's Hierarchy tab).");
            }

            // Build the inverse scaling matrix.
            Matrix3 inverseScalingMatrix;
            {
                Point3 scalingFactors(1.0f/scaleFactors(0), 1.0f/scaleFactors(1), 1.0f/scaleFactors(2));
                inverseScalingMatrix.SetScale(scalingFactors);
            }

            // Perform the actual (de)scaling.
            nodeToWorldTransform3dsMax = inverseScalingMatrix * nodeToWorldTransform3dsMax;
        }
    }

    // Set the node's WORLD SPACE matrix
    maxNode->SetNodeTM(now, nodeToWorldTransform3dsMax);

    HCT_SCOPED_CONVERSIONS;

    // Set the node's name. Either take the supplied name or use the parent node's name extended by a postfix.
    MSTR maxNodeName = meshName ? TO_MAX(meshName) : ( parentMaxNode ? MSTR(parentMaxNode->GetName()) + TEXT("_child") : TEXT("") );
    ip->MakeNameUnique(maxNodeName);
    maxNode->SetName(maxNodeName);

    m_vertices.clear();
    m_triangles.clear();
}


// ===============================================================================================
// MESH
// ===============================================================================================

hctModelerMeshBuilder::VertexId hctMaxMeshBuilder::addMeshVertexDelayed(const hkVector4& position)
{
    int vertexId = m_vertices.getSize();

    m_vertices.pushBack(position);

    return vertexId;
}


int hctMaxMeshBuilder::addMeshTriangleDelayed(VertexId positionIndex0, VertexId positionIndex1, VertexId positionIndex2)
{
    TriangleData triangleData;
    {
        triangleData.m_positionIndex0 = positionIndex0;
        triangleData.m_positionIndex1 = positionIndex1;
        triangleData.m_positionIndex2 = positionIndex2;
    }

    int triangleId = m_vertices.getSize();

    m_triangles.pushBack(triangleData);

    return triangleId;
}


void hctMaxMeshBuilder::realizeDelayedMesh()
{
    if ( m_maxMesh.VNum() > 0 )
    {
        HK_WARN_ALWAYS(0xaf41e123, "Havok Max Mesh Builder - Mesh has already been created.");
        return;
    }

    // Create vertices.
    {
        for (int v = 0; v < m_vertices.getSize(); v++)
        {
            HK_ON_DEBUG(const int index = )m_maxMesh.NewVert(Point3(m_vertices[v](0), m_vertices[v](1), m_vertices[v](2)));
            HK_ASSERT_NO_MSG(0xaf324324, index == v);
        }
    }

    // Create triangles.
    {
        for (int t = 0; t < m_triangles.getSize(); t++)
        {
            // Note: Smoothing Group and Material ID will be set later.
            m_maxMesh.NewTri(m_triangles[t].m_positionIndex0, m_triangles[t].m_positionIndex1, m_triangles[t].m_positionIndex2, 0, 0);
        }
    }
}


// ===============================================================================================
// VERTEX COLORS
// ===============================================================================================

void hctMaxMeshBuilder::selectColorChannel(int channel)
{
    if ( channel > 0 )
    {
        HK_WARN_ALWAYS(0xafe13289, "Havok Max Mesh Builder - Currently only one Color Channel is supported for 3ds Max.");
        return;
    }

    m_colorChannel = channel;
}


hctModelerMeshBuilder::ColorId hctMaxMeshBuilder::addVertexColorImmediate(float r, float g, float b)
{
    MNMap* colorChannel = m_maxMesh.M(m_colorChannel);
    int colorId = colorChannel->NewVert( VertColor(r, g, b) );
    return colorId;
}


hctModelerMeshBuilder::ColorId hctMaxMeshBuilder::addVertexColorDelayed(float r, float g, float b)
{
    // 3ds Max does not seem to provide a 'buffered' interface function for adding Vertex Colors, so
    // we simply 'immediately' set the values then.
    int colorId = addVertexColorImmediate(r, g, b);
    return colorId;
}


void hctMaxMeshBuilder::realizeDelayedVertexColors()
{
    // Nothing to be done here for 3ds Max.
}


int hctMaxMeshBuilder::addVertexColorTriangleImmediate(int triangleIndex, ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2)
{
    MNMap* colorChannel = m_maxMesh.M(m_colorChannel);
    int triangleId = colorChannel->NewTri(colorIndexVertex0, colorIndexVertex1, colorIndexVertex2);
    return triangleId;
}


int hctMaxMeshBuilder::addVertexColorTriangleDelayed(ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2)
{
    // 3ds Max does not seem to provide a 'buffered' interface function for adding triangle information to the
    // Vertex Color channel, so we simply 'immediately' set the values then.
    // 'triangleIndex' is not needed for the 'immediate' function on 3ds Max, so we can pass in any value here (-1 in this case).
    int triangleId = addVertexColorTriangleImmediate(-1, colorIndexVertex0, colorIndexVertex1, colorIndexVertex2);
    return triangleId;
}


void hctMaxMeshBuilder::realizeDelayedVertexColorTriangles()
{
    // Nothing to be done here for 3ds Max.
}


// ===============================================================================================
// TEXTURES
// ===============================================================================================

int hctMaxMeshBuilder::mapVirtualToActualUvChannel(int virtualChannelId)
{
    if ( virtualChannelId >= m_mapChannelsInUse.getSize() )
    {
        return 0;
    }

    return m_mapChannelsInUse[virtualChannelId];
}


void hctMaxMeshBuilder::selectUvChannel(int virtualChannel)
{
    m_uvChannel = mapVirtualToActualUvChannel(virtualChannel);
}


hctModelerMeshBuilder::UvId hctMaxMeshBuilder::addVertexUvImmediate(float u, float v)
{
    MNMap* uvChannel = m_maxMesh.M(m_uvChannel);
    int uvId = uvChannel->NewVert( UVVert(u, v, 0.0f) );
    return uvId;
}


hctModelerMeshBuilder::UvId hctMaxMeshBuilder::addVertexUvDelayed(float u, float v)
{
    // 3ds Max does not seem to provide a 'buffered' interface function for adding vertex UVs, so
    // we simply 'immediately' set the values then.
    int uvId = addVertexUvImmediate(u, v);
    return uvId;
}


void hctMaxMeshBuilder::realizeDelayedVertexUvs()
{
    // Nothing to be done here for 3ds Max.
}


int hctMaxMeshBuilder::addVertexUvTriangleImmediate(int triangleIndex, UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2)
{
    MNMap* uvChannel = m_maxMesh.M(m_uvChannel);
    int triangleId = uvChannel->NewTri(uvIndexVertex0, uvIndexVertex1, uvIndexVertex2);
    return triangleId;
}


int hctMaxMeshBuilder::addVertexUvTriangleDelayed(UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2)
{
    // 3ds Max does not seem to provide a 'buffered' interface function for adding triangle information to the
    // UV channel, so we simply 'immediately' set the values then.
    // 'triangleIndex' is not needed for the 'immediate' function on 3ds Max, so we can pass in any value here (-1 in this case).
    int triangleId = addVertexUvTriangleImmediate(-1, uvIndexVertex0, uvIndexVertex1, uvIndexVertex2);
    return triangleId;
}


void hctMaxMeshBuilder::realizeDelayedVertexUvTriangles()
{
    // Nothing to be done here for 3ds Max.
}


// ===============================================================================================
// NORMALS
// ===============================================================================================

hctModelerMeshBuilder::NormalId hctMaxMeshBuilder::addNormalDelayed(const hkVector4& normal)
{
    int normalId = m_vertices.getSize();

    m_normals.pushBack(normal);

    return normalId;
}


int hctMaxMeshBuilder::addNormalTriangleDelayed(NormalId triangleVertex0, NormalId triangleVertex1, NormalId triangleVertex2)
{
    int triangleId = m_normalTriangleIndices.getSize() / 3;

    m_normalTriangleIndices.pushBack(triangleVertex0);
    m_normalTriangleIndices.pushBack(triangleVertex1);
    m_normalTriangleIndices.pushBack(triangleVertex2);

    return triangleId;
}


void hctMaxMeshBuilder::realizeDelayedNormals()
{
    if (m_normals.getSize() == 0)
    {
        // If we don't have any normals at all we need to abort as 3dsMax does not like it if we
        // initialize the normals but don't specify any then.
        return;
    }

    m_maxMesh.buildNormals(); // force allocation
    m_maxMesh.SpecifyNormals();

    MNNormalSpec* meshNormals = m_maxMesh.GetSpecifiedNormals();
    meshNormals->SetParent(&m_maxMesh);
    meshNormals->CheckNormals();

    meshNormals->SetNumNormals(m_normals.getSize());
    meshNormals->MakeNormalsExplicit(false, NULL, true);

    // Initialize the actual normals.
    for (int i = 0; i < m_normals.getSize(); i++)
    {
        const hkVector4& normal = m_normals[i];
        meshNormals->Normal(i) = Point3(normal(0), normal(1), normal(2));
    }

    // Initialize the normal triangles. These have to match the mesh triangles 1:1!
    const int numTriangles = m_normalTriangleIndices.getSize() / 3;
    for (int i = 0; i < numTriangles; i++)
    {
        meshNormals->SetNormalIndex(i, 0, m_normalTriangleIndices[i*3+0]);
        meshNormals->SetNormalIndex(i, 1, m_normalTriangleIndices[i*3+1]);
        meshNormals->SetNormalIndex(i, 2, m_normalTriangleIndices[i*3+2]);
    }
}


// ===============================================================================================
// MATERIALS
// ===============================================================================================

void hctMaxMeshBuilder::registerMaterial(Mtl* modelerMaterial)
{
    m_modelerMaterial = modelerMaterial;
}


void hctMaxMeshBuilder::setTriangleMaterialImmediate(int triangleIndex, const char* materialName)
{
    int matId = -1;
    const int numSubMtls = hctMaxUtils::getNumSubMaterials(m_modelerMaterial);

    if ( numSubMtls && materialName )
    {
        for (int subMtlIdx = 0; subMtlIdx < numSubMtls; subMtlIdx++)
        {
            hkStringPtr strb;
            hctMaxUtils::findSubMaterialName(m_modelerMaterial, subMtlIdx, strb);

            if ( materialName && !hkString::strCmp(strb.cString(), materialName) )
            {
                matId = subMtlIdx;
                break;
            }
        }

        HK_ASSERT_NO_MSG(0xaf4ee134, matId != -1);
    }
    else
    {
        matId = 0;
    }

    m_maxMesh.F(triangleIndex)->material = matId;
    m_maxMesh.F(triangleIndex)->smGroup  = matId+1;
}


void hctMaxMeshBuilder::setTriangleMaterialDelayed(int triangleIndex, const char* materialName)
{
    // There's no need to buffer this for 3ds Max as we are only setting integer values in the mesh faces.
    setTriangleMaterialImmediate(triangleIndex, materialName);
}


void hctMaxMeshBuilder::realizeDelayedTriangleMaterials()
{
    // Nothing to be done here.
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
