// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/GUP/Viewport/hctMaxViewWindow.h>
#include <ContentTools/Max/MaxSceneExport/resource.h>

#include <ContentTools/Common/ViewportUtils/hctviewportutils.h>
#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>
#include <ContentTools/Common/SceneExport/Filters/hctFilterProcessingUtil.h>

#include <ContentTools/Max/MaxSceneExport/hctMaxIncludes.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxFpInterfaces/Export/hctSceneExportUtilityInterface.h>
//#include <maxscrpt/maxscrpt.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>

extern HINSTANCE hInstance;

hctMaxViewWindow::hctMaxViewWindow()
: m_active(false), m_hWnd(NULL), m_hIntermediateParent(NULL), m_viewportDll(NULL), m_viewportBase(NULL), m_viewportUtils(NULL)
{

}

hctMaxViewWindow::~hctMaxViewWindow()
{
    if (m_hWnd)
    {
        DestroyViewWindow(m_hWnd);
    }

    if (m_viewportDll)
    {
        unloadViewportDLL();
    }
}

HWND hctMaxViewWindow::CreateViewWindow(HWND hParent, int x, int y, int w, int h)
{
    if (!m_viewportUtils)
    {
        loadViewportDLL();
        if (!m_viewportUtils)
        {
            return NULL;
        }
    }

    // Make a intermediate window that just has a context sensitive menu to allow the Max->Viewport interaction
    // Assumes that the Viewport itself is allowing the Right click event to be handled in this parent
    {
        const TCHAR* className = TEXT("INTERMEDIATEHAVOKMAXPREVIEW");
        WNDCLASS wc;
        wc.style = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc = (WNDPROC) hctMaxViewWindow::WndProc;
        wc.cbClsExtra = 0;
        wc.cbWndExtra = 0;
        wc.hInstance = hInstance;
        wc.hIcon = LoadIcon(NULL, IDI_WINLOGO );
        wc.hCursor = LoadCursor(NULL, IDC_ARROW );
        wc.hbrBackground = NULL;
        wc.lpszMenuName = NULL;
        wc.lpszClassName = className;

        if(! RegisterClass(&wc) && (ERROR_CLASS_ALREADY_EXISTS != GetLastError()))
        {
            // failed.. ignore
        }

        // Set the window's initial style
        DWORD windowStyle = WS_CHILD;

        // Create the render window
        m_hIntermediateParent = CreateWindow( className, TEXT("HavokMaxPreviewIntermediate"), windowStyle,
            x, y, w, h, hParent, HK_NULL, hInstance, (LPVOID) this );

        setupContextMenu(m_hIntermediateParent);
    }

    RECT cw;
    GetWindowRect( m_hIntermediateParent, &cw );
    m_hWnd = (HWND)m_viewportUtils->createPreviewWindow( (void*) m_hIntermediateParent, 0, 0, cw.right - cw.left , cw.bottom - cw.top );

    UpdateWindow( m_hIntermediateParent );
    ShowWindow( m_hIntermediateParent, SW_NORMAL);

    m_active = true;

    // Start with current max scene?
    UpdateScene();

    return m_hIntermediateParent;
}

#define IDC_UPDATE_SCENE        0x9990
#define IDC_RESTART_VIEWPORT    0x9991

void hctMaxViewWindow::setupContextMenu(HWND hWnd )
{
    m_contextMenu = CreatePopupMenu();
    InsertMenu( m_contextMenu, -1, MF_BYPOSITION | MF_STRING, IDC_UPDATE_SCENE, TEXT("&Update from Max"));
    InsertMenu( m_contextMenu, -1, MF_BYPOSITION | MF_STRING, IDC_RESTART_VIEWPORT, TEXT("&Restart"));
}

void hctMaxViewWindow::showContextMenu(HWND hWnd, int x, int y)
{
    if (m_contextMenu)
    {
        TrackPopupMenu( m_contextMenu, TPM_RIGHTBUTTON, x, y, 0, hWnd, NULL);
    }
}

struct NMHDR_WITH_POS
{
    HWND      hwndFrom;
    UINT_PTR  idFrom;
    UINT      code;
    INT       x;
    INT       y;
};

LRESULT CALLBACK hctMaxViewWindow::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    hctMaxViewWindow* wnd = reinterpret_cast<hctMaxViewWindow*>( (hkUlong) GetWindowLongPtr(hWnd, GWLP_USERDATA) );

    switch(message)
    {
        case WM_CREATE:
        {
            wnd = (hctMaxViewWindow*) ((CREATESTRUCT*)lParam)->lpCreateParams;
            SetWindowLongPtr(hWnd, GWLP_USERDATA, *(LONG_PTR*)&wnd); // so that it can be retrieved later
            break;
        }

        case WM_NOTIFY:
            switch (((LPNMHDR)lParam)->code)
            {
                case WM_RBUTTONUP:
                {
                    NMHDR_WITH_POS* lp = (NMHDR_WITH_POS*)lParam;
                    RECT cr;
                    GetWindowRect(hWnd, &cr);
                    int x = lp->x + cr.left;
                    int y = lp->y + cr.top;
                    if ((lp->x > 20) && (lp->y > 20))
                    {
                        wnd->showContextMenu(hWnd, x, y);
                    }
                    else
                    {
                        Interface* ip = GetCOREInterface();
                        POINT pt; pt.x = lp->x; pt.y = lp->y;
                        ip->PutUpViewMenu(hWnd,  pt);
                    }
                    return 0;
                }
            }
            break;

        case WM_COMMAND:
            {
                switch ( LOWORD(wParam) )
                {
                    case IDC_UPDATE_SCENE:
                        wnd->UpdateScene(); break;

                    case IDC_RESTART_VIEWPORT:
                        wnd->Restart(); break;
                }
                break;
            }

        case WM_SIZE:
            {
                WORD w=LOWORD(lParam);
                WORD h=HIWORD(lParam);
                // always pass on full resize
                if (wnd)
                {
                    wnd->Resize(w,h);
                }
            }
            break;
        case WM_ERASEBKGND:
            return 1;

        default: break;
    }

    return DefWindowProc(hWnd, message, wParam, lParam);
}


void hctMaxViewWindow::DestroyViewWindow(HWND hWnd)
{
    if (m_viewportUtils)
    {
        // hWnd == intermediate

        m_viewportUtils->destroyPreviewWindow( m_hWnd );
    }

    ShowWindow(m_hIntermediateParent, SW_HIDE);
    CloseWindow(m_hIntermediateParent);

    m_active = false;
}

void hctMaxViewWindow::loadViewportDLL()
{
    hkStringOld filterManagerPath;
    hctMaxUtils::getFilterManagerPath( filterManagerPath );
    hkStringOld viewportUtilDll = filterManagerPath + "\\utils\\hctViewportUtils.dll";

    // call the utility
    MaxProgressUpdater progressUpdater(_T("Loading Viewport Util DLL.."));

    hkIfstream testStream( viewportUtilDll.cString() );
    if (!testStream.isOk())
    {
        HK_ERROR(0x29e3fb7d,"Viewport Util DLL not found in filter manager's util dir");
        return;
    }

    m_viewportDll = LoadLibraryA( viewportUtilDll.cString() );
    if (!m_viewportDll)
    {
        HK_ERROR(0x71cbca52,"Viewport Util DLL found but could not load. The DLL is corrupt.");
        return;
    }

    hctGetBaseDllInterfaceFunc viewportBaseDllFunc = ( hctGetBaseDllInterfaceFunc ) GetProcAddress(m_viewportDll, "getBaseDllInterface");
    if (!viewportBaseDllFunc)
    {
        HK_ERROR(0x44f121b0,"Viewport Util DLL found but could not find baseDLL sync function. Install is invalid.");
        unloadViewportDLL();
        return;
    }

    progressUpdater.progression(0, _T("Waiting for Havok Content Tools init to finish caching filters..") );

    MSTR waitCmd = _T("hkSceneExportUtility.waitForFilterLoad()");
    if ( !ExecuteMAXScriptScript( waitCmd ) )
    {
        HK_ERROR(0x37ec740f,"Could not wait on background filter load, Viewport Util DLL statics may get corrupted.");
    }

    if (progressUpdater.didUserCancel())
    {
        HK_ERROR(0x512398ba,"Viewport Util did not complete, user canceled op.");
        unloadViewportDLL();
        return;
    }

    progressUpdater.progression(1, _T("Initializing Viewport Util DLL..") );

    m_viewportBase = viewportBaseDllFunc();
    if (m_viewportBase)
    {
        if ( m_viewportBase->getDllVersion() != HCT_CURRENT_VERSION )
        {
            HK_ERROR(0x76f4fa19, "Could not load Viewport Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall." );
            unloadViewportDLL();
            return;
        }

        hkMemoryInitUtil::SyncInfo baseSystemInfo;
        hkSceneExportMemory::getBaseSystemSyncInfo( baseSystemInfo );
        m_viewportBase->initDll( baseSystemInfo, HK_NULL );

        hctGetViewportUtilInterfaceFunc getUtilInterface = ( hctGetViewportUtilInterfaceFunc ) GetProcAddress(m_viewportDll, "getViewportUtilInterface");
        m_viewportUtils = getUtilInterface? getUtilInterface() : HK_NULL;
    }

    if (!m_viewportUtils || !m_viewportBase )
    {
        hctGetViewportDllErrorFunc destDllErrorFunc = ( hctGetViewportDllErrorFunc ) GetProcAddress(m_viewportDll, "getViewportDllError");
        if (!destDllErrorFunc)
        {
            HK_ERROR(0x102ea4d, "Could not query Viewport Util DLL. Maybe your install is corrupt." );
            unloadViewportDLL();
            return;
        }

        hctViewportUtils::DllError destError = destDllErrorFunc();
        if (destError & hctViewportUtils::DLL_INIT_ERROR)
        {
            HK_ERROR(0x3993a8ee, "The Viewport Util DLL failed to initialize properly. Please try restarting the Modeller. If this error repeats, please contact Havok support at http://support.havok.com" );
        }
        else
        {
            HK_ERROR(0x74479ff2, "The Viewport Util DLL failed to load. Maybe your install is corrupt. If this error repeats, please contact Havok support at http://support.havok.com" );
        }
        unloadViewportDLL();
        return;
    }

    // All good:

    m_viewportUtils->setFilterManagerPath(filterManagerPath.cString());

    if (progressUpdater.didUserCancel())
    {
        HK_ERROR(0x1ac62e9f, "Viewport Util init did complete, but user canceled op. Try again without cancel to get viewport." );
        unloadViewportDLL();
        return;
    }

    m_viewportUtils->registerThreadCallback(hctFilterProcessingUtil::getDefaultThreadCallback());

}

void hctMaxViewWindow::unloadViewportDLL()
{
    if (m_viewportUtils)
    {
        delete m_viewportUtils;
        m_viewportUtils = HK_NULL;
    }

    if (m_viewportBase)
    {
        m_viewportBase->quitDll();
        m_viewportBase = HK_NULL;
    }

    if ( m_viewportDll )
    {
        FreeLibrary( m_viewportDll );
        m_viewportDll = NULL;
    }
}


bool hctMaxViewWindow::IsActive()
{
   return m_active;
}

void hctMaxViewWindow::ResetNode()
{
   InvalidateRect(m_hWnd,NULL,FALSE);
}

void hctMaxViewWindow::Redraw()
{
   InvalidateRect(m_hWnd,NULL,FALSE);
}

void hctMaxViewWindow::UpdateScene()
{
    Interface* ip = GetCOREInterface();

    SetCursor(LoadCursor(NULL, IDC_WAIT));
    ip->PushPrompt( _T("Updating Havok scene..") );

    hctMaxSceneExportOptions exportOptions;
    exportOptions.m_batchMode = true;
    exportOptions.m_doNotSplitVertices = false;
    exportOptions.m_visibleOnly = true;

    hctFilterSetOptions filterSetOptions;
    filterSetOptions.m_data.SetCount(0);
    filterSetOptions.m_version = 0;

    hctSceneExportUtilityFPInterface* sceneExportInterface = (hctSceneExportUtilityFPInterface*) ::GetInterface( UTILITY_CLASS_ID, HK_SCENE_EXPORT_UTILITY_CLASS_ID, HK_SCENE_EXPORT_UTILITY_FPINTERFACE_ID );
    if (sceneExportInterface)
    {
        void* optionsPtr = HK_NULL;
        int optionsSize = 0;
        int optionsVer = 0;

        // Filter Info
        if (sceneExportInterface->iGetFilterSetOptions( &optionsPtr, &optionsSize, &optionsVer ) )
        {
            filterSetOptions.m_data.SetCount(optionsSize );
            if (optionsSize > 0)
            {
                hkString::memCpy( filterSetOptions.m_data.Addr(0), optionsPtr, optionsSize );
            }
            filterSetOptions.m_version = optionsVer;
        }

        // Export options (frames ranges etc)
        sceneExportInterface->iGetExportOptions( &exportOptions );
    }

    {
        MaxProgressUpdater progressUpdater(_T("Updating Havok Preview scene.."));

        {
            hctMaxSceneExporter exporter( exportOptions, filterSetOptions ); // sets base sys error handler to Modellers

            bool haveScene = exporter.createScene();
            if (progressUpdater.didUserCancel())
            {
                HK_WARN_ALWAYS(0xabba1767, "Updating Havok Preview did not complete, user canceled op." );
                exporter.cleanupScene();
                goto CLEAN_EXIT;
            }

            hkRootLevelContainer* sceneData = exporter.getCurrentRootContainer();
            if (!haveScene || (sceneData == HK_NULL))
            {
                HK_WARN_ALWAYS(0xabba4534, "No scene data exported, so nothing to update yet");
                exporter.cleanupScene();
                goto CLEAN_EXIT;
            }

            if (filterSetOptions.m_data.Count() > 0)
            {
                m_viewportUtils->filterAndUpdateScene( sceneData, filterSetOptions.m_data.Addr(0), filterSetOptions.m_data.Count() );
            }
            else
            {
                m_viewportUtils->updateScene( sceneData );
            }
            // Cleanup scene, we are done
            exporter.cleanupScene();
        }
    }

CLEAN_EXIT:

    ip->PopPrompt();
    SetCursor(LoadCursor(NULL, IDC_ARROW));
    ip->RedrawViews( ip->GetTime() );
}

void hctMaxViewWindow::Restart()
{
    // Usefull to change renderer
    // Will loose current scene though
    if (m_hIntermediateParent)
    {
        m_viewportUtils->destroyPreviewWindow( m_hWnd );

        RECT cw;
        GetWindowRect( m_hIntermediateParent, &cw );

        m_viewportUtils->createPreviewWindow( m_hIntermediateParent, 0, 0, cw.right - cw.left, cw.bottom - cw.top );
        UpdateScene();
    }
}

void hctMaxViewWindow::Resize(int w, int h)
{
    if (m_viewportUtils)
    {
        m_viewportUtils->resizeWindow( w, h );
    }
}

BOOL hctMaxViewWindow::CanCreate()
{
   // Disabling the creation of the extended viewport as it is not properly supported
   //return !m_active; // only allow one for the moment.
   return false;
}

void hctMaxViewWindow::SetUtility( hctMaxViewport* util)
{
   m_utility = util;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
