// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/GUP/ConvexDecomp/hctConvexDecompUtility.h>
#include <ContentTools/Max/MaxSceneExport/GUP/ConvexDecomp/hctConvexDecompUtilityInterfaceImpl.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.3dsmax"
#include <Common/Base/System/Log/hkLog.hxx>


//
// Max interface
//

hctConvexDecompUtilityGUP* getConvexDecompUtilityInstance()
{
    static hctConvexDecompUtilityGUP theInstance;
    return &theInstance;
}

class hkConvexDecompositionUtilityClassDesc : public ClassDesc2
{
public:
    int IsPublic() { return TRUE; }
    void* Create( BOOL loading = FALSE ) { return getConvexDecompUtilityInstance(); }
    const MCHAR * ClassName() { return GetString( IDS_CONVEX_DECOMPOSITION_UTILITY_CLASS_NAME ); }
    SClass_ID SuperClassID() { return GUP_CLASS_ID; }
    Class_ID ClassID() { return HK_CONVEX_DECOMPOSITION_UTILITY_CLASS_ID; }
    const MCHAR*  Category() { return GetString( IDS_CONVEX_DECOMPOSITION_UTILITY_CATEGORY ); }
    const MCHAR* InternalName() { return _T("hkConvexDecompUtility"); } // Returns fixed parsable name (scripter-visible name)
    HINSTANCE HInstance() { return hInstance; } // Returns owning module handle
};

ClassDesc2* getHkConvexDecompUtilityDesc()
{
    static hkConvexDecompositionUtilityClassDesc maxConvexDecompDesc;
    return &maxConvexDecompDesc;
}

// Begin/EndEditParams calls which create and destroy the toolbar
DWORD hctConvexDecompUtilityGUP::Start()
{
    return GUPRESULT_KEEP;
}

void hctConvexDecompUtilityGUP::Stop()
{
}

//
// Progress bar job-handler
//

void hctMaxConvexDecompProgressHandler::processJobs()
{
    m_isCancelled_mainThread = m_isCancelled = false;
    char message[160];
    int percentage;

    HCT_SCOPED_CONVERSIONS;

    while ( m_newJobEvent->acquire(), true )
    {
        m_jobQueueLock->enter();
        strcpy( message, m_message );
        percentage = m_percentage;
        m_jobQueueLock->leave();

        if ( 0 == strcmp( "quit", message ) )
        {
            break;
        }

        Interface* ip = GetCOREInterface();
        ip->ProgressUpdate( percentage, FALSE, TO_MAX(message) );

        m_interfaceLock->enter();
        m_isCancelled_mainThread = m_isCancelled = ( ip->GetCancel() != FALSE );
        m_interfaceLock->leave();
    }
}

void hctMaxConvexDecompProgressHandler::setupBar()
{
    Interface* ip = GetCOREInterface();
    LPVOID arg = 0;
    MCHAR *message = TEXT("Press ESC to cancel");

    ip->ProgressStart( message, TRUE, __dummyUpdateFn, arg );

    m_nFinished = 0;
    m_nJobs = 0;
    m_isCancelled = false;
    m_wasCancelPressed = false;
}

void hctMaxConvexDecompProgressHandler::cleanupBar( const bool result )
{
    Interface* ip = GetCOREInterface();
    ip->ProgressUpdate( 0, FALSE, TEXT("") );
    ip->ProgressEnd();

    if ( m_isCombine )
    {
        ip->ReplacePrompt( result ? TEXT("Combine complete") : TEXT("Combine failed") );
    }
    else
    {
        ip->ReplacePrompt( result ? TEXT("Decomposition complete") : TEXT("Decomposition failed") );
    }
}

//
// Utility functions
//

// Find and load the convex decomposition DLL
bool _loadLibs( hctConvexDecompUtils** convexDecompUtils, hctBaseDll** convexDecompBaseDll, HMODULE& convexDecompHmodule, hkStringBuf& errorMessage, hkBool& displayErrorMessage )
{
    hkStringOld convexDecompUtilsDllName;
    {
        hkStringOld filterManagerPath;
        hctMaxUtils::getFilterManagerPath( filterManagerPath );
        convexDecompUtilsDllName = filterManagerPath + "\\utils\\hctConvexDecompUtils.dll";

        displayErrorMessage = FALSE;

        // Find library file
        hkIfstream testStream( convexDecompUtilsDllName.cString() );
        if ( !testStream.isOk() )
        {
            errorMessage = "Convex Decomposition Util DLL not found in filter manager's util dir";
            HK_WARN_ALWAYS( 0x7b5ac0db, errorMessage.cString() );
            return false;
        }

        displayErrorMessage = TRUE;

        // Load library
        convexDecompHmodule = LoadLibraryA( convexDecompUtilsDllName.cString() );
        if ( !convexDecompHmodule )
        {
            errorMessage = "Convex Decomposition Util DLL found but could not load. The DLL is corrupt.";
            HK_WARN_ALWAYS( 0x299d37e4, errorMessage.cString() );
            return false;
        }

        // Get base DLL interface
        hctGetBaseDllInterfaceFunc convexDecompBaseDllFunc = ( hctGetBaseDllInterfaceFunc ) GetProcAddress( convexDecompHmodule, "getBaseDllInterface" );
        if ( !convexDecompBaseDllFunc )
        {
            errorMessage = "Convex Decomposition DLL found but could not find baseDLL sync function. Install is invalid.";
            HK_WARN_ALWAYS( 0x6fdba42b, errorMessage.cString() );
            return false;
        }

        // Get convexDecompUtil DLL interface
        *convexDecompBaseDll = convexDecompBaseDllFunc();
        if ( *convexDecompBaseDll )
        {
            if ( (*convexDecompBaseDll)->getDllVersion() != HCT_CURRENT_VERSION )
            {
                errorMessage = "Could not load Convex Decomposition Util DLL interface as it was built off an older version of Havok than your current tools install. Please reinstall.";
                HK_WARN_ALWAYS( 0x7f425fa0, errorMessage.cString() );
                return false;
            }

            // Get the hkMemoryRouter and hkMemorySystem used here, and set these in the dll
            hkMemoryInitUtil::SyncInfo baseSystemInfo;
            hkSceneExportMemory::getBaseSystemSyncInfo( baseSystemInfo );
            (*convexDecompBaseDll)->initDll( baseSystemInfo, HK_NULL  );

            hctGetConvexDecompUtilInterfaceFunc convexDecompInterfaceFunc = ( hctGetConvexDecompUtilInterfaceFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompUtilInterface" );
            *convexDecompUtils = convexDecompInterfaceFunc ? convexDecompInterfaceFunc() : HK_NULL;

            if ( *convexDecompUtils )
            {
                // Set callback in dll such that a memory router created on a thread can be set back here in the modeler dll
                (*convexDecompUtils)->setThreadCallback( &hkSceneExportMemory::g_defaultThreadCallback );
            }
        }

        // Handle errors
        if ( !(*convexDecompUtils) || !(*convexDecompBaseDll) )
        {
            hctGetConvexDecompDllErrorFunc convexDecompErrorFunc = ( hctGetConvexDecompDllErrorFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompDllError" );
            if ( !convexDecompErrorFunc )
            {
                errorMessage = "Could not query Convex Decomposition Util DLL. Maybe your install is corrupt.";
                HK_WARN_ALWAYS( 0x34e5aa70, errorMessage.cString() );
                return false;
            }

            hctConvexDecompUtils::DllError convexDecompError = convexDecompErrorFunc();

            if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_ILLEGAL )
            {
                errorMessage = "The keycode for the Convex Decomposition Util DLL is invalid or is for a different product version. Please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x5a9a1283, errorMessage.cString() );
                return false;
            }

            if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_EXPIRED )
            {
                errorMessage = "The keycode for the Convex Decomposition Util DLL is expired or is for a different product version. Please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x7441b3ff, errorMessage.cString() );
                return false;
            }

            if ( convexDecompError & hctConvexDecompUtils::DLL_INIT_ERROR )
            {
                errorMessage = "The Convex Decomposition Util DLL failed to initialize properly. Please try restarting Maya. If this error repeats, please contact Havok support at http://support.havok.com";
                HK_WARN_ALWAYS( 0x146fc145, errorMessage.cString() );
                return false;
            }

            errorMessage = "The Convex Decomposition Util DLL failed to load. Maybe your install is corrupt. If this error repeats, please contact Havok support at http://support.havok.com";
            HK_WARN_ALWAYS( 0x21270eed, errorMessage.cString() );
            return false;
        }

        // Warn user if keycode is about to expire
        hctGetConvexDecompDllErrorFunc convexDecompErrorFunc = ( hctGetConvexDecompDllErrorFunc ) GetProcAddress( convexDecompHmodule, "getConvexDecompDllError" );
        if ( !convexDecompErrorFunc )
        {
            errorMessage = "Could not query Convex Decomposition Util DLL. Maybe your install is corrupt.";
            HK_WARN_ALWAYS( 0x7b16680f, errorMessage.cString() );
            return false;
        }

        hctConvexDecompUtils::DllError convexDecompError = convexDecompErrorFunc();

        if ( convexDecompError & hctConvexDecompUtils::DLL_KEYCODE_WARNING )
        {
            errorMessage = "The Convex Decomposition keycode expires in 10 days or less.";
            displayErrorMessage = TRUE;
            HK_WARN_ALWAYS( 0x6fbe3ac2, errorMessage.cString() );
        }
    }

    return true;
}

void _cleanupLibs( hctConvexDecompUtils* convexDecompUtils, hctBaseDll* convexDecompBaseDll, HMODULE& convexDecompHmodule )
{
    if ( convexDecompUtils )
    {
        delete convexDecompUtils;
    }

    // Cleanup
    if ( convexDecompBaseDll )
    {
        convexDecompBaseDll->quitDll();
    }

    if ( convexDecompHmodule )
    {
        FreeLibrary( convexDecompHmodule );
    }
}

//
// Convex decomposition
//

bool g_displayDllAlert = true;

BOOL hctConvexDecompUtilityGUP::checkValidity()
{
    HMODULE convexDecompHmodule = HK_NULL;
    hctBaseDll* convexDecompBaseDll = HK_NULL;
    hkStringBuf errorMessage;
    hkBool displayErrorMessage;
    m_convexDecompUtils = HK_NULL;
    bool isLibLoaded = _loadLibs( &m_convexDecompUtils, &convexDecompBaseDll, convexDecompHmodule, errorMessage, displayErrorMessage );

    if ( ( 0 != errorMessage.compareTo( "" ) ) && g_displayDllAlert && displayErrorMessage )
    {
        HCT_SCOPED_CONVERSIONS;

        MSTR script;
        script.printf( TEXT("print \"%s\""), TO_MAX(errorMessage.cString()) );
        ExecuteMAXScriptScript( script );

        g_displayDllAlert = false;
    }

    _cleanupLibs( m_convexDecompUtils, convexDecompBaseDll, convexDecompHmodule );

    return isLibLoaded;
}

BOOL hctConvexDecompUtilityGUP::decompose( Tab< INode* >& nodesIn, Tab< INode* >& nodesOut, BOOL detachParts, BOOL singleOut, double guardTolerance, BOOL hideMesh, INT reduceOverlapPasses, INT octreeDepth, INT method, INT reduceMethod, INT witerations, double wconcavity)
{
    Log_Info( "Method:{} ReduceMethod:{} Iterations:{} Concavity:{}", method, reduceMethod, witerations, wconcavity );
    bool result = false;

    HMODULE convexDecompHmodule = HK_NULL;
    hctBaseDll* convexDecompBaseDll = HK_NULL;
    hkStringBuf errorMessaage;
    hkBool displayErrorMessage;
    m_convexDecompUtils = HK_NULL;
    bool isLibLoaded = _loadLibs( &m_convexDecompUtils, &convexDecompBaseDll, convexDecompHmodule, errorMessaage, displayErrorMessage );

    if ( isLibLoaded )
    {
        // Get either single mesh, or group parent of hierarchy of meshes
        INode* baseNode = nodesIn[0];
        while ( baseNode->IsGroupMember() )
        {
            baseNode = baseNode->GetParentNode();
        }

        hctConvexDecompResults results;
        hkGeometry geometryFromScene;

        // Create geometry from original group selected in scene
        if ( 1 == nodesIn.Count() )
        {
            if ( !createGeometryFromSingleNode( baseNode, baseNode, geometryFromScene, results ) )
            {
                goto CLEANUP_LIBS;
            }
        }
        // Create geometry from group hierarchy
        else
        {
            if ( !createGeometryFromBaseNode( baseNode, geometryFromScene, results ) )
            {
                goto CLEANUP_LIBS;
            }
        }

        if( !displayWarning( results, true ) )
        {
            goto CLEANUP_LIBS;
        }

        SetCursor( LoadCursor( NULL, IDC_WAIT ) );

        // There are a lot of absolute epsilons in convex decomposition, therefore better to work with a normalized mesh
        hkTransform normalizedToWorld;
        {
            hkVector4 normalizeCenter;
            m_convexDecompUtils->normalizeGeometry( geometryFromScene, 10.0f, &normalizedToWorld, &normalizeCenter );
        }

        bool decomposeResult;
        hkArray< hkArray< hkVector4 > > outputShapesFromCD;
        {
            hctConvexDecompUtils::Input input;
            hctMaxConvexDecompProgressHandler progressHandler;
            input.m_progressJobHandler = (hctConvexDecompProgressHandler*)&progressHandler;
            input.setupDecomposeInput( detachParts, reduceOverlapPasses, guardTolerance, octreeDepth, method, reduceMethod, witerations, wconcavity);

            // Call utility
            decomposeResult = m_convexDecompUtils->decompose( input, geometryFromScene, outputShapesFromCD );
        }

        if ( decomposeResult )
        {
            {
                Interface* ip = GetCOREInterface();
                ip->ClearNodeSelection();
            }

            createOutputNodesFromPoints( baseNode, outputShapesFromCD, nodesOut, results, singleOut, &normalizedToWorld );

            if ( true == ( hideMesh != FALSE ) )
            {
                hideOriginalMesh( nodesIn );
            }

            MSTR text;
            text.printf( TEXT("hvkConvexDecomp_Results_DisplayResults %d %d %d %d %d %d"),
                results.nInShapes, results.nOutShapes, results.nInTriangles, results.nInPlanes,
                results.nOutPlanes, 0 );

            result = ( ExecuteMAXScriptScript( text ) != 0 );
        }

        if ( result )
        {
            result = ( nodesOut.Count() != 0 ) ? true : false;
        }
    }

CLEANUP_LIBS:
    _cleanupLibs( m_convexDecompUtils, convexDecompBaseDll, convexDecompHmodule );
    SetCursor( LoadCursor( NULL, IDC_ARROW ) );

    return result;
}

BOOL hctConvexDecompUtilityGUP::combine ( INode* baseNode, Tab< INode* >& origMeshNodes, Tab< INode* >& nodesIn, Tab< INode* >& nodesOut, double accuracyCombine, BOOL singleOutCombine, BOOL genGuards, INT octreeDepth, INT reduceMethod )
{
    bool result = false;

    HMODULE convexDecompHmodule = HK_NULL;
    hctBaseDll* convexDecompBaseDll = HK_NULL;
    hkStringBuf errorMessaage;
    hkBool displayErrorMessage;
    m_convexDecompUtils = HK_NULL;
    bool isLibLoaded = _loadLibs( &m_convexDecompUtils, &convexDecompBaseDll, convexDecompHmodule, errorMessaage, displayErrorMessage );

    if ( isLibLoaded )
    {
        hctConvexDecompResults results;
        hkGeometry baseGeometry;
        if (false == createGeometryFromOrigNodes( baseNode, origMeshNodes, baseGeometry, results ))
        {
            return false;
        }

        // There are a lot of absolute epsilons in convex decomposition, therefore better to work with a normalized mesh
        hkTransform normalizedToWorld;
        hkVector4 normalizeCenter;
        hkReal normalizeScale;
        normalizeScale = m_convexDecompUtils->normalizeGeometry( baseGeometry, 10.0f, &normalizedToWorld, &normalizeCenter );

        results.reset();

        hkArray< hkArray< hkVector4 > > shapesToCombine;
        createVertexArrayFromNodes( baseNode, nodesIn, shapesToCombine, results );

        m_convexDecompUtils->normalizeShapes( shapesToCombine, normalizeScale, normalizeCenter );

        if( !displayWarning( results, true ) )
        {
            goto CLEANUP_LIBS;
        }

        SetCursor( LoadCursor( NULL, IDC_WAIT ) );

        bool combineResult;
        hkArray< hkArray< hkVector4 > > outputShapesFromCD;
        {
            hctConvexDecompUtils::Input input;
            hctMaxConvexDecompProgressHandler progressHandler;
            input.m_progressJobHandler = (hctConvexDecompProgressHandler*)&progressHandler;
            input.setupCombineInput( genGuards, accuracyCombine, octreeDepth, reduceMethod );

            // Call utility
            combineResult = m_convexDecompUtils->combine( input, baseGeometry, outputShapesFromCD, &shapesToCombine );
        }

        if ( combineResult )
        {
            Interface* ip = GetCOREInterface();
            ip->ClearNodeSelection();

            createOutputNodesFromPoints( baseNode, outputShapesFromCD, nodesOut, results, singleOutCombine, &normalizedToWorld );

            INode* groupHead = nodesIn[0]->GetParentNode();

            for (int i = 0; i < nodesOut.Count(); ++i)
            {
                groupHead->AttachChild( nodesOut[i] );
            }

            result = (nodesOut.Count() != 0) ? true : false;

            if ( result )
            {
                for (int i = 0; i < nodesIn.Count(); ++i)
                {
                    ip->DeleteNode( nodesIn[i], 0 );
                }

                MSTR text;
                text.printf( TEXT("hvkConvexDecomp_Results_DisplayResults %d %d %d %d %d %d"),
                    results.nInShapes, results.nOutShapes, results.nInTriangles, results.nInPlanes,
                    results.nOutPlanes, 1 );

                result = (ExecuteMAXScriptScript( text ) != 0);
            }
        }
    }

CLEANUP_LIBS:
    _cleanupLibs( m_convexDecompUtils, convexDecompBaseDll, convexDecompHmodule );
    SetCursor( LoadCursor( NULL, IDC_ARROW ) );

    return result;
}

//
// Scene data processing
//

bool hctConvexDecompUtilityGUP::createGeometryFromSingleNode ( INode* baseMesh, INode* node, hkGeometry &geometry, hctConvexDecompResults& results )
{
    if ( !node || !baseMesh )
    {
        return false;
    }

    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();
    Object *obj = node->EvalWorldState( now ).obj;
    if ( !obj->CanConvertToType( Class_ID( TRIOBJ_CLASS_ID, 0 ) ) )
    {
        // ignore cameras, lights, etc. in the group
        return true;
    }

    TriObject *triObject = (TriObject*) obj->ConvertToType( now, Class_ID( TRIOBJ_CLASS_ID, 0 ) );
    Mesh& nodeMesh = triObject->mesh;
    hkGeometry nodeGeometry;
    {
        // copy vertices to geometry
        hkArray<hkVector4>& nodeVertices = nodeGeometry.m_vertices;
        nodeVertices.setSize( nodeMesh.getNumVerts() );

        int numVerts = nodeMesh.getNumVerts();
        Matrix3 objectTM = node->GetObjectTM( now );
        Matrix3 worldToNodeTM = Inverse( baseMesh->GetNodeTM( now ) );
        Matrix3 finalTM = objectTM * worldToNodeTM;

        for ( int vi = 0; vi < numVerts; ++vi )
        {
            Point3 pWorld = nodeMesh.verts[vi] * finalTM;

            hkVector4 v( pWorld.x, pWorld.y, pWorld.z );
            nodeVertices[vi] = v;
        }

        // copy triangles to geometry
        hkArray<hkGeometry::Triangle>& nodeTriangles = nodeGeometry.m_triangles;
        nodeTriangles.setSize( nodeMesh.getNumFaces() );

        int numFaces = nodeMesh.getNumFaces();
        for ( int ti = 0; ti < numFaces; ++ti )
        {
            Face& f = nodeMesh.faces[ti];
            nodeTriangles[ti].m_a = f.getVert(0);
            nodeTriangles[ti].m_b = f.getVert(1);
            nodeTriangles[ti].m_c = f.getVert(2);
        }

        ++results.nInShapes;
        results.nInTriangles += nodeMesh.getNumFaces();
    }

    m_convexDecompUtils->mergeGeometries( nodeGeometry, geometry );

    if ( obj != triObject )
    {
        triObject->DeleteThis();
    }

    if ( geometry.m_vertices.getSize() < 3 )
    {
        return false;
    }

    return true;
}

bool hctConvexDecompUtilityGUP::createGeometryFromBaseNode( INode* baseNode, hkGeometry& geometryFromScene, hctConvexDecompResults& results )
{
    if ( baseNode->IsGroupHead() )
    {
        hkArray<INode*> tempNodeStack;
        tempNodeStack.pushBack( baseNode );

        while ( 0 != tempNodeStack.getSize() )
        {
            INode* node = tempNodeStack.back();
            tempNodeStack.popBack();

            for ( int i = 0; i < node->NumberOfChildren(); ++i )
            {
                INode* child = node->GetChildNode(i);

                if ( child->IsGroupHead() )
                {
                    tempNodeStack.pushBack( child );
                }
                else
                {
                    if ( !createGeometryFromSingleNode( baseNode, child, geometryFromScene, results ) )
                    {
                        return false;
                    }
                }
            }
        }
    }

    return true;
}

BOOL hctConvexDecompUtilityGUP::createGeometryFromOrigNodes( INode* baseNode, Tab<INode*>& origMeshNodes, hkGeometry& geometryFromScene, hctConvexDecompResults& results )
{
    for (int i = 0; i < origMeshNodes.Count(); ++i)
    {
        INode* node = origMeshNodes[i];

        if (false == createGeometryFromSingleNode( baseNode, node, geometryFromScene, results ))
        {
            return false;
        }
    }

    return true;
}

bool hctConvexDecompUtilityGUP::createVertexArrayFromNodes( INode* baseNode, Tab< INode* >& nodesIn, hkArray< hkArray< hkVector4 > >& shapesToCombine, hctConvexDecompResults& results )
{
    Interface* ip = GetCOREInterface();
    const TimeValue now = ip->GetTime();
    for (int i = 0; i < nodesIn.Count(); ++i)
    {
        INode* node = nodesIn[i];
        Object *obj = node->EvalWorldState( now ).obj;
        TriObject *triObject = (TriObject*) obj->ConvertToType( now, Class_ID( TRIOBJ_CLASS_ID, 0 ));
        Mesh& nodeMesh = triObject->mesh;
        hkArray<hkVector4>& vertices = shapesToCombine.expandOne();
        int nVertices = nodeMesh.getNumVerts();
        vertices.setSize(nVertices);

        Matrix3 objectTM = node->GetObjectTM( now );
        Matrix3 worldToNodeTM = Inverse( baseNode->GetNodeTM( now ));
        Matrix3 finalTM = objectTM * worldToNodeTM;

        for (int vi = 0; vi < nVertices; ++vi)
        {
            Point3 pWorld = nodeMesh.verts[vi] * finalTM;

            hkVector4 v(pWorld.x, pWorld.y, pWorld.z);
            vertices[vi] = v;
        }

        // calculate n planes for results
        {
            hkGeometry geometry;
            geometry.m_vertices = vertices;

            hkArray<hkGeometry::Triangle>& nodeTriangles = geometry.m_triangles;
            nodeTriangles.setSize( nodeMesh.getNumFaces() );

            int numFaces = nodeMesh.getNumFaces();
            for (int ti = 0; ti < numFaces; ++ti)
            {
                Face& f = nodeMesh.faces[ti];
                nodeTriangles[ti].m_a = f.getVert(0);
                nodeTriangles[ti].m_b = f.getVert(1);
                nodeTriangles[ti].m_c = f.getVert(2);
            }

            int nPlanes = m_convexDecompUtils->getConvexHullPlanes( geometry );

            results.nInPlanes += nPlanes;
        }
    }

    results.nInShapes = nodesIn.Count();

    return true;
}

// Alert user if convex decomposition may take a long time
bool hctConvexDecompUtilityGUP::displayWarning( const hctConvexDecompResults& results, const bool isDecompose )
{
    if ( isDecompose && results.nInTriangles > m_triangleWarningThreshold )
    {
        char text[320];
        sprintf( text, "%d triangles are selected\nDecompose may take a long time\nSelecting 'Process elements independently' and setting \n'Reduce overlap steps' to 0 may reduce execution time", results.nInTriangles );

        if ( IDOK != MessageBoxA( NULL, text, "Warning", MB_OKCANCEL ) )
        {
            return false;
        }
    }
    else if ( !isDecompose && results.nInPlanes > m_triangleWarningThreshold * 0.5f )
    {
        char text[320];
        sprintf( text, "%d planes are selected\nCombine may take a long time", results.nInPlanes );

        if ( IDOK != MessageBoxA( NULL, text, "Warning", MB_OKCANCEL ) )
        {
            return false;
        }
    }
    return true;
}

void hctConvexDecompUtilityGUP::createOutputNodesFromPoints( INode* mainNode, hkArray< hkArray< hkVector4 > >& shapes, Tab< INode* >& nodesOut, hctConvexDecompResults& results, const BOOL& singleNodeOutput, const hkTransform* normalizedToWorld )
{
    hkGeometry combinedGeometry;
    int nShapes = shapes.getSize();

    hkPseudoRandomGenerator randomGenerator ( ++m_randomColorSeed * m_randomColorSeed * 12417 + m_randomColorSeed );

    for( int si = 0; si < nShapes; ++si )
    {
        hkGeometry shapeGeometry;
        m_convexDecompUtils->createConvexGeometry( shapes[si], shapeGeometry );

        if ( singleNodeOutput )
        {
            m_convexDecompUtils->mergeGeometries( shapeGeometry, combinedGeometry );
        }
        else
        {
            createNodeFromGeometry( shapeGeometry, mainNode, nodesOut, results, randomGenerator, normalizedToWorld );
        }
    }

    if ( singleNodeOutput )
    {
        createNodeFromGeometry( combinedGeometry, mainNode, nodesOut, results, randomGenerator, normalizedToWorld );
    }
}


// Create INode from single hkGeometry
void hctConvexDecompUtilityGUP::createNodeFromGeometry( hkGeometry& geometry, INode* mainNode, Tab< INode* >& nodesOut, hctConvexDecompResults& results, hkPseudoRandomGenerator& randomGenerator, const hkTransform* normalizedToWorld )
{
    Mesh newMesh;
    newMesh.setNumVerts( geometry.m_vertices.getSize() );
    newMesh.setNumFaces( geometry.m_triangles.getSize() );

    int numVerts = geometry.m_vertices.getSize();

    bool doTransform = false;
    hkTransform ntw;
    if ( NULL != normalizedToWorld )
    {
        ntw = *normalizedToWorld;
        doTransform = true;
    }
    for ( int vi = 0; vi < numVerts; ++vi )
    {
        hkVector4& v = geometry.m_vertices[vi];
        if ( doTransform )
        {
            v.setTransformedPos( ntw, v );
        }
        newMesh.setVert( vi, v(0), v(1), v(2) );
    }

    int numFaces = geometry.m_triangles.getSize();
    for ( int ti = 0; ti < numFaces; ++ti )
    {
        hkGeometry::Triangle& t = geometry.m_triangles[ti];
        newMesh.faces[ti].setVerts( t.m_a, t.m_b, t.m_c );
        newMesh.faces[ti].setEdgeVisFlags( EDGE_VIS, EDGE_VIS, EDGE_VIS );
        newMesh.faces[ti].setSmGroup(1);
    }
    newMesh.AutoSmooth( 0, false );

    int nPlanes = m_convexDecompUtils->getConvexHullPlanes( geometry );

    ++results.nOutShapes;
    results.nOutPlanes += nPlanes;

    // Create node
    TriObject* triObj = CreateNewTriObject();
    triObj->mesh = newMesh;
    Interface* ip = GetCOREInterface();
    INode *node = ip->CreateObjectNode( triObj );
    MSTR newNodeName = MSTR( mainNode->GetName() ) + TEXT("__hkConvDec");
    ip->MakeNameUnique(newNodeName);
    node->SetName(newNodeName);

    COLORREF wireColor = RGB( randomGenerator.getRandChar(256), randomGenerator.getRandChar(256), randomGenerator.getRandChar(256) );
    node->SetWireColor( wireColor );

    const TimeValue now = ip->GetTime();
    node->SetNodeTM( now, mainNode->GetNodeTM( now ) );
    node->CenterPivot( now, FALSE );

    nodesOut.Append( 1, &node );
}

void hctConvexDecompUtilityGUP::selectOutputNodes( Tab< INode* >& nodesOut )
{
    Interface* ip = GetCOREInterface();

    for ( int i = 0; i < nodesOut.Count(); ++i )
    {
        ip->SelectNode( nodesOut[i], 0 );
    }
}

void hctConvexDecompUtilityGUP::hideOriginalMesh( Tab< INode* >& nodesIn )
{
    Tab< INode* > hiddenNodes;

    if ( 1 == nodesIn.Count() )
    {
        hiddenNodes.Append( 1, &nodesIn[0] );
    }
    else
    {
        INode* baseNode = nodesIn[0];
        while ( baseNode->IsGroupMember() )
        {
            baseNode = baseNode->GetParentNode();
        }

        if ( baseNode->IsGroupHead() )
        {
            hkArray<INode*> tempNodeStack;
            tempNodeStack.pushBack(baseNode);

            while ( 0 != tempNodeStack.getSize() )
            {
                INode* node = tempNodeStack.back();
                tempNodeStack.popBack();

                for ( int i = 0; i < node->NumberOfChildren(); ++i )
                {
                    HCT_SCOPED_CONVERSIONS;

                    INode* child = node->GetChildNode(i);

                    if ( child->IsGroupHead() )
                    {
                        tempNodeStack.pushBack( child );
                    }
                    else if ( NULL == strstr( FROM_MAX(child->GetName()), "__hkConvDec" ) )
                    {
                        hiddenNodes.Append( 1, &child );
                    }
                }
            }
        }
    }

    for ( int i = 0; i < hiddenNodes.Count(); ++i )
    {
        hiddenNodes[i]->Hide( true );
    }

    if ( theHold.Holding() )
    {
        theHold.Put( new HideRestore( hiddenNodes ) );
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
