// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>

#include <Common/SceneData/Camera/hkxCamera.h>
#include <Common/SceneData/Light/hkxLight.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>

void hctMaxSceneExporter::fixupSkins()
{
    // For printing warnings when skin bindings cannot be fixed up.
    int processedSkinBindingsCount = 0;

    for( int skinBindingIndex = 0; skinBindingIndex < m_currentScene.m_skinBindings.getSize(); ++skinBindingIndex )
    {
        hkxSkinBinding* curSkin = m_currentScene.m_skinBindings[skinBindingIndex];
        processedSkinBindingsCount++;

        
        hkArray<INode*> inodes;
        reinterpret_cast< hkArray<INode*>& >(curSkin->m_nodeNames).swap(inodes);
        curSkin->m_nodeNames.setSize( inodes.getSize() );

        for( int mappingIndex = 0; mappingIndex < inodes.getSize(); ++mappingIndex )
        {
            // Get the nodeId.
            INode* maxNode = inodes[mappingIndex];

            // Make sure the node is in the map before setting the mapping.
            hkxNode* node = m_nodeMap.getWithDefault( maxNode, HK_NULL );
            if (node)
            {
                curSkin->m_nodeNames[mappingIndex] = node->m_name;
            }
            else
            {
                // Remove any references to the soon-to-be-removed skin binding node from the nodes in the scene.
                m_currentScene.m_rootNode->replaceAllObjects(m_currentScene.m_skinBindings[skinBindingIndex], curSkin->m_mesh);

                // Remove the skin binding, because not all of the necessary bones are in the exported scene.
                // The mesh will still be exported, but will not be skinned to the skeleton.
                m_currentScene.m_skinBindings.removeAtAndCopy( skinBindingIndex );
                --skinBindingIndex;

                HK_WARN_ALWAYS( 0xabba6655, "Skin binding " << processedSkinBindingsCount << " will not be created because not all of the required bones have been included for export." );
                break;
            }
        }
    }
}

hkxMesh* hctMaxSceneExporter::lookForAlreadyExportedMesh (INode* inode)
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;

    const Matrix3 localToWorld = inode->GetNodeTM (staticTick);
    const Matrix3 objectToWorld = inode->GetObjTMAfterWSM(staticTick);
    const Matrix3 objectToLocal = objectToWorld * Inverse(localToWorld);

    for (int i=0; i<m_exportedMeshesInstanceData.getSize(); i++)
    {
        if ( (obj == m_exportedMeshesInstanceData[i].m_maxObject) &&
             (objectToLocal.Equals(m_exportedMeshesInstanceData[i].m_objectToLocal, 0.01f))
             )
        {
            return m_exportedMeshesInstanceData[i].m_havokMesh;
        }
    }

    return HK_NULL;
}

void hctMaxSceneExporter::registerExportedMesh (INode* inode, hkxMesh* mesh)
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;
    const Matrix3 localToWorld = inode->GetNodeTM (staticTick);
    const Matrix3 objectToWorld = inode->GetObjTMAfterWSM(staticTick);
    const Matrix3 objectToLocal = objectToWorld * Inverse(localToWorld);

    ExportedMeshInstanceData exportedMeshData;
    exportedMeshData.m_havokMesh = mesh;
    exportedMeshData.m_maxObject = obj;
    exportedMeshData.m_objectToLocal = objectToLocal;

    m_exportedMeshesInstanceData.pushBack(exportedMeshData);
}

void hctMaxSceneExporter::processPolyMesh( INode *inode, hkRefVariant& object, bool forceSkinned )
{
    // EXP-861
    hkxMesh* instancedMesh = lookForAlreadyExportedMesh(inode);
    if (instancedMesh)
    {
        object = instancedMesh;
        return;
    }

    hkxMesh* newMesh = HK_NULL;
    hkxSkinBinding* newSkin = HK_NULL;
    m_meshWriter.init( inode, m_iStaticFrame );
    m_meshWriter.extract();
    object = HK_NULL;
    m_meshWriter.write(newMesh, newSkin, forceSkinned);

    if (newSkin)
    {
        object = newSkin;
        m_currentScene.m_skinBindings.pushBack(newSkin);
        m_currentScene.m_meshes.pushBack(newMesh);

        newSkin->removeReference();
        newMesh->removeReference();
    }
    else if (newMesh)
    {
        object = newMesh;
        m_currentScene.m_meshes.pushBack(newMesh);
        registerExportedMesh(inode, newMesh);
        newMesh->removeReference();
    }
}

hkxSpline* hctMaxSceneExporter::processSpline( INode* inode )
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;

    SplineShape* splineShape = (SplineShape*) (obj->ConvertToType(staticTick, (Class_ID(SPLINESHAPE_CLASS_ID,0))));

    if (!splineShape)
    {
        return HK_NULL;
    }

        /// If the conversion required a new object, we are responsible for deleting it
  const bool mustDelete = splineShape!=obj;

  BezierShape bezierShape; splineShape->MakeBezier(staticTick, bezierShape);

  if (bezierShape.splineCount == 0)
    {
        HK_WARN_ALWAYS( 0xabba6355, "Shape " << inode->GetName() << " has not splines. Ignoring it." );
        if (mustDelete)
        {
                delete splineShape;
        }
        return HK_NULL;
    }

    hkxSpline* newSpline = new hkxSpline;


    Spline3D *spline3D = bezierShape.GetSpline(0);

    for (int j = 0; j < spline3D->KnotCount(); j++)
    {
          Point3 knotPos = spline3D->GetKnot(j).Knot();
          Point3 knotIn = spline3D->GetKnot(j).InVec();
          Point3 knotOut = spline3D->GetKnot(j).OutVec();

          //Create cp
          hkxSpline::ControlPoint& cp = newSpline->m_controlPoints.expandOne();
          cp.m_position.set(knotPos.x, knotPos.y, knotPos.z);
          cp.m_tangentIn.set(knotIn.x, knotIn.y, knotIn.z);
          cp.m_tangentOut.set(knotOut.x, knotOut.y, knotOut.z);

      int iInType = spline3D->GetKnot(j).Ktype();
      switch (iInType)
      {
        case KTYPE_AUTO : cp.m_inType = hkxSpline::BEZIER_SMOOTH; break;
        case KTYPE_CORNER : cp.m_inType = hkxSpline::LINEAR; break;
        case KTYPE_BEZIER_CORNER : cp.m_inType = hkxSpline::BEZIER_CORNER; break;
        default: cp.m_inType = hkxSpline::CUSTOM; break;
      }
      int iOutType = spline3D->GetKnot(j).Ktype();
      switch (iOutType)
      {
        case KTYPE_AUTO : cp.m_outType = hkxSpline::BEZIER_SMOOTH; break;
        case KTYPE_CORNER : cp.m_outType = hkxSpline::LINEAR; break;
        case KTYPE_BEZIER_CORNER : cp.m_outType = hkxSpline::BEZIER_CORNER; break;
        default: cp.m_outType = hkxSpline::CUSTOM; break;
      }
    }

      newSpline->m_isClosed = (splineShape->CurveClosed(staticTick, 0) == TRUE);

    if (mustDelete)
    {
        splineShape->DeleteThis();
    }

    m_currentScene.m_splines.pushBack(newSpline);

    return newSpline;
}

hkxLight* hctMaxSceneExporter::processLight( INode *inode )
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;
    LightObject* maxLight = (LightObject*) obj;
    GenLight* gl = dynamic_cast<GenLight*>(maxLight);
    LightState lightState;
    // Is it enabled?
    if (!(maxLight->EvalLightState(staticTick, FOREVER, &lightState)) || !lightState.on)
    {
        return HK_NULL;
    }

    hkxLight* newLight = new hkxLight();

    bool hasTarget = false;
    switch (obj->ClassID().PartA())
    {
        case OMNI_LIGHT_CLASS_ID:   // Omni light
        {
            newLight->m_type = hkxLight::POINT_LIGHT;
            break;

        }
        case SPOT_LIGHT_CLASS_ID:   // target spotlight
        {
            hasTarget = true;
            newLight->m_type = hkxLight::SPOT_LIGHT;
            break;
        }
        case FSPOT_LIGHT_CLASS_ID:  // free spotlight
        {
            hasTarget = false;
            newLight->m_type = hkxLight::SPOT_LIGHT;
            break;
        }
        case DIR_LIGHT_CLASS_ID:    // free directional
        {
            hasTarget = false;
            newLight->m_type = hkxLight::DIRECTIONAL_LIGHT;
            break;
        }
        case TDIR_LIGHT_CLASS_ID:   // target directional
        {
            hasTarget = true;
            newLight->m_type = hkxLight::DIRECTIONAL_LIGHT;
            break;
        }
        default:
        {
            // Sometimes IGame returns values not in the range of the enum
            // for example for IES Sun lights (see EXP-474)
            HK_WARN_ALWAYS(0xabba99ce, "Unsupported light type");
            return HK_NULL;
        }
    }


    INode* targetNode = inode->GetTarget();
    hasTarget = hasTarget && (targetNode!=NULL);

    Matrix3 lightMatrix = inode->GetObjTMAfterWSM(staticTick);

    // Position
    {
        Point3 pos = lightMatrix.GetTrans();
        newLight->m_position.set( pos[0], pos[1], pos[2]);
    }

    // Direction
    {
        Point3 dir;
        if (!hasTarget)
        {
            dir = -lightMatrix.GetRow(2); // negative z
        }
        else
        {
            Point3 tpos = targetNode->GetObjTMAfterWSM(staticTick).GetTrans();
            dir = ( tpos - lightMatrix.GetTrans() );
        }
        dir = Normalize( dir );
        newLight->m_direction.set( dir[0], dir[1], dir[2] );
    }

    // Light color
    {
        Point3 rgb = maxLight->GetRGBColor(staticTick);
        unsigned int crgb = 0xffffffff;
        crgb = hkSceneExportUtils::floatsToARGB(rgb.x, rgb.y, rgb.z, 1.0f);
        newLight->m_color = crgb;
    }

    // Angle
    {
        const float fo = maxLight->GetFallsize(staticTick);
        newLight->m_innerAngle = fo * HK_REAL_DEG_TO_RAD * 0.75;
        newLight->m_outerAngle = fo * HK_REAL_DEG_TO_RAD;
    }

    // Intensity
    newLight->m_intensity = lightState.intens;

    if(gl)
    {
        // Decay Rate
        newLight->m_decayRate = (hkInt16)gl->GetDecayType();

        // Radius
        if (gl->Type() == TSPOT_LIGHT)
        {
            theHold.Begin();
            gl->SetType(FSPOT_LIGHT);
            newLight->m_range = gl->GetTDist(0);
            theHold.Cancel();
        }
        else if (newLight->m_type != hkxLight::POINT_LIGHT)
        {
            newLight->m_range = maxLight->GetTDist(0);
        }
        else // POINT LIGHT
        {
            if (newLight->m_decayRate)
            {
                float cutOff = 0.01f;

                // Calculate range of new light
                newLight->m_range = hkMath::pow((hkReal)(newLight->m_intensity / cutOff), (hkReal)(1.f / newLight->m_decayRate));
            }
            else
            {
                // No decay
                HK_WARN_ALWAYS(0xabba92ce, "Point lights with no decay are not supported. Please use a directional light, or ambient lighting instead.");
                return HK_NULL;
            }
        }
    }

    // This is the range that the camera will start to fade the light out (because it is too far away)
    newLight->m_fadeStart = newLight->m_range * 2;
    newLight->m_fadeEnd = newLight->m_range * 3;

    // Shadow
    newLight->m_shadowCaster = lightState.shadow;

    m_currentScene.m_lights.pushBack(newLight);

    return newLight;
}

hkxCamera* hctMaxSceneExporter::processCamera(INode *inode)
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;

    hkxCamera* newCamera = new hkxCamera();

    CameraObject* maxCamera = (CameraObject*) obj;

    Matrix3 camMatrix = inode->GetObjTMAfterWSM(staticTick);

    INode* targetInode = inode->GetTarget();

    // Target (point of interest)
    {
        Point3 poi;
        if (targetInode)
        {
            Matrix3 tm = targetInode->GetObjTMAfterWSM(staticTick);
            poi = tm.GetTrans();
        }
        else
        {
            float dist = maxCamera->GetTDist(staticTick);
            poi = (Point3)( (camMatrix.GetRow(2)*-dist) + camMatrix.GetRow(3) ); //-z is front on free cam, row 3 == trans
        }
        newCamera->m_focus.set( poi[0], poi[1], poi[2] );
    }

    // From (eye pos)
    {
        Point3 eye = camMatrix.GetTrans();
        newCamera->m_from.set( eye[0], eye[1], eye[2] );
    }

    // Up
    {
        Point3 up = camMatrix.GetRow(1);
        newCamera->m_up.set( up[0], up[1], up[2] );
    }

    // fov
    {
        float horFov = maxCamera->GetFOV(staticTick);
        newCamera->m_fov = hctMaxUtils::convertHorizontalToVerticalFOV(horFov);
    }

    // clipping distances
    newCamera->m_far = 100000.0f;   // some large value
    newCamera->m_near = 0.1f;       // some not-too-small value
    if( maxCamera->GetManualClip() )
    {
        newCamera->m_near = maxCamera->GetClipDist(staticTick, CAM_HITHER_CLIP);
        newCamera->m_far = maxCamera->GetClipDist(staticTick, CAM_YON_CLIP);
    }

    // Cameras in max are right-handed
    newCamera->m_leftHanded = false;

    m_currentScene.m_cameras.pushBack(newCamera);

    return newCamera;
}


/*
** Default camera  : see EXP-69
*/


static inline ViewExp* _getActiveViewport ()
{
    Interface* coreInterface = GetCOREInterface();
#if MAX_VERSION_MAJOR>=15
    return &coreInterface->GetActiveViewExp();
#else
    return coreInterface->GetActiveViewport();
#endif
}

static inline ViewExp* _getViewport(HWND hWnd)
{
    Interface* coreInterface = GetCOREInterface();
#if MAX_VERSION_MAJOR>=15
    return &coreInterface->GetViewExp(hWnd);
#else
    return coreInterface->GetViewport(hWnd);
#endif
}

static BOOL CALLBACK _enumViewportsProc( HWND hWnd, LPARAM lParam )
{
    ViewExp** goodViewportRef = (ViewExp**) (lParam);

    ViewExp* thisViewport = _getViewport(hWnd);

    if (thisViewport!=NULL)
    {
        const int viewportType = thisViewport->GetViewType();

        // If this viewport is a perspective viewport or camera viewport, that's the one!
        if ((viewportType == VIEW_PERSP_USER) || (viewportType == VIEW_CAMERA))
        {
            (*goodViewportRef) = thisViewport;
            return FALSE;
        }

    }

    return TRUE;
}

static ViewExp* _getBestViewport()
{

    ViewExp* currentViewport = _getActiveViewport();

    const int viewportType = currentViewport->GetViewType();

    // If the current viewport is a perspective viewport or camera viewport, that's the one!
    if ((viewportType == VIEW_PERSP_USER) || (viewportType == VIEW_CAMERA))
    {
        return currentViewport;
    }

    // Otherwise, we start choosing the current viewport
    ViewExp* goodViewport = currentViewport;

    // But try to find another viewport that is better
    HWND parent = GetParent( currentViewport->GetHWnd() );

    EnumChildWindows( parent, _enumViewportsProc, (LPARAM) &goodViewport );

    return goodViewport;
}

hkxCamera* hctMaxSceneExporter::createDefaultCamera()
{
    ViewExp* currentViewport = _getBestViewport();
    const int viewportType = currentViewport->GetViewType();

    switch (viewportType)
    {
        case VIEW_PERSP_USER:
            {
                Matrix3 cameraMat(1);
                const float focalDist = currentViewport->GetFocalDist();

                currentViewport->GetAffineTM(cameraMat);

                cameraMat.Invert();

                const Point3 toAxis = -cameraMat.GetRow(2);
                const hkVector4 toAxisNorm = hctMaxUtils::asHkVector4(Normalize(toAxis));

                const hkVector4 from = hctMaxUtils::asHkVector4(cameraMat.GetTrans());
                hkVector4 to; to.setAddMul4(from, toAxisNorm, focalDist);
                const hkVector4 up = hctMaxUtils::asHkVector4(cameraMat.GetRow(1));

                const float hFov = currentViewport->GetFOV();

                hkxCamera* newCamera = new hkxCamera();

                newCamera->m_from = from;
                newCamera->m_focus = to;
                newCamera->m_up = up;

                newCamera->m_fov = hctMaxUtils::convertHorizontalToVerticalFOV(hFov);
                newCamera->m_leftHanded = false;

                // Clipping planes, based on distance
                hkVector4 fromTo; fromTo.setSub4(to, from);
                const hkReal distance = fromTo.length3();

                newCamera->m_far = distance * 8.0f;
                newCamera->m_near = distance * 0.001f;

                return newCamera;
            }

        default:
            {
                HK_WARN_ALWAYS (0xabba0d39, "Couldn't find a perspective viewport to create a default camera");
                return HK_NULL;
            }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
