// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>

#include <ContentTools/Max/MaxSceneExport/Exporter/CommonInterfaces/hctCommonParameterInterface.h>

#include <custattrib.h>
#include <icustattribcontainer.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Attributes/hkxAttributeGroup.h>

#include <ContentTools/Common/SceneExport/AttributeProcessing/hctAttributeDescription.h>

hctAttributeSelectionUtil::UserAction hctMaxSceneExporter::filterAttribute (const hctCommonParameterInterface& pblock, int index)
{
    HCT_SCOPED_CONVERSIONS;

    // EXP-913 - User selected attributes
    hkStringOld internalClassName = FROM_MAX(pblock.getInternalClassName());
    hkStringOld uiClassName = FROM_MAX(pblock.getUIClassName());
    hkStringOld pblockName  = FROM_MAX(pblock.getPBlockName());

    hkStringOld internalParamName = FROM_MAX(pblock.getParameterInternalName (index));
    hkStringOld uiParamName = FROM_MAX(pblock.getParameterUIName (index));

    // We check combinations of internal/ui object and parameter names
    hctAttributeSelectionUtil::UserAction desiredAction = hctAttributeSelectionUtil::USER_UNDEFINED;

    desiredAction = m_attributeSelection.filterAttribute(uiClassName.cString(), pblockName.cString(), uiParamName.cString());

    if (desiredAction!=hctAttributeSelectionUtil::USER_UNDEFINED)
    {
        return desiredAction;
    }

    desiredAction = m_attributeSelection.filterAttribute(internalClassName.cString(), pblockName.cString(), internalParamName.cString());

    if (desiredAction!=hctAttributeSelectionUtil::USER_UNDEFINED)
    {
        return desiredAction;
    }

    desiredAction = m_attributeSelection.filterAttribute(uiClassName.cString(), pblockName.cString(), internalParamName.cString());

    if (desiredAction!=hctAttributeSelectionUtil::USER_UNDEFINED)
    {
        return desiredAction;
    }

    desiredAction = m_attributeSelection.filterAttribute(internalClassName.cString(), pblockName.cString(), uiParamName.cString());

    return desiredAction;

}

hkResult hctMaxSceneExporter::convertSingleAttrib(const hctCommonParameterInterface& pblock, int idx, hkxAttribute& attributeOut, const hkReflect::Type* klass)
{
    HCT_SCOPED_CONVERSIONS;

    hctCommonParameterInterface::AttributeType attributeType = pblock.getAttributeType(idx);
    MSTR        paramNameStr      = pblock.getParameterInternalName(idx);
    const char* paramName         = FROM_MAX(paramNameStr);
    int         tpf               = GetTicksPerFrame();
    int         tickWholeDuration = m_exportOptions.m_animationEnd - m_exportOptions.m_animationStart;
    int         numFrames         = (int)(tickWholeDuration / (float)tpf) + 1; // frame range -> num frames (inc each p_end)
    const bool  isParamAnimated   = pblock.isParameterAnimated(idx);
    const int   framesToSample    = isParamAnimated ? numFrames : 1;
    TimeValue   startTime         = m_exportOptions.m_animationStart;
    float       fps               = (float)GetFrameRate();
    int         numTicks          = GetTicksPerFrame();

    hkRefVariant var;

    switch ( attributeType )
    {
    case hctCommonParameterInterface::AT_STRING:
        {
            MSTR stringValue;
            hkResult result = pblock.getParameterValue(idx, startTime, stringValue);

            if (result.isSuccess())
            {
                hkxSparselyAnimatedString* v = new hkxSparselyAnimatedString();
                v->m_times.setSize(1);
                v->m_strings.setSize(1);
                v->m_times[0] = 0;
                if (stringValue)
                {
                    v->m_strings[0] = FROM_MAX(stringValue);
                }

                var = v;
                v->removeReference();
            }
            break;
        }

    case hctCommonParameterInterface::AT_FLOAT:
        {
            hkxAnimatedFloat* v = new hkxAnimatedFloat();
            v->m_floats.setSize(framesToSample);

            float floatValue = 0.0f;
            int currentNumticks = GetTicksPerFrame();
            for (int f=0; f<framesToSample; f++)
            {
                pblock.getParameterValue(idx, startTime + f*currentNumticks, floatValue);
                v->m_floats[f] = floatValue;
            }

            v->m_hint = pblock.getAttributeHint(idx);

            var = v;
            v->removeReference();

            break;
        }

    case hctCommonParameterInterface::AT_BOOL:
        {
            hkxSparselyAnimatedBool* v = new hkxSparselyAnimatedBool();

            // Get the sparse animated bool array
            hkArray<hkBool>& sparseValues = v->m_bools;
            hkArray<float>& sparseTimes = v->m_times;
            sparseTimes.reserve(framesToSample);
            sparseValues.reserve(framesToSample);

            hkBool boolValue=0;
            for (int f=0; f<framesToSample; f++)
            {
                pblock.getParameterValue(idx, startTime + f*numTicks, boolValue );

                bool newValue = true;
                if (f > 0)
                {
                    newValue = sparseValues[sparseValues.getSize()-1] != boolValue;
                }

                if (newValue)
                {
                    sparseValues.pushBack (boolValue);
                    sparseTimes.pushBack (f / fps );
                }
            }

            var = v;
            v->removeReference();

            break;
        }

    case hctCommonParameterInterface::AT_INT:
        {
            hkxSparselyAnimatedInt* v = new hkxSparselyAnimatedInt();

            // Get the sparse animated int array
            hkArray<int>& sparseValues = v->m_ints;
            hkArray<float>& sparseTimes = v->m_times;
            sparseTimes.reserve(framesToSample);
            sparseValues.reserve(framesToSample);

            int intValue=0;
            for (int f=0; f<framesToSample; f++)
            {
                pblock.getParameterValue(idx, startTime + f*numTicks, intValue );

                bool newValue = true;
                if (f > 0)
                {
                    newValue = sparseValues[sparseValues.getSize()-1] != intValue;
                }

                if (newValue)
                {
                    sparseValues.pushBack (intValue);
                    sparseTimes.pushBack (f / fps );
                }
            }

            // Fix enums because we increased the values by one because of maxscript
            if( klass )
            {
                if( const hkReflect::RecordType* rec = klass->asRecord() )
                {
                    hkReflect::FieldDecl member = rec->findField(paramName, true);
                    if ( member && member.getType()->findAttribute<hk::Presets>() )
                    {
                        const int numValues = sparseValues.getSize();
                        for ( int i=0; i<numValues; i++ )
                        {
                            sparseValues[i]--;
                        }
                    }
                }
            }

            var = v;
            v->removeReference();

            break;
        }

    case hctCommonParameterInterface::AT_VECTOR:
        {
            hkxAnimatedVector* v = new hkxAnimatedVector();
            v->m_vectors.setSize(4*framesToSample);

            hkVector4 vectorValue; vectorValue.setZero();

            int currentNumTicks = GetTicksPerFrame();
            for (int f=0; f<framesToSample; f++)
            {
                pblock.getParameterValue(idx, startTime + f*currentNumTicks, vectorValue);
                vectorValue.store<4,HK_IO_NATIVE_ALIGNED>(&v->m_vectors[4*f]);
            }

            v->m_hint = pblock.getAttributeHint(idx);

            var = v;
            v->removeReference();

            break;
        }

    case hctCommonParameterInterface::AT_MATRIX:
        {
            hkxAnimatedMatrix* v = new hkxAnimatedMatrix();
            v->m_matrices.setSize(16*framesToSample);

            hkMatrix4 matrixValue; matrixValue.setIdentity();

            int currentNumTicksM = GetTicksPerFrame();
            for (int f=0; f<framesToSample; f++)
            {
                pblock.getParameterValue(idx, startTime + f*currentNumTicksM, matrixValue);
                matrixValue.get4x4ColumnMajor(&v->m_matrices[16*f]);
            }

            v->m_hint = pblock.getAttributeHint(idx);

            var = v;
            v->removeReference();

            break;

        }

    case hctCommonParameterInterface::AT_UNSUPPORTED:
    default:
        {
            break;
        }
    }

    if ( var.val() != HK_NULL )
    {
        attributeOut.m_name = paramName;
        attributeOut.m_value = var;
        return HK_SUCCESS;
    }
    else
    {
        return HK_FAILURE;
    }
}

hkxAttributeGroup* hctMaxSceneExporter::convertAttribGroup( const hctCommonParameterInterface& pblock, bool exportAll, ReferenceTarget* curObject )
{
    MSTR pblockName = pblock.getPBlockName();
    MSTR groupName = pblockName;

    //const char* groupName = groupNameStr.cString();

    TimeValue startTime = m_exportOptions.m_animationStart;

    hkxAttributeGroup* group = new hkxAttributeGroup();
    hkArray<hkxAttribute>&  havokAttribs = group->m_attributes;

    int num_params = pblock.getNumParameters();

    HCT_SCOPED_CONVERSIONS;

    // Get class
    MSTR modifierName;
    curObject->GetClassName(modifierName);
    const hkReflect::Type* klass = hctSdkUtils::getTypeByName(FROM_MAX(modifierName));

    for ( int idx = 0 ; idx < num_params; idx++ )
    {
        hctCommonParameterInterface::AttributeType attributeType = pblock.getAttributeType(idx);
        MSTR paramNameStr = pblock.getParameterInternalName(idx);
        const char* paramName = FROM_MAX(paramNameStr);

        // Special case : string attribute "hkType" changes the type of the group
        if ((hkString::strCasecmp(paramName, "hkType")==0) && (attributeType==hctCommonParameterInterface::AT_STRING))
        {
            pblock.getParameterValue(idx, startTime, groupName);
            continue;
        }

        // Ignore attributes with no name
        if (hkString::strLen(paramName)==0)
        {
            continue;
        }

        hctAttributeSelectionUtil::UserAction userAction = filterAttribute(pblock, idx);

        if (userAction == hctAttributeSelectionUtil::USER_REMOVE)
        {
            continue;
        }

        if ( (userAction == hctAttributeSelectionUtil::USER_UNDEFINED) && !exportAll )
        {
            continue;
        }

        hkxAttribute attribute;
        hkResult res = convertSingleAttrib(pblock, idx, attribute, klass);
        if ( res.isSuccess() )
        {
            havokAttribs.pushBack(attribute);
        }
    }


    if (havokAttribs.getSize() > 0)
    {
        // Name
        group->m_name = FROM_MAX(groupName);
        return group;
    }
    else
    {
        delete group;
        return HK_NULL;
    }
}

void hctMaxSceneExporter::exportAttributes(ReferenceTarget* curObject, hkArray<hkxAttributeGroup>& havokAttribGroups)
{
    // Get an initial list of groups belonging to this object/modifier
    hkArray<hkxAttributeGroup> thisObjectGroups;
    {

        // All pblocks
        {
            for(int i=0;i<curObject->NumRefs();i++)
            {
                ReferenceTarget* subRef = curObject->GetReference(i);

                if (subRef == HK_NULL) continue;

                hctCommonParameterInterface* pblockInterface = HK_NULL;
                {
                    if (subRef->SuperClassID()== PARAMETER_BLOCK2_CLASS_ID)
                    {
                        IParamBlock2* pblock2 = (IParamBlock2*) subRef;
                        pblockInterface = new hctParamBlock2Interface (pblock2);

                    }
                    else if (subRef->SuperClassID()== PARAMETER_BLOCK_CLASS_ID)
                    {
                        pblockInterface = new hctParamBlock1Interface (curObject, i);

                    }
                    else
                    {
                        // Avoid recursing into submaterials since we'll do that at a different level
                        if (curObject->SuperClassID() == MATERIAL_CLASS_ID)
                        {
                            if (subRef->SuperClassID() == MATERIAL_CLASS_ID)
                            {
                                continue;
                            }
                        }

                        // Do not recurse references to node or derived objects
                        // since they will sampled later on
                        Class_ID classID = subRef->ClassID();
                        if (classID.PartB() == 0)
                        {
                            switch (classID.PartA())
                            {
                                case BASENODE_CLASS_ID:
                                case GEN_DERIVOB_CLASS_ID:
                                case DERIVOB_CLASS_ID:
                                    continue;
                            }
                        }

                        // Recurse in case the subanim has pblocks (cameras, booleans for example)
                        exportAttributes (subRef,  thisObjectGroups);
                    }
                }

                if (pblockInterface)
                {
                    HCT_SCOPED_CONVERSIONS;

                    const hkStringOld pblockName = hkStringOld (FROM_MAX(pblockInterface->getPBlockName())).asLowerCase();

                    const bool exportAllAttributes = pblockName.beginsWith("hk");

                    hkxAttributeGroup* g = convertAttribGroup (*pblockInterface, exportAllAttributes, curObject);
                    if (g)
                    {
                        thisObjectGroups.pushBack(*g);
                        delete g;
                    }

                    delete pblockInterface;
                }

            }

        }
        // Check for custom attributes
        ICustAttribContainer* cc = curObject->GetCustAttribContainer();
        if ( cc )
        {
            int num_attribs = cc->GetNumCustAttribs();

            // Each attrib in Max is a roll down, so we can group them as such
            for (int a= 0; a < num_attribs; a++ )
            {
                CustAttrib* attr = (CustAttrib*)cc->GetCustAttrib( a );
                if (attr) // just to be safe
                {
                    // Usually CA only have one PBlock, but they can have more than one
                    const int numParamBlocks = attr->NumParamBlocks();

                    for (int i=0; i<numParamBlocks; i++)
                    {
                        IParamBlock2* pblock2 = attr->GetParamBlock(i);
                        hctParamBlock2Interface pblockInterface (pblock2);

                        // EXP-474 : Sometimes pblock2 is NULL
                        if (!pblock2) continue;

                        hkxAttributeGroup* g = convertAttribGroup(pblockInterface, true, curObject);
                        if (g)
                        {
                            thisObjectGroups.pushBack(*g); // copy.
                            delete g;
                        }
                    }

                    //
                    // Special treatment for Havok Destruction's Generic Modifier:
                    // merge all separate Attribute Groups into one, using the modifier's classname as group name.
                    //
                    //XXCK : Since the change to mem managed (hkStringPtr etc) all these copies etc are slow
                    //       and should be refactored to cleaner, fast, code.
                    {
                        MSTR modifierNameStr;
                        curObject->GetClassName(modifierNameStr);

                        HCT_SCOPED_CONVERSIONS;

                        const char* modifierName = FROM_MAX(modifierNameStr);

                        // Verify that this modifier is actually a Generic Modifier.
                        if ( (hctSdkUtils::getTypeByName(modifierName) != HK_NULL) && (thisObjectGroups.getSize() > 1) )
                        {
                            int totalNumAttributes = 0;
                            for ( int i = 0; i < thisObjectGroups.getSize(); i++)
                            {
                                totalNumAttributes += thisObjectGroups[i].m_attributes.getSize();
                            }

                            hkxAttributeGroup mergedGroup;
                            mergedGroup.m_name = modifierName;
                            mergedGroup.m_attributes.reserve( totalNumAttributes );

                            // Copy the individual attributes from all original groups into the new 'merged' group.
                            for ( int ii = 0; ii < thisObjectGroups.getSize(); ii++)
                            {
                                const hkxAttributeGroup& sourceAttributeGroup = thisObjectGroups[ii];
                                mergedGroup.m_attributes.append(sourceAttributeGroup.m_attributes.begin(), sourceAttributeGroup.m_attributes.getSize() );
                            }

                            thisObjectGroups.clear();
                            thisObjectGroups.pushBack(mergedGroup);
                        }
                    }
                }
            }
        }
    }

    // Finally add these groups to the node list
    {
        havokAttribGroups.insertAt(havokAttribGroups.getSize(), thisObjectGroups.begin(), thisObjectGroups.getSize());
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
