// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>

#if defined(HK_COMPILER_MSVC)
#   pragma warning(disable: 4244)   // conversion from 'int' to 'BYTE', possible loss of data
#   pragma warning(disable: 4996)   // deprecated
#endif

#include <ContentTools/Common/SdkUtils/ClassHierarchy/hctClassHierarchyUtil.h>

#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/CommonInterfaces/hctCommonParameterInterface.h>

#include <ContentTools/Common/Filters/FilterManager/hctFilterManagerImpl.h>

#include <XRef/iXrefObj.h>
#include <notetrck.h>
#include <modstack.h>
#include <custattrib.h>
#include <icustattribcontainer.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Light/hkxLight.h>
#include <Common/SceneData/Camera/hkxCamera.h>
#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Attributes/hkxAttributeGroup.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>
#include <Common/SceneData/Selection/hkxNodeSelectionSet.h>

#include <ContentTools/Common/SceneExport/AttributeProcessing/hctAttributeDescription.h>
#include <ContentTools/Common/SceneExport/Utils/hctSceneExportUtils.h>
#include <ContentTools/Common/Filters/Common/Utils/hctFilterUtils.h>

#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#include <Common/Base/Types/Uuid/hkUuidMapOperations.cxx>
#include <Common/Base/Container/PointerMap/hkMap.hxx>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.export.3dsmax"
#include <Common/Base/System/Log/hkLog.hxx>

// This registry key contains the path to the filter manager.
// The most recently used filter set is also stored from here.
#ifdef HK_PLATFORM_WIN64
static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT("Software\\Havok\\hkFilters_x64");
#else
static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT("Software\\Havok\\hkFilters");
#endif

#define  XREF_MERGE_MODIFIERS   (1<<2)

hctMaxSceneExportOptions::hctMaxSceneExportOptions()
{
    Interface* iface = GetCOREInterface();
    TimeValue now = iface->GetTime();

    m_batchMode = false;
    m_optionsFile = "";
    m_visibleOnly = false;
    m_selectedOnly = false;
    m_expandSelectionToChildren = false; // Vision implicitly expands selection to children nodes (prefab export of groups for instance)
    m_expandSelectionToParents = true;   // so that world transform of selected child will be correct. Dont want this for partial anims though
    m_expandSelectionToSkeleton = false; // Backwards compatible with Havok Export
    m_animationStart = now;
    m_animationEnd = now;
    m_exportMeshes = true;
    m_exportMaterials = true;
    m_exportAttributes = true;
    m_exportAnnotations = true;
    m_exportLights = true;
    m_exportCameras = true;
    m_exportSplines = true;
    m_exportVertexTangents = false; // false just to keep back compat with older Havok exports, but can be set to true. Vision requires it to be true.
    m_exportVertexAnimations = false;
    m_respectModifierSkinPose = false; // false just to keep back compat with older Havok exports, but can be set to true. Vision requires it to be true.
    m_autoSkinAttachments = false; // false  just to keep back compat with older Havok exports, but can be set to true. Vision requires it to be true.
    m_exportWireColorsAsVertexColors = false; // false just to keep back compat with older Havok exports, but can be set to true. Vision requires it to be true.
    m_exportMergeVertsOnlyIfSameSrcId = false; // false will increase amount of welding that is done. For Vision it is prefered that only verts with same src id are able to be merged
    m_addDefaultCamera = true;
    m_doNotSplitVertices = false; // We want to split based on texture, normals etc. This used to default true, but that dates back to Physics only export.
    m_environmentVariables = "";
    m_storeKeyframeSamplePoints = false;
}


hctMaxSceneExporter::hctMaxSceneExporter (const hctMaxSceneExportOptions& exportOptions, hctFilterSetOptions& filterSetOptions)
    : m_exportOptions (exportOptions),
    m_filterSetOptions(filterSetOptions),
    m_filters(),
    m_configurationSetIndex(-1),
    m_currentRootContainer(HK_NULL),
    m_meshWriter(exportOptions),
    m_totalNumNodes(0)
{
    m_meshWriter.setOwner( this );

    Interface* iface = GetCOREInterface();

    // EXP-1276 : Hack, where we force an evaluation of the scene by moving the time forward and back
    TimeValue now = iface->GetTime();
    iface->SetTime(now+GetTicksPerFrame()); // move one frame forward
    iface->SetTime(now);

    HWND ownerWnd = iface->GetMAXHWnd();
    m_filters.setOwnerHandle( ownerWnd );
}

hctMaxSceneExporter::~hctMaxSceneExporter()
{
}


static void __findAnimatedNodesRecursive( INode* inode, hkArray<INode*>& animatedNodes )
{
    for(int i=0;i<inode->NumberOfChildren();i++)
    {
        INode* child = inode->GetChildNode(i);

        // we deal with targets in the light/camera section (not a node in our graph)
        if( child->IsTarget() )
        {
            continue;
        }

        __findAnimatedNodesRecursive( child, animatedNodes );
    }

    Control* controller = inode->GetTMController();
    if (controller)
    {
        animatedNodes.pushBack( inode );
    }
}

float hctMaxSceneExporter::getSceneLength()
{
    int tpf = GetTicksPerFrame();
    int fps = GetFrameRate();
    int tickWholeDuration = m_exportOptions.m_animationEnd - m_exportOptions.m_animationStart;
    float frames = tickWholeDuration / (float)tpf;
    float framesLenInSec = frames / (float)fps;
    return framesLenInSec;
}

int hctMaxSceneExporter::getNumFrames()
{
    int tpf = GetTicksPerFrame();
    int tickWholeDuration = m_exportOptions.m_animationEnd - m_exportOptions.m_animationStart;
    float frames = tickWholeDuration / (float)tpf;
    return hkMath::ceil(frames); // round up partial frame?
}

void hctMaxSceneExporter::addAnnotations( INode* iNode, hkxNode* hknode )
{
    // two arrays to hold the annotation data
    hkArray<hkStringOld> noteTrackTexts;
    hkArray<hkReal> noteTrackTimes;
  bool bFoundTMCEvent = false;
  bool bFoundObjectEvent = false;

    // get access to the Max node
    Control* maxTMControl = iNode->GetTMController();


    if( maxTMControl )
    {

    // check for note tracks
    const int tracks = maxTMControl->NumNoteTracks();

    if( tracks )
    {
      for( int j = 0; j < tracks; j++ )
      {
        NoteTrack* ntrack = maxTMControl->GetNoteTrack(j);

        if ((ntrack == NULL) || (ntrack->ClassID() != Class_ID(NOTETRACK_CLASS_ID,0)))
        {
          continue;
        }

        DefNoteTrack* noteTrack = static_cast<DefNoteTrack*> (ntrack);

        const int numKeys = noteTrack->NumKeys();

        HCT_SCOPED_CONVERSIONS;

        for( int k = 0; k < numKeys; k++ )
        {
          NoteKey *noteKey = noteTrack->keys[k];
          if( !noteKey )
          {
            continue;
          }

          noteTrackTexts.pushBack( FROM_MAX(noteKey->note) );

          TimeValue noteTime    = noteKey->time - m_exportOptions.m_animationStart;
          noteTrackTimes.pushBack( float(noteTime) );
          bFoundTMCEvent = true;

        }
      }
    }

    // check for sub anims
    const int numSubAnims = maxTMControl->NumSubs();

    for( int i = 0; i < numSubAnims; i++ )
    {
      Animatable *anim = maxTMControl->SubAnim(i);

      if( anim )
      {
        const int numNoteTracks = anim->NumNoteTracks();
        for( int j = 0; j < numNoteTracks; j++ )
        {
          DefNoteTrack* noteTrack = (DefNoteTrack*)anim->GetNoteTrack(j);

          if( !noteTrack )
          {
            continue;
          }

          const int numKeys = noteTrack->NumKeys();

          HCT_SCOPED_CONVERSIONS;

          for( int k = 0; k < numKeys; k++ )
          {
            NoteKey *noteKey = noteTrack->keys[k];
            if( !noteKey )
            {
              continue;
            }

            noteTrackTexts.pushBack( FROM_MAX(noteKey->note) );

            TimeValue noteTime  = noteKey->time - m_exportOptions.m_animationStart;
            noteTrackTimes.pushBack( float(noteTime) );
            bFoundTMCEvent = true;
          }
        }
      }
    }
  }

  //Search for Vision events
  const int iVisionEventTracks = iNode->NumNoteTracks();

  if( iVisionEventTracks )
  {
    for( int j = 0; j < iVisionEventTracks; j++ )
    {
      NoteTrack* ntrack = iNode->GetNoteTrack(j);
      if ((ntrack == NULL) || (ntrack->ClassID() != Class_ID(NOTETRACK_CLASS_ID,0)))
      {
        continue;
      }

      DefNoteTrack* noteTrack = static_cast<DefNoteTrack*> (ntrack);
      const int numKeys = noteTrack->NumKeys();

      HCT_SCOPED_CONVERSIONS;

      for( int k = 0; k < numKeys; k++ )
      {
        NoteKey *noteKey = noteTrack->keys[k];
        if( !noteKey )
        {
          continue;
        }

        noteTrackTexts.pushBack( FROM_MAX(noteKey->note) );
        TimeValue noteTime  = noteKey->time - m_exportOptions.m_animationStart;
        noteTrackTimes.pushBack( float(noteTime) );
        bFoundObjectEvent = true;
      }
    }
  }

    int numTracks = noteTrackTexts.getSize();
    if( numTracks )
    {
        // allocate enough space
        hknode->m_annotations.setSize(numTracks);

        // max ticks per second
        const hkReal maxTicks = GetTicksPerFrame() * GetFrameRate();

        for( int i = 0; i < numTracks; i++ )
        {
            hkxNode::AnnotationData& desc = hknode->m_annotations[i];
            // time stamp
            desc.m_time = noteTrackTimes[i] / maxTicks;
            // description
            desc.m_description = noteTrackTexts[i].cString();
        }
    }

  if(bFoundObjectEvent && bFoundTMCEvent)
  {
    //Warning about mix usage of animation events of Vision and Havok Animation
    HK_WARN_ALWAYS(0xabba78ab,"Note tracks discovered for object and transform controller, please make sure only to use one of them, as appropriate for you usage scenario(Vision Animation/Havok Animation)!");
  }
}

// Calculates the node world TM
Matrix3 hctMaxSceneExporter::calculateNodeWorldTM (const INode* constinode, TimeValue time)
{
    // Max SDK is not great using "const"
    INode* inode = const_cast<INode*> (constinode);

    // Stretch has already been applied to the nodeTM so no need to multiply the nodeTM by the stretchTM as this will
    // produce incorrect results
    const Matrix3 nodeTM = inode->GetNodeTM(time);

    return nodeTM;
}

// Calculates the node local TM
Matrix3 hctMaxSceneExporter::calculateNodeLocalTM (const INode* constinode, TimeValue time)
{
    // Max SDK is not great using "const"
    INode* inode = const_cast<INode*> (constinode);
    const Matrix3 nodeToWorld = calculateNodeWorldTM (inode, time);

    const INode* parentNode = inode->GetParentNode();

    const Matrix3 parentToWorld = parentNode ? calculateNodeWorldTM(parentNode, time) : Matrix3(1);

    // local = nodeToParent = nodeToWorld * inv (parentToWorld)
    const Matrix3 invParentToWorld = Inverse(parentToWorld);
    const Matrix3 localTM = nodeToWorld * invParentToWorld;

    return localTM;
}

int _recursiveCount (INode* inode)
{
    int total = 1;
    for(int i=0;i<inode->NumberOfChildren();i++)
    {
        INode* child = inode->GetChildNode(i);
        total += _recursiveCount(child);
    }

    return total;
}

void hctMaxSceneExporter::countNumberOfNodesInScene()
{
    m_totalNumNodes = _recursiveCount(GetCOREInterface()->GetRootNode());
}

// Better (faster) to query all nodes at the same time instant, advance time, and query again and so on.
void hctMaxSceneExporter::precomputeAnimation(bool updateProgress)
{
    Interface* ip = GetCOREInterface();
    m_animatedNodes.reserve(128);
    m_animatedNodes.setSize(0);
    int i;

    INode* rootNode = ip->GetRootNode();

    __findAnimatedNodesRecursive( rootNode, m_animatedNodes );

    int tpf = GetTicksPerFrame();
    int tickWholeDuration = m_exportOptions.m_animationEnd - m_exportOptions.m_animationStart;
    int numFrames = (int)(tickWholeDuration / (float)tpf) + 1; // frame range -> num frames (inc each p_end)

    int numNodes = m_animatedNodes.getSize();
    m_animatedNodeTracks.setSize(numNodes);
    for (i=0; i < numNodes; ++i)
    {
        m_animatedNodeTracks[i].m_keyFrames.setSize(numFrames);
    }

    // Keys, all at the same frame
    for (int f = 0; f < numFrames; ++f)
    {
        MSTR tempT; tempT.printf(TEXT("Preprocessing Frame %d"), f);

        if (updateProgress)
        {
            ip->ProgressUpdate( int((f/(float)(numFrames-1))*100), FALSE, tempT );
        }
        TimeValue curT = m_exportOptions.m_animationStart + (f*tpf);

        for (i=0; i < numNodes; ++i)
        {
            INode* inode = m_animatedNodes[i];

            Matrix3 localTM = calculateNodeLocalTM (inode, curT);

            hctMaxUtils::convertToMatrix4( localTM, m_animatedNodeTracks[i].m_keyFrames[f] );
        }
    }

    // Analyse the keys for redundancy
    for (int t=0; t < m_animatedNodeTracks.getSize(); t++)
    {
        bool redundant = true;
        for (int k=1; k < m_animatedNodeTracks[t].m_keyFrames.getSize(); k++)
        {
            if ( m_animatedNodeTracks[t].m_keyFrames[k].isApproximatelyEqual( m_animatedNodeTracks[t].m_keyFrames[0], hkSimdReal::fromFloat(1e-3f) ) == false)
            {
                redundant = false;
            }
        }
        if (redundant && (m_animatedNodeTracks[t].m_keyFrames.getSize() > 2))
        {
            m_animatedNodeTracks[t].m_keyFrames.setSize(2);
        }
    }
}

static Object* _findBaseObject (INode* inode)
{
    // Look for modifiers and base objects to walk
    Object* baseObject = inode->GetObjectRef();
    bool finished = false;
    while (!finished)
    {
        IDerivedObject* devObject = baseObject->SuperClassID() == GEN_DERIVOB_CLASS_ID? (IDerivedObject*) baseObject : NULL;
        if (devObject) // has modifiers
        {
            baseObject = devObject->GetObjRef(); // drop down one level the stack
        }
        else // p_end of the line, the base object
        {
            finished = true;
        }
    }

    return baseObject;
}

// Standard bones : we check the base object
bool hctMaxSceneExporter::isBone (INode* inode) const
{
    // EXP-1667 - Detect other bones, not just max standard bones.
    if (inode->GetBoneNodeOnOff()) return true;

    Object* obj = _findBaseObject(inode);

    Class_ID objClassID = obj->ClassID();

    if (objClassID == BONE_OBJ_CLASSID)
    {
        return true;
    }

    return false;
}

// Biped bones : we check the TM controller
bool hctMaxSceneExporter::isBipedBone (INode* inode) const
{
    Control* tmController = inode->GetTMController();

    if (tmController)
    {
        Class_ID controllerClassId = tmController->ClassID();

        if ( (controllerClassId == BIPSLAVE_CONTROL_CLASS_ID) || (controllerClassId == BIPBODY_CONTROL_CLASS_ID) )
        {
            return true;
        }

    }

    return false;
}

// Footsteps : we check the TM controller
bool hctMaxSceneExporter::isFootSteps (INode* inode) const
{
    Control* tmController = inode->GetTMController();
    if (tmController)
    {
        Class_ID controllerClassId = tmController->ClassID();

        if (controllerClassId == FOOTPRINT_CLASS_ID)
        {
            return true;
        }
    }
    return false;
}

// Light : we check the base object
bool hctMaxSceneExporter::isLight (INode* inode) const
{
    Object* obj = _findBaseObject(inode);

    if (obj->SuperClassID() == LIGHT_CLASS_ID)
    {
        return true;
    }

    return false;
}

bool hctMaxSceneExporter::isSpline (INode* inode) const
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;

    if ( (obj->SuperClassID() == SHAPE_CLASS_ID ) && (obj->CanConvertToType(Class_ID(SPLINESHAPE_CLASS_ID,0)) ))
    {
        return true;
    }

    return false;

}

// Camera : we check the base object
bool hctMaxSceneExporter::isCamera (INode* inode) const
{
    Object* obj = _findBaseObject(inode);

    if (obj->SuperClassID() == CAMERA_CLASS_ID)
    {
        return true;
    }

    return false;
}

// Mesh : we evaluate the pipeline, check that is a geometric object that can be converted to triangles
bool hctMaxSceneExporter::isMesh (INode* inode) const
{
    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();
    Object *obj=inode->EvalWorldState(staticTick).obj;

    Class_ID objClassID = obj->ClassID();

    if ( (obj->SuperClassID() == GEOMOBJECT_CLASS_ID ) && (obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID,0)) ))
    {
        return true;
    }

    return false;
}

// Recursively finds if any children are selected
static bool _isAnySelectedRecursive(INode* inode)
{
    bool result = inode->Selected();
    if (result)
        return true;

    int numChild = inode->NumberOfChildren();
    if (numChild > 0)
    {
        for (int bi = 0; bi < inode->NumberOfChildren(); bi ++)
        {
            if (_isAnySelectedRecursive(inode->GetChildNode(bi)))
            {
                return true;
            }
        }
    }
    return false;
}

hkxNode* hctMaxSceneExporter::exportNode( INode* inode, bool parentTreeHasBones, bool ancestorWasSelected, bool updateProgress )
{
    // have to export children before exporting parent
    // so that we know how many children there are for the node (so that
    // we don't have to keep resizing the arrays)
    hkArray< hkRefPtr<hkxNode> > exportedChildren;
    bool currentNodeIsBone = hctMaxUtils::isBone(inode);
    bool isSelected = inode->Selected();

    // Children:
    for(int i=0;i<inode->NumberOfChildren();i++)
    {
        INode* child = inode->GetChildNode(i);

        // we deal with targets in the light/camera section
        if( child->IsTarget() )
        {
            continue;
        }

        hkxNode* childNode = exportNode( child, (parentTreeHasBones || currentNodeIsBone), (ancestorWasSelected || isSelected), updateProgress );
        if( childNode )
        {
            exportedChildren.pushBack( childNode );
            childNode->removeReference();
        }
    }

    m_curNodeIndex++;
    MSTR tempBuf = MSTR(TEXT("Processing: ")) + inode->GetName();
    Interface* ip = GetCOREInterface();

    if (updateProgress)
    {
        ip->ProgressUpdate((int)((float) m_curNodeIndex / m_totalNumNodes * 100.0f), FALSE, tempBuf);
    }

    // Exclude hidden nodes, if "visibleOnly".
    if( (exportedChildren.getSize() == 0) && m_exportOptions.m_visibleOnly && inode->IsHidden() )
    {
        return HK_NULL;
    }

    // If m_selectedOnly is true, export selected nodes only.
    bool exportAsParentSelected = ancestorWasSelected && m_exportOptions.m_expandSelectionToChildren;
    bool exportAsChildSelected = (exportedChildren.getSize() != 0) && m_exportOptions.m_expandSelectionToParents;
    bool exportAsSkeleton = m_exportOptions.m_expandSelectionToSkeleton && currentNodeIsBone;
    bool exportAsSelected = isSelected;
    if ( m_exportOptions.m_selectedOnly && (!exportAsSelected && !exportAsChildSelected && !exportAsParentSelected) )
    {
        if (exportAsSkeleton)
        {
            // get skeleton root node
            INode* sceneRoot = ip->GetRootNode();
            INode* rootBone = HK_NULL;
            INode* curBone = inode;
            while (curBone != sceneRoot)
            {
                if (hctMaxUtils::isBone(curBone))
                {
                    rootBone = curBone;
                }
                curBone = curBone->GetParentNode();
            }

            if (!_isAnySelectedRecursive(rootBone))
            {
                return HK_NULL;
            }
        }
        else
        {
            if ( exportedChildren.getSize() == 1)
            {
                exportedChildren[0]->addReference();
                return exportedChildren[0];
            }
            else if ( exportedChildren.getSize() > 1)
            {
                // Slightly dodgy: Have multiple children selected, but have not enabled m_expandSelectionToParents
                // Create a node to hold then nodes
                hkxNode* dummyNode = new hkxNode();
                dummyNode->m_selected = false;
                dummyNode->m_bone = false;
                dummyNode->m_name = "__DUMMY_SELECTION_CONTAINER";
                dummyNode->m_keyFrames.setSize(1);
                dummyNode->m_keyFrames[0].setIdentity();
                dummyNode->m_children.append(exportedChildren.begin(), exportedChildren.getSize());
                return dummyNode;
            }
            return HK_NULL;
        }
    }

    //int nodeId = node->GetNodeID();
    Matrix3 transform;

    // Create a node
    hkxNode* curNode = new hkxNode();

    // HKA-176
    // User properties
    {
        HCT_SCOPED_CONVERSIONS;

        MSTR buffer;
        inode->GetUserPropBuffer(buffer);
        curNode->m_userProperties = FROM_MAX(buffer);
    }

    // Selection status
    curNode->m_selected = isSelected;
    curNode->m_bone = currentNodeIsBone;

    // Object off the node
    if(!inode->IsGroupHead())
    {
        // Look for modifiers and base objects to walk
        hkArray<BaseObject*> objectsToWalk;
        {
            Object* curObject = inode->GetObjectRef();
            while (curObject)
            {
#if MAX_VERSION_MAJOR >= 8
                /// EXP-1600 - Support for XRefs (only in max sdk 8 and later)
                if (curObject->SuperClassID() == SYSTEM_CLASS_ID && curObject->ClassID() == XREFOBJ_CLASS_ID)
                {
                    IXRefObject8* ix = (IXRefObject8*)curObject;

                    Tab<Modifier*> modifiers;
                    ix->GetSourceObject(true, &modifiers);
                    for (int i=0; i < modifiers.Count(); i++)
                    {
                        if (modifiers[i]->IsEnabled())
                        {
                            objectsToWalk.pushBack(modifiers[i]);
                        }
                    }
                    curObject = HK_NULL;
                }
                else
#endif
                    if (curObject->SuperClassID() == GEN_DERIVOB_CLASS_ID)
                    {
                        IDerivedObject* devObject = (IDerivedObject*)curObject;

                        for (int m=devObject->NumModifiers()-1; m >=0 ; --m)
                        {
                            Modifier* mod = devObject->GetModifier(m);
                            if (mod->IsEnabled())
                            {
                                objectsToWalk.pushBack(mod);
                            }
                        }
                        curObject = devObject->GetObjRef(); // drop down one level the stack
                    }
                    else // p_end of the line, the base object
                    {
                        objectsToWalk.pushBack(curObject);
                        curObject = HK_NULL;
                    }
            }
        }

        if( m_exportOptions.m_exportAttributes )
        {
            // Walk those objects and fill attribute groups
            {
                // Do it backwards since the list of objects goes from top to bottom
                for (int i=objectsToWalk.getSize()-1; i>=0; --i)
                {
                    exportAttributes(objectsToWalk[i], curNode->m_attributeGroups);
                }
            }

            // Process and possibly merge attributes
            m_attributeProcessing.processAttributes(curNode);
        }
        else
        {
            curNode->m_attributeGroups.setSize(0);
        }

        if ( currentNodeIsBone ) //isBone (inode) || isBipedBone (inode) )
        {
            // Export node only
        }
        else if ( isLight (inode) )
        {
            if( !m_exportOptions.m_exportLights )
            {
                curNode->removeReference();
                return HK_NULL;
            }
            hkxLight* newLight = processLight( inode );
            curNode->m_object = newLight;
            if( newLight )
            {
                newLight->removeReference();
            }

        }
        else if ( isFootSteps(inode) )
        {
            // We ignore footsteps since that was IGame was doing. At some stage this should be optional
            // See EXP-509
            curNode->removeReference();
            return HK_NULL;
        }
        else if ( isCamera (inode) )
        {
            if( !m_exportOptions.m_exportCameras )
            {
                curNode->removeReference();
                return HK_NULL;
            }
            hkxCamera* newCamera = processCamera( inode );
            curNode->m_object = newCamera;
            if( newCamera )
            {
                newCamera->removeReference();
            }
        }
        else if ( isSpline (inode) && inode->Renderable() )
        {
            if ( !m_exportOptions.m_exportSplines)
            {
                curNode->removeReference();
                return HK_NULL;
            }
            hkxSpline* spline = processSpline ( inode );
            curNode->m_object = spline;
            if (spline)
            {
                spline->removeReference();
            }
        }
        else if ( isMesh (inode) && inode->Renderable() )
        {
            if( !m_exportOptions.m_exportMeshes )
            {
                curNode->removeReference();
                return HK_NULL;
            }
            processPolyMesh( inode, curNode->m_object, parentTreeHasBones && m_exportOptions.m_autoSkinAttachments );
        }
    }

    TimeValue staticTick = m_iStaticFrame * GetTicksPerFrame ();
    transform = calculateNodeLocalTM (inode, staticTick);

    // Name:
    HCT_SCOPED_CONVERSIONS;

    curNode->m_name = FROM_MAX(inode->GetName());

    //bool sampledTM = false;
    int ni = m_animatedNodes.indexOf(inode);
    if ( (ni>=0) && (m_animatedNodeTracks[ni].m_keyFrames.getSize() > 1) )
    {
        // can use the memory directly from the anim array (as long as it
        // is not deleted too soon)
        curNode->m_keyFrames.insertAt(0, m_animatedNodeTracks[ni].m_keyFrames.begin(), m_animatedNodeTracks[ni].m_keyFrames.getSize() );

        if (m_exportOptions.m_storeKeyframeSamplePoints && m_animatedNodeTracks[ni].m_keyFrames.getSize() > 2) // 2 frames get exported if totally redudant track
        {
            // fill keyframe times array (for Visions simple linear blending)
            hkArray<int> sampleTicks;
            hctMaxUtils::getControllerKeyTimes( inode, sampleTicks, m_exportOptions.m_animationStart / GetTicksPerFrame(), m_exportOptions.m_animationEnd / GetTicksPerFrame() );
            if (sampleTicks.getSize() > 0)
            {
                const int numFrames = m_animatedNodeTracks[ni].m_keyFrames.getSize();
                const int numKeys = sampleTicks.getSize();
                for (int ki =0; ki < numKeys; ++ki)
                {
                    int keyFrameIndex = (sampleTicks[ki] - m_exportOptions.m_animationStart) / GetTicksPerFrame();
                    float keyFrameTime = float(sampleTicks[ki] - m_exportOptions.m_animationStart) / TIME_TICKSPERSEC;
                    if (keyFrameIndex >= numFrames)
                        break;
                    if (curNode->m_linearKeyFrameHints.indexOf(keyFrameTime) < 0) // need to check if have it due to sub fram timer accuracy so can get more than one in one frame (just one per tick though as getKeyTimes() sorts etc)
                        curNode->m_linearKeyFrameHints.pushBack(keyFrameTime);
                }
            }
        }




    }
    else // place one key in, the current local transform
    {
        curNode->m_keyFrames.setSize(1);
        hctMaxUtils::convertToMatrix4( transform, curNode->m_keyFrames[0] );
    }

    // check and write if the node has any Note Tracks associated with this node
    if( m_exportOptions.m_exportAnnotations )
    {
        addAnnotations( inode, curNode );
    }
    else
    {
        curNode->m_annotations.setSize(0);
    }

    // setup the children ptr array
    curNode->m_children.setSize(0);
    if (exportedChildren.getSize() > 0)
    {
        curNode->m_children.append(exportedChildren.begin(), exportedChildren.getSize());
    }

    // Store the node in the node map for skin lookup.
    if( inode )
    {
        m_nodeMap.insert( inode, curNode );
    }

    return curNode;
}


void hctMaxSceneExporter::exportNodeSelectionSets()
{
    // find all object level named selection sets
#if (MAX_VERSION_MAJOR < 14)
    Interface* ip = GetCOREInterface();
#else
    INamedSelectionSetManager* ip = static_cast<INamedSelectionSetManager*>(GetCOREInterface(IID_NAMED_SELECTION_SET_MANAGER));
#endif


    int numNamedSets = ip->GetNumNamedSelSets();


    for(int setNum = 0; setNum < numNamedSets; ++setNum)
    {
        HCT_SCOPED_CONVERSIONS;

        const MCHAR *setName = ip->GetNamedSelSetName(setNum);
        int numItems   = ip->GetNamedSelSetItemCount(setNum);

        // create a selection set
        hkxNodeSelectionSet* selectionSet = new hkxNodeSelectionSet();

        // set the name of the selection set
        selectionSet->m_name = FROM_MAX(setName);

        // find nodes in the selection set
        hkArray< hkRefPtr<hkxNode> > nodesInSet;
        {
            for (int item=0; item<numItems; ++item)
            {
                INode* maxNode = ip->GetNamedSelSetItem(setNum, item);

                // corresponding hkxNode
                hkxNode* node = m_nodeMap.getWithDefault( maxNode, HK_NULL );
                if (!node) continue;

                nodesInSet.pushBack( node );
            }
        }

        // Allocate and assign nodes in the selection set if necessary
        // EXP-1477
        if( nodesInSet.getSize() > 0 )
        {
            // Allocate nodes
            selectionSet->m_selectedNodes.append(nodesInSet.begin(), nodesInSet.getSize() );
        }

        m_currentScene.m_selectionSets.pushBack(selectionSet);
        selectionSet->removeReference();
    }
}

void hctMaxSceneExporter::exportAll(bool updateProgress)
{
    // Make sure our scene helper stuct is clean
    m_currentScene.clear();

    // find all children
    hkArray< hkRefPtr<hkxNode> > exportedChildren;
    INode* rootINode = GetCOREInterface()->GetRootNode();

    for(int i = 0; i < rootINode->NumberOfChildren(); i++)
    {
        INode* childINode = rootINode->GetChildNode(i);
        if ( childINode->IsTarget() )
        {
            continue;
        }

        hkxNode* childNode = exportNode( childINode, false, false, updateProgress );

        if (childNode)
        {
            exportedChildren.pushBack( childNode );
            childNode->removeReference();
        }
    }

    // object level named selection sets
    exportNodeSelectionSets();

    // EXP-69 : Default camera
    if (m_exportOptions.m_addDefaultCamera) // if "export default camera"
    {
        if (m_currentScene.m_cameras.getSize()==0)
        {
            hkxCamera* defaultCamera = createDefaultCamera();

            if (defaultCamera)
            {
                // Create a fake node to hold the camera
                hkxNode* defaultCameraNode = new hkxNode();

                defaultCameraNode->m_name = "defaultCamera";

                // one key, identity
                defaultCameraNode->m_keyFrames.setSize(1);
                defaultCameraNode->m_keyFrames[0].setIdentity();

                defaultCameraNode->m_object = defaultCamera;

                exportedChildren.pushBack(defaultCameraNode);
                defaultCameraNode->removeReference();

                m_currentScene.m_cameras.pushBack(defaultCamera);
                defaultCamera->removeReference();
            }
        }
    }

    // Children
    if (exportedChildren.getSize() > 0)
    {
        m_currentScene.m_rootNode = new hkxNode();

        if (exportedChildren.getSize() > 0)
        {
            m_currentScene.m_rootNode->m_children.append( exportedChildren.begin(), exportedChildren.getSize() );
        }

        // one key, the identity
        m_currentScene.m_rootNode->m_keyFrames.setSize(1);
        m_currentScene.m_rootNode->m_keyFrames[0].setIdentity();

        m_currentScene.m_rootNode->m_name = "ROOT_NODE";
        m_currentScene.m_rootNode->removeReference(); // its a refptr
    }

}

// Dummy function for progress bar
DWORD WINAPI fn(LPVOID arg)
{
    return(0);
}

bool hctMaxSceneExporter::exportAndProcess()
{
#if !defined (HCT_NO_FILTERS)
    hkStringOld filterManagerPath;
    hctMaxUtils::getFilterManagerPath( filterManagerPath );

    // Try to load the filter manager.
    m_filters.load( filterManagerPath.cString() );

    // Load options to use with the asset.
    if ( !loadFilterSetOptions() )
    {
        m_filters.unload();
        return false;
    }
#endif
    HK_REPORT_SECTION_BEGIN(0xabba934, "3ds Max Scene Export");

    // EXP-969
    if (m_exportOptions.m_selectedOnly)
    {
        Log_Info( "*Exporting Selected Only*" );
    }
    if (m_exportOptions.m_visibleOnly)
    {
        Log_Info( "*Exporting Visible Only*" );
    }

    createScene();

    HK_REPORT_SECTION_END();

    // Do filtering
    bool okVal = m_currentRootContainer != HK_NULL;
    if (m_currentRootContainer)
    {
#if !defined (HCT_NO_FILTERS)
        // Assumes that we have set the correct error handler here..
        m_filters.mergeErrors( (hctSceneExportError*)&hkError::getInstance() );

        if( m_exportOptions.m_batchMode )
        {
            hkBool saveConfig;
            m_filters.processBatch( *m_currentRootContainer, saveConfig, m_configurationSetIndex );
            if( saveConfig )
            {
                saveFilterSetOptions(); //EXP-1377 (we may have upgraded them)
            }
            // TODO: give a message
        }
        else
        {
            hkBool saveConfig = true;
            m_filters.openFilterManager( *m_currentRootContainer, saveConfig);
            if( saveConfig )
            {
                saveFilterSetOptions();
            }
        }

#else
        Interface* ip = GetCOREInterface();

        HCT_SCOPED_CONVERSIONS;

        hkStringBuf assetFileName= FROM_MAX(ip->GetCurFileName());
        hkStringBuf assetFilePath= FROM_MAX(ip->GetCurFilePath());
        wchar_t filenameBuffer[1024];
        if( (assetFilePath.getLength() == 0) || (assetFileName.getLength() == 0) )
        {
            assetFileName = "untitled.max";
            _wgetcwd(filenameBuffer, 1024);
            hkUtf8::Utf8FromWide wf(filenameBuffer);
            assetFilePath.setJoin(wf.cString(), "\\", assetFileName.cString(), ".max" );
        }


        bool saveFile = true;
        if (!m_exportOptions.m_batchMode)
        {
            assetFilePath.replace(".max",".hkt");
            assetFileName.replace(".max",".hkt");
            if (!assetFilePath.endsWithCase(".hkt"))
            {
                assetFilePath.append(".hkt");
                assetFileName.append(".hkt");
            }

            OPENFILENAMEW op;
            hkString::memSet( &op, 0, sizeof(OPENFILENAMEW) );
            op.lStructSize = sizeof(OPENFILENAMEW);

            op.hwndOwner = ip->GetMAXHWnd();
            op.lpstrFilter = L"Havok Files (*.hkx;*.hkt;*.xml)\0*.hkx;*.hkt;*.xml;\0Binary Packfiles (*.hkx)\0*.hkx\0Tagfiles (*.hkt)\0*.hkt\0XML Packfiles (*.xml)\0*.xml\0All Files\0*.*\0\0";
            op.lpstrDefExt = L"hkt";
            op.nFilterIndex = 3;

            hkUtf8::WideFromUtf8 assetFileNameWide(assetFileName.cString());
            hkUtf8::WideFromUtf8 assetFilePathWide(assetFilePath.cString());

            op.lpstrInitialDir = assetFilePathWide.cString();
            wcscpy_s( filenameBuffer, assetFileNameWide.cString() );

            op.lpstrFile = filenameBuffer;

            op.nMaxFile = 1023;
            op.Flags = OFN_ENABLESIZING | OFN_EXPLORER | OFN_NOREADONLYRETURN | OFN_PATHMUSTEXIST;
            if ( GetSaveFileNameW(&op) )
            {
                hkUtf8::Utf8FromWide utf8File( op.lpstrFile );
                assetFilePath = utf8File.cString();
            }
            else
            {
                saveFile = false;
            }
        }
        else
        {
            //XX assume tools batch mode. ADD SCRIPT OPTION FOR FILE NAME
            assetFilePath.append(".hktAuto"); // so fullname.mb.hktAuto  so to not conflict with max ver etc of twp of same asset etc
        }

        if (saveFile)
        {
            assetFilePath.replace( '/', '\\' );
            hkSerializeUtil::save(m_currentRootContainer, assetFilePath.cString());
        }
#endif
    }


    cleanupScene();

    return okVal;
}

bool hctMaxSceneExporter::getFileDependencies(hkArray<hkStringPtr>& fileDependencies)
{
#if !defined (HCT_NO_FILTERS)
    hkStringOld filterManagerPath;
    hctMaxUtils::getFilterManagerPath(filterManagerPath);

    // Try to load the filter manager.
    m_filters.load(filterManagerPath.cString());

    // Load options to use with the asset.
    if (!loadFilterSetOptions())
    {
        m_filters.unload();
        return false;
    }
#endif

    // Create an empty scene with environment variables set up instead of createScene()
    {
        bool callStartEndProgress = true;
        bool updateProgress = true;

        Interface* ip = GetCOREInterface();

        //
        // Do export.
        //


        m_currentRootContainer = new hkRootLevelContainer();
        m_currentRootContainer->m_namedVariants.setSize(3);

        hkRootLevelContainer::NamedVariant& environmentVariant = m_currentRootContainer->m_namedVariants[0];
        hkRootLevelContainer::NamedVariant& sceneVariant = m_currentRootContainer->m_namedVariants[1];
        hkRootLevelContainer::NamedVariant& registeredClassNodesVariant = m_currentRootContainer->m_namedVariants[2];

        m_curNodeIndex = 0;

        // HKD-514: Because of what seems to be a bug in 3ds Max, we need to be able to skip the
        //          explicit ProgressStart/End call (in case the calling function has already
        //          started the progress bar) to avoid the floating toolbars/windows from disappearing
        //          permanently.
        if (callStartEndProgress && updateProgress)
        {
            ip->ProgressStart(_T("Exporting Scene.."), TRUE, fn, NULL);
        }

        if (updateProgress)
        {
            ip->ProgressUpdate(100, FALSE, _T("Invoking filter manager.."));
        }

        // Create environment
        hkxEnvironment* envPtr = new hkxEnvironment();
        hkxEnvironment& environment = *envPtr;

        hkxScene* scenePtr = new hkxScene();
        hkxScene& scene = *scenePtr;

        hkMemoryResourceContainer* resourceContainerPtr = new hkMemoryResourceContainer();
        hkMemoryResourceContainer& resourceContainer = *resourceContainerPtr;

        environmentVariant.set("Environment Data", &environment);
        envPtr->removeReference();
        sceneVariant.set("Scene Data", &scene);
        scenePtr->removeReference();
        registeredClassNodesVariant.set("Resource Data", &resourceContainer);
        resourceContainerPtr->removeReference();

        hkStringBuf modellerString;
        modellerString.printf("3ds max %d.%d.%d", MAX_VERSION_MAJOR, MAX_VERSION_MINOR, MAX_VERSION_POINT);
        scene.m_modeller = modellerString.cString();

        // Asset name.
        HCT_SCOPED_CONVERSIONS;

        scene.m_asset = FROM_MAX(ip->GetCurFilePath());
        if (scene.m_asset.getLength() == 0)
        {
            scene.m_asset = "untitled";
        }

        // Fill environment variables
        {
            // Copy asset & modeler variables from the scene
            hkxSceneUtils::fillEnvironmentFromScene(scene, environment);
        }

        if (callStartEndProgress && updateProgress)
        {
            ip->ProgressEnd();
        }
    }

    // Do filtering
    bool okVal = m_currentRootContainer != HK_NULL;
    if (m_currentRootContainer)
    {
#if !defined (HCT_NO_FILTERS)
        // Assumes that we have set the correct error handler here..
        m_filters.mergeErrors((hctSceneExportError*)&hkError::getInstance());

        if (m_exportOptions.m_batchMode)
        {
            m_filters.getFileDependencies(*m_currentRootContainer, fileDependencies, m_configurationSetIndex);
        }
#endif
    }

    cleanupScene();

    return okVal;
}

bool hctMaxSceneExporter::createScene(bool callStartEndProgress, bool updateProgress)
{

    Interface* ip = GetCOREInterface();

    //
    // Do export.
    //


    m_currentRootContainer = new hkRootLevelContainer();
    m_currentRootContainer->m_namedVariants.setSize(3);

    hkRootLevelContainer::NamedVariant& environmentVariant = m_currentRootContainer->m_namedVariants[0];
    hkRootLevelContainer::NamedVariant& sceneVariant = m_currentRootContainer->m_namedVariants[1];
    hkRootLevelContainer::NamedVariant& registeredClassNodesVariant = m_currentRootContainer->m_namedVariants[2];

    m_curNodeIndex = 0;

    // Try to load the attribute type info for this export.
    // Loaded each export so that it can be edited easily.
    // Init the attribute database
    hkStringOld filterManagerPath;
    bool filtersOk = hctMaxUtils::getFilterManagerPath( filterManagerPath );
    hkStringOld attrDescsRoot = filterManagerPath + "\\attributeProcessing";
    hkStringOld attrSelsRoot = filterManagerPath + "\\attributeSelection";
    {
        m_attributeProcessing.init( );

        // Load common attribute descriptions
        m_attributeProcessing.loadAttributeDescriptions( attrDescsRoot.cString() );

        // Load max-specific attribute descriptions
        attrDescsRoot += "\\max";
        m_attributeProcessing.loadAttributeDescriptions( attrDescsRoot.cString() );

        // Only complain about no Attribs of there was a FilterManager etc (and so proper tools install).
        if (m_attributeProcessing.isEmpty() && filtersOk)
        {
            HK_WARN_ALWAYS(0x5711ce79, "Couldn't find any attribute description .xml file(s)." );
            return false;
        }
    }

    {
        m_attributeSelection.init( );

        // Load max-specific attribute selections
        attrSelsRoot += "\\max";
        m_attributeSelection.loadAttributeSelections( attrSelsRoot.cString() );

    }

    // HKD-514: Because of what seems to be a bug in 3ds Max, we need to be able to skip the
    //          explicit ProgressStart/End call (in case the calling function has already
    //          started the progress bar) to avoid the floating toolbars/windows from disappearing
    //          permanently.
    if ( callStartEndProgress && updateProgress )
    {
        ip->ProgressStart(_T("Exporting Scene.."), TRUE, fn, NULL);
    }


    // Count the number of nodes so we can report progress
    countNumberOfNodesInScene();

    m_iStaticFrame = m_exportOptions.m_animationStart / GetTicksPerFrame();
    {
        // Clear the nodeId pointer map.
        m_nodeMap.clear();

        if (updateProgress)
        {
            ip->ProgressUpdate(0, FALSE, _T("Precomputing Animations.."));
        }

        if( m_exportOptions.m_animationEnd > m_exportOptions.m_animationStart )
        {
            // Precompute all of the animated nodes.
            precomputeAnimation(updateProgress);
        }
        else
        {
            m_animatedNodes.clear();
        }

        if (updateProgress)
        {
            ip->ProgressUpdate(0, FALSE, _T("Processing SceneGraph.."));
        }

        // Export the scene.
        exportAll(updateProgress);

        if( m_exportOptions.m_exportMeshes )
        {
            fixupSkins();
        }
        m_meshWriter.cleanup();
    }

    if (updateProgress)
    {
        ip->ProgressUpdate(100, FALSE, _T("Invoking filter manager.."));
    }

    // Create environment
    hkxEnvironment* envPtr = new hkxEnvironment();
    hkxEnvironment& environment = *envPtr;

    hkxScene* scenePtr = new hkxScene();
    hkxScene& scene = *scenePtr;

    hkMemoryResourceContainer* resourceContainerPtr = new hkMemoryResourceContainer();
    hkMemoryResourceContainer& resourceContainer = *resourceContainerPtr;

    environmentVariant.set("Environment Data", &environment);
    envPtr->removeReference();
    sceneVariant.set("Scene Data", &scene);
    scenePtr->removeReference();
    registeredClassNodesVariant.set("Resource Data", &resourceContainer);
    resourceContainerPtr->removeReference();

    hkStringBuf modellerString;
    modellerString.printf("3ds max %d.%d.%d", MAX_VERSION_MAJOR, MAX_VERSION_MINOR, MAX_VERSION_POINT);
    scene.m_modeller = modellerString.cString();

    // Asset name.
    HCT_SCOPED_CONVERSIONS;

    scene.m_asset = FROM_MAX(ip->GetCurFilePath());
    if( scene.m_asset.getLength() == 0 )
    {
        scene.m_asset = "untitled";
    }

    // Scene length.
    scene.m_sceneLength = getSceneLength();
    scene.m_numFrames = getNumFrames();

    // Root node.
    scene.m_rootNode = m_currentScene.m_rootNode;

    // Named selection sets.
    if (m_currentScene.m_selectionSets.getSize())
    {
        scene.m_selectionSets.append( m_currentScene.m_selectionSets.begin(), m_currentScene.m_selectionSets.getSize() );
    }

    // Cameras.
    if (m_currentScene.m_cameras.getSize())
    {
        scene.m_cameras.insertAt(0, m_currentScene.m_cameras.begin(), m_currentScene.m_cameras.getSize() );
    }

    // Lights.
    if (m_currentScene.m_lights.getSize())
    {
        scene.m_lights.insertAt(0, m_currentScene.m_lights.begin(), m_currentScene.m_lights.getSize() );
    }

    // Splines
    if (m_currentScene.m_splines.getSize())
    {
        scene.m_splines = m_currentScene.m_splines;
    }

    // If meshes aren't exported there'll be no skin bindings either.
    if( m_exportOptions.m_exportMeshes )
    {
        // Meshes.
        if (m_currentScene.m_meshes.getSize())
        {
            scene.m_meshes.insertAt(0, m_currentScene.m_meshes.begin(), m_currentScene.m_meshes.getSize() );
        }

        // Skin bindings.
        if (m_currentScene.m_skinBindings.getSize())
        {
            scene.m_skinBindings.insertAt(0, m_currentScene.m_skinBindings.begin(), m_currentScene.m_skinBindings.getSize() );
        }
    }

    // If materials aren't exported textures won't be either.
    if( m_exportOptions.m_exportMaterials )
    {
        // Materials.
        if (m_currentScene.m_materials.getSize())
        {
            scene.m_materials.insertAt(0, m_currentScene.m_materials.begin(), m_currentScene.m_materials.getSize() );
        }

        // External textures.
        if (m_currentScene.m_externalTextures.getSize())
        {
            scene.m_externalTextures.insertAt(0, m_currentScene.m_externalTextures.begin(), m_currentScene.m_externalTextures.getSize() );
        }
    }

    // Assign node UUIDs
    assignHkxSceneUUIDs(&scene);

    // Fill in the registered classes data.
    {
        createGenericObjectsFromMaxNodes(&resourceContainer);
    }

    // Fill environment variables
    {
        // Copy asset & modeler variables from the scene
        hkxSceneUtils::fillEnvironmentFromScene(scene, environment);

        // Then interpret user variables
        environment.interpretString(m_exportOptions.m_environmentVariables.cString());
    }


    // EXP-441 : Warn when nodes have the same name
    {
        // If the user defines "IGNORE_DUPLICATES", we don't check for duplicates
        if (environment.getVariableValue("ignore_duplicates")==HK_NULL)
        {
            checkForDuplicatedNames (scene);
        }
    }

    // EXP-867 : Reorder nodes by name
    {
        // If the user defined "do_not_reorder_nodes", we don't do so
        if ( (environment.getVariableValue("do_not_reorder_nodes")==HK_NULL)

            // check the Windows environment variables also (EXP-1418)
            && (::getenv("do_not_reorder_nodes")==HK_NULL) )
        {
            hkxSceneUtils::reorderNodesAlphabetically(scene);
        }
    }

    
#if defined (HCT_NO_FILTERS)
    // GetMasterScale(UNITS_METERS) == meters per system unit (e.g. 0.01 when SU==cm, 1 when SU==m)
    const hkFloat32 metersFromSystemUnits = static_cast<hkFloat32>(GetMasterScale(UNITS_METERS));
    const hkSimdReal systemUnitsFromMeters = hkSimdReal::fromFloat(1.0f / metersFromSystemUnits);

    // Emit maxFromHke transform in scene.m_appliedTransform (assumes HKE +X forward, +Z up, meter units)
    hkMatrix3 maxFromHke;
    maxFromHke.setCols(
        hkVector4(0.0f,-1.0f, 0.0f, 0.0f),
        hkVector4(1.0f, 0.0f, 0.0f, 0.0f),
        hkVector4(0.0f, 0.0f, 1.0f, 0.0f)
        );
    maxFromHke.mul(hkSimdReal::fromFloat(systemUnitsFromMeters));

    scene.m_appliedTransform = maxFromHke;
#endif

    // EXP-969
    hkSceneExportUtils::reportSceneData(&scene);

    if ( callStartEndProgress && updateProgress )
    {
        ip->ProgressEnd();
    }

    return TRUE;
}

void hctMaxSceneExporter::assignHkxSceneUUIDs(hkxScene* scene)
{
    hkArray<char> memBuffer;
    hkMap<hkUuid, int> knownUUIDs;

    hkxNode* rootNode = scene->m_rootNode;
    assignHkxNodeUUID(rootNode, memBuffer, knownUUIDs);
}

void hctMaxSceneExporter::assignHkxNodeUUID(hkxNode* node, hkArray<char>& memBuffer, hkMap<hkUuid, int>& knownUUIDs)
{
    if ( !node )
    {
        return;
    }

    // Assign UUIDs to all its children
    {
        hkArray< hkRefPtr<hkxNode> >& childNodes = node->m_children;

        for (int k = childNodes.getSize() - 1; k >= 0; k--)
        {
            hkxNode* childNode = childNodes[k];
            assignHkxNodeUUID(childNode, memBuffer, knownUUIDs);
        }
    }

    // Assign a UUID to this node
    {
        memBuffer.clear();

        // Get the contents of the node
        node->m_uuid = hkUuid::getNil();
        hkArrayStreamWriter memWriter(&memBuffer, hkArrayStreamWriter::ARRAY_BORROW);
        const hkResult ret = hkSerializeUtil::save<hkxNode>(node, &memWriter);

        if ( ret.isSuccess() )
        {
            // Got the contents, digest it into an UUID
            hkUuid uuid;
            int hashSize = 16;
            if ( hctSdkUtils::computeMd5Hash(memBuffer.begin(), memBuffer.getSize(), reinterpret_cast<hkUint8*>(&uuid), hashSize).isSuccess() )
            {
                HK_ASSERT_NO_MSG(0x7979b975, hashSize == 16);

                // Make sure the UUID is unique, i.e. does not collide with another in this file
                bool hadCollisions = false;
                for (hkMap<hkUuid, int>::Iterator it = knownUUIDs.findKey(uuid); knownUUIDs.isValid(it); it = knownUUIDs.findKey(uuid))
                {
                    uuid.setRandom();
                    hadCollisions = true;
                }

                // Found an unique UID!
                node->m_uuid = uuid;
                knownUUIDs.insert(uuid, 1);
                if ( hadCollisions )
                {
                    HK_WARN_ALWAYS(0xabbacddc, "The UUID of node " << node->m_name.cString() << " generated from its contents collides with another node. It will be randomly generated.");
                }
            }
        }
        else
        {
            HK_WARN_ALWAYS(0xabbacddc, "Failed to serialize the contents of node " << node->m_name.cString() << ". Its UUID will be randomly generated");
        }
    }
}

void hctMaxSceneExporter::cleanupScene()
{
    //
    // Cleanup
    //
    m_currentScene.clear();
    m_animatedNodes.setSize(0);
    m_animatedNodeTracks.setSize(0);

    if (m_currentRootContainer)
    {
        delete m_currentRootContainer;
        m_currentRootContainer = HK_NULL;
    }

    // Release the filter DLL.
    m_filters.unload();

}



static void _exploreNodeTree (const hkxNode* node, hkArray<const hkxNode*>& nodeList)
{
    if (!node) return;

    nodeList.pushBack(node);

    for (int i=0; i<node->m_children.getSize(); i++)
    {
        _exploreNodeTree (node->m_children[i], nodeList);
    }
}

void hctMaxSceneExporter::checkForDuplicatedNames(class hkxScene& scene)
{
    // Start by constructing an array of all nodes (depth first)
    hkArray<const hkxNode*> allNodes;
    _exploreNodeTree(scene.m_rootNode, allNodes);

    // Then construct an array of all names and how many times they've been used
    hkArray<hkStringOld> allNames;
    hkArray<int> uses;
    bool duplicates = false;

    hkStringOld nodeName;
    for (int i=0; i<allNodes.getSize(); ++i)
    {
        nodeName = allNodes[i]->m_name;
        const int idx = allNames.indexOf(nodeName);
        if (idx!=-1)
        {
            duplicates = true;
            ++uses[idx];
        }
        else
        {
            allNames.pushBack(nodeName);
            uses.pushBack(1);
        }
    }

    if (duplicates)
    {
        hkStringOld str( "Some nodes have duplicated names. This may cause some filters to get confused:\n" );
        for( int i=0; i<allNames.getSize(); ++i )
        {
            if( uses[i] > 1 )
            {
                hkStringOld text;
                text.printf( "  '%s'(x%d)", allNames[i].cString(), uses[i] );

                str += text;
            }
        }

        HK_WARN_ALWAYS( 0xabba8dc8, str.cString() );
    }
}

//
// Loading and saving options.
//
bool hctMaxSceneExporter::loadFilterSetOptions()
{
    // Try to load the options.
    char* optionData = HK_NULL;
    int optionSize = 0;

    // Buffer for reading the XML options from a file/the registry
    hkArray<char> readOptionsBuf;

    bool loadedFromFile = false;

    // Check if the user specified an options file to use.
    if ( m_exportOptions.m_optionsFile.getLength() > 0 )
    {
        if( hkLoadUtil(m_exportOptions.m_optionsFile.cString()).toArray(readOptionsBuf))
        {
            optionData = readOptionsBuf.begin();
            optionSize = readOptionsBuf.getSize();
            loadedFromFile = true;
        }
        else
        {
            Interface* ip = GetCOREInterface();

            // Name:
            HCT_SCOPED_CONVERSIONS;

            hkStringPtr filePath = FROM_MAX(ip->GetCurFilePath());
            hkStringBuf assetDir = filePath.cString();
            assetDir.pathDirname();

            hkStringBuf searchloc;
            searchloc.printf("%s/%s", assetDir.cString(), m_exportOptions.m_optionsFile.cString());

            if( hkLoadUtil(searchloc.cString()).toArray(readOptionsBuf))
            {
                optionData = readOptionsBuf.begin();
                optionSize = readOptionsBuf.getSize();
                loadedFromFile = true;
            }
            else
            {
                HK_ERROR(0x5dc34d98, "The specified .hko options file could not be found. Export cancelled." );
                return false;
            }
        }
    }

    // If we couldn't find the options file or none was specified check the asset itself, then the registry.
    if ( !loadedFromFile )
    {
        // Check if there are options stored with the asset.
        if ( m_filterSetOptions.m_data.Count() > 0 )
        {
            optionData = &m_filterSetOptions.m_data[0];
            optionSize = m_filterSetOptions.m_data.Count();
        }
        else
        {
            // If the asset hasn't any options check the registry.
            HKEY animReg;
            DWORD dispos;

            RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

            // Read the buffer size first
            DWORD bufSize = 128*1024;
            RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, NULL, &bufSize );
            readOptionsBuf.setSize(bufSize);

            if ( RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, (LPBYTE)readOptionsBuf.begin(), &bufSize ) == ERROR_SUCCESS )
            {
                optionData = readOptionsBuf.begin();
                optionSize = bufSize;
            }
            else
            {
                HK_WARN_ALWAYS(0x5dc34d98, "Could not read filter options from registry." );
            }

            RegCloseKey( animReg );
        }
    }

    // Load the options into the filter manager.
    m_filters.setOptions( optionData, optionSize );

    if (m_exportOptions.m_configurationSet != "")
    {
        hctFilterManagerImpl *manager = (hctFilterManagerImpl *)((hctFilterManagerInterface *)m_filters.getFilterManagerInterface());
        if (manager != NULL && manager->m_configurationSet != NULL)
        {
            for (int curConfig = 0; curConfig < manager->m_configurationSet->m_configurations.getSize(); ++curConfig)
            {
                const hctFilterConfigurationSet::Configuration& config = manager->m_configurationSet->m_configurations[curConfig];
                if (m_exportOptions.m_configurationSet == config.m_configName.cString())
                {
                    m_configurationSetIndex = curConfig;
                }
            }
        }
    }

    return true;
}

void hctMaxSceneExporter::saveFilterSetOptions()
{
    // Buffered so that we can see if we need to set the save flag or not.
    hkArray<char> options;

    int optionSize = m_filters.getOptionsSize();
    options.setSize(optionSize);

    if (optionSize > 0)
    {
        m_filters.getOptions( options.begin() );
    }

    // Check if there's no need to set the options (and thus stop the save flag being raised).
    // 'no need' == extact manager version match + size the same + mem the same.
    if ( (m_filterSetOptions.m_version != m_filters.getCurrentOptionsVersion()) ||
        ( options.getSize() != m_filterSetOptions.m_data.Count() ) ||
        ( hkString::memCmp( &m_filterSetOptions.m_data[0], options.begin(), optionSize ) !=0 ) )
    {
        m_filterSetOptions.m_version = m_filters.getCurrentOptionsVersion();
        m_filterSetOptions.m_data.SetCount( optionSize );
        if (optionSize > 0)
        {
            hkString::memCpy( &m_filterSetOptions.m_data[0], options.begin(), optionSize );
        }
        SetSaveRequiredFlag(TRUE);
    }

    // Save the option data to the registry.
    {
        HKEY animReg;
        DWORD dispos;

        if ( RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL,
            REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos ) == ERROR_SUCCESS )
        {
            RegSetValueEx( animReg, TEXT("FilterSetup"), 0, REG_BINARY, LPBYTE( options.begin() ), optionSize );
        }
        RegCloseKey( animReg );
    }
}


bool hctMaxSceneExporter::createGenericObjectsFromMaxNodes(hkMemoryResourceContainer* resourceContainer)
{
    INode* rootINode = GetCOREInterface()->GetRootNode();

    hkInplaceArray<INode*,10> nodePath; nodePath.pushBackUnchecked( rootINode );
    createGenericObjectsFromMaxNodesRecursive(nodePath, resourceContainer);

    {
        hkError::getInstance().setEnabled(0xaf12e114, false);

        hkContainerResourceMap map(resourceContainer) ;
        resourceContainer->tryToResolveLinks(map);

        hkError::getInstance().setEnabled(0xaf12e114, true);
    }

    return true;
}


void hctMaxSceneExporter::createGenericObjectsFromMaxNodesRecursive( hkArray<INode*>& nodePath, hkMemoryResourceContainer* resourceContainer)
{
    INode* maxNode = nodePath.back();

    // Early abort: no need to handle Targets.
    if ( maxNode->IsTarget() )
    {
        return;
    }

    // Do not export hidden nodes if "visibleOnly" option is set.
    if ( m_exportOptions.m_visibleOnly && maxNode->IsHidden() )
    {
        return;
    }

    // Recurse into children first. This can be either a true group created by the artist or an implicit group
    // like SceneRoot.
    {
        nodePath.pushBack(HK_NULL);
        for (int i = 0; i < maxNode->NumberOfChildren(); i++)
        {
            INode* childINode = maxNode->GetChildNode(i);
            nodePath[ nodePath.getSize()-1 ] = childINode;
            createGenericObjectsFromMaxNodesRecursive(nodePath, resourceContainer);
        }
        nodePath.popBack();
    }

    // Skip any group heads.
    if ( maxNode->IsGroupHead() )
    {
        return;
    }

    // Look for modifiers and base objects to walk
    hkArray<Modifier*> modifierList;
    {
        Object* curObject = maxNode->GetObjectRef();
        while ( curObject )
        {
            IDerivedObject* devObject = curObject->SuperClassID() == GEN_DERIVOB_CLASS_ID ? (IDerivedObject*)curObject : NULL;
            if ( devObject ) // has modifiers
            {
                for (int modifierIdx = devObject->NumModifiers()-1; modifierIdx >= 0; modifierIdx--)
                {
                    Modifier* modifierItem = devObject->GetModifier(modifierIdx);
                    if (modifierItem->IsEnabled() )
                    {
                        modifierList.pushBack(modifierItem);
                    }
                }
                curObject = devObject->GetObjRef(); // drop down one level the stack
            }
            else // p_end of the line, the base object
            {
                curObject = HK_NULL;
            }
        }
    }

    if ( m_exportOptions.m_exportAttributes )
    {
        // Go through all modifiers on this node's stack and try to create objects from them.
        {
            for (int i = 0; i < modifierList.getSize(); i++)
            {
                Modifier* modifierItem = modifierList[i];

                const hkReflect::Type* klass = getClassFromMaxModifier(modifierItem);
                if ( !klass || !hctSdkUtils::getTypeByName(klass->getName()) )
                {
                    continue;
                }

                hkReflect::Var object = createEmptyHavokObject(klass);

                hkResourceContainer* childContainer = resourceContainer;
                // find the final resource container
                {
                    for (int nodePathIndex = 1; nodePathIndex < nodePath.getSize(); nodePathIndex++)    // skip the root
                    {
                        HCT_SCOPED_CONVERSIONS;

                        INode* ancestor = nodePath[nodePathIndex];
                        const char* nodeName = FROM_MAX(ancestor->GetName());
                        childContainer = childContainer->createContainer( nodeName );
                    }
                }

                {
                    HCT_SCOPED_CONVERSIONS;

                    MSTR resourceName;
                    resourceName.printf(TEXT("%s:0x%x"), maxNode->GetName(), hkUlong(modifierItem));
                    hkResourceHandle* resourceHandle = childContainer->createResource(FROM_MAX(resourceName), object.getAddress(), object.getType());

                    convertMaxModifierToClassAndObject(modifierItem, resourceHandle, FROM_MAX(maxNode->GetName()), maxNode);

                    if ( hkReflect::StringVar sh = object["name"] )
                    {
                        sh.setValue( hkString::strDup(FROM_MAX(maxNode->GetName())) );
                    }
                }
            }
        }
    }
}



const hkReflect::Type* hctMaxSceneExporter::getClassFromMaxModifier(Modifier* modifierItem)
{
    HCT_SCOPED_CONVERSIONS;

    const hkArray<hctModelerNodeIdToClassMap>& classMap = hctSdkUtils::getNodeIdToClassMap();

    MSTR classNameStr;
    modifierItem->GetClassName(classNameStr);

    const char* className = FROM_MAX(classNameStr);

    // Search for the type in our mapping table.
    {
        for (int index = 0; index < classMap.getSize() && classMap[index].m_class != HK_NULL; index++ )
        {
            if ( hkString::strCasecmp(className, classMap[index].m_class->getName()) == 0 )
            {
                const hkReflect::Type* klass = classMap[index].m_class;
                return klass;
            }
        }
    }

    return HK_NULL;
}


hkReflect::Var hctMaxSceneExporter::createEmptyHavokObject(const hkReflect::Type* klass)
{
    
    // Keep compatibility with the old behavior. Use a zeroed buffer so that reference count is disabled and objects
    // are never deleted (that might call destructors which were never meant to be called).
    void* object = hkAllocate<char>(klass->getSizeOf(), HK_MEMORY_CLASS_EXPORT);
    hkString::memSet(object, 0x0, klass->getSizeOf());

    hkReflect::Var res(object, klass);
    hkReflect::Detail::initializeVtables(object, klass);

    // Set the defaults.
    res.writeDefault();
    return res;
}

namespace
{
    hkReflect::Var getFieldByNameCaseInsensitive(const char* name, const hkReflect::Var& object)
    {
        for (hkReflect::DeclIter<hkReflect::FieldDecl> it(object.getType()); it.advance();)
        {
            if (hkString::strCasecmp(it.current().getName(), name) == 0)
            {
                return object[it.current()];
            }
        }
        return hkReflect::Var();
    }
}


void hctMaxSceneExporter::convertMaxModifierToClassAndObject(Modifier* modifierItem, hkResourceHandle* resourceHandle, const char* nodeName, INode* maxNode)
{
    hkReflect::Var object(resourceHandle->getObject(), resourceHandle->getObjectType());

    ICustAttribContainer* customAttributeContainer = modifierItem->GetCustAttribContainer();
    if ( customAttributeContainer )
    {
        int numCustomAttributes = customAttributeContainer->GetNumCustAttribs();

        // Each attrib in Max is a roll down, so we can group them as such
        {
            for (int customAttributesIndex = 0; customAttributesIndex < numCustomAttributes; customAttributesIndex++)
            {
                CustAttrib* customAttribute = (CustAttrib*)customAttributeContainer->GetCustAttrib(customAttributesIndex);
                if ( customAttribute ) // just to be safe
                {
                    // Usually CA only have one PBlock, but they can have more than one
                    const int numParamBlocks = customAttribute->NumParamBlocks();

                    for (int i = 0; i < numParamBlocks; i++)
                    {
                        HCT_SCOPED_CONVERSIONS;

                        IParamBlock2* pblock2 = customAttribute->GetParamBlock(i);
                        hctParamBlock2Interface pblockInterface (pblock2);

                        // EXP-474 : Sometimes pblock2 is NULL
                        if ( !pblock2 ) continue;

                        int numPblockParameters = pblockInterface.getNumParameters();

                        for (int pblockIndex = 0 ; pblockIndex < numPblockParameters; pblockIndex++)
                        {
                            //hctCommonParameterInterface::AttributeType attributeType = pblockInterface.getAttributeType(idx);
                            MSTR paramNameStr = pblockInterface.getParameterInternalName(pblockIndex);
                            const char* paramName = FROM_MAX(paramNameStr);

                            hkReflect::Var actualObject = object;
                            const char* remainingName = paramName;
                            const char* colonPos;
                            while ( (colonPos = hkString::strChr(remainingName, '_')) != 0 )
                            {
                                int substringLen = int(colonPos - remainingName);
                                hkStringOld structName(remainingName, substringLen);
                                hkReflect::Var strukt = actualObject[structName.cString()];
                                if ( !strukt )
                                {
                                    break;
                                }
                                actualObject = strukt;
                                remainingName = colonPos + 1;
                            }

                            hkStringOld revertedRemainingNameBuffer;
                            const char* memberName = hctSdkUtils::revertValidName(remainingName, revertedRemainingNameBuffer);

                            // Because Max does some strange manipulations on symbol names, use the case insensitive version of the getMemberIndexByName method
                            hkReflect::Var field = getFieldByNameCaseInsensitive(memberName, actualObject);

                            if ( field.isValid() )
                            {
                                hkStringOld revertedRemainingPathAndNameBuffer;
                                const char* memberPathAndName = hctSdkUtils::revertValidName(paramName, revertedRemainingPathAndNameBuffer);

                                attributeToClassMember(pblockIndex, field, memberPathAndName, pblockInterface, resourceHandle, nodeName, maxNode);
                            }
                            else
                            {
                                // If there's no class member that would match the Custom Attribute's name, we will try to add the Attribute and its value
                                // as a property to the class (if the class supports this).
                                hkReflect::Var attributesField = actualObject["attributes"];
                                if ( attributesField.isValid() )
                                {
                                    if ( hkArray<hkxAttribute>* attributes = attributesField.dynCast< hkArray<hkxAttribute> >() )
                                    {
                                        hkxAttribute attribute;
                                        hkResult res = convertSingleAttrib(pblockInterface, pblockIndex, attribute, object.getType());
                                        if ( res.isSuccess() )
                                        {
                                            attributes->pushBack(attribute);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    resourceHandle->setObject(object.getAddress(), object.getType());
}


void hctMaxSceneExporter::attributeToClassMember(int pblockIndex, const hkReflect::Var& field,
    const char* memberPathAndName, const hctCommonParameterInterface& pblock, hkResourceHandle* resourceHandle, const char* nodeName, INode* maxNode)
{
    struct Visitor : public hkReflect::VarVisitor<Visitor, void, const hctCommonParameterInterface&, int>
    {
        Visitor(const char* nodeName, const char* memberPathAndName, hkResourceHandle* resourceHandle, hkPointerMap< const INode*, hkxNode*>& nodeMap)
            : nodeName(nodeName)
            , memberPathAndName(memberPathAndName)
            , resourceHandle(resourceHandle)
            , nodeMap(nodeMap)
        {
        }

        const char* nodeName;
        const char* memberPathAndName;
        hkResourceHandle* resourceHandle;
        hkPointerMap< const INode*, hkxNode*>& nodeMap;

        static void setString(const hkReflect::StringVar& var, const char* str)
        {
            // hack to handle raw char*
            if (char** raw = var.dynCast<char*>())
            {
                *raw = hkString::strDup(str);
            }
            else
            {
                var.setValue(str);
            }
        }

        void visit(const hkReflect::VoidVar&, const hctCommonParameterInterface&, int) { HK_UNREACHABLE(0x179a2eae, "Void field found" ); }
        void visit(const hkReflect::ArrayVar&, const hctCommonParameterInterface&, int) { HK_WARN(0xe89fd61, "Field should not be an array"); }
        void visit(const hkReflect::RecordVar&, const hctCommonParameterInterface&, int) { HK_WARN(0x1e46313, "Field should not be a record"); }

        void visitUuid(hkUuid* uuid, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            HCT_SCOPED_CONVERSIONS;

            MSTR val;
            const hkResult res = pblock.getParameterValue(pblockIndex, 0, val);
            if (res.isSuccess())
            {
                hkStringPtr str = hkString::strDup(FROM_MAX(val));
                uuid->setFromFormattedString(str.cString());
            }
            else
            {
                *uuid = hkUuid::getNil();
            }
        }

        void visit(const hkReflect::IntVar& var, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            if (const hk::Presets* presets = var.getType()->findAttribute<hk::Presets>())
            {
                int itemIdx;
                pblock.getParameterValue(pblockIndex, 0, itemIdx);
                itemIdx--;
                if (itemIdx >= presets->getNumPresets())
                {
                    itemIdx = 0;
                }

                var.assign(presets->getPreset(itemIdx));
            }
            else
            {
                int val;
                pblock.getParameterValue(pblockIndex, 0, val);
                
                var.setValue(val);
            }
        }

        void visit(const hkReflect::FloatVar& var, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            hkReal val;
            pblock.getParameterValue(pblockIndex, 0, val);
            var.setValue(val);
        }

        void visit(const hkReflect::BoolVar& var, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            int val;
            pblock.getParameterValue(pblockIndex, 0, val);
            var.setValue(val ? true : false);
        }

        void visit(const hkReflect::StringVar& var, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            const hkLinkAttribute*      linkAttributes      = var.getType()->findAttribute<hkLinkAttribute>();
            if ( linkAttributes && linkAttributes->m_type == hkLinkAttribute::PARENT_NAME )
            {
                setString(var, nodeName);
            }
            else
            {
                HCT_SCOPED_CONVERSIONS;

                MSTR val;
                const hkResult res = pblock.getParameterValue(pblockIndex, 0, val);
                if ( res.isSuccess() )
                {
                    setString(var, FROM_MAX(val));
                }
                else
                {
                    setString(var, "");
                }
            }
        }

        void visitVector4(hkVector4* vec, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            hkVector4 val;
            pblock.getParameterValue(pblockIndex, 0, val);
            vec->setComponent<0>(val(0));
            vec->setComponent<1>(val(1));
            vec->setComponent<2>(val(2));
            vec->setComponent<3>(val(3));
        }

        void visit(const hkReflect::PointerVar& var, const hctCommonParameterInterface& pblock, int pblockIndex)
        {
            hkReflect::FieldDecl field = var.getType()->isField();
            HK_ASSERT_NO_MSG(0x3332937b, field);

            const hkLinkAttribute*      linkAttributes      = var.getType()->findAttribute<hkLinkAttribute>();
            hkStringOld memberName = memberPathAndName;
            memberName = memberName.replace("_", "."); // sync id: <0xaf1ee232>

            if ( linkAttributes && linkAttributes->m_type == hkLinkAttribute::NODE_UUID )
            {
                // Get the referenced node
                INode* linkedMaxNode    = HK_NULL;
                const hkResult res      = pblock.getParameterValue(pblockIndex, 0, linkedMaxNode);
                hkStringBuf linkName;   linkName.printf("UndefinedLink_%s", field.getName());

                if ( res.isSuccess() )
                {
                    // Locate its hkxNode
                    const hkxNode* linkedHkxNode = nodeMap.getWithDefault(linkedMaxNode, HK_NULL);
                    if ( linkedHkxNode )
                    {
                        const hkUuid& linkUuid = linkedHkxNode->m_uuid;
                        linkUuid.toString(linkName);
                    }
                }

                resourceHandle->addExternalLink(memberName.cString(), linkName.cString());
            }
            else if ( linkAttributes && linkAttributes->m_type == hkLinkAttribute::MESH )
            {
                HCT_SCOPED_CONVERSIONS;

                MSTR linkedNodeName;
                const hkResult res = pblock.getParameterValue(pblockIndex, 0, linkedNodeName);

                // Only create a new external (unresolved) link if the INODE was valid.
                if ( res.isSuccess() )
                {
                    resourceHandle->addExternalLink(memberName.cString(), FROM_MAX(linkedNodeName));
                }
                else
                {
                    hkStringOld replacementLinkName;
                    replacementLinkName.printf("UndefinedLink_%s", field.getName());
                    resourceHandle->addExternalLink(memberName.cString(), replacementLinkName.cString());
                }
            }
            else
            {
                ReferenceTarget* refTarget;
                pblock.getParameterValue(pblockIndex, 0, refTarget);

                if ( refTarget )
                {
                    hkStringOld linkedId;
                    linkedId.printf("%s:0x%x", nodeName, refTarget);
                    resourceHandle->addExternalLink(memberName.cString(), linkedId.cString());
                }
                else
                {
                    hkStringOld replacementLinkName;
                    replacementLinkName.printf("UndefinedLink_%s", field.getName());
                    resourceHandle->addExternalLink(memberName.cString(), replacementLinkName.cString());
                }
            }

            // We set the member value to HK_NULL because we cannot fix up the pointer now.
            var.setValue(hkReflect::Var());
        }
    } visitor(nodeName, memberPathAndName, resourceHandle, m_nodeMap);

    if (hkUuid* uuid = field.dynCast<hkUuid>())
    {
        visitor.visitUuid(uuid, pblock, pblockIndex);
    }
    else if (hkVector4* vec = field.dynCast<hkVector4>())
    {
        visitor.visitVector4(vec, pblock, pblockIndex);
    }
    else
    {
        visitor.dispatch(field, pblock, pblockIndex);
    }
}


void hctMaxSceneExporter::collectINodesRecursively(INode* iNode, hkArray<INode*>& iNodesOut)
{
    // Ignore Group Heads.
    if ( !iNode->IsGroupHead() )
    {
        iNodesOut.pushBack(iNode);
    }

    // Process all child nodes.
    {
        for (int i = 0; i < iNode->NumberOfChildren(); i++)
        {
            collectINodesRecursively(iNode->GetChildNode(i), iNodesOut);
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
