// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Max/MaxSceneExport/hctMaxSceneExport.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/hctMaxSceneExporter.h>
#include <ContentTools/Max/MaxSceneExport/Utils/hctMaxUtils.h>
#include <ContentTools/Max/MaxSceneExport/Exporter/CommonInterfaces/hctCommonSkinInterface.h>

#include <Common/SceneData/Attributes/hkxAttributeGroup.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>
#include <Common/SceneData/Material/hkxMaterialEffect.h>

#include <Common/SceneData/Mesh/Channels/hkxVertexSelectionChannel.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/SceneData/Skin/hkxSkinUtils.h>

#include <Common/SceneData/Mesh/hkxMeshSectionUtil.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>

#include <ContentTools/Common/SceneExport/Userchannel/hctUserChannelUtil.h>
#include <ContentTools/Common/SceneExport/Utils/hctSceneExportUtils.h>

#include <ContentTools/Max/MaxFpInterfaces/Physics/ExportChannel/hctExportChannelModifierInterface.h>

#include <cs/Phyexp.h>
#include <MeshNormalSpec.h>

// Max 6 or higher:
#if defined(MAX_VERSION_MAJOR) && (MAX_VERSION_MAJOR >= 6)
#define MAX6_OR_HIGHER
#include <IDxMaterial.h>
#if (MAX_VERSION_MAJOR >= 7)
#define MAX7_OR_HIGHER
#endif
#if (MAX_VERSION_MAJOR >= 12)
#define MAX2010_OR_HIGHER
#endif
#endif

#include "shaders.h"


#define MAX_NUM_HAVOK_BONES 4

hctMaxMeshWriter::hctMaxMeshWriter(const hctMaxSceneExportOptions& exportOptions)
    : m_mesh(HK_NULL), m_skin(HK_NULL), m_exportOptions(exportOptions), m_inode(HK_NULL), m_triObject(HK_NULL), m_deleteTriObject(false), m_wasntAMesh(false)
{


}

hctMaxMeshWriter::~hctMaxMeshWriter()
{
    cleanup();
}

BOOL hctMaxMeshWriter::init(INode* inode, int staticFrame)
{
    // Deallocate memory for previous mesh (if any)
    cleanup();

    m_iStaticFrame = staticFrame;

    if (!inode) return FALSE;

    const TimeValue staticTick = m_iStaticFrame*GetTicksPerFrame();

    Object* curObject = inode->GetObjectRef();
    while (curObject)
    {
        IDerivedObject* devObject = curObject->SuperClassID() == GEN_DERIVOB_CLASS_ID ? (IDerivedObject*)curObject : NULL;
        if (devObject) // has modifiers
        {
            for (int m = devObject->NumModifiers() - 1; m >= 0; --m)
            {
                Modifier* modifierItem = devObject->GetModifier(m);
                if (modifierItem->IsEnabled())
                {
                    ISkin* skinMod = (ISkin*)modifierItem->GetInterface(I_SKIN);
                    if (skinMod)
                    {
                        m_skin = new hctStandardSkinInterface(skinMod, inode);
                    }

                    if (modifierItem->ClassID() == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
                    {
                        IPhysiqueExport* physique = (IPhysiqueExport *)modifierItem->GetInterface(I_PHYINTERFACE);
                        if (physique)
                        {
                            m_skin = new hctPhysiqueSkinInterface(physique, inode);
                        }
                    }

                }
            }
            curObject = devObject->GetObjRef(); // drop down one level the stack
        }
        else // p_end of the line, the base object
        {
            curObject = HK_NULL;
        }
    }

    if (m_skin)
    {
        if (m_owner->m_exportOptions.m_respectModifierSkinPose)
        {
            m_skin->toggleSkinPose(true);
        }

        if (!m_skin->initialize())
        {
            m_skin->toggleSkinPose(false);
            delete m_skin;
            m_skin = HK_NULL;
        }
    }
    else
    {
        //Vision only
        //Check if there is no bone above otherwise toggle the found bone's
        //bind pose as this also influences the unskinned mesh's transform
        if (m_owner->m_exportOptions.m_respectModifierSkinPose)
        {
            toggleAllParentINodeBindPose(inode, true);
        }
    }

    // Initialize mesh
    {
        // We ask the object to convert itself to a TriObject
        Object *obj = inode->EvalWorldState(staticTick).obj;
        if (!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))
        {
            return FALSE;
        }

        // EXP-848 : Keep track if the object was of an unsupported type (non-mesh)
        static const Class_ID supportedClassIDs[] =
        {
            Class_ID(EDITTRIOBJ_CLASS_ID,0),
            Class_ID(TRIOBJ_CLASS_ID, 0),
            Class_ID(POLYOBJ_CLASS_ID, 0),
            EPOLYOBJ_CLASS_ID
        };

        m_wasntAMesh = true;
        Class_ID objClassID = obj->ClassID();
        for (int i = 0; i < sizeof(supportedClassIDs) / sizeof(Class_ID); i++)
        {
            if (supportedClassIDs[i] == objClassID)
            {
                m_wasntAMesh = false;
                break;
            }
        }

        m_triObject = (TriObject *)obj->ConvertToType(staticTick, Class_ID(TRIOBJ_CLASS_ID, 0));

        // if the obj!=triobject, it means that the triobj has been created for us, and we will have to destroy it
        m_deleteTriObject = (obj != m_triObject);
    }

    m_inode = inode;
    m_mesh = &m_triObject->GetMesh();

    return TRUE;
}

void hctMaxMeshWriter::cleanup()
{
    if (m_triObject && m_deleteTriObject)
    {
        delete m_triObject;
    }

    if (m_skin)
    {
        if (m_owner->m_exportOptions.m_respectModifierSkinPose)
        {
            m_skin->toggleSkinPose(false);
        }
        delete m_skin;
    }
    else
    {
        //Vision only
        //Check if there is no bone above otherwise toggle the found bone's
        //bind pose as this also influences the unskinned mesh's transform
        if (m_owner->m_exportOptions.m_respectModifierSkinPose)
        {
            toggleAllParentINodeBindPose(m_inode, false);
        }
    }

    m_mesh = HK_NULL;
    m_skin = HK_NULL;
    m_inode = HK_NULL;
    m_triObject = HK_NULL;
    m_deleteTriObject = false;
    m_wasntAMesh = false;
}

BOOL hctMaxMeshWriter::extract()
{
    return true; // do the extraction on the fly in Max (unlike in Maya)
}

static int _containsNode(hkArray<const INode*>& tab, const INode* search)
{
    int nb = tab.getSize();
    for (int i = nb - 1; i >= 0; --i)
    {
        if (tab[i] == search) return i;
    }
    return -1;
}


static void _countChildNodesRecursive(INode* currNode, int &numChildren)
{
    for (int c = 0; c < currNode->NumberOfChildren(); ++c)
    {
        INode* child = currNode->GetChildNode(c);
        _countChildNodesRecursive(child, numChildren);
    }
    numChildren++;
}
static int _getNumberOfBonesInSkeleton(const INode* bone)
{
    INode* root = const_cast<INode*> (bone);

    int numBonesInSkeleton = 0;

    if (root == NULL)
        return numBonesInSkeleton;

    //Get the root node of this bone's hierarchy, make sure it is not the scene's root node (in the case where the bone is not part of a hierarchy)
    while (root->GetParentNode() != NULL && root->GetParentNode() != GetCOREInterface()->GetRootNode())
    {
        root = root->GetParentNode();
    }

    //Count children
    _countChildNodesRecursive(root, numBonesInSkeleton);

    return numBonesInSkeleton;
}

static INode* _getBoneRoot(const INode* bone)
{
    INode* root = const_cast<INode*> (bone);

    if (root == NULL)
        return NULL;

    //Get the root node of this bone's hierarchy, make sure it is not the scene's root node (in the case where the bone is not part of a hierarchy)
    while (root->GetParentNode() != NULL && root->GetParentNode() != GetCOREInterface()->GetRootNode())
    {
        root = root->GetParentNode();
    }

    return root;
}

static inline int _findVert(const hkArray< hctMaxMeshWriter::FullVert >& verts, DWORD v)
{
    const int nv = verts.getSize();
    for (int i = 0; i < nv; ++i)
    {
        if (v == (unsigned)(verts[i].vertIndex))
            return i;
    }
    return -1;
}

void hctMaxMeshWriter::createSkin(hkxMesh*& newMesh, hkxSkinBinding*& newSkin)
{
    const TimeValue staticTick = m_iStaticFrame * GetTicksPerFrame();

    const int numMeshVerts = m_mesh->getNumVerts();
    const int numSkinnedVerts = m_skin ? m_skin->getNumSkinnedVerts() : numMeshVerts; // support for forced skinned mesh (m_skin will be null)

    // EXP-848 : Skinned nurbs not supported
    if (m_wasntAMesh && m_skin)
    {
        HK_WARN_ALWAYS(0xabba1baa, "Skin only supported on polygonal meshes. Ignoring skin (" << m_inode->GetName() << ").");
        return;
    }

    // EXP-778
    if (numSkinnedVerts > numMeshVerts)
    {
        HK_WARN_ALWAYS(0xabbaeda7, "Topology changes detected after skin modifier - can't export skin for node \"" << m_inode->GetName() << "\"");
        HK_WARN_ALWAYS(0xabbaeda7, "Remove or disable modifiers above the skin modifier to avoid this error.");
        return;
    }


    hkArray<float> weights; // 4 per vert for havok default mesh format
    hkArray<int> indices; // same, always 4
    weights.setSize(numMeshVerts * 4, -1.0f);
    indices.setSize(numMeshVerts * 4, -1);

    int forceRigidSkinnedBone = 0;
    hkArray<const INode*> boneNodes;
    Tab<bool> doneBones;

    // EXP-2492 : Surface with no bones export as mesh
    if (m_skin)
    {
        bool hasBones = false;
        if (m_skin)
        {
            for (int x = 0; x < m_skin->getNumSkinnedVerts(); x++)
            {
                if (m_skin->getNumVertexBones(x))
                {
                    hasBones = true;
                    break;
                }
            }
        }
        if (!hasBones)
        {
            HK_WARN_ALWAYS(0xabbaed57, "Found a skin in node " << m_inode->GetName() << " with no bones, treating as a mesh.");
            return;
        }
    }

    if (!m_skin)
    {
        // Add fake rigid skin for node - get first bone parent.
        forceRigidSkinnedBone = boneNodes.getSize();
        INode* parent = m_inode;
        while (parent && !hctMaxUtils::isBone(parent))
        {
            parent = parent->GetParentNode();
        }

        if (!parent)
        {
            parent = m_inode;
        }
        boneNodes.pushBack(parent);
    }

    for (int x = 0; x < numSkinnedVerts; x++)
    {
        const int fourX = x * 4;

        int numBones = m_skin ? m_skin->getNumVertexBones(x) : 1;

        int usedBones = fourX;

        if (numBones == 1) // RIGID
        {
            // rigid (one bone influence only)
            if (m_skin)
            {
                hkReal weight = m_skin->getVertexBoneWeight(x, 0);
                if (weight > 0.0f)
                {
                    weights[usedBones] = 1.0f;
                    const INode* boneNode = m_skin->getVertexBoneNode(x, 0);
                    int bidx = _containsNode(boneNodes, boneNode);
                    if (bidx < 0)
                    {
                        bidx = boneNodes.getSize();
                        boneNodes.pushBack(boneNode);
                    }
                    indices[usedBones++] = bidx;
                }
                else
                {
                    weights[usedBones] = 0;
                    indices[usedBones++] = 0;
                }
            }
            else
            {
                weights[usedBones] = 1;
                indices[usedBones++] = forceRigidSkinnedBone;
            }
        }
        else // blended
        {
            if (numBones > MAX_NUM_HAVOK_BONES) // pick the 4 with most influence and normalize? Ok for now:
            {
                doneBones.SetCount(numBones);
                memset(&doneBones[0], 0, sizeof(bool)*numBones);
                for (int b = 0; b < MAX_NUM_HAVOK_BONES; ++b)
                {
                    int curMax = -1;
                    float curMaxWeight = -1;
                    for (int y = 0; y < numBones;y++)
                    {
                        if (!doneBones[y])
                        {
                            float w = m_skin->getVertexBoneWeight(x, y);
                            if (w > curMaxWeight)
                            {
                                curMax = y; curMaxWeight = w;
                            }
                        }
                    }

                    if (curMax >= 0)
                    {
                        const INode* boneNode = m_skin->getVertexBoneNode(x, curMax);
                        int bidx = _containsNode(boneNodes, boneNode);
                        if (bidx < 0)
                        {
                            bidx = boneNodes.getSize();
                            boneNodes.pushBack(boneNode);
                        }
                        weights[usedBones] = curMaxWeight;
                        indices[usedBones++] = bidx;
                        doneBones[curMax] = true;
                    }
                }
                // will be renormalized later
                numBones = MAX_NUM_HAVOK_BONES;
            }
            else
            {
                for (int y = 0; y < numBones;y++)
                {
                    weights[usedBones] = m_skin->getVertexBoneWeight(x, y);
                    const INode* boneNode = m_skin->getVertexBoneNode(x, y);
                    int bidx = _containsNode(boneNodes, boneNode);
                    if (bidx < 0)
                    {
                        bidx = boneNodes.getSize();
                        boneNodes.pushBack(boneNode);
                    }
                    indices[usedBones++] = bidx;
                }
            }
        }

#define WANT_RENORMALIZE
#ifdef WANT_RENORMALIZE

        // normalize so that they sum to 1
        float tw = 0;
        int nbi;
        for (nbi = 1; nbi <= numBones; ++nbi) // 1 as index offset backwards
        {
            tw += weights[usedBones - nbi]; // total weigth
        }

        for (nbi = 1; nbi <= numBones; ++nbi)
            weights[usedBones - nbi] /= tw; // make them sum to 1
#endif

        while (usedBones < (fourX + 4))
        {
            indices[usedBones] = 0; // doesn't matter as zero weight
            weights[usedBones++] = 0;
        }
    }

    // Verts that aren't skinned in the mesh (bad envelopes)
    int usedBones = numSkinnedVerts * 4;
    for (int j = numSkinnedVerts; j < numMeshVerts; ++j)
    {
        for (int b = 0; b < 4; ++b)
        {
            indices[usedBones] = 0; // doesn't matter as zero weight
            weights[usedBones++] = 0;
        }
    }

    if (boneNodes.getSize() == 0)
    {
        HK_WARN_ALWAYS(0xabba1baa, "No bones have been assigned to skinned vertices - mesh will not be animated");
    }

    Matrix3 worldTM = hctMaxSceneExporter::calculateNodeWorldTM(m_inode, staticTick);

    // Need to count the number of bones in the skeleton to see what size to use for the bone indices
    bool useInt8Indices = true;
    {
        hkArray <const INode* > uniqueRootNodes;
        for (int b = 0; b < boneNodes.getSize(); b++)
        {
            const INode* boneRoot = _getBoneRoot(boneNodes[b]);

            if (boneRoot != NULL)
            {
                if (uniqueRootNodes.indexOf(boneRoot) == -1)
                {
                    uniqueRootNodes.pushBack(boneRoot);
                }
            }
        }

        int totalNumSkeletalBones = 0;

        for (int urb = 0; urb < uniqueRootNodes.getSize(); urb++)
        {
            totalNumSkeletalBones += _getNumberOfBonesInSkeleton(uniqueRootNodes[urb]);
        }

        if (totalNumSkeletalBones > 255)
            useInt8Indices = false;
    }
    // output the skin mesh (with weights and indices per vert)
    createMesh(newMesh, &weights, &indices, &worldTM, useInt8Indices);

    if (newMesh == HK_NULL) // no mesh created.. so we can't create a skin for it..
    {
        newSkin = HK_NULL;
        return;
    }

    newSkin = new hkxSkinBinding();
    newSkin->m_mesh = newMesh;

    // Bone bindings
    int numBones = boneNodes.getSize();
    newSkin->m_nodeNames.setSize(numBones);

    
    // just store the INode for now. Will do a fixup later on once all hkxNodes are actually written out
    HK_COMPILE_TIME_ASSERT(sizeof(INode*) == sizeof(hkStringPtr));
    reinterpret_cast<hkArray<const INode*>&>(newSkin->m_nodeNames) = boneNodes;

    // Init bone bind pose.
    newSkin->m_bindPose.setSize(numBones);
    for (int bmi = 0; bmi < numBones; ++bmi)
    {
        const INode* boneINode = boneNodes[bmi];

        if (boneINode)
        {
            Matrix3 btm;
            btm = hctMaxSceneExporter::calculateNodeWorldTM(boneINode, m_iStaticFrame*GetTicksPerFrame());
            hctMaxUtils::convertToMatrix4(btm, newSkin->m_bindPose[bmi]);
        }
        else // ehh.. no inode?
        {
            newSkin->m_bindPose[bmi].setIdentity();
        }
    }

    // Init skin tm for bind pose (physique is a WSM and so modifies the skin tm etc)
    hctMaxUtils::convertToMatrix4(worldTM, newSkin->m_initSkinTransform);
}

inline hkUint8 quantizeTo255(float v)
{
    if (v < 0) v = 0;
    float v2 = (v * 255);
    if (v2 < 254.0f) return (hkUint8)v2;
    else return 255;
}

Point3 _getObjectVertexNormal(Mesh* mesh, int faceNo, RVertex* rv, int faceVertexIndexHint)
{
    Face* f = &mesh->faces[faceNo];
    DWORD smGroup = f->smGroup;
    Point3 vertexNormal;

    //[EXP-2361] Is normal specified
    MeshNormalSpec * meshNorm = mesh->GetSpecifiedNormals();
    if (meshNorm && meshNorm->GetNumNormals())
    {
        vertexNormal = meshNorm->GetNormal(faceNo, faceVertexIndexHint);
    }
    else
    {
        // SPCIFIED is not currently used, but may be used in future versions.
        if (rv->rFlags & SPECIFIED_NORMAL)
        {
            vertexNormal = rv->rn.getNormal();
        }
        // If normal is not specified it'sas only available if the face belongs
        // to a smoothing group
        else
        {
            int numNormals = rv->rFlags & NORCT_MASK;

            if (numNormals && smGroup)
            {
                // If there is only one vertex is found in the rn member.
                if (numNormals == 1)
                {
                    vertexNormal = rv->rn.getNormal();
                }
                else
                {
                    // If two or more vertices are there you need to step through them
                    // and find the vertex with the same smoothing group as the current face.
                    // You will find multiple normals in the ern member.
                    for (int i = 0; i < numNormals; i++)
                    {
                        if (rv->ern[i].getSmGroup() & smGroup)
                        {
                            vertexNormal = rv->ern[i].getNormal();
                        }
                    }
                }
            }
            else
            {
                // Get the normal from the Face if no smoothing groups are there
                vertexNormal = mesh->getFaceNormal(faceNo);
            }
        }
    }
    return vertexNormal;
}

void hctMaxMeshWriter::createVertexBuffer(hkxVertexBuffer*& newVB, std::deque<FullVert>& verts, hkArray<float>* weights, hkArray<int>* indices, hkArray<int>* channelIDs, bool hasMaterial, Matrix3* worldTM, bool useInt8BoneIndices)
{
    const TimeValue staticTick = m_iStaticFrame * GetTicksPerFrame();

    bool hasNormals = true; // !! When could this be false?

    BOOL vcSupport = m_mesh->mapSupport(0); // 0 = Vertex Color
    BOOL alphaSupport = m_mesh->mapSupport(MAP_ALPHA);

    int numTCoords = channelIDs ? channelIDs->getSize() : 0; // used texture channels, as found from the material used, in texture stage order

    BOOL hasVColors = vcSupport;
    BOOL hasAColors = alphaSupport;
    union { DWORD i; unsigned char c[4]; } wc;
    wc.i = m_inode->GetWireColor();
    Point3 wireCol((float)wc.c[0] / 255.f, (float)wc.c[1] / 255.f, (float)wc.c[2] / 255.f);

    bool isSkinned = indices && weights;

    // Write a vertex color channel if we have vertex colors on the mesh, or the following holds:
    // - no explicit vertex colors
    // - no material applied
    // - wire colors should be exported as vertex colors
    bool useWireColorAsVertexColor = !hasVColors && m_exportOptions.m_exportWireColorsAsVertexColors && !hasMaterial;
    bool needVColor = hasAColors || hasVColors || useWireColorAsVertexColor;

    // Get world to local transform
    Matrix3 worldToLocal(true);
    if (worldTM != NULL)
    {
        worldToLocal = Inverse(*worldTM);
    }

    // Describe what we are going to export
    hkxVertexDescription desiredVertDesc;
    desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_POSITION, hkxVertexDescription::HKX_DT_FLOAT, 3));
    if (hasNormals)
    {
        desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_NORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3));
    }
    if (needVColor)
    {
        desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_COLOR, hkxVertexDescription::HKX_DT_UINT32, 1));
    }

    if (isSkinned)
    {
        desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, hkxVertexDescription::HKX_DT_UINT8, 4));

        if (useInt8BoneIndices)
        {
            desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, hkxVertexDescription::HKX_DT_UINT8, 4));
        }
        else
        {
            desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, hkxVertexDescription::HKX_DT_INT16, 4));
        }
    }

    int numTCoordsInFormat = hkMath::max2(1, numTCoords); // always try to export the default channel anyway
    int maxHandledTCoords = hkMath::clamp<int>(numTCoordsInFormat, 0, MAX_EXPORTED_CHANNELS); // can't fit more than MAX_EXPORTED_CHANNELS in the local FullVert struct
    for (int tcr = 0; tcr < maxHandledTCoords; ++tcr)
    {
        // EXP-2344 uv-channelIDs need to be preserved in the descriptor so they can be referenced later on.
        hkStringBuf channelIDString = "";
        if (tcr < (*channelIDs).getSize())
        {
            channelIDString.printf("%d", (*channelIDs)[tcr]);
        }
        else
        {
            channelIDString = "default";
        }

        desiredVertDesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_TEXCOORD, hkxVertexDescription::HKX_DT_FLOAT, 2, channelIDString.cString()));
    }

    // Allocate buffer
    newVB = new hkxVertexBuffer();
    newVB->setNumVertices((int)verts.size(), desiredVertDesc);


    const Matrix3 localToWorld = m_inode->GetNodeTM(staticTick);
    const Matrix3 objectToWorld = m_inode->GetObjTMAfterWSM(staticTick);
    const Matrix3 objectToLocal = objectToWorld * Inverse(localToWorld);

    // EXP-387
    // The normals given by the mesh are in object space
    // but we want them in local space
    Matrix3 normalTransform;
    {
        // Normals are transformed by the inverse transposed
        Matrix3 temp = objectToLocal;
        temp.SetTrans(Point3(0, 0, 0)); // no translation
        Matrix3 inverse = Inverse(temp);
        Matrix3& inverseTransposed = normalTransform;
        inverseTransposed.SetRow(0, inverse.GetColumn3(0));
        inverseTransposed.SetRow(1, inverse.GetColumn3(1));
        inverseTransposed.SetRow(2, inverse.GetColumn3(2));
        inverseTransposed.SetRow(3, Point3(0, 0, 0));
    }

    const hkxVertexDescription& vertDesc = newVB->getVertexDesc();
    const hkxVertexDescription::ElementDecl* posDecl = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
    const hkxVertexDescription::ElementDecl* normDecl = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
    const hkxVertexDescription::ElementDecl* colorDecl = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_COLOR, 0);
    const hkxVertexDescription::ElementDecl* weightsDecl = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0);
    const hkxVertexDescription::ElementDecl* indicesDecl = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0);

    hkArray<const hkxVertexDescription::ElementDecl*> texCoordDecls;
    texCoordDecls.setSize(maxHandledTCoords);

    hkArray<int> texCoordStride;
    hkArray<char*> texCoordBuf;

    texCoordStride.setSize(maxHandledTCoords);
    texCoordBuf.setSize(maxHandledTCoords);
    for (int tc = 0; tc < maxHandledTCoords; ++tc)
    {
        texCoordDecls[tc] = vertDesc.getElementDecl(hkxVertexDescription::HKX_DU_TEXCOORD, tc);
        if (texCoordDecls[tc])
        {
            texCoordStride[tc] = texCoordDecls[tc]->m_byteStride;
            texCoordBuf[tc] = static_cast<char*>(newVB->getVertexDataPtr(*texCoordDecls[tc]));
        }
        else
        {
            texCoordStride[tc] = 0;
            texCoordBuf[tc] = HK_NULL;
        }
    }

    int posStride = posDecl ? posDecl->m_byteStride : 0;
    int normStride = normDecl ? normDecl->m_byteStride : 0;
    int colorStride = colorDecl ? colorDecl->m_byteStride : 0;
    int weightsStride = weightsDecl ? weightsDecl->m_byteStride : 0;
    int indicesStride = indicesDecl ? indicesDecl->m_byteStride : 0;

    char* posBuf = static_cast<char*>(posDecl ? newVB->getVertexDataPtr(*posDecl) : HK_NULL);
    char* normBuf = static_cast<char*>(normDecl ? newVB->getVertexDataPtr(*normDecl) : HK_NULL);
    char* colorBuf = static_cast<char*>(colorDecl ? newVB->getVertexDataPtr(*colorDecl) : HK_NULL);
    char* weightsBuf = static_cast<char*>(weightsDecl ? newVB->getVertexDataPtr(*weightsDecl) : HK_NULL);
    char* indicesBuf = static_cast<char*>(indicesDecl ? newVB->getVertexDataPtr(*indicesDecl) : HK_NULL);

    int numVerts = newVB->getNumVertices();
    for (int i = 0; i < numVerts; ++i)
    {
        const FullVert& fv = verts[i];
        const int vertexIndex = fv.vertIndex;

        // Pos in local space
        {
            const Point3 posObject = m_mesh->getVert(vertexIndex); // objectSpace
            const Point3 posLocal = posObject * objectToLocal;

            float* _pos = (float*)(posBuf);
            _pos[0] = posLocal[0]; _pos[1] = posLocal[1]; _pos[2] = posLocal[2]; _pos[3] = 0;
        }

        // Normal in local space
        {
            Point3 normal;
            if (hasNormals)
            {
                RVertex* rv = m_mesh->getRVertPtr(fv.vertIndex);
                normal = _getObjectVertexNormal(m_mesh, fv.faceIndex, rv, fv.faceVertexIndexHint);
                normal = normalTransform * normal; // EXP-387
            }
            else
            {
                normal = Point3(0, 0, 0);
            }
            float* _normal = (float*)(normBuf);
            _normal[0] = normal[0]; _normal[1] = normal[1]; _normal[2] = normal[2]; _normal[3] = 0;
        }

        if (needVColor) // otherwise the format will skip it.
        {
            // Vert color
            unsigned color = 0xffffffff;

            assert(!hasVColors || (fv.vertColorIndex >= 0));
            assert(!hasAColors || (fv.alphaIndex >= 0));
            Point3 rgb(1, 1, 1);
            if (useWireColorAsVertexColor)
                rgb = wireCol;
            else if (hasVColors && (fv.vertColorIndex >= 0))
                rgb = m_mesh->vertCol[fv.vertColorIndex];

            float alpha = 1.0;
            if (hasAColors && (fv.alphaIndex >= 0))
            {
                MeshMap& alphaMap = m_mesh->Map(MAP_ALPHA);
                alpha = alphaMap.tv[fv.alphaIndex].x;
            }

            color = hkSceneExportUtils::floatsToARGB(rgb.x, rgb.y, rgb.z, alpha);

            hkUint32* _color = (hkUint32*)(colorBuf);
            *_color = color;
        }

        // Weights and indices, and uvs
        if (isSkinned)
        {
            // Blending
            int vfour = fv.vertIndex * 4;

            hkUint32* _i = (hkUint32*)(indicesBuf);
            if (indices)
            {
                if (useInt8BoneIndices)
                {
                    unsigned int compressedI = unsigned int((*indices)[vfour]) << 24 |
                        unsigned int((*indices)[vfour + 1]) << 16 |
                        unsigned int((*indices)[vfour + 2]) << 8 |
                        unsigned int((*indices)[vfour + 3]);

                    *_i = compressedI;
                }
                else if (!useInt8BoneIndices)
                {
                    unsigned int compressedIndices01 = unsigned int((*indices)[vfour]) << 16 | unsigned int((*indices)[vfour + 1]);
                    unsigned int compressedIndices23 = unsigned int((*indices)[vfour + 2]) << 16 | unsigned int((*indices)[vfour + 3]);

                    *_i = compressedIndices23;
                    _i++;
                    *_i = compressedIndices01;
                }
            }
            else
            {
                *_i = 0;
            }

            unsigned int compressedW = 0;
            if (weights) // quantized into 255 range
            {
                hkReal tempWeights[4];
                for (int tempWeightIndex = 0; tempWeightIndex < 4; tempWeightIndex++)
                {
                    tempWeights[tempWeightIndex] = (*weights)[vfour + tempWeightIndex];
                }

                hkUint8 tempQWeights[4];
                hkxSkinUtils::quantizeWeights(tempWeights, tempQWeights);

                compressedW = unsigned int(tempQWeights[0]) << 24 |
                    unsigned int(tempQWeights[1]) << 16 |
                    unsigned int(tempQWeights[2]) << 8 |
                    unsigned int(tempQWeights[3]);

            }
            hkUint32* _w = (hkUint32*)(weightsBuf);
            *_w = compressedW;
        }

        if (maxHandledTCoords > 0)
        {
            if ((numTCoords == 0) && (fv.texCoordIndex >= 0)) // might as well export the default channel
            {
                UVVert uv = m_mesh->tVerts[fv.texCoordIndex];
                float* _uv = (float*)(texCoordBuf[0]);
                _uv[0] = uv.x; _uv[1] = uv.y;
            }
            else if (channelIDs && channelIDs->getSize())
            {
                for (int ch = 0; ch < maxHandledTCoords; ++ch)
                {
                    const int channelID = (*channelIDs)[ch];
                    int cIndex = fv.mapChannels[ch];
                    if (m_mesh->mapSupport(channelID)) // EXP-1000
                    {
                        MeshMap& map = m_mesh->Map(channelID);
                        Point3 mapChannel = map.tv[cIndex];
                        float* _uv = (float*)(texCoordBuf[ch]);
                        _uv[0] = mapChannel.x; _uv[1] = mapChannel.y;
                    }
                }
            }
            else // always 1
            {
                float* _uv = (float*)(texCoordBuf[0]);
                _uv[0] = _uv[1] = 0;
            }
        }

        // next
        posBuf += posStride;
        normBuf += normStride;
        colorBuf += colorStride;
        weightsBuf += weightsStride;
        indicesBuf += indicesStride;
        for (int tcn = 0; tcn < maxHandledTCoords; ++tcn)
            texCoordBuf[tcn] += texCoordStride[tcn];

    }
}

static void _getMaterialIdsInMesh(const Mesh* mesh, int maxMaterialIds, hkArray<MtlID>& materialsOut)
{
    materialsOut.clear();
    for (int f = 0; f < mesh->numFaces; f++)
    {
        Face& face = mesh->faces[f];
        const MtlID mId = face.getMatID() % maxMaterialIds;
        bool alreadyFound = false;
        for (int k = 0; k < materialsOut.getSize(); k++)
        {
            if (materialsOut[k] == mId)
            {
                alreadyFound = true; break;
            }
        }
        if (!alreadyFound)
        {
            materialsOut.pushBack(mId);
        }
    }
}

static int _maxMatIds(Mtl* m)
{
    int subMats = m ? m->NumSubMtls() : 0;
    int max = hkMath::max2<int>(subMats, 1);
    if (m && (m->ClassID() == Class_ID(MIXMAT_CLASS_ID, 0))) // blends act as one, not N sub mats
    {
        max = 1;
    }
    for (int mi = 0; mi < subMats; ++mi)
    {
        int childMax = _maxMatIds(m->GetSubMtl(mi));
        max = hkMath::max2<int>(childMax, max);
    }
    return max;
}


// Same as IGame functionality
struct MyFaceEx
{
    // Index into the vertex array
    int m_vertIdx[3];
    // Index into the standard mapping channel
    int m_texCoordIdx[3];
    // Index into the vertex color array
    int m_vcolorIdx[3];
    // Index into the vertex alpha array
    int m_alphaIdx[3];
    // The smoothing group of the face
    DWORD m_faceSmGrp;
    // The smoothing group of each normal
    DWORD m_normalSmGrp[3];
    // The material ID of the face
    int m_matId;
    // Additional flags
    int m_flags;
    // Index of corresponding face in the original mesh
    int m_meshFaceIndex;
    // Index into edge visibility array.
    /*! 1 for visible, 0 if the edge is invisible.*/
    BOOL m_edgeVis[3];

    MyFaceEx(Mesh* mesh, int faceIdx)
    {
        Face& face = mesh->faces[faceIdx];
        TVFace* tvFace = mesh->tvFace ? &mesh->tvFace[faceIdx] : HK_NULL;
        TVFace* vcFace = mesh->vcFace ? &mesh->vcFace[faceIdx] : HK_NULL;
        TVFace* alphaFace = HK_NULL;
        {
            if (mesh->mapSupport(MAP_ALPHA))
            {
                alphaFace = &mesh->mapFaces(MAP_ALPHA)[faceIdx];
            }
        }

        // Per vertex info
        for (int vi = 0; vi < 3; vi++)
        {
            m_vertIdx[vi] = face.v[vi];
            m_texCoordIdx[vi] = tvFace ? tvFace->t[vi] : -1;
            m_vcolorIdx[vi] = vcFace ? vcFace->t[vi] : -1;
            m_alphaIdx[vi] = alphaFace ? alphaFace->t[vi] : -1;
            m_edgeVis[vi] = face.getEdgeVis(vi);

            // EXP-798 : Track normals
            //           We store the smoothing groups of the rendering normal we will use
            //           Later on we will compare it (&-and) and reuse. We use 0 (no smoothing groups) for unique normals (never reused).
            //           Logic here is very similar to _getObjectVertexNormal() above
            m_normalSmGrp[vi] = 0; // vertex normal unique by default
            {
                RVertex* rv = mesh->getRVertPtr(face.v[vi]);
                MeshNormalSpec * meshNorm = mesh->GetSpecifiedNormals();

                // If normal is not specified it's only available if the face belongs
                // to a smoothing group
                if (((!meshNorm || (meshNorm->GetNumNormals() == 0)) ||                     // No UNIFIED normal has been specified for ANY corner (ignores smoothing groups)
                    !meshNorm->GetNormalExplicit(meshNorm->GetNormalIndex(faceIdx, vi)))    // The current normal has not been explicitly MODIFIED via "Edit Normals" or otherwise
                    && ((rv->rFlags & SPECIFIED_NORMAL) == 0))                              // The specified normal flag is not set
                {
                    // then calculated from smoothing groups
                    int numNormals = rv->rFlags & NORCT_MASK;

                    if (numNormals && face.smGroup)
                    {
                        // If there is only one vertex is found in the rn member.
                        if (numNormals == 1)
                        {
                            m_normalSmGrp[vi] = rv->rn.getSmGroup();
                        }
                        else
                        {
                            for (int i = 0; i < numNormals; i++)
                            {
                                if (rv->ern[i].getSmGroup() & face.smGroup)
                                {
                                    m_normalSmGrp[vi] = rv->ern[i].getSmGroup();
                                }
                            }
                        }
                    }
                }
            }
        }

        m_faceSmGrp = face.smGroup;
        m_matId = face.getMatID();
        m_flags = face.flags;
        m_meshFaceIndex = faceIdx;
    }
};

bool hctMaxMeshWriter::getMeshChannelName(INode* inode, int channelId, MSTR& nameOut)
{
    MSTR propName; propName.printf(TEXT("MapChannel:%i"), channelId);

    return (inode->GetUserPropString(propName, nameOut) != 0);
}



void hctMaxMeshWriter::getAllExportChannels(hkArray<ExportChannel>& exportChannelsOut)
{
    Object* curObject = m_inode->GetObjectRef();
    if (!curObject) return;

    hkArray<hkxAttributeGroup> groups;
    m_owner->exportAttributes(curObject, groups);

    for (int i = 0; i < groups.getSize(); ++i)
    {
        const hkxAttributeGroup& group = groups[i];
        if (hkString::strCmp(group.m_name, "hkExportChannelModifier") == 0)
        {
            ExportChannel& expChann = exportChannelsOut.expandOne();

            {
                hkxSparselyAnimatedString* sas = group.findStringAttributeByName("channelExportName");
                if (sas->m_strings.getSize() == 1)
                {
                    expChann.m_channnelExportName = sas->m_strings[0].cString();
                }
            }
            {
                hkxSparselyAnimatedInt* sai = group.findIntAttributeByName("channelID");
                if (sai->m_ints.getSize() == 1)
                {
                    expChann.m_channelID = sai->m_ints[0];
                }
            }
            {
                hkxSparselyAnimatedInt* sai = group.findIntAttributeByName("channelType");
                if (sai->m_ints.getSize() == 1)
                {
                    expChann.m_channelType = sai->m_ints[0];
                }
            }
            bool rescale = false;
            {
                hkxSparselyAnimatedBool* spb = group.findBoolAttributeByName("rescale");
                if (spb->m_bools.getSize() == 1)
                {
                    rescale = spb->m_bools[0];
                }
            }
            if (rescale)
            {
                const bool isDistance = (expChann.m_channelType == HAVOK_CHANNEL_DISTANCE);
                const char* rescaleMinName = (isDistance) ? "rescaleMinD" : "rescaleMinF";
                const char* rescaleMaxName = (isDistance) ? "rescaleMaxD" : "rescaleMaxF";

                hkxAnimatedFloat* af = group.findFloatAttributeByName(rescaleMinName);
                if (af->m_floats.getSize() == 1)
                {
                    expChann.m_rescaleMin = af->m_floats[0];
                }
                af = group.findFloatAttributeByName(rescaleMaxName);
                if (af->m_floats.getSize() == 1)
                {
                    expChann.m_rescaleMax = af->m_floats[0];
                }
            }
            else
            {
                expChann.m_rescaleMin = .0f;
                expChann.m_rescaleMax = 1.0f;
            }
        }
    }
}

bool _compareChannelsNames(const ExportChannel& a, const ExportChannel& b)
{
    return (hkString::strCmp(a.m_channnelExportName.cString(), b.m_channnelExportName.cString()) < 0);
}

void _checkForDuplicateExportChannelsAndWarn(hkArray<ExportChannel>& channels)
{
    // Sort channels by name
    hkAlgorithm::quickSort(channels.begin(), channels.getSize(), _compareChannelsNames);

    // Warn if some duplicates are found
    for (int i = 1; i < channels.getSize(); ++i)
    {
        bool warned = false;
        while ((i < channels.getSize()) && (hkString::strCmp(channels[i].m_channnelExportName.cString(), channels[i - 1].m_channnelExportName.cString()) == 0))
        {
            if (!warned)
            {
                HK_WARN_ALWAYS(0x571e715b, "More than one Export Channel with name \"" << channels[i - 1].m_channnelExportName.cString() << "\" found. Export Channel names should be unique.");
                warned = true;
            }
            ++i;
        }
    }
}


static void addHiddenEdgesChannel(const Mesh* mesh, hctUserChannelUtil& userChannelUtil)
{
    hctUserChannelUtil::GlobalChannel invisibleEdgeChannel;
    invisibleEdgeChannel.m_channelName = "hidden_edges";
    invisibleEdgeChannel.m_channelType = hctUserChannelUtil::CT_EDGE_SELECTION;

    for (int fi = 0; fi < mesh->numFaces; fi++)
    {
        const Face& face = mesh->faces[fi];
        const int faceIndex = 3 * fi;

        if (!(face.flags & EDGE_A))
        {
            hctUserChannelUtil::ChannelDataItem item;
            item.m_index = faceIndex;
            invisibleEdgeChannel.m_channelData.pushBack(item);
        }

        if (!(face.flags & EDGE_B))
        {
            hctUserChannelUtil::ChannelDataItem item;
            item.m_index = faceIndex + 1;
            invisibleEdgeChannel.m_channelData.pushBack(item);
        }

        if (!(face.flags & EDGE_C))
        {
            hctUserChannelUtil::ChannelDataItem item;
            item.m_index = faceIndex + 2;
            invisibleEdgeChannel.m_channelData.pushBack(item);
        }
    }

    // add hidden edges if found any
    if (invisibleEdgeChannel.m_channelData.getSize() > 0)
    {
        userChannelUtil.addGlobalChannel(invisibleEdgeChannel);
    }
}


hkxVertexAnimation* hctMaxMeshWriter::getVertexAnimation(INode* pNode, const hkxMeshSection* section, hkxVertexAnimationStateCache& c, const hctUserChannelUtil::SectionToGlobalMap& dataMap, int sampleTicks)
{
    // As we do this per material section, we chould prob move the eval state of the mesh up to per node only later as an optimization
    ObjectState state = pNode->EvalWorldState(sampleTicks);
    TriObject *triangleObject = dynamic_cast<TriObject*>(state.obj->ConvertToType(sampleTicks, triObjectClassID));
    TriObject *triangleObjectToDelete = HK_NULL;
    if (triangleObject != state.obj)
        triangleObjectToDelete = triangleObject;

    Mesh* meshAtCurTime = &triangleObject->GetMesh();
    meshAtCurTime->checkNormals(true); // allocate rverts and calc normals
    const int maxNumVerts = meshAtCurTime->getNumVerts();

    const Matrix3 localToWorld = pNode->GetNodeTM(sampleTicks);
    const Matrix3 objectToWorld = pNode->GetObjTMAfterWSM(sampleTicks);
    const Matrix3 objectToLocal = objectToWorld * Inverse(localToWorld);

    // EXP-387
    // The normals given by the mesh are in object space
    // but we want them in local space
    Matrix3 normalTransform;
    {
        // Normals are transformed by the inverse transposed
        Matrix3 temp = objectToLocal;
        temp.SetTrans(Point3(0, 0, 0)); // no translation
        Matrix3 inverse = Inverse(temp);
        Matrix3& inverseTransposed = normalTransform;
        inverseTransposed.SetRow(0, inverse.GetColumn3(0));
        inverseTransposed.SetRow(1, inverse.GetColumn3(1));
        inverseTransposed.SetRow(2, inverse.GetColumn3(2));
        inverseTransposed.SetRow(3, Point3(0, 0, 0));
    }

    hkxVertexAnimation* vanim = HK_NULL;

    const hkxVertexBuffer& curVertexState = c.getState();
    const hkxVertexDescription& curVertexDesc = curVertexState.getVertexDesc();

    // The hkxVertexAnimaion can store animate points for any data (uvs, colors, etc)
    // The orig vMax2 etc exporters only supported Pos and Normal, so we will just do the same here
    const hkxVertexDescription::ElementDecl* posDecl = curVertexDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
    const hkxVertexDescription::ElementDecl* normalDecl = curVertexDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
    const hkVector4* curPosDataPtr = (hkVector4*)curVertexState.getVertexDataPtr(*posDecl);
    const hkVector4* curNormalDataPtr = (hkVector4*)curVertexState.getVertexDataPtr(*normalDecl);
    int posStride = posDecl->m_byteStride / sizeof(hkVector4);
    int normalStride = normalDecl->m_byteStride / sizeof(hkVector4);

    // Note we try to extract the information here from the same Mesh type (Mesh not MNMesh say) with the same mappings
    int numVerts = dataMap.m_sectionVertexIdToGlobalVertexId.getSize();

    hkArray<hkVector4> animatedPos;
    hkArray<hkVector4> animatedNormal;
    hkArray<int> animatedVid;

    // Need faceid to get proper smoothing group flags etc
    hkArray<int> faceIdPerVert;
    faceIdPerVert.setSize(numVerts, -1);
    for (int fi = 0; fi < section->m_indexBuffers[0]->getNumTriangles(); fi++)
    {
        int localTriIndex = fi;
        int globalTriIndex = dataMap.m_sectionTriangleIdToGlobalFaceId[localTriIndex];

        // set globalTriIndex on the three verts using this tri (we know they are split for smooth groups before now anyway for hKSceneData etc)
        int va, vb, vc;
        if (section->m_indexBuffers[0]->m_indices16.getSize())
        {
            va = section->m_indexBuffers[0]->m_indices16[fi * 3];
            vb = section->m_indexBuffers[0]->m_indices16[fi * 3 + 1];
            vc = section->m_indexBuffers[0]->m_indices16[fi * 3 + 2];
        }
        else
        {
            va = section->m_indexBuffers[0]->m_indices32[fi * 3];
            vb = section->m_indexBuffers[0]->m_indices32[fi * 3 + 1];
            vc = section->m_indexBuffers[0]->m_indices32[fi * 3 + 2];
        }

        faceIdPerVert[va] = globalTriIndex;
        faceIdPerVert[vb] = globalTriIndex;
        faceIdPerVert[vc] = globalTriIndex;
    }


    // Work put minimal set of animated verts
    for (int vi = 0; vi < numVerts; vi++)
    {
        int mainMeshVertId = dataMap.m_sectionVertexIdToGlobalVertexId[vi];
        if ((mainMeshVertId < 0) || (mainMeshVertId >= maxNumVerts))
        {
            // now out of range.. topology changed?
            HK_WARN_ALWAYS(0xabba0000, "Inconsistent topology during animated vert export");
            if (vanim) delete vanim;
            return HK_NULL;
        }

        const Point3 posObject = meshAtCurTime->getVert(mainMeshVertId); // objectSpace
        const Point3 posLocal = posObject * objectToLocal;

        RVertex* rv = meshAtCurTime->getRVertPtr(mainMeshVertId);
        int faceId = faceIdPerVert[vi];
        HK_ASSERT_NO_MSG(0x43762610, faceId >= 0);
        Point3 normal = _getObjectVertexNormal(meshAtCurTime, faceId, rv, 0); //XX Explicit Normal support will not work here with animated verts (used?)
        normal = normalTransform * normal; // EXP-387

        // If normal or pos != last anim pos for that vert then is new vert
        const hkVector4* curPos = curPosDataPtr + posStride*vi;
        const hkVector4* curNormal = curNormalDataPtr + normalStride*vi;
        hkVector4 animPos; animPos.set(posLocal.x, posLocal.y, posLocal.z);
        hkVector4 animNormal; animNormal.set(normal.x, normal.y, normal.z);

        // assume if one different, both are, just to keep code down
        if (!animPos.equals3(*curPos) || !animNormal.equals3(*curNormal))
        {
            animatedPos.pushBack(animPos);
            animatedNormal.pushBack(animNormal);
            animatedVid.pushBack(vi);
        }
    }

    //do we have any, if so, create new hkxVertexAnimation
    if (animatedVid.getSize() > 0)
    {
        vanim = new hkxVertexAnimation();
        {
            hkxVertexDescription reqdesc;
            reqdesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_POSITION, hkxVertexDescription::HKX_DT_FLOAT, 3));
            reqdesc.m_decls.pushBack(hkxVertexDescription::ElementDecl(hkxVertexDescription::HKX_DU_NORMAL, hkxVertexDescription::HKX_DT_FLOAT, 3));
            vanim->m_vertData.setNumVertices(animatedVid.getSize(), reqdesc);

            // Associate our P and N with the original ones:
            vanim->m_componentMap.setSize(2);
            vanim->m_componentMap[0].m_use = hkxVertexDescription::HKX_DU_POSITION;
            vanim->m_componentMap[0].m_useIndexLocal = 0;
            vanim->m_componentMap[0].m_useIndexOrig = 0;

            vanim->m_componentMap[1].m_use = hkxVertexDescription::HKX_DU_NORMAL;
            vanim->m_componentMap[1].m_useIndexLocal = 0;
            vanim->m_componentMap[1].m_useIndexOrig = 0;
        }

        const hkxVertexDescription& curAnimDesc = vanim->m_vertData.getVertexDesc();
        const hkxVertexDescription::ElementDecl* aposDecl = curAnimDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
        const hkxVertexDescription::ElementDecl* anormalDecl = curAnimDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
        hkVector4* aPosDataPtr = (hkVector4*)vanim->m_vertData.getVertexDataPtr(*aposDecl);
        hkVector4* aNormalDataPtr = (hkVector4*)vanim->m_vertData.getVertexDataPtr(*anormalDecl);
        int aposStride = aposDecl->m_byteStride / sizeof(hkVector4);
        int anormalStride = anormalDecl->m_byteStride / sizeof(hkVector4);

        vanim->m_vertexIndexMap.setSize(animatedVid.getSize());
        for (int ai = 0; ai < animatedVid.getSize(); ++ai)
        {
            vanim->m_vertexIndexMap[ai] = animatedVid[ai];
            hkVector4* destP = aPosDataPtr + (ai * aposStride);
            hkVector4* destN = aNormalDataPtr + (ai * anormalStride);
            const hkVector4& srcP = animatedPos[ai];
            const hkVector4& srcN = animatedNormal[ai];
            *destP = srcP;
            *destN = srcN;

        }
    }

    // Cleanup

    if (triangleObjectToDelete)
    {
        delete triangleObjectToDelete;
    }


    return vanim;

}

void hctMaxMeshWriter::createMesh(hkxMesh*& newMesh, hkArray<float>* weights, hkArray<int>* indices, Matrix3* worldTM, bool useInt8BoneIndices)
{
    // EXP-315. We could also make it optional
    const bool optimizeMatIds = true;
    int maxMaterialIds = 1000000;

    Mtl* nodeMaterial = hctMaxUtils::getViewportMaterial(m_inode->GetMtl());

    if (optimizeMatIds)
    {
        maxMaterialIds = _maxMatIds(nodeMaterial);
    }

    // This tells 3ds max to get normal data ready, as we will query it
    m_mesh->checkNormals(true);

    newMesh = HK_NULL;
    int numFaces = m_mesh->getNumFaces();
    if (numFaces < 1) return;

    hkArray<int> keyListTicks;
    hkArray<int> linearKeyFrameTicks;
    int tickPerSecond = GetTicksPerFrame() * GetFrameRate();
    if (m_exportOptions.m_exportVertexAnimations)
    {
        // EXP-2435 save key frames
        int startFrame = hctMaxUtils::getFramesFromTicks(m_exportOptions.m_animationStart);
        int endFrame = hctMaxUtils::getFramesFromTicks(m_exportOptions.m_animationEnd);

        if (m_exportOptions.m_forceVertexAnimationSamplePoints.getSize() > 0)
        {
            keyListTicks.insertAt(0, m_exportOptions.m_forceVertexAnimationSamplePoints.begin(), m_exportOptions.m_forceVertexAnimationSamplePoints.getSize());
            hctMaxUtils::getAnimatableKeyTimes(m_inode, linearKeyFrameTicks, startFrame, endFrame);

            // EXP-2838 Add key frames as well as m_forceVertexAnimationSamplePoints may not correlate with the key frames at all.
            keyListTicks.insertAt(0, linearKeyFrameTicks.begin(), linearKeyFrameTicks.getSize());
            hkSceneExportUtils::sortAndRemoveDuplicates(keyListTicks);
        }
        else // just key base
        {
            hctMaxUtils::getAnimatableKeyTimes(m_inode, keyListTicks, startFrame, endFrame);
        }
    }

    std::deque<FullVert> fullVerts;

    // This will reorder the face list into face usage order (better for caches etc anyway)
    // and reduced the indices to a list of unique full verts.
    BOOL vcSupport = m_mesh->mapSupport(0); // 0 = Vertex Color
    BOOL alphaSupport = m_mesh->mapSupport(MAP_ALPHA);
    bool noDefaultTCoords = m_mesh->numTVerts == 0;
    bool noVColors = !vcSupport;
    bool noAColors = !alphaSupport;

    hkArray<MtlID> matIds; _getMaterialIdsInMesh(m_mesh, maxMaterialIds, matIds);

    hkArray<DWORD> faceChannels;

    // Each matId maps to a mesh section.
    hkArray< hkxMeshSection* > exportedSections;
    exportedSections.reserve(matIds.getSize());

    // Vertex Selection
    hctUserChannelUtil userChannelUtil;

    int anyExportedMapID = -1;
    // Get the vertex float channels marked for export
    {
        hkArray<ExportChannel> exportChannels;
        getAllExportChannels(exportChannels);
        // warn if some export channels names are duplicated
        _checkForDuplicateExportChannelsAndWarn(exportChannels);

        const int numChannels = exportChannels.getSize();
        for (int ind = 0; ind < numChannels; ++ind)
        {
            const ExportChannel& expChannel = exportChannels[ind];
            const int channelID = expChannel.m_channelID;

            // if the channel is supported
            if (m_mesh->mapSupport(channelID))
            {
                // get number of values
                int numValues = m_mesh->getNumMapVerts(channelID);
                if (numValues > 0)
                {
                    anyExportedMapID = channelID; // any supported map
                    const float scaleMin = expChannel.m_rescaleMin;
                    const float scaleMax = expChannel.m_rescaleMax;

                    // Create a new user channel for the float array
                    hctUserChannelUtil::GlobalChannel clothChannel;
                    clothChannel.m_channelName = expChannel.m_channnelExportName;
                    clothChannel.m_channelType = hctUserChannelUtil::CT_SAMPLE_FLOAT;
                    clothChannel.m_scaleMin = scaleMin;
                    clothChannel.m_scaleMax = scaleMax;

                    // set channel dimensions
                    switch (expChannel.m_channelType)
                    {
                    case HAVOK_CHANNEL_FLOAT:
                        clothChannel.m_channelDimensions = hctUserChannelUtil::CD_FLOAT;
                        break;
                    case HAVOK_CHANNEL_DISTANCE:
                        clothChannel.m_channelDimensions = hctUserChannelUtil::CD_DISTANCE;
                        break;
                    case HAVOK_CHANNEL_ANGLE:
                        clothChannel.m_channelDimensions = hctUserChannelUtil::CD_ANGLE;
                        break;
                    default:
                        clothChannel.m_channelDimensions = hctUserChannelUtil::CD_INVALID;
                    }

                    // resize the user channel array
                    clothChannel.m_channelData.setSize(numValues);

                    // get the pointer to the values array
                    UVVert* values = m_mesh->mapVerts(channelID);
                    for (int i = 0; i < numValues; ++i)
                    {
                        float value = (values[i].x + values[i].y + values[i].z) / 3.0f;

                        // HCL-1804 - Sometimes max gives us weird values....
                        if (value < 0.0f) value = 0.0f;
                        if (value > 1.0f) value = 1.0f;

                        clothChannel.m_channelData[i].m_float = value;

                    }
                    // add the channel to the mesh
                    userChannelUtil.addGlobalChannel(clothChannel);
                }
            }
        }
    }

    // Fill global (mesh) channels
    addSelectionChannels(userChannelUtil);

    // XXX testing edge selection
    addHiddenEdgesChannel(m_mesh, userChannelUtil);

    for (int curMat = 0; curMat < matIds.getSize(); ++curMat)
    {
        // See if we have a material to export for this material section
        // This will also see what channel(s) we want to export for this section
        // as the material will be mapped to one or more (if mulitple slots used)
        hkxMaterial* sectMat = HK_NULL;
        hkArray<int> usedChannels;

        if (m_exportOptions.m_exportMaterials)
        {
            Mtl* theMaterial = nodeMaterial;
            hkArray<int> blends;
            if (theMaterial)
            {
                Class_ID theId = theMaterial->ClassID();

                // EXP-365 - We don't support stand-in materials
                if (theId.PartA() != STANDIN_CLASS_ID)
                {
                    hkArray<Mtl*> mats; // May be more than 1 mat that will statisfy a matId. Say Blend of a lightmap with a multimap for the textures etc is common for quick prelit scenes
                    hctMaxUtils::findSubMaterials(theMaterial, matIds[curMat], mats, blends);

                    createMaterial(theMaterial, mats, blends, sectMat, usedChannels);
                }
            }
        }

        // [HS#9419] There used to be an else on this line (with regard to "if (theMaterial)" but that meant that UV channels used in the mesh which weren't used in the material would
        // not be added to the usedChannels array, which meant that they would not end up in the UV export pipeline.
        for (int i = 1; i <= MAX_EXPORTED_CHANNELS; i++)
        {
            if (m_mesh->mapSupport(i) && (usedChannels.indexOf(i) == -1))
            {
                usedChannels.pushBack(i);
            }
        }

        if (!m_exportOptions.m_exportWireColorsAsVertexColors)
        {
            // If we don't have any material at all, then the object is displayed in Max with the object color
            // so we can use that
            if (!sectMat)
            {
                HCT_SCOPED_CONVERSIONS;

                sectMat = new hkxMaterial();
                hkStringBuf n = "wirecolor_"; n.append(FROM_MAX(m_inode->GetName()));
                sectMat->m_name = n.cString();
                DWORD wc = m_inode->GetWireColor();

                getSpecularParameters(*sectMat, nodeMaterial);

                // BGRA not RGBA in wc, our macros assume RGBA, hence the flip here:
                sectMat->m_diffuseColor.set(hkColor::getBlueAsFloat(wc), hkColor::getGreenAsFloat(wc), hkColor::getRedAsFloat(wc), 1.0);
                sectMat->m_ambientColor.setAll(0);
                sectMat->m_specularColor = sectMat->m_diffuseColor;
                sectMat->m_specularColor(3) = 75.0f; // spec power // Not used in Vision; kept for backwards compatibility
                sectMat->m_emissiveColor.setAll(0);
                sectMat->m_uvMapAlgorithm = hkxMaterial::UVMA_3DSMAX_STYLE;
                sectMat->m_uvMapOffset[0] = 0; sectMat->m_uvMapOffset[1] = 0;
                sectMat->m_uvMapScale[0] = 1.f; sectMat->m_uvMapScale[1] = 1.f;
                sectMat->m_uvMapRotation = 0;
                sectMat->m_userData = 0;

                m_owner->m_currentScene.m_materials.pushBack(sectMat); // add to the master top level list
            }
        }

        bool defaultTextureChannel = false; // XXXTEST (!noDefaultTCoords) && (usedChannels.Count() == 1) && (usedChannels[0] == 1); // ch 1 == default tcoords
        int maxUsedChannels = usedChannels.getSize(); // Export all channels, (used to clamp here)

        fullVerts.resize(0); //XX TODO: Make only one vertex buffer per mesh

        hctUserChannelUtil::SectionToGlobalMap sectionToGlobalMap;

        hkArray<hkArray<int> > vertsToFullVerts;
        vertsToFullVerts.setSize(m_mesh->numVerts);

        hkArray<int> faceVertRemap;
        faceVertRemap.reserve(numFaces * 3);
        faceVertRemap.setSize(0);

        for (int f = 0;f < numFaces;f++)
        {
            MyFaceEx faceInfo(m_mesh, f);

            // per material
            if ((faceInfo.m_matId % maxMaterialIds) != matIds[curMat])
            {
                continue;
            }

            if (!defaultTextureChannel)
            {
                faceChannels.setSize(3 * maxUsedChannels, 0);
                for (int chi = 0; chi < maxUsedChannels; ++chi)
                {
                    if (m_mesh->mapSupport(usedChannels[chi]))
                    {
                        MeshMap& map = m_mesh->maps[usedChannels[chi]];
                        for (int i = 0; i < 3; i++)
                        {
                            faceChannels[chi * 3 + i] = map.tf[f].t[i];
                        }
                    }
                    else
                    {
                        for (int i = 0; i < 3; i++)
                        {
                            faceChannels[chi * 3 + i] = 0;
                        }
                    }
                }
            }

            for (int j = 0; j < 3; ++j)
            {
                const int vertexIndex = faceInfo.m_vertIdx[j];

                hkArray<int>& fullVertsWithThatIndex = vertsToFullVerts[vertexIndex];

                bool reusedVertex = false;
                for (int fvi = 0; fvi < fullVertsWithThatIndex.getSize(); fvi++)
                {
                    const int fullVertexIndex = fullVertsWithThatIndex[fvi];

                    // EXP- : Do not split
                    if (m_owner->m_exportOptions.m_doNotSplitVertices)
                    {
                        faceVertRemap.pushBack(fullVertexIndex);
                        reusedVertex = true;
                        break;
                    }

                    FullVert& fv = fullVerts[fullVertexIndex];
                    if ((noVColors || (fv.vertColorIndex == faceInfo.m_vcolorIdx[j]) || (m_mesh->vertCol[fv.vertColorIndex].Equals(m_mesh->vertCol[faceInfo.m_vcolorIdx[j]]))) &&
                        (noAColors || (fv.alphaIndex == faceInfo.m_alphaIdx[j])) &&
                        ((fv.m_smoothingGroup != 0) && (fv.m_smoothingGroup & faceInfo.m_normalSmGrp[j])) && // EXP-798. group=0 means unique, never merge (which it is for Explict Normals)
                        ((!m_owner->m_exportOptions.m_exportMergeVertsOnlyIfSameSrcId) || (fv.vertIndex == faceInfo.m_vertIdx[j]))) // EXP-2312 : Preserve different src vert indices on Vision export
                    {
                        // check default tex coords
                        bool tcoordsOk = false;
                        if (defaultTextureChannel)
                        {
                            tcoordsOk = (fv.texCoordIndex == faceInfo.m_texCoordIdx[j]);
                        }
                        else
                        {
                            // has to have same map uv in all channels used.
                            tcoordsOk = true;
                            for (int chi = 0; tcoordsOk && (chi < maxUsedChannels); ++chi)
                            {
                                const int ch = usedChannels[chi];
                                const UVVert* uvVerts = m_mesh->mapVerts(ch);
                                if (uvVerts)
                                {
                                    const int chIndex = faceChannels[(3 * chi) + j];
                                    if (chIndex < m_mesh->getNumMapVerts(ch))
                                    {
                                        UVVert mapChannelA = uvVerts[chIndex];
                                        if ((int)fv.mapChannels[chi] < (int)m_mesh->getNumMapVerts(ch))
                                        {
                                            UVVert mapChannelB = uvVerts[fv.mapChannels[chi]];
                                            tcoordsOk &= (bool)hkMath::equal(mapChannelA[0], mapChannelB[0], 0.001f); // U
                                            tcoordsOk &= (bool)hkMath::equal(mapChannelA[1], mapChannelB[1], 0.001f); // V    , don't use W
                                        }
                                        else
                                        {
                                            tcoordsOk = false;
                                        }
                                    }
                                    else
                                    {
                                        tcoordsOk = false;
                                    }
                                }
                            }
                        }

                        if (tcoordsOk)
                        {
                            // don't need a new one, continue on;
                            faceVertRemap.pushBack(fullVertexIndex);
                            reusedVertex = true;
                            break;
                        }
                    }
                }

                if (!reusedVertex)
                {
                    // need a new one
                    fullVerts.push_back(FullVert());
                    FullVert& nv = fullVerts.back();

                    nv.vertIndex = faceInfo.m_vertIdx[j];
                    nv.texCoordIndex = noDefaultTCoords ? -1 : faceInfo.m_texCoordIdx[j];
                    nv.vertColorIndex = noVColors ? -1 : faceInfo.m_vcolorIdx[j];
                    nv.alphaIndex = noAColors ? -1 : faceInfo.m_alphaIdx[j];
                    nv.faceIndex = f;
                    nv.faceVertexIndexHint = j;
                    nv.m_smoothingGroup = faceInfo.m_normalSmGrp[j];
                    nv.normalId = -1;

                    nv.mapChannels.setSize(maxUsedChannels);
                    if (defaultTextureChannel)
                    {
                        for (int chi = 0; chi < maxUsedChannels; chi++)
                        {
                            nv.mapChannels[chi] = 0;
                        }
                    }
                    else
                    {
                        for (int chi = 0; chi < maxUsedChannels; chi++)
                        {
                            nv.mapChannels[chi] = faceChannels[j + (3 * chi)];
                        }
                    }

                    const int fullVertexIndex = (int)fullVerts.size() - 1;
                    faceVertRemap.pushBack(fullVertexIndex);
                    fullVertsWithThatIndex.pushBack(fullVertexIndex);

                    sectionToGlobalMap.m_sectionVertexIdToGlobalVertexId.pushBack(faceInfo.m_vertIdx[j]);

                    // save the index of the corresponding value in the samples data array.
                    // This will be used to retrieve the color value for each channel.
                    if (anyExportedMapID >= 0)
                    {
                        const int idx = m_mesh->mapFaces(anyExportedMapID)[f].getTVert(j);
                        sectionToGlobalMap.m_sectionVertexIdToGlobalSampleId.pushBack(idx);
                    }
                }

            }

            sectionToGlobalMap.m_sectionTriangleIdToGlobalFaceId.pushBack(f);
        }

        if (faceVertRemap.getSize() < 1)
            continue; // nothing to do..

        //
        // Construct unique normal IDs from smoothing groups. These will be stored in a vertex int channel in the mesh, to
        // encode the mesh smoothing group information (for rendering of hard edges).
        //
        sectionToGlobalMap.m_sectionVertexIdToGlobalNormalId.setSize((int)fullVerts.size());
        hkUint16 currentNormalID = 0;

        for (int vi = 0; vi < m_mesh->numVerts; ++vi)
        {
            const hkArray<int>& fullVertsWithThatIndex = vertsToFullVerts[vi];

            // If there is only one full (exported) vertex corresponding to this geometric vertex, it gets a unique normal ID.
            if (fullVertsWithThatIndex.getSize() == 1)
            {
                FullVert& fv = fullVerts[fullVertsWithThatIndex[0]];
                fv.normalId = currentNormalID;
                currentNormalID++;
            }
            else
            {
                // Iterate through all the full (exported) vertices corresponding to the same geometric vertex, and assign them normal IDs based on their smoothing groups.
                // (Note that this algorithm does not necessarily cause all faces connected "transitively" by smoothing group bits to be smoothed, it depends on the
                // order in which faces are visited, but Max does the same thing in computing its smoothed normals).
                int fvi_current = -1;
                while (1)
                {
                    // Find first full vertex with unset normal ID
                    bool found = false;
                    for (int fvi = fvi_current + 1; fvi < fullVertsWithThatIndex.getSize(); fvi++)
                    {
                        FullVert& fv = fullVerts[fullVertsWithThatIndex[fvi]];
                        if (fv.normalId == -1)
                        {
                            fvi_current = fvi;
                            found = true;
                            break;
                        }
                    }

                    // If all full vertices have had normal IDs assigned already, we're done
                    if (!found) break;

                    FullVert& currentfv = fullVerts[fullVertsWithThatIndex[fvi_current]];
                    currentfv.normalId = currentNormalID;

                    // Find all other full vertices (with the same geometric index) which should be smoothed with the current one
                    for (int fvi = fvi_current + 1; fvi < fullVertsWithThatIndex.getSize(); fvi++)
                    {
                        FullVert& fv = fullVerts[fullVertsWithThatIndex[fvi]];
                        if ((fv.m_smoothingGroup & currentfv.m_smoothingGroup) && (fv.normalId == -1))
                        {
                            fv.normalId = currentNormalID;
                        }
                    }

                    currentNormalID++;
                }
            }

            // Store the normal IDs in hctUserChannelUtil::SectionToGlobalMap::m_sectionVertexIdToGlobalNormalId.
            for (int fvi = 0; fvi < fullVertsWithThatIndex.getSize(); fvi++)
            {
                const int fullVertexIndex = fullVertsWithThatIndex[fvi];
                FullVert& fv = fullVerts[fullVertexIndex];
                sectionToGlobalMap.m_sectionVertexIdToGlobalNormalId[fullVertexIndex] = hkUint16(fv.normalId);
            }
        }

        hkxMeshSection* newSection = new hkxMeshSection();

        // Vertex buffer
        hkxVertexBuffer* newVB;
        createVertexBuffer(newVB, fullVerts, weights, indices, &usedChannels, !!sectMat, worldTM, useInt8BoneIndices);

        hkxIndexBuffer* newIB = new hkxIndexBuffer();
        bool need32BitsIndices = newVB->getNumVertices() > (int)0x0ffff; // if 16bits can't index
        newIB->m_indexType = hkxIndexBuffer::INDEX_TYPE_TRI_LIST; // XX todo, add strips (or a strip filter at least)
        newIB->m_vertexBaseOffset = 0;
        newIB->m_length = faceVertRemap.getSize();
        if (need32BitsIndices)
        {
            newIB->m_indices32.setSize(newIB->m_length);
        }
        else
        {
            newIB->m_indices16.setSize(newIB->m_length);
        }

        if (need32BitsIndices)
        {
            hkUint32* curIndex = newIB->m_indices32.begin();
            for (unsigned i = 0; i < newIB->m_length; i++)
            {
                *curIndex = (hkUint32)faceVertRemap[i];
                curIndex++;
            }
        }
        else
        {
            hkUint16* curIndex = newIB->m_indices16.begin();
            for (unsigned i = 0; i < newIB->m_length; i++)
            {
                *curIndex = (hkUint16)faceVertRemap[i];
                curIndex++;

            }
        }

        // write the actual sections to link the buffers together
        newSection->m_material = sectMat;
        newSection->m_vertexBuffer = newVB;
        newSection->m_indexBuffers.setSize(1);
        newSection->m_indexBuffers[0] = newIB;
        exportedSections.pushBack(newSection);
        // max doesnt have non triangular faces so global triangle ids are identical to global face ids
        sectionToGlobalMap.m_sectionTriangleIdToGlobalTriangleId = sectionToGlobalMap.m_sectionTriangleIdToGlobalFaceId;
        userChannelUtil.registerSection(sectionToGlobalMap);

        if (sectMat)
            sectMat->removeReference();
        newVB->removeReference();
        newIB->removeReference();
    }

    // Create a user channel of per-vertex ints, to associate each exported vertex with a unique normal index (the normal 'ids').
    // Vertices which share a normal id we know should be rendered with the same normal, even though they may be exported as different vertices due
    // to differing UVs. This is used in Cloth to preserve hard/soft edges. A vertex int channel with the name "normalIds" is expected to be present by Cloth.
    // Note that each section will have in general completely different normal IDs. So this is not like a float channel, where vertices in different
    // sections corresponding to the same geometric vertex get the same float value as stored per-geometric-vertex in the channel data. Here, the
    // int is stored in a per-section map hctUserChannelUtil::SectionToGlobalMap::m_sectionVertexIdToGlobalNormalId, not in hctUserChannelUtil::GlobalChannel::m_channelData.
    // We use the channel type CT_NORMAL_IDS for this specific case.
    hctUserChannelUtil::GlobalChannel normalIdChannel;
    normalIdChannel.m_channelName = "normalIds";
    normalIdChannel.m_channelType = hctUserChannelUtil::CT_NORMAL_IDS;
    userChannelUtil.addGlobalChannel(normalIdChannel);

    if (exportedSections.getSize() < 1)
    {
        newMesh = HK_NULL;
        return; // could not write any of them
    }

    newMesh = new hkxMesh();
    newMesh->m_sections.setSize(exportedSections.getSize());
    for (int cs = 0; cs < newMesh->m_sections.getSize(); ++cs)
    {
        newMesh->m_sections[cs] = exportedSections[cs];
        exportedSections[cs]->removeReference();
    }

    userChannelUtil.storeChannelsInMesh(newMesh);

    // The animation query will Eval the INode at different times and invalidate current m_mesh state
    if (keyListTicks.getSize() > 0)
    {
        // A bit nasty that we are hijacking the usersectionutil info here, should split it out
        const hkArray<hctUserChannelUtil::SectionToGlobalMap>& globalMaps = userChannelUtil.getSectionGlobalMaps();

        // Potential optimization: if we reverse this so we iter over keys, then sections we would save on redundant Evals on the INodes
        for (int si = 0; si < newMesh->m_sections.getSize(); ++si)
        {
            hkxVertexAnimationStateCache vcache(newMesh->m_sections[si]->m_vertexBuffer, false); // copy
            for (int ki = 0; ki < keyListTicks.getSize(); ++ki)
            {
                hkxVertexAnimation* vanim = getVertexAnimation(m_inode, newMesh->m_sections[si], vcache, globalMaps[si], keyListTicks[ki]);
                if (vanim)
                {
                    int exportRelativeTicks = keyListTicks[ki] - m_exportOptions.m_animationStart;
                    vanim->m_time = (float)exportRelativeTicks / (float)tickPerSecond;
                    newMesh->m_sections[si]->m_vertexAnimations.pushBack(vanim);
                    vanim->removeReference();

                    // make it the current state
                    vcache.apply(vanim);
                }
            }

            // EXP-2435 store list in mesh section
            const int numLinearKeyFrames = linearKeyFrameTicks.getSize();
            for (int ki = 0; ki < numLinearKeyFrames; ki++)
            {
                float keyFrameTime = (linearKeyFrameTicks[ki] - m_exportOptions.m_animationStart) / (float)tickPerSecond;
                newMesh->m_sections[si]->m_linearKeyFrameHints.pushBack(keyFrameTime);
            }

        }
        // the EvalState in the getVertexAnimation will have invalidated our m_mesh, so make sure not able to use again now:
        m_mesh = HK_NULL;
    }

    // Finally, we can compute tangents. Note that tangent computation, as there may be mirrored verts (so verts that share the
    // same uv, normals etc, but that have divergent computed normals (due to uv mirror seam down the middle of a character for instance)
    if (m_owner->m_exportOptions.m_exportVertexTangents)
    {
        HCT_SCOPED_CONVERSIONS;

        hkxMeshSectionUtil::computeTangents(newMesh, true, FROM_MAX(m_inode->GetName()));
    }
}

void hctMaxMeshWriter::addSelectionChannels(hctUserChannelUtil& userChannelUtil)
{
    HCT_SCOPED_CONVERSIONS;

    // 1b - Add vertex soft selection (if any)

    // Removed (see EXP-1043) since soft selection is not fully supported as exportable data by max
    /*
    if (m_mesh->vDataSupport(VDATA_SELECT))
    {
    float* softSelection = m_mesh->getVSelectionWeights();
    hctUserChannelUtil::GlobalChannel softSelectionGChannel;
    softSelectionGChannel.m_channelName = "Soft Selection";
    softSelectionGChannel.m_channelType = hctUserChannelUtil::CT_VERTEX_FLOAT;

    hctUserChannelUtil::ChannelDataItem item;
    for (int i=0; i<m_mesh->numVerts; i++)
    {
    item.m_float = softSelection[i];
    softSelectionGChannel.m_channelData.pushBack(item);
    }

    userChannelUtil.addGlobalChannel(softSelectionGChannel);
    }
    */

    // 2 - Add named selections from modifiers and/or object
    {
        // Traverse modifiers stack and store objects with the right interface
        hkArray<IMeshSelectData*> meshSelectDataInterfaces;
        {
            Object* curObject = m_inode->GetObjectRef();
            while (curObject)
            {
                IDerivedObject* devObject = curObject->SuperClassID() == GEN_DERIVOB_CLASS_ID ? (IDerivedObject*)curObject : NULL;
                if (devObject) // has modifiers
                {
                    for (int m = devObject->NumModifiers() - 1; m >= 0; --m)
                    {

                        Modifier* mod = devObject->GetModifier(m);
                        Class_ID modClassID = mod->ClassID();

                        // EXP-1248 : Cloth modifier gives trouble, so ignore it if found.
                        static const Class_ID maxClothModifierClassID(0x67c1cd47, 0x7782f878);
                        if (mod->IsEnabled() && (modClassID != maxClothModifierClassID))
                        {
                            ModContext* mc = devObject->GetModContext(m);
                            if (!mc) continue;

                            LocalModData * mseldata = mc->localData;
                            if (!mseldata) continue;

                            IMeshSelectData* selectInterface = (IMeshSelectData *)(mseldata->GetInterface(I_MESHSELECTDATA));

                            if (selectInterface)
                            {
                                meshSelectDataInterfaces.pushBack(selectInterface);
                            }

                        }
                    }
                    curObject = devObject->GetObjRef(); // drop down one level the stack
                }
                else // p_end of the line, the base object
                {
                    IMeshSelectData* selectInterface = (IMeshSelectData *)(curObject->GetInterface(I_MESHSELECTDATA));
                    Class_ID objClassID = curObject->ClassID();

                    if (selectInterface)
                    {
                        meshSelectDataInterfaces.pushBack(selectInterface);
                    }

                    curObject = HK_NULL; // finish
                }
            }
        }

        // Add named vertex/triangle selections
        for (int meshSelectDataIndex = 0; meshSelectDataIndex < meshSelectDataInterfaces.getSize(); meshSelectDataIndex++)
        {
            IMeshSelectData* selectInterface = meshSelectDataInterfaces[meshSelectDataIndex];

            {
                GenericNamedSelSetList& vertSelList = selectInterface->GetNamedVertSelList();

                for (int vs = 0; vs < vertSelList.Count(); vs++)
                {
                    hctUserChannelUtil::GlobalChannel vertexSelectionGChannel;
                    vertexSelectionGChannel.m_channelName = FROM_MAX(*vertSelList.names[vs]);
                    vertexSelectionGChannel.m_channelType = hctUserChannelUtil::CT_VERTEX_SELECTION;

                    if ((*vertSelList.sets[vs]).GetSize() != m_mesh->numVerts) // EXP-1491
                    {
                        HK_WARN_ALWAYS(0xabba9231, "Invalid named vertex selection \"" << *vertSelList.names[vs] << "\" found (inconsistent size). Ignoring it");
                        continue;
                    }

                    hctUserChannelUtil::ChannelDataItem item;
                    for (int vertexIndex = 0; vertexIndex < m_mesh->numVerts; vertexIndex++)
                    {
                        if ((*vertSelList.sets[vs])[vertexIndex])
                        {
                            item.m_index = vertexIndex;
                            vertexSelectionGChannel.m_channelData.pushBack(item);
                        }
                    }

                    userChannelUtil.addGlobalChannel(vertexSelectionGChannel);
                }
            }

            {
                GenericNamedSelSetList& triSelList = selectInterface->GetNamedFaceSelList();

                for (int fs = 0; fs < triSelList.Count(); fs++)
                {
                    hctUserChannelUtil::GlobalChannel faceSelectionGChannel;
                    faceSelectionGChannel.m_channelName = FROM_MAX(*triSelList.names[fs]);
                    faceSelectionGChannel.m_channelType = hctUserChannelUtil::CT_FACE_SELECTION;

                    if ((*triSelList.sets[fs]).GetSize() != m_mesh->numFaces)   //EXP-1491
                    {
                        HK_WARN_ALWAYS(0xabba9231, "Invalid named face selection \"" << *triSelList.names[fs] << "\" found (inconsistent size). Ignoring it");
                        continue;
                    }

                    hctUserChannelUtil::ChannelDataItem item;
                    for (int faceIndex = 0; faceIndex < m_mesh->numFaces; faceIndex++)
                    {
                        if ((*triSelList.sets[fs])[faceIndex])
                        {
                            item.m_index = faceIndex;
                            faceSelectionGChannel.m_channelData.pushBack(item);
                        }
                    }

                    userChannelUtil.addGlobalChannel(faceSelectionGChannel);
                }
            }

            {
                GenericNamedSelSetList& edgeSelList = selectInterface->GetNamedEdgeSelList();

                for (int es = 0; es < edgeSelList.Count(); es++)
                {
                    hctUserChannelUtil::GlobalChannel edgeSelectionGChannel;
                    edgeSelectionGChannel.m_channelName = FROM_MAX(*edgeSelList.names[es]);
                    edgeSelectionGChannel.m_channelType = hctUserChannelUtil::CT_EDGE_SELECTION;

                    const int numEdges = 3 * m_mesh->numFaces;
                    if ((*edgeSelList.sets[es]).GetSize() != numEdges)  //EXP-1491
                    {
                        HK_WARN_ALWAYS(0xabba9231, "Invalid named edge selection \"" << *edgeSelList.names[es] << "\" found (inconsistent size). Ignoring it");
                        continue;
                    }

                    hctUserChannelUtil::ChannelDataItem item;
                    for (int i = 0; i < numEdges; i++)
                    {
                        if ((*edgeSelList.sets[es])[i])
                        {
                            item.m_index = i;
                            edgeSelectionGChannel.m_channelData.pushBack(item);
                        }
                    }

                    userChannelUtil.addGlobalChannel(edgeSelectionGChannel);
                }
            }

        }

    }
}

hkxTextureFile* hctMaxMeshWriter::findTextureByBitmap(BitmapTex* bmTexture)
{
    HCT_SCOPED_CONVERSIONS;

    hkxTextureFile* curTex = m_exportedFileTextures.getWithDefault(bmTexture, HK_NULL);
    if (curTex)
    {
        return curTex;
    }

    MSTR thisName = bmTexture->GetMapName();
    thisName.toUpper();

    // check them all by name (as a lot of artists don't reuse the same Map for the same bitmap)
    hkPointerMap<BitmapTex*, hkxTextureFile*>::Iterator it = m_exportedFileTextures.getIterator();
    while (m_exportedFileTextures.isValid(it))
    {
        BitmapTex* other = m_exportedFileTextures.getKey(it);
        if (other)
        {
            MSTR otherName = other->GetMapName();
            otherName.toUpper();
            if (thisName == otherName)
            {
                return m_exportedFileTextures.getValue(it);
            }
        }

        it = m_exportedFileTextures.getNext(it);
    }

    return HK_NULL;
}

void hctMaxMeshWriter::createTextures(const hkArray<Mtl*>& mats, const hkArray<int>& blends, hkArray< hkxMaterial::TextureStage >& newTextureVariants, hkArray< int >& textureChannels, hkArray< int >& textureBlends)
{
    // Special case for blend materials, do not export sub-materials per se, just add texture slots
    for (int mi = 0; mi < mats.getSize(); ++mi)
    {
        Mtl* mat = mats[mi];
        // Default case, get the number of maps, export the ones that are enabled
        int numSlots = mat->NumSubTexmaps();
        for (int slot = 0;slot < numSlots;slot++)
        {
            Texmap* texmap = mat->GetSubTexmap(slot);
            int st = mat->SubTexmapOn(slot);

            if (!st || !texmap)
            {
                // Map not present or disabled
                continue;
            }

            hkxMaterial::TextureType hkslot = hctMaxUtils::maxSlotToTextureType(slot);
            if (hkslot == hkxMaterial::TEX_NOTEXPORTED) // supported by the exporters
            {
                continue;
            }

            // texmap itself might be multi map
            hkArray< Texmap* > sourceTexMaps;
            hctMaxUtils::getSubTexMaps(texmap, sourceTexMaps);

            HCT_SCOPED_CONVERSIONS;

            for (int ti = 0; ti < sourceTexMaps.getSize(); ++ti)
            {
                //make sure that maxMap->GetUVWSource() == UVWSRC_EXPLICIT, or gen them?
                int ch = sourceTexMaps[ti]->GetMapChannel(); // 1 == default ch

                BitmapTex* bmTexture = HK_NULL;
                if (sourceTexMaps[ti]->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
                {
                    bmTexture = (BitmapTex*)sourceTexMaps[ti];
                }

                if (bmTexture)
                {
                    // if a bitmmap is used for different usages (both reflect + diffuse in the scene this check will be wrong., but that is unlikely)
                    hkxTextureFile* curTex = findTextureByBitmap(bmTexture);
                    if (!curTex)
                    {
                        curTex = new hkxTextureFile();
                        m_exportedFileTextures.insert(bmTexture, curTex); // keep track of the association

                        const char* assetPathName = FROM_MAX(bmTexture->GetMapName());
                        bool isNotRelativePath = (assetPathName[0] != '.') && ((hkString::indexOf(assetPathName, '/') >= 0) || (hkString::indexOf(assetPathName, '\\') >= 0));
                        if (isNotRelativePath)
                        {
                            //XX [EXP-2332] To keep compat with older Havok exports, iof the asset manager uses relative paths, then leave them like that, otherwise use full path as found by Max
                            BitmapInfo _bi(bmTexture->GetMapName());
                            BMMGetFullFilename(&_bi);
                            const MCHAR* bitmap = _bi.Name();
                            curTex->m_filename = FROM_MAX(bitmap);
                        }
                        else
                        {
                            curTex->m_filename = assetPathName;
                        }

                        curTex->m_originalFilename = curTex->m_filename;
                        const char* texName = FROM_MAX(bmTexture->GetName());
                        if (texName)
                        {
                            curTex->m_name = texName;
                        }
                        else
                        {
                            curTex->m_name = HK_NULL;
                        }

                        m_owner->m_currentScene.m_externalTextures.pushBack(curTex); // add to the master top level list
                    }

                    hkxMaterial::TextureStage& texState = newTextureVariants.expandOne();
                    texState.m_texture = curTex;
                    texState.m_usageHint = hkslot;
                    // HCL-106: Don't add channel more than once
                    int channelIndex = textureChannels.indexOf(ch);
                    if (channelIndex < 0)
                    {
                        channelIndex = textureChannels.getSize();
                        textureChannels.pushBack(ch);
                    }
                    texState.m_tcoordChannel = channelIndex;

                    if (blends.indexOf(mi) >= 0)
                    {
                        textureBlends.pushBack(newTextureVariants.getSize() - 1);
                    }
                }
                else// try to just make a bitmap for it (Checker, Noise, etc procedural style
                {
                    hkxTextureInplace* curTex = m_exportedInplaceTextures.getWithDefault(sourceTexMaps[ti], HK_NULL);
                    if (!curTex)
                    {
                        BitmapInfo bi;
                        bi.SetType(BMM_TRUE_32);
                        bi.SetWidth(256);
                        bi.SetHeight(256);
                        bi.SetFlags(MAP_HAS_ALPHA);
                        bi.SetCustomFlag(0);

                        Bitmap* bm = TheManager->Create(&bi);
                        if (bm)
                        {
                            sourceTexMaps[ti]->RenderBitmap(0, bm, 1.0f, FALSE);// no scale, no filter
                            curTex = new hkxTextureInplace();

                            // save as a raw image. Uncompressed TGA will do.
                            curTex->m_fileType[0] = 'T';
                            curTex->m_fileType[1] = 'G';
                            curTex->m_fileType[2] = 'A';
                            curTex->m_fileType[3] = '\0';
                            curTex->m_name = FROM_MAX(sourceTexMaps[ti]->GetName());
                            curTex->m_originalFilename = ""; // none

                            struct TargaHeader
                            {
                                unsigned char   IDLength; // id length, can be 0
                                unsigned char   ColormapType; // 0 == no map
                                unsigned char   ImageType; // 2 == raw rgb(a) (not packet based)
                                unsigned char   ColormapSpecification[5]; // all 0 for no color map
                                unsigned short  XOrigin;
                                unsigned short  YOrigin;
                                unsigned short  ImageWidth;
                                unsigned short  ImageHeight;
                                unsigned char   PixelDepth; // 32 in this case
                                unsigned char   ImageDescriptor; // how the data is stored
                            };

                            int bmType;
                            BMM_Color_32* bmData32 = (BMM_Color_32*)bm->GetStoragePtr(&bmType);
                            BMM_Color_24* bmData24 = (BMM_Color_24*)bmData32;

                            int bytes = (bmType == BMM_TRUE_24 ? 3 : 4);
                            int numData = (256 * 256 * bytes) + sizeof(TargaHeader); // 4 bytes per pixel + header
                            curTex->m_data.setSize(numData);

                            TargaHeader* header = (TargaHeader*)(curTex->m_data.begin());
                            hkString::memSet(header, 0, sizeof(TargaHeader));
                            header->ImageType = 2;
                            header->ImageWidth = 256;
                            header->ImageHeight = 256;
                            header->PixelDepth = (unsigned char)(bytes * 8);
                            header->ImageDescriptor = bytes == 4 ? 8 : 0; // 8 bit alpha, lower left origin (bits 5 and 6 are flags)

                            if ((bmType == BMM_TRUE_32) || (bmType == BMM_TRUE_24))
                            {
                                int byteIndex = sizeof(TargaHeader);
                                for (int x = 255; x >= 0; --x)
                                {
                                    for (int y = 0; y < 256; ++y)
                                    {
                                        int bmDataIndex = x * 256 + y;
                                        // TGA is a BGR(A) format
                                        if (bmType == BMM_TRUE_32)
                                        {
                                            curTex->m_data[byteIndex++] = bmData32[bmDataIndex].b;
                                            curTex->m_data[byteIndex++] = bmData32[bmDataIndex].g;
                                            curTex->m_data[byteIndex++] = bmData32[bmDataIndex].r;
                                            curTex->m_data[byteIndex++] = bmData32[bmDataIndex].a;
                                        }
                                        else
                                        {
                                            curTex->m_data[byteIndex++] = bmData24[bmDataIndex].b;
                                            curTex->m_data[byteIndex++] = bmData24[bmDataIndex].g;
                                            curTex->m_data[byteIndex++] = bmData24[bmDataIndex].r;
                                        }
                                    }
                                }
                            }

                            m_exportedInplaceTextures.insert(sourceTexMaps[ti], curTex); // keep track of the association

                            bm->DeleteThis();
                        }
                    }

                    if (curTex)
                    {
                        hkxMaterial::TextureStage& texState = newTextureVariants.expandOne();
                        texState.m_texture = curTex;
                        curTex->removeReference();
                        texState.m_usageHint = hkslot;
                        int channelIndex = textureChannels.indexOf(ch);
                        if (channelIndex < 0)
                        {
                            channelIndex = textureChannels.getSize();
                            textureChannels.pushBack(ch);
                        }
                        texState.m_tcoordChannel = channelIndex;

                        if (blends.indexOf(mi) >= 0)
                        {
                            textureBlends.pushBack(newTextureVariants.getSize() - 1);
                        }
                    }
                }
            }
        }
    }
}

void hctMaxMeshWriter::createMaterialDX(class IDxMaterial* dxMat, hkxMaterial* newMat, hkArray<int>& usedChannels)
{
    HCT_SCOPED_CONVERSIONS;

#ifdef MAX6_OR_HIGHER

    IDxMaterial2* dx2Mat = (IDxMaterial2*)dxMat->GetInterface(IDXMATERIAL2_INTERFACE);

    int numBitmaps = dxMat->GetNumberOfEffectBitmaps();
    for (int bi = 0; bi < numBitmaps; ++bi)
    {
        PBBitmap* pbm = dxMat->GetEffectBitmap(bi);
        if (pbm)
        {
            const MCHAR* bmFile = pbm->bi.Filename();
            if (bmFile)
            {
                //XXX TODO: see if this has a texture file already

                // if a bitmap is used for different usages (both reflect + diffuse in the scene this check will be wrong., but that is unlikely)
                hkxTextureFile* curTex = new hkxTextureFile();

                curTex->m_filename = FROM_MAX(bmFile);
                curTex->m_originalFilename = curTex->m_filename;

                const MCHAR* texName = pbm->bi.Name();
                if (texName)
                {
                    curTex->m_name = FROM_MAX(texName);
                }
                else
                {
                    curTex->m_name = HK_NULL;
                }

                m_owner->m_currentScene.m_externalTextures.pushBack(curTex); // add to the master top level list

                int texChannel = dx2Mat ? dx2Mat->GetBitmapMappingChannel(bi) : 0;
                if ((texChannel >= 0) && (usedChannels.indexOf(texChannel) < 0)) // EXP-1835: Don't add invalid channels, HCL-106: Don't add channel more than once
                {
                    usedChannels.pushBack(texChannel);
                }
            }
        }
    }

#ifdef MAX2010_OR_HIGHER
    const MaxSDK::AssetManagement::AssetUser& assetUser = dxMat->GetEffectFile();
    MSTR fxFilenameStr = assetUser.GetFileName();
    const char* fxFilename = FROM_MAX(fxFilenameStr);
#else
    const char* fxFilename = FROM_MAX(dxMat->GetEffectFilename());
#endif

    // read the fxFile and pop it into the effect
    hkArray<char>::Temp pFile;
    if (hkLoadUtil(fxFilename).toArray(pFile))
    {
        hkxMaterialEffect* matEffect = new hkxMaterialEffect();

        matEffect->m_name = fxFilename;
        matEffect->m_data.append(reinterpret_cast<hkUint8*>(pFile.begin()), pFile.getSize());
        matEffect->m_type = hkxMaterialEffect::EFFECT_TYPE_HLSL_FX_INLINE;

        // add to the mat
        newMat->m_extraData = matEffect;
    }

    return;

#endif

}
static int _findTextureIndex(hkxMaterial* curMat, hkReferencedObject* lastTexture)
{
    for (int mi = 0; mi < curMat->m_stages.getSize(); ++mi)
    {
        if (curMat->m_stages[mi].m_texture.val() == lastTexture)
            return mi;
    }
    return 0;
}

hkxMaterial* hctMaxMeshWriter::findMaterialMatch(Mtl* keyMat, const hkArray<Mtl*>& matsUsed)
{
    for (int mi = 0; mi < m_exportedMaterials.getSize(); ++mi)
    {
        UniqueMatConversion& um = m_exportedMaterials[mi];
        if (um.keyMtl == keyMat)
        {
            if (um.usedMats.getSize() != matsUsed.getSize())
                continue;

            // for multi/sub mats, different parts may have been used for each actual export real mat
            int si = 0;
            for (; si < matsUsed.getSize(); ++si)
            {
                if (matsUsed[si] != um.usedMats[si])
                    break;
            }
            if (si == matsUsed.getSize())
                return um.result;

        }
    }
    return HK_NULL;
}


void hctMaxMeshWriter::createMaterial(Mtl* keyMat, const hkArray<Mtl*>& mats, const hkArray<int>& blends, hkxMaterial*& newMat, hkArray<int>& usedChannels)
{
    HCT_SCOPED_CONVERSIONS;

    if ((mats.getSize() < 1) || (keyMat == HK_NULL))
    {
        newMat = HK_NULL;
        return;
    }

    const TimeValue staticTick = m_iStaticFrame * GetTicksPerFrame();

    hkxMaterial* curMat = findMaterialMatch(keyMat, mats);
    if (curMat)
    {
        newMat = curMat;
        newMat->addReference();
        for (int mi = 0; mi < mats.getSize(); ++mi)
            hctMaxUtils::findUsedChannels(mats[mi], usedChannels);
        return; // done already.
    }

    // EXP-365 - Cannot work with stand-in materials
    {
        Class_ID theId = keyMat->ClassID();

        if (theId.PartA() == STANDIN_CLASS_ID)
        {
            newMat = HK_NULL;
            return;
        }
    }

    newMat = curMat = new hkxMaterial();

    // Set the material name
    if (keyMat->GetName().Length()>0)
    {
        MSTR matName = keyMat->GetName();
        hkStringBuf b(FROM_MAX(matName));
        for (int ms = 0; ms < mats.getSize(); ++ms)
        {
            if (mats[ms] != keyMat)
            {
                MSTR subMatName = mats[ms]->GetName();
                b.appendJoin("/", FROM_MAX(subMatName));
            }
        }
        newMat->m_name = b.cString();
    }

    UniqueMatConversion& um = m_exportedMaterials.expandOne();
    um.keyMtl = keyMat;
    um.result = curMat; // keep track of the association
    um.usedMats.append(mats);

    m_owner->m_currentScene.m_materials.pushBack(curMat); // add to the master top level list

    // see if it is a DX hardware shader perhaps
    hkArray<hkxMaterial::TextureStage> newTextures;

    bool dxExport = false;
    hkArray<int> textureBlends;
#ifdef MAX6_OR_HIGHER
    IDxMaterial* dxMat = (IDxMaterial*)(keyMat->GetInterface(IDXMATERIAL_INTERFACE));
    if (dxMat)
    {
        dxExport = true;
        createMaterialDX(dxMat, newMat, usedChannels);
    }
    else
#endif
    {
        // not a dx texture, so do normal export of the material
        createTextures(mats, blends, newTextures, usedChannels, textureBlends);
    }

    Mtl* colorPropsMat = mats[0]; // better way to look for them? blend them?
    {
        float opacity = 1.0f;

        if (colorPropsMat->ClassID() == Class_ID(DMTL_CLASS_ID, 0) && colorPropsMat->SupportsShaders())
        {
            StdMat2* stdMat = (StdMat2*)colorPropsMat;
            opacity = stdMat->GetOpacity(staticTick);
        }

        Point3 diffuseColor = colorPropsMat->GetDiffuse();
        curMat->m_diffuseColor.set(diffuseColor[0], diffuseColor[1], diffuseColor[2], opacity);
    }
    {
        Point3 ambientColor = colorPropsMat->GetAmbient();
        curMat->m_ambientColor.set(ambientColor[0], ambientColor[1], ambientColor[2]);
    }
    {
        getSpecularParameters(*curMat, colorPropsMat);
        Point3 specularColor = colorPropsMat->GetSpecular();
        float specularLevel = colorPropsMat->GetShinStr();
        float specularPower = colorPropsMat->GetShininess();

        specularColor *= specularLevel; // reduce the brightness of the specular color so that it is scaled by the level in Max. Okay approx.
        specularPower = (specularPower * 100) + 1; // 1..101 for now.
        // both for the for the level being 0..100 in Max.

        curMat->m_specularColor.set(specularColor[0], specularColor[1], specularColor[2], specularPower);
    }

    // Vision engine Transparency
    curMat->m_transparency = hkxMaterial::transp_none;
    {
        //Only set transparency if Opacity channel is enabled
        StdMat* pStdMat = dynamic_cast<StdMat2*> (colorPropsMat);
        if (pStdMat)
        {
            if (pStdMat->MapEnabled(ID_OP))
            {
                // Map for it should already be gathered by the createTextures() above and aleady be in the m_stages
                switch (pStdMat->GetTransparencyType())
                {
                case TRANSP_ADDITIVE:
                {
                    curMat->m_transparency = hkxMaterial::transp_additive;
                }
                break;
                case TRANSP_SUBTRACTIVE:
                {
                    curMat->m_transparency = hkxMaterial::transp_subtractive;
                }
                break;
                case TRANSP_FILTER:
                {
                    if (pStdMat->GetFalloffOut())
                    {
                        curMat->m_transparency = hkxMaterial::transp_colorkey;
                    }
                    else
                    {
                        curMat->m_transparency = hkxMaterial::transp_alpha;
                    }
                }
                break;
                default:
                {
                    curMat->m_transparency = hkxMaterial::transp_none;
                }
                break;
                }
            }
        }
    }


    {
        Point3 emissive(0, 0, 0);
        if (colorPropsMat->GetSelfIllumColorOn())
        {
            emissive = colorPropsMat->GetSelfIllumColor();
        }
        else
        {
            emissive = colorPropsMat->GetDiffuse() * colorPropsMat->GetSelfIllum(staticTick);

        }

        curMat->m_emissiveColor.set(emissive[0], emissive[1], emissive[2], 1.0f);
    }

    //[EXP-2316] UV map info for Vision
    newMat->m_uvMapAlgorithm = hkxMaterial::UVMA_3DSMAX_STYLE;
    newMat->m_uvMapOffset[0] = 0; newMat->m_uvMapOffset[1] = 0;
    newMat->m_uvMapScale[0] = 1.f; newMat->m_uvMapScale[1] = 1.f;
    newMat->m_uvMapRotation = 0;
    newMat->m_userData = 0;
    {
        // VGeom2 etc have one uv transform per mat, not per texture, can they have historically
        // just taken this from the first Diffuse texture map found
        Texmap *pDiffuseTexMap = colorPropsMat->GetSubTexmap(ID_DI); //XX asumes no remap here. Blend mats may disagree.. xx
        if (pDiffuseTexMap && (pDiffuseTexMap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0)))
        {
            BitmapTex *pTex = (BitmapTex*)pDiffuseTexMap;
            StdUVGen *pUVGen = pTex->GetUVGen();
            if (pUVGen)
            {
                newMat->m_uvMapOffset[0] = pUVGen->GetUOffs(0);
                newMat->m_uvMapOffset[1] = pUVGen->GetVOffs(0);
                newMat->m_uvMapScale[0] = pUVGen->GetUScl(0);
                newMat->m_uvMapScale[1] = pUVGen->GetVScl(0);
                newMat->m_uvMapRotation = pUVGen->GetAng(0);
            }
        }
    }

    // textures
    if (newTextures.getSize())
    {
        curMat->m_stages.setSize(newTextures.getSize());

        // fill out the stages
        for (int s = 0; s < newTextures.getSize(); ++s)
        {
            hkxMaterial::TextureStage& ts = curMat->m_stages[s];
            ts = newTextures[s];
        }

        hkReferencedObject* firstFoundBlendTexture = HK_NULL;
        if (textureBlends.getSize() > 0)
        {
            firstFoundBlendTexture = curMat->m_stages[textureBlends[0]].m_texture.val();
        }

        curMat->sortTextureStageOrder();

        if (firstFoundBlendTexture != HK_NULL) // assume last texture was the mix texture (mat should have been the last in the mat array for it anyway). TODO remove assumption
        {
            // Can only add one as it is a prop. Expand to multiple if > 1 blend found and needed in future.
            curMat->addProperty(hkxMaterial::PROPERTY_MTL_TYPE_BLEND, _findTextureIndex(curMat, firstFoundBlendTexture));
        }

    }

    {
        // add double/two-sided attribute
        bool bIsDoubleSided = false;
        StdMat2* pStdMat = dynamic_cast<StdMat2*> (colorPropsMat);
        if (pStdMat)
        {
            bIsDoubleSided = pStdMat->GetTwoSided();
        }

        // use hkxSparselyAnimatedBool to store bool value
        hkxSparselyAnimatedBool* pIsDoubleSidedUsed = new hkxSparselyAnimatedBool();
        pIsDoubleSidedUsed->m_times.pushBack(0.0f); // needed if only one bool value should be stored
        pIsDoubleSidedUsed->m_bools.pushBack(bIsDoubleSided);

        // create the double sided attribute
        hkxAttribute doubleSidedAttribute;
        doubleSidedAttribute.m_name = "DoubleSided";
        doubleSidedAttribute.m_value = pIsDoubleSidedUsed;
        pIsDoubleSidedUsed->removeReference();

        // add Vision attribute group
        hkxAttributeGroup visionGroup;
        visionGroup.m_name = "Vision";
        visionGroup.m_attributes.pushBack(doubleSidedAttribute);
        curMat->m_attributeGroups.pushBack(visionGroup);
    }

    // EXP-921 : Export material attributes
    {
        m_owner->exportAttributes(keyMat, curMat->m_attributeGroups);

        // Processas and possibly merge attributes
        m_owner->m_attributeProcessing.processAttributes(curMat);


    }
}

void hctMaxMeshWriter::write(hkxMesh*& writtenMesh, hkxSkinBinding*& writtenSkin, bool forceSkinned)
{
    writtenMesh = HK_NULL;
    writtenSkin = HK_NULL;

    // Look for modifiers and base objects to walk
    if (m_skin || forceSkinned)
    {
        createSkin(writtenMesh, writtenSkin);
    }

    // create the mesh if no skin
    if (!writtenSkin)
    {
        Matrix3 worldTM = m_inode->GetObjTMAfterWSM(m_iStaticFrame*GetTicksPerFrame());
        createMesh(writtenMesh, NULL, NULL, &worldTM);
    }
}

void hctMaxMeshWriter::getSpecularParameters(hkxMaterial& mat, Mtl* pMaxMtl)
{
    StdMat2* pMat2 = dynamic_cast<StdMat2*>(pMaxMtl);
    if (pMat2)
    {
        Shader* pShader = pMat2->GetShader();
        if (pShader)
        {
            mat.m_specularMultiplier = pShader->GetSpecularLevel(0, 0);
            mat.m_specularExponent = hkMath::pow(2.f, pShader->GetGlossiness(0, 0) * 10.0f);
        }
        else
        {
            mat.m_specularMultiplier = 1.0f;
            mat.m_specularExponent = 75.0f;
        }
    }
    else
    {
        mat.m_specularMultiplier = 1.0f;
        mat.m_specularExponent = 75.0f;
    }

    if (mat.m_specularMultiplier > 10.0f) mat.m_specularMultiplier = 10.0f;
    if (mat.m_specularMultiplier < 0.0f) mat.m_specularMultiplier = 0.0f;

}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
