// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

//#define EMBED_IN_MODELLER

using System;
using System.Collections.Generic;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Net.Sockets;
using Microsoft.Win32;
using System.Runtime.InteropServices;

using Havok.Graphics;
using Havok.Tool.UI;


namespace Havok.Tool
{
    
    public delegate void toolProgressReportDelegate(string message, float percentDone);
    public delegate void setContentsFromMemoryDelegate(IntPtr root /* hkRootLevelContainer* */ , IntPtr vtableReg, bool copy, bool allowInstances);
    

    public partial class ToolUI : UserControl, /*IBaseSystem,*/ IDisposable, IPluginOwner
    {
        [DllImport("user32.dll")]
        static extern bool SetForegroundWindow(IntPtr hWnd);

        [DllImport("User32.dll")]
        public static extern IntPtr GetParent(IntPtr hwnd);

        [DllImport("User32.dll")]
        public static extern IntPtr SendMessage(IntPtr hWnd, Int32 msg, IntPtr wParam, IntPtr lParam);


        [StructLayout(LayoutKind.Sequential)]
        struct NMHDR_WITH_POS
        {
            public IntPtr hwndFrom;
            public IntPtr idFrom;
            public int code;
            public int x;
            public int y;
        };
        

        //static private Int32 WM_NOTIFY = 0x004E;
        //static private Int32 WM_RBUTTONDOWN = 0x0204;
        //static private Int32 WM_RBUTTONUP = 0x0205; 
        
        static private String s_havokFileExtenstions = "Havok Files (*.hkx;*.hkt;*.xml)|*.HKX;*.HKT;*.XML;|Binary Packfiles (*.hkx)|*.HKX|Tagfiles (*.hkt)|*.HKT|XML Packfiles (*.xml)|*.XML|All Files|*.*";

        // Renderer can be null, or the text name for the hkg renderer to use (instead of using the setting in the registry)
        public ToolUI(string renderer, IntPtr baseSystem)
        {
            m_splash = new SplashScreen();
            m_splash.start();
            m_splash.setMessageAndProgress("Initializing Control UI..", 0);

            InitializeComponent();

            m_splash.setMessageAndProgress("Initializing Graphics Context..", 0);

            m_graphicsCtrl = new GraphicsControl(renderer, baseSystem, delegate { m_splash.showSplash(); }, delegate { m_splash.hideSplash(); });

            m_graphicsCtrl.m_toolUI = this;

            m_errorConsole = new ErrorConsoleForm();

            m_gridForm = new UI.GridForm(m_graphicsCtrl);

            m_graphicsCtrl.Dock = System.Windows.Forms.DockStyle.Fill;
            m_graphicsCtrl.Size = mainSplitContainer.Panel1.Size;
            mainToolStripContainer.ContentPanel.Controls.Add(m_graphicsCtrl);

            this.DragEnter += new DragEventHandler(OnMyDragEnter);
            this.DragDrop += new DragEventHandler(OnMyFileDragDrop);
            this.DragLeave += new EventHandler(OnMyFileDragLeave);

            m_graphicsCtrl.MouseDoubleClick += new MouseEventHandler(graphicsMouseSelect);
            m_graphicsCtrl.KeyDown += new KeyEventHandler(graphicsKeyDown);
            m_graphicsCtrl.KeyUp += new KeyEventHandler(graphicsKeyUp);

            m_graphicsCtrl.MouseDown += new MouseEventHandler(graphicsMouseDown);
            m_graphicsCtrl.MouseMove += new MouseEventHandler(graphicsMouseMove);
            m_graphicsCtrl.MouseUp += new MouseEventHandler(graphicsMouseUp);

            m_graphicsCtrl.LostFocus += new EventHandler(OnFocusLost);

            m_splash.setMessageAndProgress("Initializing Plugins..", 0);

            Populate(new toolProgressReportDelegate(m_splash.setMessageAndProgress));

            m_splash.setMessageAndProgress("Starting Plugins..", 0.99f);

            tabControl.CreateControl(); // necessary workaround so that TabControl.TabPages.Insert() later on will work
            m_plugins.StartPlugins(tabControl, mainMenu, m_graphicsCtrl, m_errorConsole, navToolStrip);
            m_plugins.StartTweakers(tabControl, mainMenu, m_errorConsole);
            if (tabControl.TabCount > 0)
            {
                tabControl.SelectTab(0);
            }
            m_plugins.SetOwnerOnPlugins(this);

            m_graphicsCtrl.m_plugins = m_plugins;

            bool shadows = (int)m_graphicsCtrl.m_windowHKG.getShadowMapSupport() > (int)hkgWindowCLR.ShadowMapSupport.NOSUPPORT;
            shadowsToolStripMenuItem.Enabled = shadows;

            m_serverForm = new UI.ServerForm(m_plugins);

            m_stepTimer = new System.Windows.Forms.Timer();
            m_stepTimer.Tick += new EventHandler(TimerEventProcessor);
            m_stepTimer.Interval = 4; // run at as high a speed as required
            m_stepTimer.Tag = this;
            m_stepTimer.Start();

            m_flyModeUpVectorExplicit = false;

            const int defaultPort = 25010;
            int port = defaultPort;
            {
                const string userRoot0 = "HKEY_CURRENT_USER\\Software\\Havok";
                string subkey;
                bool x64Run = IntPtr.Size > 4;
                if (x64Run)
                {
                    subkey = "ManagedTools_x64";
                }
                else
                {
                    subkey = "ManagedTools";
                }
                string keyName = userRoot0 + "\\" + subkey;
                string keyValue = (string)Registry.GetValue(keyName, "ListenPort", "25010");
                if (keyValue != null)
                {
                    Int32.TryParse(keyValue, out port);
                    if (port == 0) 
                    {
                        port = defaultPort;
                    }
                }
                Registry.SetValue(keyName, "ListenPort", ((Int32)port).ToString() ); // always save so that users can edit it / see it
            }

            m_tcpUpdateListener = new TcpListener(System.Net.IPAddress.Any, port);
            try
            {
                m_tcpUpdateListener.Start();
                m_serverForm.m_ownListener = m_tcpUpdateListener;

                m_errorConsole.addMessage(IConsole.MessageType.REPORT, "Listening for Remote Preview connections on port " + ((Int32)port).ToString());
            }
            catch (SocketException)
            {
                m_tcpUpdateListener = null;
            }

            // if you want a more responsive sim then on idle you should 
            // return processing to the sim loop
            Application.Idle += new EventHandler(Application_Idle);

#if FULLDEBUG
            // If we own the base sys, then we are in standalone.
            // In debug standalone we need to refresh the debug symbols
            if (hkgBaseSystemCLR.ownMemSystem())
            {
                m_splash.setMessageAndProgress("Loading debug symbols..", 0.95f);
                hkgBaseSystemCLR.refreshDebugSymbols();
            }
#endif

            m_splash.setMessageAndProgress("Loading complete, starting..", 1.0f);

        }
            
        public IPlugin[] getAllPlugins()
        {
            int numPlugs = m_plugins.GetNumPlugins();
            IPlugin[] plugins = new IPlugin[numPlugs]; 

            for (int pi = 0; pi < numPlugs; ++pi)
            {
                IPlugin ed = (IPlugin)m_plugins.GetPlugin(pi);
                if (ed != null)
                {
                    plugins[pi] = ed;
                }
        	}

            return plugins;
        }
       
        public void OnIdle()
        {
           if (this.Visible && (m_plugins != null) && (m_plugins.IsPlaying() || m_graphicsCtrl.m_constantUpdateKeysDown))
           {
               this.Invalidate();
           }
        }

        public void Application_Idle(object sender, EventArgs e)
        {
            OnIdle();
        }

        public void HideSplashScreen()
        {
            if (m_splash != null)
            {
                m_splash.Opacity = 0.75f;
            }
        }

        public void CloseSplashScreen()
        {
            if (m_splash != null)
            {
                SetForegroundWindow( this.Handle );

                m_splash.end(0);
                m_splash = null;

                if ((m_plugins.m_assembliesWithPlugins.Count < 1) && (m_plugins.m_assembliesWithTweakers.Count < 1))
                {
                    string str = "No Tool Plugins found in \r\n[ " + m_plugins.m_lastPlugPath + " ]\r\nCheck your HKCU\\Software\\Havok\\ManagedTools\\ToolsDir path";
                    MessageBox.Show(str, "Tool Plugins", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            loadPreferences();
        }

        void OnMyDragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop, false))
            {
                e.Effect = DragDropEffects.All;
                foreach (string fileName in (string[])e.Data.GetData(DataFormats.FileDrop))
                {
                    if (System.IO.File.Exists(fileName))
                    {
                         ShowDragDropTargetsForFile(fileName);
                    }
                } 
            }
        }
    
        void OnMyFileDragDrop(object sender, DragEventArgs e)
        {
            foreach (string fileName in (string[])e.Data.GetData(DataFormats.FileDrop))
            {
                if ( System.IO.File.Exists(fileName) )
                {
                    DragDropAddFile(fileName, e.X, e.Y);
                }
            }
            EndDragDropTargets();
        }

        void OnMyFileDragLeave(object sender, EventArgs e)
        {
            EndDragDropTargets();
        }

        void OnFocusLost(object sender, EventArgs e)
        {
            if ( toolStripFirstPersonButton.Checked == true )
            {
                toolStripFlyModeButton_Click(this, EventArgs.Empty);
            }
        }

        void performSelectAndTweak()
        {
             // Selection (different to pick and pull)
            if (m_currentPickedObject != null) m_currentPickedObject.m_object.Dispose();

            m_currentPickedObject = m_graphicsCtrl.pick(ref m_currentPickedObjectDepth);
            
            if (m_currentPickedObject != null)
            {
                if ( (m_lastPickedObject != null) && (m_lastPickedObject != m_currentPickedObject.m_object) )
                {
                    m_lastPickedObject.Dispose();
                }

                m_lastPickedObject = m_currentPickedObject.m_object;
                
            // get the data from plugin and give to the tweaker..
                int numPlugs = m_plugins.GetNumPlugins();
                for (int pi = 0; pi < numPlugs; ++pi)
                {
                    IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                    if (ed != null)
                    {
                        IDataSource ds = (IDataSource)ed;
                        if (ds != null)
                        {
                            HavokReflectedObject ro = new HavokReflectedObject();
                            ds.getDataAndClassForObject(m_currentPickedObject, ref ro);
                            if ((ro.instance.ToInt64() != 0) && (ro.klass.ToInt64() != 0)) // then it knew the picked object 
                            {
                                int numTweakers = m_plugins.GetNumTweakers();
                                for (int ti = 0; ti < numTweakers; ++ti)
                                {
                                    IDataEditor tweaker = m_plugins.GetTweaker(ti);
                                    if (tweaker != null)
                                    {
                                        tweaker.editData(ro, m_currentPickedObject.m_object.getName());
                                    }
                                    // continue to all tweakers?
                                 }
                                 break; // can stop asking the plugs
                            }
                        }

                        hkgVector3CLR mouseWorldPos = m_graphicsCtrl.getMouseInWorld(m_currentPickedObjectDepth);
                        ed.objectPicked(m_currentPickedObject, mouseWorldPos);
                    }
                }
            }
        }

        void performPick()
        {
            if (m_currentPickedObject != null) m_currentPickedObject.m_object.Dispose();
            
            m_currentPickedObject = m_graphicsCtrl.pick(ref m_currentPickedObjectDepth);
            
            if (m_currentPickedObject != null)
            {
                if (m_lastPickedObject != null)
                {
                    m_lastPickedObject.Dispose();
                }

                m_lastPickedObject = m_currentPickedObject.m_object;

                 // get the data from plugin and give to the tweaker..
                int numPlugs = m_plugins.GetNumPlugins();
                for (int pi = 0; pi < numPlugs; ++pi)
                {
                    IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                    if (ed != null)
                    {
                        hkgVector3CLR mouseWorldPos = m_graphicsCtrl.getMouseInWorld(m_currentPickedObjectDepth);
                        ed.objectPicked(m_currentPickedObject, mouseWorldPos);
                    }
                }
            }
        }

        void graphicsKeyDown( Object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Space && (m_currentPickedObject == null))
            {
                performPick();
                m_graphicsCtrl.m_constantUpdateKeysDown = false; // The HKG Control will update constantly when keys are down (good for fly, not so good for sim)
            }

        }

        void graphicsKeyUp(Object sender, KeyEventArgs e)
        {
            if ((e.KeyCode == Keys.Space) && (m_currentPickedObject != null))
            {
                int numPlugs = m_plugins.GetNumPlugins();
                for (int pi = 0; pi < numPlugs; ++pi)
                {
                    IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                    if (ed != null)
                    {
                        ed.objectReleased();
                    }
                }
                // Don't dispose as m_lastPickedObject still uses it
                
                m_currentPickedObject = null;
                m_currentPickedObjectDepth = -1;
            }

        }

        // Double click
        void graphicsMouseSelect( Object sender, MouseEventArgs e)
        {
            performSelectAndTweak();  
        }

        // Single click:
        void graphicsMouseDown(Object sender, MouseEventArgs e)
        {

#if EMBED_IN_MODELLER
            // Special upper left mouse notify up to parent (so handy for Max embed etc)
            IntPtr master = GetParent(GetParent(this.Handle));

            if (e.Button == MouseButtons.Right)
            {
                m_rightMouseDownLoc = e.Location;

                if ((master != IntPtr.Zero) && e.Button == MouseButtons.Right)
                {
                    NMHDR_WITH_POS nmh;
                    nmh.code = WM_RBUTTONDOWN;    // Message type defined by control.
                    nmh.idFrom = IntPtr.Zero;
                    nmh.hwndFrom = this.Handle;
                    nmh.x = e.X;
                    nmh.y = e.Y;

                    IntPtr msg = new IntPtr(WM_NOTIFY);
                    IntPtr val = Marshal.AllocHGlobal(Marshal.SizeOf(nmh));
                    Marshal.StructureToPtr(nmh, val, false);
                    SendMessage(master, WM_NOTIFY, msg, val);
                    Marshal.FreeHGlobal(val);

                    // It can call show view menu if it likes
                }
            }
#else
            if (e.Button == MouseButtons.Right)
            {
                m_rightMouseDownLoc = e.Location;
            }
#endif

        }

        public void showViewMenu(Point loc)
        {
            if ( !firstPersonModeToolStripMenuItem.Checked )
            {
                // Right click menu should set current view too
                int nv = m_graphicsCtrl.m_windowHKG.getNumViewports();
                for (int i = 0; i < nv; ++i)
                {
                    using (hkgViewportCLR v = m_graphicsCtrl.m_windowHKG.getViewport(i))
                    {
                        if (v.containsWindowCoord(loc.X, (int)(m_graphicsCtrl.m_windowHKG.getHeight() - loc.Y)))
                        {
                            m_graphicsCtrl.m_windowHKG.setCurrentViewport(i);
                            reflectViewportSettings();
                            break;
                        }
                    }
                }
                viewRenderMenuStrip.Show(this, loc);
            }
        }

        void graphicsMouseMove(Object sender, MouseEventArgs e)
        {
            // get the data from plugin and give to the tweaker..
            if (m_currentPickedObject != null)
            {
                int numPlugs = m_plugins.GetNumPlugins();
                for (int pi = 0; pi < numPlugs; ++pi)
                {
                    IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                    if (ed != null)
                    {
                        hkgVector3CLR mouseWorldPos = m_graphicsCtrl.getMouseInWorld(m_currentPickedObjectDepth);
                        ed.objectMoved(mouseWorldPos);
                    }
                }
            }
        }

        void graphicsMouseUp(Object sender, MouseEventArgs e)
        {

#if EMBED_IN_MODELLER
            // Special upper left mouse notify up to parent (so handy for Max embed etc)
            IntPtr master = GetParent(GetParent(this.Handle));
            if (e.Button == MouseButtons.Right)
            {
                if (master != IntPtr.Zero)
                {
                    NMHDR_WITH_POS nmh;
                    nmh.code = WM_RBUTTONUP;    // Message type defined by control.
                    nmh.idFrom = IntPtr.Zero;
                    nmh.hwndFrom = this.Handle;
                    nmh.x = e.X;
                    nmh.y = e.Y;

                    IntPtr msg = new IntPtr(WM_NOTIFY);
                    IntPtr val = Marshal.AllocHGlobal(Marshal.SizeOf(nmh));
                    Marshal.StructureToPtr(nmh, val, false);
                    SendMessage(master, WM_NOTIFY, msg, val);
                    Marshal.FreeHGlobal(val);
                }
                else if (e.Location == m_rightMouseDownLoc)
                {
                    showViewMenu(new Point(e.X, e.Y));
                }
            }
#else
            if (e.Button == MouseButtons.Right)
            {
                if (e.Location == m_rightMouseDownLoc)
                {
                    showViewMenu(new Point(e.X, e.Y));
                }
            }
            
#endif

            if (m_currentPickedObject != null)
           {
                int numPlugs = m_plugins.GetNumPlugins();
                for (int pi = 0; pi < numPlugs; ++pi)
                {
                    IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                    if (ed != null)
                    {
                        ed.objectReleased();
                    }
                }

                // Don't dispose as m_lastPickedObject still uses it
                
                m_currentPickedObject = null;
                m_currentPickedObjectDepth = -1;
            }
        }

        ~ToolUI()
        {
            savePreferences();
            Cleanup();
        }

        public void Populate(toolProgressReportDelegate progress)
        {
            m_plugins = new Plugins();

            bool try64bit = IntPtr.Size == 8;

            string pathSource = "none";
        RETRY_WITHOUT_64BIT:

            // Check Tools EnvVar
            string subkey;
            string userToolPath;
            string filterRoot;
            if (try64bit)
            {
                subkey = "ManagedTools_x64";
                filterRoot = "HKEY_CURRENT_USER\\Software\\Havok\\hkFilters_x64";
                userToolPath = Environment.GetEnvironmentVariable("HAVOK_TOOLS_ROOT_X64");
                if ((userToolPath != null) && (userToolPath.Length > 0)) 
                {
                    pathSource = "envvar HAVOK_TOOLS_ROOT_X64";
                    userToolPath += "\\tools";
                }
            }
            else
            {
                subkey = "ManagedTools";
                filterRoot = "HKEY_CURRENT_USER\\Software\\Havok\\hkFilters";
                userToolPath = Environment.GetEnvironmentVariable("HAVOK_TOOLS_ROOT");
                if ((userToolPath != null) && (userToolPath.Length > 0))
                {
                    pathSource = "envvar HAVOK_TOOLS_ROOT";
                    userToolPath += "\\tools";
                }
            }
             
            if ((userToolPath != null) && (userToolPath.Length > 0) && (!System.IO.Directory.Exists(userToolPath)))
            {
                userToolPath = null;
            }

            if ((userToolPath == null) || (userToolPath.Length < 1))
            {
                const string userRoot0 = "HKEY_CURRENT_USER\\Software\\Havok";
                const string userRoot1 = "HKEY_LOCAL_MACHINE\\SOFTWARE\\Havok";
                
                // Check Managed Tools RegKey
                string keyName = userRoot0 + "\\" + subkey;
                pathSource = keyName + "\\ToolsDir";
                userToolPath = (String)Registry.GetValue(keyName, "ToolsDir", "");
                if ((userToolPath != null) && (userToolPath.Length > 0) && (!System.IO.Directory.Exists(userToolPath)))
                {
                    userToolPath = null;
                }

                // Check Machine Tools RegKey
                if ((userToolPath == null) || (userToolPath.Length < 1))
                {
                    keyName = userRoot1 + "\\" + subkey;
                    pathSource = keyName + "\\ToolsDir";
                    userToolPath = (String)Registry.GetValue(keyName, "ToolsDir", "");
                    if ((userToolPath != null) && (userToolPath.Length > 0) && (!System.IO.Directory.Exists(userToolPath)))
                    {
                        userToolPath = null;
                    }
                }

                // Check FilterTools RegKey
                if ((userToolPath == null) || (userToolPath.Length < 1))
                {
                    userToolPath = (String)Registry.GetValue(filterRoot, "FilterPath", "");
                    pathSource = filterRoot + "\\FilterPath";
                    if ((userToolPath != null) && (userToolPath.Length > 0))
                    {
                        // use the 'tools' sub dir if it exists
                        userToolPath += "\\tools";
                        if (!System.IO.Directory.Exists(userToolPath))
                        {
                            userToolPath = null;
                        }
                    }
                }
            }
           
            if ((userToolPath != null) && (userToolPath.Length > 0))
            {
                System.Console.WriteLine("Havok.Tool.ToolUI: Using user Plugin path: [{0:S}]", userToolPath);
                m_errorConsole.addMessage(IConsole.MessageType.REPORT, "Loading plugins from : " + userToolPath + " (from: " + pathSource + ")" );
                m_plugins.PopulateAll(userToolPath, progress);
            }
            else
            {
                if (try64bit)
                {
                    System.Console.WriteLine("Havok.Tool.ToolUI: x64 build but no x64 plugin paths found, trying non x64 specific paths..");
                    try64bit = false;
                    goto RETRY_WITHOUT_64BIT;
                }

                System.Console.WriteLine("Havok.Tool.ToolUI: Using Current Working Dir (./tools) as no plugin paths found" );
                m_errorConsole.addMessage(IConsole.MessageType.REPORT, "Loading plugins from './tools' as no env var or reg key found.");
                m_plugins.PopulateAll(".\\tools", progress);
            }
        }

        public void Cleanup()
        {
            clearCachedViewports();

            if (m_plugins != null)
            {
                m_plugins.Cleanup();
                m_plugins = null;
            }

            if (m_currentPickedObject != null)
            {
                m_currentPickedObject.m_object.Dispose();
                m_currentPickedObject = null;
                m_lastPickedObject = null;
            }
            else if (m_lastPickedObject != null)
            {
                m_lastPickedObject.Dispose();
                m_lastPickedObject = null;
            }

            if (m_currentSelectedObject != null)
            {
                m_currentSelectedObject.Dispose();
                m_currentSelectedObject = null;
            }

            if (m_graphicsCtrl.m_displayWorldHKG != null)
            {
                m_graphicsCtrl.Cleanup();
                if (m_graphicsCtrl.m_hkgOwner)
                {
                    hkgSystemCLR.quit();
                }
            }

            if (m_serverForm != null)
            {
                m_serverForm.Dispose();
                m_serverForm = null;
            }

            if (m_tcpUpdateListener != null)
            {
                m_tcpUpdateListener.Stop();
                m_tcpUpdateListener = null;
            }
        }


        public hkgDisplayObjectCLR getSelectedObject()
        {
            hkgDisplayObjectCLR ret = null;
            int numPlugs = m_plugins.GetNumPlugins();
            for (int pi=0; pi < numPlugs; ++pi)
            {
                IPluginDisplay ed = (IPluginDisplay)m_plugins.GetPlugin(pi);
                if (ed != null)
                {
                    ret = ed.getSelectedDisplayObject();
                    if (ret != null)
                    {
                        if (m_currentSelectedObject != null)
                        {
                            m_currentSelectedObject.Dispose();
                        }
                        m_currentSelectedObject = ret;
                        break;
                    }
                }
            }
              
            if (ret == null)
            {
                if ((m_currentPickedObject != null) && (m_currentPickedObject.m_object != null))
                {
                    ret = m_currentPickedObject.m_object;
                }
                else
                {
                    ret = m_lastPickedObject;
                }
            }

            if (ret != null)
            {
                int idx = m_graphicsCtrl.m_displayWorldHKG.getDisplayObjectIndex(ret);
                if (idx < 0) // not in the world
                {
                    if ((m_currentPickedObject != null) && (m_currentPickedObject.m_object != null))
                    {
                        m_currentPickedObject.m_object.Dispose();
                        m_currentPickedObject = null;
                        m_lastPickedObject = null;
                    }
                    else if (m_lastPickedObject != null)
                    {
                        m_lastPickedObject.Dispose();
                        m_lastPickedObject = null;
                    }
                }
                else
                {
                    return ret;
                }
            }
            return null;     
        }

        GraphicsControl m_graphicsCtrl;
        Plugins m_plugins;
        System.Windows.Forms.Timer m_stepTimer;
        SplashScreen m_splash;
        TcpListener m_tcpUpdateListener;
        ErrorConsoleForm m_errorConsole;
        PickInfo m_currentPickedObject;
        hkgDisplayObjectCLR m_lastPickedObject;
        hkgDisplayObjectCLR m_currentSelectedObject;
        float m_currentPickedObjectDepth;
        UI.GridForm m_gridForm;
        UI.ServerForm m_serverForm;

        Point m_rightMouseDownLoc;

        ArrayList m_cachedViewports;

        bool m_flyModeUpVectorExplicit;
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
