// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Windows.Forms;
using Havok.Tool;
using Havok.Graphics;

namespace Havok.Tool
{
	struct PluginAssembly
	{
		public Assembly assm;
		public IBaseSystem baseSystem;
		public bool baseSystemIsPluginToo;
		public bool baseSystemIsTweakerToo;
		
		public System.Collections.ArrayList pluginTypes; // Type[],  non basesys plugins(and tweakers etc)

		public System.Collections.ArrayList pluginInterfaceTypeIndices;
		public System.Collections.ArrayList tweakerInterfaceTypeIndices;

	}

	public class Plugins
	{
		public Plugins()
		{
			m_pluginAssemblies = new System.Collections.ArrayList();
			m_assembliesWithPlugins = new System.Collections.ArrayList();
			m_assembliesWithTweakers = new System.Collections.ArrayList();
		
			m_pluginInstances = new System.Collections.ArrayList();
			m_tweakerInstances = new System.Collections.ArrayList();
		}

		~Plugins()
		{
			Cleanup();
		}

		public void PopulateAll(string path, toolProgressReportDelegate progress)
		{
			m_lastPlugPath = path;
			
			try
			{
				// Populate on cur dir dlls
				string[] files = System.IO.Directory.GetFiles(path, "*.dll");
				int fi =0; 
				foreach (string f in files)
				{
					if (progress != null)
					{
						int lastBack = f.LastIndexOf('\\');
						string filename = f.Substring(lastBack > 0 ? lastBack + 1 : 0);
						string stat = "Checking DLL: " + filename;
						progress(stat, fi / (float)files.Length);
					}

					Populate(f);
					++fi;
				}

				// no recurse to make loads faster on proper setup system
				/*
					foreach (string d in System.IO.Directory.GetDirectories(path))
					{
						PopulateAll(d, progress);
					}
				*/
			}
			catch (System.Exception excpt)
			{
				Console.WriteLine(excpt.Message);
			}
		}

		public static bool MyInterfaceFilter(Type typeObj, Object criteriaObj)
		{
			if (typeObj.ToString() == criteriaObj.ToString())
				return true;
			else
				return false;
		}

		public void Populate(string assmName)
		{
			
			Assembly pluginAssm;
			try
			{
				pluginAssm = Assembly.LoadFrom(assmName);
			}
			catch (System.Exception excpt)
			{
				Console.WriteLine(excpt.Message);
				return; // prob just not a proper assm 
			}

			if (pluginAssm == null)
			{
				return;
			}

			string requiredInterfaceVersion = null;
			string requiredHavokSdkVersion = null;
			VersionCheck.getVersionInfo(ref requiredInterfaceVersion, ref requiredHavokSdkVersion);

			System.Collections.ArrayList plugins = new System.Collections.ArrayList();
			
			// find all types that use IPlugin
			Type[] pluginTypes = { };
			try
			{
				pluginTypes = pluginAssm.GetTypes();
			}
			catch (System.Exception e)
			{
				Console.WriteLine("Could not load assembly " + assmName + ". Ignoring");
				Console.WriteLine("Error was : " + e.Message);

			}

			Type baseType = null;

			TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);

			String pluginInterfaceName = "Havok.Tool.IPlugin";
			String tweakerInterfaceName = "Havok.Tool.IDataEditor";
			String baseInterfaceName = "Havok.Tool.IBaseSystem";

			bool baseSysIsPlugin = false;
			bool baseSysIsTweaker = false;
			bool havePlugins = false;

			System.Collections.ArrayList pluginIndices = new System.Collections.ArrayList();
			System.Collections.ArrayList tweakerIndices = new System.Collections.ArrayList();
		  
			for (int ti = 0; ti < pluginTypes.Length; ++ti)
			{
				if (pluginTypes[ti].IsAbstract)
					continue;
				
				Type[] baseList = pluginTypes[ti].FindInterfaces(myPluginFilter, baseInterfaceName);
				bool isBaseSystemImpl = (baseList.Length > 0) && (baseType == null); // first found base sys type gets the honors..
				if (isBaseSystemImpl)
				{
					baseType = pluginTypes[ti];// assume just the one
				}

				// allow multiple interface (so base and the plugin (or tweaker) in one type impl, so not an 'else'
				Type[] pluginList = pluginTypes[ti].FindInterfaces(myPluginFilter, pluginInterfaceName);
				Type[] tweakerList = pluginTypes[ti].FindInterfaces(myPluginFilter, tweakerInterfaceName);
				
				if ((pluginList.Length > 0) || (tweakerList.Length > 0) )
				{
					havePlugins = true;
					if (isBaseSystemImpl)
					{
						baseSysIsPlugin = pluginList.Length > 0;
						baseSysIsTweaker = tweakerList.Length > 0;
					}
					else
					{
						plugins.Add(pluginTypes[ti]);
						if (pluginList.Length > 0)
							pluginIndices.Add(plugins.Count - 1);
						if (tweakerList.Length > 0)
							tweakerIndices.Add(plugins.Count - 1);

					}
				}
			}

			IBaseSystem baseSys = null;
			if (havePlugins && (baseType != null))
			{
				// found the base system impl, so create it before any plugins that rely on it
				baseSys = (Havok.Tool.IBaseSystem)Activator.CreateInstance(baseType);

				string interfaceVer = null;
				string havokSdkVer = null;
				baseSys.getVersionInfo(ref interfaceVer, ref havokSdkVer);
				if ((interfaceVer != requiredInterfaceVersion) || ( havokSdkVer != requiredHavokSdkVersion) )
				{
					// can do whatever you like here, for the moment just warn
					string err = string.Format("Found interface version [{0}], sdk [{1}] in plugin [{2}] but main interface assm is [{3}], sdk [{4}]. Could lead to odd behavior or crashes.",
						interfaceVer, havokSdkVer, assmName, requiredInterfaceVersion, requiredHavokSdkVersion );

					MessageBox.Show(err, "Havok Plugin Warning", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					Console.WriteLine(err);
				}

				try
				{
					baseSys.initBaseSystem(hkgBaseSystemCLR.getBaseSyncInfoAsStaticPtr(), hkgSystemCLR.getGraphicsInfoAsStaticPtr());
					hkgSystemCLR.addSystemReference(baseSys);
				}
				catch (System.Exception excpt)
				{
					Console.WriteLine(assmName);
					Console.WriteLine(excpt.Message);
					return; // dodgy init or base has become corrupt
				}
			}

			// If we have found some plugins etc, add to our list:
			if ( (baseSys != null) || ((plugins != null) && (plugins.Count > 0)) )
			{
				PluginAssembly ea = new PluginAssembly();
				ea.assm = pluginAssm;
				ea.baseSystem = baseSys;
				ea.baseSystemIsPluginToo = baseSysIsPlugin;
				ea.baseSystemIsTweakerToo = baseSysIsTweaker;
				ea.pluginTypes = plugins; // non basesys plugins
				ea.pluginInterfaceTypeIndices = pluginIndices;
				ea.tweakerInterfaceTypeIndices = tweakerIndices;
				m_pluginAssemblies.Add(ea);

				if ((pluginIndices.Count > 0) || baseSysIsPlugin)
					m_assembliesWithPlugins.Add(m_pluginAssemblies.Count - 1);
				if ((tweakerIndices.Count > 0) || baseSysIsTweaker)
					m_assembliesWithTweakers.Add(m_pluginAssemblies.Count - 1);
			}
		}

		public void Cleanup()
		{
			if (m_pluginAssemblies != null)
			{
				DeletePlugins();
				DeleteTweakers();

				foreach (PluginAssembly ea in m_pluginAssemblies)
				{
					ea.baseSystem.quitBaseSystem();
					hkgSystemCLR.removeSystemReference(ea.baseSystem);
					if ( (ea.baseSystemIsPluginToo || ea.baseSystemIsTweakerToo) && typeof(IDisposable).IsAssignableFrom(ea.baseSystem.GetType()) )
					{
						((IDisposable)ea.baseSystem).Dispose();
					}
				}
			}

			m_pluginAssemblies = null;
		}

		public void StartPlugins(System.Windows.Forms.TabControl parentTabControl, System.Windows.Forms.MenuStrip menu, GraphicsControl graphicsControl, Havok.Tool.IConsole errorConsole, System.Windows.Forms.ToolStrip navigationToolBar)
		{
			CreatePlugins();

			SetDisplayContextOnPlugins(graphicsControl.m_displayWorldHKG, graphicsControl.m_windowHKG);
			SetParentUIOnPlugins(parentTabControl, menu, errorConsole, navigationToolBar);
			
			for (int i=0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.init();
			}
		}

		public void StartTweakers(System.Windows.Forms.TabControl parentTabControl, System.Windows.Forms.MenuStrip menu, Havok.Tool.IConsole errorConsole)
		{
			CreateTweakers();

			SetParentUIOnTweakers(parentTabControl, menu, errorConsole);

			//for (int i = 0; i < m_tweakerInstances.Count; ++i)
			//{
			//  IDataEditor tweakerInterface = (IDataEditor)m_tweakerInstances[i];
			//} 
		}

		public void CreatePlugins()
		{
			for (int ai=0; ai < m_assembliesWithPlugins.Count; ++ai)
			{
				int actualAssmIndex = (int)m_assembliesWithPlugins[ai];
				PluginAssembly ea = (PluginAssembly)m_pluginAssemblies[actualAssmIndex];
				if (ea.baseSystemIsPluginToo)
				{
					m_pluginInstances.Add( (IPlugin)ea.baseSystem );
				}
				for (int pi=0; pi < ea.pluginInterfaceTypeIndices.Count; ++pi)
				{
					int pluginTypeIndex = (int)ea.pluginInterfaceTypeIndices[pi];
					m_pluginInstances.Add( (IPlugin)Activator.CreateInstance((Type)ea.pluginTypes[pluginTypeIndex]) );
				}
			}
		}

		public void CreateTweakers()
		{
			for (int ai=0; ai < m_assembliesWithTweakers.Count; ++ai)
			{
				int actualAssmIndex = (int)m_assembliesWithTweakers[ai];
				PluginAssembly ea = (PluginAssembly)m_pluginAssemblies[actualAssmIndex];
				if (ea.baseSystemIsTweakerToo)
				{
					m_tweakerInstances.Add( (IDataEditor)ea.baseSystem );
				}
				for (int pi=0; pi < ea.tweakerInterfaceTypeIndices.Count; ++pi)
				{
					int tweakerTypeIndex = (int)ea.tweakerInterfaceTypeIndices[pi];
					m_tweakerInstances.Add( (IDataEditor)Activator.CreateInstance((Type)ea.pluginTypes[tweakerTypeIndex]) );
				}
			}
		}

		public int GetNumPlugins()
		{
			return m_pluginInstances.Count;
		}

		public int GetNumTweakers()
		{
			return m_tweakerInstances.Count;
		}

		public IPlugin GetPlugin(int i)
		{
			return (IPlugin) m_pluginInstances[i];
		}

		public IDataEditor GetTweaker(int i)
		{
			return (IDataEditor)m_tweakerInstances[i];
		}

		public void SetContentsFromMemory(IntPtr root /* hkRootLevelContainer* */, IntPtr vtableReg, bool copy, bool allowInstances)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.setContentsFromMemory(root, vtableReg, copy, allowInstances);
			}
		}

		public void SetContentsFromFile(String path, bool dragIn, Int32 x, Int32 y, Boolean prompt)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.setContentsFromFile(path, dragIn, x, y, prompt);
			}
		}

		public void ServeContentsToHost(String hostName, int port, System.Windows.Forms.Label connectionStatusLabel, Form form)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.serveContentsToHost(hostName, port, connectionStatusLabel, form);
			}
		}

		public void SaveContentsToFile(string fileName, IPlugin.ExportFormat format, String target, Boolean writeMeta )
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.saveContentsToFile(fileName, format, target, writeMeta);
			}
		}

		public bool IsPlaying()
		{
			bool isPlaying = false;
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				isPlaying = isPlaying || plugInterface.isPlaying();
			}
			return isPlaying;
		}


		public void EndDragDropTargets()
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.endDragDrop();
			}
		}

		public void ShowDragDropTargets(String path)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.startDragDrop(path);
			}
		}

		public void CreateFirstPersonViewer(hkgVector3CLR pos, hkgVector3CLR dir, hkgVector3CLR up)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.createFirstPersonController(pos, dir, up);
			}
		}

		public void DeleteFirstPersonViewer()
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.deleteFirstPersonController();
			}
		}

		public void UpdateContentFromPackfile(byte[] data)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				IPlugin plugInterface = (IPlugin)m_pluginInstances[i];
				plugInterface.updateContentsFromPackfile(data);
			}
		}

		public void SetDisplayContextOnPlugins(hkgDisplayWorldCLR displayWorld, hkgWindowCLR window)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				// see if the plugin supports the display interface:
				String displayInterfaceName = "Havok.Tool.IPluginDisplay";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] displayList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, displayInterfaceName);
				if (displayList.Length > 0)
				{
					Havok.Tool.IPluginDisplay ed = (Havok.Tool.IPluginDisplay) (m_pluginInstances[i]);
					using (hkgDisplayContextCLR ctx = window.getContext())
					{
						ed.setContext(displayWorld, ctx, window);
					}
				}      
			}
		}

		public void FirePreWorldRender(hkgViewportCLR view)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				String displayInterfaceName = "Havok.Tool.IPluginDisplay";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] displayList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, displayInterfaceName);
				if (displayList.Length > 0)
				{
					Havok.Tool.IPluginDisplay ed = (Havok.Tool.IPluginDisplay)(m_pluginInstances[i]);
					ed.preWorldRender(view);
				}  
			}
		}

		public void FirePostWorldRender(hkgViewportCLR view)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				String displayInterfaceName = "Havok.Tool.IPluginDisplay";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] displayList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, displayInterfaceName);
				if (displayList.Length > 0)
				{
					Havok.Tool.IPluginDisplay ed = (Havok.Tool.IPluginDisplay)(m_pluginInstances[i]);
					ed.postWorldRender(view);
				}
			}
		}

		public void SetOwnerOnPlugins( IPluginOwner owner)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				// see if the plugin supports the display interface:
				String uiInterfaceName = "Havok.Tool.IPluginUI";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] uiList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, uiInterfaceName);
				if (uiList.Length > 0)
				{
					Havok.Tool.IPluginUI eui = (Havok.Tool.IPluginUI)(m_pluginInstances[i]);
					eui.setPluginOwner( owner );
				}
			}
		}

		public void SetParentUIOnPlugins(System.Windows.Forms.TabControl parentCtrl, System.Windows.Forms.MenuStrip parentMenu, Havok.Tool.IConsole errorConsole, System.Windows.Forms.ToolStrip navigationToolBar)
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				// see if the plugin supports the display interface:
				String uiInterfaceName = "Havok.Tool.IPluginUI";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] uiList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, uiInterfaceName);
				if (uiList.Length > 0)
				{
					Havok.Tool.IPluginUI eui = (Havok.Tool.IPluginUI)(m_pluginInstances[i]);
					string name = eui.getName();
                    int priority = eui.getPriority();
					TabPage plugTab = AddPlugTab(parentCtrl, name, priority);
					eui.setParentControl(plugTab, errorConsole);
					eui.setParentMenu(parentMenu);
					eui.setNavigationToolBar(navigationToolBar);
				}
			}
		}

		public void SetParentUIOnTweakers(System.Windows.Forms.TabControl parentCtrl, System.Windows.Forms.MenuStrip parentMenu, Havok.Tool.IConsole errorConsole)
		{
			for (int i = 0; i < m_tweakerInstances.Count; ++i)
			{
				// see if the plugin supports the display interface:
				String uiInterfaceName = "Havok.Tool.IPluginUI";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] uiList = m_tweakerInstances[i].GetType().FindInterfaces(myPluginFilter, uiInterfaceName);
				if (uiList.Length > 0)
				{
					Havok.Tool.IPluginUI eui = (Havok.Tool.IPluginUI)m_tweakerInstances[i];
					string name = eui.getName();
                    int priority = eui.getPriority();
                    TabPage plugTab = AddPlugTab(parentCtrl, name, priority);
					
					eui.setParentControl(plugTab, errorConsole);
					eui.setParentMenu(parentMenu);
				}
			}
		}

		public void RestorePluginDefaultProperties()
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				// see if the plugin supports the display interface:
				String uiInterfaceName = "Havok.Tool.IPluginUI";
				TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
				Type[] uiList = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, uiInterfaceName);
				if (uiList.Length > 0)
				{
					Havok.Tool.IPluginUI eui = (Havok.Tool.IPluginUI)m_pluginInstances[i];
					eui.restoreDefaultProperties();
				}
			}
		}

		public void DeletePlugins()
		{
			for (int i = 0; i < m_pluginInstances.Count; ++i)
			{
				Havok.Tool.IPlugin plug = (Havok.Tool.IPlugin)m_pluginInstances[i];
				
				plug.quit();

				bool isBaseSystem = false;
				for (int ai=0; ai < m_assembliesWithPlugins.Count; ++ai)
				{
					int actualAssmIndex = (int)m_assembliesWithPlugins[ai];
					PluginAssembly ea = (PluginAssembly)m_pluginAssemblies[actualAssmIndex];
					if (ea.baseSystemIsPluginToo && ((IPlugin)ea.baseSystem == plug)) 
					{
						isBaseSystem = true;
						break;
					}
				}

				if (!isBaseSystem) 
				{
					String disposableInterfaceName = "System.IDisposable";
					TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
					Type[] implDisp = m_pluginInstances[i].GetType().FindInterfaces(myPluginFilter, disposableInterfaceName);
					if (implDisp.Length > 0)
					{
						System.IDisposable d = (System.IDisposable)m_pluginInstances[i];
						d.Dispose();
					}
				}
			}

			m_pluginInstances.Clear();
		}

		public void DeleteTweakers()
		{
			for (int i = 0; i < m_tweakerInstances.Count; ++i)
			{
				bool isBaseSystem = false;
				IDataEditor plug = (IDataEditor)m_tweakerInstances[i];

				plug.setDataSource(null);

				for (int ai = 0; ai < m_assembliesWithTweakers.Count; ++ai)
				{
					int actualAssmIndex = (int)m_assembliesWithTweakers[ai];
					PluginAssembly ea = (PluginAssembly)m_pluginAssemblies[actualAssmIndex];
					if (ea.baseSystemIsTweakerToo && ((IDataEditor)ea.baseSystem == plug))
					{
						isBaseSystem = true;
						break;
					}
				}

				if (!isBaseSystem)
				{
					String disposableInterfaceName = "System.IDisposable";
					TypeFilter myPluginFilter = new TypeFilter(MyInterfaceFilter);
					Type[] implDisp = m_tweakerInstances[i].GetType().FindInterfaces(myPluginFilter, disposableInterfaceName);
					if (implDisp.Length > 0)
					{
						System.IDisposable d = (System.IDisposable)m_tweakerInstances[i];
						d.Dispose();
					}
				}
			}

			m_tweakerInstances.Clear();
		}

        public class PluginTweakerTabPage : System.Windows.Forms.TabPage
        {
            public int m_priority;
        }

        public System.Windows.Forms.TabPage AddPlugTab(TabControl tabControl, string name, int priority)
		{
            PluginTweakerTabPage tabPage = new PluginTweakerTabPage();

            tabPage.Location = new System.Drawing.Point(4, 22);
            tabPage.Name = name + "_TabPage";
            tabPage.Padding = new System.Windows.Forms.Padding(3);
			tabPage.Size = new System.Drawing.Size(317, 520);
			tabPage.TabIndex = 1;
			tabPage.Text = name;
			tabPage.UseVisualStyleBackColor = true;
            tabPage.m_priority = priority;

            int i = 0;
            for (; i < tabControl.TabPages.Count; i++)
            {
                if (tabControl.TabPages[i] is PluginTweakerTabPage) // safeguard in case some non PluginTweakerTabPages are part of the control
                {
                    if (priority > (tabControl.TabPages[i] as PluginTweakerTabPage).m_priority)
                    {
                        break;
                    }
                }
            }
            tabControl.TabPages.Insert(i, tabPage);

            return tabPage;
		}

		public System.Collections.ArrayList m_pluginAssemblies;
		public System.Collections.ArrayList m_assembliesWithPlugins;
		public System.Collections.ArrayList m_assembliesWithTweakers;

		private System.Collections.ArrayList /*IPlugin*/ m_pluginInstances;
		private System.Collections.ArrayList /*IDataEditor*/ m_tweakerInstances;
		
		public string m_lastPlugPath;

	}
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
