// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using Havok.Graphics;
using Microsoft.Win32;

namespace Havok.Tool
{
    public partial class GraphicsControl : UserControl
    {
        public MethodInvoker m_hideSplashMethod;
        public MethodInvoker m_showSplashMethod;

        public GraphicsControl(string renderer, IntPtr baseSystem, MethodInvoker showSplashMethod, MethodInvoker hideSplashMethod)
        {
            m_hideSplashMethod = hideSplashMethod;
            m_showSplashMethod = showSplashMethod;

            this.SetStyle(ControlStyles.Opaque | ControlStyles.ResizeRedraw, true);
            AttachToWindow(this.Handle, this, renderer, baseSystem);

            m_lastStep = 0.008f;
            m_pluginStepTimer = Stopwatch.StartNew();

            m_followCamActive = false;
            m_followCamSmoothness = 0.1f;

            m_viewportOptions = new ViewportOptions[4];
            m_viewportOptions[0] = new ViewportOptions();
            m_viewportOptions[1] = new ViewportOptions();
            m_viewportOptions[2] = new ViewportOptions();
            m_viewportOptions[3] = new ViewportOptions();


            
            m_gridCellSize = 1.0f;
            m_gridOrientation = GridOrientation.GRID_XY;
        }

        protected void eventKeyDown(Object sender, System.Windows.Forms.KeyEventArgs e)
        {
            m_constantUpdateKeysDown = true;
        }

        protected void eventKeyUp(Object sender, System.Windows.Forms.KeyEventArgs e)
        {
            m_constantUpdateKeysDown = m_windowHKG.getKeyboard().numKeysPressed() > 0;
        }

        protected hkgSystemCLR.Renderer stringToRenderer( string str )
        {
            string lowerCaseStr = str.ToLower();
            if (lowerCaseStr.StartsWith("d3d") || lowerCaseStr.StartsWith("direct"))
            {
                if (lowerCaseStr.EndsWith("11") || lowerCaseStr.EndsWith("10"))
                    return hkgSystemCLR.Renderer.DIRECT3D11;
                else if (lowerCaseStr.EndsWith("9s") || lowerCaseStr.EndsWith("9"))
                    return hkgSystemCLR.Renderer.DIRECT3D9;
            }
            else if (lowerCaseStr.StartsWith("ogl") || lowerCaseStr.StartsWith("opengl"))
            {
                if (lowerCaseStr.EndsWith("es2"))
                    return hkgSystemCLR.Renderer.OPENGLES2; // GLSL shaders, ala iOS
                if (lowerCaseStr.EndsWith("ls"))
                    return hkgSystemCLR.Renderer.OPENGLS; // Cg shaders, ala PS3
                else // assume just ogles, fixed func
                    return hkgSystemCLR.Renderer.OPENGLES;
            }
            return hkgSystemCLR.Renderer.DIRECT3D11; // safe option (as will try anyway, and fall back to plain 9 if no sm2 or higher)
        }

        // The hostCtrl can be null, but then the window will not have user interaction
        protected void AttachToWindow(IntPtr hostWindowHandle, Control hostWindowCtrl, string renderer, IntPtr baseSystem)
        {
            // Choose which renderer you would like to use here. All should work fine.
            hkgSystemCLR.Renderer actualRenderer;

            const string userRoot0 = "HKEY_CURRENT_USER\\Software\\Havok";
            Boolean msaa = true;
            string subkey;
            bool x64Run = IntPtr.Size > 4;
            if (x64Run) 
            {
                subkey = "ManagedTools_x64";
            }
            else
            {
                subkey = "ManagedTools";
            } 
            
            string keyName = userRoot0 + "\\" + subkey;
            bool changedRenderer = false;
            m_hkgOwner = !hkgBaseSystemCLR.isInitialized();
            
            {
                if ((renderer == null) || (renderer.Length < 0))
                {
                    string defaultRenderer = "Direct3D11";
                    renderer = (string)Registry.GetValue(keyName, "Renderer", defaultRenderer);
                    msaa = ((string)Registry.GetValue(keyName, "MSAA", "True")) == "True";
                    if (x64Run && ((renderer == null) || (renderer.Length < 0)) )// try x86 version
                    {
                        keyName = userRoot0 + "\\ManagedTools";
                        renderer = (string)Registry.GetValue(keyName, "Renderer", defaultRenderer);
                        msaa = ((string)Registry.GetValue(keyName, "MSAA", "True")) == "True";
                    } 
                }

                changedRenderer = (renderer == null) || (renderer.Length < 0);

                hkgSystemCLR.Renderer requestedRenderer = (renderer != null ? stringToRenderer(renderer) : hkgSystemCLR.Renderer.DIRECT3D11);

                actualRenderer = m_hkgOwner? hkgSystemCLR.init(requestedRenderer, baseSystem) : hkgSystemCLR.getCurrentRenderer();

                if ((m_hkgOwner) && (renderer != null) && (actualRenderer != stringToRenderer(renderer)))
                {
                    changedRenderer = true;

                    string requestedRendererStr = hkgSystemCLR.getRendererAsString(requestedRenderer);
                    string actualRendererStr = hkgSystemCLR.getRendererAsString(actualRenderer);
                    string extraInfo = "\nNo reason given.";
                    if (requestedRenderer == hkgSystemCLR.Renderer.DIRECT3D9)
                    {
                        extraInfo = "\nProbably missing d3dx9_43.dll in the dll search path or Havok Content Tools bin dir.";
                    }
                    else if (requestedRenderer == hkgSystemCLR.Renderer.OPENGLS)
                    {
                        extraInfo = "\nProbably missing Cg 2.0 (Cg.dll and CgGL.dll) in the dll search path or Havok Content Tools bin dir.";
                    }
                    else if (requestedRenderer == hkgSystemCLR.Renderer.DIRECT3D11)
                    {
                        extraInfo = "\nProbably no working DirectX install, or just not running in older than XP SP3?";
                    }

                    if (m_hideSplashMethod!=null) m_hideSplashMethod.Invoke();
                    string warningString = "Tried to initialize the " + requestedRendererStr + " but your platform does not support it. " + extraInfo + "\nReverting to the " + actualRendererStr + " renderer.";
                    MessageBox.Show(warningString, "Havok Graphics Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    if (m_showSplashMethod!=null) m_showSplashMethod.Invoke();
                }
            }

            m_windowHKG = new hkgWindowCLR(hostWindowHandle, (msaa ? hkgWindowCLR.WindowCreateFlags.MSAA : hkgWindowCLR.WindowCreateFlags.DEFAULT));
            bool shaderBased = (actualRenderer == hkgSystemCLR.Renderer.DIRECT3D11) || (actualRenderer == hkgSystemCLR.Renderer.DIRECT3D9) || (actualRenderer == hkgSystemCLR.Renderer.OPENGLS) || (actualRenderer == hkgSystemCLR.Renderer.OPENGLES2);
            if (m_hkgOwner && shaderBased)
            {
                if ((m_windowHKG.vertexShaderMajorVersion() < 2) || (m_windowHKG.pixelShaderMajorVersion() < 2))
                {
                    changedRenderer = true;

                    Console.WriteLine("HKG: You requested a shader driven renderer but the platform does nor support Shader Model 2 or higher. Reverting to fixed function.");
                    MessageBox.Show("Tried to initialize a shader driven renderer but your platform does not support Shader Model 2 or higher. Reverting to fixed function renderer", "Havok Graphics Warning", MessageBoxButtons.OK, MessageBoxIcon.Warning );
                    
                    // ignore the request, as graphics card doesn't support it
                    m_windowHKG.Dispose();
                    hkgSystemCLR.quit();
                    hkgSystemCLR.init( hkgSystemCLR.Renderer.OPENGLES, baseSystem);
                    m_windowHKG = new hkgWindowCLR(hostWindowHandle, (msaa ? hkgWindowCLR.WindowCreateFlags.MSAA : hkgWindowCLR.WindowCreateFlags.DEFAULT) );
                }
            }

            // Always revert reg to what worked, so that we don't keep warning etc. Users can change it in the Preview options or via cmdline in standalone etc anyway
            if (changedRenderer)
            {
                Registry.SetValue(keyName, "Renderer", hkgSystemCLR.getRendererAsString(hkgSystemCLR.getCurrentRenderer()));
            }

            hkgDisplayContextCLR context = m_windowHKG.getContext();
            context.lockContext();

            m_windowHKG.setWantDrawHavokLogo(false); // no need as the Form etc is all branded anyway

            if (hostWindowCtrl != null)
            {
                m_windowHKG.hookUpEvents(hostWindowCtrl);

                m_windowHKG.setMouseButtonEventMask(hkgMouseCLR.Button.ANY);

                // Need to continually redraw while a key is pressed (for smooth strafe etc)
                hostWindowCtrl.KeyDown += new System.Windows.Forms.KeyEventHandler(eventKeyDown);
                hostWindowCtrl.KeyUp += new System.Windows.Forms.KeyEventHandler(eventKeyUp);

            }

            m_displayWorldHKG = new hkgDisplayWorldCLR(context, true);
            m_displayWorldHKG.setMutableObjectsUpdatedPerFrame(-1); // Will force all Mutables (cloth and skin) to update AABBs every frame, as we use the AABB for camera tracking and viz aids

            // Some lights
            hkgLightManagerCLR lightManager = m_displayWorldHKG.getLightManager();
            {
                m_defaultLight = new hkgLightCLR();
                m_defaultLight.setType(hkgLightCLR.LightType.DIRECTIONAL);

                hkgVector3CLR v = new hkgVector3CLR();
                v.set(0, 3, 10);
                m_defaultLight.setPosition(v);

                hkgVector3CLR p = new hkgVector3CLR();
                p.set(0, -1, -1);
                p.normalize();
                m_defaultLight.setDirection(p);

                hkgVector3CLR col = new hkgVector3CLR();
                col.set(1, 1, 1);
                m_defaultLight.setDiffuse(col);
                m_defaultLight.setSpecular(col);
                m_defaultLight.setDesiredEnabledState(true);

                lightManager.addLight(m_defaultLight);// will always be light 0 
            }

            // Setup the lighting
            {
                hkgVector3CLR amb = new hkgVector3CLR();
                amb.set(0.1f, 0.1f, 0.1f);
                lightManager.addSceneAmbient(amb);

                hkgVector3CLR zero = new hkgVector3CLR();
                zero.setZero();
                lightManager.computeActiveSet(zero);
            }

            hkgViewportCLR viewport = m_windowHKG.getCurrentViewport();

            // setup the main camera
            {
                hkgCameraCLR cam = viewport.getCamera();
                hkgVector3CLR v = new hkgVector3CLR();
                v.set(10, 10, 10);
                cam.setFrom(v);
                v.set(0, 0, 0);
                cam.setTo(v);
                v.set(0, 0, 1);
                cam.setUp(v);
                viewport.setWorldUp(v);
                cam.computeModelView(false);

                cam.Dispose();
            }
            // set our render state
            {
                viewport.setDesiredState(hkgDisplayContextCLR.EnabledState.TEXTURE2D | hkgDisplayContextCLR.EnabledState.LIGHTING | hkgDisplayContextCLR.EnabledState.ZREAD | hkgDisplayContextCLR.EnabledState.ZWRITE | hkgDisplayContextCLR.EnabledState.ALPHABLEND);
                viewport.setDesiredCullFaceMode(hkgDisplayContextCLR.CullFaceMode.CCW);
                m_windowHKG.setWantViewportBorders(false);
            }

            // make a font
            {
                m_displayFont = new hkgFontCLR();
                m_displayFont.loadFromBuiltin(context, false);
            }

            viewport.setNavigationMode(hkgViewportCLR.CameraNavigationMode.TRACKBALL);
            viewport.setMouseConvention(hkgViewportCLR.MouseConvention.DEFAULT);

            viewport.Dispose();
            lightManager.Dispose();

            context.unlockContext();
            context.Dispose();
        }

        public void EnableHardwareSkinning(bool enabled)
        {
            m_displayWorldHKG.setHardwareSkinning(enabled);
        }

        public void SetState(hkgDisplayContextCLR.EnabledState es, bool on)
        {
            hkgDisplayContextCLR context = m_windowHKG.getContext();
            context.lockContext();

            hkgViewportCLR view = m_windowHKG.getCurrentViewport();
            hkgDisplayContextCLR.EnabledState ds = view.getDesiredState();
            if (on) ds |= es;
            else ds &= ~es;
            view.setDesiredState(ds);

            view.Dispose();
            context.unlockContext();
            context.Dispose();
        }

        public void SetForcedMaterial(bool on)
        {
            hkgDisplayContextCLR context = m_windowHKG.getContext();
            context.lockContext();

            hkgViewportCLR view = m_windowHKG.getCurrentViewport();
            hkgDisplayContextCLR.EnabledState ds = view.getDesiredState();
            hkgDisplayContextCLR.EnabledState es = hkgDisplayContextCLR.EnabledState.MATERIALOVERRIDE;
     
            // Get default material and force
            if ( on )
            {
                ds |= es;
                context.setMaterialOverride(context.getDefaultMaterialOverride());
            }
            else
            {
                ds &= ~es;
                context.setMaterialOverride(null);
            }

            view.setDesiredState(ds);
            view.Dispose();

            context.unlockContext();
            context.Dispose();
        }

        public void SetCullMode(hkgDisplayContextCLR.CullFaceMode cm)
        {
            hkgDisplayContextCLR context = m_windowHKG.getContext();
            context.lockContext();

            hkgViewportCLR view = m_windowHKG.getCurrentViewport();
            view.setDesiredCullFaceMode(cm); 
            
            view.Dispose();
            context.unlockContext();
            context.Dispose();
        }

        public bool SceneHasLighting()
        {
            hkgLightManagerCLR lightManager = m_displayWorldHKG.getLightManager();
            int numLights = lightManager.getNumLights();
            lightManager.Dispose();
            return numLights > 1; // first == flashlight
        }

        public void SetSceneLighting(bool useSceneLights)
        {
            hkgDisplayContextCLR context = m_windowHKG.getContext();
            context.lockContext();

            hkgLightManagerCLR lightManager = m_displayWorldHKG.getLightManager();
            lightManager.setUseSceneLights(useSceneLights);
            
            int numLights = lightManager.getNumLights();
            if (!useSceneLights) // enable 0, disable all others
            {
                lightManager.enable(0);
                for (int li = 1; li < numLights; ++li)
                {
                    lightManager.disable(li);
                }
            }
            else // disable 0, enable all others
            {
                lightManager.disable(0);
                for (int li = 1; li < numLights; ++li)
                {
                    lightManager.enable(li);
                }
            }

            hkgVector3CLR zero = new hkgVector3CLR();
            zero.setZero();
            lightManager.computeActiveSet(zero);

            lightManager.Dispose();
            context.unlockContext();
            context.Dispose();
        }

        public void SetCameraLightTracking(bool trackCam)
        {
            m_defaultLightTrackCamera = trackCam;
        }

        public void Cleanup()
        {
            if (m_displayWorldHKG != null)
            {
                m_displayFont.Dispose();
                m_defaultLight.Dispose();
                m_displayWorldHKG.Dispose();
                m_windowHKG.Dispose();

                m_displayFont = null;
                m_defaultLight = null;
                m_displayWorldHKG = null;
                m_windowHKG = null;
            }
        }

        public ViewportOptions getCurrentViewportOptions()
        {
            return m_viewportOptions[m_windowHKG.getCurrentViewportIndex()];
        }

        public ToolUI m_toolUI;

        public bool m_constantUpdateKeysDown = false;
      
        // Rendering
        public hkgWindowCLR m_windowHKG;
        public bool m_hkgOwner; 

        public hkgDisplayWorldCLR m_displayWorldHKG;
        public bool m_frustumCullEnabled = true;
        public bool m_invalidOrUnknownDevice = true;
        public hkgLightCLR m_defaultLight;

        public class ViewportOptions 
        {
            public bool m_shadowsEnabled = false;
            public bool m_gridEnabled = true;
            
            public bool m_displayEdgedFaces = false;
            public bool m_displayNormals = false;
            public bool m_displayTangents = false;
            public bool m_displayBiTangents = false;

            public enum CameraType
            {
                Perspective = 0,
                OrthoUser   = 1,
                OrthoFront  = 2,
                OrthoBack   = 3,
                OrthoLeft   = 4,
                OrthoRight  = 5,
                OrthoTop    = 6,
                OrthoBottom = 7
            };
            public CameraType m_cameraType = CameraType.Perspective;
        }

        // size 4:
        public ViewportOptions[] m_viewportOptions;

        public int m_currentViewportSettings = 0;

        // Stepping
        public Plugins m_plugins;
        public Stopwatch m_pluginStepTimer;
        public float m_lastStep;

        // Tracking camera
        public Boolean m_defaultLightTrackCamera;
        public Boolean m_followCamActive;
        public float m_followCamSmoothness;

		// Grid 
		public float m_gridCellSize;
        public enum GridOrientation
        {
            GRID_XY,
            GRID_XZ,
            GRID_YZ
        };
        public GridOrientation m_gridOrientation;

        // Display info
        public hkgFontCLR m_displayFont;
        


    } // ctrl
}// namespace

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
