// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Tool/Plugins/Tweaker/hctTweakerPlugin.h>

#pragma unmanaged

#include <Common/Base/Container/String/Deprecated/hkStringOld.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>

namespace
{
    void getSanitizedTypeName(const hkReflect::Type* type, hkStringBuf& nameOut)
    {
        hkStringBuf nameBuf;
        const char* typeName = type->getFullName(nameBuf);

        nameOut.append(typeName);
        nameOut.replace("::", "");
        nameOut.replace("<", "");
        nameOut.replace(">", "");
        nameOut.replace(",", "");
        nameOut.replace(" ", "");
    }
}

#pragma managed

#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

#include <ContentTools/Common/Tool/Plugins/Tweaker/Reflection/hctClassReflection.h>
#include <ContentTools/Common/Tool/Plugins/Tweaker/Reflection/hctClassReflectionTypes.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/Container/String/hkUtf8.hxx>

using namespace System::Threading;
using namespace System::Runtime::InteropServices; // for Marshal of strings to ANSI
using namespace Havok;

#if HK_POINTER_SIZE == 4
#define MGD_PTR_TYPE UInt32;
#define TO_MGD_PTR(VAL) System::Convert::ToUInt32(VAL)
#else
#define MGD_PTR_TYPE UInt64;
#define TO_MGD_PTR(VAL) System::Convert::ToUInt64(VAL)
#endif

namespace
{
    // Use the old class member enum as an index into the type/opcode arrays.
    struct OldClassMember
    {
        enum Type
        {
            /// No type
            TYPE_VOID = 0,
            /// hkBool, boolean type
            TYPE_BOOL,
            /// hkChar, signed char type
            TYPE_CHAR,
            /// hkInt8, 8 bit signed integer type
            TYPE_INT8,
            /// hkUint8, 8 bit unsigned integer type
            TYPE_UINT8,
            /// hkInt16, 16 bit signed integer type
            TYPE_INT16,
            /// hkUint16, 16 bit unsigned integer type
            TYPE_UINT16,
            /// hkInt32, 32 bit signed integer type
            TYPE_INT32,
            /// hkUint32, 32 bit unsigned integer type
            TYPE_UINT32,
            /// hkInt64, 64 bit signed integer type
            TYPE_INT64,
            /// hkUint64, 64 bit unsigned integer type
            TYPE_UINT64,
            /// hkReal, float type
            TYPE_REAL,
            /// hkVector4 type
            TYPE_VECTOR4,
            /// hkQuaternion type
            TYPE_QUATERNION,
            /// hkMatrix3 type
            TYPE_MATRIX3,
            /// hkRotation type
            TYPE_ROTATION,
            /// hkQsTransform type
            TYPE_QSTRANSFORM,
            /// hkMatrix4 type
            TYPE_MATRIX4,
            /// hkTransform type
            TYPE_TRANSFORM,

            TYPE_MAX
        };
    };

    static OldClassMember::Type getBuiltinTypeEnum(const hkReflect::Type* type)
    {
        if (type->asBool())
        {
            return OldClassMember::TYPE_BOOL;
        }
        else if (type->equals<char>())
        {
            return OldClassMember::TYPE_CHAR;
        }
        else if (type->asInteger())
        {
            switch (type->getFormat().get())
            {
            case hkReflect::Format::OfInt<hkInt8>::Value:
                return OldClassMember::TYPE_INT8;
            case hkReflect::Format::OfInt<hkUint8>::Value:
                return OldClassMember::TYPE_UINT8;
            case hkReflect::Format::OfInt<hkInt16>::Value:
                return OldClassMember::TYPE_INT16;
            case hkReflect::Format::OfInt<hkUint16>::Value:
                return OldClassMember::TYPE_UINT16;
            case hkReflect::Format::OfInt<hkInt32>::Value:
                return OldClassMember::TYPE_INT32;
            case hkReflect::Format::OfInt<hkUint32>::Value:
                return OldClassMember::TYPE_UINT32;
            case hkReflect::Format::OfInt<hkInt64>::Value:
                return OldClassMember::TYPE_INT64;
            case hkReflect::Format::OfInt<hkUint64>::Value:
                return OldClassMember::TYPE_UINT64;
            }
        }
        else if (type->asFloat())
        {
            return OldClassMember::TYPE_REAL;
        }
        else if (type->equals<hkVector4>())
        {
            return OldClassMember::TYPE_VECTOR4;
        }
        else if (type->equals<hkQuaternion>())
        {
            return OldClassMember::TYPE_QUATERNION;
        }
        else if (type->equals<hkMatrix3>())
        {
            return OldClassMember::TYPE_MATRIX3;
        }
        else if (type->equals<hkRotation>())
        {
            return OldClassMember::TYPE_ROTATION;
        }
        else if (type->equals<hkQsTransform>())
        {
            return OldClassMember::TYPE_QSTRANSFORM;
        }
        else if (type->equals<hkMatrix4>())
        {
            return OldClassMember::TYPE_MATRIX4;
        }
        else if (type->equals<hkTransform>())
        {
            return OldClassMember::TYPE_TRANSFORM;
        }
        return OldClassMember::TYPE_MAX;
    }

    struct ConvertTypeVisitor : public hkReflect::TypeVisitor < ConvertTypeVisitor, Type^, ReflectionManager% >
    {
        Type^ visit(const hkReflect::VoidType*, ReflectionManager%) { HK_UNREACHABLE(0x3716692f, "Void field found" ); }
        Type^ visit(const hkReflect::OpaqueType*, ReflectionManager%) { return nullptr; }

        Type^ visit(const hkReflect::StringType*, ReflectionManager%)
        {
            return String::typeid;
        }

        Type^ visit(const hkReflect::ValueType* type, ReflectionManager% manager)
        {
            if (const hkReflect::IntType* it = type->asInteger())
            {
                if (Type^ enumType = manager.DefineEnum(it))
                {
                    return enumType;
                }
            }

            // if simple type or one of our value types, look it up
            return manager.GetBuiltinType(type);
        }

        Type^ visit(const hkReflect::RecordType* type, ReflectionManager% manager)
        {
            return manager.CreateType(type);
        }

        Type^ visit(const hkReflect::PointerType* type, ReflectionManager% manager)
        {
            if (!type->getSubType())
            {
                // variant
                return Object::typeid;
            }
            else if (type->getSubType()->asVoid())
            {
                // void*
                return UInt32::typeid;
            }
            else
            {
                return manager.ConvertMemberToType(type->getSubType());
            }
        }

        Type^ visit(const hkReflect::ArrayType* type, ReflectionManager%)
        {
            return ArrayList::typeid;
        }
    };
}

Type^ ReflectionManager::ConvertMemberToType(const hkReflect::Type* type)
{
    if (type->equals<hkVector4>())
    {
        return Vector4::typeid;
    }
    else if (type->equals<hkQuaternion>())
    {
        return Quaternion::typeid;
    }
    else if (type->equals<hkMatrix3>() || type->equals<hkRotation>())
    {
        return Matrix3::typeid;
    }
    else if (type->equals<hkQsTransform>())
    {
        return QsTransform::typeid;
    }
    else if (type->equals<hkMatrix4>())
    {
        return Matrix4::typeid;
    }
    else if (type->equals<hkTransform>())
    {
        return Transform::typeid;
    }
    else
    {
        return ConvertTypeVisitor().dispatch(type, *this);
    }
}

bool ReflectionManager::GetLoadOpcode(Type^ t, OpCode% code)
{
    System::Collections::IEnumerator^ klassEnumerator = m_memberTypes->GetEnumerator();
    int idx = 0;
    while (klassEnumerator->MoveNext() && (idx <= OldClassMember::TYPE_REAL))
    {
        Type^ tp = static_cast<Type^>(klassEnumerator->Current);
        if (tp == t)
        {
            code = (OpCode)m_loadValueOpcodes[idx];
            return true;
        }
        idx++;
    }

    code = OpCodes::Nop;
    return false;
}

Type^ ReflectionManager::GetBuiltinType(const hkReflect::Type* type)
{
    OldClassMember::Type t = getBuiltinTypeEnum(type);
    if (t < m_memberTypes->Count)
    {
        return (Type^)( m_memberTypes[t] );
    }
    return nullptr;
}

Type^ ReflectionManager::DefineEnum(const hkReflect::IntType* type)
{
    if (const hk::Presets* presets = type->findAttribute<hk::Presets>())
    {
        // see if we have created this enum already?
        hkStringBuf enumName;
        getSanitizedTypeName(type, enumName);

        EnumBuilder^ myEnumBuilder = m_moduleBuilder->DefineEnum(gcnew String(enumName.cString()),
            TypeAttributes::Public, Int32::typeid);

        for (int ei = 0; ei < presets->getNumPresets(); ++ei)
        {
            FieldBuilder^ fb = myEnumBuilder->DefineLiteral(gcnew String(presets->getPresetName(ei)),
                hkReflect::IntVar(presets->getPreset(ei)).getValue().convertTo<hkInt32>());
        }

        return myEnumBuilder->CreateType();
    }
    return nullptr;
}

Type^ ReflectionManager::CreateType( const hkReflect::RecordType* klass )
{
    // see if we have created this type already?
    System::Collections::IEnumerator^ klassEnumerator = m_classTypes->GetEnumerator();
    while ( klassEnumerator->MoveNext() )
    {
        TypePair^ tp = (TypePair^)klassEnumerator->Current;
        if (tp->klass->equals(klass))
            return tp->type;
    }

    // It is possible that a member might require this type, if so we just say that the
    // type is Object

    TypePair^ tp = gcnew TypePair;
    tp->klass = klass;
    tp->type = Object::typeid;
    m_classTypes->Add(tp);

    hkStringBuf reflectedName;
    getSanitizedTypeName(klass, reflectedName);
    TypeBuilder^ myTypeBuilder = m_moduleBuilder->DefineType(gcnew String(reflectedName.cString()), TypeAttributes::Public);

    // make parents first
    Type^ parentType = nullptr;
    if (klass->getParentRecord() != HK_NULL)
    {
        parentType = CreateType( klass->getParentRecord() );
        myTypeBuilder->SetParent( parentType );
    }

    // Make our default constructor
    ConstructorBuilder^ ctorBuilder = myTypeBuilder->DefineDefaultConstructor( MethodAttributes::Public );

    // for each member, add a property
    int numMembers = klass->getNumFields();

    Type^ thisType = ReflectionManager::typeid;
    MethodInfo^ genericSetMethod = thisType->GetMethod( "genericSetHelper" );
    MethodInfo^ genericGetMethod = thisType->GetMethod( "genericGetHelper" );

    // Add our ReflectionInfo so we know how to deal with get and sets
    // It is either the parents or we create one
    FieldBuilder^ infoFieldBldr = nullptr;
    FieldInfo^ infoField = nullptr;
    if (parentType != nullptr)
    {
        infoField = parentType->GetField("HavokInternalInfo");
    }
    else /// make one
    {
        infoFieldBldr = myTypeBuilder->DefineField( "HavokInternalInfo", ReflectionInfo::typeid, FieldAttributes::Public );
    }

    int memberOffset = parentType? hkReflectUtil::getRecordNumFieldsWithAncestors(klass->getParentRecord()) : 0;
    hkArrayView<const hkReflect::FieldDecl> fields = klass->getFields();
    for (int mi = 0; mi < fields.getSize(); ++mi)
    {
        hkReflect::FieldDecl member = fields[mi];
        int memberIndex = memberOffset + mi;

        // Add some properties (with get and set methods)
        Type^ memberType = ConvertMemberToType( member.getType() );
        if (memberType == nullptr) // maybe be null if a "zero pointer" etc
            continue;

        PropertyBuilder^ prop = myTypeBuilder->DefineProperty(gcnew String(member.getName()), PropertyAttributes::None,
                                                            memberType, nullptr);

        Type^ nullType = nullptr;
        array<Type^>^ memberSetTypes = gcnew array<Type^>( 1 ); memberSetTypes[0] = memberType;

        hkStringOld propName;
        propName = "get_";
        propName += member.getName();
        MethodBuilder^ getPropMthdBldr = myTypeBuilder->DefineMethod(
            gcnew String(propName.cString()),
            MethodAttributes::Public,
            memberType,
            nullptr);

        ILGenerator^ getILGen = getPropMthdBldr->GetILGenerator();
        {
            LocalBuilder^ data = getILGen->DeclareLocal( Object::typeid ); // local data object

            getILGen->Emit(OpCodes::Ldarg_0); // this instance ptr (arg0, for use with ldfld)
            if (infoField)
                getILGen->Emit(OpCodes::Ldfld, infoField); // load our Havok field given the instance
            else
                getILGen->Emit(OpCodes::Ldfld, infoFieldBldr); // load our Havok field given the instance
            getILGen->Emit(OpCodes::Ldc_I4, memberIndex); // load the member index, 32bit
            getILGen->Emit(OpCodes::Ldloca, data); //load addr of the local data object

            // call the genertic get method
            getILGen->EmitCall(OpCodes::Call, genericGetMethod, nullptr); //Get the value

            getILGen->Emit(OpCodes::Ldloc_0); //ret object (local data object)

            // unbox the returned data if possible
            if (  memberType->IsValueType )
            {
                getILGen->Emit(OpCodes::Unbox, memberType );
                OpCode opCode;
                if (GetLoadOpcode(memberType, opCode) ) //maybe a no op
                    getILGen->Emit(opCode);// convert the int (the unboxed top of the stack) to a natural int
                else
                    getILGen->Emit(OpCodes::Ldobj, memberType); // value object load
            }

            getILGen->Emit(OpCodes::Ret);
        }
        prop->SetGetMethod(getPropMthdBldr);

        if (m_canModify)
        {
            propName = "set_";
            propName += member.getName();
            MethodBuilder^ setPropMthdBldr = myTypeBuilder->DefineMethod(
                gcnew String(propName.cString()),
                MethodAttributes::Public,
                nullptr,
                memberSetTypes);

            ILGenerator^ setILGen = setPropMthdBldr->GetILGenerator();
            {
                setILGen->Emit(OpCodes::Ldarg_0); // this instance ptr (arg0, for use with ldfld)
                if (infoField)
                    setILGen->Emit(OpCodes::Ldfld, infoField); // load our Havok field given the instance
                else
                    setILGen->Emit(OpCodes::Ldfld, infoFieldBldr); // load our Havok field given the instance
            setILGen->Emit(OpCodes::Ldc_I4, memberIndex); // load the member index
                setILGen->Emit(OpCodes::Ldarg_1); // data to set (first arg)
                if (memberType->IsValueType)
                {
                    setILGen->Emit(OpCodes::Box, memberType); // we know it is a value type (int etc.)
                }

                // call the genertic set method
                setILGen->EmitCall(OpCodes::Call, genericSetMethod, nullptr); //Set the value

                // we're done, return
                setILGen->Emit(OpCodes::Ret);
            }
            prop->SetSetMethod(setPropMthdBldr);
        }
    }

    Type^ t = myTypeBuilder->CreateType(); // have to pin it so that we can ref it in out class map
    tp->type = t; // finally we can set the proper type in the cache.

    return t;
}

void ReflectionManager::GetReflectedSource( Object^ obj, const hkReflect::Type*& klass, void*& data)
{
    Type^ type = obj->GetType();
    FieldInfo^ theHavokField = type->GetField("HavokInternalInfo");
    if (theHavokField != nullptr)
    {
        ReflectionInfo^ rf = (ReflectionInfo^) theHavokField->GetValue(obj);
        if (rf != nullptr)
        {
            klass = rf->klass;
            data = rf->data;
            return;
        }
    }
    klass = HK_NULL;
    data = HK_NULL;
}

Object^ ReflectionManager::ReflectInstance(const hkReflect::RecordType* klass, void* data)
{
    // Current type (see if it is the real instance type)
    Type^ type = CreateType( klass );


    array<Type^>^ derivedTypes = gcnew array<Type^>(1); derivedTypes[0] = Object::typeid;
    ConstructorInfo^ ci = type->GetConstructor(derivedTypes);
    Object^ inst = Activator::CreateInstance(type, nullptr);

    // Create the actual type
    ReflectionInfo^ thisInfo = gcnew ReflectionInfo();
    thisInfo->data = (char*)data;
    thisInfo->klass = klass;
    thisInfo->manager = this;

    FieldInfo^ theHavokField = type->GetField("HavokInternalInfo");
    theHavokField->SetValue(inst, thisInfo);

    return inst;
}

static inline void _copyhkVector4(Havok::Vector4% v4, const hkVector4* hv4)
{
    v4.x = (*hv4)(0);
    v4.y = (*hv4)(1);
    v4.z = (*hv4)(2);
    v4.w = (*hv4)(3);
}

static inline void _copyVector4(hkVector4* hv4, Havok::Vector4^ v4 )
{
    hv4->set(v4->x, v4->y, v4->z, v4->w );
}

static inline void _copyhkQuaternion(Havok::Quaternion% q, const hkQuaternion* hq)
{
    q.x = (*hq)(0);
    q.y = (*hq)(1);
    q.z = (*hq)(2);
    q.w = (*hq)(3);
}

static inline void _copyQuaternion(hkQuaternion* hq, Havok::Quaternion^ q )
{
    hq->set( q->x, q->y, q->z, q->w);
}

static inline void _copyhkMatrix3(Matrix3% m3, const hkMatrix3* hm3)
{
    _copyhkVector4(m3.m_col0, &hm3->getColumn(0));
    _copyhkVector4(m3.m_col1, &hm3->getColumn(1));
    _copyhkVector4(m3.m_col2, &hm3->getColumn(2));
}

static inline void _copyMatrix3(hkMatrix3* hm3, Matrix3^ m3)
{
    _copyVector4(&hm3->getColumn(0), m3->col0);
    _copyVector4(&hm3->getColumn(1), m3->col1);
    _copyVector4(&hm3->getColumn(2), m3->col2);
}

static void _copyhkMatrix4(Matrix4% m4, const hkMatrix4* hm4)
{
    _copyhkVector4(m4.m_col0, &hm4->getColumn(0));
    _copyhkVector4(m4.m_col1, &hm4->getColumn(1));
    _copyhkVector4(m4.m_col2, &hm4->getColumn(2));
    _copyhkVector4(m4.m_col3, &hm4->getColumn(3));
}

static void _copyMatrix4(hkMatrix4* hm4, Matrix4^ m4)
{
    _copyVector4(&hm4->getColumn(0), m4->m_col0 );
    _copyVector4(&hm4->getColumn(1), m4->m_col1 );
    _copyVector4(&hm4->getColumn(2), m4->m_col2 );
    _copyVector4(&hm4->getColumn(3), m4->m_col3 );
}


void ReflectionManager::genericSetHelper(Havok::ReflectionInfo^ info, int memberIndex, Object^ data)
{
    hkReflect::FieldDecl fieldType = hkReflectUtil::getRecordFieldWithAncestors(info->klass, memberIndex);

    hkReflect::RecordVar obj(info->data, info->klass);
    hkReflect::Var field = obj[fieldType];

    info->manager->genericSet(field, data);
}

void ReflectionManager::genericGetHelper(Havok::ReflectionInfo^ info, int memberIndex, Object^% data)
{
    hkReflect::FieldDecl fieldType = hkReflectUtil::getRecordFieldWithAncestors(info->klass, memberIndex);

    hkReflect::RecordVar obj(info->data, info->klass);
    hkReflect::Var field = obj[fieldType];

    data = info->manager->genericGet(field);
}

namespace
{
    struct SetVisitor : public hkReflect::VarVisitor < SetVisitor, void, Object^, ReflectionManager% >
    {
        void visit(const hkReflect::VoidVar& var, Object^ data, ReflectionManager%) { HK_UNREACHABLE(0x39d4332, "Void field found" ); }
        void visit(const hkReflect::OpaqueVar& var, Object^ data, ReflectionManager%) { /* do nothing */ }

        void visit(const hkReflect::BoolVar& var, Object^ data, ReflectionManager%)
        {
            var.setValue(System::Convert::ToBoolean(data));
        }

        void visit(const hkReflect::IntVar& var, Object^ data, ReflectionManager%)
        {
            if (var.getType()->findAttribute<hk::Presets>())
            {
                // Enums are forced to hkInt32.
                var.setValue(System::Convert::ToUInt32(data));
            }
            else if (var.dynCast<char>())
            {
                var.setValue((char)System::Convert::ToUInt16(data)); /*xx unicode loss*/
            }
            else
            {
                switch (var.getType()->getFormat().get())
                {
                case hkReflect::Format::OfInt<hkInt8>::Value:
                    var.setValue(System::Convert::ToSByte(data)); break;
                case hkReflect::Format::OfInt<hkUint8>::Value:
                    var.setValue(System::Convert::ToByte(data)); break;
                case hkReflect::Format::OfInt<hkInt16>::Value:
                    var.setValue(System::Convert::ToInt16(data)); break;
                case hkReflect::Format::OfInt<hkUint16>::Value:
                    var.setValue(System::Convert::ToUInt16(data)); break;
                case hkReflect::Format::OfInt<hkInt32>::Value:
                    var.setValue(System::Convert::ToInt32(data)); break;
                case hkReflect::Format::OfInt<hkUint32>::Value:
                    var.setValue(System::Convert::ToUInt32(data)); break;
                case hkReflect::Format::OfInt<hkInt64>::Value:
                    var.setValue(System::Convert::ToInt64(data)); break;
                case hkReflect::Format::OfInt<hkUint64>::Value:
                    var.setValue(System::Convert::ToUInt64(data)); break;
                }
            }
        }

        void visit(const hkReflect::FloatVar& var, Object^ data, ReflectionManager%)
        {
            var.setValue(System::Convert::ToSingle(data));
        }

        void visit(const hkReflect::ArrayVar& var, Object^ data, ReflectionManager% manager)
        {
            ArrayList^ arr = (ArrayList^)data;
            if (arr)
            {
                var.setArraySize(arr->Count);
                for (int i = 0; i < arr->Count; ++i)
                {
                    manager.genericSet(var[i], arr[i]);
                }
            }
        }

        void visit(const hkReflect::PointerVar& var, Object^ data, ReflectionManager% manager)
        {
            if (hkReflect::Var tgt = var.getValue())
            {
                if (hkReflect::RecordVar recTgt = tgt)
                {
                    // reflect in changed data??
                }
                else
                {
                    if (var.getSubType()) // if subtype is null we should try to do a lookup from the address
                    {
                        // set ptr as integer
                        hkUlong value = TO_MGD_PTR(data);
                        var.setValue(hkReflect::Var(reinterpret_cast<void*>(value), var.getSubType()));
                    }
                }
            }
        }

        void visit(const hkReflect::StringVar& var, Object^ data, ReflectionManager% manager)
        {
            String^ s = data->ToString();
            // copy the string in case this is a char*
            char* copy = hkString::strDup(hkUtf8::Utf8FromWide(s).cString());
            manager.RememberAlloc(IntPtr(copy));
            var.setValue(copy);
        }

        void visit(const hkReflect::RecordVar& var, Object^ data, ReflectionManager% manager)
        {
            void* addr;
            const hkReflect::Type* type;
            manager.GetReflectedSource(data, type, addr);
            if (addr && type)
            {
                var.assign(hkReflect::Var(addr, type));
            }
        }
    };

    struct GetVisitor : public hkReflect::VarVisitor < GetVisitor, Object^, ReflectionManager% >
    {
        Object^ visit(const hkReflect::VoidVar& var, ReflectionManager%) { HK_UNREACHABLE(0x282cf034, "Void field found" ); }
        Object^ visit(const hkReflect::OpaqueVar& var, ReflectionManager%) { return nullptr; }

        Object^ visit(const hkReflect::BoolVar& var, ReflectionManager%)
        {
            return (Boolean)var.getValue();
        }

        Object^ visit(const hkReflect::IntVar& var, ReflectionManager%)
        {
            if (var.getType()->findAttribute<hk::Presets>())
            {
                // Enums are forced to hkInt32.
                return (Int32)var.getValue().convertTo<hkInt32>();
            }

            if (const char* value = var.dynCast<char>())
            {
                return (UInt16)value;
            }

            switch (var.getType()->getFormat().get())
            {
            case hkReflect::Format::OfInt<hkInt8>::Value:
                return (SByte)var.getValue().convertTo<hkInt8>();
            case hkReflect::Format::OfInt<hkUint8>::Value:
                return (Byte)var.getValue().convertTo<hkUint8>();
            case hkReflect::Format::OfInt<hkInt16>::Value:
                return (Int16)var.getValue().convertTo<hkInt16>();
            case hkReflect::Format::OfInt<hkUint16>::Value:
                return (UInt16)var.getValue().convertTo<hkUint16>();
            case hkReflect::Format::OfInt<hkInt32>::Value:
                return (Int32)var.getValue().convertTo<hkInt32>();
            case hkReflect::Format::OfInt<hkUint32>::Value:
                return (UInt32)var.getValue().convertTo<hkUint32>();
            case hkReflect::Format::OfInt<hkInt64>::Value:
                return (Int64)var.getValue().convertTo<hkInt64>();
            case hkReflect::Format::OfInt<hkUint64>::Value:
                return (UInt64)var.getValue().convertTo<hkUint64>();
            default:
                return nullptr;
            }
        }

        Object^ visit(const hkReflect::FloatVar& var, ReflectionManager%)
        {
            return (Single)var.getValue();
        }

        Object^ visit(const hkReflect::ArrayVar& var, ReflectionManager% manager)
        {
            // regular array
            ArrayList^ arr = gcnew ArrayList();
            for (int i = 0; i < var.getCount(); ++i)
            {
                arr->Add(manager.genericGet(var[i]));
            }
            return arr;
        }

        Object^ visit(const hkReflect::StringVar& var, ReflectionManager%)
        {
            if (hkReflect::StringValue value = var.getValue())
            {
                return gcnew String(value);
            }
            return nullptr;
        }

        Object^ visit(const hkReflect::RecordVar& var, ReflectionManager% manager)
        {
            return manager.ReflectInstance(var.getType(), var.getAddress());
        }

        Object^ visit(const hkReflect::PointerVar& var, ReflectionManager% manager)
        {
            if (hkReflect::Var tgt = var.getValue())
            {
                if (hkReflect::RecordVar recTgt = tgt)
                {
                    return manager.genericGet(recTgt);
                }
                else
                {
                    // return ptr as integer
                    return gcnew MGD_PTR_TYPE(reinterpret_cast<hkUlong>(tgt.getAddress()));
                }
            }
            return nullptr;
        }
    };
}

void ReflectionManager::genericSet(const hkReflect::Var& field, Object^ data)
{
    if (hkVector4* hv4 = field.dynCast<hkVector4>())
    {
        Vector4^ v4 = (Vector4^)(data);
        _copyVector4(hv4, v4); // same layout
        return;
    }
    else if (hkQuaternion* hq = field.dynCast<hkQuaternion>())
    {
        Quaternion^ q = (Quaternion^)(data);
        _copyQuaternion(hq, q); // same layout
        return;
    }
    else if (hkMatrix3* hm3 = field.dynCast<hkMatrix3>())
    {
        Matrix3^ m3 = (Matrix3^)(data);
        _copyMatrix3(hm3, m3);
        return;
    }
    else if (hkRotation* hm3 = field.dynCast<hkRotation>())
    {
        Matrix3^ m3 = (Matrix3^)(data);
        _copyMatrix3(hm3, m3);
        return;
    }
    else if (hkQsTransform* hqs = field.dynCast<hkQsTransform>())
    {
        QsTransform^ qs = (QsTransform^)(data);
        _copyQuaternion(&hqs->m_rotation, qs->m_rot);
        _copyVector4(&hqs->m_translation, qs->m_trans);
        _copyVector4(&hqs->m_scale, qs->m_scale);
        return;
    }
    else if (hkMatrix4* hm4 = field.dynCast<hkMatrix4>())
    {
        Matrix4^ m4 = (Matrix4^)data;
        _copyMatrix4(hm4, m4);
        return;
    }
    else if (hkTransform* ht = field.dynCast<hkTransform>())
    {
        Transform^ t = (Transform^)(data);
        _copyMatrix3(&ht->getRotation(), t->m_rot);
        _copyVector4(&ht->getTranslation(), t->m_trans);
        return;
    }
    else
    {
        SetVisitor().dispatch(field, data, *this);
    }
}

Object^ ReflectionManager::genericGet(const hkReflect::Var& field)
{
    Object^ data;
    if (const hkVector4* hv4 = field.dynCast<hkVector4>())
    {
        Vector4 v4;
        _copyhkVector4(v4, hv4); // same layout
        data  = v4;
    }
    else if (const hkQuaternion* hq = field.dynCast<hkQuaternion>())
    {
        Quaternion q;
        _copyhkQuaternion(q, hq); // same layout
        data  = q;
    }
    else if (const hkMatrix3* hm3 = field.dynCast<hkMatrix3>())
    {
        Matrix3 m3;
        _copyhkMatrix3(m3, hm3);
        data  = m3;
    }
    else if (const hkQsTransform* hqs = field.dynCast<hkQsTransform>())
    {
        QsTransform qs;
        _copyhkQuaternion(qs.m_rot, &hqs->m_rotation);
        _copyhkVector4(qs.m_trans, &hqs->m_translation);
        _copyhkVector4(qs.m_scale, &hqs->m_scale);
        data  = qs;
    }
    else if (const hkMatrix4* hm4 = field.dynCast<hkMatrix4>())
    {
        Matrix4 m4;
        _copyhkMatrix4(m4, hm4);
        data  = m4;
    }
    else if (const hkTransform* ht = field.dynCast<hkTransform>())
    {
        Transform t;
        _copyhkMatrix3(t.m_rot, &ht->getRotation());
        _copyhkVector4(t.m_trans, &ht->getTranslation());
        data  = t;
    }
    else
    {
        data = GetVisitor().dispatch(field, *this);
    }
    return data;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
