// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : PHYSICS_2012
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <Plugins/Preview/hctPreviewPlugin.h> //PCH

#pragma unmanaged

#include <Common/Base/hkBase.h>
#include <Common/Base/KeyCode.h>
#include <Common/Base/Math/Matrix/hkMatrixDecomposition.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Plugins/Preview/hctPreviewAsset.h>

#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>

#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>

#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Collide/Dispatch/hkpAgentRegisterUtil.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Dynamics/Collide/Filter/Constraint/hkpConstraintCollisionFilter.h>
#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.h>
#include <Physics2012/Utilities/Dynamics/SaveContactPoints/hkpSaveContactPointsUtil.h>
#include <Physics2012/Utilities/Dynamics/SaveContactPoints/hkpPhysicsSystemWithContacts.h>
#include <Physics2012/Utilities/Dynamics/RigidBodyReset/hkpRigidBodyResetUtil.h>

#include <Common/Visualize/hkProcess.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpConvexRadiusViewer.h>

#include <Plugins/Preview/hctPreviewUnmanaged.h>
#include <Plugins/Preview/Physics/hctPreviewPhysics.h>

using namespace PreviewPlugin;

static hkgDisplayObject* _findDisplayObject( hkxNode* meshNode, hkArray<hkgAssetConverter::NodeMapping>& mappings)
{
    for (int mi=0; mi < mappings.getSize(); ++mi)
    {
        if (mappings[mi].m_origNode == meshNode)
            return (hkgDisplayObject*)( mappings[mi].m_hkgObject );
    }
    return HK_NULL;
}



static void _addOrRemoveShapeNodes (hkgDisplayObject* dispObj, hkxNode* node, hkArray<hkgAssetConverter::NodeMapping>& meshes, const hkArray<hkpPhysicsSystem*>& systems, bool rbHasShape )
{
    // Check
    {
        // RB children should not be merged or removed
        const bool nodeIsRb = node->findAttributeGroupByName("hkRigidBody")!=HK_NULL;
        if (nodeIsRb) return;

        // If it wasn't a shape originally then we don't do anything with it
        const bool isShape = node->findAttributeGroupByName("hkShape")!=HK_NULL;
        if (!isShape) return;
    }

    // could be an old style export scene, so will not have attributes
    // check for rbs of the same name..
    if (node->m_name)
    {
        for (int psi=0; psi < systems.getSize(); ++psi)
        {
            const hkArray<hkpRigidBody*>& rbs = systems[psi]->getRigidBodies();
            for (int rbl = 0; rbl < rbs.getSize(); ++rbl)
            {
                if (rbs[rbl]->getName() && (hkString::strCmp( rbs[rbl]->getName(), node->m_name) == 0) )
                    return; // rb by the same name as this node so can assume it is this one
            }
        }
    }

    // See EXP-630 - We merge meshes for shapes for compound rigid bodies, but not for proxied rigid bodies
    const bool merge = dispObj && rbHasShape;

    // Look for the mesh of the shape, and merge (transforming it) it or remove it
    if ( node->m_object && hkReflect::exactMatchDynCast<hkxMesh>(node->m_object) )
    {
        hkgDisplayObject* shapeDisplay = _findDisplayObject( node, meshes );

        if (merge) // Compound rigid body - we merge the meshes of the shapes
        {
            const float* tA = shapeDisplay->getTransform();
            const float* tB = dispObj->getTransform();
            float bInv[16];
            float geomT[16];
            hkgMat4Invert(bInv, tB);
            hkgMat4Mult(geomT, bInv, tA);

            const int numGeom = shapeDisplay->getNumGeometry();
            for (int gi=numGeom-1; gi >= 0; --gi)
            {
                hkgGeometry* geom = shapeDisplay->removeGeometry(gi);
                geom->transform(geomT); // bake in relative transform
                dispObj->addGeometry(geom);
                geom->removeReference(); // remove ref given back by previous remove
            }
        }
        else    // Rigid Body with proxy shapes - we don't display the meshes of the shapes - remove them
        {
            const int numGeom = shapeDisplay->getNumGeometry();
            for (int gi=numGeom-1; gi >= 0; --gi)
            {
                hkgGeometry* geom = shapeDisplay->removeGeometry(gi);
                geom->removeReference(); // remove ref given back by previous remove
            }

        }
    }

    // recurse
    for (int ci=0; ci < node->m_children.getSize(); ++ci)
    {
        _addOrRemoveShapeNodes(dispObj, node->m_children[ci], meshes, systems, rbHasShape );
    }
}



void PreviewAssetManager::addPhysicsObjects( PreviewAsset* asset )
{
    hkRootLevelContainer* root = asset->m_container;
    hkxScene* sceneContainer = root->findObject<hkxScene>();
    hkpPhysicsData* physicsContainer = root->findObject<hkpPhysicsData>();

    if (!physicsContainer)
    {
        return;
    }

    // create a world (only if don't have one already from prev / other asset)
    const bool mustCreateWorld = (m_currentWorld == HK_NULL);
    hkpWorldCinfo* wcinfo = physicsContainer->getWorldCinfo();
    if ( mustCreateWorld )
    {
        // Make a default one if none found
        if ( !wcinfo )
        {
            wcinfo = new hkpWorldCinfo;
            physicsContainer->setWorldCinfo(wcinfo);
            wcinfo->removeReference();

            // usually y is the up axis
            wcinfo->m_gravity.set(0.0f,-9.8f,0.0f);

            hkxScene* theScene = root->findObject<hkxScene>();
            if(theScene)
            {
                // 3ds max uses z as up axis
                hkStringOld modellerName(theScene->m_modeller);
                if (modellerName.substr(0,3)=="3ds")
                {
                    wcinfo->m_gravity.set(0.0f,0.0f,-9.8f);
                }

                // Transform the direction of gravity with the scene transform
                hkMatrixDecomposition::Decomposition decomp;
                hkMatrixDecomposition::decomposeMatrix(theScene->m_appliedTransform, decomp);
                wcinfo->m_gravity.setRotatedDir(decomp.m_basis, wcinfo->m_gravity);
            }
        }

        //Don't remove objects from the world when they leave the broadphase, as this can break some plugins.
        wcinfo->m_broadPhaseBorderBehaviour = hkpWorldCinfo::BROADPHASE_BORDER_DO_NOTHING;

        if (HK_PREVIEW_MULTITHREADED_PHYSICS==0)
        {
            // EXP-2501 Simulation type used to be always set to Continuous, now it is kept how the user set it, and a check is carried out.
            HK_ASSERT(0xabba5a34,
                wcinfo->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS ||
                wcinfo->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_DISCRETE,
                "Unknown simulation type found. Must be DISCRETE or CONTINUOUS");
        }
        else
        {
            wcinfo->m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED;
        }

        m_currentWorld = new hkpWorld( *wcinfo );

        m_currentWorld->lock();

        hkpAgentRegisterUtil::registerAllAgents( m_currentWorld->getCollisionDispatcher() );

        if (HK_PREVIEW_MULTITHREADED_PHYSICS!=0)
        {
            hkpMultithreadConfig config;
            m_currentWorld->setMultithreadConfig( config, m_jobQueue );
            m_currentWorld->registerWithJobQueue( m_jobQueue );
        }

        // EXP-2501 Set the collision filter
        if ((wcinfo->m_collisionFilter && wcinfo->m_collisionFilter->m_type != hkpCollisionFilter::HK_FILTER_CONSTRAINT ) ||
            (!wcinfo->m_collisionFilter))
        {
            hkpCollisionFilter* newFilter = new hkpConstraintCollisionFilter(wcinfo->m_collisionFilter);
            m_currentWorld->setCollisionFilter(newFilter);
            newFilter->removeReference();
        }

        m_ghostDisplayHandler = new hkgDisplayHandler( m_displayWorld, m_displayContext, m_displayContext->getOwner() );
        m_ghostPhysicsContext = new hkpPhysicsContext();

        hkArray< hkProcessContext* > ctxs; ctxs.pushBack( m_ghostPhysicsContext );

        m_ghostProcesses.pushBack( hkProcessFactory::getInstance().createProcess( hkpShapeDisplayViewer::getName(), ctxs ) );
        {
            hkpShapeDisplayViewer* p = (hkpShapeDisplayViewer*)( m_ghostProcesses.back() );

            p->setAutoColorMode(true);
            p->setAutoColorModeOverride(true);
            p->setFixedObjectColor( m_ghostsEnabled? ALPHA_PHYSICS_COLOR : ALPHA_PHYSICS_TRANSPARENT );
            p->setMovableObjectColor( m_ghostsEnabled? ALPHA_PHYSICS_COLOR : ALPHA_PHYSICS_TRANSPARENT );

            p->m_displayHandler = m_ghostDisplayHandler;
            p->init();
        }

        m_ghostProcesses.pushBack( hkProcessFactory::getInstance().createProcess( hkpConvexRadiusViewer::getName(), ctxs ) );
        {
            hkpConvexRadiusViewer* p = (hkpConvexRadiusViewer*)( m_ghostProcesses.back() );
            p->setFixedObjectColor( m_ghostsEnabled? ALPHA_PHYSICS_RADIUS_COLOR : ALPHA_PHYSICS_TRANSPARENT );
            p->setMovableObjectColor( m_ghostsEnabled? ALPHA_PHYSICS_RADIUS_COLOR : ALPHA_PHYSICS_TRANSPARENT );

            p->m_displayHandler = m_ghostDisplayHandler;
            p->init();
        }

        m_desiredGravity = wcinfo->m_gravity;
        m_loadedGravity = wcinfo->m_gravity;

        m_currentWorld->unlock();
    }

    // Create collision filter if there's none
    if ( !m_currentWorld->m_collisionFilter )
    {
        if ((wcinfo->m_collisionFilter && wcinfo->m_collisionFilter->m_type != hkpCollisionFilter::HK_FILTER_CONSTRAINT ) ||
            (!wcinfo->m_collisionFilter))
        {
            hkpCollisionFilter* newFilter = new hkpConstraintCollisionFilter(wcinfo->m_collisionFilter);
            m_currentWorld->setCollisionFilter(newFilter);
            newFilter->removeReference();
        }

        if ((wcinfo->m_collisionFilter && wcinfo->m_collisionFilter->m_type == hkpCollisionFilter::HK_FILTER_CONSTRAINT ))
        {
            m_currentWorld->setCollisionFilter(wcinfo->m_collisionFilter);
        }
    }

    if ( m_ghostPhysicsContext )
    {
        m_ghostPhysicsContext->addWorld( m_currentWorld );
    }

    const hkArray<hkpPhysicsSystem*>& ps = physicsContainer->getPhysicsSystems();

    // Physics Display
    bool needGraphicsHookup = sceneContainer && asset->m_convertedGraphicsObjects && m_displayHandler;
    for (int psi=0; needGraphicsHookup && (psi < ps.getSize()); ++psi)
    {
        const hkArray<hkpRigidBody*>& rbs = ps[psi]->getRigidBodies();
        for (int rbi=0; rbi < rbs.getSize(); ++rbi)
        {
            hkpRigidBody* rigidBody = rbs[rbi];
            if ( rigidBody->m_breakableBody )
            {
                continue;   // breakable bodies will get their own graphics
            }

            // precreate objects linked up:
            const char* name = rigidBody->getName();
            if (!name)
            {
                continue;
            }

            hkxNode* node = sceneContainer->findNodeByName(name);
            if ( !node )
            {
                continue;
            }

            // Note that mesh may be null if the rigid body is not a geometric object (see EXP-614)
            hkUint64 id = hkUint64( rigidBody->getCollidable() );
            hkgDisplayObject* dispObj =_findDisplayObject( node, asset->m_convertedGraphicsObjects->m_meshNodes );

            // EXP-630 : If a rigid body with proxies do not merge the children
            // Compound rigid bodies have a shape in the rigid body node, but proxied rbs don't
            const bool rbHasShape = node->findAttributeGroupByName("hkShape")!=HK_NULL;

            // the body may be a compound object
            // so we should search for all mesh shape nodes
            // below this one (not recursing into ones with rigid body attributes)
            // and merge the display objects together for them
            for (int ci=0; ci < node->m_children.getSize(); ++ci)
            {
                _addOrRemoveShapeNodes (dispObj, node->m_children[ci], asset->m_convertedGraphicsObjects->m_meshNodes, ps , rbHasShape);
            }

            if (dispObj)
            {
                dispObj->computeAABB();

                m_displayHandler->addPrecreatedDisplayObject( id, dispObj );

                //m_physicsDisplayObjects.pushBack( dispObj );

                if ( !rigidBody->isFixed() ) // make a shadow caster.
                {
                    // EXP-760 removeDisplayObject doesn't decrease ref counting (returns worlds one) we need to check its
                    // result (as the display object may not be removed) and decrease references ourselves
                    dispObj->setStatusFlags( dispObj->getStatusFlags() | HKG_DISPLAY_OBJECT_DYNAMIC | HKG_DISPLAY_OBJECT_SHADOWCASTER );
                }
            }
            else
            {
                // EXP-630 (example 2)
                // The rb has no mesh - that's fine, pretend we did create one
                m_displayHandler->addPrecreatedDisplayObject( id, HK_NULL );
            }
        }
    }

    // due to merging, we may be left with some empty display objects at this stage, so we can remove them
    // to avoid work
    for (int doi=0; m_displayWorld && (doi < m_displayWorld->getNumDisplayObjects());)
    {
        if (m_displayWorld->getDisplayObject(doi)->getNumGeometry() < 1)
        {
            m_displayWorld->removeDisplayObject(doi)->removeReference();
        }
        else
        {
            doi++;
        }
    }


    //
    // Add all new bodies to the world
    // As we have associated the precreated (from HKX) display object
    // they will not autocreate
    //
    m_currentWorld->lock();

    const hkArray<hkpPhysicsSystem*>& systems = physicsContainer->getPhysicsSystems();
    for (int p=0; p < systems.getSize(); ++p)
    {
        m_currentWorld->addPhysicsSystem(systems[p]);

        if (systems[p]->hasContacts())
        {
            hkpSaveContactPointsUtil::LoadPointsInput input;
            hkpPhysicsSystem* ps = systems[p];
            hkpPhysicsSystemWithContacts* systemWithContacts = static_cast<hkpPhysicsSystemWithContacts*>(ps);
            hkpSaveContactPointsUtil::loadContactPoints(input, systemWithContacts, m_currentWorld);
        }
    }

    m_currentWorld->updateCollisionFilterOnWorld(HK_UPDATE_FILTER_ON_WORLD_FULL_CHECK, HK_UPDATE_COLLECTION_FILTER_PROCESS_SHAPE_COLLECTIONS);

    toggleRbDisplay( !m_hideRbMeshes );

    m_currentWorld->unlock();
}


void PreviewAssetManager::removePhysicsObjects( PreviewAsset* asset )
{
    hkRootLevelContainer* root = asset->m_container;
    hkpPhysicsData* physicsContainer = root->findObject<hkpPhysicsData>();
    if (physicsContainer && m_currentWorld)
    {
        m_currentWorld->lock();

        // physics
        {
            const hkArray<hkpPhysicsSystem*>& systems = physicsContainer->getPhysicsSystems();
            for (int p=0; p < systems.getSize(); ++p)
            {
                // we need a safe way, as the destruction might already have removed some objects in the physics system
                hkArray<hkpRigidBody*> removedBodies;
                hkArray<hkpConstraintInstance*> removedConstraints;

                hkpPhysicsSystem* system =  systems[p];
                const hkArray<hkpRigidBody*>& bodies = system->getRigidBodies();
                for (int bi = bodies.getSize()-1; bi>=0; bi-- )
                {
                    hkpRigidBody* body = bodies[bi];
                    if ( body->getWorld() == HK_NULL)
                    {
                        body->addReference();
                        removedBodies.pushBack( body );
                        system->removeRigidBody( bi );
                    }
                }
                const hkArray<hkpConstraintInstance*>& constraints = system->getConstraints();
                for (int c=constraints.getSize()-1; c>=0; c--)
                {
                    hkpConstraintInstance* constraint = constraints[c];
                    if (!constraint->getOwner())
                    {
                        constraint->addReference();
                        removedConstraints.pushBack(constraint);
                        system->removeConstraint(c);
                    }
                }
                m_currentWorld->removePhysicsSystem(system);
                for (int n=0; n < removedConstraints.getSize(); n++)
                {
                    system->addConstraint( removedConstraints[n] );
                    removedConstraints[n]->removeReference();
                }
                for (int n=0; n < removedBodies.getSize(); n++)
                {
                    system->addRigidBody( removedBodies[n] );
                    removedBodies[n]->removeReference();
                }
            }
        }

        // Remove collision filter as well if it was created from the packfile
        {
            hkpWorldCinfo* wcinfo = physicsContainer->getWorldCinfo();

            if ( wcinfo && wcinfo->m_collisionFilter )
            {
                // The collision filter uses a pointer that came from the pack-file, we must remove it!
                hkpCollisionFilter* filter = m_currentWorld->m_collisionFilter;
                if ( filter->m_type == hkpCollisionFilter::HK_FILTER_CONSTRAINT )
                {
                    hkpConstraintCollisionFilter* ccf = (hkpConstraintCollisionFilter*)filter;
                    hkpConstraintListener* cl = (hkpConstraintListener*)ccf;
                    if ( m_currentWorld->m_constraintListeners.indexOf(cl) >= 0 )
                    {
                        m_currentWorld->removeConstraintListener(cl);
                    }
                }

                if ( filter != wcinfo->m_collisionFilter )
                {
                    filter->removeReference();
                }
                m_currentWorld->m_collisionFilter = HK_NULL;
            }
        }

        m_currentWorld->unlock();
    }

    if ( m_ghostPhysicsContext )
    {
        m_ghostPhysicsContext->removeWorld(m_currentWorld);
    }
}

void PreviewAssetManager::deleteRbResetUtils()
{
#ifdef HK_ENABLE_DESTRUCTION_2012
    hkReferencedObject::removeReferences(m_rbResetUtils.begin(), m_rbResetUtils.getSize());
    m_rbResetUtils.clear();
#endif
}

void PreviewAssetManager::deletePhysicsWorld( )
{
    if (m_currentWorld)
    {
        m_currentWorld->markForWrite();

        m_currentWorld->removeReference();
        m_currentWorld = HK_NULL;
    }

    if (m_ghostDisplayHandler)
    {
        // clean up an dangling old ghosts (XX clean me up)
        {
            const hkPointerMap<hkUint64, hkgDisplayObject*>& ghosts = m_ghostDisplayHandler->getObjectMap();
            hkPointerMap<hkUint64, hkgDisplayObject*>::Iterator it = ghosts.getIterator();

            for ( ; ghosts.isValid(it); it = ghosts.getNext(it))
            {
                hkgDisplayObject* dobject = ghosts.getValue(it);
                dobject->removeReference();
            }
        }

        m_ghostDisplayHandler->removeReference();
        m_ghostDisplayHandler = HK_NULL;
    }

    for (int pi=0; pi < m_ghostProcesses.getSize(); ++pi)
    {
        delete m_ghostProcesses[pi];
    }
    m_ghostProcesses.setSize(0);

    if (m_ghostPhysicsContext)
    {
        m_ghostPhysicsContext->removeReference();
        m_ghostPhysicsContext = HK_NULL;
    }
}

#pragma managed

/*
 * Havok SDK - Product file, BUILD(#20171210)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2017 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
