// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>

#include <ContentTools/Common/StandAloneFilterManager/standaloneFilterManager.h>
#include <ContentTools/Common/StandAloneFilterManager/standaloneFilterManagerOptions.h>

#include <ContentTools/Common/Filters/Common/hctFilterCommon.h>

#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/hkBaseSystem.h>

#include <Common/Base/Serialize/Resource/hkResource.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Base/Serialize/Format/Compat/hkCompatFormats.h>

#include <ContentTools/Common/SceneExport/Filters/hctFilterProcessingUtil.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>

#include <Common/SceneData/hkSceneData.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Camera/hkxCamera.h>
#include <Common/SceneData/Light/hkxLight.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxIndexBuffer.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Skin/hkxSkinBinding.h>
#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>
#include <Common/SceneData/Selection/hkxNodeSelectionSet.h>
#include <Common/SceneData/Environment/hkxEnvironment.h>

#ifdef HK_FEATURE_PRODUCT_ANIMATION
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Animation/hkaAnimationBinding.h>
#include <Animation/Animation/Animation/hkaAnnotationTrack.h>
#include <Animation/Animation/Animation/hkaAnimation.h>
#include <Animation/Animation/Animation/Interleaved/hkaInterleavedUncompressedAnimation.h>
#include <Animation/Animation/Motion/Default/hkaDefaultAnimatedReferenceFrame.h>
#include <Animation/Animation/Motion/hkaAnimatedReferenceFrame.h>

#include <Common/GeometryUtilities/Skeleton/hkaBone.h>
#include <Common/GeometryUtilities/Skeleton/hkaSkeleton.h>
#include <Animation/Animation/Rig/hkaBoneAttachment.h>
#include <Common/GeometryUtilities/Skeleton/hkaSkeletonUtils.h>
#include <Animation/Animation/Deform/Skinning/hkaMeshBinding.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#ifndef _DLL
#pragma message ("The standalonefiltermanager project should be built in a multithreaded DLL configuration.")
#endif

// This registry key contains the path to the filter manager.
// The most recently used filter set is also stored from here.

#ifdef HK_PLATFORM_WIN64
    static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT("Software\\Havok\\hkFilters_x64");
#else
    static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT("Software\\Havok\\hkFilters");
#endif

// RegisterAllClasses functionality

#ifdef HK_FEATURE_PRODUCT_ANIMATION

inline hkaAnimationContainer* _mergeAnims(hkaAnimationContainer* master, const hkaAnimationContainer* other)
{
    master->m_skeletons.insertAt(master->m_skeletons.getSize(), other->m_skeletons.begin(), other->m_skeletons.getSize());
    master->m_animations.insertAt(master->m_animations.getSize(), other->m_animations.begin(), other->m_animations.getSize());
    master->m_bindings.insertAt(master->m_bindings.getSize(), other->m_bindings.begin(), other->m_bindings.getSize());
    master->m_attachments.insertAt(master->m_attachments.getSize(), other->m_attachments.begin(), other->m_attachments.getSize());
    master->m_skins.insertAt(master->m_skins.getSize(), other->m_skins.begin(), other->m_skins.getSize());
    return master;
}

#endif

void _userMessage(const char* title, const char* message, bool interactive)
{
    if (interactive)
    {
        MessageBox(NULL, message,
            title,
            MB_OK | MB_ICONWARNING | MB_TASKMODAL | MB_TOPMOST);
    }
    else
    {
        puts(title);
        puts(message);
    }
}

hkxScene* _mergeScene( hkxScene* master, hkxScene* other )
{
    if (other->m_asset) master->m_asset = other->m_asset;

    if (other->m_sceneLength > master->m_sceneLength) master->m_sceneLength = other->m_sceneLength;

    if (other->m_rootNode)
    {
        if (!master->m_rootNode)
        {
            master->m_rootNode = other->m_rootNode;
        }
        else // two root nodes.. add children in as root (root never animated etc anyway)
        {
            master->m_rootNode->m_children.append( other->m_rootNode->m_children.begin(), other->m_rootNode->m_children.getSize() );
        }
    }

    if ( other->m_cameras.getSize())
    {
        master->m_cameras.append( other->m_cameras.begin(), other->m_cameras.getSize() );
    }

    if ( other->m_lights.getSize())
    {
        master->m_lights.append( other->m_lights.begin(), other->m_lights.getSize() );
    }

    if ( other->m_meshes.getSize())
    {
        master->m_meshes.append( other->m_meshes.begin(), other->m_meshes.getSize() );
    }

    if ( other->m_materials.getSize())
    {
        master->m_materials.append( other->m_materials.begin(), other->m_materials.getSize() );
    }

    if ( other->m_inplaceTextures.getSize())
    {
        master->m_inplaceTextures.append( other->m_inplaceTextures.begin(), other->m_inplaceTextures.getSize() );
    }

    if ( other->m_externalTextures.getSize())
    {
        master->m_externalTextures.append( other->m_externalTextures.begin(), other->m_externalTextures.getSize() );
    }

    if ( other->m_skinBindings.getSize())
    {
        master->m_skinBindings.append( other->m_skinBindings.begin(), other->m_skinBindings.getSize() );
    }

    return master;
}

void _mergeEnvironment (hkxEnvironment* master,class hkxEnvironment* other )
{
    for (int v=0; v<other->getNumVariables(); v++)
    {
        const char* name = other->getVariableName(v);
        const char* value = other->getVariableValue(v);

        // We don't want to override new values - we only set non-existing variables
        if (master->getVariableValue(name)==HK_NULL)
        {
            master->setVariable(name, value);
        }
    }

}

void _mergeContainers( hkRootLevelContainer* masterContainer, hkRootLevelContainer* otherContainer, hkArray<hkStringPtr>& assetPaths )
{
    hkStringMap<hkBool32> knownContainers;

    const char* newAssetPath = HK_NULL;

    // Check for the scene containers
    {
        hkxScene* masterScene = masterContainer->findObject<hkxScene>();
        hkxScene* otherScene = otherContainer->findObject<hkxScene>();

        if (otherScene)
        {
            newAssetPath = otherScene->m_asset;
        }

        if (masterScene && otherScene)
        {
            do
            {
                _mergeScene(masterScene, otherScene);
            }
            while ((otherScene = otherContainer->findObject<hkxScene>(otherScene))!=HK_NULL);

            // Add hkxScene to the knownContainers
            knownContainers.insert( hkReflect::getName<hkxScene>(), true );
        }

    }

    // Check for environment containers
    {
        hkxEnvironment* masterEnvironment = masterContainer->findObject<hkxEnvironment>();
        hkxEnvironment* otherEnvironment = otherContainer->findObject<hkxEnvironment>();

        if (otherEnvironment)
        {
            newAssetPath = otherEnvironment->getVariableValue("assetPath");
        }

        if (masterEnvironment && otherEnvironment)
        {

            do
            {
                _mergeEnvironment(masterEnvironment, otherEnvironment);
            }
            while ((otherEnvironment = otherContainer->findObject<hkxEnvironment>(otherEnvironment))!=HK_NULL);

            // Add hkxEnvironment to the knownContainers
            knownContainers.insert( hkReflect::getName<hkxEnvironment>(), true );
        }

    }

    if (newAssetPath!=HK_NULL)
    {
        assetPaths.pushBack(newAssetPath);
    }

#ifdef HK_FEATURE_PRODUCT_ANIMATION

    // Check for the anim containers
    {
        hkaAnimationContainer* masterAnim = masterContainer->findObject<hkaAnimationContainer>();

        hkaAnimationContainer* otherAnim = HK_NULL;
        while ((otherAnim = otherContainer->findObject<hkaAnimationContainer>(otherAnim))!=HK_NULL)
        {
            if (masterAnim == HK_NULL)
            {
                // Add a new animation container
                masterAnim = new hkaAnimationContainer();

                masterContainer->m_namedVariants.expandOne().set("Merged Animation Container", masterAnim);

                masterAnim->removeReference();
            }

            _mergeAnims(masterAnim, otherAnim);
        }
    }

    // Add hkxAnimationContainer to the knownContainers
    knownContainers.insert( hkReflect::getName<hkaAnimationContainer>(), true );

#endif

    // Iterate through the rest and append.
    {
        for (int v=0; v < otherContainer->m_namedVariants.getSize(); ++v)
        {
            const char* nvName = otherContainer->m_namedVariants[v].getTypeName();

            // Check knownContainers
            if( !knownContainers.getWithDefault(nvName, false) )
            {
                // Add a new container
                masterContainer->m_namedVariants.expandOne() = otherContainer->m_namedVariants[v];
            }
        }
    }
}

static hkRootLevelContainer* _loadSceneFile(
    const char* filename, hkArray<hkResource*>& resources, hkSerializeUtil::ErrorDetails* errorDetails)
{
    // Open the stream
    hkIfstream fileIn(filename);

    if (!fileIn.isOk())
    {
        HK_WARN_ALWAYS(0xabba6ab5, "Unable to load scene file " << filename << ".");
        return HK_NULL;
    }
    // warn user if failed
    hkResource* resource = hkSerializeUtil::load(fileIn.getStreamReader(), errorDetails);
    if( !resource )
    {
        HK_WARN_ALWAYS(0xabbacd8a, errorDetails->defaultMessage.cString() << "\nUnable to load scene file " << filename << ".");
        return HK_NULL;
    }
    resources.pushBack(resource);

    return resource->getContents<hkRootLevelContainer>();
}

static void _loadOptionsFromRegistry( hctFilterProcessingUtil* sceneExport )
{
    HKEY animReg;
    DWORD dispos;

    RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

    // Read the buffer size first
    DWORD bufSize = 128*1024;
    RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, NULL, &bufSize );
    hkArray<char> optionsBuf;
    optionsBuf.setSize(bufSize);

    if ( RegQueryValueEx( animReg, TEXT("FilterSetup"), NULL, NULL, (LPBYTE)optionsBuf.begin(), &bufSize ) == ERROR_SUCCESS )
    {
        sceneExport->setOptions( optionsBuf.begin(), bufSize );
    }
    RegCloseKey( animReg );
}

static void _saveOptionsToRegistryAndFile( hctFilterProcessingUtil* sceneExport, const char* optionsFile )
{
    int bufSize = sceneExport->getOptionsSize();

    if (bufSize > 0)
    {
        char* buf = hkAllocateChunk<char>( bufSize, HK_MEMORY_CLASS_EXPORT);
        sceneExport->getOptions(buf);
        // Save to registry.
        {
            HKEY animReg;
            DWORD dispos;

            if ( RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL,
                REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos) == ERROR_SUCCESS )
            {
                RegSetValueEx(animReg, TEXT("FilterSetup"), 0, REG_BINARY, LPBYTE(buf), bufSize);
            }
            RegCloseKey(animReg);
        }

        // Save to file.
        if (optionsFile != HK_NULL && optionsFile[0] != '\0')
        {
            hkRefPtr<hkStreamWriter> optionsWriter = hkFileSystem::getInstance().openWriter(optionsFile);
            if (optionsWriter != HK_NULL)
            {
                optionsWriter->write(buf, bufSize);
            }
        }
        hkDeallocateChunk<char>( buf, bufSize, HK_MEMORY_CLASS_EXPORT);
    }
}

    // Sets environment variables specific to the processing
static hkxEnvironment* _getOrAddEnvironment (hkRootLevelContainer* rootLevelcontainer)
{
    // Look for the environment
    hkxEnvironment* environment = HK_NULL;
    hkxScene* scene = HK_NULL;
    {
        for (int i=0; i<rootLevelcontainer->m_namedVariants.getSize(); i++)
        {
            if (hkxEnvironment* e = rootLevelcontainer->m_namedVariants[i].getByTypeName<hkxEnvironment>())
            {
                environment = e;
            }
            if (hkxScene* s = rootLevelcontainer->m_namedVariants[i].getByTypeName<hkxScene>())
            {
                scene = s;
            }
        }
    }

    if (!environment)
    {
        // Add one
        environment = new hkxEnvironment();

        // Put the new one first
        rootLevelcontainer->m_namedVariants.expandOne().set("Environment Data", environment);
        environment->removeReference();

        // If there was a scene, fill it with that
        if (scene)
        {
            hkxSceneUtils::fillEnvironmentFromScene(*scene, *environment);
        }
    }

    return environment;

}

int main()
{
    // Grab the current dir
    char currentDir[1024];
    GetCurrentDirectory(1024, currentDir);

    OleInitialize(NULL);

#ifdef HK_DEBUG_SLOW
    hkSceneExportMemory::baseSystemInit(true);
#else
    hkSceneExportMemory::baseSystemInit(false);
#endif

    int status = -1;
    int numFilesLoaded = 0;
    {
        // Gets the filter manager path.
        // The DLL should be in the same directory as the exe.
        hkStringOld filterManagerPath = currentDir;
        hkBool filterManagerFound = false;
        {
            // If it's not there, check the regsitry, then look for the HAVOK_TOOLS_ROOT environment variable.
            //
            // NOTE:
            // The HAVOK_TOOLS_ROOT variable is used for development and debugging only.
            // It isn't needed to run the previewer, won't be created by installing the Animation
            // Tools package, and deliberately isn't mentioned in the Havok documentation.
            //
            // Only create the HAVOK_TOOLS_ROOT variable if you wish to run/debug the
            // previewer using a compiler such as Visual Studio.
            //
            // Set it to the directory containing the hkfiltermanager.dll file.
            hkStringOld dllPath;
            {
                // Look for the HAVOK_TOOLS_ROOT environment variable.
                filterManagerPath = ::getenv(ENVVAR_FILTER_ROOT);
                dllPath = filterManagerPath + "\\hctFilterManager.dll";
                hkIfstream edllFile(dllPath.cString());
                filterManagerFound = edllFile.isOk();
            }

            if (!filterManagerFound)
            {
                HKEY animReg;
                DWORD dispos;

                RegCreateKeyEx(
                    HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

                LPSTR filtersPath[1024];
                DWORD filtersPathSize = 1024;
                if (RegQueryValueEx(animReg, TEXT("FilterPath"), NULL, NULL, (LPBYTE)filtersPath, &filtersPathSize) == ERROR_SUCCESS)
                {
                    filterManagerPath = (const char*)filtersPath;
                    dllPath = filterManagerPath + "\\hctFilterManager.dll";
                    hkIfstream rdllFile(dllPath.cString());
                    filterManagerFound = rdllFile.isOk();
                }

                RegCloseKey(animReg);
            }
        }

        // Load hkCompatFormats from the filter manager path.
        if (!hkCompatFormats::getInstance().isLoaded() && filterManagerFound)
        {
            hkStringOld compatPath = filterManagerPath + "\\utils";
            hkCompatFormats::replaceInstance(new hkCompatFormats(compatPath.cString()));
        }

        hctStandaloneFilterManagerOptions options;
        hkArray<hkStringOld> buffer;

        if (!hkGetOptions(options, buffer))
        {
            return 0;
        }

        hkStringOld optionsFile(options.m_optionsPath);
        hkStringOld optionsAssetPath(options.m_assetPath);
        hkStringOld optionsOutput(options.m_outputPath);

        // Interactive mode
        bool interactive = options.m_forceInteractive;

        if (interactive)
        {
            FreeConsole();
            hkShowSplash(GetModuleHandle(NULL));
        }

        hkRootLevelContainer* master = new hkRootLevelContainer();

        // Read multiple files into these buffers
        if (options.m_arguments.getSize() == 0)
        {
            hkGetFiles( options.m_arguments );
        }

        if (options.m_arguments.getSize() == 0)
        {
            _userMessage("Warning: Load failed", "No files have been opened. The standalone filter manager will now exit.", interactive);
        }

        hkArray<hkResource*> resources;
        hkArray<hkStringPtr> assetPaths;
        hkArray<hkStringPtr> sceneFilePaths;
        for (int f=0; f < options.m_arguments.getSize(); f++)
        {
            // Attempt to load file
            hkSerializeUtil::ErrorDetails errorDetails;
            hkRootLevelContainer* loadedData = _loadSceneFile( options.m_arguments[f].cString(), resources, &errorDetails );

            if (loadedData)
            {
                _mergeContainers( master, loadedData, assetPaths ); // so that we can expose all the data to the filters as one scene
                numFilesLoaded++;

                sceneFilePaths.pushBack(options.m_arguments[f].cString());
            }
            else
            {
                hkStringOld message = "Unable to load scene file ";
                message += options.m_arguments[f];
                message += ": ";
                message += errorDetails.defaultMessage.cString() ? errorDetails.defaultMessage.cString() : "unknown reason" ;
                message += ".";

                _userMessage("Warning: Load failed", message.cString(), interactive);
            }
        }

        // Attempt to fixup RLC classes with no metadata
        for (int v=0 ; v < master->m_namedVariants.getSize(); v++)
        {
            hkRootLevelContainer::NamedVariant& nVar  = master->m_namedVariants[v];
            // Class name but no meta data - potentially fixup class
            const char* typeName = nVar.getTypeName();
            if (typeName && (!nVar.getType()) )
            {
                // fix it up
                nVar.set(nVar.getTypeName(), nVar.getObject(),hkReflect::typeFromName( typeName ) );
            }
        }

        if (numFilesLoaded)
        {
            hkxEnvironment* environment = _getOrAddEnvironment(master);

            if ((optionsAssetPath.getLength() == 0) && interactive)
            {
                hkGetAssetPath(assetPaths, sceneFilePaths, optionsAssetPath);
            }

            // Set asset path, asset and asset folder
            if (optionsAssetPath.getLength() > 0)
            {
                hkxScene tempScene;
                tempScene.m_asset = optionsAssetPath.cString();
                hkxSceneUtils::fillEnvironmentFromScene(tempScene, *environment);
            }

            if (optionsOutput.getLength()>0)
            {
                environment->setVariable("output", optionsOutput.cString());
            }

            // Create a Scene Export manager
            hctFilterProcessingUtil* sceneExport =
                new hctFilterProcessingUtil(HK_NULL, interactive, options.m_logFile, options.m_verboseLog, options.m_verbose);

            sceneExport->setOwnerHandle( HK_NULL );

            // Load the filter manager.
            if( filterManagerFound )
            {
                if (sceneExport->load(filterManagerPath.cString()))
                {
                    {
                        // Load the default configuration from the passed in .hko file.
                        hkArray<char>::Temp optionsBuf;
                        if( hkLoadUtil(optionsFile.cString()).toArray(optionsBuf) )
                        {
                            sceneExport->setOptions( optionsBuf.begin(), optionsBuf.getSize() );
                        }
                        else
                        {
                            // If the file can't be found check the registry for previously used options.
                            _loadOptionsFromRegistry( sceneExport );
                        }
                    }

                    if (interactive)
                    {
                        hkBool saveOptions = true;

                        // Run the filter manager interactively
                        sceneExport->openFilterManager(*master, saveOptions);
                        if (saveOptions)
                        {
                            // Save option data to the registry and to the passed in .hko file if it is valid.
                            _saveOptionsToRegistryAndFile(sceneExport, optionsFile.cString());
                        }
                    }
                    else
                    {
                        hkBool optionsModifiedUnused;
                        sceneExport->processBatch( *master, optionsModifiedUnused );
                    }
                    status = sceneExport->getResult();
                }
                else
                {
                    _userMessage("Error: Load failed", "Could not load the filter manager dll and associated filters.", interactive);
                }
            }
            else
            {
                _userMessage("Error: Load failed", "Could not find hctFilterManager.dll.", interactive);
            }

            delete master;
            master = HK_NULL;

            // free loaded resources
            for( int i = 0; i < resources.getSize(); ++i )
            {
                resources[i]->removeReference();
            }

            sceneExport->unload();
            delete sceneExport;
        }

        if (master)
        {
            delete master;
        }
    }

    hkSceneExportMemory::baseSystemQuit();

    OleUninitialize();

    if (numFilesLoaded == 0)
    {
        status = -2;
    }

    return status;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
