// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/Filters/Common/version.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Container/String/Deprecated/hkStringOld.h>
#include <Common/Base/Container/String/hkStringBuf.h>

#include <Common/Base/Fwd/hkcstdlib.h>

#ifndef HK_DYNAMIC_DLL
#include <Common/Base/keycode.cxx>
#endif

#include <Wincrypt.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <ContentTools/Common/Filters/Common/hctBaseDll.h>

//
//  Initialize hctSdkUtils statics

hctModelerNodeIdToClassMap g_modelerNodeIdToClassBuffer[HCT_MAX_NUM_CLASS_ENTRIES];
hkArray<hctModelerNodeIdToClassMap>*    hctSdkUtils::m_modelerNodeIdToClassMap  = HK_NULL;
HMODULE                                 hctSdkUtils::m_destructionUtilsDll      = HK_NULL;
hctBaseDll*                             hctSdkUtils::m_baseDestructionUtilsDll = HK_NULL;
hctBaseDll*                             hctSdkUtils::m_baseDestructionClassesDll = HK_NULL;

// This registry key contains the path to the filter manager.
// The most recently used filter set is also stored here.

#ifdef HK_PLATFORM_WIN64
static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT( "Software\\Havok\\hkFilters_x64" );
#else
static const TCHAR* _HAVOK_FILTERS_REG_KEY = TEXT( "Software\\Havok\\hkFilters" );
#endif

static bool getValidClassPath(hkStringOld& pathOut)
{
    hkStringOld path;
    bool found = false;

    // If the filter manager still hasn't been found, see if the environment variable is set up
    const char* cpath = ::getenv( ENVVAR_FILTER_ROOT );
    if (cpath)
    {
        path = hkStringOld(cpath) + hkStringOld("\\classes\\");
        hkFileSystem::DirectoryListing listing;
        hkResult result = hkFileSystem::getInstance().listDirectory( path.cString(), listing );
        found = (result.isSuccess());
    }

    // Look for a path in the registry
    if( !found )
    {
        HKEY animReg;
        DWORD dispos;
        RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

        BYTE filtersPath[1024];
        DWORD filtersPathSize = 1024;

        if( RegQueryValueEx( animReg, TEXT("FilterPath"), NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
        {
            path = (const char*)filtersPath;
            path += "\\classes\\";

            hkFileSystem::DirectoryListing listing;
            hkResult result = hkFileSystem::getInstance().listDirectory( path.cString(), listing );

            found = (result.isSuccess());
        }

        RegCloseKey(animReg);
    }

    if (found)
    {
        pathOut = path;
    }

    return found;
}

/* static */ HMODULE hctSdkUtils::loadClassDll(const char* dllName)
{
    hkStringOld dllPath;
    bool found = getValidClassPath(dllPath);

    HMODULE dllHandle = 0;
    if (found)
    {
        dllPath += dllName;

        dllHandle = LoadLibrary( dllPath.cString() );
    }

    return dllHandle;
}

static bool getValidUtilsPath(hkStringOld& pathOut)
{
    hkStringOld path;
    bool found = false;

    // If the filter manager still hasn't been found, see if the environment variable is set up
    const char* cpath = ::getenv( ENVVAR_FILTER_ROOT );
    if (cpath)
    {
        path = hkStringOld(cpath) + hkStringOld("\\utils\\");
        hkFileSystem::DirectoryListing listing;
        hkResult result = hkFileSystem::getInstance().listDirectory( path.cString(), listing );
        found = (result.isSuccess());
    }

    // Look for a path in the registry
    if( !found )
    {
        HKEY animReg;
        DWORD dispos;
        RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

        BYTE filtersPath[1024];
        DWORD filtersPathSize = 1024;

        if( RegQueryValueEx( animReg, TEXT("FilterPath"), NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
        {
            path = (const char*)filtersPath;
            path += "\\utils\\";

            hkFileSystem::DirectoryListing listing;
            hkResult result = hkFileSystem::getInstance().listDirectory( path.cString(), listing );

            found = (result.isSuccess());
        }

        RegCloseKey(animReg);
    }

    if (found)
    {
        pathOut = path;
    }

    return found;
}

/* static */ HMODULE hctSdkUtils::loadUtilDll(const char* dllName)
{
    hkStringOld dllPath;
    bool found = getValidUtilsPath(dllPath);

    HMODULE dllHandle = 0;
    if (found)
    {
        dllPath += dllName;

        dllHandle = LoadLibrary( dllPath.cString() );
    }

    return dllHandle;
}

/* static */ void hctSdkUtils::loadAllClasses()
{
    

    if (m_baseDestructionClassesDll)
    {
        return;
    }

    // Allocate enough room
    if ( !m_modelerNodeIdToClassMap )
    {
        m_modelerNodeIdToClassMap = new hkArray<hctModelerNodeIdToClassMap>();
    }
    m_modelerNodeIdToClassMap->setDataUserFree( g_modelerNodeIdToClassBuffer, m_modelerNodeIdToClassMap->getSize(), HCT_MAX_NUM_CLASS_ENTRIES );

    HMODULE dllHandle = hctSdkUtils::loadClassDll( "hctDestructionClasses.dll" );

    // If we've found the DLL we can use it.
    if ( dllHandle )
    {
        
        typedef hctBaseDll* (*GetBaseDllFunc)(HMODULE);
        GetBaseDllFunc getBaseDllFunc = (GetBaseDllFunc)GetProcAddress( dllHandle, "getBaseDll");
        m_baseDestructionClassesDll = getBaseDllFunc(dllHandle);
        hkMemoryInitUtil::SyncInfo syncInfo;
        hkMemoryInitUtil::SyncInfo::getLocalInfo(syncInfo);
        m_baseDestructionClassesDll->initDll(syncInfo, hkError::getInstancePtr());

        hkGetClassEntriesFunc getClassEntriesFunc = (hkGetClassEntriesFunc) GetProcAddress( dllHandle, "getClassEntries");

        // retrieve class list
        getClassEntriesFunc(*m_modelerNodeIdToClassMap);
    }
}

/* static */ void hctSdkUtils::unloadAllClasses()
{
    if (m_modelerNodeIdToClassMap)
    {
        delete m_modelerNodeIdToClassMap;
        m_modelerNodeIdToClassMap = HK_NULL;
    }

    if (m_baseDestructionClassesDll)
    {
        m_baseDestructionClassesDll->quitDll();
        m_baseDestructionClassesDll = HK_NULL;
    }
}

//
//  Loads all available utility Dlls

void hctSdkUtils::loadAllUtils()
{
    // Unload any existing utils
    unloadAllUtils();

    // Load all utils
    loadDestructionUtils();
}

//
//  Unloads all loaded utility Dlls

void hctSdkUtils::unloadAllUtils()
{
    hctSdkUtils::unloadDestructionUtils();
}

//
//  Attempts to load the hctDestructionUtils.dll

void hctSdkUtils::loadDestructionUtils()
{
    m_destructionUtilsDll = loadUtilDll("hctDestructionUtils.dll");
    if ( !m_destructionUtilsDll )
    {
        return; // Destruction Util DLL found but could not load. The DLL is corrupt.
    }
}

//
//  Attempts to unload the hctDestructionUtils.dll

void hctSdkUtils::unloadDestructionUtils()
{
    if ( m_destructionUtilsDll )
    {
        FreeLibrary(m_destructionUtilsDll);
        m_destructionUtilsDll = HK_NULL;
    }
}

//
//  Returns the UI scheme

hctSdkUtils::UiScheme hctSdkUtils::getUiScheme()
{
    UiScheme uiScheme(UI_SCHEME_NONE);

    // Check if we've already loaded Destruction
    bool doUnloadDestruction = false;
    if ( !m_destructionUtilsDll )
    {
        // We haven't, try to do that now
        loadDestructionUtils();

        if ( m_destructionUtilsDll)
        {
            doUnloadDestruction = true;
        }
    }

    // Check if we have Destruction available
    if ( m_destructionUtilsDll )
    {
        typedef bool (HK_CALL *FunPtr)(void);
        FunPtr fun = reinterpret_cast<FunPtr>(getUtilityFunction("getUseDestructionUi"));

        if ( fun )
        {
            const bool useNdUi = fun();
            if ( useNdUi )
            {
                uiScheme.orWith(UI_SCHEME_DESTRUCTION);
            }
            else
            {
                uiScheme.orWith(UI_SCHEME_DESTRUCTION_2012);
            }
        }
    }

    // Unload Destruction if required
    if ( doUnloadDestruction )
    {
        unloadDestructionUtils();
    }

    return uiScheme;
}

//
//  Returns the UI attributes hide criteria

hkAttributeHideCriteria::Types HK_CALL hctSdkUtils::getUiAttributesHideCriteria(hkAttributeHideCriteria::Types modelerId)
{
    hkUint32 hideCriteria = modelerId;
    {
        const hctSdkUtils::UiScheme uiScheme = hctSdkUtils::getUiScheme();

        // Hide all members mark as hidden for this UI scheme
        if      ( uiScheme.anyIsSet(hctSdkUtils::UI_SCHEME_DESTRUCTION) )       {   hideCriteria |= hkAttributeHideCriteria::UI_SCHEME_IS_DESTRUCTION;      }
        else if ( uiScheme.anyIsSet(hctSdkUtils::UI_SCHEME_DESTRUCTION_2012) )  {   hideCriteria |= hkAttributeHideCriteria::UI_SCHEME_IS_DESTRUCTION_2012; }
    }
    return hkAttributeHideCriteria::Types(hideCriteria);
}

//
//  Attempts to retrieve the given utility function from one of the loaded utility Dlls

void* HK_CALL hctSdkUtils::getUtilityFunction(const char* functionName)
{
    // Try the Destruction Dll
    if ( m_destructionUtilsDll )
    {
        return GetProcAddress(m_destructionUtilsDll, functionName);
    }

    return HK_NULL;
}

/* static */ int hctSdkUtils::getMatchingClasses(const char* targetNodeName, const char* targetAttributeName, hkArray<hkStringOld>& matchingClassesOut)
{
    // First find the target class in the definition.
    const hkReflect::Type* targetClass = HK_NULL;
    {
        for (int i = 0; m_modelerNodeIdToClassMap && (i < m_modelerNodeIdToClassMap->getSize()) && ((*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL); i++)
        {
            const char* testClassName = (*m_modelerNodeIdToClassMap)[i].m_nodeName;
            if (hkString::strCasecmp(targetNodeName, testClassName) == 0)
            {
                const hkReflect::RecordType* klass = (*m_modelerNodeIdToClassMap)[i].m_class->asRecord();
                HK_ASSERT_NO_MSG(0x18605bdf, klass);
                hkReflect::FieldDecl member = klass->findField(targetAttributeName, true);
                if (member && member.getType()->asPointer() && member.getType()->asPointer()->getSubType()->asRecord())
                {
                    targetClass = member.getType()->asPointer()->getSubType();
                    break;
                }
            }
        }
    }

    // If the target class is not registered, we have no results to return
    if (targetClass == 0)
    {
        return 0;
    }

    // Iterate through all the available classes and examine if they inherit from the target one.
    for (int i = 0; m_modelerNodeIdToClassMap && (i < m_modelerNodeIdToClassMap->getSize()) && (*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL; i++)
    {
        const hkReflect::Type* klass = (*m_modelerNodeIdToClassMap)[i].m_class;

        // We don't want to consider pure virtual classes as they won't be defined in the modeler
        if ( klass->extendsOrEquals(targetClass) )
        {
            const char* matchingNodeName = (*m_modelerNodeIdToClassMap)[i].m_nodeName;
            matchingClassesOut.pushBack(matchingNodeName);
        }
    }
    return matchingClassesOut.getSize();
}

/*static*/ const hkReflect::Type* hctSdkUtils::getTypeByName(const char* className)
{
    for (int i = 0; m_modelerNodeIdToClassMap && (i < m_modelerNodeIdToClassMap->getSize()) && ((*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL); i++)
    {
        if ( hkString::strCmp((*m_modelerNodeIdToClassMap)[i].m_class->getName(), className) == 0 ||
             hkString::strCmp((*m_modelerNodeIdToClassMap)[i].m_nodeName, className) == 0 )
        {
            return (*m_modelerNodeIdToClassMap)[i].m_class;
        }
    }

    return HK_NULL;
}


/*static*/ void hctSdkUtils::getClassesByBaseName(const char* baseName, hkArray<hctModelerNodeIdToClassMap>& classesOut)
{
    // For each entry in the class map: traverse the class's derivation path upwards until we reach the root base class
    // or we find the supplied base class amongst the parent classes.
    for (int i = 0; m_modelerNodeIdToClassMap && (i < m_modelerNodeIdToClassMap->getSize()) && ((*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL); i++)
    {
        const hkReflect::Type* klass = (*m_modelerNodeIdToClassMap)[i].m_class;
        while ( klass )
        {
            if ( hkString::strCmp(klass->getName(), baseName) == 0 )
            {
                classesOut.pushBack((*m_modelerNodeIdToClassMap)[i]);
                break;
            }
            if( const hkReflect::RecordType* r = klass->asRecord() )
            {
                klass = r->getParent();
            }
            else
            {
                break;
            }
        }
    }
}


/*static*/ int hctSdkUtils::getIndexByNodeName(const char* className)
{
    for (int i = 0; m_modelerNodeIdToClassMap && (i < m_modelerNodeIdToClassMap->getSize()) && ((*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL); i++)
    {
        if ( hkString::strCmp(className, (*m_modelerNodeIdToClassMap)[i].m_nodeName ) == 0)
        {
            return i;
        }
    }
    return -1;
}


/*static */int hctSdkUtils::getNumUiClasses()
{
    int numUiClasses = 0;

    {
        for (int i = 0; m_modelerNodeIdToClassMap && i < m_modelerNodeIdToClassMap->getSize(); i++)
        {
            hctModelerNodeIdToClassMap& mapEntry = (*m_modelerNodeIdToClassMap)[i];
            if ( mapEntry.m_class != HK_NULL )
            {
                numUiClasses++;
            }
        }
    }

    return numUiClasses;
}


/*static */hctModelerNodeIdToClassMap& hctSdkUtils::getUiClass(int maxClassId)
{
    HK_ASSERT(0xaf31ef12, m_modelerNodeIdToClassMap, "No ui classes loaded!");

    {
        for (int i = 0; i < m_modelerNodeIdToClassMap->getSize() && (*m_modelerNodeIdToClassMap)[i].m_class != HK_NULL; i++)
        {
            hctModelerNodeIdToClassMap& mapEntry = (*m_modelerNodeIdToClassMap)[i];
            if ( mapEntry.m_3dsMaxClassId == maxClassId )
            {
                return mapEntry;
            }
        }
    }

    HK_ASSERT(0xaf31ef13, false, "Supplied index is out-of-scope for the registered table.");

    return (*m_modelerNodeIdToClassMap)[0];
}


// VERY IMPORTANT NOTE: changing this uniquifier string will break all existing assets! Do not touch!
#define HCT_GET_VALID_NAME_UNIQUIFIER "HkUnique"

/*static*/ const char* hctSdkUtils::getValidName(const char* name, hkStringOld& buffer)
{
    bool modifyFlag = false;

    // 3DSMax has a Fracture helper that might cause a conflict.
    if ( hkString::strCmp(name, "fracture") == 0 )
    {
        modifyFlag = true;
    }
    // 3DSMax has a groupname internal type that might cause a conflict.
    else if ( hkString::strCmp(name, "groupName") == 0 )
    {
        modifyFlag = true;
    }
    // 3DSMax has a controller internal type that might cause a conflict.
    else if ( hkString::strCmp(name, "controller") == 0 )
    {
        modifyFlag = true;
    }

    if ( !modifyFlag )
    {
        return name;
    }

    buffer = name;
    buffer += HCT_GET_VALID_NAME_UNIQUIFIER;
    return buffer.cString();
}


/*static*/ const char* hctSdkUtils::revertValidName(const char* name, hkStringOld& buffer)
{
    int uniquifierLength = int(strlen(HCT_GET_VALID_NAME_UNIQUIFIER));
    int inputNameLength  = int(strlen(name));

    if ( inputNameLength <= uniquifierLength )
    {
        // Original (unmodified) name would have been empty (or strlen would have been negative): abort...
        return name;
    }

    if ( hkString::strCmp(&name[inputNameLength-uniquifierLength], HCT_GET_VALID_NAME_UNIQUIFIER) != 0 )
    {
        // Passed-in name doesn't end with uniquifier: abort...
        return name;
    }

    buffer = name;
    buffer.replaceInplace(HCT_GET_VALID_NAME_UNIQUIFIER, "");
    return buffer.cString();
}

// Get the filterManagerPath
/*static */ void hctSdkUtils::getFilterManagerPath(hkStringOld& filterManagerPath)
{
    HKEY animReg;
    DWORD dispos;

    RegCreateKeyEx( HKEY_CURRENT_USER, _HAVOK_FILTERS_REG_KEY, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &animReg, &dispos);

    BYTE filtersPath[1024];
    DWORD filtersPathSize = 1024;

    if ( RegQueryValueEx( animReg, TEXT("FilterPath"), NULL, NULL, filtersPath, &filtersPathSize ) == ERROR_SUCCESS )
    {
        filterManagerPath = (const char*)filtersPath;
    }
    else
    {
        filterManagerPath = ::getenv( ENVVAR_FILTER_ROOT );
    }

    RegCloseKey(animReg);
}

/*static*/ void hctSdkUtils::getHelpFileName(hkStringOld& helpFile)
{
    hkStringBuf version(HAVOK_SDK_VERSION_NUM_STRING);                         // 2010.1.0-r1
    version.chompEnd( version.getLength() - version.indexOf('-') );            // 2010.1.0
    version.replace('.', '-', hkStringBuf::REPLACE_ALL);                       // 2010-1-0

    helpFile.printf("\\Havok_%s_Content_Tools_Manual.chm", version.cString()); // Havok_2010-1-0_Content_Tools_Manual.chm
}

/*static*/ void hctSdkUtils::getPathToAndNameOfDocumentationChmFile(hkStringOld& docsPathAndFileName)
{
    hkStringOld filterManagerPath;
    hctSdkUtils::getFilterManagerPath(filterManagerPath);
    hkStringOld helpFileName;
    hctSdkUtils::getHelpFileName(helpFileName);

    docsPathAndFileName += filterManagerPath;
    docsPathAndFileName += helpFileName;
}

/*static*/ bool hctSdkUtils::openVideoBrowser(const char* commandLineArgs)
{
    hkStringOld filterManagerPath;
    hctSdkUtils::getFilterManagerPath(filterManagerPath);

    hkStringOld videoBrowserExe = hkStringOld(filterManagerPath) + "\\HavokVideoBrowser.exe";

    hkIfstream testStream( videoBrowserExe.cString() );
    if (!testStream.isOk())
    {
        return false;
    }

    PROCESS_INFORMATION ProcessInfo;
    STARTUPINFOW StartupInfo = {0};
    StartupInfo.cb = sizeof(STARTUPINFO);

    wchar_t cmdLine[4096];

    hkStringOld videoBrowserCmd = videoBrowserExe + " " + hkStringOld(commandLineArgs);

    int len = videoBrowserCmd.getLength() < 4095 ? videoBrowserCmd.getLength() + 1 : 4095;

    mbstowcs( cmdLine, videoBrowserCmd.cString(), len );
    cmdLine[4095] = TEXT('0');

    CreateProcessW(NULL, cmdLine, NULL, NULL, FALSE, 0, NULL, NULL, &StartupInfo, &ProcessInfo);

    return true;
}

/*static*/ bool hctSdkUtils::openDocumentationChmFile()
{
    hkStringOld chmFilePath;
    hctSdkUtils::getPathToAndNameOfDocumentationChmFile(chmFilePath);

    hkIfstream testStream( chmFilePath.cString() );
    if (!testStream.isOk())
    {
        return false;
    }

    PROCESS_INFORMATION ProcessInfo;
    STARTUPINFOW StartupInfo = {0};
    StartupInfo.cb = sizeof(STARTUPINFO);

    wchar_t cmdLine[4096];

    hkStringOld openCHMFileCmd = hkStringOld("hh.exe ") + chmFilePath;
    int len = openCHMFileCmd.getLength() < 4095 ? openCHMFileCmd.getLength() + 1 : 4095;
    mbstowcs( cmdLine, openCHMFileCmd.cString(), len );
    cmdLine[4095] = TEXT('0');

    CreateProcessW(NULL, cmdLine, NULL, NULL, FALSE, 0, NULL, NULL, &StartupInfo, &ProcessInfo);

    return true;
}

//
//  Returns the MD5 hash of the given data buffer

hkResult HK_CALL hctSdkUtils::computeMd5Hash(const void* bufferIn, hkUint32 bufferSize, hkUint8* hashOut, int& hashSizeOut)
{
    // Initialize return value
    hkResult ret = HK_FAILURE;

    // Get cryptographic context
    HCRYPTPROV hProvider = 0;
    if ( CryptAcquireContext(&hProvider, HK_NULL, HK_NULL, PROV_RSA_FULL, CRYPT_VERIFYCONTEXT | CRYPT_SILENT) )
    {
        // Create MD5 hash
        HCRYPTHASH hHash = 0;
        if ( CryptCreateHash(hProvider, CALG_MD5, 0, 0, &hHash) )
        {
            // Hash the contents of the node
            if ( CryptHashData(hHash, reinterpret_cast<const BYTE*>(bufferIn), bufferSize, 0) )
            {
                DWORD hashLen = hashSizeOut;
                if ( CryptGetHashParam(hHash, HP_HASHVAL, hashOut, &hashLen, 0) )
                {
                    // Success!
                    hashSizeOut = hashLen;
                    ret = HK_SUCCESS;
                }
                else
                {
                    HK_WARN_ALWAYS(0xabbacddc, "Failed to read MD5 hash value");
                }
            }
            else
            {
                HK_WARN_ALWAYS(0xabbacddc, "Failed to hash data");
            }

            // Discard hash
            CryptDestroyHash(hHash);
        }
        else
        {
            HK_WARN_ALWAYS(0xabbacddc, "Failed to create MD5 hash");
        }

        // Discard context
        CryptReleaseContext(hProvider, 0);
    }
    else
    {
        HK_WARN_ALWAYS(0xabbacddc, "Failed to acquire cryptographic context");
    }

    return ret;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
