// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <ContentTools/Common/SdkUtils/Cloth/BonesFromMesh/hctBonesFromMeshUtil.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

static void HK_INLINE _swapColumns (hkMatrix3& rotation, int colA, int colB)
{
    hkVector4 temp; temp = rotation.getColumn(colA);
    rotation.getColumn(colA) = rotation.getColumn(colB);
    rotation.getColumn(colB) = temp;
}

/*static*/ void hctBonesFromMeshUtil::_calculateRotation(hkVector4 &boneNormalAxis, hkVector4 &boneTangentAxis, hkVector4& boneOtherAxis, BoneAxis normalAxis, BoneAxis tangentAxis, hkRotation& rotationOut)
{
    if(normalAxis == B_X_AXIS)
    {
        if(tangentAxis == B_Y_AXIS)
        {
            rotationOut.setCols(boneNormalAxis, boneTangentAxis, boneOtherAxis);
        }
        else
        {
            rotationOut.setCols(boneNormalAxis, boneOtherAxis, boneTangentAxis);
        }

        if ( rotationOut.getDeterminant().isLessZero() )
        {
            _swapColumns (rotationOut, 1,2);
        }
    }
    else if(normalAxis == B_Y_AXIS)
    {
        if(tangentAxis == B_X_AXIS)
        {
            rotationOut.setCols(boneTangentAxis, boneNormalAxis, boneOtherAxis);
        }
        else
        {
            rotationOut.setCols(boneOtherAxis, boneNormalAxis, boneTangentAxis);
        }

        if (rotationOut.getDeterminant().isLessZero())
        {
            _swapColumns (rotationOut, 0,2);
        }
    }
    else if(normalAxis == B_Z_AXIS)
    {
        if(tangentAxis == B_X_AXIS)
        {
            rotationOut.setCols(boneTangentAxis, boneOtherAxis, boneNormalAxis);
        }
        else
        {
            rotationOut.setCols(boneOtherAxis, boneTangentAxis, boneNormalAxis);
        }

        if ( rotationOut.getDeterminant().isLessZero())
        {
            _swapColumns (rotationOut, 0,1);
        }
    }

    if(!rotationOut.isOrthonormal())
    {
        HK_WARN_ALWAYS( 0x7f425fa1, "Bone rotation is not orthonormal." );
    }

}
/*static*/ bool hctBonesFromMeshUtil::createBones (hkGeometry& simulationMesh, hkArray<hkTransform>& boneTransformsOut, BoneAxis normalAxis, BoneAxis tangentAxis, GlobalTangentAxis globalTangentAxis, hkArray<hkBool>& boneAxisChanged, BoneCreationMethod method)
{
    // tangent and normal axis can't be the same
    if( normalAxis == tangentAxis )
    {
        HK_WARN_ALWAYS( 0x7f425fa1, "Bone Normal and Tangent Axis are the same, could not create bones." );
        return false;
    }

    const int numVertices = simulationMesh.m_vertices.getSize();
    const int numFaces = simulationMesh.m_triangles.getSize();

    // Construct the globalTangentAligmentAxis hkVector4
    hkVector4 globalTangentAligmentAxis;
    {
        hkVector4 posXAxis; posXAxis.set(1,0,0);
        hkVector4 negXAxis; negXAxis.set(-1,0,0);
        hkVector4 posYAxis; posYAxis.set(0,1,0);
        hkVector4 negYAxis; negYAxis.set(0,-1,0);
        hkVector4 posZAxis; posZAxis.set(0,0,1);
        hkVector4 negZAxis; negZAxis.set(0,0,-1);

        switch (globalTangentAxis )
        {
        case GT_X_POSITIVE_AXIS:
            globalTangentAligmentAxis = posXAxis;
            break;
        case GT_X_NEGATIVE_AXIS:
            globalTangentAligmentAxis = negXAxis;
            break;
        case GT_Y_POSITIVE_AXIS:
            globalTangentAligmentAxis = posYAxis;
            break;
        case GT_Y_NEGATIVE_AXIS:
            globalTangentAligmentAxis = negYAxis;
            break;
        case GT_Z_POSITIVE_AXIS:
            globalTangentAligmentAxis = posZAxis;
            break;
        case GT_Z_NEGATIVE_AXIS:
            globalTangentAligmentAxis = negZAxis;
            break;
        }
    }

    //calculate the per face normals
    hkArray<hkVector4> faceNormals;
    {
        faceNormals.setSize(numFaces);
        for(int faceIdx = 0; faceIdx < numFaces; faceIdx++)
        {
            hkVector4 vertA = simulationMesh.m_vertices[simulationMesh.m_triangles[faceIdx].m_a];
            hkVector4 vertB = simulationMesh.m_vertices[simulationMesh.m_triangles[faceIdx].m_b];
            hkVector4 vertC = simulationMesh.m_vertices[simulationMesh.m_triangles[faceIdx].m_c];

            hkVector4 edgeA; edgeA.setSub4(vertB, vertA);
            hkVector4 edgeB; edgeB.setSub4(vertC, vertA);
            hkVector4 faceNormal; faceNormal.setCross(edgeA, edgeB);
            faceNormal.normalize3();
            faceNormals[faceIdx] = faceNormal;
        }
    }

    //calculate bone transforms (per vertex)
    if(method == CREATE_BONE_PER_VERTEX)
    {
        boneTransformsOut.setSize(numVertices);

        //calculate vertex normals
        hkArray<hkVector4> vertexNormals;
        {
            hkVector4 zeroVec; zeroVec.setZero4();
            vertexNormals.setSize(numVertices, zeroVec);
            {
                for(int faceIdx = 0; faceIdx < numFaces; faceIdx++)
                {
                    vertexNormals[simulationMesh.m_triangles[faceIdx].m_a].add4(faceNormals[faceIdx]);
                    vertexNormals[simulationMesh.m_triangles[faceIdx].m_b].add4(faceNormals[faceIdx]);
                    vertexNormals[simulationMesh.m_triangles[faceIdx].m_c].add4(faceNormals[faceIdx]);
                }

                for(int vertIdx = 0; vertIdx < numVertices; vertIdx++)
                {
                    vertexNormals[vertIdx].normalize3();
                }
            }
        }

        for(int vertIdx = 0; vertIdx < numVertices; vertIdx++)
        {
            //Calculate the rotation
            hkRotation rotation;
            {
                hkVector4 boneNormalAxis;
                boneNormalAxis = vertexNormals[vertIdx];
                boneNormalAxis.normalize3();

                hkVector4 boneOtherAxis;
                hkVector4 boneTangentAxis;

                hkReal dotProd = hkMath::fabs(boneNormalAxis.dot3(globalTangentAligmentAxis));

                if(hkMath::equal(dotProd, 1.0f, 1e-3f))
                {
                    hkVector4 newGlobalTangentAligmentAxis;
                    hkVector4Util::calculatePerpendicularVector(boneNormalAxis, newGlobalTangentAligmentAxis);
                    newGlobalTangentAligmentAxis.normalize3();

                    boneOtherAxis.setCross(boneNormalAxis, newGlobalTangentAligmentAxis);
                    boneOtherAxis.normalize3();

                    boneTangentAxis.setCross(boneOtherAxis, boneNormalAxis);
                    boneTangentAxis.normalize3();

                    boneAxisChanged[vertIdx] = true;
                }
                else
                {
                    boneOtherAxis.setCross(boneNormalAxis, globalTangentAligmentAxis);
                    boneOtherAxis.normalize3();

                    boneTangentAxis.setCross(boneOtherAxis, boneNormalAxis);
                    boneTangentAxis.normalize3();

                    boneAxisChanged[vertIdx] = false;
                }

                hctBonesFromMeshUtil::_calculateRotation(boneNormalAxis, boneTangentAxis, boneOtherAxis, normalAxis, tangentAxis, rotation);
            }

            boneTransformsOut[vertIdx].setTranslation(simulationMesh.m_vertices[vertIdx]);
            boneTransformsOut[vertIdx].setRotation(rotation);
        }
    }
    //calculate bone transforms (per triangle)
    else if(method == CREATE_BONE_PER_TRIANGLE)
    {
        boneTransformsOut.setSize(numFaces);

        for(int faceIdx = 0; faceIdx < numFaces; faceIdx++)
        {
            //Calculate the rotation
            hkRotation rotation;
            {
                hkVector4 boneNormalAxis;
                boneNormalAxis = faceNormals[faceIdx];

                hkVector4 boneOtherAxis;
                hkVector4 boneTangentAxis;

                hkReal dotProd = hkMath::fabs(boneNormalAxis.dot3(globalTangentAligmentAxis));
                if(hkMath::equal(dotProd, 1.0f, 1e-3f))
                {
                    hkVector4 newGlobalTangentAligmentAxis;
                    hkVector4Util::calculatePerpendicularVector(boneNormalAxis, newGlobalTangentAligmentAxis);
                    newGlobalTangentAligmentAxis.normalize3();

                    boneOtherAxis.setCross(boneNormalAxis, newGlobalTangentAligmentAxis);
                    boneOtherAxis.normalize3();

                    boneTangentAxis.setCross(boneOtherAxis, boneNormalAxis);
                    boneTangentAxis.normalize3();

                    boneAxisChanged[faceIdx] = true;

                }
                else
                {
                    boneOtherAxis.setCross(boneNormalAxis, globalTangentAligmentAxis);
                    boneOtherAxis.normalize3();

                    boneTangentAxis.setCross(boneOtherAxis, boneNormalAxis);
                    boneTangentAxis.normalize3();

                    boneAxisChanged[faceIdx] = false;
                }

                hctBonesFromMeshUtil::_calculateRotation(boneNormalAxis, boneTangentAxis, boneOtherAxis, normalAxis, tangentAxis, rotation);
            }

            //Calculate the triangle's center and use it for the bone's position
            hkVector4 bonePosition;
            {
                const hkVector4& vertexA = simulationMesh.m_vertices[ simulationMesh.m_triangles[faceIdx].m_a ];
                const hkVector4& vertexB = simulationMesh.m_vertices[ simulationMesh.m_triangles[faceIdx].m_b ];
                const hkVector4& vertexC = simulationMesh.m_vertices[ simulationMesh.m_triangles[faceIdx].m_c ];

                bonePosition.setAdd4(vertexA, vertexB);
                bonePosition.add4(vertexC);
                bonePosition.mul4(1.0f/3.0f);
            }

            boneTransformsOut[faceIdx].setTranslation(bonePosition);
            boneTransformsOut[faceIdx].setRotation(rotation);
        }
    }
    return true;
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
