// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <ContentTools/Common/SdkUtils/ClassHierarchy/hctClassHierarchyUtil.h>

#include <Common/Base/Container/String/Deprecated/hkStringOld.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Reflect/Util/hkReflectAny.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

void hctClassHierarchyUtil::createDisplayHierarchyFromClass(const hkReflect::Type* klass, std::string& groupPathDelimiter, hkReal unitConversionModifier, std::vector<ParamGroup*>& groupsArrayOut)
{
    //
    // Build the hierarchical representation of all the class's members.
    //
    {
        std::vector<ParamGroup*> virtualRecursionStack;
        ParamGroup* classGroup = new ParamGroup("General", "general");
        virtualRecursionStack.push_back(classGroup);

        std::vector<ParamGroup*> advancedGroupVirtualRecursionStack;
        ParamGroup* advancedGroup = new ParamGroup("Advanced", "advanced");
        advancedGroupVirtualRecursionStack.push_back(advancedGroup);

        for (hkReflect::DeclIter<hkReflect::FieldDecl> it(klass); it.advance();)
        {
            hkReflect::FieldDecl member = it.current();
            std::string namePrefix("");
            hctClassHierarchyUtil::buildTreeFromClassDescription(klass, member, &namePrefix, groupPathDelimiter, unitConversionModifier, virtualRecursionStack, advancedGroupVirtualRecursionStack);
        }

        // We will have at least those 2 groups in our array (until we reach the pruning below).
        groupsArrayOut.push_back(classGroup);
        groupsArrayOut.push_back(advancedGroup);
    }

    //
    // Flatten Hierarchies, Step 1: all subgroups of the GENERAL group will become root-level groups.
    //
    {
        ParamGroup* generalGroup = groupsArrayOut[0];
        {
            for (int i = 0; i < int(generalGroup->m_subGroups.size()); i++)
            {
                groupsArrayOut.push_back(generalGroup->m_subGroups[i]);
            }
        }
        generalGroup->m_subGroups.clear();
    }

    //
    // Flatten Hierarchies, Step 2: all subgroups deeper than level 2 will be moved to level 2.
    //
    {
        for (int i = 1; i < int(groupsArrayOut.size()); i++)
        {
            ParamGroup* group = groupsArrayOut[i];
            {
                for (int subgroupIndex = 0; subgroupIndex < int(group->m_subGroups.size()); subgroupIndex++)
                {
                    ParamGroup* subgroup = group->m_subGroups[subgroupIndex];
                    hctClassHierarchyUtil::flattenHierarchy(subgroup, group->m_subGroups);
                }
            }
        }
    }

    //
    // Move ADVANCED group to end of array.
    //
    {
        // Right here we should still have at least 2 groups in the array (GENERAL and ADVANCED).
        groupsArrayOut.push_back(groupsArrayOut[1]);
        groupsArrayOut.erase(groupsArrayOut.begin() + 1);
    }

    //
    // Merge all groups on the same level that share the same name.
    //
    hctClassHierarchyUtil::mergeGroupsOfSameName(groupsArrayOut);

    //
    // Remove all empty subtrees from hierarchies.
    //
    hctClassHierarchyUtil::pruneEmptyGroups(groupsArrayOut);
}

//
// This function will recursively traverse a class's member hierarchy, building two full hierarchy trees (one for members
// tagged as ADVANCED, one for the rest) from any enclosed STRUCTs or manually tagged GROUPs. It uses two virtual stacks
// to be able to cope with the manual GROUP tags. The virtual stacks have to be initialized with exactly one group each
// before calling this function for the first time.
//
void hctClassHierarchyUtil::buildTreeFromClassDescription(const hkReflect::Type* klass, hkReflect::FieldDecl member, const std::string* namePrefix, const std::string groupPathDelimiter, hkReal unitConversionModifier, std::vector<ParamGroup*>& virtualRecursionStack, std::vector<ParamGroup*>& advancedGroupVirtualRecursionStack)
{
    // Skip any non-serialized member.
    if ( !member.isSerializable() )
    {
        return;
    }

    // Retrieve UI tags for member (if available).
    const hkUiAttribute* uiAttributes = member.getType()->findAttribute<hkUiAttribute>();

    // Extract the user defined label (if available).
    const char* memberLabel = hctClassHierarchyUtil::getMemberName(member, uiAttributes);

    // Save the group we are currently in.
    ParamGroup* currentGroup = virtualRecursionStack.back();
    ParamGroup* advancedCurrentGroup = advancedGroupVirtualRecursionStack.back();

    // Check whether this is a UUID, will handle that differently
    if ( member.getType()->asRecord() && !member.getType()->equals<hkUuid>() )
    {
        // Regular struct
        // Build the name for the new subgroup (from the supplied group path and the STRUCT's name).
        std::string structPathAsName = *namePrefix;
        structPathAsName.append(member.getName());

        // Create a new GENERAL subgroup for STRUCT and link it to the current GENERAL group.
        ParamGroup* subGroup = new ParamGroup(memberLabel, structPathAsName.c_str());
        currentGroup->addChild(subGroup);

        // Create a new ADVANCED subgroup for STRUCT and link it to the current ADVANCED group.
        ParamGroup* advancedSubGroup = new ParamGroup(memberLabel, structPathAsName.c_str());
        advancedCurrentGroup->addChild(advancedSubGroup);

        virtualRecursionStack.push_back(subGroup);
        advancedGroupVirtualRecursionStack.push_back(advancedSubGroup);
        {
            // Prepare the group path for further subgroups.
            structPathAsName.append(groupPathDelimiter);

            // Iterate over all STRUCT's members
            for (hkReflect::DeclIter<hkReflect::FieldDecl> it(member.getType()); it.advance();)
            {
                hkReflect::FieldDecl structMember = it.current();
                hctClassHierarchyUtil::buildTreeFromClassDescription(member.getType(), structMember, &structPathAsName, groupPathDelimiter, unitConversionModifier, virtualRecursionStack, advancedGroupVirtualRecursionStack);
            }
        }
        virtualRecursionStack.pop_back(); // subGroup
        advancedGroupVirtualRecursionStack.pop_back(); // advancedSubGroup
        return;
    }

    bool visible = false;
    bool editable = true;

    if ( !uiAttributes )
    {
        return;
    }

    {
        // Store the visibility flag.
        visible = uiAttributes->m_visible;
        editable = uiAttributes->m_editable;

        // Retrieve UI tags for class itself (if available).
        const hkUiAttribute* classUiAttributes = klass->findAttribute<hkUiAttribute>();

        // Hide those visible members that have specified in the class's UI attribute.
        if ( visible &&
                classUiAttributes &&
                classUiAttributes->m_hideBaseClassMembers &&
                (hkString::strStr(classUiAttributes->m_hideBaseClassMembers, member.getName()) != HK_NULL) )
        {
            visible = false;
        }

        // Create a new subgroup if there's the GROUP tag (with a valid display name) on the member.
        if ( uiAttributes->m_group && uiAttributes->m_group[0] != 0 )
        {
            // Build the group's internal name (from group path, member name and postfix)
            std::string groupInternalName(*namePrefix);
            groupInternalName.append(uiAttributes->m_group);
            groupInternalName.append("Group");
            verifyAlphaNumericString(groupInternalName);

            // Create a new GENERAL subgroup and link it to the current GENERAL group.
            ParamGroup* subGroup = new ParamGroup(uiAttributes->m_group, groupInternalName.c_str());
            currentGroup->addChild(subGroup);

            // Create a new ADVANCED subgroup and link it to the current ADVANCED group.
            ParamGroup* advancedSubGroup = new ParamGroup(uiAttributes->m_group, groupInternalName.c_str());
            advancedCurrentGroup->addChild(advancedSubGroup);

            // Push the new subgroups on the virtual stacks.
            virtualRecursionStack             .push_back(subGroup);
            advancedGroupVirtualRecursionStack.push_back(advancedSubGroup);

            // Continue working on the newly created subgroups.
            currentGroup         = subGroup;
            advancedCurrentGroup = advancedSubGroup;
        }
    }

    // Create a new entry for the class's member.
    ParamGroup::ClassAndMember newClassAndMember;
    {
        newClassAndMember.m_class             = klass;
        newClassAndMember.m_member            = member;
        newClassAndMember.m_memberLabel       = hctClassHierarchyUtil::prettifyName(memberLabel).cString();
        newClassAndMember.m_memberInternalName. append(*namePrefix);
        newClassAndMember.m_memberInternalName. append(member.getName());
        newClassAndMember.m_visible           = visible;
        newClassAndMember.m_editable          = editable;

        hctClassHierarchyUtil::getDefaultValueAsString(newClassAndMember, unitConversionModifier,newClassAndMember.m_defaultValue);
    }

    // Add the member to the respective group (GENERAL or ADVANCED).
    if ( uiAttributes->m_advanced )
    {
        advancedCurrentGroup->m_members.push_back(newClassAndMember);
    }
    else
    {
        currentGroup->m_members.push_back(newClassAndMember);
    }

    // Close the current subgroup.
    if ( uiAttributes->m_endGroup )
    {
        //HK_ASSERT(0xaf1f2e13, virtualRecursionStack.size() > 1, "Tagging error: 'endGroup' tag is missing a corresponding 'group' tag!");
        if ( virtualRecursionStack.size() > 1 )
        {
            // Pop the current group from the virtual stack.
            virtualRecursionStack             .pop_back();
            advancedGroupVirtualRecursionStack.pop_back();
        }
    }

    // Close another subgroup. Hack.
    if ( uiAttributes->m_endGroup2 )
    {
        if ( virtualRecursionStack.size() > 1 )
        {
            // Pop the current group from the virtual stack.
            virtualRecursionStack             .pop_back();
            advancedGroupVirtualRecursionStack.pop_back();
        }
    }
}



hkStringOld hctClassHierarchyUtil::prettifyName(const char* nameIn)
{
    int nameInLen = (int)strlen(nameIn);
    hkArray<char> buf;
    buf.reserve( nameInLen+10 );

    {
        char c = nameIn[0];
        if (c >= 'a' && c <= 'z')
        {
            buf.pushBack(c + 'A' - 'a');
        }
        else
        {
            buf.pushBack(c);
        }
    }

    int numChars = nameInLen;
    {
        for (int i = 1; i < numChars; i++)
        {
            char c = nameIn[i - 1];
            char cNext = nameIn[i];

            if (c >= 'a' && c <= 'z' && cNext >= 'A' && cNext <= 'Z')
            {
                buf.pushBack(' ');
            }

            buf.pushBack(cNext);
        }
    }

    buf.pushBack(0);
    return buf.begin();
}

namespace
{
    struct PrintDefaultVisitor : public hkReflect::VarVisitor<PrintDefaultVisitor, void, hkReal, hkStringBuf&>
    {
        void visit(const hkReflect::VoidVar&, hkReal, hkStringBuf&) { HK_UNREACHABLE(0x2cb0f1b4, "Void field found" ); }
        void visit(const hkReflect::CompoundVar&, hkReal, hkStringBuf&) { /* do nothing */ }

        void visit(const hkReflect::IntVar& var, hkReal, hkStringBuf& out)
        {
            var.toString(out);
        }

        void visit(const hkReflect::FloatVar& var, hkReal unitConversionModifier, hkStringBuf& out)
        {
            hkReal defaultValue = static_cast<hkReal>(var.getValue());
            defaultValue /= unitConversionModifier;
            out.printf("%f", defaultValue);
        }

        void visit(const hkReflect::BoolVar& var, hkReal, hkStringBuf& out)
        {
            out.append(var.getValue() ? "1" : "0");
        }

        void visit(const hkReflect::StringVar& var, hkReal, hkStringBuf& out)
        {
            out = var.getValue();
        }
    };
}

void hctClassHierarchyUtil::getDefaultValueAsString(const ParamGroup::ClassAndMember& memberData, hkReal unitConversionModifier, std::string& defaultValueStringOut)
{
    const hkReflect::Type* klass  = memberData.m_class;
    hkReflect::FieldDecl member = memberData.m_member;

    // Special case for hkdActions, m_priority has a different default value
    if ( klass && klass->getName() && member && member.getName() && !hkString::strCmp(member.getName(), "priority") )
    {
        hkStringOld strb;
        if (!hkString::strCmp(klass->getName(), "hkdConvexHullAction"))             { strb.printf("7"); defaultValueStringOut = strb.cString(); return; }       // PRIORITY_CONVEX_HULL
        if (!hkString::strCmp(klass->getName(), "hkdRemoveWeakConnectionsAction"))  { strb.printf("60");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_REMOVE_WEAK_CONNECTIONS
        if (!hkString::strCmp(klass->getName(), "hkdConvexDecompositionAction"))        { strb.printf("10");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_CONVEX_DECOMPOSITION
        if (!hkString::strCmp(klass->getName(), "hkdSplitByPhysicsIslandsAction"))  { strb.printf("8"); defaultValueStringOut = strb.cString(); return; }       // PRIORITY_SPLITBYPHYSICSISLANDS
        if (!hkString::strCmp(klass->getName(), "hkdGlueFixedPiecesAction"))            { strb.printf("50");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_GLUE_FIXED_PIECES
        if (!hkString::strCmp(klass->getName(), "hkdDecorateFractureFaceAction"))       { strb.printf("5"); defaultValueStringOut = strb.cString(); return; }       // PRIORITY_DECORATE_FRACTURE_FACE
        if (!hkString::strCmp(klass->getName(), "hkdMeshSimplifierAction"))         { strb.printf("30");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_MESH_SIMPLIFIER
        if (!hkString::strCmp(klass->getName(), "hkdFlattenHierarchyAction"))           { strb.printf("55");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_FLATTEN_HIERARCHY
        if (!hkString::strCmp(klass->getName(), "hkdDecalMapAction"))                   { strb.printf("6"); defaultValueStringOut = strb.cString(); return; }       // PRIORITY_CREATE_DECAL_MAP
        if (!hkString::strCmp(klass->getName(), "hkdShareVerticesAction"))          { strb.printf("20");    defaultValueStringOut = strb.cString(); return; }       // PRIORITY_SHARE_VERTICES
    }

    const hkSemanticsAttribute* semanticsAttributes = member.getType()->findAttribute<hkSemanticsAttribute>();

    hkStringBuf defaultValueString;

    hkReflect::Var defaultValue = member.getType()->getDefault();

    if (member.getType()->equals<hkVector4>())
    {
        if (!semanticsAttributes || semanticsAttributes->m_type != hkSemanticsAttribute::POSITION)
        {
            unitConversionModifier = 1.0f;
        }
        hkVector4 vec = defaultValue ? *defaultValue.dynCast<hkVector4>() : hkVector4::getZero();
        hkReal defaultValue0 = vec.getComponent<0>();
        hkReal defaultValue1 = vec.getComponent<1>();
        hkReal defaultValue2 = vec.getComponent<2>();
        defaultValue0 /= unitConversionModifier;
        defaultValue1 /= unitConversionModifier;
        defaultValue2 /= unitConversionModifier;
        defaultValueString.printf("%f, %f, %f", defaultValue0, defaultValue1, defaultValue2);
    }
    else if (const hk::Presets* presets = member.getType()->findAttribute<hk::Presets>())
    {
        // Look up the index of the default preset.
        // If the type has no default, stick to the first value.
        int defaultIntValue = 0;
        if (defaultValue)
        {
            hkReflect::IntVar defaultValueEnumVariant = defaultValue;
            // Scan through all enum values and find the entry whose numerical value matches the default value.
            // This is necessary in cases where the enum entries' values won't start with 0.
            for (int i = 0; i < presets->getNumPresets(); i++)
            {
                if(hkReflect::IntVar(presets->getPreset(i)).getValue() == defaultValueEnumVariant.getValue())
                {
                    // use the matching entry's index (+1) as the index into the dropdownlist's string array.
                    defaultIntValue = i;
                    break;
                }
            }
        }
        defaultValueString.printf("%d", defaultIntValue);
    }
    else if (member.getType()->equals<hkUuid>())
    {
        hkUuid uuid;
        if (memberData.m_editable)  { uuid = hkUuid::getNil(); }    // Editable, set to NULL
        else                            { uuid.setRandom(); }   // Non-editable, auto-assign a new UUID

        hkStringBuf strb;
        uuid.toString(strb);
        defaultValueString.printf("%s", strb.cString());
    }
    else
    {
        if (!semanticsAttributes || semanticsAttributes->m_type != hkSemanticsAttribute::DISTANCE)
        {
            unitConversionModifier = 1.0f;
        }

        // use zero if no default
        hkUint64 aux = 0;
        HK_ASSERT_NO_MSG(0xa24f12d, member.getType()->asCompound() 
            || member.getType()->getSizeOf() <= sizeof(hkUint64));
        if (!defaultValue.isValid())
        {
            defaultValue = hkReflect::Var(&aux, member.getType());
        }

        PrintDefaultVisitor().dispatch(defaultValue, unitConversionModifier, defaultValueString);
    }

    defaultValueStringOut = defaultValueString.cString();

    return;
}


const char* hctClassHierarchyUtil::getMemberName(hkReflect::FieldDecl member, const hkUiAttribute* uiAttributes)
{
    if ( uiAttributes && uiAttributes->m_label && uiAttributes->m_label[0] != 0 )
    {
        return uiAttributes->m_label;
    }
    else
    {
        return member.getName();
    }
}


void hctClassHierarchyUtil::pruneEmptyGroups(std::vector<ParamGroup*>& groupsArray)
{
    for (int i = int(groupsArray.size())-1; i >= 0; i--)
    {
        ParamGroup* group = groupsArray[i];
        pruneEmptyGroupsRecursively(group);
        if ( group->m_members.size() == 0 && group->m_subGroups.size() == 0 )
        {
            groupsArray.erase(groupsArray.begin() + i);
            delete group;
        }
    }
}


//
// Recursively removes all empty subtrees from a given group.
// Returns TRUE if the supplied group is empty (i.e. had neither members nor subgroups), FALSE otherwise.
//
bool hctClassHierarchyUtil::pruneEmptyGroupsRecursively(ParamGroup* group)
{
    {
        for (int i = int(group->m_subGroups.size())-1; i >= 0; i--)
        {
            ParamGroup* subGroup = group->m_subGroups[i];

            // If the subgroup is empty, remove it from this group's subgroup list.
            if ( pruneEmptyGroupsRecursively(subGroup) )
            {
                group->m_subGroups.erase(group->m_subGroups.begin() + i);
                delete subGroup;
            }
        }
    }

    if ( group->m_members.size() == 0 && group->m_subGroups.size() == 0 )
    {
        return true;
    }

    return false;
}


void hctClassHierarchyUtil::pruneInvisibleMembers(hkAttributeHideCriteria::Types hideCriteria, std::vector<ParamGroup*>& groupsArray)
{
    for (int i = 0; i < int(groupsArray.size()); i++)
    {
        pruneInvisibleMembersRecursively(hideCriteria, groupsArray[i]);
    }
}


void hctClassHierarchyUtil::pruneInvisibleMembersRecursively(hkAttributeHideCriteria::Types hideCriteria, ParamGroup* group)
{
    {
        for (int i = int(group->m_subGroups.size())-1; i >= 0; i--)
        {
            ParamGroup* subGroup = group->m_subGroups[i];
            pruneInvisibleMembersRecursively(hideCriteria, subGroup);
        }
    }

    {
        for (int i = int(group->m_members.size())-1; i >= 0; i--)
        {
            ParamGroup::ClassAndMember& member = group->m_members[i];
            const hkUiAttribute* uiAttributes = member.m_member.getType()->findAttribute<hkUiAttribute>();
            if ( uiAttributes )
            {
                if ( member.m_visible == false || (uiAttributes->m_hideCriteria & hideCriteria) )
                {
                    group->m_members.erase(group->m_members.begin() + i);
                }
            }
        }
    }

    return;
}


void hctClassHierarchyUtil::pruneMembersByClassType(const hkReflect::Type* type, std::vector<ParamGroup*>& groupsArray)
{
    for (int i = 0; i < int(groupsArray.size()); i++)
    {
        pruneMembersByClassTypeRecursively(type, groupsArray[i]);
    }
}


void hctClassHierarchyUtil::pruneMembersByClassTypeRecursively(const hkReflect::Type* type, ParamGroup* group)
{
    {
        for (int i = int(group->m_subGroups.size())-1; i >= 0; i--)
        {
            ParamGroup* subGroup = group->m_subGroups[i];
            pruneMembersByClassTypeRecursively(type, subGroup);
        }
    }

    {
        for (int i = int(group->m_members.size())-1; i >= 0; i--)
        {
            ParamGroup::ClassAndMember& member = group->m_members[i];
            if ( member.m_member.getType()->extendsOrEquals(type) )
            {
                group->m_members.erase(group->m_members.begin() + i);
            }
        }
    }

    return;
}


void hctClassHierarchyUtil::markVisuallyEmptyGroup(hkAttributeHideCriteria::Types modelerId, std::vector<ParamGroup*>& groupsArray)
{
    for (int i = 0; i < int(groupsArray.size()); i++)
    {
        ParamGroup* group = groupsArray[i];
        markVisuallyEmptyGroupRecursively(modelerId, group);
    }
}


//
// Recursively removes all empty subtrees from a given group.
// Returns TRUE if the supplied group is empty (i.e. had neither members nor subgroups), FALSE otherwise.
//
void hctClassHierarchyUtil::markVisuallyEmptyGroupRecursively(hkAttributeHideCriteria::Types modelerId, ParamGroup* group)
{
    //
    // Count the number of visible subgroups (i.e. subgroups that have visible members).
    // This will also recurse into the subgroup itself and update its state.
    //
    int numVisibleSubgroups = 0;
    {
        for (int i = 0; i < int(group->m_subGroups.size()); i++)
        {
            ParamGroup* subGroup = group->m_subGroups[i];
            markVisuallyEmptyGroupRecursively(modelerId, subGroup);
            if ( !subGroup->m_visuallyEmpty )
            {
                numVisibleSubgroups++;
            }
        }
    }

    //
    // Count the number of visible members (not including any subgroups)
    //
    int numVisibleMembers = 0;
    {
        for (int i = 0; i < int(group->m_members.size()); i++)
        {
            ParamGroup::ClassAndMember& member = group->m_members[i];
            const hkUiAttribute* uiAttributes = member.m_member.getType()->findAttribute<hkUiAttribute>();
            if ( uiAttributes )
            {
                if ( member.m_visible && !(uiAttributes->m_hideCriteria & modelerId) )
                {
                    numVisibleMembers++;
                }
            }
        }
    }

    if ( numVisibleMembers == 0 && numVisibleSubgroups == 0 )
    {
        group->m_visuallyEmpty = true;
    }
    else
    {
        group->m_visuallyEmpty = false;
    }
}


void hctClassHierarchyUtil::mergeGroupsOfSameName(std::vector<ParamGroup*>& groupsArray)
{
    for (int a = 0; a < int(groupsArray.size()); a++)
    {
        ParamGroup* groupA = groupsArray[a];

        for (int b = a+1; b < int(groupsArray.size()); b++)
        {
            ParamGroup* groupB = groupsArray[b];

            if ( hkString::strCasecmp(groupA->m_groupLabel, groupB->m_groupLabel) == 0 )
            {
                {
                    for (int c = 0; c < int(groupB->m_members.size()); c++)
                    {
                        ParamGroup::ClassAndMember& member = groupB->m_members[c];
                        groupA->m_members.push_back(member);
                    }
                    groupB->m_members.clear();
                }
                {
                    for (int c = 0; c < int(groupB->m_subGroups.size()); c++)
                    {
                        ParamGroup* subGroup = groupB->m_subGroups[c];
                        groupA->m_subGroups.push_back(subGroup);
                    }
                    groupB->m_subGroups.clear();
                }
            }
        }

        mergeGroupsOfSameName(groupA->m_subGroups);
    }
}


void hctClassHierarchyUtil::flattenHierarchy(ParamGroup* group, std::vector<ParamGroup*>& rootLevelArray)
{
    for (int i = 0; i < int(group->m_subGroups.size()); i++)
    {
        ParamGroup* subGroup = group->m_subGroups[i];
        rootLevelArray.push_back(subGroup);
        flattenHierarchy(subGroup, rootLevelArray);
    }
    group->m_subGroups.clear();
}


bool hctClassHierarchyUtil::isParentClassOf(const char* parentClassName, const hkReflect::Type* klass)
{
    while ( klass )
    {
        if ( hkString::strCasecmp(klass->getName(), parentClassName ) == 0 )
        {
            return true;
        }
        klass = klass->getParent();
    }

    return false;
}


void hctClassHierarchyUtil::verifyAlphaNumericString(std::string& inputString)
{
    //
    // Remove all non-alpha-numeric characters from string.
    //
    {
        for (int i = int(inputString.size())-1; i >= 0; i--)
        {
            if ( !isalnum(inputString[i]) )
            {
                inputString.erase(i, 1);
            }
        }
    }

    //
    // If string would be empty after the above removal, at least add one dummy character back.
    //
    if ( inputString.size() == 0 )
    {
        inputString.push_back('a');
    }
    else
    {
        //
        // Prepend an underscore if first letter is a digit.
        // Otherwise turn the first character to lowercase.
        //
        if ( isdigit(inputString[0]) )
        {
            inputString.insert(0, "_");
        }
        else
        {
            inputString[0] = char(tolower(inputString[0]));
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
