// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

    /// This class stores component data (vertex, edge, triangle selection and data) using global indices for the components.
    /// Then, with the help of maps between section indices and global indices, it creates and stores that data in the sections
    /// of an hkxMesh.
    /// This is used by all exporters to easily map between data in the modeller to user channels in the hkxMesh sections
class hctUserChannelUtil
{
    public:

            /// Defined the type of global component data. Currently only "CT_VERTEX_SELECTION" is supported.
            /// Note that FACE channels will become TRIANGLE channels.
        enum ChannelType
        {
            CT_INVALID = -1,

            CT_VERTEX_SELECTION =0,
            CT_VERTEX_FLOAT,            // floats per vertex counting duplicated vertices as identical
            CT_SAMPLE_FLOAT,            // floats per vertex counting duplicated vertices separately

            CT_VERTEX_INT,              // not implemented yet
            CT_VERTEX_VECTOR,           // not implemented yet
            CT_EDGE_SELECTION = 10 ,    // not implemented yet
            CT_FACE_SELECTION = 20 ,    // face selection (set of triangle indices)

            CT_NORMAL_IDS               // unique normal IDs per vertex, for smoothing
        };

        enum ChannelDimensions
        {
            CD_INVALID = -1,

            CD_FLOAT = 0,
            CD_DISTANCE,
            CD_ANGLE
        };

            /// Union of index, int, float and vector elements (depending on the type of component data)
        union ChannelDataItem
        {
            int m_index;
            int m_integer;
            float m_float;
            float m_vector[4];
        };

            /// Data describing a channel for component data. Indices are global.
        struct GlobalChannel
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, GlobalChannel );
            hkStringOld m_channelName;
            ChannelType m_channelType;
            ChannelDimensions m_channelDimensions;
            hkArray<ChannelDataItem> m_channelData;
            hkReal m_scaleMin;
            hkReal m_scaleMax;

            GlobalChannel (const GlobalChannel& other)
            {
                m_channelName = other.m_channelName;
                m_channelType = other.m_channelType;
                m_channelDimensions = other.m_channelDimensions;
                m_channelData = other.m_channelData;
                m_scaleMin = other.m_scaleMin;
                m_scaleMax = other.m_scaleMax;
            }

            GlobalChannel () : m_channelName (""), m_channelType (CT_INVALID), m_scaleMin(0.0f), m_scaleMax(1.0f) {}
        };

            /// A map between the indices of this section and the original indices of the global mesh
        struct SectionToGlobalMap
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, SectionToGlobalMap );

            // Maps vertices in the section to vertices in the mesh, counting duplicated vertices as identical
            hkArray<int> m_sectionVertexIdToGlobalVertexId;

            // Maps vertices in the section to vertices in the mesh, counting duplicated vertices separately
            hkArray<int> m_sectionVertexIdToGlobalSampleId;

            hkArray<int> m_sectionTriangleIdToGlobalFaceId;

            hkArray<int> m_sectionTriangleIdToGlobalTriangleId;

            // Maps vertices to normal IDs, where vertices sharing normal IDs should be assigned equal normals. This is used to maintain mesh smoothness
            // even in the presence of UV seams which cause vertex duplication.
            hkArray<hkUint16> m_sectionVertexIdToGlobalNormalId;

            SectionToGlobalMap (const SectionToGlobalMap& other)
            {
                m_sectionVertexIdToGlobalVertexId = other.m_sectionVertexIdToGlobalVertexId;
                m_sectionVertexIdToGlobalSampleId = other.m_sectionVertexIdToGlobalSampleId;
                m_sectionTriangleIdToGlobalFaceId = other.m_sectionTriangleIdToGlobalFaceId;
                m_sectionVertexIdToGlobalNormalId = other.m_sectionVertexIdToGlobalNormalId;
                m_sectionTriangleIdToGlobalTriangleId = other.m_sectionTriangleIdToGlobalTriangleId;
            }

            SectionToGlobalMap () {}
        };

            /// Add a new user channel to the mesh
        void addGlobalChannel (const GlobalChannel& globalChannel);

            /// Register a new section in the mesh and provide a map between indices in the section and indices in the original mesh
        void registerSection (const SectionToGlobalMap& sectionMap);

            /// Converts the stored global channels to individual per-section channels, which are added to the hkxMeshSections of the
            /// given mesh. Allocations are done using the given hctFilterMemoryTracker object.
        void storeChannelsInMesh (hkxMesh* theMesh);

            /// Clears all the stored data. Useful if the object is reused for multiple meshes
        void clear();

    const hkArray<SectionToGlobalMap>& getSectionGlobalMaps() const { return m_sectionMaps; }

    private:

        hkArray<GlobalChannel> m_globalChannels;

        hkArray<SectionToGlobalMap> m_sectionMaps;

        void processVertexSelection ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const;
        void processFaceSelection ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const;
        void processFloatData ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const;
        void processNormalIDs ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const;
        void processEdgeSelection ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const;

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
