// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/SceneExport/hctSceneExport.h> // PCH
#include <ContentTools/Common/SceneExport/Userchannel/hctUserChannelUtil.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/Channels/hkxVertexSelectionChannel.h>
#include <Common/SceneData/Mesh/Channels/hkxVertexFloatDataChannel.h>
#include <Common/SceneData/Mesh/Channels/hkxVertexIntDataChannel.h>
#include <Common/SceneData/Mesh/Channels/hkxTriangleSelectionChannel.h>
#include <Common/SceneData/Mesh/Channels/hkxEdgeSelectionChannel.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>


// Add a new user channel to the mesh
void hctUserChannelUtil::addGlobalChannel (const hctUserChannelUtil::GlobalChannel& globalChannel)
{
    m_globalChannels.pushBack(globalChannel);
}

// Register a new section in the mesh and provide a map between indices in the section and indices in the original mesh
void hctUserChannelUtil::registerSection (const hctUserChannelUtil::SectionToGlobalMap& sectionMap)
{
    m_sectionMaps.pushBack(sectionMap);
}


// do the work
void hctUserChannelUtil::storeChannelsInMesh (hkxMesh* theMesh)
{
    if (theMesh->m_sections.getSize() != m_sectionMaps.getSize())
    {
        HK_WARN_ALWAYS(0xabba0000, "Internal error - inconsistent number of mesh sections");
        return;
    }

    const int numChannels = m_globalChannels.getSize();

    if ((theMesh->m_sections.getSize() == 0) || (numChannels == 0))
    {
        // Nothing to do
        return;
    }

    // Initialize the arrays
    {
        // Channel info array in hkxmesh
        theMesh->m_userChannelInfos.setSize(numChannels);

        // Userchannels (variants) in each section
        for (int si=0; si<theMesh->m_sections.getSize(); si++)
        {
            hkxMeshSection* section = theMesh->m_sections[si];
            section->m_userChannels.setSize(numChannels);
        }
    }

    // Now, process each channel
    for (int ci=0; ci< m_globalChannels.getSize(); ci++)
    {
        const hctUserChannelUtil::GlobalChannel& channel = m_globalChannels[ci];

        // Process each section
        for (int si=0; si<theMesh->m_sections.getSize(); si++)
        {
            switch (channel.m_channelType)
            {
                case CT_VERTEX_SELECTION:
                    {
                        processVertexSelection ( ci, si, theMesh);
                        break;
                    }
                case CT_FACE_SELECTION:
                    {
                        processFaceSelection( ci, si, theMesh);
                        break;
                    }
                case CT_EDGE_SELECTION:
                    {
                        processEdgeSelection( ci, si, theMesh);
                        break;
                    }

                case CT_VERTEX_FLOAT:
                case CT_SAMPLE_FLOAT:
                    {
                        processFloatData( ci, si, theMesh);
                        break;
                    }

                case CT_NORMAL_IDS:
                    {
                        processNormalIDs( ci, si, theMesh);
                        break;
                    }

                default:
                    {
                        HK_ASSERT(0xabbae66d, 0, "Unsupported mesh channel type");
                        continue;
                    }
            }

        }

        // And set the channel info
        theMesh->m_userChannelInfos[ci].setAndDontIncrementRefCount(new hkxMesh::UserChannelInfo);
        theMesh->m_userChannelInfos[ci]->m_name = channel.m_channelName.cString();
        theMesh->m_userChannelInfos[ci]->m_className = theMesh->m_sections[0]->m_userChannels[ci].getType()->getName();
    }

}

static bool _compareIndices (const hkInt32& one, const hkInt32& two)
{
    return one < two;
}

void hctUserChannelUtil::processVertexSelection (int channelNumber, int sectionNumber, hkxMesh* theMesh) const
{
    const hctUserChannelUtil::GlobalChannel& channel = m_globalChannels[channelNumber];
    const hkxMeshSection* section = theMesh->m_sections[sectionNumber];
    const SectionToGlobalMap& sectionMap = m_sectionMaps[sectionNumber];
    hkRefVariant& sectionChannel = const_cast<hkRefVariant&>( section->m_userChannels[channelNumber] );

    hkxVertexSelectionChannel* selectionChannel = new hkxVertexSelectionChannel();

    hkArray<hkInt32>& sectionIndices = selectionChannel->m_selectedVertices;
    for (int i=0; i<channel.m_channelData.getSize(); i++)
    {
        int selectedGlobalVertexIndex = channel.m_channelData[i].m_index;

        // Now see if that vertex is selected in this section (it could be many times)
        for (int j=0; j<sectionMap.m_sectionVertexIdToGlobalVertexId.getSize(); j++)
        {
            if (sectionMap.m_sectionVertexIdToGlobalVertexId[j] == selectedGlobalVertexIndex)
            {
                sectionIndices.pushBack(j);
            }
        }
    }

    // Sort indices into ascending order
    hkAlgorithm::quickSort(sectionIndices.begin(), sectionIndices.getSize(), _compareIndices);

    sectionChannel = selectionChannel;
    selectionChannel->removeReference();
}

void hctUserChannelUtil::processFaceSelection (int channelNumber, int sectionNumber, hkxMesh* theMesh) const
{
    const hctUserChannelUtil::GlobalChannel& channel = m_globalChannels[channelNumber];
    const hkxMeshSection* section = theMesh->m_sections[sectionNumber];
    const SectionToGlobalMap& sectionMap = m_sectionMaps[sectionNumber];
    hkRefVariant& sectionChannel = const_cast<hkRefVariant&>( section->m_userChannels[channelNumber] );

    hkxTriangleSelectionChannel* selectionChannel = new hkxTriangleSelectionChannel();

    hkArray<hkInt32>& sectionIndices = selectionChannel->m_selectedTriangles;
    for (int i=0; i<channel.m_channelData.getSize(); i++)
    {
        int selectedGlobalFaceIndex = channel.m_channelData[i].m_index;

        // Now see if that vertex is selected in this section (it could be many times)
        for (int j=0; j<sectionMap.m_sectionTriangleIdToGlobalFaceId.getSize(); j++)
        {
            if (sectionMap.m_sectionTriangleIdToGlobalFaceId[j] == selectedGlobalFaceIndex)
            {
                sectionIndices.pushBack(j);
            }
        }
    }

    sectionChannel = selectionChannel;
    selectionChannel->removeReference();
}

void hctUserChannelUtil::processFloatData ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const
{
    const hctUserChannelUtil::GlobalChannel& channel = m_globalChannels[channelNumber];
    const hkxMeshSection* section = theMesh->m_sections[sectionNumber];
    const SectionToGlobalMap& sectionMap = m_sectionMaps[sectionNumber];
    hkRefVariant& sectionChannel = const_cast<hkRefVariant&>( section->m_userChannels[channelNumber] );

    hkxVertexFloatDataChannel* floatChannel = new hkxVertexFloatDataChannel();

    hkArray<float>& sectionFloats = floatChannel->getPerVertexFloats();

    switch (channel.m_channelType)
    {
        case CT_VERTEX_FLOAT:
        {
            for (int i=0; i<sectionMap.m_sectionVertexIdToGlobalVertexId.getSize(); i++)
            {
                const int globalIdx = sectionMap.m_sectionVertexIdToGlobalVertexId[i];

                if ( globalIdx>=0 && globalIdx<channel.m_channelData.getSize() )
                {
                    const float value = channel.m_channelData[globalIdx].m_float;
                    sectionFloats.pushBack(value);
                }
                else
                {
                    sectionFloats.pushBack(0.0f);
                }
            }
            break;
        }

        case CT_SAMPLE_FLOAT:
        {
            for (int i=0; i<sectionMap.m_sectionVertexIdToGlobalSampleId.getSize(); i++)
            {
                const int globalIdx = sectionMap.m_sectionVertexIdToGlobalSampleId[i];

                if ( globalIdx>=0 && globalIdx<channel.m_channelData.getSize() )
                {
                    const float value = channel.m_channelData[globalIdx].m_float;
                    sectionFloats.pushBack(value);
                }
                else
                {
                    sectionFloats.pushBack(0.0f);
                }
            }
            break;
        }

        default:
            break;
    }

    // Set the dimensions
    floatChannel->m_dimensions = (hkxVertexFloatDataChannel::VertexFloatDimensions) channel.m_channelDimensions;
    floatChannel->m_scaleMin = channel.m_scaleMin;
    floatChannel->m_scaleMax = channel.m_scaleMax;

    // set the variant
    sectionChannel = floatChannel;
    floatChannel->removeReference();
}


struct VertexID
{
    hkUint32 m_vi;
    hkUint16 m_id;
};

static bool _compareVertexIDs (const VertexID& one, const VertexID& two)
{
    return one.m_id < two.m_id;
}

static inline void _remapNormalIDs(hkArray<hkUint16>& normalIDsOut, const hkArray<hkUint16>& normalIDsIn)
{
    // Normal IDs must be less than the number of vertices in the selection (as required by Cloth hclSetupMesh).
    // Since the normalIDs are just arbitrary labels, remap into this range. In fact, we go ahead and remap into
    // the range [0, (number of unique normal IDs)-1].
    hkArray<VertexID> vertexIDs;

    for (int vi=0; vi<normalIDsIn.getSize(); ++vi)
    {
        VertexID vertexID;
        vertexID.m_vi = hkUint32(vi);
        vertexID.m_id = normalIDsIn[vi];
        vertexIDs.pushBack(vertexID);
    }

    hkAlgorithm::heapSort(vertexIDs.begin(), vertexIDs.getSize(), _compareVertexIDs);

    int newId = -1;
    int origId = -1;
    for (int i=0; i<vertexIDs.getSize(); ++i)
    {
        VertexID& vertexID = vertexIDs[i];
        int id = int(vertexID.m_id);
        if ( id != origId )
        {
            origId = id;
            newId++;
        }
        vertexID.m_id = hkUint16(newId);
    }

    normalIDsOut.setSize(normalIDsIn.getSize());

    for (int i=0; i<vertexIDs.getSize(); ++i)
    {
        VertexID& vertexID = vertexIDs[i];
        normalIDsOut[vertexID.m_vi] = vertexID.m_id;
    }

    for (int i=0; i<normalIDsOut.getSize(); ++i)
    {
        HK_ASSERT_NO_MSG(0x14e84a58, normalIDsOut[i] < normalIDsIn.getSize());
    }
}


void hctUserChannelUtil::processNormalIDs ( int channelNumber, int sectionNumber, hkxMesh* theMesh) const
{
    const hkxMeshSection* section = theMesh->m_sections[sectionNumber];
    const SectionToGlobalMap& sectionMap = m_sectionMaps[sectionNumber];
    hkRefVariant& sectionChannel = const_cast<hkRefVariant&>( section->m_userChannels[channelNumber] );

    hkxVertexIntDataChannel* intChannel = new hkxVertexIntDataChannel();

    hkArray<hkUint16> normalIDs;

    // We assume one int per section vertex
    for (int vi=0; vi<section->m_vertexBuffer->getNumVertices(); vi++)
    {
        hkUint16 normalID = 65535;

        if ( vi < sectionMap.m_sectionVertexIdToGlobalNormalId.getSize() )
        {
            normalID = sectionMap.m_sectionVertexIdToGlobalNormalId[vi];
        }
        normalIDs.pushBack(normalID);
    }

    // Exported Normal IDs should lie in the range [0, (num. verts. in section)-1]. They are not necessarily, e.g. from Maya. Make it so.
    hkArray<hkUint16> remappedNormalIDs;
    _remapNormalIDs(remappedNormalIDs, normalIDs);

    intChannel->m_perVertexInts.setSize(remappedNormalIDs.getSize());
    for (int vi=0; vi<remappedNormalIDs.getSize(); ++vi)
    {
        intChannel->m_perVertexInts[vi] = hkInt32( remappedNormalIDs[vi] );
    }

    sectionChannel = intChannel;
    intChannel->removeReference();
}

// speed up hctUserChannelUtil::processEdgeSelection for large poly meshes
namespace
{
class TriangleOfEdgeFinder
{
public:
    TriangleOfEdgeFinder(const hkArray<int>& triangleIdToGlobalTriangleId)
    {
        edgeIndicesByTriangleIndex.reserveExactly(triangleIdToGlobalTriangleId.getSize());

        for (int i = 0; i < triangleIdToGlobalTriangleId.getSize(); ++i)
        {
            edgeIndicesByTriangleIndex.pushBackUnchecked(EdgeTriangleIndexes(i, triangleIdToGlobalTriangleId[i]));
        }

        hkAlgorithm::quickSort<EdgeTriangleIndexes, CompareTriangleIndex >(
            edgeIndicesByTriangleIndex.begin(),
            edgeIndicesByTriangleIndex.getSize(),
            CompareTriangleIndex());
    }

    int findEdgeIndexWithTriangleIndex(int triangleIndex)
    {
        const hkArray<EdgeTriangleIndexes>& a = edgeIndicesByTriangleIndex;
        int hi(a.getSize() - 1);
        int lo(0);

        do
        {
            int index = (hi + lo) / 2;

            if (triangleIndex == a[index].triangleIndex)
                return a[index].edgeIndex;

            else if (triangleIndex < a[index].triangleIndex)
                hi = index - 1;

            else
                lo = index + 1;
        }
        while (lo <= hi);

        return -1;
    }

private:
    struct EdgeTriangleIndexes
    {
        HK_INLINE EdgeTriangleIndexes() {}
        HK_INLINE EdgeTriangleIndexes(int e, int t) : edgeIndex(e), triangleIndex(t) {}
        int edgeIndex;
        int triangleIndex;
    };

    class CompareTriangleIndex
    {
    public:
        HK_INLINE hkBool32 operator() ( const EdgeTriangleIndexes& a, const EdgeTriangleIndexes& b ) { return a.triangleIndex < b.triangleIndex; }
    };

    hkArray<EdgeTriangleIndexes> edgeIndicesByTriangleIndex;
};

} // namespace

void hctUserChannelUtil::processEdgeSelection( int channelNumber, int sectionNumber, hkxMesh* theMesh ) const
{
    const hctUserChannelUtil::GlobalChannel& channel = m_globalChannels[channelNumber];
    const SectionToGlobalMap& sectionMap = m_sectionMaps[sectionNumber];

    ///*const*/ hkxMeshSection* section = theMesh->m_sections[sectionNumber];
    hkRefVariant& sectionChannel = static_cast<hkRefVariant&>( theMesh->m_sections[sectionNumber]->m_userChannels[channelNumber] );
    hkxEdgeSelectionChannel* selectionChannel = new hkxEdgeSelectionChannel();
    TriangleOfEdgeFinder mapper(sectionMap.m_sectionTriangleIdToGlobalTriangleId);

    hkArray<hkInt32>& sectionIndices = selectionChannel->m_selectedEdges;
    for (int i=0; i<channel.m_channelData.getSize(); i++)
    {
        // by convention: edge_index = 3*face_index + local_edge_id, for local_edge_id={0,1,2}
        const int selectedGlobalEdgeIndex = channel.m_channelData[i].m_index;
        const int triangleIndex = selectedGlobalEdgeIndex / 3;
        const int edgeIndex = selectedGlobalEdgeIndex % 3;

        // Now see if the triangle of this edge is contained in this section
        int index = mapper.findEdgeIndexWithTriangleIndex(triangleIndex);
        if (index != -1)
        {
            sectionIndices.pushBack(3*index + edgeIndex);
        }
    }

    sectionChannel = selectionChannel;
    selectionChannel->removeReference();
}


void hctUserChannelUtil::clear()
{
    m_globalChannels.clear();
    m_sectionMaps.clear();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
