// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once


///
/// This Mesh Builder class is an abstract interface that hides the modeler-specific implementations
/// for common tasks that are required for creating a new mesh/node in all modelers.
///
/// Most functions offer an 'immediate' variant (which performs the respective action right away) as
/// well as a 'delayed' variant (which, if supported by the modeler, collects that data and performs
/// the operation only once the corresponding 'realize' function is called).
///
/// Functions working on channels (e.g. Color and Texture) require the 'selectXXXChannel()' function
/// to be called before data is added to the respective channel.
///
class hctModelerMeshBuilder
{
    public:

        typedef int VertexId;
        typedef int ColorId;
        typedef int UvId;
        typedef int NormalId;

    public:

        //
        // Mesh
        //
        virtual VertexId    addMeshVertexDelayed(const hkVector4& position) = 0;
        virtual int         addMeshTriangleDelayed(VertexId positionIndex0, VertexId positionIndex1, VertexId positionIndex2) = 0;
        virtual void        realizeDelayedMesh() = 0;

        //
        // Vertex Colors
        //
        // Note: Vertex Color triangles and Mesh triangles have to match 1:1
        // Note: If you are adding Vertex Colors using addVertexColorDelayed(), make sure to call realizeDelayedVertexColors() *before*
        //       your (first) call to addVertexColorTriangleImmediate() or realizeDelayedVertexColorTriangles().
        //
        virtual void        selectColorChannel(int channel) = 0;

        virtual ColorId     addVertexColorImmediate(float r, float g, float b) = 0;
        virtual ColorId     addVertexColorDelayed  (float r, float g, float b) = 0;
        virtual void        realizeDelayedVertexColors() = 0;

        virtual int         addVertexColorTriangleImmediate(int triangleIndex, ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2) = 0;
        virtual int         addVertexColorTriangleDelayed                     (ColorId colorIndexVertex0, ColorId colorIndexVertex1, ColorId colorIndexVertex2) = 0;
        virtual void        realizeDelayedVertexColorTriangles() = 0;

        //
        // Textures
        //
        // Note: Vertex UV triangles and Mesh triangles have to match 1:1
        // Note: If you are adding Vertex Colors using addVertexUvDelayed(), make sure to call realizeDelayedVertexUvs() *before*
        //       your (first) call to addVertexUvTriangleImmediate() or realizeDelayedVertexUvTriangles().
        //
        virtual void        selectUvChannel(int channel) = 0;

        virtual UvId        addVertexUvImmediate(float u, float v) = 0;
        virtual UvId        addVertexUvDelayed  (float u, float v) = 0;
        virtual void        realizeDelayedVertexUvs() = 0;

        virtual int         addVertexUvTriangleImmediate(int triangleIndex, UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2) = 0;
        virtual int         addVertexUvTriangleDelayed                     (UvId uvIndexVertex0, UvId uvIndexVertex1, UvId uvIndexVertex2) = 0;
        virtual void        realizeDelayedVertexUvTriangles() = 0;

        //
        // Normals
        //
        virtual NormalId    addNormalDelayed(const hkVector4& normal) = 0;
        virtual int         addNormalTriangleDelayed(NormalId triangleVertex0, NormalId triangleVertex1, NormalId triangleVertex2) = 0;
        virtual void        realizeDelayedNormals() = 0;

        //
        // Materials
        //
        virtual void        setTriangleMaterialImmediate(int triangleIndex, const char* materialName) = 0;
        virtual void        setTriangleMaterialDelayed  (int triangleIndex, const char* materialName) = 0;
        virtual void        realizeDelayedTriangleMaterials() = 0;

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
