// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/SceneExport/hctSceneExport.h> // PCH

#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>
#include <Common/GeometryUtilities/Mesh/hkMeshVertexBuffer.h>
#include <Common/GeometryUtilities/Mesh/Utils/VertexBufferUtil/hkMeshVertexBufferUtil.h>
#include <Common/GeometryUtilities/Mesh/Utils/MeshSectionLockSet/hkMeshSectionLockSet.h>
#include <Common/GeometryUtilities/Mesh/Utils/PrimitiveUtil/hkMeshPrimitiveUtil.h>
#include <Common/GeometryUtilities/Mesh/Utils/FindUniquePositionsUtil/hkFindUniquePositionsUtil.h>

#include <ContentTools/Common/SceneExport/Importer/MeshImport/hctModelerMeshBuilder.h>

#include <ContentTools/Common/SceneExport/Importer/MeshImport/hctMeshImportUtilities.h>


/*

Some limitations/issues:

1) Vertices with the same position are shared
2) If one section has uv coordinates, then all faces will have uv faces. The faces that have no uvs, will just index uvs (0, 0, 0)
3) Non position indices are shared in the same way as original data set (ie if uv coordinates are the same they will not be shared), but if the indices are shared
   in the original data they will be shared here.
   Note: This sharing is done at the mesh section level. If multiple mesh sections index a single vertex buffer, and those indices are shared
         via another mesh section, they will not be shared here (they will be duplicated). The algorithm could be made to do so, but adds extra complexity
4) The algorithm assumes nothing about the layout of vertex buffers/sharing etc

*/


struct ChannelData
{
    hkArray<hkVector4>  m_vertexValues;
    hkArray<int>        m_triangleIndices;
};


struct SectionData
{
    hkArray<ChannelData> m_channels;
};


/*static*/ int hctModelerMeshBuilderUtilities::prepareMeshSectionLockSet(const hkMeshBody& meshBody, hkMeshSectionLockSet* lockSet)
{
    const hkMeshShape* meshShape = meshBody.getMeshShape();

    if ( !meshShape || (meshShape->getNumSections() < 1) )
    {
        return 0;
    }

    // Lock access to all of the mesh sections.
    lockSet->addMeshSections(meshShape, hkMeshShape::ACCESS_INDICES | hkMeshShape::ACCESS_VERTEX_BUFFER);

    //
    // Remove all sections that don't contain triangles.
    //
    for (int i = 0; i < lockSet->getNumSections(); i++)
    {
        const hkMeshSection& section = lockSet->getSection(i);
        hkMeshPrimitiveUtil::PrimitiveStyle style = hkMeshPrimitiveUtil::getPrimitiveStyle(section.m_primitiveType);
        if ( (style != hkMeshPrimitiveUtil::PRIMITIVE_STYLE_TRIANGLE) || (section.m_numPrimitives <= 0) )
        {
            lockSet->removeSectionAtIndex(i);
            i--;
        }
    }

    const int numSections = lockSet->getNumSections();
    return numSections;
}


static void _findUsedIndices(const hkMeshSection& section, hkArray<int>& mapSectionIndicesToUsedIndices, hkArray<int>& usedIndices)
{
    hkMeshVertexBuffer* vertexBuffer = section.m_vertexBuffer;

    usedIndices.clear();

    const int numVertices = vertexBuffer->getNumVertices();

    mapSectionIndicesToUsedIndices.clear();
    mapSectionIndicesToUsedIndices.setSize(numVertices, -1);

    switch ( section.m_indexType )
    {
        case hkMeshSection::INDEX_TYPE_UINT16:
        {
            const hkUint16* indices = (const hkUint16*)section.m_indices;

            for (int i = 0; i < section.m_numIndices; i++)
            {
                const int index = int(indices[i]);
                if ( mapSectionIndicesToUsedIndices[index] < 0 )
                {
                    mapSectionIndicesToUsedIndices[index] = usedIndices.getSize();
                    usedIndices.pushBack(index);
                }
            }
            break;
        }
        case hkMeshSection::INDEX_TYPE_UINT32:
        {
            const hkUint32* indices = (const hkUint32*)section.m_indices;

            for (int i = 0; i < section.m_numIndices; i++)
            {
                const int index = int(indices[i]);
                if ( mapSectionIndicesToUsedIndices[index] < 0 )
                {
                    mapSectionIndicesToUsedIndices[index] = usedIndices.getSize();
                    usedIndices.pushBack(index);
                }
            }
            break;
        }
    }
}


// Note that this function is only called for secondary vertex data like e.g. vertex color, UV coordinates or vertex normal and not for the vertex's actual position!
static void _extractVertexValuesAndRemapTriangleIndices(const hkMeshSection& section, hkVertexFormat::ComponentUsage usage, int subUsage, const hkArray<int>& mapSectionIndicesToUsedIndices, const hkArray<hkUint32>& originalTriangleIndices, const hkArray<int>& usedIndices, ChannelData& channelData)
{
    hkMeshVertexBuffer* vertexBuffer = section.m_vertexBuffer;

    hkVertexFormat format;
    vertexBuffer->getVertexFormat(format);

    if ( format.findElementIndex(usage, subUsage) >= 0 )
    {
        // Retrieve the actual vertex values.
        hkArray<hkVector4> values;
        {
            HK_ON_DEBUG(hkResult res =) hkMeshVertexBufferUtil::getElementVectorArray(vertexBuffer, usage, subUsage, values);
            HK_ASSERT_NO_MSG(0x23423432, res.isSuccess());
        }

        // Save the base index for *this* section in the current channel before we start appending new values (as all sections
        // get flattened on a per-channel basis).
        const int baseIndex = channelData.m_vertexValues.getSize();

        // Append the actual vertex values (e.g. RBG color, UVs values, normal direction) as they are to the channel's value storage array.
        {
            const int numUsedIndices = usedIndices.getSize();
            hkVector4* dst = channelData.m_vertexValues.expandBy(numUsedIndices);
            for (int i = 0; i < numUsedIndices; i++)
            {
                dst[i] = values[usedIndices[i]];
            }
        }

        // Append the triangle indices to the channel's triangle indices array. The indices will get remapped to the usedIndices array first.
        // Also the flattening of all sections on a per-channel basis is taken into account by adding the current section's base index).
        {
            const int numTriangleIndices = originalTriangleIndices.getSize();
            int* dst = channelData.m_triangleIndices.expandBy(numTriangleIndices);
            for (int i = 0; i < numTriangleIndices; i++)
            {
                dst[i] = baseIndex + mapSectionIndicesToUsedIndices[originalTriangleIndices[i]];
            }
        }
    }
}


/*static*/ hkResult hctModelerMeshBuilderUtilities::convertHavokMeshToModelerNode(const hkMeshBody&           meshBody,
                                                                                  const hkMeshSectionLockSet& lockSet,
                                                                                  hctModelerMeshBuilder&      modelerMeshBuilder)
{
    const int                   numSections         = lockSet.getNumSections();

    // We will create at least one Color Channel and one UV channel (required for 3dsMax).
    // If there's no respective data available in the vertex buffer then these default channels
    // will automatically be filled with color WHITE / UV (0,0,0).
    int                         numColorChannels    = 1;
    int                         numUvChannels       = 1;

    hkFindUniquePositionsUtil   positionUtil;               // This utility is used to find (and share) identical vertices.

    hkArray<SectionData>        colorChannelSections;       // Array is built like that: section[channel[colors+triangleIndices]
    hkArray<SectionData>        uvChannelSections;          // Array is built like that: section[channel[uvs+triangleIndices]
    hkArray<SectionData>        normalChannelSections;      // Array is built like that: section[1[normals+triangleIndices]

    ChannelData                 normals;

    hkArray<int>                triangleIndices;            // Holds the indices into the list of (shared!) vertices for each triangle. Format is: {vi_0, vi_1, vi_2} consecutively for each triangle.
    hkArray<int>                numTrianglesPerSection;     // Holds the number of triangles for each section.

    //
    // Retrieve data from hkMeshShape.
    //
    {
        //
        // Extract vertex/color/uv/normal information for all sections in this hkMeshShape.
        //
        for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
        {
            const hkMeshSection& section = lockSet.getSection(sectionIndex);

            // This array holds the indices into the (unshared, original) vertex list for each triangle in the
            // *current* section. Format is: {vi_0, vi_1, vi_2} consecutively for each triangle.
            // Note that if we come across degenerate triangles they will get pruned from this array before
            // actual data (like colors, UVs or normals) is extracted from the mesh.
            hkArray<hkUint32> originalTriangleIndices;

            // This array holds the indices into the (unshared) vertex list for all vertices used by this section.
            hkArray<int> usedIndices;

            // This map serves 2 purposes:
            // a) it assists the creation of the usedIndices array where every index is unique (i.e. showing up only
            //    once in case it would show up several times in the original section indices list)
            // b) it maps from the original section indices to the usedIndices list
            hkArray<int> mapSectionIndicesToUsedIndices;

            //
            // Extract mesh vertices for this section.
            //
            {
                hkMeshVertexBuffer* vertexBuffer = section.m_vertexBuffer;

                // Get the used vertex indices.
                _findUsedIndices(section, mapSectionIndicesToUsedIndices, usedIndices);

                // Get the vertex positions.
                hkArray<hkVector4>  vertices;
                HK_ON_DEBUG(hkResult res =) hkMeshVertexBufferUtil::getElementVectorArray(vertexBuffer, hkVertexFormat::USAGE_POSITION, 0, vertices);
                HK_ASSERT_NO_MSG(0x23432432, res.isSuccess());

                const int numUsedIndices = usedIndices.getSize();

                // This array maps from the unshared vertex indices to the shared vertex indices.
                hkArray<int> mapUnsharedToSharedVertexIndex(numUsedIndices);

                // NOTE: Don't call positionUtil.reset() as we want to share vertices between mesh sections.

                for (int j = 0; j < numUsedIndices; j++)
                {
                    const int index = usedIndices[j];
                    mapUnsharedToSharedVertexIndex[j] = positionUtil.addPosition(vertices[index]);
                }

                // Get the vertex indices (into the original, unshared vertex list) for all triangles in the current section.
                hkMeshPrimitiveUtil::appendTriangleIndices(section, originalTriangleIndices);

                // Map the original vertex indices of all triangles to the shared vertices. Also prune degenerate triangles.
                int numTrianglesInThisSection = 0;
                {
                    const int numTriangleIndices = originalTriangleIndices.getSize();

                    hkArray<int> indicesBuffer;
                    indicesBuffer.reserve(numTriangleIndices);

                    for (int triangleId = (numTriangleIndices/3)-1; triangleId >= 0; triangleId--)
                    {
                        int vertex0 = mapUnsharedToSharedVertexIndex[mapSectionIndicesToUsedIndices[originalTriangleIndices[triangleId*3+0]]];
                        int vertex1 = mapUnsharedToSharedVertexIndex[mapSectionIndicesToUsedIndices[originalTriangleIndices[triangleId*3+1]]];
                        int vertex2 = mapUnsharedToSharedVertexIndex[mapSectionIndicesToUsedIndices[originalTriangleIndices[triangleId*3+2]]];

                        if ( (vertex0 == vertex1) || (vertex1 == vertex2) || (vertex0 == vertex2) )
                        {
                            // Remove the degenerate triangle without destroying the triangle order
                            // of the triangles that follow behind (as we are traversing from back
                            // to front)
                            originalTriangleIndices.removeAtAndCopy((triangleId*3)+2);
                            originalTriangleIndices.removeAtAndCopy((triangleId*3)+1);
                            originalTriangleIndices.removeAtAndCopy((triangleId*3)+0);
                        }
                        else
                        {
                            // We need to insert the vertex indices in reverse order as they will get copied below
                            // in reverse order as well (restoring the original order).
                            indicesBuffer.pushBack(vertex2);
                            indicesBuffer.pushBack(vertex1);
                            indicesBuffer.pushBack(vertex0);
                            numTrianglesInThisSection++;
                        }
                    }

                    // Copy the vertex indices in reverse order from temporary storage array into their
                    // final destination array.
                    for (int index = indicesBuffer.getSize()-1; index >= 0; index--)
                    {
                        triangleIndices.pushBack(indicesBuffer[index]);
                    }
                }

                numTrianglesPerSection.pushBack(numTrianglesInThisSection);
            }

            //
            // Extract Vertex Colors for all channels on this section.
            //
            {
                int numVertexColorChannelsForThisSection;
                {
                    hkVertexFormat format;
                    section.m_vertexBuffer->getVertexFormat(format);
                    numVertexColorChannelsForThisSection = format.findNextSubUsage(hkVertexFormat::USAGE_COLOR);
                }

                // Store the maximum number of Vertex Color channels.
                numColorChannels = hkMath::max2(numColorChannels, numVertexColorChannelsForThisSection);

                SectionData& colorChannelSection = colorChannelSections.expandOne();

                for (int channelIndex = 0; channelIndex < numVertexColorChannelsForThisSection; channelIndex++)
                {
                    ChannelData& colorChannel = colorChannelSection.m_channels.expandOne();
                    _extractVertexValuesAndRemapTriangleIndices(section, hkVertexFormat::USAGE_COLOR, channelIndex, mapSectionIndicesToUsedIndices, originalTriangleIndices, usedIndices, colorChannel);
                }
            }

            //
            // Extract UV vertices for all channels on this section.
            //
            {
                int numUvChannelsForThisSection;
                {
                    hkVertexFormat format;
                    section.m_vertexBuffer->getVertexFormat(format);
                    numUvChannelsForThisSection = format.findNextSubUsage(hkVertexFormat::USAGE_TEX_COORD);
                }

                // Store the maximum number of map channels.
                numUvChannels = hkMath::max2(numUvChannels, numUvChannelsForThisSection);

                SectionData& uvChannelSection = uvChannelSections.expandOne();

                for (int channelIndex = 0; channelIndex < numUvChannelsForThisSection; channelIndex++)
                {
                    ChannelData& uvChannel = uvChannelSection.m_channels.expandOne();
                    _extractVertexValuesAndRemapTriangleIndices(section, hkVertexFormat::USAGE_TEX_COORD, channelIndex, mapSectionIndicesToUsedIndices, originalTriangleIndices, usedIndices, uvChannel);
                }
            }

            //
            // Extract normals for this section.
            //
            {
                SectionData& normalChannelSection = normalChannelSections.expandOne();
                ChannelData& normalChannel        = normalChannelSection.m_channels.expandOne();

                _extractVertexValuesAndRemapTriangleIndices(section, hkVertexFormat::USAGE_NORMAL, 0, mapSectionIndicesToUsedIndices, originalTriangleIndices, usedIndices, normals);
                _extractVertexValuesAndRemapTriangleIndices(section, hkVertexFormat::USAGE_NORMAL, 0, mapSectionIndicesToUsedIndices, originalTriangleIndices, usedIndices, normalChannel);

//              // Check that they all match up in terms of amount of indices... else we can't do this
//              if ( (normalChannel.m_triangleIndices.getSize() > 0) && (normalChannel.m_triangleIndices.getSize() != triangleIndices.getSize()) )
//              {
//                  HK_ASSERT_NO_MSG(0x324234, !"Can only convert if each mesh section has the same combination of vertices/normals");
//                  return HK_FAILURE;
//              }
            }
        }
    }

    //
    // Create the actual Mesh from vertices and triangles.
    //
    {
        for (int v = 0; v < positionUtil.m_positions.getSize(); v++)
        {
            modelerMeshBuilder.addMeshVertexDelayed(positionUtil.m_positions[v]);
        }

        for (int t = 0; t < (triangleIndices.getSize() / 3); t++)
        {
            modelerMeshBuilder.addMeshTriangleDelayed(triangleIndices[t*3+0], triangleIndices[t*3+1], triangleIndices[t*3+2]);
        }

        modelerMeshBuilder.realizeDelayedMesh();
    }

    //
    // Setup the Vertex UV Channels.
    //
    // The triangles in the Vertex UV channels have to match the mesh triangles 1:1, so we will simply loop over all sections and recreate
    // each section's triangle list. All sections will be 'flattened' on a 'per map channel' basis, i.e. for each channel all sections will
    // be concatenated.
    //
    {
        for (int channelIndex = 0; channelIndex < numUvChannels; channelIndex++)
        {
            modelerMeshBuilder.selectUvChannel(channelIndex);

            // Always create one default (white) Vertex Color to be used as fallback for those sections
            // that don't have an explicit Vertex Color channel.
            modelerMeshBuilder.addVertexUvDelayed(1, 1);

            // All Vertex UVs of all mesh sections get concatenated (flattened) into the Vertex UV channel, so
            // whenever we want to reference a Vertex UV in a channel we need to offset its index by the number of
            // all preceding sections' UVs (in this particular channel). As we always have one default UV value
            // (0.0, 0.0) at the channel's start, this value has to be initialized to 1.
            int uvBase = 1;

            for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
            {
                SectionData& uvChannelSection = uvChannelSections[sectionIndex];

                // Create UV Vertices for all sections that use the current map channel.
                if ( channelIndex < uvChannelSection.m_channels.getSize() )
                {
                    hkArray<hkVector4>& uvs = uvChannelSection.m_channels[channelIndex].m_vertexValues;
                    for (int uvIndex = 0; uvIndex < uvs.getSize(); uvIndex++)
                    {
                        const hkVector4& uv = uvs[uvIndex];
                        modelerMeshBuilder.addVertexUvDelayed((float)uv(0), (float)uv(1));
                    }
                }

                for (int triangleIndex = 0; triangleIndex < numTrianglesPerSection[sectionIndex]; triangleIndex++)
                {
                    // Default if no such channel is available for this section.
                    int uvIndex[3] = { 0, 0, 0 };

                    // Extract triangle data if the current UV channel is available for this section and if it is not empty.
                    if ( channelIndex < uvChannelSection.m_channels.getSize() )
                    {
                        ChannelData& channelData = uvChannelSection.m_channels[channelIndex];
                        if ( !channelData.m_vertexValues.isEmpty() )
                        {
                            // We have a channel and it is properly filled.
                            const int* triangleUvIndices = &channelData.m_triangleIndices[triangleIndex*3];
                            uvIndex[0] = uvBase + triangleUvIndices[0];
                            uvIndex[1] = uvBase + triangleUvIndices[1];
                            uvIndex[2] = uvBase + triangleUvIndices[2];
                        }
                    }

                    // Create the actual map triangle.
                    modelerMeshBuilder.addVertexUvTriangleDelayed(uvIndex[0], uvIndex[1], uvIndex[2]);
                }

                // Advance the offset into the flattened UV vertices list.
                if ( channelIndex < uvChannelSection.m_channels.getSize() )
                {
                    uvBase += uvChannelSection.m_channels[channelIndex].m_vertexValues.getSize();
                }
            }

            modelerMeshBuilder.realizeDelayedVertexUvs();
            modelerMeshBuilder.realizeDelayedVertexUvTriangles();
        }
    }

    //
    // Setup the Vertex Color Channels.
    //
    // The triangles in the Vertex Color channels have to match the mesh triangles 1:1, so we will simply loop over all sections and recreate
    // each section's triangle list. All sections will be 'flattened' on a 'per map channel' basis, i.e. for each channel all sections will
    // be concatenated.
    //
    {
        for (int channelIndex = 0; channelIndex < numColorChannels; channelIndex++)
        {
            modelerMeshBuilder.selectColorChannel(channelIndex);

            // Always create one default (white) Vertex Color to be used as fallback for those sections
            // that don't have an explicit Vertex Color channel.
            modelerMeshBuilder.addVertexColorDelayed(1, 1, 1);

            // All Vertex Colors of all mesh sections get concatenated (flattened) into the Vertex Color channel, so
            // whenever we want to reference a Vertex Color in a channel we need to offset its index by the number of
            // all preceding sections' colors (in this particular channel). As we always have one default (white)
            // color at the channel's start, this value has to be initialized to 1.
            int colorBase = 1;

            for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
            {
                SectionData& colorChannelSection = colorChannelSections[sectionIndex];

                if ( channelIndex < colorChannelSection.m_channels.getSize() )
                {
                    hkArray<hkVector4>& colors = colorChannelSection.m_channels[channelIndex].m_vertexValues;
                    for (int colorIndex = 0; colorIndex < colors.getSize(); colorIndex++)
                    {
                        const hkVector4& colorBGRA = colors[colorIndex];
                        modelerMeshBuilder.addVertexColorDelayed((float)colorBGRA(2), (float)colorBGRA(1), (float)colorBGRA(0));
                    }
                }

                for (int triangleIndex = 0; triangleIndex < numTrianglesPerSection[sectionIndex]; triangleIndex++)
                {
                    // Default to the first (white) Vertex Color in the Vertex Color channel in case section does not have a dedicated Color channel of its own.
                    int colorIndex[3] = { 0, 0, 0 };

                    // Extract triangle data if the current Color channel is available for this section and if it is not empty.
                    if ( channelIndex < colorChannelSection.m_channels.getSize() )
                    {
                        ChannelData& channelData = colorChannelSection.m_channels[channelIndex];
                        if ( !channelData.m_vertexValues.isEmpty() )
                        {
                            // We have a channel and it is properly filled.
                            const int* triangleColorIndices = &channelData.m_triangleIndices[triangleIndex*3];
                            colorIndex[0] = colorBase + triangleColorIndices[0];
                            colorIndex[1] = colorBase + triangleColorIndices[1];
                            colorIndex[2] = colorBase + triangleColorIndices[2];
                        }
                    }

                    // Create the actual Vertex Color triangle.
                    modelerMeshBuilder.addVertexColorTriangleDelayed(colorIndex[0], colorIndex[1], colorIndex[2]);
                }

                // Advance the offset into the (flattened) Vertex Color list if the current channel was defined for this section.
                if ( channelIndex < colorChannelSection.m_channels.getSize() )
                {
                    colorBase += colorChannelSection.m_channels[channelIndex].m_vertexValues.getSize();
                }
            }

            // Realize all delayed operations.
            modelerMeshBuilder.realizeDelayedVertexColors();
            modelerMeshBuilder.realizeDelayedVertexColorTriangles();
        }
    }

    //
    // Setup normals.
    //
    {
        // All normals of all mesh sections get concatenated (flattened), so whenever we want to reference
        // a normal we need to offset its index by the number of all preceding sections' normals.
        int normalBase = 0;

        {
            for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
            {
                SectionData& normalChannelSection = normalChannelSections[sectionIndex];
                ChannelData& normalChannel        = normalChannelSection.m_channels[0]; // Currently there's exactly one normal channel.

                if (!normalChannel.m_vertexValues.isEmpty())
                {
                    for (int normalIndex = 0; normalIndex < normalChannel.m_vertexValues.getSize(); normalIndex++)
                    {
                        modelerMeshBuilder.addNormalDelayed(normalChannel.m_vertexValues[normalIndex]);
                    }

                    for (int triangleIndex = 0; triangleIndex < numTrianglesPerSection[sectionIndex]; triangleIndex++)
                    {
                        int normalId0 = normalBase + normalChannel.m_triangleIndices[(triangleIndex*3)+0];
                        int normalId1 = normalBase + normalChannel.m_triangleIndices[(triangleIndex*3)+1];
                        int normalId2 = normalBase + normalChannel.m_triangleIndices[(triangleIndex*3)+2];
                        modelerMeshBuilder.addNormalTriangleDelayed(normalId0, normalId1, normalId2);
                    }

                    normalBase += normalChannel.m_vertexValues.getSize();
                }
            }

            modelerMeshBuilder.realizeDelayedNormals();
        }
    }

    //
    // Assign materials to triangles.
    //
    {
        int globalTriangleIndex = 0; // i.e. mesh 'global'

        for (int sectionIndex = 0; sectionIndex < numSections; sectionIndex++)
        {
            const hkMeshSection& meshSection = lockSet.getSection(sectionIndex);
            const char* materialName = meshSection.m_material ? meshSection.m_material->getName() : HK_NULL;

            for (int triangleIndex = 0; triangleIndex < numTrianglesPerSection[sectionIndex]; triangleIndex++)
            {
                modelerMeshBuilder.setTriangleMaterialDelayed(globalTriangleIndex, materialName);
                globalTriangleIndex++;
            }
        }

        modelerMeshBuilder.realizeDelayedTriangleMaterials();
    }

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
