// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/SceneExport/hctSceneExport.h> // PCH
#include <ContentTools/Common/SceneExport/Filters/hctFilterProcessingUtil.h>
#include <ContentTools/Common/SceneExport/Memory/hctSceneExportMemory.h>
#include <ContentTools/Common/SceneExport/Error/hctSceneExportError.h>

#include <Common/Base/hkBase.h>
#include <Common/Base/System/hkBaseSystem.h>

#include <Common/Serialize/Util/hkRootLevelContainer.h>

#include <stdlib.h> // for getenv


hctFilterProcessingUtil::hctFilterProcessingUtil(void* ownerWindowHandle, hkBool interactive, const char* logFile, unsigned int logVerboseLevel, bool stdOuput)
:
    m_filterManagerDll(HK_NULL),
    m_filterManager(HK_NULL),
    m_interactive(interactive),
    m_logFileName(logFile),
    m_logVerboseLevel(logVerboseLevel),
    m_stdOutput(stdOuput)
{
}

hctFilterManagerInterface* hctFilterProcessingUtil::getFilterManagerInterface()
{
    return m_filterManager;
}

const char* hctFilterProcessingUtil::getFilterManagerPath() const
{
    return m_filterManagerPath.cString();
}

/// Get the path used to load attribute descriptions (for processing)
const char* hctFilterProcessingUtil::getAttributeProcessingPath() const
{
    m_attributeProcessingPath = m_filterManagerPath+"\\attributeProcessing";
    return m_attributeProcessingPath.cString();
}

/// Get the path used to load attribute selections
const char* hctFilterProcessingUtil::getAttributeSelectionPath() const
{
    m_attributeSelectionPath = m_filterManagerPath+"\\attributeSelection";
    return m_attributeSelectionPath.cString();
}

#include <ContentTools/Common/Filters/Common/FilterManager/hctFilterManagerInterface.h>

hkBool hctFilterProcessingUtil::load( const char* havokPath )
{
    if (m_filterManagerDll)
    {
        unload();
    }

    hctFilterProcessingUtil::waitOnBackgroundFilterLoad();

    m_filterManagerPath = havokPath;
    hkStringOld managerPath = m_filterManagerPath + "\\hctFilterManager.dll";
    HMODULE dllHandle = LoadLibrary( managerPath.cString() );

    // If we've found the DLL we can use it.
    if (dllHandle)
    {
        hkGetFilterManagerDllFunc getFilterManagerDllFunc = (hkGetFilterManagerDllFunc) GetProcAddress( dllHandle, "getFilterManagerDll");

        if( getFilterManagerDllFunc )
        {
            m_filterManagerDll = getFilterManagerDllFunc (dllHandle);

            unsigned int dllVersion = m_filterManagerDll->getDllVersion();
            unsigned int currentDllVersion = hctBaseDll::getCurrentDllVersion();

            if ( HCT_IS_MINOR_COMPATIBLE( currentDllVersion, dllVersion ) ) // same version (ignoring patch)
            {
                // Init the shared mem system.
                hkMemoryInitUtil::SyncInfo baseSystemSyncInfo;
                hkSceneExportMemory::getBaseSystemSyncInfo(baseSystemSyncInfo);

                m_filterManagerDll->initDll(baseSystemSyncInfo, HK_NULL );

                // Create a manager
                // Pass the same init info directlty to filters too (so that they see the exact same singleton list)
                m_filterManager = m_filterManagerDll->createFilterManager(&baseSystemSyncInfo, m_interactive);

                if (m_filterManager)
                {
                    m_filterManager->getErrorHandler()->setLogFile(m_logFileName);
                    m_filterManager->getErrorHandler()->setLogVerboseLevel(m_logVerboseLevel);
                    m_filterManager->getErrorHandler()->setStdOutput(m_stdOutput);
                    return true;
                }
            }
        }

        unload();

        MessageBox( NULL, "The Havok Filter Manager DLL could not be loaded.\nYou may need to reinstall the Havok Content Tools package.\0", "Warning: Load failed", MB_OK | MB_ICONWARNING | MB_TASKMODAL | MB_TOPMOST );
    }
    else
    {
        char buf[500];
        hkString::sprintf( buf, "The Havok Filter Manager DLL was not found in the expected location:\n%s.\n\nYou may need to reinstall the Havok Content Tools package.\0", havokPath );
        MessageBox( NULL, buf, "Warning: Missing DLL", MB_OK | MB_ICONWARNING | MB_TASKMODAL | MB_TOPMOST );
    }

    return false;
}

hkBool hctFilterProcessingUtil::unload()
{
    hctFilterProcessingUtil::waitOnBackgroundFilterLoad();

    if (m_filterManager)
    {
        m_filterManager->getErrorHandler()->setLogVerboseLevel(0);
        m_filterManagerDll->deleteFilterManager(m_filterManager);
        m_filterManager = HK_NULL;
    }

    // Release the manager DLL
    if (m_filterManagerDll)
    {
        m_filterManagerDll->quitDll();

        // We have unloaded dlls that would have registered classes with this dll, so have to wipe them:
        //hkBuiltinTypeRegistry::getInstance().reinitialize();

        FreeLibrary( m_filterManagerDll->getDllModule() );

        m_filterManagerDll = HK_NULL;
    }

    return true;
}


void hctFilterProcessingUtil::openFilterManager(hkRootLevelContainer& data, hkBool& shouldSaveConfigOut)
{
    if ( m_filterManager )
    {
        // EXP-841
        registerDefaultThreadCallback();

        m_filterManager->openFilterManager( (HWND) m_ownerHwnd, data, shouldSaveConfigOut );
    }
    else
    {
        shouldSaveConfigOut = false;
    }
}

void hctFilterProcessingUtil::processBatch( hkRootLevelContainer& data, hkBool& configModifiedOut, int configToRun, bool allowModelessFilters )
{
    if (!m_filterManager)
    {
        return;
    }

    // EXP-841
    registerDefaultThreadCallback();

    m_filterManager->processBatch( (HWND)m_ownerHwnd, data, configModifiedOut, configToRun, allowModelessFilters );

    // no processes should still be active at this stage
}


hkRootLevelContainer* hctFilterProcessingUtil::processBatchReturnData( hkRootLevelContainer& data, int configToRun, bool allowModelessFilters, hkResource*& sceneCopyStorage )
{
    if (!m_filterManager )
    {
        return HK_NULL;
    }

    // EXP-841
    registerDefaultThreadCallback();

    return m_filterManager->processBatchReturnData( (HWND)m_ownerHwnd, data, configToRun, allowModelessFilters, sceneCopyStorage );
    // no processes should still be active at this stage
}

void hctFilterProcessingUtil::getFileDependencies(hkRootLevelContainer& data, hkArray<hkStringPtr>& fileDependencies, int configToRun)
{
    if (!m_filterManager)
    {
        return;
    }

    // EXP-841
    registerDefaultThreadCallback();

    m_filterManager->getFileDependencies((HWND)m_ownerHwnd, data, fileDependencies, configToRun);

    // no processes should still be active at this stage
}

unsigned int hctFilterProcessingUtil::getCurrentOptionsVersion () const
{
    if (m_filterManager)
    {
        return m_filterManager->getFilterManagerVersion();
    }
    return 0;
}

int hctFilterProcessingUtil::getOptionsSize() const
{
    if (m_filterManager)
    {
        return m_filterManager->getConfigurationSet( HK_NULL );
    }
    return 0;
}

void hctFilterProcessingUtil::getOptions(void* buf) const
{
    if (m_filterManager)
    {
        m_filterManager->getConfigurationSet(buf);
    }
}

void hctFilterProcessingUtil::setOptions(const void* buf, int bufSize)
{
    if (m_filterManager)
    {
        m_filterManager->setConfigurationSet( buf, bufSize );
    }
}

int hctFilterProcessingUtil::getResult() const
{
    if (m_filterManager)
    {
        return m_filterManager->getResult();
    }
    return -1;
}

void hctFilterProcessingUtil::registerThreadCallback( const hctFilterThreadCallback* cb )
{
    if (m_filterManager)
    {
        m_filterManager->registerThreadCallback(cb);
    }
}

void hctFilterProcessingUtil::mergeErrors (const hctSceneExportError* otherHandler)
{
    hctSceneExportError* handler = m_filterManager->getErrorHandler();
    if( handler )
    {
        handler->merge( otherHandler );
    }
}

class DefaultThreadCallback : public hctFilterThreadCallback
{
    public:

        /*virtual*/ void newThreadCreated (hkMemoryRouter* memRouter) const
        {
            if (memRouter)
            {
                hkBaseSystem::initThread(memRouter);
            }
            else
            {
                hkBaseSystem::quitThread();
            }
        }
};

// EXP-841
static DefaultThreadCallback g_defaultThreadCallback;
void hctFilterProcessingUtil::registerDefaultThreadCallback ()
{
    m_filterManager->registerThreadCallback(&g_defaultThreadCallback);
}

hctFilterThreadCallback* hctFilterProcessingUtil::getDefaultThreadCallback()
{
    return &g_defaultThreadCallback;
}

static HANDLE g_backgroundDllLoaderThread = HK_NULL;
#define MAX_FILTER_PATH_SIZE 4096
static char g_havokPath[MAX_FILTER_PATH_SIZE];

DWORD WINAPI _backgroundFilterLoadFunc( LPVOID lpParam )
{
    hkMemoryInitUtil::SyncInfo dummyBaseSystemInfo;
    ::memset( &dummyBaseSystemInfo, 0, sizeof(hkMemoryInitUtil::SyncInfo) );

    char dllName[MAX_FILTER_PATH_SIZE];

    // Search for any DLLs in the provided path
    ::strncpy( dllName, g_havokPath, MAX_FILTER_PATH_SIZE);
    ::strncat( dllName, "\\filters\\*.dll", MAX_FILTER_PATH_SIZE);
    dllName[MAX_FILTER_PATH_SIZE-1] = 0;

    WIN32_FIND_DATA FindFileData;
    HANDLE hFind = FindFirstFile( dllName, &FindFileData );
    while( hFind != INVALID_HANDLE_VALUE )
    {
        ::strncpy( dllName, g_havokPath, MAX_FILTER_PATH_SIZE);
        ::strncat( dllName, "\\filters\\", MAX_FILTER_PATH_SIZE);
        ::strncat( dllName, FindFileData.cFileName, MAX_FILTER_PATH_SIZE);
        dllName[MAX_FILTER_PATH_SIZE-1] = 0;

        // Load the DLL
        HMODULE filterDllModule = LoadLibrary( dllName);
        if( !filterDllModule )
        {
            goto nextFile;
        }

        // Require our exported function
        hkGetFilterDLLFunc getFilterDllFunc = (hkGetFilterDLLFunc) GetProcAddress( filterDllModule, "getFilterDll" );
        if( !getFilterDllFunc )
        {
            goto nextFile;
        }

        hctFilterDll* filterDll = getFilterDllFunc( filterDllModule );
        if( !filterDll )
        {
            goto nextFile;
        }

        // Require correct version
        if( filterDll->getDllVersion() != hctBaseDll::getCurrentDllVersion() )
        {
            goto nextFile;
        }

        // Require some filters
        if( filterDll->getNumberOfFilters() <= 0 )
        {
            goto nextFile;
        }

        // Init and keep the DLL
        filterDll->initDll( dummyBaseSystemInfo, HK_NULL );
        filterDll->quitDll();

        // never delete the filterdll, as they are normally statics in the dll, not allocated.


    nextFile:

        if (filterDllModule)
        {
            FreeLibrary( filterDllModule );
        }

        if( FindNextFile( hFind, &FindFileData ) == 0 )
        {
            break;
        }
    }

    FindClose( hFind );

    return 1;
}

/*static*/ void hctFilterProcessingUtil::startBackgroundFilterLoad(const char* havokPath)
{
    if (!g_backgroundDllLoaderThread)
    {
        // Create preview thread:
        DWORD dwThreadId;

        ::strncpy(g_havokPath, havokPath, MAX_FILTER_PATH_SIZE - 1);
        g_havokPath[MAX_FILTER_PATH_SIZE - 1] = 0;

        g_backgroundDllLoaderThread = CreateThread(
            NULL, // default security attributes
            0, // default stack size
            _backgroundFilterLoadFunc,
            (LPVOID)HK_NULL, // arg
            0, // default creation flags
            &dwThreadId); // returns the thread identifier
    }
}

/*static*/ void hctFilterProcessingUtil::waitOnBackgroundFilterLoad()
{
    DWORD ec;
    while ( g_backgroundDllLoaderThread && GetExitCodeThread(g_backgroundDllLoaderThread, &ec) && (ec == STILL_ACTIVE) )
    {
        Sleep(1);
    }

    if (g_backgroundDllLoaderThread )
    {
        CloseHandle(g_backgroundDllLoaderThread);
        g_backgroundDllLoaderThread = HK_NULL;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
