// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/SceneExport/hctSceneExport.h> // PCH
#include <ContentTools/Common/SceneExport/AttributeProcessing/hctAttributeProcessingUtil.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Container/String/hkUtf8.h>

#include <Common/Base/Config/hkConfigVersion.h>
#include <ContentTools/Common/SceneExport/AttributeProcessing/hctAttributeDescription.h>

#include <Common/SceneData/Attributes/hkxAttributeGroup.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/System/Io/OStream/hkOStream.h>
#include <Common/Base/Serialize/Format/Compat/hkCompatFormats.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.sceneexport"
#include <Common/Base/System/Log/hkLog.hxx>


hkBool hctAttributeProcessingUtil::init ()
{
    m_database.m_groupDescriptions.clear();

    return true;
}

hctAttributeProcessingUtil::~hctAttributeProcessingUtil()
{
}

hkBool hctAttributeProcessingUtil::isEmpty () const
{
    int numAttributesDescribed = 0;

    for (int gIt=0; gIt<m_database.m_groupDescriptions.getSize(); gIt++)
    {
        const hctAttributeGroupDescription& group = m_database.m_groupDescriptions[gIt];

        for (int aIt=0; aIt<group.m_attributeDescriptions.getSize(); aIt++)
        {
            numAttributesDescribed++;
        }
    }

    return (numAttributesDescribed == 0 );
}

hkBool hctAttributeProcessingUtil::loadAttributeDescriptions ( const char* attributeDescriptionPath )
{
    // look for all files in attributeDescriptionPath with starting with hct and ending in xml (hctAttributeProcessingUtil.xml is the default master desc)
    WIN32_FIND_DATA FindFileData;
    HANDLE hFindHandle;
    hkStringOld databasePath = attributeDescriptionPath ;
    databasePath += "\\hct*.xml";
    hFindHandle = FindFirstFile( TEXT( databasePath.cString() ), &FindFileData);
    bool finished = (hFindHandle == INVALID_HANDLE_VALUE);
    while( !finished )
    {
        hkStringOld xmlName = attributeDescriptionPath;
        xmlName += "\\";
        xmlName += FindFileData.cFileName;

        hkIstream file(xmlName.cString());
        if (file.isOk())
        {
            hkRefPtr<hkResource> loaded;

            // try using the compatibility dll
            if (hkSerializeUtil::FORMAT_PACKFILE_XML == hkSerializeUtil::detectFormat(file.getStreamReader()))
            {
                if (hkCompatFormats::getInstance().isLoaded())
                {
                    hkArray<char> buffer;
                    hkOstream ostr(buffer);
                    hkCompatFormats::getInstance().convertAttributeDescriptionOptions(file.getStreamReader(), ostr.getStreamWriter());
                    loaded = hkSerialize::Load().toResource(buffer.begin(), buffer.getSize());
                }
                else
                {
                    HK_WARN_ALWAYS(0xabba1458, "Attribute descriptions could not be loaded, a valid hkCompatFormats.dll was not found");
                }
            }
            else
            {
                loaded = hkSerialize::Load().toResource(file.getStreamReader());
            }

            if (loaded)
            {
                if (hctAttributeDescriptionDatabase* db = loaded->getContents<hctAttributeDescriptionDatabase>())
                {
                    mergeAttributeDescriptionDatabase(*db);
                }
            }
        }

        // get next xml file:
        if (FindNextFile( hFindHandle, &FindFileData ) == 0) // zero == error
        {
            finished = true;
            FindClose(hFindHandle);
        }
    }

    return true;
}

void hctAttributeProcessingUtil::mergeAttributeDescriptionDatabase (hctAttributeDescriptionDatabase& newDatabase)
{
    for (int i=0; i<newDatabase.m_groupDescriptions.getSize(); i++)
    {
        hctAttributeGroupDescription& newGroupDesc = newDatabase.m_groupDescriptions[i];

        bool found = false;

        for (int j=0; j<m_database.m_groupDescriptions.getSize(); j++)
        {
            hctAttributeGroupDescription& currentGroupDesc = m_database.m_groupDescriptions[j];

            if (hkString::strCmp(newGroupDesc.m_name, currentGroupDesc.m_name) == 0)
            {
                mergeAttributeGroupDescriptions(newGroupDesc, currentGroupDesc);
            }
        }

        if (!found)
        {
            // New attribute group description
            
            hctAttributeGroupDescription& currentGroupDesc = m_database.m_groupDescriptions.expandOne();
            currentGroupDesc.move(newGroupDesc);
        }
    }
}

void hctAttributeProcessingUtil::mergeAttributeGroupDescriptions (hctAttributeGroupDescription& newGroupDesc, hctAttributeGroupDescription& currentGroupDesc)
{
    for (int i=0; i<newGroupDesc.m_attributeDescriptions.getSize(); i++)
    {
        hctAttributeDescription& newAttDesc = newGroupDesc.m_attributeDescriptions[i];

        bool found = false;

        for (int j=0; j<currentGroupDesc.m_attributeDescriptions.getSize(); j++)
        {
            hctAttributeDescription& currentAttDesc = currentGroupDesc.m_attributeDescriptions[j];

            if (hkString::strCmp(newAttDesc.m_name, currentAttDesc.m_name) == 0)
            {
                mergeAttributeDescriptions(newAttDesc, currentAttDesc);
            }
        }

        if (!found)
        {
            // New attribute group description
            
            hctAttributeDescription& currentAttDesc = currentGroupDesc.m_attributeDescriptions.expandOne();
            currentAttDesc.move(newAttDesc);
        }
    }

}

void hctAttributeProcessingUtil::mergeAttributeDescriptions (hctAttributeDescription& newAttDesc, hctAttributeDescription& currentAttDesc)
{
    // Non-defaults win
    if ( currentAttDesc.m_enabledBy == HK_NULL)
    {
        currentAttDesc.m_enabledBy = newAttDesc.m_enabledBy;
    }

    if ( currentAttDesc.m_forcedType == hctAttributeDescription::LEAVE )
    {
        currentAttDesc.m_forcedType = newAttDesc.m_forcedType;
        currentAttDesc.m_enum.reset(newAttDesc.m_enum.steal());
    }

    if ( currentAttDesc.m_floatScale == 0.0f )
    {
        currentAttDesc.m_floatScale = newAttDesc.m_floatScale;
    }

    // We do a bitwise OR with the hint
    currentAttDesc.m_hint = (hctAttributeDescription::Hint) ( (int) currentAttDesc.m_hint | (int) newAttDesc.m_hint );
}


void hctAttributeProcessingUtil::processAttributeGroup (hkxAttributeGroup& attrGroup)
{
    const char* groupName = attrGroup.m_name;
    if( !groupName )
    {
        return;
    }

    // Get the attribute group description. Will be NULL if not present in the XML.
    const hctAttributeGroupDescription* attrGroupDesc = m_database.findAttributeDescriptionGroupByName( groupName );

    // 1 - Merge _X,_Y,_Z to vector
    mergeXYZToVector (attrGroup.m_attributes);

    // 2 - Scale floats
    scaleFloats (attrGroup.m_attributes, attrGroupDesc);

    // 3 - Use Force types
    enforceTypes (attrGroup.m_attributes, attrGroupDesc);

    // 4 - Merge _Trans,_Rot to matrix3  (doesn't depend on desc directly, but type may have changed in theory in prev pass)
    mergeTransAndRotToMatrix (attrGroup.m_attributes);

    // 5 - Check pairs of attributes ("name", "changeName").
    // Removes "name" depending on the value of "changeName". Always removes "changeName"
    enforceChangePairs (attrGroup.m_attributes);

    // 6 - Remove using "enabled by" property (deprecated by the above)
    enforceEnabledBy (attrGroup.m_attributes, attrGroupDesc);

    // 7 - Enforce Hints
    enforceHints (attrGroup.m_attributes, attrGroupDesc);

}


void hctAttributeProcessingUtil::scaleFloats (hkArray<hkxAttribute>& attributes, const struct hctAttributeGroupDescription* attrGroupDesc)
{
    if (!attrGroupDesc)
    {
        // No further instructions, leave it as it is
        return;
    }

    for( int i=0; i<attributes.getSize(); ++i )
    {
        // The attribute is not in the database? do nothing
        const hctAttributeDescription *attrDesc = attrGroupDesc->findAttributeDescriptionByName(attributes[i].m_name);
        if (!attrDesc)
        {
            continue;
        }

        // The scale is 0.0 (not defined) >? do nothing
        if (attrDesc->m_floatScale == 0.0f)
        {
            continue;
        }

        hkRefVariant& attr = attributes[i].m_value;

        // Float attributes : scale
        if (hkxAnimatedFloat* animatedFloat = hkDynCast<hkxAnimatedFloat>(attr) )
        {
            for (int f=0; f<animatedFloat->m_floats.getSize(); f++)
            {
                animatedFloat->m_floats[f] *= (float)attrDesc->m_floatScale;
            }
        }

        // Vector attributes (euler) : scale
        if (hkxAnimatedVector* animatedVector = hkDynCast<hkxAnimatedVector>(attr))
        {
            const hkSimdReal floatScale = hkSimdReal::fromFloat(attrDesc->m_floatScale);
            for (int v=0; v<animatedVector->m_vectors.getSize()/4; v++)
            {
                hkVector4 vec;
                vec.load<4,HK_IO_NATIVE_ALIGNED>(&animatedVector->m_vectors[4*v]);
                vec.mul(floatScale);
                vec.store<4,HK_IO_NATIVE_ALIGNED>(&animatedVector->m_vectors[4*v]);
            }
        }

    }
}

void hctAttributeProcessingUtil::mergeXYZToVector (hkArray<hkxAttribute>& attributes)
{
    for( int i=0; i<attributes.getSize(); ++i )
    {
        // Get base name if any (look for underscore)
        hkStringBuf basename( attributes[i].m_name );
        int li = basename.lastIndexOf( '_' );
        if( li == -1 ) continue;

        basename.slice(0, li);

        // Search for translation/rotation pairs
        int xIndex = -1;
        int yIndex = -1;
        int zIndex = -1;

        hkStringBuf x; x.setJoin(basename, "_X");
        hkStringBuf y; y.setJoin(basename, "_Y");
        hkStringBuf z; z.setJoin(basename, "_Z");

        for( int k=i; k<attributes.getSize(); ++k )
        {
            if( x.compareToIgnoreCase( attributes[k].m_name ) == 0 )
            {
                xIndex = k;
            }
            else if( y.compareToIgnoreCase( attributes[k].m_name ) == 0 )
            {
                yIndex = k;
            }
            else if ( z.compareToIgnoreCase(attributes[k].m_name) == 0)
            {
                zIndex = k;
            }
        }

        // One of the attributes should be this one; otherwise continue
        if ((xIndex!=i) && (yIndex!=i) && (zIndex!=i))
        {
            continue;
        }

        // Process the pairs
        if ( (xIndex>=0) && (yIndex>=0) && (zIndex>=0))
        {
            hkxAnimatedFloat* animX = hkDynCast<hkxAnimatedFloat>(attributes[xIndex].m_value);
            hkxAnimatedFloat* animY = hkDynCast<hkxAnimatedFloat>(attributes[yIndex].m_value);
            hkxAnimatedFloat* animZ = hkDynCast<hkxAnimatedFloat>(attributes[zIndex].m_value);

            // Check the types
            if (animX == HK_NULL )
            {
                Log_Warning( "Attribute with _X postfix should be of type Float" );
                continue;
            }

            if ( animY == HK_NULL )
            {
                Log_Warning( "Attribute with _Y postfix should be of type Float" );
                continue;
            }

            if ( animZ == HK_NULL )
            {
                Log_Warning( "Attribute with _Z postfix should be of type Float" );
                continue;
            }


            if ( ( animX->m_floats.getSize() != animY->m_floats.getSize() ) || ( animX->m_floats.getSize() != animZ->m_floats.getSize() ) )
            {
                Log_Warning( "Mismatched number of animated elements in x/y/z attributes" );
                continue;
            }

            // Build the animated matrix
            hkxAnimatedVector* animatedVector = new hkxAnimatedVector();
            {
                animatedVector->m_vectors.setSize(4*animX->m_floats.getSize());

                // Assume that if the floats were distances, this is a vector in space
                if ((animX->m_hint & hkxAttribute::HINT_SCALE) != 0)
                {
                    animatedVector->m_hint = hkxAttribute::HINT_TRANSFORM_AND_SCALE;
                }
                else
                {
                    animatedVector->m_hint = hkxAttribute::HINT_NONE;
                }

                for( int v=0; v<animatedVector->m_vectors.getSize()/4; ++v )
                {
                    animatedVector->m_vectors[4*v  ] = animX->m_floats[v];
                    animatedVector->m_vectors[4*v+1] = animY->m_floats[v];
                    animatedVector->m_vectors[4*v+2] = animZ->m_floats[v];
                    animatedVector->m_vectors[4*v+3] = 0;
                }
            }

            // Replace first hkx attribute with the vector
            {
                hkxAttribute& vectorAttribute = attributes[i];

                // Set the new name
                vectorAttribute.m_name = basename;

                // Set the type and object
                vectorAttribute.setValue( animatedVector );
                animatedVector->removeReference();
            }

            // Remove the other two attributes
            {
                // We are being overzealous and not assuming X,Y and Z are properly ordered...
                // We remove them backwards

                int rest1;
                int rest2;

                if ((zIndex<yIndex)&&(zIndex<xIndex))
                {
                    rest1 = xIndex; rest2 = yIndex;
                }
                else if ((yIndex<zIndex)&&(yIndex<xIndex))
                {
                    rest1 = zIndex; rest2 = xIndex;
                }
                else
                {
                    rest1 = yIndex; rest2 = zIndex;
                }

                if (rest1>rest2)
                {
                    attributes.removeAt(rest1);
                    attributes.removeAt(rest2);
                }
                else
                {
                    attributes.removeAt(rest2);
                    attributes.removeAt(rest1);
                }
            }
        }
    }
}

void hctAttributeProcessingUtil::mergeTransAndRotToMatrix(hkArray<hkxAttribute>& attributes)
{
    for( int i=0; i<attributes.getSize(); ++i )
    {
        // Get base name if any (loom for underscore)
        hkStringOld basename( attributes[i].m_name );
        int li = basename.lastIndexOf( '_' );
        if( li == -1 ) continue;

        basename = basename.substr(0, li);

        // Search for translation/rotation pairs
        int translationIndex = -1;
        int rotationIndex = -1;

        for( int k=i; k<attributes.getSize(); ++k )
        {
            if( ( basename + "_Trans" ).compareToIgnoreCase( attributes[k].m_name ) == 0 )
            {
                translationIndex = k;
            }
            else if( ( basename + "_Rot" ).compareToIgnoreCase( attributes[k].m_name ) == 0 )
            {
                rotationIndex = k;
            }
        }

        // One of the attributes should be this one; otherwise continue
        if ((translationIndex!=i) && (rotationIndex!=i))
        {
            continue;
        }

        // Process the pairs
        if ( (translationIndex>=0) && (rotationIndex>=0) )
        {
            hkxAnimatedVector* animTranslation = hkDynCast<hkxAnimatedVector>(attributes[translationIndex].m_value);
            hkxAnimatedQuaternion* animRotation = hkDynCast<hkxAnimatedQuaternion>(attributes[rotationIndex].m_value);
            // Check the types
            if ( animTranslation == HK_NULL )
            {
                Log_Warning( "Attribute with _Trans postfix should be of type Vector" );
                continue;
            }

            if ( animRotation == HK_NULL )
            {
                Log_Warning( "Attribute with _Rot postfix should be of type Vector. Use \"forceType\" to convert from Euler Angles." );
                continue;
            }

            if ( animTranslation->m_vectors.getSize() != animRotation->m_quaternions.getSize() )
            {
                Log_Warning( "Mismatched number of animated elements in trans/rot pair" );
                continue;
            }

            // Build the animated matrix
            hkxAnimatedMatrix* animatedMatrix = new hkxAnimatedMatrix();
            {
                animatedMatrix->m_matrices.setSize( 4 * animTranslation->m_vectors.getSize() );
                animatedMatrix->m_hint = hkxAttribute::HINT_TRANSFORM_AND_SCALE;
                for( int m=0; m<animatedMatrix->m_matrices.getSize()/16; ++m )
                {
                    hkVector4 translation; translation.load<4,HK_IO_NATIVE_ALIGNED>(&animTranslation->m_vectors[4*m]);
                    hkQuaternion rotation; rotation.m_vec.load<4,HK_IO_NATIVE_ALIGNED>(&animRotation->m_quaternions[4*m]);
                    hkQsTransform qsTransform(translation, rotation);
                    qsTransform.get4x4ColumnMajor(&animatedMatrix->m_matrices[16*m]);
                }
            }

            // Replace first hkx attribute with the matrix
            {
                hkxAttribute& matrixAttribute = attributes[i];

                // Set the new name
                matrixAttribute.m_name = basename.cString();

                matrixAttribute.setValue( animatedMatrix );
                animatedMatrix->removeReference();
            }

            // Remove the second attribute
            {
                int secondAttrIdx = (translationIndex==i) ? rotationIndex : translationIndex;
                attributes.removeAt(secondAttrIdx);
            }
        }
    }
}

inline int _findAttributeByName (const hkArray<hkxAttribute>& attributes, const char* name)
{
    for (int i=0; i<attributes.getSize(); ++i)
    {
        if (hkString::strCasecmp(attributes[i].m_name, name)==0)
        {
            return i;
        }
    }

    return -1;
}

inline const hkReflect::Type* _attrTypeToClass (hctAttributeDescription::ForcedType type)
{
    switch (type)
    {
        case hctAttributeDescription::FORCE_BOOL:
        {
            return hkReflect::getType<hkxSparselyAnimatedBool>();
        }
        case hctAttributeDescription::FORCE_INT:
        {
            return hkReflect::getType<hkxSparselyAnimatedInt>();
        }
        case hctAttributeDescription::FORCE_ENUM:
        {
            return hkReflect::getType<hkxSparselyAnimatedEnum>();
        }
        case hctAttributeDescription::FORCE_STRING:
        {
            return hkReflect::getType<hkxSparselyAnimatedString>();
        }
        case hctAttributeDescription::FORCE_FLOAT:
        {
            return hkReflect::getType<hkxAnimatedFloat>();
        }
        case hctAttributeDescription::FORCE_VECTOR:
        {
            return hkReflect::getType<hkxAnimatedVector>();
        }
        case hctAttributeDescription::FORCE_MATRIX:
        {
            return hkReflect::getType<hkxAnimatedMatrix>();
        }
        case hctAttributeDescription::FORCE_QUATERNION:
        {
            return hkReflect::getType<hkxAnimatedQuaternion>();
        }
        default: // LEAVE
        {
            return HK_NULL;
        }
    }
}

static hkxEnum* _getEnum( hkPointerMap<const hctAttributeDescription::Enum*, hkxEnum*>& m, const hctAttributeDescription::Enum* e)
{
    hkxEnum* r = m.getWithDefault(e, HK_NULL);
    if (!r)
    {
        r = new hkxEnum();
        r->m_items.setSize( e->getNumItems() );
        for (int ii=0; ii < e->getNumItems(); ++ii)
        {
            const hctAttributeDescription::Enum::Item& it = e->getItem(ii);
            hkxEnum::Item& ie = r->m_items[ii];
            ie.m_value = it.getValue();
            ie.m_name = it.getName();
        }
    }
    return r;
}
void hctAttributeProcessingUtil::enforceTypes (hkArray<hkxAttribute>& attributes, const hctAttributeGroupDescription* attrGroupDesc)
{
    if (!attrGroupDesc)
    {
        // No instructions, leave it as it is
        return;
    }

    for (int i=0; i<attrGroupDesc->m_attributeDescriptions.getSize(); ++i)
    {
        const hctAttributeDescription& attrDesc = attrGroupDesc->m_attributeDescriptions[i];

        const int attrIndex = _findAttributeByName (attributes, attrDesc.m_name);
        if (attrIndex<0)
        {
            continue;
        }

        const hkReflect::Type* currentAttrClass = attributes[attrIndex].m_value.getType();
        const hkReflect::Type* desiredAttrClass = _attrTypeToClass(attrDesc.m_forcedType);

        if (!desiredAttrClass || (currentAttrClass == desiredAttrClass))
        {
            continue;
        }

        // Supported conversions

        // Vector (EULER) to Quaternion
        if ( currentAttrClass->extendsOrEquals<hkxAnimatedVector>() && desiredAttrClass->extendsOrEquals<hkxAnimatedQuaternion>() )
        {
            hkxAnimatedVector* oldAttribute = (hkxAnimatedVector*) attributes[attrIndex].m_value.val();
            hkxAnimatedQuaternion* newAttribute = convertEulerToQuaternion (oldAttribute);

            attributes[attrIndex].m_value = newAttribute;
            newAttribute->removeReference();

            continue;
        }

        // Matrix to Quaternion
        if ( (currentAttrClass->extendsOrEquals<hkxAnimatedMatrix>()) && (desiredAttrClass->extendsOrEquals<hkxAnimatedQuaternion>()))
        {
            hkxAnimatedMatrix* oldAttribute = (hkxAnimatedMatrix*) attributes[attrIndex].m_value.val();
            hkxAnimatedQuaternion* newAttribute = convertMatrixToQuaternion (oldAttribute);

            attributes[attrIndex].m_value = newAttribute;
            newAttribute->removeReference();

            continue;
        }

        // Int to Enum
        if ( (currentAttrClass->extendsOrEquals<hkxSparselyAnimatedInt>()) && (desiredAttrClass->extendsOrEquals<hkxSparselyAnimatedEnum>()) && (attrDesc.m_enum!=HK_NULL))
        {
            hkxSparselyAnimatedInt* oldAttribute = (hkxSparselyAnimatedInt*) attributes[attrIndex].m_value.val();

            const hctAttributeDescription::Enum* enumDesc = attrDesc.m_enum.get();
            hkxEnum* exportEnum = _getEnum( m_enumMap, enumDesc );
            hkxSparselyAnimatedEnum* newAttribute = convertIntToEnum (oldAttribute, exportEnum);
            exportEnum->removeReference();

            attributes[attrIndex].m_value = newAttribute;
            newAttribute->removeReference();

            continue;
        }

        // Enum To String
        if ( (currentAttrClass->extendsOrEquals<hkxSparselyAnimatedEnum>()) && (desiredAttrClass->extendsOrEquals<hkxSparselyAnimatedString>()))
        {
            hkxSparselyAnimatedEnum* oldAttribute = (hkxSparselyAnimatedEnum*) attributes[attrIndex].m_value.val();
            hkxSparselyAnimatedString* newAttribute = convertEnumToString (oldAttribute);

            attributes[attrIndex].m_value = newAttribute;
            newAttribute->removeReference();

            continue;
        }

        // Int To String (requires enum definition)
        if ( (currentAttrClass->extendsOrEquals<hkxSparselyAnimatedInt>()) && (desiredAttrClass->extendsOrEquals<hkxSparselyAnimatedString>()) && (attrDesc.m_enum!=HK_NULL))
        {
            hkxSparselyAnimatedInt* oldAttribute = (hkxSparselyAnimatedInt*) attributes[attrIndex].m_value.val();
            const hctAttributeDescription::Enum* enumDesc = attrDesc.m_enum.get();
            hkxEnum* exportEnum =  _getEnum( m_enumMap, enumDesc );
            hkxSparselyAnimatedString* newAttribute = convertIntToString (oldAttribute, exportEnum);
            exportEnum->removeReference();

            attributes[attrIndex].m_value = newAttribute;
            newAttribute->removeReference();

            continue;
        }



        // Unsupported conversions
        // We leave Attribute untouched.. or should we remove it?
    }

}

void hctAttributeProcessingUtil::enforceChangePairs (hkArray<hkxAttribute>& attributes)
{
    for (int i=0; i<attributes.getSize(); ++i)
    {
        // Look for an attribute called changeNAME
        hkStringBuf changeName("change", attributes[i].m_name);
        int enablerAttrIndex = _findAttributeByName (attributes, changeName.cString());

        if (enablerAttrIndex<0)
        {
            // Not found? continue
            continue;
        }

        // Look whether we are enabling or not
        bool enabled = true;
        {
            if (hkxSparselyAnimatedBool* enablerBool = hkDynCast<hkxSparselyAnimatedBool>(attributes[enablerAttrIndex].m_value))
            {
                enabled = enablerBool->m_bools[0];
            }
            else if (hkxSparselyAnimatedInt* enablerInt = hkDynCast<hkxSparselyAnimatedInt>(attributes[enablerAttrIndex].m_value))
            {
                enabled = enablerInt->m_ints[0] != 0;
            }
        }

        // No matter what, remove the enabler ("changeXXXX") attribute
        {
            attributes.removeAtAndCopy(enablerAttrIndex);
            if (enablerAttrIndex<i)
            {
                --i;
            }
        }

        // And, if it was set to false, remove this attribute as well
        if (!enabled)
        {
            // removeAtAndCopy keeps the order in the array
            attributes.removeAtAndCopy(i);
            i--;
        }
    }
}

void hctAttributeProcessingUtil::enforceEnabledBy (hkArray<hkxAttribute>& attributes, const hctAttributeGroupDescription* attrGroupDesc)
{
    if (!attrGroupDesc)
    {
        // No further instructions, leave it as it is
        return;
    }

    for (int i=0; i<attrGroupDesc->m_attributeDescriptions.getSize(); ++i)
    {
        const hctAttributeDescription& attrDesc = attrGroupDesc->m_attributeDescriptions[i];

        if (attrDesc.m_enabledBy== HK_NULL)
        {
            continue;
        }

        const int enablerAttrIndex = _findAttributeByName (attributes, attrDesc.m_enabledBy);
        if (enablerAttrIndex<0)
        {
            continue;
        }

        // Look whether we are enabling or not
        bool enabled = true;
        {
            if (hkxSparselyAnimatedBool* enablerBool = hkDynCast<hkxSparselyAnimatedBool>(attributes[enablerAttrIndex].m_value))
            {
                enabled = enablerBool->m_bools[0];
            }
            else if (hkxSparselyAnimatedInt* enablerInt = hkDynCast<hkxSparselyAnimatedInt>(attributes[enablerAttrIndex].m_value))
            {
                enabled = enablerInt->m_ints[0] != 0;
            }
        }

        // If we are enabled, do nothing
        if (enabled)
        {
            continue;
        }

        // Otherwise remove the other attribute
        {
            const int attrIndex = _findAttributeByName (attributes, attrDesc.m_name);
            if (attrIndex<0)
            {
                continue;
            }

            // TODO : Destroy attribute data

            attributes.removeAtAndCopy(attrIndex);
            if(attrIndex <= i)
            {
                i--;
            }
        }

    }
}

void hctAttributeProcessingUtil::enforceHints (hkArray<hkxAttribute>& attributes, const hctAttributeGroupDescription* attrGroupDesc)
{
    if (!attrGroupDesc)
    {
        // No instructions, leave it as it is
        return;
    }

    for (int i=0; i<attrGroupDesc->m_attributeDescriptions.getSize(); ++i)
    {
        const hctAttributeDescription& attrDesc = attrGroupDesc->m_attributeDescriptions[i];

        const int attrIndex = _findAttributeByName (attributes, attrDesc.m_name);
        if (attrIndex<0)
        {
            continue;
        }

        hkRefVariant& attributeObject = attributes[attrIndex].m_value;

        if (attrDesc.m_clearHints)
        {
            if (hkxAnimatedFloat* a = hkDynCast<hkxAnimatedFloat>(attributeObject))
            {
                a->m_hint = hkxAttribute::HINT_NONE;
            }

            if (hkxAnimatedVector* a = hkDynCast<hkxAnimatedVector>(attributeObject))
            {
                a->m_hint = hkxAttribute::HINT_NONE;
            }

            if (hkxAnimatedMatrix* a = hkDynCast<hkxAnimatedMatrix>(attributeObject))
            {
                a->m_hint = hkxAttribute::HINT_NONE;
            }
        }

        hctAttributeDescription::Hint desiredHint = attrDesc.m_hint;

        if (desiredHint==hctAttributeDescription::HINT_IGNORE)
        {
            // Remove attribute according to hint
            attributes.removeAt(attrIndex);
            continue;
        }

        // Otherwise, we do an OR of the hint flags
        // We need to cast to the right type though...

        if (hkxAnimatedFloat* a = hkDynCast<hkxAnimatedFloat>(attributeObject))
        {
            (int&)a->m_hint |= (int) desiredHint;
            continue;
        }

        if (hkxAnimatedVector* a = hkDynCast<hkxAnimatedVector>(attributeObject))
        {
            (int&)a->m_hint |= (int) desiredHint;
            continue;
        }

        if (hkxAnimatedMatrix* a = hkDynCast<hkxAnimatedMatrix>(attributeObject))
        {
            (int&)a->m_hint |= (int) desiredHint;
            continue;
        }
    }
}

// Conversions
hkxAnimatedQuaternion* hctAttributeProcessingUtil::convertEulerToQuaternion(hkxAnimatedVector* animatedVector)
{
    hkxAnimatedQuaternion* animatedQuaternion = new hkxAnimatedQuaternion();

    const int numQuaternions = animatedVector->m_vectors.getSize()/4;
    animatedQuaternion->m_quaternions.setSize(4*numQuaternions);

    for (int i=0; i<numQuaternions; i++)
    {
        hkVector4 v;
        v.load<4,HK_IO_NATIVE_ALIGNED>(&animatedVector->m_vectors[4*i]);

        hkQuaternion qx; qx.setAxisAngle( hkVector4::getConstant<HK_QUADREAL_1000>(), v.getComponent<0>() );
        hkQuaternion qy; qy.setAxisAngle( hkVector4::getConstant<HK_QUADREAL_0100>(), v.getComponent<1>() );
        hkQuaternion qz; qz.setAxisAngle( hkVector4::getConstant<HK_QUADREAL_0010>(), v.getComponent<2>() );

        // Create composite quaternion
        hkQuaternion q;
        q.setMul( qz, qy );
        q.mul( qx );

        q.m_vec.store<4,HK_IO_NATIVE_ALIGNED>(&animatedQuaternion->m_quaternions[4*i]);
    }

    return animatedQuaternion;
}

hkxAnimatedQuaternion* hctAttributeProcessingUtil::convertMatrixToQuaternion (class hkxAnimatedMatrix* animatedMatrix)
{
    hkxAnimatedQuaternion* animatedQuaternion = new hkxAnimatedQuaternion();

    const int numQuaternions = animatedMatrix->m_matrices.getSize() >> 4;
    animatedQuaternion->m_quaternions.setSize(4*numQuaternions);

    for (int i=0; i<numQuaternions; i++)
    {
        hkMatrix4 mat; mat.set4x4ColumnMajor(&animatedMatrix->m_matrices[16*i]);

        hkRotation rot;
        rot.setCols(mat.getColumn<0>(),mat.getColumn<1>(),mat.getColumn<2>());

        hkQuaternion q; q.set(rot);
        q.m_vec.store<4,HK_IO_NATIVE_ALIGNED>(&animatedQuaternion->m_quaternions[4*i]);
    }

    return animatedQuaternion;
}

class hkxSparselyAnimatedString* hctAttributeProcessingUtil::convertEnumToString (class hkxSparselyAnimatedEnum* animatedEnum)
{
    return convertIntToString (animatedEnum, animatedEnum->m_enum);
};

class hkxSparselyAnimatedString* hctAttributeProcessingUtil::convertIntToString (class hkxSparselyAnimatedInt* animatedInt, const hkxEnum* enumClass)
{
    hkxSparselyAnimatedString* animatedString = new hkxSparselyAnimatedString();

    const int numStrings = animatedInt->m_ints.getSize();
    animatedString->m_times.insertAt(0, animatedInt->m_times.begin(), animatedInt->m_times.getSize() );
    animatedString->m_strings.setSize(numStrings);

    for (int i=0; i<numStrings; i++)
    {
        int value = animatedInt->m_ints[i];
        const char* name = HK_NULL;
        enumClass->getNameOfValue(value, &name);
        animatedString->m_strings[i] = name;
    }

    return animatedString;
};


hkxSparselyAnimatedEnum* hctAttributeProcessingUtil::convertIntToEnum (hkxSparselyAnimatedInt* animatedInt, hkxEnum* enumClass)
{
    hkxSparselyAnimatedEnum* animatedEnum = new hkxSparselyAnimatedEnum();

    // We know animated enums inherit from animated int, so we can do a normal memcopy
    animatedEnum->m_times.insertAt(0, animatedInt->m_times.begin(), animatedInt->m_times.getSize() );
    animatedEnum->m_ints.insertAt(0, animatedInt->m_ints.begin(), animatedInt->m_ints.getSize() );

    // and then fill the class enum pointer
    animatedEnum->m_enum = enumClass;

    // We could check the enum range here...

    return animatedEnum;
}

void hctAttributeProcessingUtil::processAttributes (hkxAttributeHolder* attributeHolder)
{

    //
    // First, merge any groups with the same name
    //
    {
        hkArray<hkxAttributeGroup>& groups = attributeHolder->m_attributeGroups;

        // We assume attribute groups are ordered hkX, hkX, hkXMerge, hkXMerge, hkX, hkxMerge..
        // That is, mergeable groups should always follow the group they should merge.
        for (int firstGroupIdx = 0; firstGroupIdx < groups.getSize(); firstGroupIdx++)
        {
            const hkxAttributeGroup& firstGroup = groups [firstGroupIdx];

            hkStringBuf firstGroupName(firstGroup.m_name);
            firstGroupName.lowerCase();

            if (firstGroupName.endsWith("merge"))
            {
                HK_WARN_ALWAYS(0xabbaa6c9,"Cannot merge attribute group "<<firstGroup.m_name);
                continue;
            }

            hkStringBuf firstGroupNameMerge; firstGroupNameMerge.setJoin(firstGroupName, "merge");

            for (int secondGroupIdx = firstGroupIdx+1; secondGroupIdx < groups.getSize(); secondGroupIdx++)
            {
                const hkxAttributeGroup& secondGroup = groups [secondGroupIdx];
                hkStringBuf secondGroupName = secondGroup.m_name;
                secondGroupName.lowerCase();

                // We found another group of the same type, nothing to merge further on
                if (secondGroupName==firstGroupName)
                {
                    break;
                }

                // Second group
                if (secondGroupName == firstGroupNameMerge)
                {
                    hkxAttributeGroup newGroup;
                    mergeTwoAttributeGroups(firstGroup, secondGroup, newGroup);

                    groups[firstGroupIdx] = newGroup;
                    groups.removeAtAndCopy(secondGroupIdx); //EXP-1144: removeAtAndCopy vs. removeAt
                    secondGroupIdx--;
                }
            }
        }

    }

    //
    // Second, process all the attribute groups
    //
    {
        const int numGroups = attributeHolder->m_attributeGroups.getSize();
        for( int i=0; i<numGroups; ++i )
        {
            processAttributeGroup( attributeHolder->m_attributeGroups[i] );
        }
    }
}

void hctAttributeProcessingUtil::mergeTwoAttributeGroups(const hkxAttributeGroup& groupOne, const hkxAttributeGroup& groupTwo, hkxAttributeGroup& mergedGroup)
{
    // Name : Should be the same
    {
        mergedGroup.m_name = groupOne.m_name;
    }

    // Attributes: store them in an hkArray, then copy
    {
        mergedGroup.m_attributes.setSize(0);
        mergedGroup.m_attributes.reserve(groupOne.m_attributes.getSize() + groupTwo.m_attributes.getSize());

        // We assume attributes are not duplicated inside a single group, but may be duplicated on each group
        // We start by merging the first group, comparing with second one, removing
        for (int firstAttrIdx=0; firstAttrIdx<groupOne.m_attributes.getSize(); firstAttrIdx++)
        {
            hkxAttribute firstAttr = groupOne.m_attributes[firstAttrIdx];

            const char* attrName = firstAttr.m_name;

            int secondAttrIdx = groupTwo.findAttributeIndexByName(attrName);

            // We only add attributes if they are not in the second group
            if (secondAttrIdx<0)
            {
                mergedGroup.m_attributes.pushBack(firstAttr);
            }
        }

        // From the second group, we add all attributes
        for (int secondAttrIdx=0; secondAttrIdx<groupTwo.m_attributes.getSize(); secondAttrIdx++)
        {
            hkxAttribute secondAttr = groupTwo.m_attributes[secondAttrIdx];
            mergedGroup.m_attributes.pushBack(secondAttr);
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
