// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterTexture/hctFilterTexture.h>
#include <ContentTools/Common/Filters/FilterTexture/GatherAndConvertTexture/hctGatherAndConvertTextureFilter.h>
#include <ContentTools/Common/Filters/FilterTexture/hctFilterTextureUtils.h>

#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>
#include <Common/Base/Serialize/Resource/hkResource.h>

#include <FreeImage.h>

#include <windowsx.h>
#include <commctrl.h>
#include <shlobj.h>

extern HINSTANCE hInstance;


static void _getCurrentOptions( HWND hWnd, hctGatherAndConvertTextureFilterOptions::ConvertOptions& opt )
{
    int formatIndex = ComboBox_GetCurSel( GetDlgItem( hWnd, IDC_GATHER_FORMAT ) );
    switch (formatIndex)
    {
        case 0: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_BMP; break;
        case 1: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_JPEG; break;
        case 2: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_PNG; break;
        case 3: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_TARGA; break;
        case 4: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_TIFF; break;
        case 5: opt.m_format = hctGatherAndConvertTextureFilterOptions::OF_HDR; break;
    }

    opt.m_resizeFilter = (hctGatherAndConvertTextureFilterOptions::ResizeFilter) ( ComboBox_GetCurSel( GetDlgItem( hWnd, IDC_GATHER_RESIZE ) ) - 1 );

    BOOL transOk;
    int v = GetDlgItemInt( hWnd, IDC_GATHER_XRES, &transOk, FALSE);
    if (transOk)
    {
        opt.m_xTexels = (hkUint16) v;
    }

    v = GetDlgItemInt( hWnd, IDC_GATHER_YRES, &transOk, FALSE);
    if (transOk)
    {
        opt.m_yTexels = (hkUint16) v;
    }

    opt.m_useOriginalFilename = IsDlgButtonChecked( hWnd, IDC_GATHER_USEORIGNAME ) == TRUE;
}

static void _getCurrentSelectedOptions( HWND hWnd, hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*>& options, hctGatherAndConvertTextureFilter* filter )
{
    options.setSize(1);
    options[0] = &filter->m_options.m_perTextureOptions[0];

    if( IsDlgButtonChecked( hWnd, IDC_GATHER_APPLYTOALL ) == FALSE)
    {
        HWND texListWnd = GetDlgItem( hWnd, IDC_GATHER_TEXTURE_LIST);
        int numItems = ListView_GetItemCount( texListWnd );
        hkArray<int> optionsIndices;

        for (int i=0; i < numItems; ++i)
        {
            if (ListView_GetItemState( texListWnd, i, LVIS_SELECTED ) != 0)
            {
                bool fileBased = i < filter->m_listedFileTextures.getSize();
                const char* textureName = fileBased?
                    filter->m_listedFileTextures[i]->m_name :
                    filter->m_listedInplaceTextures[i-filter->m_listedFileTextures.getSize()]->m_name;

                int haveOptions= false;
                for (int tt=0; textureName && (tt < filter->m_options.m_perTextureOptions.getSize()); ++tt)
                {
                    if (filter->m_options.m_perTextureOptions[tt].m_textureName && hkString::strCmp( textureName, filter->m_options.m_perTextureOptions[tt].m_textureName) ==0 )
                    {
                        // found
                        haveOptions = true;
                        optionsIndices.pushBack(tt);
                        break;
                    }
                }

                // make options for any object without them (and change icon acordingly)
                if (!haveOptions)
                {
                    hctGatherAndConvertTextureFilterOptions::ConvertOptions& opt = filter->m_options.m_perTextureOptions.expandOne();
                    // copy settings from last
                    _getCurrentOptions(hWnd, opt);
                    opt.m_textureName = const_cast<char *>(textureName);

                    optionsIndices.pushBack(filter->m_options.m_perTextureOptions.getSize() - 1);

                    LV_ITEM item;
                    item.mask = LVIF_IMAGE;
                    item.iImage = fileBased? 2 : 3 ;
                    item.iItem = i;
                    item.iSubItem = 0;
                    ListView_SetItem( texListWnd, &item);
                }
            }
        }

        for (int oii=0; oii < optionsIndices.getSize(); ++oii)
        {
            options.pushBack( &filter->m_options.m_perTextureOptions[ optionsIndices[oii] ] );
        }
    }
}

static void _reflectCurrentOptions( HWND hWnd, hctGatherAndConvertTextureFilterOptions::ConvertOptions& opt )
{
    //XX assumes the same enum etc:
    int formatIndex = 0;
    switch (opt.m_format)
    {
        case hctGatherAndConvertTextureFilterOptions::OF_BMP: formatIndex = 0; break;
        case hctGatherAndConvertTextureFilterOptions::OF_JPEG: formatIndex = 1; break;
        case hctGatherAndConvertTextureFilterOptions::OF_PNG: formatIndex = 2; break;
        case hctGatherAndConvertTextureFilterOptions::OF_TARGA: formatIndex = 3; break;
        case hctGatherAndConvertTextureFilterOptions::OF_TIFF: formatIndex = 4; break;
        case hctGatherAndConvertTextureFilterOptions::OF_HDR: formatIndex = 5; break;
    }

    ComboBox_SetCurSel( GetDlgItem( hWnd, IDC_GATHER_FORMAT ), formatIndex);
    ComboBox_SetCurSel( GetDlgItem( hWnd, IDC_GATHER_RESIZE ), (int)(opt.m_resizeFilter+1) );
    SetDlgItemInt( hWnd, IDC_GATHER_XRES, opt.m_xTexels, FALSE);
    SetDlgItemInt( hWnd, IDC_GATHER_YRES, opt.m_yTexels, FALSE);
    EnableWindow(  GetDlgItem( hWnd, IDC_GATHER_XRES ), opt.m_resizeFilter >= 0 );
    EnableWindow(  GetDlgItem( hWnd, IDC_GATHER_YRES ), opt.m_resizeFilter >= 0 );
    CheckDlgButton( hWnd, IDC_GATHER_USEORIGNAME, opt.m_useOriginalFilename);
}

static void _reflectCurrentOptions( HWND hWnd, int texIndex, hctGatherAndConvertTextureFilter* filter )
{
    // find the texture options (if any)
    const char* texName = HK_NULL;
    if (texIndex < filter->m_listedFileTextures.getSize())
    {
        texName = filter->m_listedFileTextures[texIndex]->m_name;
    }
    else
    {
        texIndex -= filter->m_listedFileTextures.getSize();
        texName = filter->m_listedInplaceTextures[texIndex]->m_name;
    }

    if (texName)
    {
        for (int pti=0; pti < filter->m_options.m_perTextureOptions.getSize(); ++pti)
        {
            const char* oname = filter->m_options.m_perTextureOptions[pti].m_textureName;
            if (oname && (hkString::strCmp(oname, texName)==0))
            {
                // found it.
                hctGatherAndConvertTextureFilterOptions::ConvertOptions& opt = filter->m_options.m_perTextureOptions[pti];
                _reflectCurrentOptions( hWnd, opt );
            }
        }
    }
}

static bool _toggleOptions( HWND hWnd, int texIndex, hctGatherAndConvertTextureFilter* filter )
{
    // find the texture options (if any)
    const char* texName = HK_NULL;
    if (texIndex < filter->m_listedFileTextures.getSize())
    {
        texName = filter->m_listedFileTextures[texIndex]->m_name;
    }
    else
    {
        texIndex -= filter->m_listedFileTextures.getSize();
        texName = filter->m_listedInplaceTextures[texIndex]->m_name;
    }

    if (texName)
    {
        for (int pti=0; pti < filter->m_options.m_perTextureOptions.getSize(); ++pti)
        {
            const char* oname = filter->m_options.m_perTextureOptions[pti].m_textureName;
            if (oname && (hkString::strCmp(oname, texName)==0))
            {
                filter->m_options.m_perTextureOptions.removeAtAndCopy(pti);
                return false; // no options now
            }
        }

        // if we get this far the it does not have options yet
        hctGatherAndConvertTextureFilterOptions::ConvertOptions& opt = filter->m_options.m_perTextureOptions.expandOne();
        _getCurrentOptions(hWnd, opt);
        opt.m_textureName = const_cast<char *>(texName);

        return true; // has options now
    }

    return false;
}


INT_PTR CALLBACK hkFilterGatherAndConvertTextureDialogProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
    hctGatherAndConvertTextureFilter* filter = reinterpret_cast<hctGatherAndConvertTextureFilter*>( (hkUlong) GetWindowLongPtr(hWnd,GWLP_USERDATA)) ;

    switch( message )
    {
    case WM_INITDIALOG:
        {
            filter = (hctGatherAndConvertTextureFilter*)lParam;
            SetWindowLongPtr( hWnd, GWLP_USERDATA, (LONG_PTR)lParam ); // so that it can be retrieved later

            HIMAGELIST hImageList = ImageList_LoadImage( hInstance, MAKEINTRESOURCE(IDB_TEXTURETYPES_BITMAP), 16, 1, RGB(255,255,255), IMAGE_BITMAP, LR_CREATEDIBSECTION );

            // make our image of the current root inc processing up to our filter
            filter->getFilterManager()->getInputContentsToCurrentFilter( &filter->m_optionSceneData, filter->m_optionSceneDataTracker);

            hkStringOld infoText("FreeImage version: ");
            infoText += FreeImage_GetVersion();
            infoText += "\n";
            infoText += FreeImage_GetCopyrightMessage();
            SetDlgItemText( hWnd, IDC_GATHER_STATIC_COPYRIGHT, infoText.cString());

            hkxScene* scenePtr = filter->m_optionSceneData->findObject<hkxScene>();
            HWND texListWnd = GetDlgItem( hWnd, IDC_GATHER_TEXTURE_LIST);

            ListView_SetImageList( texListWnd, hImageList, LVSIL_SMALL /*LVSIL_STATE*/);
            LV_COLUMN col;
            col.mask = LVCF_WIDTH;
            col.cx = 139;
            ListView_InsertColumn( texListWnd, 0, &col);

            LVGROUP grp;
            grp.mask = LVGF_HEADER;
            grp.pszHeader = L"File Textures";
            ListView_InsertGroup(texListWnd, 0, &grp);
            grp.pszHeader = L"Inplace Textures";
            ListView_InsertGroup(texListWnd, 1, &grp);

            ListView_EnableGroupView(texListWnd, TRUE);

            filter->m_listedFileTextures.setSize(0);
            filter->m_listedInplaceTextures.setSize(0);

            bool hadSomeOptions= false;
            if (scenePtr != HK_NULL)
            {
                hkArray<hkxMaterial::TextureType> infoHints;
                hkArray<const char*> fileHints;

                hctFilterTextureUtils::findFileTextures( *scenePtr, filter->m_listedFileTextures, &infoHints);
                hctFilterTextureUtils::findInplaceTextures( *scenePtr, filter->m_listedInplaceTextures, &infoHints );

                int numFileTextures = 0;
                hkArray<const char*> textureNames;

                for (int t=0; t < filter->m_listedFileTextures.getSize(); ++t)
                {
                    const char* tname = filter->m_listedFileTextures[t]->m_name;
                    if (tname)
                    {
                        textureNames.pushBack(tname);
                        fileHints.pushBack(filter->m_listedFileTextures[t]->m_filename );
                        ++numFileTextures;
                    }
                    else
                    {
                        filter->m_listedFileTextures.removeAtAndCopy(t);
                        infoHints.removeAtAndCopy(t);
                    }
                }

                for (int tt=0; tt < filter->m_listedInplaceTextures.getSize(); ++tt)
                {
                    const char* tname = filter->m_listedInplaceTextures[tt]->m_name;
                    if (tname)
                    {
                        textureNames.pushBack(tname);
                        fileHints.pushBack(filter->m_listedInplaceTextures[tt]->m_originalFilename );
                    }
                    else
                    {
                        filter->m_listedInplaceTextures.removeAtAndCopy(tt);
                        infoHints.removeAtAndCopy(tt + numFileTextures);
                    }
                }

                for (int tn=0; tn < textureNames.getSize(); ++tn)
                {
                    bool hasOptions = false;
                    for (int tt=0; tt < filter->m_options.m_perTextureOptions.getSize(); ++tt)
                    {
                        if (filter->m_options.m_perTextureOptions[tt].m_textureName && hkString::strCmp( textureNames[tn], filter->m_options.m_perTextureOptions[tt].m_textureName) ==0 )
                        {
                            hasOptions = true;
                            hadSomeOptions = true;
                            break;
                        }
                    }

                    LV_ITEM item;
                    item.mask = LVIF_IMAGE | LVIF_TEXT | LVIF_STATE | LVIF_GROUPID;
                    item.iItem = tn;
                    item.iSubItem = 0;
                    hkStringOld fullText;

                    hkStringOld pathHint;
                    int plen = hkString::strLen(fileHints[tn]);
                    pathHint = fileHints[tn];
                    if (plen > 10)
                    {
                        int justFilenameA = pathHint.lastIndexOf('\\');
                        int justFilenameB = pathHint.lastIndexOf('/');
                        int lastSlash = hkMath::max2<int>( justFilenameA, justFilenameB );
                        pathHint = (lastSlash > 0? hkStringOld("...") : hkStringOld("")) + pathHint.substr(lastSlash);
                    }

                    fullText.printf( "%s, [%s:%s]", textureNames[tn], hctFilterTextureUtils::hintToText( infoHints[tn]), pathHint.cString() );
                    item.pszText = const_cast<char*>( fullText.cString() );
                    item.state = item.stateMask = 0;
                    item.iGroupId = tn < numFileTextures? 0 : 1;
                    item.iImage = tn < numFileTextures? (hasOptions? 2 : 0) : (hasOptions? 3 : 1);
                    ListView_InsertItem( texListWnd, &item);
                }
            }

            CheckDlgButton( hWnd, IDC_GATHER_APPLYTOALL, !hadSomeOptions );
            EnableWindow( texListWnd, hadSomeOptions );

            SetDlgItemTextW( hWnd, IDC_GATHER_PATH, hkUtf8::WideFromUtf8(filter->m_options.m_outputTexturePath));
            CheckDlgButton( hWnd, IDC_GATHER_OVERWRITE_ORIG, filter->m_options.m_allowOverwrite);

            HWND formatCombo = GetDlgItem( hWnd, IDC_GATHER_FORMAT);
            HWND resizeCombo = GetDlgItem( hWnd, IDC_GATHER_RESIZE);
            ComboBox_AddString( formatCombo, "BMP");
            ComboBox_AddString( formatCombo, "JPEG");
            ComboBox_AddString( formatCombo, "PNG");
            ComboBox_AddString( formatCombo, "TARGA");
            ComboBox_AddString( formatCombo, "TIFF");
            ComboBox_AddString( formatCombo, "HDR");

            ComboBox_AddString( resizeCombo, "No Resizing");
            ComboBox_AddString( resizeCombo, "Box");
            ComboBox_AddString( resizeCombo, "Bicubic");
            ComboBox_AddString( resizeCombo, "Bilinear");
            ComboBox_AddString( resizeCombo, "B-spline");
            ComboBox_AddString( resizeCombo, "CatmullRom");
            ComboBox_AddString( resizeCombo, "Lanczos");

            hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
            _getCurrentSelectedOptions( hWnd, options, filter );
            _reflectCurrentOptions( hWnd, *(options[0]) );

            return TRUE; // did handle it
        }

    case WM_COMMAND: // UI Changes
        {
            switch ( LOWORD( wParam ) )
            {
                case IDC_GATHER_APPLYTOALL:
                {
                    HWND texListWnd = GetDlgItem( hWnd, IDC_GATHER_TEXTURE_LIST);
                    if( IsDlgButtonChecked( hWnd, IDC_GATHER_APPLYTOALL ) == TRUE )
                    {
                        EnableWindow( texListWnd, FALSE);
                    }
                    else
                    {
                        EnableWindow( texListWnd, TRUE);
                    }

                    hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                    _getCurrentSelectedOptions( hWnd, options, filter );
                    _reflectCurrentOptions( hWnd, *(options[0]) );

                    break;
                }

                case IDC_GATHER_USEORIGNAME:
                {
                    bool useOrig = IsDlgButtonChecked( hWnd, IDC_GATHER_APPLYTOALL ) == TRUE;
                    hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                    _getCurrentSelectedOptions( hWnd, options, filter );
                    for (int oi=0; oi < options.getSize(); ++oi)
                    {
                        options[oi]->m_useOriginalFilename = useOrig;
                    }

                    break;
                }


                case IDC_GATHER_FORMAT:
                {
                    HWND comboBox = (HWND)lParam;
                    if (HIWORD(wParam) == CBN_SELCHANGE)
                    {
                        int formatValue = ComboBox_GetCurSel(comboBox);
                        hctGatherAndConvertTextureFilterOptions::OutputFormat formatEnum = hctGatherAndConvertTextureFilterOptions::OF_PNG;
                        switch (formatValue)
                        {
                            case 0: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_BMP; break;
                            case 1: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_JPEG;  break;
                            case 2: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_PNG; break;
                            case 3: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_TARGA; break;
                            case 4: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_TIFF; break;
                            case 5: formatEnum = hctGatherAndConvertTextureFilterOptions::OF_HDR; break;
                        }

                        hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                        _getCurrentSelectedOptions( hWnd, options, filter );
                        for (int oi=0; oi < options.getSize(); ++oi)
                        {
                            options[oi]->m_format = formatEnum;
                        }
                    }
                    break;
                }

                case IDC_GATHER_RESIZE:
                {
                    HWND comboBox = (HWND)lParam;
                    if (HIWORD(wParam) == CBN_SELCHANGE)
                    {
                        int resizeValue = ComboBox_GetCurSel(comboBox);
                        EnableWindow(  GetDlgItem( hWnd, IDC_GATHER_XRES ), (resizeValue > 0) );
                        EnableWindow(  GetDlgItem( hWnd, IDC_GATHER_YRES ), (resizeValue > 0) );

                        hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                        _getCurrentSelectedOptions( hWnd, options, filter );
                        for (int oi=0; oi < options.getSize(); ++oi)
                        {
                            options[oi]->m_resizeFilter = hctGatherAndConvertTextureFilterOptions::ResizeFilter(resizeValue - 1);
                        }
                    }
                    break;
                }

                case IDC_GATHER_XRES:
                {
                    if (HIWORD(wParam) == EN_CHANGE)
                    {
                        BOOL bSuccess;
                        int xRes = GetDlgItemInt( hWnd, IDC_GATHER_XRES, &bSuccess, false);
                        if (bSuccess)
                        {
                            hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                            _getCurrentSelectedOptions( hWnd, options, filter );
                            for (int oi=0; oi < options.getSize(); ++oi)
                            {
                                options[oi]->m_xTexels = (hkUint16)xRes;
                            }
                        }
                    }
                    break;
                }

                case IDC_GATHER_YRES:
                {
                    if (HIWORD(wParam) == EN_CHANGE)
                    {
                        BOOL bSuccess;
                        int yRes = GetDlgItemInt( hWnd, IDC_GATHER_YRES, &bSuccess, false);
                        if (bSuccess)
                        {
                            hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                            _getCurrentSelectedOptions( hWnd, options, filter );
                            for (int oi=0; oi < options.getSize(); ++oi)
                            {
                                options[oi]->m_yTexels = (hkUint16)yRes;
                            }
                        }
                    }
                    break;
                }

                case IDC_GATHER_EXPLORE:
                {
                    // Open a browser so the user can choose a new path to replace the selected one.
                    wchar_t displayName[MAX_PATH];

                    LPMALLOC pMalloc;
                    SHGetMalloc(&pMalloc);

                    BROWSEINFOW pbi;
                    hkString::memSet( &pbi, 0, sizeof(pbi));
                    pbi.hwndOwner = hWnd;
                    pbi.lpszTitle = L"Please select an output for new file based textures";
                    pbi.ulFlags = BIF_USENEWUI;
                    pbi.pszDisplayName = displayName;

                    LPITEMIDLIST folders = SHBrowseForFolderW(&pbi);
                    if (folders && SHGetPathFromIDListW(folders, displayName) )
                    {
                        SetDlgItemTextW( hWnd, IDC_GATHER_PATH, displayName);
                    }

                    pMalloc->Free(folders);
                    pMalloc->Release();

                    break;
                }
            }
        }

    case WM_NOTIFY: // Listview state changes etc
        {
            int idCtrl = (int) wParam;
            LPNMHDR pnmh = (LPNMHDR) lParam;
            if (idCtrl == IDC_GATHER_TEXTURE_LIST)
            {
                switch (pnmh->code)
                {
                    case LVN_ITEMCHANGING:
                    {
                        LPNMLISTVIEW pnmv = (LPNMLISTVIEW)pnmh;
                        if (pnmv->uChanged & LVIF_STATE)
                        {
                            int texIndex = pnmv->iItem;
                            bool selected = (pnmv->uNewState & LVIS_SELECTED) != 0;
                            if (selected)
                            {
                                _reflectCurrentOptions(hWnd, texIndex, filter );
                            }
                        }

                        return FALSE; // will allow state change
                    }

                    case NM_DBLCLK: // toggle options on or off for this file
                    {
                        LPNMITEMACTIVATE lpnmitem = (LPNMITEMACTIVATE)pnmh;
                        bool hasOptions = _toggleOptions( hWnd, lpnmitem->iItem, filter );

                        // change image
                        LV_ITEM item;
                        item.mask = LVIF_IMAGE | LVIF_STATE;
                        item.iImage = lpnmitem->iItem < filter->m_listedFileTextures.getSize()? (hasOptions? 2 : 0) : (hasOptions? 3 : 1);
                        item.iItem = lpnmitem->iItem;
                        item.iSubItem = 0;
                        item.state = hasOptions? LVIS_SELECTED : 0; // select it if we have create options
                        item.stateMask = LVIS_SELECTED;
                        ListView_SetItem( pnmh->hwndFrom, &item);

                        // update the opt selection
                        hkArray<hctGatherAndConvertTextureFilterOptions::ConvertOptions*> options;
                        _getCurrentSelectedOptions( hWnd, options, filter );
                        _reflectCurrentOptions( hWnd, *(options[0]) );
                    }
                }
            }
            break;
        }
    }

    return FALSE;
}

HWND hctGatherAndConvertTextureFilter::showOptions( HWND owner)
{
    if( m_optionsDialog )
    {
        hideOptions();
    }

    // dealloc previous
    if (m_optionSceneDataTracker)
    {
        m_optionSceneDataTracker->removeReference();
        m_optionSceneDataTracker = HK_NULL;
    }

    m_optionsDialog = CreateDialogParamW( hInstance, MAKEINTRESOURCEW( IDD_GATHERCONVERT_DIALOG ),
        owner, hkFilterGatherAndConvertTextureDialogProc, (LPARAM) this );

    return m_optionsDialog;
}

void hctGatherAndConvertTextureFilter::hideOptions()
{
    if( m_optionsDialog )
    {
        if( IsDlgButtonChecked( m_optionsDialog, IDC_GATHER_APPLYTOALL ) == TRUE )
        {
            m_options.m_perTextureOptions.setSize(1);
            m_options.m_perTextureOptions[0].m_textureName = HK_NULL;
            // leave rest as is
        }

        m_options.m_outputTexturePath = HK_NULL;
        hctGetDlgItemHkStringPtr(m_optionsDialog, IDC_GATHER_PATH, m_options.m_outputTexturePath);

        m_options.m_allowOverwrite = IsDlgButtonChecked(m_optionsDialog, IDC_GATHER_OVERWRITE_ORIG) != 0;

        DestroyWindow( m_optionsDialog );
    }


    m_optionsDialog = NULL;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
