// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterTexture/hctFilterTexture.h>
#include <ContentTools/Common/Filters/FilterTexture/GatherAndConvertTexture/hctGatherAndConvertTextureFilter.h>
#include <ContentTools/Common/Filters/FilterTexture/hctFilterTextureUtils.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>

#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>

#include <Common/Base/Serialize/Resource/hkResource.h>


#include <FreeImage.h>
#include <tchar.h>


#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.texture.convert"
#include <Common/Base/System/Log/hkLog.hxx>


hctGatherAndConvertTextureFilterDesc g_gatherAndConvertTextureDesc;

hctGatherAndConvertTextureFilter::hctGatherAndConvertTextureFilter(const hctFilterManagerInterface* owner)
: hctFilterInterface (owner),
  m_optionSceneData(HK_NULL),
  m_optionSceneDataTracker(HK_NULL),
  m_optionsDialog(HK_NULL)
{
    m_options.m_perTextureOptions.setSize(1);
    m_options.m_perTextureOptions[0].m_format = hctGatherAndConvertTextureFilterOptions::OF_PNG;
    m_options.m_perTextureOptions[0].m_dither = hctGatherAndConvertTextureFilterOptions::DTH_NONE;
    m_options.m_perTextureOptions[0].m_resizeFilter = hctGatherAndConvertTextureFilterOptions::RF_NONE;
    m_options.m_perTextureOptions[0].m_xTexels = 512;
    m_options.m_perTextureOptions[0].m_xTexels = 512;
    m_options.m_perTextureOptions[0].m_textureName = HK_NULL;
    m_options.m_perTextureOptions[0].m_useOriginalFilename = false;

    m_options.m_outputTexturePath = ".";
    m_options.m_allowOverwrite = false;
}

hctGatherAndConvertTextureFilter::~hctGatherAndConvertTextureFilter()
{
    if (m_optionSceneDataTracker)
        m_optionSceneDataTracker->removeReference();

}

void hctGatherAndConvertTextureFilter::process( hkRootLevelContainer& data )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL)
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "No scene data found");
        return;
    }
    hkxScene& scene = *scenePtr;

    hkArray<hkxTextureFile*> doneFileTextures;
    hkArray<hkxTextureInplace*> doneInplaceTextures;

    bool optionsRun = getFilterManager()->getProcessMode() == hctFilterManagerInterface::OPTIONS_BATCH;

    if (!optionsRun)
    {
        Log_Info( "FreeImage version: {}", FreeImage_GetVersion() );
    }

    if (scene.m_materials.getSize() > 0)
    {
        hkArray<hkxMaterial*> sceneMaterials;
        hctFilterTextureUtils::findSceneMaterials(scene, sceneMaterials);

        for (int cim = 0; cim < sceneMaterials.getSize(); ++cim)
        {
            hkxMaterial* material = sceneMaterials[cim];
            // all external textures (can extend to inplace aswell later on)
            for (int net = 0; net < material->m_stages.getSize(); ++net)
            {
                if ( !(material->m_stages[net].m_texture.getType()) || !(material->m_stages[net].m_texture) )
                    continue;

                // File based texture
                if ( hkxTextureFile* textureFile = hkReflect::exactMatchDynCast<hkxTextureFile>(material->m_stages[net].m_texture) )
                {
                    int doneIdx = doneFileTextures.indexOf(textureFile);
                    if ( doneIdx < 0)
                    {
                        doneFileTextures.pushBack(textureFile);

                        bool skipTexture = (m_options.m_perTextureOptions.getSize() > 1);
                        int curIndex = -1;
                        for (int toi=0; skipTexture && (toi < m_options.m_perTextureOptions.getSize()); ++toi)
                        {
                            const char* tname = textureFile->m_name? textureFile->m_name : textureFile->m_filename;
                            if ( m_options.m_perTextureOptions[toi].m_textureName && (hkString::strCmp(m_options.m_perTextureOptions[toi].m_textureName, tname) == 0) )
                            {
                                curIndex = toi;
                                skipTexture = false;
                                break;
                            }
                        }
                        if (skipTexture)
                        {
                            continue;
                        }

                        hctGatherAndConvertTextureFilterOptions::ConvertOptions currentOptions = m_options.m_perTextureOptions[curIndex >= 0? curIndex : 0];

                        hkStringOld filename = (currentOptions.m_useOriginalFilename && (textureFile->m_originalFilename != HK_NULL))? textureFile->m_originalFilename : textureFile->m_filename;
                        hkStringOld filenameLowerCase = filename.asLowerCase();
                        hkStringOld fullPath;
                        hctFilterTextureUtils::getFullTexturePath( filename.cString(), data, fullPath );


                        // Make an output filename
                        hkStringOld outputFilename;
                        if (m_options.m_outputTexturePath != HK_NULL)
                        {
                            hkStringOld texPath(m_options.m_outputTexturePath);
                            if ( texPath.beginsWith(".") )
                            {
                                hkStringOld assetFolder( hctFilterUtils::getEnvironmentVariable(data, "assetFolder") );
                                bool needsSep = (!assetFolder.endsWith("/")) && (!assetFolder.endsWith("\\"));
                                texPath = assetFolder + ( needsSep? "/" : "" ) + texPath;
                            }

                            if ( !texPath.endsWith("/") && !texPath.endsWith("\\") )
                            {
                                texPath += "\\";
                            }

                            // strip the old path and add the new one
                            int lastIndexBack = fullPath.lastIndexOf('\\');
                            int lastIndexFront = fullPath.lastIndexOf('/');
                            int lastIndex = hkMath::max2<int>( lastIndexBack, lastIndexFront );

                            outputFilename = texPath + fullPath.substr( lastIndex + 1 );
                        }
                        else // assume same dir as orig tex, just with new ending
                        {
                            outputFilename = fullPath;
                        }

                        // make new ext

                        int dotExt = outputFilename.lastIndexOf('.');
                        if (dotExt < 0)
                        {
                            HK_WARN_ALWAYS(0xabba0000, "Could not resolve input file ext for [" << textureFile->m_name << "]. Skipping it." );
                            continue;
                        }
                        FREE_IMAGE_FORMAT outputFIF = (FREE_IMAGE_FORMAT)(int)currentOptions.m_format;
                        const char* threeCC = FreeImage_GetFormatFromFIF( outputFIF );
                        outputFilename.setAsSubstr(0, dotExt + 1);
                        outputFilename += threeCC;

                        if (!optionsRun)
                        {
                            // check if input == output
                            if (!m_options.m_allowOverwrite)
                            {
                                const int MaxBuf = 4096;
                                TCHAR   winInputFullPath[MaxBuf] = TEXT("");
                                TCHAR   winOutputFullPath[MaxBuf] = TEXT("");

                                int retvalIn = GetFullPathName(fullPath.cString(), MaxBuf, winInputFullPath, HK_NULL);
                                int retvalOut = GetFullPathName(outputFilename.cString(), MaxBuf, winOutputFullPath, HK_NULL);
                                if ((retvalIn == 0) ||( retvalOut == 0))
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not resolve input and output filenames for [" << textureFile->m_name << "]. Skipping it." );
                                    continue;
                                }

                                if (_tcsncmp(winInputFullPath, winOutputFullPath, MaxBuf) == 0)
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Input and output files the same and options disallow it, on [" << textureFile->m_name << "], both end up as [" << winInputFullPath << "]. Skipping it." );
                                    continue;
                                }
                            }

                            // Do the convert
                            FREE_IMAGE_FORMAT fileFormat = FreeImage_GetFileType(fullPath.cString());

                            if (fileFormat != FIF_UNKNOWN)
                            {
                                FIBITMAP* dib = FreeImage_Load( fileFormat, fullPath.cString() );

                                // resize?
                                if (currentOptions.m_resizeFilter != hctGatherAndConvertTextureFilterOptions::RF_NONE)
                                {
                                    FIBITMAP* rdib = FreeImage_Rescale( dib, currentOptions.m_xTexels, currentOptions.m_yTexels, (FREE_IMAGE_FILTER)(int)currentOptions.m_resizeFilter );
                                    if (!rdib)
                                    {
                                        HK_WARN_ALWAYS(0xabba0000, "Rescale failed on [" << textureFile->m_name << "]. Leaving it the input size." );
                                    }
                                    else
                                    {
                                        Log_Info( "-> Resized texture [{}] from [{}, {}] to [{}, {}]", textureFile->m_name, FreeImage_GetWidth(dib), FreeImage_GetHeight(dib), currentOptions.m_xTexels, currentOptions.m_yTexels );
                                        FreeImage_Unload(dib);
                                        dib = rdib;
                                    }
                                }


                                if ( FreeImage_Save( outputFIF, dib, outputFilename.cString(), 0 ) )
                                {
                                    Log_Info( "Converted file based texture [{}] from [{}] to [{}].", textureFile->m_name, fullPath.cString(), outputFilename.cString() );

                                    // alter the texture path
                                    textureFile->m_filename = outputFilename.cString();
                                }
                                else
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not save file base texture [" << textureFile->m_name << "] to [" << outputFilename.cString() << "]. Leaving it as it was." );
                                }

                                FreeImage_Unload(dib);
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Unkown file texture format in texture [" << textureFile->m_name << "]" );
                            }
                        }
                        else // just alter filename so options can see what it would be
                        {
                            textureFile->m_filename = outputFilename.cString();
                        }

                    }
                }
                else // Inplace texture
                {
                    hkxTextureInplace* textureInplace = (hkxTextureInplace*)( material->m_stages[net].m_texture.val() );
                    int doneIdx = doneInplaceTextures.indexOf(textureInplace);
                    if ( doneIdx < 0)
                    {
                        doneInplaceTextures.pushBack(textureInplace);

                        bool skipTexture = (m_options.m_perTextureOptions.getSize() > 1);
                        int curIndex = -1;
                        for (int toi=0; skipTexture && (toi < m_options.m_perTextureOptions.getSize()); ++toi)
                        {
                            if ( textureInplace->m_name && ( hkString::strCmp(m_options.m_perTextureOptions[toi].m_textureName, textureInplace->m_name) == 0 ) )
                            {
                                curIndex = toi;
                                skipTexture = false;
                                break;
                            }
                        }
                        if (skipTexture)
                        {
                            continue;
                        }

                        hctGatherAndConvertTextureFilterOptions::ConvertOptions currentOptions = m_options.m_perTextureOptions[curIndex >= 0? curIndex : 0];

                        FIMEMORY* fimem = HK_NULL;
                        FIBITMAP* dib = HK_NULL;
                        FREE_IMAGE_FORMAT fileFormat = FIF_UNKNOWN;

                        if ( currentOptions.m_useOriginalFilename && (textureInplace->m_originalFilename != HK_NULL) )
                        {
                            hkStringOld filename = textureInplace->m_originalFilename;
                            hkStringOld filenameLowerCase = filename.asLowerCase();
                            hkStringOld fullPath;
                            hctFilterTextureUtils::getFullTexturePath( filename.cString(), data, fullPath );

                            fileFormat = FreeImage_GetFileType(fullPath.cString());
                            if (fileFormat != FIF_UNKNOWN)
                            {
                                dib = FreeImage_Load( fileFormat, fullPath.cString() );
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Could not make input stream for inplace texture (orig filename requested) [" << textureInplace->m_name << "], orig filename [" << fullPath.cString() << "]" );
                            }
                        }
                        else // load from inplace mem
                        {
                            // Do the convert
                            fimem = FreeImage_OpenMemory(textureInplace->m_data.begin(), textureInplace->m_data.getSize() );
                            if (fimem)
                            {
                                FREE_IMAGE_FORMAT fileFormatFree = FreeImage_GetFIFFromFormat( reinterpret_cast<const char*>(textureInplace->m_fileType) );
                                if (fileFormatFree == FIF_UNKNOWN)
                                {
                                    fileFormatFree = FreeImage_GetFileTypeFromMemory( fimem, 0 );
                                }

                                if (fileFormatFree != FIF_UNKNOWN)
                                {
                                    dib = FreeImage_LoadFromMemory(fileFormatFree, fimem, 0 );

                                }
                                else
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Unkown inplace texture format in texture [" << textureInplace->m_name << "]" );
                                }
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Could not make input stream for inplace texture [" << textureInplace->m_name << "]" );
                            }
                        }

                        if (dib != HK_NULL)
                        {
                            // resize?
                            if (currentOptions.m_resizeFilter != hctGatherAndConvertTextureFilterOptions::RF_NONE)
                            {
                                FIBITMAP* rdib = FreeImage_Rescale( dib, currentOptions.m_xTexels, currentOptions.m_yTexels, (FREE_IMAGE_FILTER)(int)currentOptions.m_resizeFilter );
                                if (!rdib)
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Rescale failed on [" << textureInplace->m_name << "]. Leaving it the input size." );
                                }
                                else
                                {
                                    Log_Info( "-> Resized texture [{}] from [{}, {}] to [{}, {}]", textureInplace->m_name, FreeImage_GetWidth(dib), FreeImage_GetHeight(dib), currentOptions.m_xTexels, currentOptions.m_yTexels );

                                    FreeImage_Unload(dib);
                                    dib = rdib;
                                }
                            }

                            FIMEMORY* fomem = FreeImage_OpenMemory();
                            if (fomem)
                            {
                                FREE_IMAGE_FORMAT outputFIF = (FREE_IMAGE_FORMAT)(int)currentOptions.m_format;
                                const char* threeCC = FreeImage_GetFormatFromFIF( outputFIF );

                                if ( FreeImage_SaveToMemory( outputFIF, dib, fomem, 0 ) )
                                {
                                    Log_Info( "Converted inplace texture [{}] from [{}] to [{}].", textureInplace->m_name, textureInplace->m_fileType, threeCC );

                                    // alter the texture
                                    int bufSize = FreeImage_TellMemory(fomem);
                                    if (bufSize > 0)
                                    {
                                        textureInplace->m_data.setSize(bufSize);
                                        FreeImage_SeekMemory(fomem, 0, SEEK_SET);
                                        FreeImage_ReadMemory(textureInplace->m_data.begin(), bufSize, 1, fomem );
                                    }
                                    else
                                    {
                                        textureInplace->m_data.setSize(0); // failed..
                                    }

                                    textureInplace->m_fileType[0] = threeCC[0];
                                    textureInplace->m_fileType[1] = threeCC[1];
                                    textureInplace->m_fileType[2] = threeCC[2];
                                    textureInplace->m_fileType[3] = '\0';
                                }
                                else
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not converted inplace texture [" << textureInplace->m_name << "] from [" << textureInplace->m_fileType << "] to [" << threeCC << "]. Leaving it as it was." );
                                }

                                FreeImage_CloseMemory( fomem );
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Could not make output stream for inplace texture [" << textureInplace->m_name << "]" );
                            }


                            FreeImage_Unload(dib);
                        }

                        if (fimem)
                        {
                            FreeImage_CloseMemory( fimem );
                        }
                    }
                }
            }

        } // all mats
    }
}

void hctGatherAndConvertTextureFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctGatherAndConvertTextureFilterOptions* options = hctFilterUtils::getNativeOptions<hctGatherAndConvertTextureFilterOptions>(optVar))
    {
        m_options = *options;
        delete options;
    }
}

void hctGatherAndConvertTextureFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
