// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <ContentTools/Common/Filters/FilterTexture/hctFilterTexture.h>
#include <ContentTools/Common/Filters/FilterTexture/CompressTexture/hctCompressTextureFilter.h>
#include <ContentTools/Common/Filters/FilterTexture/hctFilterTextureUtils.h>

#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>

#include <Common/SceneData/Material/hkxMaterial.h>
#include <Common/SceneData/Material/hkxTextureFile.h>
#include <Common/SceneData/Material/hkxTextureInplace.h>

#include <Common/Base/Serialize/Resource/hkResource.h>


#include <nvtt.h>
#include <FreeImage.h>
#include <tchar.h>


#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "hct.texture.compress"
#include <Common/Base/System/Log/hkLog.hxx>


hctCompressTextureFilterDesc g_compressTextureDesc;

hctCompressTextureFilter::hctCompressTextureFilter(const hctFilterManagerInterface* owner)
: hctFilterInterface (owner),
  m_optionsDialog(HK_NULL), m_optionSceneData(HK_NULL), m_optionSceneDataTracker(HK_NULL)
{
    m_options.m_perTextureOptions.setSize(1);
    m_options.m_perTextureOptions[0].m_format = hctCompressTextureFilterOptions::OF_DXT5;
    m_options.m_perTextureOptions[0].m_mipMapFilter = hctCompressTextureFilterOptions::MF_Box;
    m_options.m_perTextureOptions[0].m_quality = hctCompressTextureFilterOptions::OQ_Production;
    m_options.m_perTextureOptions[0].m_textureName = HK_NULL;
    m_options.m_perTextureOptions[0].m_flipX = false;
    m_options.m_perTextureOptions[0].m_flipY = false;
    m_options.m_perTextureOptions[0].m_useOriginalFilename = false;

    m_options.m_outputTexturePath = ".";
    m_options.m_allowOverwrite = false;
}

hctCompressTextureFilter::~hctCompressTextureFilter()
{
    if (m_optionSceneDataTracker)
        m_optionSceneDataTracker->removeReference();
}

static nvtt::Format _getNvttFormat( hctCompressTextureFilterOptions::OutputFormat f )
{
    switch (f)
    {
        case hctCompressTextureFilterOptions::OF_DXT1: return nvtt::Format_DXT1;
        case hctCompressTextureFilterOptions::OF_DXT3: return nvtt::Format_DXT3;
        case hctCompressTextureFilterOptions::OF_DXT5: return nvtt::Format_DXT5;
        case hctCompressTextureFilterOptions::OF_DXT5n: return nvtt::Format_DXT5n;
        case hctCompressTextureFilterOptions::OF_BC4: return nvtt::Format_BC4;
        case hctCompressTextureFilterOptions::OF_BC5: return nvtt::Format_BC5;
        default: return nvtt::Format_DXT5;
    }
}

static const char* _getNvttFormatString( hctCompressTextureFilterOptions::OutputFormat f )
{
    switch (f)
    {
        case hctCompressTextureFilterOptions::OF_DXT1: return "DXT1/BC1";
        case hctCompressTextureFilterOptions::OF_DXT3: return "DXT3/BC2";
        case hctCompressTextureFilterOptions::OF_DXT5: return "DXT5/BC3";
        case hctCompressTextureFilterOptions::OF_DXT5n: return "DXT5n/BC3n";
        case hctCompressTextureFilterOptions::OF_BC4: return "BC4";
        case hctCompressTextureFilterOptions::OF_BC5: return "BC5";
        default: return "unknown";
    }
}


static nvtt::Quality _getNvttQuality( hctCompressTextureFilterOptions::OutputQuality q )
{
    switch (q)
    {
        case hctCompressTextureFilterOptions::OQ_Fastest: return nvtt::Quality_Fastest;
        case hctCompressTextureFilterOptions::OQ_Normal: return nvtt::Quality_Normal;
        case hctCompressTextureFilterOptions::OQ_Production: return nvtt::Quality_Production;
        case hctCompressTextureFilterOptions::OQ_Highest: return nvtt::Quality_Highest;
    }
    return nvtt::Quality_Production;;
}

static nvtt::MipmapFilter _getNvttMipFilter( hctCompressTextureFilterOptions::MipmapFilter mf )
{
    switch (mf)
    {
        case hctCompressTextureFilterOptions::MF_Box: return nvtt::MipmapFilter_Box;
        case hctCompressTextureFilterOptions::MF_Triangle: return nvtt::MipmapFilter_Triangle;
        case hctCompressTextureFilterOptions::MF_Kaiser: return nvtt::MipmapFilter_Kaiser;
        default: return nvtt::MipmapFilter_Kaiser;
    }
}

class StreamNvttHandler : public nvtt::OutputHandler
{
public:

    void beginImage(int size, int width, int height, int depth, int face, int miplevel);
    bool writeData(const void * data, int size);

    hkArray<char> m_data;
};

void StreamNvttHandler::beginImage(int size, int width, int height, int depth, int face, int miplevel)
{
    /* ignore */
}

bool StreamNvttHandler::writeData(const void * data, int size)
{
    m_data.insertAt( m_data.getSize(), (char*)data, size );
    return true;
}


// We use FreeImage to load the texture to increase the range of formats we can convert to DDS
// It also allows us to do transforms, such as flip x, y etc easily
// We then use NVTT to write the texture to DDS (which in turn just uses squish)

void hctCompressTextureFilter::process( hkRootLevelContainer& data  )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL)
    {
        HK_WARN_ALWAYS(0xabbaa5f0, "No scene data found");
        return;
    }
    hkxScene& scene = *scenePtr;

    hkArray<hkxTextureFile*> doneFileTextures;
    hkArray<hkxTextureInplace*> doneInplaceTextures;

    bool optionsRun = getFilterManager()->getProcessMode() == hctFilterManagerInterface::OPTIONS_BATCH;

    if (scene.m_materials.getSize() > 0)
    {
        hkArray<hkxMaterial*> sceneMaterials;
        hctFilterTextureUtils::findSceneMaterials(scene, sceneMaterials);

        for (int cim = 0; cim < sceneMaterials.getSize(); ++cim)
        {
            hkxMaterial* material = sceneMaterials[cim];

            for (int net = 0; net < material->m_stages.getSize(); ++net)
            {
                if ( !(material->m_stages[net].m_texture.getType()) || !(material->m_stages[net].m_texture) )
                    continue;

                // File based texture
                if ( hkxTextureFile* textureFile = hkReflect::exactMatchDynCast<hkxTextureFile>(material->m_stages[net].m_texture) )
                {
                    int doneIdx = doneFileTextures.indexOf(textureFile);
                    if ( doneIdx < 0)
                    {
                        doneFileTextures.pushBack(textureFile);

                        bool skipTexture = (m_options.m_perTextureOptions.getSize() > 1);
                        int curIndex = -1;
                        for (int toi=0; skipTexture && (toi < m_options.m_perTextureOptions.getSize()); ++toi)
                        {
                            const char* tname = textureFile->m_name? textureFile->m_name : textureFile->m_filename;
                            if ( m_options.m_perTextureOptions[toi].m_textureName && (hkString::strCmp(m_options.m_perTextureOptions[toi].m_textureName, tname) == 0) )
                            {
                                curIndex = toi;
                                skipTexture = false;
                                break;
                            }
                        }
                        if (skipTexture)
                        {
                            continue;
                        }


                        hkStringOld filename = textureFile->m_filename;
                        hkStringOld filenameLowerCase = filename.asLowerCase();
                        hkStringOld fullPath;
                        hctFilterTextureUtils::getFullTexturePath( filename.cString(), data, fullPath );

                        hctCompressTextureFilterOptions::ConvertOptions currentOptions = m_options.m_perTextureOptions[curIndex >= 0? curIndex : 0];

                        // Make an output filename
                        hkStringOld outputFilename;
                        if (m_options.m_outputTexturePath.getLength() > 0)
                        {
                            hkStringOld texPath(m_options.m_outputTexturePath);
                            if ( texPath.beginsWith(".") )
                            {
                                hkStringOld assetFolder( hctFilterUtils::getEnvironmentVariable(data, "assetFolder") );
                                bool needsSep = (!assetFolder.endsWith("/")) && (!assetFolder.endsWith("\\"));
                                texPath = assetFolder + ( needsSep? "/" : "" ) + texPath;
                            }

                            if ( !texPath.endsWith("/") && !texPath.endsWith("\\") )
                            {
                                texPath += "\\";
                            }

                            // strip the old path and add the new one
                            int lastIndexBack = fullPath.lastIndexOf('\\');
                            int lastIndexFront = fullPath.lastIndexOf('/');
                            int lastIndex = hkMath::max2<int>( lastIndexBack, lastIndexFront );

                            outputFilename = texPath + fullPath.substr( lastIndex + 1 );
                        }
                        else // assume same dir as orig tex, just with new ending
                        {
                            outputFilename = fullPath;
                        }

                        // make new ext

                        int dotExt = outputFilename.lastIndexOf('.');
                        if (dotExt < 0)
                        {
                            HK_WARN_ALWAYS(0xabba0000, "Could not resolve input file ext for [" << textureFile->m_name << "]. Skipping it." );
                            continue;
                        }

                        nvtt::Format outputFormat = _getNvttFormat( currentOptions.m_format );

                        outputFilename.setAsSubstr(0, dotExt + 1);
                        outputFilename += "DDS";

                        if (!optionsRun)
                        {
                            // check if input == output
                            if (!m_options.m_allowOverwrite)
                            {
                                const int MaxBuf = 4096;
                                TCHAR   winInputFullPath[MaxBuf] = TEXT("");
                                TCHAR   winOutputFullPath[MaxBuf] = TEXT("");

                                int retvalIn = GetFullPathName(fullPath.cString(), MaxBuf, winInputFullPath, HK_NULL);
                                int retvalOut = GetFullPathName(outputFilename.cString(), MaxBuf, winOutputFullPath, HK_NULL);
                                if ((retvalIn == 0) ||( retvalOut == 0))
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not resolve input and output filenames for [" << textureFile->m_name << "]. Skipping it." );
                                    continue;
                                }

                                if (_tcsncmp(winInputFullPath, winOutputFullPath, MaxBuf) == 0)
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Input and output files the same and options disallow it, on [" << textureFile->m_name << "], both end up as [" << winInputFullPath << "]. Skipping it." );
                                    continue;
                                }
                            }

                            // Do the convert
                            FREE_IMAGE_FORMAT fileFormat = FreeImage_GetFileType(fullPath.cString());

                            if (fileFormat != FIF_UNKNOWN)
                            {
                                FIBITMAP* dib = FreeImage_Load( fileFormat, fullPath.cString() );

                                if (currentOptions.m_flipX)
                                {
                                    if (!FreeImage_FlipHorizontal(dib))
                                    {
                                        HK_WARN_ALWAYS(0xabba0000, "Could not flip horizontal for [" << textureFile->m_name << "]." );
                                    }
                                    else
                                    {
                                        Log_Info( "-> Flipped along X (horizontal) [{}]", textureFile->m_name );
                                    }
                                }

                                if (currentOptions.m_flipY)
                                {
                                    if (!FreeImage_FlipVertical(dib))
                                    {
                                        HK_WARN_ALWAYS(0xabba0000, "Could not flip vertical for [" << textureFile->m_name << "]." );
                                    }
                                    else
                                    {
                                        Log_Info( "-> Flipped along Y (vertical) [{}]", textureFile->m_name );
                                    }
                                }

                                FREE_IMAGE_COLOR_TYPE colorType = FreeImage_GetColorType(dib);

                                if ( colorType != FIC_RGBALPHA )
                                {
                                    FIBITMAP* convDib = FreeImage_ConvertTo32Bits(dib);
                                    FreeImage_Unload(dib);
                                    dib = convDib;
                                }

                                void* rawData = FreeImage_GetBits( dib );
                                int w = FreeImage_GetWidth(dib);
                                int h = FreeImage_GetHeight(dib);
                                //int bpp = FreeImage_GetBPP(dib);
                                bool hasAlpha = FreeImage_GetColorType(dib) == FIC_RGBALPHA; // will be FIC_RGB otherwise

                                unsigned int rmask = FreeImage_GetRedMask(dib);
                                unsigned int gmask = FreeImage_GetGreenMask(dib);
                                unsigned int bmask = FreeImage_GetBlueMask(dib);
                                unsigned int amask = hasAlpha ? 0xff000000 : 0; // always last, BGR[A] convention

                                nvtt::InputOptions inputOptions;
                                inputOptions.setTextureLayout(nvtt::TextureType_2D, w, h);
                                inputOptions.setFormat(nvtt::InputFormat_BGRA_8UB);
                                inputOptions.setMipmapData(rawData, w, h);
                                inputOptions.setMipmapGeneration( currentOptions.m_mipMapFilter != hctCompressTextureFilterOptions::MF_NoMipmaps );

                                nvtt::OutputOptions outputOptions;
                                outputOptions.setFileName(outputFilename.cString());

                                nvtt::CompressionOptions compressionOptions;
                                compressionOptions.setPixelFormat(hasAlpha? 32 : 24, rmask, gmask, bmask, amask);
                                compressionOptions.setQuality( _getNvttQuality(currentOptions.m_quality) );
                                compressionOptions.setFormat(outputFormat);

                                nvtt::Compressor compressor;
                                if (!compressor.process(inputOptions, compressionOptions, outputOptions))
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not compress [" << textureFile->m_name << "]. Will still use orig file." );
                                }
                                else
                                {
                                    Log_Info( "Compressed [{}] to [{}].", textureFile->m_name, _getNvttFormatString(currentOptions.m_format) );

                                    textureFile->m_filename = outputFilename.cString();
                                }
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Unkown texture file format in texture [" << textureFile->m_name << "], [" << fullPath.cString() << "]" );
                            }
                        }
                        else // just alter filename so options can see what it would be
                        {
                            textureFile->m_filename = outputFilename.cString();
                        }
                    }
                }
                else // Inplace texture
                {
                    hkxTextureInplace* textureInplace = (hkxTextureInplace*)( material->m_stages[net].m_texture.val() );
                    int doneIdx = doneInplaceTextures.indexOf(textureInplace);
                    if ( doneIdx < 0)
                    {
                        doneInplaceTextures.pushBack(textureInplace);

                        bool skipTexture = (m_options.m_perTextureOptions.getSize() > 1);
                        int curIndex = -1;
                        for (int toi=0; skipTexture && (toi < m_options.m_perTextureOptions.getSize()); ++toi)
                        {
                            if ( textureInplace->m_name && ( hkString::strCmp(m_options.m_perTextureOptions[toi].m_textureName, textureInplace->m_name) == 0 ) )
                            {
                                curIndex = toi;
                                skipTexture = false;
                                break;
                            }
                        }
                        if (skipTexture)
                        {
                            continue;
                        }

                        hctCompressTextureFilterOptions::ConvertOptions currentOptions = m_options.m_perTextureOptions[curIndex >= 0? curIndex : 0];

                        nvtt::Format outputFormat = _getNvttFormat( currentOptions.m_format );

                        // if (!optionsRun), as inplace can do the convert and not affect anything else

                        // Do the convert
                        FIMEMORY* fimem = FreeImage_OpenMemory(textureInplace->m_data.begin(), textureInplace->m_data.getSize() );
                        if (fimem)
                        {
                            FREE_IMAGE_FORMAT fileFormat = FreeImage_GetFIFFromFormat( reinterpret_cast<const char*>(textureInplace->m_fileType) );
                            if (fileFormat == FIF_UNKNOWN)
                            {
                                fileFormat = FreeImage_GetFileTypeFromMemory( fimem, 0 );
                            }

                            if (fileFormat != FIF_UNKNOWN)
                            {
                                FIBITMAP* dib = FreeImage_LoadFromMemory( fileFormat, fimem, 0 );

                                if (currentOptions.m_flipX)
                                {
                                    if (!FreeImage_FlipHorizontal(dib))
                                    {
                                        HK_WARN_ALWAYS(0xabba0000, "Could not flip horizontal for [" << textureInplace->m_name << "]." );
                                    }
                                    else
                                    {
                                        Log_Info( "-> Flipped along X (horizontal) [{}]", textureInplace->m_name );
                                    }
                                }

                                if (currentOptions.m_flipY)
                                {
                                    if (!FreeImage_FlipVertical(dib))
                                    {
                                        HK_WARN_ALWAYS(0xabba0000, "Could not flip vertical for [" << textureInplace->m_name << "]." );
                                    }
                                    else
                                    {
                                        Log_Info( "-> Flipped along Y (vertical) [{}]", textureInplace->m_name );
                                    }
                                }

                                FREE_IMAGE_COLOR_TYPE colorType = FreeImage_GetColorType(dib);
                                if ( colorType != FIC_RGBALPHA )
                                {
                                    FIBITMAP* convDib = FreeImage_ConvertTo32Bits(dib);
                                    FreeImage_Unload(dib);
                                    dib = convDib;
                                }

                                void* rawData = FreeImage_GetBits( dib );
                                int w = FreeImage_GetWidth(dib);
                                int h = FreeImage_GetHeight(dib);
                                //int bpp = FreeImage_GetBPP(dib);
                                bool hasAlpha = FreeImage_GetColorType(dib) == FIC_RGBALPHA; // will be FIC_RGB otherwise
                                unsigned int rmask = FreeImage_GetRedMask(dib);
                                unsigned int gmask = FreeImage_GetGreenMask(dib);
                                unsigned int bmask = FreeImage_GetBlueMask(dib);
                                unsigned int amask = hasAlpha? 0xff000000 : 0; // always last, BGR[A] convention

                                nvtt::InputOptions inputOptions;
                                inputOptions.setTextureLayout(nvtt::TextureType_2D, w, h);
                                inputOptions.setMipmapData(rawData, w, h);
                                inputOptions.setFormat(nvtt::InputFormat_BGRA_8UB);
                                inputOptions.setMipmapGeneration( currentOptions.m_mipMapFilter != hctCompressTextureFilterOptions::MF_NoMipmaps );

                                nvtt::OutputOptions outputOptions;
                                StreamNvttHandler streamHandler;
                                outputOptions.setOutputHeader(true);
                                outputOptions.setOutputHandler(&streamHandler);

                                nvtt::CompressionOptions compressionOptions;
                                compressionOptions.setPixelFormat(hasAlpha? 32 : 24, rmask, gmask, bmask, amask);
                                compressionOptions.setQuality( _getNvttQuality(currentOptions.m_quality) );
                                compressionOptions.setFormat(outputFormat);

                                nvtt::Compressor compressor;
                                if (!compressor.process(inputOptions, compressionOptions, outputOptions))
                                {
                                    HK_WARN_ALWAYS(0xabba0000, "Could not compress [" << textureInplace->m_name << "]. Will still use orig file." );
                                }
                                else if (streamHandler.m_data.getSize() > 0)
                                {
                                    Log_Info( "Compressed [{}] to [{}]. WILL OVERWRITE ORIGINAL INPLACE TEXTURE IN SCENE.", textureInplace->m_name, _getNvttFormatString(currentOptions.m_format) );

                                    textureInplace->m_data.setSize(streamHandler.m_data.getSize());
                                    hkString::memCpy( textureInplace->m_data.begin(), streamHandler.m_data.begin(), streamHandler.m_data.getSize() );

                                    textureInplace->m_fileType[0] = 'D';
                                    textureInplace->m_fileType[1] = 'D';
                                    textureInplace->m_fileType[2] = 'S';
                                    textureInplace->m_fileType[3] = '\0';
                                }
                                else
                                {
                                    HK_WARN_ALWAYS(0xabba0001, "Could not compress [" << textureInplace->m_name << "] (no data written). Will still use orig file." );
                                }

                                FreeImage_Unload(dib);
                            }
                            else
                            {
                                HK_WARN_ALWAYS(0xabba0000, "Unkown inplace texture format in texture [" << textureInplace->m_name << "]" );
                            }

                            FreeImage_CloseMemory( fimem );
                        }
                        else
                        {
                            HK_WARN_ALWAYS(0xabba0000, "Could not make input stream for inplace texture [" << textureInplace->m_name << "]" );
                        }
                    }
                }
            }
        } // all mats
    }
}


void hctCompressTextureFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctCompressTextureFilterOptions* options = hctFilterUtils::getNativeOptions<hctCompressTextureFilterOptions>(optVar))
    {
        m_options = *options;
        hctFilterUtils::emptyIfNull(m_options.m_outputTexturePath);
        delete options;
    }
}

void hctCompressTextureFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
