// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <ContentTools/Common/Filters/FilterScene/hctFilterScene.h>
#include <ContentTools/Common/SdkUtils/hctSdkUtils.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <ContentTools/Common/Filters/FilterScene/SceneTransform/hctSceneTransformFilter.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

hctSceneTransformFilterDesc g_sceneTransformDesc;

hctSceneTransformFilter::hctSceneTransformFilter(const hctFilterManagerInterface* owner)
:   hctFilterInterface (owner),
    m_optionsDialog(NULL), m_fillingControls(false), m_doNotRefreshEdit(false)
{
    m_options.m_matrix.setIdentity();
    m_options.m_applyToBuffers = true;
    m_options.m_applyToFloatChannels = true;
    m_options.m_applyToCameras = true;
    m_options.m_applyToLights = true;
    m_options.m_applyToNodes = true;
    m_options.m_flipWinding = false;
    m_options.m_preset = hctSceneTransformOptions::IDENTITY;

    // Needed for destruction data transformation.
    hctSdkUtils::loadAllClasses();
}

hctSceneTransformFilter::~hctSceneTransformFilter()
{
}


void hctSceneTransformFilter::process( class hkRootLevelContainer& data  )
{
    // Find the scene in the root level container
    hkxScene* scenePtr = data.findObject<hkxScene>();
    if (scenePtr == HK_NULL)
    {
        HK_WARN_ALWAYS (0xabbaa5f0, "No scene data found");
        return;
    }
    hkxScene& scene = *scenePtr;

    // we don't need to register any classes as we are not adding any extra types that are not 'known' already.

    // which options to process with?
    hkxSceneUtils::SceneTransformOptions options;

    //
    options.m_applyToBuffers = m_options.m_applyToBuffers;
    options.m_applyToFloatChannels = m_options.m_applyToFloatChannels;
    options.m_applyToCameras = m_options.m_applyToCameras;
    options.m_applyToLights = m_options.m_applyToLights;
    options.m_applyToNodes = m_options.m_applyToNodes;
    options.m_transformMatrix.setCols(m_options.m_matrix.getColumn(0), m_options.m_matrix.getColumn(1), m_options.m_matrix.getColumn(2));
    options.m_flipWinding = m_options.m_flipWinding;

    // use hkxSceneUtils to process the scene
    hkxSceneUtils::transformScene( scene, options );

    // Transform Destruction specific data if available.
    hkResourceContainer* resourceContainer = (hkResourceContainer*)data.findObjectByName("Resource Data");
    if ( resourceContainer != HK_NULL )
    {
        transformResourceData(options, resourceContainer);
    }
}

namespace
{
    struct ResourceDataTransformInfo : public hkxSceneUtils::TransformInfo
    {
    public:

        hkReal m_floatScale;

        hkReal m_angleMultiplier;
    };

    void transformFloat(const hkReflect::FloatVar& f,
        hkSemanticsAttribute::Semantics type,
        const ResourceDataTransformInfo& resourceDataTransformInfo,
        const char* member)
    {
        if ( type == hkSemanticsAttribute::DISTANCE )
        {
            f.setValue(f.getValue() * resourceDataTransformInfo.m_floatScale);
        }
        else if ( type == hkSemanticsAttribute::ANGLE )
        {
            f.setValue(f.getValue() * resourceDataTransformInfo.m_angleMultiplier);
        }
        else
        {
            const char* attributeEnumString = hkReflect::getPresetsOf<hkSemanticsAttribute::Semantics>().getNameByPreset(hkReflect::Var(&type));
            HK_ASSERT_NO_MSG(0x3adc3c16, attributeEnumString);
            HK_WARN_ALWAYS(0xabbaefde, "Member (" << member <<") has unrecognized hk.Semantics attribute '" << attributeEnumString << "', only 'DISTANCE' and 'ANGLE' are allowed");
        }
    }

    void transformVector(hkVector4& vec,
        hkSemanticsAttribute::Semantics type,
        const ResourceDataTransformInfo& resourceDataTransformInfo,
        const char* member)
    {
        hkVector4 vecTemp;

        if ( type == hkSemanticsAttribute::NORMAL )
        {
            vecTemp.setMul3(resourceDataTransformInfo.m_inverseTranspose, vec);
            vecTemp.normalize3IfNotZero();
        }
        else if ( type == hkSemanticsAttribute::POSITION )
        {
            vecTemp.setMul3(resourceDataTransformInfo.m_transform, vec);
        }
        else
        {
            const char* attributeEnumString = hkReflect::getPresetsOf<hkSemanticsAttribute::Semantics>().getNameByPreset(hkReflect::Var(&type));
            HK_WARN_ALWAYS(0xabbaefde, "Member (" << member <<") has unrecognized hk.Semantics attribute '" << attributeEnumString << "', only 'NORMAL' and 'POSITION' are allowed");
        }
        vec = vecTemp;
    }

    struct TransformResourceVisitor : public hkReflect::VarVisitor<TransformResourceVisitor, void>
    {
        TransformResourceVisitor(const ResourceDataTransformInfo& resourceDataTransformInfo, hkPointerMap<void*, int>& processedObjects)
            : m_resourceDataTransformInfo(resourceDataTransformInfo)
            , m_processedObjects(processedObjects)
        {
        }

        void visit(const hkReflect::RecordVar& rec)
        {
            for (hkReflect::DeclIter<hkReflect::FieldDecl> it(rec.getType()); it.advance();)
            {
                if (hkVector4* vec = rec[it.current()].dynCast<hkVector4>())
                {
                    if (const hkSemanticsAttribute* semanticsAttributes = it.current().getType()->findAttribute<hkSemanticsAttribute>())
                    {
                        transformVector(*vec, semanticsAttributes->m_type, m_resourceDataTransformInfo, it.current().getName());
                    }
                }
                else
                {
                    dispatch(rec[it.current()]);
                }
            }
        }

        void visit(const hkReflect::PointerVar& ptr)
        {
            hkReflect::Var tgt = ptr.getValue();
            if (tgt && tgt.getType()->asRecord() && m_processedObjects.getWithDefault(tgt.getAddress(), -1) < 0)
            {
                // Flag this object as being already processed.
                m_processedObjects.insert(tgt.getAddress(), 1);
                dispatch(tgt);
            }
        }

        void visit(const hkReflect::FloatVar& f)
        {
            //
            // Skip members without a specified semantics type.
            //
            if (const hkSemanticsAttribute* semanticsAttributes = f.getType()->findAttribute<hkSemanticsAttribute>())
            {
                hkSemanticsAttribute::Semantics type = semanticsAttributes->m_type;
                transformFloat(f, type, m_resourceDataTransformInfo, hkReflect::FieldDecl(f.getType()).getName());
            }
        }

        void visit(const hkReflect::ArrayVar& arr)
        {
            //
            // Skip members without a specified semantics type.
            //
            if (const hkSemanticsAttribute* semanticsAttributes = arr.getType()->findAttribute<hkSemanticsAttribute>())
            {
                hkSemanticsAttribute::Semantics type = semanticsAttributes->m_type;
                const char* member = hkReflect::FieldDecl(arr.getType()).getName();

                hkReflect::ArrayValue arrVal = arr.getValue();
                if (arrVal.getSubType()->extendsOrEquals<hkVector4>())
                {
                    for (int i = 0; i < arrVal.getCount(); ++i)
                    {
                        hkVector4* elem = arrVal[i].dynCast<hkVector4>();
                        transformVector(*elem, type, m_resourceDataTransformInfo, member);
                    }
                }
                else if (arrVal.getSubType()->asFloat())
                {
                    for (int i = 0; i < arrVal.getCount(); ++i)
                    {
                        hkReflect::FloatVar elem = arrVal[i];
                        transformFloat(elem, type, m_resourceDataTransformInfo, member);
                    }
                }
            }
        }

        void visit(const hkReflect::VoidVar&) {}
        void visit(const hkReflect::ValueVar&) {}

        const ResourceDataTransformInfo& m_resourceDataTransformInfo;
        hkPointerMap<void*, int>& m_processedObjects;
    };
}

void hctSceneTransformFilter::transformResourceData(const hkxSceneUtils::SceneTransformOptions& options, hkResourceContainer* resourceContainer)
{
    // Construct the "resourceDataTransformInfo" object.
    ResourceDataTransformInfo resourceDataTransformInfo;
    {
        // The 4x4 matrix
        resourceDataTransformInfo.m_transform = options.m_transformMatrix;

        // Its inverse
        if( resourceDataTransformInfo.m_inverse.setInverse( resourceDataTransformInfo.m_transform ).isFailure() )
        {
            HK_WARN_ALWAYS ( 0xabbaaf02, "Inversion failed. Check the Matrix is not singular" );
            return;
        }

        // The inverse, transposed (for normals)
        resourceDataTransformInfo.m_inverseTranspose = resourceDataTransformInfo.m_inverse;
        resourceDataTransformInfo.m_inverseTranspose.transpose();

        // Its decomposition
        hkMatrixDecomposition::decomposeMatrix(resourceDataTransformInfo.m_transform, resourceDataTransformInfo.m_decomposition);

        // The scaling factor for standard FLOAT values.
        const hkVector4& scaleVector = resourceDataTransformInfo.m_decomposition.m_scale;
        resourceDataTransformInfo.m_floatScale = hkReal(scaleVector.length3()) / hkMath::sqrt(3.0f);

        // Multiplier for angles.
        resourceDataTransformInfo.m_angleMultiplier = 1.0f;
        if ( resourceDataTransformInfo.m_decomposition.m_flips )
        {
            resourceDataTransformInfo.m_angleMultiplier = -1.0f;
        }
    }

    hkPointerMap<void*, int> processedObjects;

    //
    // Process all resources (i.e. objects) in the "Resource Data" container.
    //
    {
        hkArray<hkResourceHandle*> handles; resourceContainer->findAllResourceRecursively( handles );

        for ( int i = 0; i < handles.getSize(); i++  )
        {
            hkResourceHandle* handle = handles[i];

            // We need to retrieve the class information from the registry because the class information in the handle might be missing data (e.g. attributes)
            const hkReflect::Type* klass  = hctSdkUtils::getTypeByName(handle->getObjectType()->getName());
            void*          object = handle->getObject();

            //
            // Only process object if both object and class information are valid and if the object hasn't been processed yet.
            //
            if ( klass && object && (processedObjects.getWithDefault(object, -1) < 0) )
            {
                if (const hkReflect::RecordType* rec = klass->asRecord())
                {
                    // Flag this object as being already processed.
                    processedObjects.insert(object, 1);
                    TransformResourceVisitor(resourceDataTransformInfo, processedObjects).dispatch(hkReflect::Var(object, klass));
                }
            }
        }
    }
}

void hctSceneTransformFilter::setOptions(const hkReflect::Var& optVar)
{
    if (hctSceneTransformOptions* options = hctFilterUtils::getNativeOptions<hctSceneTransformOptions>(optVar))
    {
        m_options = *options;
        delete options;
    }
}

void hctSceneTransformFilter::getOptions(hkReflect::Any& buffer) const
{
    buffer.setFromObj( m_options );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
